/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/impl/matrixfuncs', [
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/matrix'
], function (ErrorCode, Utils, FormulaUtils, Matrix) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions dealing with matrixes.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     * In spreadsheet documents, matrixes can be given as literals in formulas,
     * or can be extracted from rectangular cell ranges. The client-side
     * formula engine provides the function signature type 'mat' (and its sub
     * types such as 'mat:num'), and the class Matrix whose instances contain
     * the elements of a matrix in a two-dimensional array.
     *
     *************************************************************************/

    var RCONVERT_ALL_SKIP_EMPTY = {
        valMode: 'exact',
        matMode: 'skip',
        refMode: 'skip',
        emptyParam: false, // skip empty parameter
        complexRef: false // do not accept multi-range and multi-sheet references
    };

    // private global functions ===============================================

    function sumMatrices(mat1, mat2, op) {
        if (mat1.cols() !== mat2.cols()) { throw ErrorCode.NA; }
        if (mat1.rows() !== mat2.rows()) { throw ErrorCode.NA; }

        var arr1 = this.getNumbersAsArray(mat1, RCONVERT_ALL_SKIP_EMPTY);
        var arr2 = this.getNumbersAsArray(mat2, RCONVERT_ALL_SKIP_EMPTY);

        if (arr1.length !== arr2.length) { throw ErrorCode.NA; }

        var sum = 0;
        _.each(arr1, function (val1, index) {
            var val2 = arr2[index];
            if (_.isNumber(val1) && _.isNumber(val2)) {
                sum += op(val1, val2);
            }
        });
        return sum;
    }

    // exports ================================================================

    return {

        MDETERM: {
            category: 'matrix',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'mat:num',
            resolve: (function () {

                function determ(matrix) {
                    // http://www.mathebibel.de/3x3-determinanten-berechnen
                    // https://www.youtube.com/watch?v=AnezNBuRkEY

                    var size = matrix.cols();

                    if (size !== matrix.rows()) {
                        // width of matrix must match height of matrix
                        throw ErrorCode.NUM;
                    } else if (size === 0) {
                        throw ErrorCode.VALUE;
                    } else if (size === 1) {
                        return determ1x1(matrix);
                    } else {
                        return determLaplace(matrix);
                    }
                }

                function determ1x1(matrix) {
                    return matrix.get(0, 0);
                }

                function determLaplace(matrix) {
                    var size = matrix.cols();

                    var firstCol = [];
                    _.times(size, function (index) {
                        var firstValue = matrix.get(0, index);
                        if (index % 2) {
                            firstValue *= -1;
                        }
                        firstCol.push(firstValue);
                    });

                    var result = 0;
                    _.each(firstCol, function (firstValue, index) {
                        var subMatrix = [];
                        _.times(size, function (j) {
                            if (j === index) { return; }
                            var line = [];
                            _.times(size - 1, function (i) {
                                line.push(matrix.get(i + 1, j));
                            });
                            subMatrix.push(line);
                        });

                        result += firstValue * determ(new Matrix(subMatrix));
                    });
                    return result;
                }

                return determ;
            }())
        },

        MINVERSE: {
            category: 'matrix',
            minParams: 1,
            maxParams: 1,
            type: 'mat',
            signature: 'mat:num',
            resolve: function (matrix) {
                // http://blog.acipo.com/matrix-inversion-in-javascript/
                // by  Andrew Ippoliti

                // I use Guassian Elimination to calculate the inverse:
                // (1) 'augment' the matrix (left) by the identity (on the right)
                // (2) Turn the matrix on the left into the identity by elemetry row ops
                // (3) The matrix on the right is the inverse (was the identity matrix)
                // There are 3 elemtary row ops: (I combine b and c in my code)
                // (a) Swap 2 rows
                // (b) Multiply a row by a scalar
                // (c) Add 2 rows

                var dim = matrix.cols();

                //if the matrix isn't square: exit (error)
                if (dim !== matrix.rows()) { throw ErrorCode.VALUE; }

                //create the identity matrix, and a copy of the original
                var i = 0, ii = 0, j = 0, e = 0;
                var identity = Matrix.create(dim, dim, 0), copy = Matrix.create(dim, dim, 0);
                for (i = 0; i < dim; i += 1) {
                    for (j = 0; j < dim; j += 1) {

                        //if we're on the diagonal, put a 1 (for identity)
                        if (i === j) { identity.set(i, j, 1); } else { identity.set(i, j, 0); }

                        // Also, make the copy of the original
                        copy.set(i, j, matrix.get(i, j));
                    }
                }

                // Perform elementary row operations
                for (i = 0; i < dim; i += 1) {
                    // get the element e on the diagonal
                    e = copy.get(i, i);

                    // if we have a 0 on the diagonal (we'll need to swap with a lower row)
                    if (e === 0) {
                        //look through every row below the i'th row
                        for (ii = i + 1; ii < dim; ii += 1) {
                            //if the ii'th row has a non-0 in the i'th col
                            if (copy.get(ii, i) !== 0) {
                                //it would make the diagonal have a non-0 so swap it
                                for (j = 0; j < dim; j++) {
                                    e = copy.get(i, j);       //temp store i'th row
                                    copy.set(i, j, copy.get(ii, j));//replace i'th row by ii'th
                                    copy.set(ii, j, e);      //repace ii'th by temp
                                    e = identity.get(i, j);       //temp store i'th row
                                    identity.set(i, j, identity.get(ii, j));//replace i'th row by ii'th
                                    identity.set(ii, j, e);      //repace ii'th by temp
                                }
                                //don't bother checking other rows since we've swapped
                                break;
                            }
                        }
                        //get the new diagonal
                        e = copy.get(i, i);
                        //if it's still 0, not invertable (error)
                        if (e === 0) { throw ErrorCode.NUM; }
                    }

                    // Scale this row down by e (so we have a 1 on the diagonal)
                    for (j = 0; j < dim; j++) {
                        copy.set(i, j, copy.get(i, j) / e); //apply to original matrix
                        identity.set(i, j, identity.get(i, j) / e); //apply to identity
                    }

                    // Subtract this row (scaled appropriately for each row) from ALL of
                    // the other rows so that there will be 0's in this column in the
                    // rows above and below this one
                    for (ii = 0; ii < dim; ii++) {
                        // Only apply to other rows (we want a 1 on the diagonal)
                        if (ii === i) { continue; }

                        // We want to change this element to 0
                        e = copy.get(ii, i);

                        // Subtract (the row above(or below) scaled by e) from (the
                        // current row) but start at the i'th column and assume all the
                        // stuff left of diagonal is 0 (which it should be if we made this
                        // algorithm correctly)
                        for (j = 0; j < dim; j++) {
                            copy.set(ii, j, copy.get(ii, j) - e * copy.get(i, j)); //apply to original matrix
                            identity.set(ii, j, identity.get(ii, j) - e * identity.get(i, j)); //apply to identity
                        }
                    }
                }

                //we've done all operations, copy should be the identity
                //matrix identity should be the inverse:
                return identity;
            }
        },

        MMULT: {
            category: 'matrix',
            minParams: 2,
            maxParams: 2,
            type: 'mat',
            signature: 'mat:num mat:num',
            resolve: function (matrix1, matrix2) {

                // width of first matrix must match height of second matrix
                if (matrix1.cols() !== matrix2.rows()) { throw ErrorCode.VALUE; }

                // calculate all elements of the result matrix
                return Matrix.generate(matrix1.rows(), matrix2.cols(), function (row, col) {
                    var elem = 0;
                    for (var i = 0, l = matrix1.cols(); i < l; i += 1) {
                        elem += (matrix1.get(row, i) * matrix2.get(i, col));
                    }
                    return elem;
                });
            }
        },

        MUNIT: {
            category: 'matrix',
            name: { ooxml: '_xlfn.MUNIT' },
            minParams: 1,
            maxParams: 1,
            type: 'mat',
            signature: 'val:int',
            resolve: function (dim) {
                if (dim < 1) { throw ErrorCode.VALUE; }
                if (dim > Math.min(FormulaUtils.MAX_MATRIX_ROW_COUNT, FormulaUtils.MAX_MATRIX_COL_COUNT)) { throw FormulaUtils.UNSUPPORTED_ERROR; }
                return Matrix.generate(dim, dim, function (row, col) {
                    return (row === col) ? 1 : 0;
                });
            }
        },

        SUMX2MY2: {
            category: 'matrix',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'mat mat',
            resolve: function (mat1, mat2) {
                return sumMatrices.call(this, mat1, mat2, function (v1, v2) {
                    return Math.pow(v1, 2) - Math.pow(v2, 2);
                });
            }
        },

        SUMX2PY2: {
            category: 'matrix',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'mat mat',
            resolve: function (mat1, mat2) {
                return sumMatrices.call(this, mat1, mat2, function (v1, v2) {
                    return Math.pow(v1, 2) + Math.pow(v2, 2);
                });
            }
        },

        SUMXMY2: {
            category: 'matrix',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'mat mat',
            resolve: function (mat1, mat2) {
                return sumMatrices.call(this, mat1, mat2, function (v1, v2) {
                    return Math.pow(v1 - v2, 2);
                });
            }
        },

        TRANSPOSE: {
            category: 'matrix',
            minParams: 1,
            maxParams: 1,
            type: 'mat',
            signature: 'mat:any',
            resolve: function (matrix) {
                return Matrix.generate(matrix.cols(), matrix.rows(), function (row, col) {
                    return matrix.get(col, row);
                });
            }
        }
    };

});
