/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/grammarconfig', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/locale/formatter',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/sheetref'
], function (Utils, LocaleData, Formatter, SheetUtils, FormulaUtils, SheetRef) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode,
        Address = SheetUtils.Address,
        Range = SheetUtils.Range;

    // private global functions ===============================================

    /**
     * Returns a regular expression that matches strings in R1C1 notation with
     * optional absolute row and column parts.
     *
     * @param {String} prefixChars
     *  The actual prefix characters for the row and column part. MUST be a
     *  string with exactly two characters (row and column).
     *
     * @returns {RegExp}
     *  A regular expression that matches absolute references in R1C1 notation.
     *  Either part may be missing completely (references to entire columns or
     *  rows), or the prefix character may be present without index. Matching
     *  strings are for example 'R2C3', 'R2C', 'RC3', 'RC', 'R2', 'C3', 'R',
     *  'C', or the empty string (!).
     */
    var getR1C1RegExp = _.memoize(function (prefixChars) {
        return new RegExp('^(?:' + prefixChars[0] + '(\\d*))?(?:' + prefixChars[1] + '(\\d*))?$', 'i');
    });

    /**
     * Returns whether the passed text is a simple R1C1 reference for the
     * passed prefix characters (either absolute column/row index not
     * enclosed in brackets, or missing column/row index).
     *
     * @param {SpreadsheetModel} docModel
     *  The document model needed to check the maximum column/row index of
     *  sheet names that look like cell addresses.
     *
     * @param {String} prefixChars
     *  The actual prefix characters for the row and column part. MUST be a
     *  string with exactly two characters (row and column).
     *
     * @param {String} text
     *  The string to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed text is a simple R1C1 reference for the passed
     *  prefix characters.
     */
    function isSimpleR1C1Reference(docModel, prefixChars, text) {

        // do not accept empty strings (would be matched by the regular expression)
        if (text.length === 0) { return false; }

        // check that text matches the regular expression
        var matches = getR1C1RegExp(prefixChars).exec(text);
        if (!matches) { return false; }

        // extract column and row index (row before column in R1C1 notation!)
        var row = matches[1] ? (parseInt(matches[1], 10) - 1) : 0,
            col = matches[2] ? (parseInt(matches[2], 10) - 1) : 0;

        // column and row index must be valid in the document
        return docModel.isValidAddress(new Address(col, row));
    }

    // class GrammarConfig ====================================================

    /**
     * An instance of this class represents all settings of a specific formula
     * grammar and file format.
     *
     * @constructor
     *
     * @extends BaseObject
     *
     * @property {String} GRAMMAR
     *  The identifier of the formula grammar, as passed to the constructor.
     *
     * @property {String} REF_SYNTAX
     *  The identifier of the syntax of cell references:
     *  - 'ooxml' for the syntax used in native and localized OOXML grammars
     *      (references in A1 style, exclamation mark as sheet name separator,
     *      sheets are always absolute, apostrophes enclose entire range of
     *      sheet names, e.g.: ='Sheet1:Sheet 2'!A1:B2).
     *  - 'odfui' for the syntax used in the localized ODF grammar (references
     *      in A1 style, exclamation mark as sheet name separator, sheets can
     *      be absolute or relative, apostrophes enclose single sheet names in
     *      a range of sheets, e.g.: =Sheet1:$'Sheet 2'!A1:B2).
     *  - 'of' for the native OpenFormula syntax (entire reference enclosed in
     *      brackets, references in A1 style, row or column index can be #REF!
     *      errors, period as sheet name separator, sheets can be absolute or
     *      relative, apostrophes enclose single sheet names in a range of
     *      sheets, e.g.: =[Sheet1.A1:$'Sheet 2'.B#REF!]).
     *
     * @property {String} DEC
     *  The decimal separator character.
     *
     * @property {String} SEP
     *  The separator character in function parameter lists.
     *
     * @property {String} MAT_OPEN
     *  The leading character of a matrix literal.
     *
     * @property {String} MAT_ROW
     *  The separator character between rows in a matrix literal.
     *
     * @property {String} MAT_COL
     *  The separator character between row elements in a matrix literal.
     *
     * @property {String} MAT_CLOSE
     *  The trailing character of a matrix literal.
     *
     * @property {Object} RE
     *  A map with regular expressions needed to parse formula expressions.
     */
    function GrammarConfig(resource, grammar) {

        // self reference
        var self = this;

        // special behavior depending on file format
        var odf = resource.getFileFormat() === 'odf';

        // different behavior for CalcEngine mode vs. local formula mode
        var remote = resource.isCalcEngine();

        // whether to used localized strings
        var localized = { op: false, ui: true }[grammar];

        // whether to use OpenFormula style (e.g. references as [Sheet1.A1:.B2] instead of Sheet1!A1:B2)
        var openFormula = odf && !remote && !localized;

        // the collection of boolean descriptors
        var booleanCollection = resource.getBooleanCollection();

        // the collection of error code descriptors
        var errorCollection = resource.getErrorCollection();

        // the collection of operator descriptors
        var operatorCollection = resource.getOperatorCollection();

        // the collection of function descriptors
        var functionCollection = resource.getFunctionCollection();

        // the separator characters, and specific error codes used in regular expressions
        var DEC = localized ? LocaleData.DEC : '.';
        var GROUP = localized ? LocaleData.GROUP : ',';
        var SEP = resource.getSeparator(localized);

        // a simple number formatter to generate the string representation of numbers
        var formatter = new Formatter({ dec: DEC, group: GROUP });

        // prefix characters for R1C1 notation
        var PREFIX_CHARS = resource.getR1C1PrefixChars(localized);

        // specific error codes used in regular expressions, or when generating text representations
        var REF_ERROR = errorCollection.getName(ErrorCode.REF.key, localized);
        var NA_ERROR = errorCollection.getName(ErrorCode.NA.key, localized);

        // A whitespace or other non-printable character.
        var WHITESPACE_PATTERN = '[\\s\x00-\x1f\x80-\x9f]';

        // RE character class for the leading character of a simple sheet name (without apostrophes).
        var SHEET_LEADING_CHAR_CLASS = '[a-z_\xa1-\u2027\u202a-\uffff]';

        // RE character class for other characters of a simple sheet name (without apostrophes).
        var SHEET_SIMPLE_CHAR_CLASS = odf ? '[\\w\xa1-\u2027\u202a-\uffff]' : '[\\w.\xa1-\u2027\u202a-\uffff]';

        // RE character class for any character of a complex sheet name (enclosed in apostrophes).
        var SHEET_COMPLEX_CHAR_CLASS = '[^\x00-\x1f\x80-\x9f\\[\\]\'*?:/\\\\]';

        // RE character class for a characters of a simple external file name (without apostrophes).
        var FILE_SIMPLE_CHAR_CLASS = '[\\w.\\\\\xa1-\u2027\u202a-\uffff]';

        // RE character class for the leading character of a name or identifier (OOXML: allow leading backslash).
        var NAME_LEADING_CHAR_CLASS = odf ? '[a-z_\xa1-\u2027\u202a-\uffff]' : '[a-z_\\\\\xa1-\u2027\u202a-\uffff]';

        // RE character class for other characters of a name or identifier (OOXML: allow backslash).
        var NAME_INNER_CHAR_CLASS = odf ? '[\\w?\xa1-\u2027\u202a-\uffff]' : '[\\w.\\\\?\xa1-\u2027\u202a-\uffff]';

        // RE character class for other characters of a function name (ODF: allow periods in difference to defined names).
        var FUNC_INNER_CHAR_CLASS = odf ? '[\\w?.\xa1-\u2027\u202a-\uffff]' : NAME_INNER_CHAR_CLASS;

        // RE pattern for a simple sheet name.
        // - Group 1: The sheet name.
        var SHEET_NAME_SIMPLE_PATTERN = '(' + SHEET_LEADING_CHAR_CLASS + SHEET_SIMPLE_CHAR_CLASS + '*)';

        // RE pattern for a complex sheet name (with embedded double-apostrophes, without surrounding apostrophes).
        // - Group 1: The sheet name.
        var SHEET_NAME_COMPLEX_PATTERN = '(' + SHEET_COMPLEX_CHAR_CLASS + '+(?:(?:\'\')+' + SHEET_COMPLEX_CHAR_CLASS + '+)*)';

        // RE pattern for a simple external file name (enclosed in brackets, without apostrophes).
        // - Group 1: The name of the external file (the part inside the brackets).
        var FILE_NAME_SIMPLE_PATTERN = '(?:\\[(' + FILE_SIMPLE_CHAR_CLASS + '+)\\])';

        // RE pattern for a #REF! error code
        // - Group 1: The matched #REF! error code.
        var REF_ERROR_PATTERN = '(' + _.escapeRegExp(REF_ERROR) + ')';

        // RE look-ahead pattern to exclude an opening parenthesis and an exclamation mark
        // after another pattern. All valid inner characters of names must be excluded too,
        // otherwise the previous groups would match less characters than they should.
        var TERMINATE_REF_PATTERN = '(?!\\(|!|' + NAME_INNER_CHAR_CLASS + ')';

        // RE pattern for a single (implicitly absolute) sheet name (without address separator), e.g. Sheet1 in the formula
        // =Sheet1!A1, or a broken sheet reference after deleting the referenced sheet, e.g. #REF! in the formula =#REF!!A1.
        // - Group 1: The name of the sheet (simple sheet name).
        // - Group 2: The name of the sheet (complex sheet name).
        // - Group 3: The #REF! error code as sheet name.
        // Either group 1 or group 2 will match the sheet name, or group 3 will contain the error code (the other groups will be empty).
        var ABS_SHEET_NAME_PATTERN = '(?:' + SHEET_NAME_SIMPLE_PATTERN + '|\'' + SHEET_NAME_COMPLEX_PATTERN + '\'|' + REF_ERROR_PATTERN + ')';

        // RE pattern for a range of (implicitly absolute) sheet names (without address separator), e.g. Sheet1:Sheet2 in the
        // formula =SUM(Sheet1:Sheet2!A1).
        // - Group 1: The name of the first sheet (simple sheet names).
        // - Group 2: The name of the second sheet (simple sheet names).
        // - Group 3: The name of the first sheet (complex sheet names).
        // - Group 4: The name of the second sheet (complex sheet names).
        // Either group 1 or group 3 will match the first sheet name (the other group will be empty).
        // Either group 2 or group 4 will match the second sheet name (the other group will be empty).
        var ABS_SHEET_RANGE_PATTERN = '(?:' + SHEET_NAME_SIMPLE_PATTERN + ':' + SHEET_NAME_SIMPLE_PATTERN + '|\'' + SHEET_NAME_COMPLEX_PATTERN + ':' + SHEET_NAME_COMPLEX_PATTERN + '\')';

        // RE pattern for a single (implicitly absolute) sheet name (with optional absolute marker, without address separator), e.g.
        // $Sheet1 in the formula =$Sheet1!A1, or a broken sheet reference after deleting the referenced sheet, e.g. $#REF! in the
        // formula =$#REF!!A1.
        // - Group 1: The absolute marker.
        // - Group 2: The name of the sheet (simple sheet name).
        // - Group 3: The name of the sheet (complex sheet name).
        // - Group 4: The #REF! error code as sheet name.
        // Either group 2 or group 3 will match the sheet name, or group 4 will contain the error code (the other groups will be empty).
        var REL_SHEET_NAME_PATTERN = '(\\$)?' + ABS_SHEET_NAME_PATTERN;

        // RE pattern for a range of (implicitly absolute) sheet names (with optional absolute markers, without address separator), e.g.
        // Sheet1:$Sheet2 in the formula =SUM(Sheet1:$Sheet2!A1).
        // - Group 1: The absolute marker of the first sheet.
        // - Group 2: The name of the first sheet (simple sheet name).
        // - Group 3: The name of the first sheet (complex sheet name).
        // - Group 4: The absolute marker of the second sheet.
        // - Group 5: The name of the second sheet (simple sheet name).
        // - Group 6: The name of the second sheet (complex sheet name).
        // Either group 2 or group 3 will match the first sheet name (the other group will be empty).
        // Either group 5 or group 6 will match the second sheet name (the other group will be empty).
        var REL_SHEET_RANGE_PATTERN = '(\\$)?(?:' + SHEET_NAME_SIMPLE_PATTERN + '|\'' + SHEET_NAME_COMPLEX_PATTERN + '\'):(\\$)?(?:' + SHEET_NAME_SIMPLE_PATTERN + '|\'' + SHEET_NAME_COMPLEX_PATTERN + '\')';

        // RE pattern for a column name (absolute or relative).
        // - Group 1: The absolute marker.
        // - Group 2: The column name.
        var COL_PATTERN = '(\\$)?([a-z]+)';

        // RE pattern for a row name (absolute or relative).
        // - Group 1: The absolute marker.
        // - Group 2: The row name.
        var ROW_PATTERN = '(\\$)?([0-9]+)';

        // RE pattern for a cell address.
        // - Group 1: The absolute marker for the column.
        // - Group 2: The name of the column.
        // - Group 3: The absolute marker for the row.
        // - Group 4: The name of the row.
        var CELL_PATTERN = COL_PATTERN + ROW_PATTERN;

        // RE pattern for a cell range address.
        // - Group 1: The absolute marker for the first column.
        // - Group 2: The name of the first column.
        // - Group 3: The absolute marker for the first row.
        // - Group 4: The name of the first row.
        // - Group 5: The absolute marker for the second column.
        // - Group 6: The name of the second column.
        // - Group 7: The absolute marker for the second row.
        // - Group 8: The name of the second row.
        var CELL_RANGE_PATTERN = CELL_PATTERN + ':' + CELL_PATTERN;

        // RE pattern for a column range reference, e.g. $C:$C.
        // - Group 1: The absolute marker for the first column.
        // - Group 2: The name of the first column.
        // - Group 3: The absolute marker for the second column.
        // - Group 4: The name of the second column.
        var COL_RANGE_PATTERN = COL_PATTERN + ':' + COL_PATTERN;

        // RE pattern for a row range reference, e.g. $2:$2.
        // - Group 1: The absolute marker for the first row.
        // - Group 2: The name of the first row.
        // - Group 3: The absolute marker for the second row.
        // - Group 4: The name of the second row.
        var ROW_RANGE_PATTERN = ROW_PATTERN + ':' + ROW_PATTERN;

        // RE pattern for a complete name or identifier.
        // - Group 1: The entire name.
        var NAME_PATTERN = '(' + NAME_LEADING_CHAR_CLASS + NAME_INNER_CHAR_CLASS + '*)';

        // RE pattern for a complete function name.
        // - Group 1: The entire function name.
        var FUNC_PATTERN = '(' + NAME_LEADING_CHAR_CLASS + FUNC_INNER_CHAR_CLASS + '*)';

        // RE pattern for a column name (absolute or relative) in OpenFormula syntax (#REF! error allowed).
        // - Group 1: The absolute marker.
        // - Group 2: The column name, or the #REF! error code.
        var OF_COL_PATTERN = '(\\$)?([a-z]+|' + _.escapeRegExp(REF_ERROR) + ')';

        // RE pattern for a row name (absolute or relative) in OpenFormula syntax (#REF! error allowed).
        // - Group 1: The absolute marker.
        // - Group 2: The column name, or the #REF! error code.
        var OF_ROW_PATTERN = '(\\$)?([0-9]+|' + _.escapeRegExp(REF_ERROR) + ')';

        // RE pattern for a cell address in OpenFormula syntax (with leading period, #REF! errors for column or row allowed).
        // - Group 1: The absolute marker for the column.
        // - Group 2: The name of the column, or the #REF! error code.
        // - Group 3: The absolute marker for the row.
        // - Group 4: The name of the row, or the #REF! error code.
        var OF_CELL_PATTERN = '\\.' + OF_COL_PATTERN + OF_ROW_PATTERN;

        // RE pattern for a cell address with sheet name in OpenFormula syntax (#REF! errors for column or row allowed).
        // - Group 1: The absolute marker for the sheet.
        // - Group 2: The name of the sheet (simple sheet name).
        // - Group 3: The name of the sheet (complex sheet name).
        // - Group 4: The #REF! error code as sheet name.
        // - Group 5: The absolute marker for the column.
        // - Group 6: The name of the column, or the #REF! error code.
        // - Group 7: The absolute marker for the row.
        // - Group 8: The name of the row, or the #REF! error code.
        var OF_CELL_3D_PATTERN = REL_SHEET_NAME_PATTERN + OF_CELL_PATTERN;

        // Matches a simple sheet name, or a range of simple sheet names, that do not need to be
        // enclosed in apostrophes. Supports an optional leading reference to an external file
        // (simple name), enclosed in brackets.
        // Examples: 'Sheet1', 'Sheet1:Sheet2', '[file1.xlsx]Sheet1', '[path\to\file1.xlsx]Sheet1:Sheet2'.
        // - Group 1: The name of the external file (the part inside the brackets), may be undefined.
        // - Group 2: The first sheet name.
        // - Group 3: The second sheet name, may be undefined.
        var SIMPLE_SHEET_NAME_RE = new RegExp('^' + FILE_NAME_SIMPLE_PATTERN + '?' + SHEET_NAME_SIMPLE_PATTERN + '(?::' + SHEET_NAME_SIMPLE_PATTERN + ')?$', 'i');

        // matches a valid label for a defined name
        var VALID_NAME_RE = new RegExp('^' + NAME_PATTERN + '$', 'i');

        // public properties --------------------------------------------------

        // the grammar identifier
        this.GRAMMAR = grammar;

        // the reference syntax identifier
        this.REF_SYNTAX = openFormula ? 'of' : odf ? 'odfui' : 'ooxml';

        // separator characters
        this.DEC = DEC;
        this.SEP = SEP;
        this.MAT_OPEN = '{';
        this.MAT_ROW = (localized || openFormula) ? '|' : ';';
        this.MAT_COL = (localized || openFormula) ? ';' : ',';
        this.MAT_CLOSE = '}';

        // map of all regular expressions
        this.RE = {

            // Any whitespace or other non-printable characters.
            WHITESPACE: new RegExp('^' + WHITESPACE_PATTERN + '+'),

            // String literals: Enclosed in double quotes, embedded double quotes
            // are represented by a sequence of two double quote characters.
            STRING_LIT: /^(".*?")+/,

            // Boolean literals: Must not be followed by any character valid inside an identifier.
            // This prevents to match 'trueA' or 'false_1' or similar name tokens, as well as TRUE()
            // and FALSE() function calls, or references with sheet names such as 'TRUE!A1'.
            BOOLEAN_LIT: new RegExp('^' + booleanCollection.getAllPattern(localized) + TERMINATE_REF_PATTERN, 'i'),

            // Error code literals (only supported error codes, not any string
            // starting with a hash character).
            ERROR_LIT: new RegExp('^' + errorCollection.getAllPattern(localized) + '(?!' + NAME_INNER_CHAR_CLASS + ')', 'i'),

            // Opening parenthesis of a matrix literal.
            MATRIX_OPEN: new RegExp('^' + _.escapeRegExp(this.MAT_OPEN)),

            // The column separator in a matrix literal.
            MATRIX_COL_SEPARATOR: new RegExp('^' + _.escapeRegExp(this.MAT_COL)),

            // The row separator in a matrix literal.
            MATRIX_ROW_SEPARATOR: new RegExp('^' + _.escapeRegExp(this.MAT_ROW)),

            // Closing parenthesis of a matrix literal.
            MATRIX_CLOSE: new RegExp('^' + _.escapeRegExp(this.MAT_CLOSE)),

            // The list/parameter separator character.
            SEPARATOR: new RegExp('^' + _.escapeRegExp(this.SEP)),

            // All unary and binary operators.
            // - Group 1: The name of the matched operator.
            OPERATOR: new RegExp('^' + operatorCollection.getAllPattern(localized)),

            // The opening parenthesis.
            OPEN: /^\(/,

            // The closing parenthesis.
            CLOSE: /^\)/,

            // A local cell reference, e.g. A1 or $A$1, but without trailing parenthesis,
            // e.g. F2(), and without trailing sheet separator, e.g. F2!A1.
            // - Group 1: The absolute marker for the column.
            // - Group 2: The column name.
            // - Group 3: The absolute marker for the row.
            // - Group 4: The row name.
            CELL_REF: new RegExp('^' + CELL_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A cell reference with implicitly absolute sheet, e.g. Sheet1!$A$1, but without trailing parenthesis,
            // e.g. Module1!F2() which is a call of the macro function F2() contained in Module1.
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The #REF! error code as sheet name.
            // - Group 4: The absolute marker for the column.
            // - Group 5: The column name.
            // - Group 6: The absolute marker for the row.
            // - Group 7: The row name.
            CELL_ABS3D_REF: new RegExp('^' + ABS_SHEET_NAME_PATTERN + '!' + CELL_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A cell reference with relative/absolute sheet, e.g. $Sheet1!$A$1, but without trailing parenthesis,
            // e.g. Module1!F2() which is a call of the macro function F2() contained in Module1.
            // - Group 1: The absolute marker for the sheet.
            // - Group 2: The simple sheet name.
            // - Group 3: The complex sheet name.
            // - Group 4: The #REF! error code as sheet name.
            // - Group 5: The absolute marker for the column.
            // - Group 6: The column name.
            // - Group 7: The absolute marker for the row.
            // - Group 8: The row name.
            CELL_REL3D_REF: new RegExp('^' + REL_SHEET_NAME_PATTERN + '!' + CELL_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A cell reference in multiple implicitly absolute sheets, e.g. Sheet1:Sheet2!$A$1.
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The absolute marker for the column.
            // - Group 6: The column name.
            // - Group 7: The absolute marker for the row.
            // - Group 8: The row name.
            CELL_ABSCUBE_REF: new RegExp('^' + ABS_SHEET_RANGE_PATTERN + '!' + CELL_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A cell reference in multiple relative/absolute sheets, e.g. Sheet1:$Sheet2!$A$1.
            // - Group 1: The absolute marker of the first sheet.
            // - Group 2: The name of the first sheet (simple sheet name).
            // - Group 3: The name of the first sheet (complex sheet name).
            // - Group 4: The absolute marker of the second sheet.
            // - Group 5: The name of the second sheet (simple sheet name).
            // - Group 6: The name of the second sheet (complex sheet name).
            // - Group 7: The absolute marker for the column.
            // - Group 8: The column name.
            // - Group 9: The absolute marker for the row.
            // - Group 10: The row name.
            CELL_RELCUBE_REF: new RegExp('^' + REL_SHEET_RANGE_PATTERN + '!' + CELL_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A local range reference, e.g. A1:B2 or $A$1:$B$2, but without trailing parenthesis, e.g. A1:F2(),
            // which is the cell address A1, and the result of the function call F2(), connected with a range operator.
            // - Group 1: The absolute marker for the first column.
            // - Group 2: The name of the first column.
            // - Group 3: The absolute marker for the first row.
            // - Group 4: The name of the first row.
            // - Group 5: The absolute marker for the second column.
            // - Group 6: The name of the second column.
            // - Group 7: The absolute marker for the second row.
            // - Group 8: The name of the second row.
            CELL_RANGE_REF: new RegExp('^' + CELL_RANGE_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A range reference with implicitly absolute sheet, e.g. Sheet1!$A$1:$B$2, but without trailing parenthesis,
            // e.g. Sheet1!A1:F2().
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The #REF! error code as sheet name.
            // - Group 4: The absolute marker for the first column.
            // - Group 5: The name of the first column.
            // - Group 6: The absolute marker for the first row.
            // - Group 7: The name of the first row.
            // - Group 8: The absolute marker for the second column.
            // - Group 9: The name of the second column.
            // - Group 10: The absolute marker for the second row.
            // - Group 11: The name of the second row.
            CELL_RANGE_ABS3D_REF: new RegExp('^' + ABS_SHEET_NAME_PATTERN + '!' + CELL_RANGE_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A range reference with relative/absolute sheet, e.g. $Sheet1!$A$1:$B$2, but without trailing parenthesis, e.g. Sheet1!A1:F2().
            // - Group 1: The absolute marker for the sheet.
            // - Group 2: The simple sheet name.
            // - Group 3: The complex sheet name.
            // - Group 4: The #REF! error code as sheet name.
            // - Group 5: The absolute marker for the first column.
            // - Group 6: The name of the first column.
            // - Group 7: The absolute marker for the first row.
            // - Group 8: The name of the first row.
            // - Group 9: The absolute marker for the second column.
            // - Group 10: The name of the second column.
            // - Group 11: The absolute marker for the second row.
            // - Group 12: The name of the second row.
            CELL_RANGE_REL3D_REF: new RegExp('^' + REL_SHEET_NAME_PATTERN + '!' + CELL_RANGE_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A range reference in multiple implicitly absolute sheets, e.g. Sheet1:Sheet2!$A$1:$A$1.
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The absolute marker for the first column.
            // - Group 6: The name of the first column.
            // - Group 7: The absolute marker for the first row.
            // - Group 8: The name of the first row.
            // - Group 9: The absolute marker for the second column.
            // - Group 10: The name of the second column.
            // - Group 11: The absolute marker for the second row.
            // - Group 12: The name of the second row.
            CELL_RANGE_ABSCUBE_REF: new RegExp('^' + ABS_SHEET_RANGE_PATTERN + '!' + CELL_RANGE_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A range reference in multiple relative/absolute sheets, e.g. Sheet1:$Sheet2!$A$1:$A$1.
            // - Group 1: The absolute marker of the first sheet.
            // - Group 2: The name of the first sheet (simple sheet name).
            // - Group 3: The name of the first sheet (complex sheet name).
            // - Group 4: The absolute marker of the second sheet.
            // - Group 5: The name of the second sheet (simple sheet name).
            // - Group 6: The name of the second sheet (complex sheet name).
            // - Group 7: The absolute marker for the first column.
            // - Group 8: The name of the first column.
            // - Group 9: The absolute marker for the first row.
            // - Group 10: The name of the first row.
            // - Group 11: The absolute marker for the second column.
            // - Group 12: The name of the second column.
            // - Group 13: The absolute marker for the second row.
            // - Group 14: The name of the second row.
            CELL_RANGE_RELCUBE_REF: new RegExp('^' + REL_SHEET_RANGE_PATTERN + '!' + CELL_RANGE_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A local column range, e.g. A:B or $C:$C (always with colon).
            // - Group 1: The absolute marker for the first column.
            // - Group 2: The name of the first column.
            // - Group 3: The absolute marker for the second column.
            // - Group 4: The name of the second column.
            COL_RANGE_REF: new RegExp('^' + COL_RANGE_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A column range with implicitly absolute sheet, e.g. Sheet1!$C:$C (always with colon).
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The #REF! error code as sheet name.
            // - Group 4: The absolute marker for the first column.
            // - Group 5: The name of the first column.
            // - Group 6: The absolute marker for the second column.
            // - Group 7: The name of the second column.
            COL_RANGE_ABS3D_REF: new RegExp('^' + ABS_SHEET_NAME_PATTERN + '!' + COL_RANGE_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A column range with relative/absolute sheet, e.g. $Sheet1!$C:$C (always with colon).
            // - Group 1: The absolute marker for the sheet.
            // - Group 2: The simple sheet name.
            // - Group 3: The complex sheet name.
            // - Group 4: The #REF! error code as sheet name.
            // - Group 5: The absolute marker for the first column.
            // - Group 6: The name of the first column.
            // - Group 7: The absolute marker for the second column.
            // - Group 8: The name of the second column.
            COL_RANGE_REL3D_REF: new RegExp('^' + REL_SHEET_NAME_PATTERN + '!' + COL_RANGE_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A column range in multiple implicitly absolute sheets, e.g. Sheet1:Sheet2!$C:$C (always with colon).
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The absolute marker for the first column.
            // - Group 6: The name of the first column.
            // - Group 7: The absolute marker for the second column.
            // - Group 8: The name of the second column.
            COL_RANGE_ABSCUBE_REF: new RegExp('^' + ABS_SHEET_RANGE_PATTERN + '!' + COL_RANGE_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A column range in multiple relative/absolute sheets, e.g. Sheet1:$Sheet2!$C:$C (always with colon).
            // - Group 1: The absolute marker of the first sheet.
            // - Group 2: The name of the first sheet (simple sheet name).
            // - Group 3: The name of the first sheet (complex sheet name).
            // - Group 4: The absolute marker of the second sheet.
            // - Group 5: The name of the second sheet (simple sheet name).
            // - Group 6: The name of the second sheet (complex sheet name).
            // - Group 7: The absolute marker for the first column.
            // - Group 8: The name of the first column.
            // - Group 9: The absolute marker for the second column.
            // - Group 10: The name of the second column.
            COL_RANGE_RELCUBE_REF: new RegExp('^' + REL_SHEET_RANGE_PATTERN + '!' + COL_RANGE_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A local row range, e.g. 1:2 or $3:$3 (always with colon).
            // - Group 1: The absolute marker for the first row.
            // - Group 2: The name of the first row.
            // - Group 3: The absolute marker for the second row.
            // - Group 4: The name of the second row.
            ROW_RANGE_REF: new RegExp('^' + ROW_RANGE_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A row range with implicitly absolute sheet, e.g. Sheet1!$3:$3 (always with colon).
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The #REF! error code as sheet name.
            // - Group 4: The absolute marker for the first row.
            // - Group 5: The name of the first row.
            // - Group 6: The absolute marker for the second row.
            // - Group 7: The name of the second row.
            ROW_RANGE_ABS3D_REF: new RegExp('^' + ABS_SHEET_NAME_PATTERN + '!' + ROW_RANGE_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A row range with relative/absolute sheet, e.g. $Sheet1!$3:$3 (always with colon).
            // - Group 1: The absolute marker for the sheet.
            // - Group 2: The simple sheet name.
            // - Group 3: The complex sheet name.
            // - Group 4: The #REF! error code as sheet name.
            // - Group 5: The absolute marker for the first row.
            // - Group 6: The name of the first row.
            // - Group 7: The absolute marker for the second row.
            // - Group 8: The name of the second row.
            ROW_RANGE_REL3D_REF: new RegExp('^' + REL_SHEET_NAME_PATTERN + '!' + ROW_RANGE_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A row range in multiple implicitly absolute sheets, e.g. Sheet1:Sheet2!$3:$3 (always with colon).
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The absolute marker for the first row.
            // - Group 6: The name of the first row.
            // - Group 7: The absolute marker for the second row.
            // - Group 8: The name of the second row.
            ROW_RANGE_ABSCUBE_REF: new RegExp('^' + ABS_SHEET_RANGE_PATTERN + '!' + ROW_RANGE_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A row range in multiple relative/absolute sheets, e.g. Sheet1:$Sheet2!$3:$3 (always with colon).
            // - Group 1: The absolute marker of the first sheet.
            // - Group 2: The name of the first sheet (simple sheet name).
            // - Group 3: The name of the first sheet (complex sheet name).
            // - Group 4: The absolute marker of the second sheet.
            // - Group 5: The name of the second sheet (simple sheet name).
            // - Group 6: The name of the second sheet (complex sheet name).
            // - Group 7: The absolute marker for the first row.
            // - Group 8: The name of the first row.
            // - Group 9: The absolute marker for the second row.
            // - Group 10: The name of the second row.
            ROW_RANGE_RELCUBE_REF: new RegExp('^' + REL_SHEET_RANGE_PATTERN + '!' + ROW_RANGE_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A reference error with implicitly absolute sheet reference, e.g. Sheet1!#REF!.
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The #REF! error code as sheet name.
            // - Group 4: The matched #REF! error code following the sheet name.
            ERROR_ABS3D_REF: new RegExp('^' + ABS_SHEET_NAME_PATTERN + '!' + REF_ERROR_PATTERN, 'i'),

            // A reference error with relative/absolute sheet reference, e.g. $Sheet1!#REF!.
            // - Group 1: The absolute marker for the sheet.
            // - Group 2: The simple sheet name.
            // - Group 3: The complex sheet name.
            // - Group 4: The #REF! error code as sheet name.
            // - Group 5: The matched #REF! error code following the sheet name.
            ERROR_REL3D_REF: new RegExp('^' + REL_SHEET_NAME_PATTERN + '!' + REF_ERROR_PATTERN, 'i'),

            // A reference error in multiple implicitly absolute sheets, e.g. Sheet1:Sheet2!#REF!.
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The matched #REF! error code following the sheet name.
            ERROR_ABSCUBE_REF: new RegExp('^' + ABS_SHEET_RANGE_PATTERN + '!' + REF_ERROR_PATTERN, 'i'),

            // A reference error in multiple relative/absolute sheets, e.g. Sheet1:$Sheet2!#REF!.
            // - Group 1: The absolute marker of the first sheet.
            // - Group 2: The name of the first sheet (simple sheet name).
            // - Group 3: The name of the first sheet (complex sheet name).
            // - Group 4: The absolute marker of the second sheet.
            // - Group 5: The name of the second sheet (simple sheet name).
            // - Group 6: The name of the second sheet (complex sheet name).
            // - Group 7: The matched #REF! error code following the sheet name.
            ERROR_RELCUBE_REF: new RegExp('^' + REL_SHEET_RANGE_PATTERN + '!' + REF_ERROR_PATTERN, 'i'),

            // A local function name (names followed by an opening parenthesis), e.g. SUM().
            // - Group 1: The name of the function.
            FUNCTION_REF: new RegExp('^' + FUNC_PATTERN + '(?=\\()', 'i'),

            // A function name (names followed by an opening parenthesis) with sheet, e.g. Module1!macro().
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The #REF! error code as sheet name.
            // - Group 4: The name of the function.
            FUNCTION_ABS3D_REF: new RegExp('^' + ABS_SHEET_NAME_PATTERN + '!' + FUNC_PATTERN + '(?=\\()', 'i'),

            // Defined names without sheet reference, e.g. my_name.
            // - Group 1: The label of the defined name.
            NAME_REF: new RegExp('^' + NAME_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // Defined names with sheet reference, e.g. Sheet1!my_name.
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The #REF! error code as sheet name.
            // - Group 4: The label of the defined name.
            NAME_ABS3D_REF: new RegExp('^' + ABS_SHEET_NAME_PATTERN + '!' + NAME_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A local cell reference in OpenFormula syntax, e.g. [.A1], [.$A$1], or [.A$#REF!].
            // - Group 1: The absolute marker for the column.
            // - Group 2: The name of the column, or the #REF! error code.
            // - Group 3: The absolute marker for the row.
            // - Group 4: The name of the row, or the #REF! error code.
            OF_CELL_REF: new RegExp('^\\[' + OF_CELL_PATTERN + '\\]', 'i'),

            // A cell reference with sheet in OpenFormula syntax, e.g. [Sheet1.A1], [$Sheet1.$A$1], or [#REF!.A$#REF!].
            // - Group 1: The absolute marker for the sheet.
            // - Group 2: The name of the sheet (simple sheet name).
            // - Group 3: The name of the sheet (complex sheet name).
            // - Group 4: The #REF! error code as sheet name.
            // - Group 5: The absolute marker for the column.
            // - Group 6: The name of the column, or the #REF! error code.
            // - Group 7: The absolute marker for the row.
            // - Group 8: The name of the row, or the #REF! error code.
            OF_CELL_3D_REF: new RegExp('^\\[' + OF_CELL_3D_PATTERN + '\\]', 'i'),

            // A local cell range reference in OpenFormula syntax, e.g. [.A1:.$A$1], or [.A#REF!:.B2].
            // - Group 1: The absolute marker for the first column.
            // - Group 2: The name of the first column, or the #REF! error code.
            // - Group 3: The absolute marker for the first row.
            // - Group 4: The name of the first row, or the #REF! error code.
            // - Group 5: The absolute marker for the second column.
            // - Group 6: The name of the second column, or the #REF! error code.
            // - Group 7: The absolute marker for the second row.
            // - Group 8: The name of the second row, or the #REF! error code.
            OF_CELL_RANGE_REF: new RegExp('^\\[' + OF_CELL_PATTERN + ':' + OF_CELL_PATTERN + '\\]', 'i'),

            // A cell range reference with sheet in OpenFormula syntax, e.g. [Sheet1.A1:.$A$1], or [$Sheet2.A#REF!:.B2].
            // - Group 1: The absolute marker for the sheet.
            // - Group 2: The name of the sheet (simple sheet name).
            // - Group 3: The name of the sheet (complex sheet name).
            // - Group 4: The #REF! error code as sheet name.
            // - Group 5: The absolute marker for the first column.
            // - Group 6: The name of the first column, or the #REF! error code.
            // - Group 7: The absolute marker for the first row.
            // - Group 8: The name of the first row, or the #REF! error code.
            // - Group 9: The absolute marker for the second column.
            // - Group 10: The name of the second column, or the #REF! error code.
            // - Group 11: The absolute marker for the second row.
            // - Group 12: The name of the second row, or the #REF! error code.
            OF_CELL_RANGE_3D_REF: new RegExp('^\\[' + OF_CELL_3D_PATTERN + ':' + OF_CELL_PATTERN + '\\]', 'i'),

            // A cell range reference in multiple sheets in OpenFormula syntax, e.g. [Sheet1.A1:$Sheet2.$A$1], or [$Sheet2.A#REF!:.#REF!B2].
            // - Group 1: The absolute marker for the first sheet.
            // - Group 2: The name of the first sheet (simple sheet name).
            // - Group 3: The name of the first sheet (complex sheet name).
            // - Group 4: The #REF! error code as first sheet name.
            // - Group 5: The absolute marker for the first column.
            // - Group 6: The name of the first column, or the #REF! error code.
            // - Group 7: The absolute marker for the first row.
            // - Group 8: The name of the first row, or the #REF! error code.
            // - Group 9: The absolute marker for the first sheet.
            // - Group 10: The name of the second sheet (simple sheet name).
            // - Group 11: The name of the second sheet (complex sheet name).
            // - Group 12: The #REF! error code as second sheet name.
            // - Group 13: The absolute marker for the second column.
            // - Group 14: The name of the second column, or the #REF! error code.
            // - Group 15: The absolute marker for the second row.
            // - Group 16: The name of the second row, or the #REF! error code.
            OF_CELL_RANGE_CUBE_REF: new RegExp('^\\[' + OF_CELL_3D_PATTERN + ':' + OF_CELL_3D_PATTERN + '\\]', 'i'),

            // A local function name (names followed by an opening parenthesis with optional whitespace), e.g. SUM (),
            // as allowed in the native OpenFormula syntax.
            // - Group 1: The name of the function.
            OF_FUNCTION_REF: new RegExp('^' + FUNC_PATTERN + '(?:' + WHITESPACE_PATTERN + '*)(?=\\()', 'i')
        };

        FormulaUtils.info('GrammarConfig(): grammar=' + grammar + ' config=', this);

        // private methods ----------------------------------------------------

        /**
         * Returns whether the passed text is a boolean literal according to
         * this formula grammar.
         *
         * @param {String} text
         *  The text to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed text is a boolean literal.
         */
        function isBooleanLiteral(text) {
            return _.isString(booleanCollection.getKey(text, localized));
        }

        /**
         * Returns whether the passed text is the representation of a relative
         * cell reference in A1 notation, or a cell reference in R1C1 notation
         * (either English, e.g. 'R1C1', or according to this formula grammar,
         * e.g. 'Z1S1' in the German UI grammar).
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to check the maximum column/row index.
         *
         * @param {String} text
         *  The text to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed text is the representation of a  cell reference
         *  in A1 or R1C1 notation.
         */
        function isReferenceSymbol(docModel, text) {

            // cell address in A1 notation
            var address = Address.parse(text);
            if (address && docModel.isValidAddress(address)) { return true; }

            // cell address in native R1C1 notation (regardless of the grammar)
            if (isSimpleR1C1Reference(docModel, resource.getR1C1PrefixChars(false), text)) { return true; }

            // localized R1C1 references (e.g. the German S1Z1)
            if (localized && isSimpleR1C1Reference(docModel, PREFIX_CHARS, text)) { return true; }

            // the passed text is not a cell reference
            return false;
        }

        /**
         * Returns the original name of the sheet referred by the passed sheet
         * reference.
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to resolve sheet names.
         *
         * @param {SheetRef|Null} sheetRef
         *  The sheet reference. If set to null, an empty string will be
         *  returned to indicate a sheet-local reference.
         *
         * @returns {String|Null}
         *  An empty string, if null has been passed to this method; the
         *  original name of the sheet referred by the passed sheet reference;
         *  or null, if the sheet reference cannot be resolved to a sheet name.
         */
        function getSheetName(docModel, sheetRef) {
            return !sheetRef ? '' : _.isNumber(sheetRef.sheet) ? docModel.getSheetName(sheetRef.sheet) : sheetRef.sheet;
        }

        /**
         * Encodes the passed sheet name, if it is a complex name, and adds an
         * absolute marker if needed.
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to recognize simple sheet names.
         *
         * @param {String|Null} sheetName
         *  The sheet name, or null to indicate a sheet reference error.
         *
         * @param {Boolean} abs
         *  Whether the sheet reference is absolute.
         *
         * @returns {String}
         *  The encoded sheet name. If null has been passed as sheet name, the
         *  #REF! error code will be returned. Complex sheet names will be
         *  enclosed in apostrophes. If the absolute flag has been set, and the
         *  sheet name is not empty, a dollar sign will be added before the
         *  (encoded) sheet name.
         */
        function encodeAbsSheetName(docModel, sheetName, abs) {

            // use the #REF! error code for invalid sheets, enclose complex sheet names in apostrophes
            if (!_.isString(sheetName)) {
                sheetName = REF_ERROR;
            } else if (sheetName && !self.isSimpleSheetName(docModel, sheetName)) {
                sheetName = SheetRef.encodeComplexSheetName(sheetName);
            }

            // add the leading absolute marker in ODF mode
            return ((abs && sheetName) ? '$' : '') + sheetName;
        }

        /**
         * Returns the string representation of the passed sheet reference for
         * the OpenFormula token syntax, i.e. with absolute sheet marker.
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to resolve sheet names.
         *
         * @param {SheetRef|Null} sheetRef
         *  The sheet reference. If set to null, an empty string will be
         *  returned to indicate a sheet-local reference.
         *
         * @returns {String}
         *  The string representation for the passed sheet reference; or an
         *  empty string, if no sheet reference has been passed.
         */
        function generateOFSheetName(docModel, sheetRef) {

            // empty string for missing sheet reference
            if (!sheetRef) { return ''; }

            // convert reference to sheet name, use the #REF! error code for invalid sheets,
            // enclose complex sheet names in apostrophes
            var sheetName = getSheetName(docModel, sheetRef);
            return encodeAbsSheetName(docModel, sheetName, sheetRef.abs);
        }

        /**
         * Returns the string representation of the passed sheet references for
         * regular token syntax, i.e. as range of sheet names, and with an
         * exclamation mark as sheet name separator.
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to resolve sheet names.
         *
         * @param {SheetRef|Null} sheet1Ref
         *  The first sheet reference. If set to null, an empty string will be
         *  returned.
         *
         * @param {SheetRef|Null} sheet2Ref
         *  The second sheet reference. Ignored, if the value null has been
         *  passed to parameter 'sheet1Ref'. If set to null, a single sheet
         *  name as described by the parameter 'sheet1Ref' will be used.
         *
         * @returns {String}
         *  The string representation for the passed sheet references, with a
         *  trailing exclamation mark; or an empty string, if no sheet
         *  references have been passed.
         */
        function generateSheetRangePrefix(docModel, sheet1Ref, sheet2Ref) {

            // empty string for missing sheet reference
            if (!sheet1Ref) { return ''; }

            // convert first sheet name
            var sheet1Name = getSheetName(docModel, sheet1Ref);
            if (!_.isString(sheet1Name)) { return REF_ERROR + '!'; }

            // convert second sheet name
            var sheet2Name = (sheet2Ref && !sheet1Ref.equals(sheet2Ref)) ? getSheetName(docModel, sheet2Ref) : '';
            if (!_.isString(sheet2Name)) { return REF_ERROR + '!'; }

            // the resulting sheet names
            var result = odf ? encodeAbsSheetName(docModel, sheet1Name, sheet1Ref.abs) : sheet1Name;
            if (sheet2Ref) {
                result += ':' + (odf ? encodeAbsSheetName(docModel, sheet2Name, sheet2Ref.abs) : sheet2Name);
            }

            // OOXML: enclose sheet range in apostrophes, if one of the sheet names is complex
            if (!odf && (!self.isSimpleSheetName(docModel, sheet1Name) || (sheet2Name && !self.isSimpleSheetName(docModel, sheet2Name)))) {
                result = SheetRef.encodeComplexSheetName(result);
            }

            return result + '!';
        }

        // public methods -----------------------------------------------------

        /**
         * Returns whether the passed text is a reserved symbol according to
         * this formula grammar. Boolean literals, and strings that look like
         * cell addresses are considered to be reserved symbols. Cell addresses
         * include the representation of a relative cell reference in A1
         * notation, or a cell reference in R1C1 notation (either English, e.g.
         * 'R1C1', or according to this formula grammar, e.g. 'Z1S1' in the
         * German UI grammar).
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to check the maximum column/row index of
         *  strings that look like cell addresses.
         *
         * @param {String} text
         *  The text to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed text is a reserved symbol according to this
         *  formula grammar.
         */
        this.isReservedSymbol = function (docModel, text) {
            return isBooleanLiteral(text) || isReferenceSymbol(docModel, text);
        };

        /**
         * Returns whether the passed sheet name is a simple sheet name, i.e.
         * it does not need to be enclosed in a pair of apostrophes in formula
         * expressions. Simple sheet names do not contain any reserved
         * characters (e.g. whitespace, or operators used in formulas), and do
         * not look like other reserved names (e.g. boolean literals, or cell
         * references).
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to check the maximum column/row index of
         *  sheet names that look like cell addresses.
         *
         * @param {String} text
         *  The text to be checked.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.external=false]
         *      If set to true, the passed text may contain a leading simple
         *      file name enclosed in brackets, e.g. '[doc.xlsx]Sheet1', or
         *      even '[path\to\doc.xlsx]Sheet1'.
         *  @param {Boolean} [options.range=false]
         *      If set to true, the passed text may contain a sheet range (two
         *      simple sheet names separated by a colon), e.g. 'Sheet1:Sheet2'.
         *
         * @returns {Boolean}
         *  Whether the passed sheet name is a simple sheet name.
         */
        this.isSimpleSheetName = function (docModel, text, options) {

            var // whether to accept leading document name in brackets
                external = Utils.getBooleanOption(options, 'external', false),
                // whether to accept a single colon separating two simple sheet names
                range = Utils.getBooleanOption(options, 'range', false),
                // the result of the regular expression test
                matches = SIMPLE_SHEET_NAME_RE.exec(text);

            // passed text does not match at all
            if (!matches) { return false; }

            // without 'external' flag, external document name is not allowed
            if (matches[1] && !external) { return false; }

            // without 'range' flag, second sheet name is not allowed
            if (matches[3] && !range) { return false; }

            // reserved symbols are not simple sheet names
            if (this.isReservedSymbol(docModel, matches[2])) { return false; }
            if (matches[3] && this.isReservedSymbol(docModel, matches[3])) { return false; }

            return true;
        };

        /**
         * Checks if the passed string can be used as label for a defined name.
         *
         * @param {String} text
         *  The string to be checked.
         *
         * @returns {String}
         *  The empty string, if the passed text is a valid label for a defined
         *  name; otherwise one of the following error codes:
         *  - 'name:empty': The passed label is empty.
         *  - 'name:invalid': The passed label contains invalid characters.
         *  - 'name:address': The passed label would be a valid, but conflicts
         *      with the representation of a relative cell reference in A1
         *      notation, or a cell reference in R1C1 notation (either English,
         *      e.g. 'R1C1', or according to current UI language, e.g. 'Z1S1'
         *      in German).
         */
        this.validateNameLabel = function (docModel, text) {

            // the passed string must not be empty
            if (text.length === 0) { return 'name:empty'; }

            // check that the passed string does not contain invalid characters
            if (!VALID_NAME_RE.test(text)) { return 'name:invalid'; }

            // check that the text does not look like a boolean literal (TODO: own error code?)
            if (!odf && isBooleanLiteral(text)) { return 'name:invalid'; }

            // bug 38786: check that the text does not look like a cell address
            if (isReferenceSymbol(docModel, text)) { return 'name:address'; }

            // the passed text is a valid label for a defined name
            return '';
        };

        /**
         * Converts the passed scalar value to its text representation in
         * formulas according to this formula grammar.
         *
         * @param {Number|String|Boolean|ErrorCode|Null} value
         *  The value to be converted.
         *
         * @returns {String}
         *  The string representation of the passed value.
         */
        this.formatScalar = function (value) {

            // numbers
            if (_.isNumber(value)) {
                return !isFinite(value) ? this.getErrorName(ErrorCode.NUM) :
                    (Math.abs(value) < Utils.MIN_NUMBER) ? '0' :
                    formatter.formatStandardNumber(value, SheetUtils.MAX_LENGTH_STANDARD_EDIT);
            }

            // strings
            if (_.isString(value)) {
                // duplicate inner quotes, enclose in quotes
                return '"' + value.replace(/"/g, '""') + '"';
            }

            // boolean values
            if (_.isBoolean(value)) {
                return this.getBooleanName(value);
            }

            // error codes
            if (value instanceof ErrorCode) {
                return this.getErrorName(value);
            }

            // empty function parameters
            if (!_.isNull(value)) {
                Utils.warn('GrammarConfig.formatScalar(): unsupported value type');
            }
            return '';
        };

        /**
         * Converts the passed reference structures to the text representation
         * of a complete cell range reference in formulas according to this
         * formula grammar.
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to resolve sheet names.
         *
         * @param {SheetRef|Null} sheet1Ref
         *  The first sheet reference (e.g. the particle 'Sheet1' in the cell
         *  range reference 'Sheet1:Sheet3!A1:C3'). If set to null, no sheet
         *  names will be inserted into the returned string.
         *
         * @param {SheetRef|Null} sheet2Ref
         *  The second sheet reference (e.g. the particle 'Sheet3' in the cell
         *  range reference 'Sheet1:Sheet3!A1:C3'). Ignored, if the value null
         *  has been passed to parameter 'sheet1Ref'. If set to null, a single
         *  sheet name as described by the parameter 'sheet1Ref' will be used
         *  (e.g. 'Sheet1!A1:C3').
         *
         * @param {CellRef|Null} cell1Ref
         *  The first cell reference (e.g. the cell address 'A1' in 'A1:C3').
         *  If set to null, the #REF! error code will be returned instead of a
         *  cell range address.
         *
         * @param {CellRef|Null} cell2Ref
         *  The second cell reference (e.g. the cell address 'C3' in 'A1:C3').
         *  If set to null, a single cell address will be returned.
         *
         * @param {Boolean} [rcStyle=false]
         *  If set to true, the reference will be formatted in R1C1 style. If
         *  set to false or omitted, the reference will be formatted in A1
         *  style. MUST NOT be set to true, if this instance represents a
         *  native formula grammar.
         *
         * @returns {String}
         *  The string representation of the cell range reference.
         */
        this.formatReference = function (docModel, sheet1Ref, sheet2Ref, cell1Ref, cell2Ref, rcStyle) {

            // R1C1 notation not supported in native formula grammars (UI only)
            if (!localized && rcStyle) {
                Utils.error('GrammarConfig.formatReference(): R1C1 notation not supported in native grammar');
                return NA_ERROR;
            }

            var // the cell range address (with adjusted column/row indexes)
                range = cell1Ref ? Range.createFromAddresses(cell1Ref.toAddress(), cell2Ref && cell2Ref.toAddress()) : null,
                // the resulting text representation
                result = '';

            // OpenFormula syntax: references are enclosed in brackets, absolute markers for
            // sheets, sheet names are separated by periods, e.g.: [$Sheet1.A1:$'Sheet 2'.B2]
            if (openFormula) {

                // the leading bracket, and sheet name with separator character
                result = '[' + generateOFSheetName(docModel, sheet1Ref) + '.';

                // the first cell address (ODF does not support column/row ranges);
                // reference error: ODF uses double #REF! errors (for column and row)
                result += range ? cell1Ref.refText() : (REF_ERROR + REF_ERROR);

                // add second cell address for sheet ranges, or for cell ranges (repeat
                // single cell address with sheet range, e.g. [Sheet1.A1:Sheet2.A1])
                if (sheet2Ref || (range && cell2Ref)) {
                    result += ':' + generateOFSheetName(docModel, sheet2Ref) + '.';
                    result += range ? (cell2Ref || cell1Ref).refText() : (REF_ERROR + REF_ERROR);
                }

                // always add the closing bracket
                return result + ']';
            }

            // the leading sheet name(s) with sheet/address separator character
            result = generateSheetRangePrefix(docModel, sheet1Ref, sheet2Ref);

            // no valid range: generate a #REF! error with (optional) leading sheet name
            if (!range) {
                return result + REF_ERROR;
            }

            // generate row interval (preferred over column interval for entire sheet range)
            if (cell1Ref.absCol && cell2Ref && cell2Ref.absCol && docModel.isRowRange(range)) {
                result += rcStyle ? cell1Ref.rowTextRC(PREFIX_CHARS) : cell1Ref.rowText();
                if (!rcStyle || !range.singleRow() || (cell1Ref.absRow !== cell2Ref.absRow)) {
                    result += ':' + (rcStyle ? cell2Ref.rowTextRC(PREFIX_CHARS) : cell2Ref.rowText());
                }
                return result;
            }

            // generate column interval
            if (cell1Ref.absRow && cell2Ref && cell2Ref.absRow && docModel.isColRange(range)) {
                result += rcStyle ? cell1Ref.colTextRC(PREFIX_CHARS) : cell1Ref.colText();
                if (!rcStyle || !range.singleCol() || (cell1Ref.absCol !== cell2Ref.absCol)) {
                    result += ':' + (rcStyle ? cell2Ref.colTextRC(PREFIX_CHARS) : cell2Ref.colText());
                }
                return result;
            }

            // generate range or cell address
            result += rcStyle ? cell1Ref.refTextRC(PREFIX_CHARS) : cell1Ref.refText();
            if (cell2Ref) {
                result += ':' + (rcStyle ? cell2Ref.refTextRC(PREFIX_CHARS) : cell2Ref.refText());
            }
            return result;
        };

        /**
         * Converts the passed sheet reference and label of a defined name or
         * macro function to the text representation of a complete defined name
         * reference or macro call in formulas according to this formula
         * grammar.
         *
         * @param {SpreadsheetModel} docModel
         *  The document model needed to resolve sheet names.
         *
         * @param {SheetRef|Null} sheetRef
         *  The sheet reference (e.g. the particle 'Sheet1' in 'Sheet1!name').
         *  If set to null, no sheet name will be inserted into the returned
         *  string.
         *
         * @param {String} label
         *  The label of the defined name, or the name of a macro function.
         *
         * @returns {String}
         *  The string representation of the defined name or macro call.
         */
        this.formatName = function (docModel, sheetRef, label) {
            // no sheet-local names in ODF files
            return (odf && sheetRef) ? NA_ERROR : (generateSheetRangePrefix(docModel, sheetRef) + label);
        };

        /**
         * Returns the name of the passed boolean value.
         *
         * @param {Boolean} value
         *  The boolean value to be converted to its string representation.
         *
         * @returns {String}
         *  The name of the passed boolean value.
         */
        this.getBooleanName = function (value) {
            return booleanCollection.getName(value ? 't' : 'f', localized);
        };

        /**
         * Converts the passed name of a boolean value to the boolean value.
         *
         * @param {String} name
         *  The name of a boolean value (case-insensitive).
         *
         * @returns {Boolean|Null}
         *  The boolean value for the passed name of a boolean value; or null,
         *  if the passed string is not the name of a boolean value.
         */
        this.getBooleanValue = function (name) {
            var key = booleanCollection.getKey(name, localized);
            return key ? (key === 't') : null;
        };

        /**
         * Returns the name of the passed error code.
         *
         * @param {ErrorCode} error
         *  The error code to be converted to its string representation.
         *
         * @returns {String|Null}
         *  The name of the passed error code; or null, if the passed error
         *  code cannot be resolved to a name.
         */
        this.getErrorName = function (error) {
            return errorCollection.getName(error.key, localized);
        };

        /**
         * Converts the passed error name to the error code instance.
         *
         * @param {String} name
         *  The name of an error code (case-insensitive).
         *
         * @returns {ErrorCode|Null}
         *  The error code instance for the passed error name; or null, if the
         *  passed string is not the name of a supported error code.
         */
        this.getErrorCode = function (name) {
            var key = errorCollection.getKey(name, localized);
            return key ? ErrorCode.create(key) : null;
        };

        /**
         * Returns the name of an operator for the passed unique resource key.
         *
         * @param {String} key
         *  The unique resource key of an operator.
         *
         * @returns {String|Null}
         *  The operator name for the passed resource key; or null, if the
         *  passed resource key cannot be resolved to the name of an operator.
         */
        this.getOperatorName = function (key) {
            return operatorCollection.getName(key, localized);
        };

        /**
         * Converts the passed operator name to the unique resource key of the
         * operator.
         *
         * @param {String} name
         *  The name of an operator.
         *
         * @returns {String|Null}
         *  The resource key for the passed operator name; or null, if the
         *  passed string is not the name of a supported operator.
         */
        this.getOperatorKey = function (name) {
            return operatorCollection.getKey(name, localized);
        };

        /**
         * Returns the name of a function for the passed unique resource key.
         *
         * @param {String} key
         *  The unique resource key of a function.
         *
         * @returns {String|Null}
         *  The function name for the passed resource key; or null, if the
         *  passed resource key cannot be resolved to the name of a function.
         */
        this.getFunctionName = function (key) {
            return functionCollection.getName(key, localized);
        };

        /**
         * Converts the passed function name to the unique resource key of the
         * function.
         *
         * @param {String} name
         *  The name of a function (case-insensitive).
         *
         * @returns {String|Null}
         *  The resource key for the passed function name; or null, if the
         *  passed string is not the name of a supported function.
         */
        this.getFunctionKey = function (name) {
            return functionCollection.getKey(name, localized);
        };

    } // class GrammarConfig

    // static methods ---------------------------------------------------------

    /**
     * Returns an existing grammar configuration from the internal cache, if
     * available; otherwise creates a new instance.
     *
     * @param {FormulaResource} resource
     *  A formula resource instance for the current UI language for a specific
     *  file format.
     *
     * @param {String} grammar
     *  The identifier of a formula grammar. Supported values are:
     *  - 'op': Fixed token representations as used in operations, and the file
     *      format.
     *  - 'ui': The localized token representations according to the current
     *      UI language of the application.
     *
     * @returns {GrammarConfig}
     *  A grammar configuration instance for the passed parameters.
     */
    GrammarConfig.create = (function () {

        // create a hash key for the memoize() cache
        function getConfigKey(resource, grammar) {
            return resource.getUid() + ':' + grammar;
        }

        // create a new instance of GrammarConfig
        function createConfig(resource, grammar) {
            return new GrammarConfig(resource, grammar);
        }

        // create a function that caches all created grammar configurations
        return _.memoize(createConfig, getConfigKey);
    }());

    // exports ================================================================

    return GrammarConfig;

});
