/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/cellattributesmodel', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/model/attributedmodel'
], function (Utils, Border, AttributeUtils, AttributedModel) {

    'use strict';

    var // various name for all border attributes, mapped by border attribute name
        BORDER_NAME_INFOS = {
            borderLeft:   { innerName: 'borderInsideVert', optionName: 'innerLeft' },
            borderRight:  { innerName: 'borderInsideVert', optionName: 'innerRight' },
            borderTop:    { innerName: 'borderInsideHor',  optionName: 'innerTop' },
            borderBottom: { innerName: 'borderInsideHor',  optionName: 'innerBottom' },
            borderDown:   null,
            borderUp:     null
        };

    // class CellAttributesModel ==============================================

    /**
     * Base class for objects containing cell attributes. Provides a
     * specialized implementation of the method 'setAttributes()' that is able
     * to update single border properties of visible border attributes.
     *
     * @constructor
     *
     * @extends AttributedModel
     *
     * @param {SheetModel} sheetModel
     *  The sheet model containing this cell model object.
     *
     * @param {Object} [initAttributes]
     *  An attribute set with initial formatting attributes for the cell.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options also supported by the base
     *  class AttributedModel. The following options will be set to fixed
     *  values and cannot be changed:
     *  - initOptions.silent will be set to true (no change events).
     *  - initOptions.styleFamily will be set to 'cell'.
     */
    function CellAttributesModel(sheetModel, initAttributes, initOptions) {

        var // self reference
            self = this,

            // the document model
            docModel = sheetModel.getDocModel(),

            // the collection of cell styles
            cellStyles = docModel.getCellStyles(),

            // the parsed format code
            parsedFormat = null;

        // base constructor ---------------------------------------------------

        // do not trigger any events (collection entries are potential mass objects)
        AttributedModel.call(this, docModel, initAttributes, Utils.extendOptions(initOptions, {
            trigger: 'never',
            styleFamily: 'cell'
        }));

        // private methods ----------------------------------------------------

        /**
         * Refreshes the parsed number format, after the formatting attributes
         * of this instance have been changed.
         */
        function updateParsedFormat() {
            parsedFormat = docModel.getNumberFormatter().parseNumberFormat(self.getMergedAttributes().cell.numberFormat);
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the parsed number format instance of this cell.
         *
         * @returns {ParsedFormat}
         *  The parsed number format instance of this cell.
         */
        this.getParsedFormat = function () {
            return parsedFormat;
        };

        /**
         * Changes and/or removes specific explicit formatting attributes, or
         * the style sheet reference of this model object. Overwrites the base
         * class method AttributedModel.setAttributes() in order to add special
         * treatment for the 'apply' flags in cell style sheets, and for border
         * attributes (see options below).
         *
         * @param {Object} attributeSet
         *  An (incomplete) attribute set.
         *
         * @param {Object} [options]
         *  Optional parameters. Additionally to the options supported by the
         *  method AttributedModel.setAttributes(), the following options are
         *  supported:
         *  @param {Boolean} [options.innerLeft=false]
         *      If set to true, the border attribute 'borderInsideVert' will be
         *      used for the left border of the cell, instead of the attribute
         *      'borderLeft'.
         *  @param {Boolean} [options.innerRight=false]
         *      If set to true, the border attribute 'borderInsideVert' will be
         *      used for the right border of the cell, instead of the attribute
         *      'borderRight'.
         *  @param {Boolean} [options.innerTop=false]
         *      If set to true, the border attribute 'borderInsideHor' will be
         *      used for the top border of the cell, instead of the attribute
         *      'borderTop'.
         *  @param {Boolean} [options.innerBottom=false]
         *      If set to true, the border attribute 'borderInsideHor' will be
         *      used for the bottom border of the cell, instead of the
         *      attribute 'borderBottom'.
         *  @param {Boolean} [options.visibleBorders=false]
         *      If set to true, border attributes may be incomplete. Specified
         *      border properties (line style, line width, line color) will be
         *      applied for visible borders only.
         *
         * @returns {Boolean}
         *  Whether any attributes have actually been changed.
         */
        this.setAttributes = _.wrap(this.setAttributes, function (setAttributes, attributeSet, options) {

            var // the new style identifier
                styleId = attributeSet.styleId,
                // the current merged attribute set
                mergedAttributeSet = this.getMergedAttributes(),
                // the cell attributes from the merged attribute set
                mergedCellAttrs = mergedAttributeSet.cell;

            // handle explicit formatting attributes to be cleared according to the apply flags of the cell style sheet
            if (_.isString(styleId) && (styleId !== mergedAttributeSet.styleId) && !cellStyles.isAuto(styleId)) {

                var // the merged attribute set of the cell style sheet
                    styleAttributeSet = cellStyles.getStyleAttributeSet(styleId);

                // work on a deep copy (do not modify the object passed by the caller)
                attributeSet = _.copy(attributeSet, true);

                // check the attribute definitions for an 'apply' property
                _.each(styleAttributeSet, function (styleAttrs, family) {
                    if (family !== 'apply') {
                        _.each(docModel.getAttributeDefinitions(family), function (definition, name) {
                            if (definition.apply && styleAttributeSet.apply[definition.apply]) {
                                AttributeUtils.insertAttribute(attributeSet, family, name, null, true);
                            }
                        });
                    }
                });
            }

            // process border attributes (copy inner to outer, update colors or line styles)
            if (_.isObject(attributeSet.cell)) {

                // update the border attributes in a deep copy of the passed attribute set
                attributeSet = _.clone(attributeSet);
                attributeSet.cell = _.copy(attributeSet.cell, true);
                var cellAttrs = attributeSet.cell;

                // move inner border attributes to outer border attributes if specified in the options
                _.each(BORDER_NAME_INFOS, function (borderInfo, borderName) {

                    // nothing to do for diagonal borders
                    if (!borderInfo) { return; }

                    if (Utils.getBooleanOption(options, borderInfo.optionName, false)) {
                        if (borderInfo.innerName in cellAttrs) {
                            cellAttrs[borderName] = cellAttrs[borderInfo.innerName];
                        } else {
                            delete cellAttrs[borderName];
                        }
                    }
                });

                // cells cannot contain inner border attributes
                delete cellAttrs.borderInsideVert;
                delete cellAttrs.borderInsideHor;

                // update styling of existing borders in 'visibleBorders' mode
                if (Utils.getBooleanOption(options, 'visibleBorders', false)) {
                    _.each(BORDER_NAME_INFOS, function (borderInfo, borderName) {
                        if (_.isObject(cellAttrs[borderName]) && Border.isVisibleBorder(mergedCellAttrs[borderName])) {
                            cellAttrs[borderName] = _.extend({}, mergedCellAttrs[borderName], cellAttrs[borderName]);
                            if (!Border.isVisibleBorder(cellAttrs[borderName])) {
                                cellAttrs[borderName] = Border.NONE;
                            }
                        } else {
                            delete cellAttrs[borderName];
                        }
                    });
                }
            }

            // apply the attributes using the base class method
            var changed = setAttributes.call(this, attributeSet, options);

            // refresh the parsed number format
            if (changed) { updateParsedFormat(); }

            return changed;
        });

        // initialization -----------------------------------------------------

        // initialize the parsed number format
        updateParsedFormat();

        // destroy class members on destruction
        this.registerDestructor(function () {
            self = docModel = cellStyles = sheetModel = initAttributes = initOptions = null;
        });

    } // class CellAttributesModel

    // exports ================================================================

    // derive this class from class AttributedModel
    return AttributedModel.extend({ constructor: CellAttributesModel });

});
