/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/model/slideoperationmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/config',
    'io.ox/office/presentation/utils/operations'
], function (Utils, Border, DOM, Position, Config, Operations) {

    'use strict';

    // mix-in class SlideOperationMixin ======================================

    /**
     * A mix-in class for the document model class PresentationModel providing
     * the the slide operation handling used in a presentation document.
     *
     * @constructor
     */
    function SlideOperationMixin() {

        var // self reference for local functions
            self = this;

        // private methods ----------------------------------------------------

        /**
         * Handler for inserting a new master slide.
         *
         * @param {String} id
         *  The id of the master slide.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied to the new inserted layout slide, as map of
         *  attribute maps (name/value pairs), keyed by attribute family.
         *
         * @returns {Boolean}
         *  Whether the master slide has been inserted successfully.
         */
        function implInsertMasterSlide(id, attrs) {

            var // the new slide node
                slide = null,
                // the layer node for the master slides
                masterSlideLayerNode = self.getOrCreateMasterSlideLayerNode();

            // adding the master slide into the master slide layer node
            slide = self.appendSlideToLayerNode(masterSlideLayerNode, id);

            // applying all attributes
            if (self.isImportFinished()) { self.getSlideStyles().updateElementFormatting(slide); }

            // registering the id at the slide
            DOM.setTargetContainerId(slide, id);

            // registering the slide in the model
            self.addIntoSlideModel(slide, id, { type: self.getMasterSlideType() });

            // saving the list style attributes in slide styles
            if (attrs && attrs.listStyles) { self.getSlideStyles().saveListStylesAttributes(id, attrs); }

            return true;
        }

        /**
         * Handler for inserting a new layout slide.
         *
         * @param {String} id
         *  The id of the layout slide.
         *
         * @param {String} target
         *  The id of the master slide, that is the base for the specified layout slide.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied to the new inserted layout slide, as map of
         *  attribute maps (name/value pairs), keyed by attribute family.
         *
         * @returns {Boolean}
         *  Whether the layout slide has been inserted successfully.
         */
        function implInsertLayoutSlide(id, target, attrs) {

            var // the layout slide node
                slide = null,
                // the layer node for the layout slides
                layoutSlideLayerNode = self.getOrCreateLayoutSlideLayerNode();

            // adding the master slide into the master slide layer node
            slide = self.appendSlideToLayerNode(layoutSlideLayerNode, id);

            // apply the passed slide attributes
            self.getSlideStyles().setElementAttributes(slide, attrs ? attrs : {});

            // applying all attributes
            if (self.isImportFinished()) { self.getSlideStyles().updateElementFormatting(slide); }

            // registering the id at the slide
            DOM.setTargetContainerId(slide, id);

            // saving the target also at the slide (required for generating undo operation)
            if (target) { slide.data('target', target); }

            // registering the slide in the model
            self.addIntoSlideModel(slide, id, { type: self.getLayoutSlideType(), target: target });

            // saving the list style attributes in slide styles
            if (attrs && attrs.listStyles) { self.getSlideStyles().saveListStylesAttributes(id, attrs); }

            return true;
        }

        /**
         * Handler for inserting a new slide
         *
         * @param {Number[]} start
         *  The logical position of the slide.
         *
         * @param {String} target
         *  The id of the layout slide, that is the base for the specified slide.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied to the new inserted slide, as map of
         *  attribute maps (name/value pairs), keyed by attribute family.
         *
         * @returns {Boolean}
         *  Whether the slide has been inserted successfully.
         */
        function implInsertSlide(start, target, attrs/*, external*/) {

            var // the new slide (jQueryfied)
                slide = DOM.createSlideNode(),
                // the new position of the slide
                position = _.clone(start),
                // the position number of the slide (there are only top level slides)
                slideNumber = _.isArray(position) ? position[0] : position,  // TODO: Handle operations correctly
                // the page content node (parent of the slides)
                pageContentNode = DOM.getPageContentNode(self.getNode()),
                // the current slide at the position
                currentSlide = pageContentNode[0].childNodes[slideNumber],
                // a new generated id for the slide
                id = self.getNextSlideID();

            if (currentSlide) {
                slide.insertBefore(currentSlide);
            } else {
                slide.insertAfter(pageContentNode.children(DOM.PARAGRAPH_NODE_SELECTOR).last());
            }

            // removing a following implicit slide
            // -> exchanging an implicit slide with a non-implicit slide
            if (DOM.isImplicitParagraphNode(slide.next())) {
                slide.next().remove();
            }

            // apply the passed slide attributes
            self.getSlideStyles().setElementAttributes(slide, attrs ? attrs : {});

            // insert required helper nodes and applying all attributes
            self.validateParagraphNode(slide);
            if (self.isImportFinished()) { self.getSlideStyles().updateElementFormatting(slide); }

            // registering the id at the slide
            DOM.setTargetContainerId(slide, id);

            // saving the target also at the slide (required for generating undo operation)
            if (target) { slide.data('target', target); }

            // registering the slide in the model (with locally generated id)
            self.addIntoSlideModel(slide, id, { type: self.getStandardSlideType(), target: target, index: slideNumber });

            return true;
        }

        /**
         * Trying to get the following layout slide ID relative to a specified layout id.
         *
         * @param {String} id
         *  The id of a layout slide.
         *
         * @returns {String}
         *  The id of a layout slide that follows the specified layout slide.
         */
        function getFollowingLayoutId(layoutId) {

            var // the master layout slide order array
                order = self.getMasterSlideOrder(),
                // the index of an id in the order array
                index = _.indexOf(order, layoutId);

            return index > -1 && index < order.length && self.isLayoutSlideId(layoutId) && self.isLayoutSlideId(order[index + 1]) ? order[index + 1] : null;
        }

        /**
         * Trying to find a second layout id following the first master id. If this
         * cannot be determined, the first found layout id will be returned.
         *
         * @returns {String}
         *  The id of a layout slide that can be used for a new slide.
         */
        function getSecondLayoutIdBehindMaster() {

            var // the layoutId
                layoutId = null,
                // the master layout slide order array
                order = self.getMasterSlideOrder(),
                // the index of an id in the order array
                index = 0,
                // the master id
                masterId = null;

            masterId = _.find(order, self.isMasterSlideId);

            index = _.indexOf(order, masterId);

            if (index > -1 && index < order.length - 2) { layoutId = order[index + 2]; }

            // otherwise take the first possible layout id
            if (!layoutId) { layoutId = _.find(order, self.isLayoutSlideId); }

            return layoutId;
        }

        /**
         * Checking, whether the specified id is a layout id and it is the first
         * id behind the master id.
         *
         * @param {String} layoutId
         *  The id of a layout slide.
         *
         * @returns {Boolean}
         *  Whether the specified id is the first layout id behind the master id.
         */
        function isFirstLayoutIdBehindMaster(layoutId) {

            var // the master layout slide order array
                order = self.getMasterSlideOrder(),
                // the index of the specified layout id inside the order array
                index = _.indexOf(order, layoutId);

            return index > 0 && self.isLayoutSlideId(layoutId) && self.isMasterSlideId(order[index - 1]);
        }

        /**
         * Trying to get the best possible layout id for a new slide. The specified
         * id is the id of the currently active id or of a user specified id.
         *
         * @param {String} id
         *  The id of a document (standard) slide.
         *
         * @returns {String}
         *  The id of a layout slide that can be used for a new slide.
         */
        function getValidLayoutIdForNewSlide(id) {

            var // the layout id of the specified slide
                layoutId = null,
                // the layout id of the following layout slide
                nextLayoutId = null,
                // whether the second layout slide shall be used (the default)
                useSecondLayoutSlide = true;

            if (self.isStandardSlideId(id)) {

                layoutId = self.getLayoutSlideId(id);

                if (layoutId) {

                    if (isFirstLayoutIdBehindMaster(layoutId)) {
                        // special handling required -> PowerPoint uses the next layout id
                        nextLayoutId = getFollowingLayoutId(layoutId);
                        layoutId = nextLayoutId ? nextLayoutId : layoutId;
                    }

                    useSecondLayoutSlide = false; // the layout id can also be used for the new slide
                }

            }

            // try to find any valid layout id
            if (useSecondLayoutSlide) { layoutId = getSecondLayoutIdBehindMaster(); }

            return layoutId;
        }

        // public methods -----------------------------------------------------

        /**
         * Inserting a new slide into the document.
         *
         * The layout of the new slide is the same as the layout of the active slide
         * (or a slide specified by its id). But if this layout is the first layout slide
         * behind the master slide, PowerPoint interprets this as a 'Title'. In this case
         * the following layout slide is used.
         *
         * @param {String} [layoutId]
         *  An optional layout id. If specified, it is used. If not specified, the layout
         *  of the currently active slide is used (or the following layout, if this is the
         *  first layout slide behind the master slide).
         *
         * @param {String} [id]
         *  An optional slide id. If not specified, the active slide is used.
         */
        this.insertSlide = function (layoutId, id) {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // whether the layoutId is specified
                isFixedLayoutId = layoutId && self.isLayoutSlideId(layoutId),
                // the local id
                localId = id || self.getActiveSlideId();

            // the layout id for the new slide
            layoutId = isFixedLayoutId ? layoutId : getValidLayoutIdForNewSlide(localId);

            var // an object with all place holder attributes that are handled by the layout slide
                allPlaceHoldersAttributes = self.getDrawingStyles().getAllPlaceHolderAttributesForId(layoutId),
                // the logical position of the slide to be inserted
                start = Position.increaseLastIndex(self.getActiveSlidePosition()),
                // the position of the drawing, the paragraph and the first character
                drawingPos = null, paraPos = null, textPos = null,
                // created operation
                newOperation = null,
                // the counter for the top level drawing
                drawingCounter = 0;

            // creating new slide
            newOperation = { start: _.clone(start), target: layoutId };
            generator.generateOperation(Operations.SLIDE_INSERT, newOperation);

            // iterating over all place holder with type 'title' and 'body' (TODO: What happens with the other place holders?)
            _.each(allPlaceHoldersAttributes, function (attributes, phType) {

                if (phType === 'title' || phType === 'body') {

                    // iterating over the index
                    _.each(attributes, function (attrs) {

                        drawingPos = _.clone(start);
                        drawingPos.push(drawingCounter);

                        paraPos = _.clone(drawingPos);
                        paraPos.push(0);

                        textPos = _.clone(paraPos);
                        textPos.push(0);

                        newOperation = { attrs: attrs, start: drawingPos, type: 'shape' };
                        generator.generateOperation(Operations.DRAWING_INSERT, newOperation);

                        // add a paragraph into the shape, so that the cursor can be set into the text frame
                        newOperation = { start: paraPos };
                        generator.generateOperation(Operations.PARA_INSERT, newOperation);

                        // increasing the counter for the position of the next drawing
                        drawingCounter++;

                    });
                }
            });

            // applying operation
            this.applyOperations(generator);

            // activating the new inserted slide
            self.setActiveSlideId(self.getIdOfSlideOfActiveViewAtIndex(start[0]));

            // setting cursor into text frame
            self.getSelection().setTextSelection(textPos);
        };

        /**
         * Deleting the currently active slide.
         */
        this.deleteSlide = function () {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // the active slide
                id = self.getActiveSlideId(),
                // the position of the active slide
                slidePos = this.getActiveSlidePosition(),
                // created operation
                newOperation = { start: _.clone(slidePos) },
                // whether the last slide is active in its view
                isLastSlide = self.isLastSlideActive();

            // Required checks:
            // self.isUsedMasterOrLayoutSlide(id) -> not removing master/layout slides that are referenced
            // self.isOnlySlideInView(id)         -> not removing the last slide in its view
            if ((self.isLayoutOrMasterId(id) && !self.isUsedMasterOrLayoutSlide(id)) || self.isOnlySlideInView(id)) { return; }

            // setting target at operation for layout or master slides
            if (self.isLayoutOrMasterId(id)) { newOperation.target = id; }

            generator.generateOperation(Operations.DELETE, newOperation);

            // applying operation
            this.applyOperations(generator);

            // special handling for the last slide
            if (isLastSlide) { slidePos = Position.decreaseLastIndex(slidePos); }

            // activating a neighboring slide
            self.setActiveSlideId(self.getIdOfSlideOfActiveViewAtIndex(slidePos[0]));

            // selecting the first object
            self.getSelection().setTextSelection(Position.appendNewIndex(slidePos, 0), Position.appendNewIndex(slidePos, 1));
        };

        // operation handler --------------------------------------------------

        /**
         * The handler for the insertMasterSlide operation.
         */
        this.insertMasterSlideHandler = function (operation, external) {
            if (!implInsertMasterSlide(operation.id, operation.attrs, external)) { return false; }
            return true;
        };

        /**
         * The handler for the insertLayoutSlide operation.
         */
        this.insertLayoutSlideHandler = function (operation, external) {
            if (!implInsertLayoutSlide(operation.id, operation.target, operation.attrs, external)) { return false; }
            return true;
        };

        /**
         * The handler for the insertSlide operation.
         */
        this.insertSlideHandler = function (operation, external) {
            var // the undo operation for the insertSlide operation
                undoOperation = null;

            if (!implInsertSlide(operation.start, operation.target, operation.attrs, external)) { return false; }

            if (self.getUndoManager().isUndoEnabled()) {
                undoOperation = { name: Operations.DELETE, start: _.clone(operation.start) };
                self.getUndoManager().addUndo(undoOperation, operation);
            }

            return true;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    } // class SlideOperationMixin

    // constants --------------------------------------------------------------

    // exports ================================================================

    return SlideOperationMixin;

});
