/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/model/objectoperationmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/config',
    'io.ox/office/presentation/utils/operations',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/drawinglayer/view/imageutil'
], function (Utils, Border, DOM, Position, Config, Operations, DrawingFrame, Image) {

    'use strict';

    // mix-in class ObjectOperationMixin ======================================

    /**
     * A mix-in class for the document model class PresentationModel providing
     * the style sheet containers for all attribute families used in a
     * presentation document.
     *
     * @constructor
     */
    function ObjectOperationMixin() {

        var // self reference for local functions
            self = this;

        // private methods ----------------------------------------------------

        /**
         * Calculates the size of the image defined by the given url,
         * limiting the size to the paragraph size of the current selection.
         *
         * @param {String} url
         *  The image url or base64 data url
         *
         * @returns {jQuery.Promise}
         *  A promise where the resolve result contains the size object
         *  including width and height in Hmm {width: 123, height: 456}
         */
        function getImageSize(url) {

            var // the result deferred
                def = $.Deferred(),
                // the image for size rendering
                image = $('<img>'),
                // the image url
                absUrl,
                // the clipboard holding the image
                clipboard = null;

            if (!url) { return def.reject(); }

            absUrl = Image.getFileUrl(self.getApp(), url);

            // append the clipboard div to the body and place the image into it
            clipboard = self.getApp().getView().createClipboardNode();
            clipboard.append(image);

            // if you set the load handler BEFORE you set the .src property on a new image, you will reliably get the load event.
            image.one('load', function () {

                // Workaround for a strange Chrome behavior, even if we use .one() Chrome fires the 'load' event twice.
                // One time for the image node rendered and the other time for a not rendered image node.
                // We check for the rendered image node
                if (Utils.containsNode(clipboard, image)) {

                    // maybe the slide is not so big
                    // -> reducing the image size to slide size, if required
                    def.resolve({ width: Utils.convertLengthToHmm(image.width(), 'px'), height: Utils.convertLengthToHmm(image.height(), 'px') });
                }
            })
            .error(function () {
                Utils.warn('Editor.getImageSize(): image load error');
                def.reject();
            })
            .attr('src', absUrl);

            // always remove the clipboard again
            def.always(function () {
                clipboard.remove();
            });

            return def.promise();
        }

        /**
         * Inserts a drawing component into the document DOM.
         *
         * @param {String} type
         *  The type of the drawing. Supported values are 'shape', 'group',
         *  'image', 'diagram', 'chart', 'ole', 'horizontal_line', 'undefined'
         *
         * @param {Number[]} start
         *  The logical start position for the new drawing.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied at the new drawing component, as map of
         *  attribute maps (name/value pairs), keyed by attribute family.
         *
         * @param {Object} [target]
         *  If exists, defines node, to which start position is related.
         *
         * @returns {Boolean}
         *  Whether the drawing has been inserted successfully.
         */
        function implInsertDrawing(type, start, attrs, target) {

            var // text span that will precede the field
                span = null,
                // deep copy of attributes, because they are modified in webkit browsers
                attributes = _.copy(attrs, true),
                // new drawing node
                drawingNode = null,
                // root node container
                rootNode = self.getRootNode(target),
                // image aspect ratio
                currentElement = Position.getContentNodeElement(rootNode, start.slice(0, -1), { allowDrawingGroup: true }),
                // whether the drawing is inserted into a drawing group
                insertIntoDrawingGroup = false,
                // the function used to insert the new drawing frame
                insertFunction = 'insertAfter',
                // the number of children in the drawing group
                childrenCount = 0;

            // helper function to insert a drawing frame into an existing drawing group
            function addDrawingFrameIntoDrawingGroup() {

                childrenCount = DrawingFrame.getGroupDrawingCount(currentElement);

                if (_.isNumber(childrenCount)) {
                    if (childrenCount === 0) {
                        if (_.last(start) === 0) {
                            span = $(currentElement).children().first(); // using 'span' for the content element in the drawing group
                            insertFunction = 'appendTo';
                        }
                    } else {
                        if (_.last(start) === 0) {
                            // inserting before the first element
                            span = DrawingFrame.getGroupDrawingChildren(currentElement, 0);
                            insertFunction = 'insertBefore';
                        } else if (_.last(start) <= childrenCount) {
                            // inserting after the span element
                            span = DrawingFrame.getGroupDrawingChildren(currentElement, _.last(start) - 1);
                        }
                    }
                    insertIntoDrawingGroup = true;
                }
            }

            try {
                span = self.prepareTextSpanForInsertion(start, {}, target);
            } catch (ex) {
                // do nothing, try to repair missing text spans
            }

            // check, if the drawing is inserted into a drawing group
            if (!span && DrawingFrame.isGroupDrawingFrame(currentElement)) { addDrawingFrameIntoDrawingGroup(); }

            // insert the drawing with default settings between the two text nodes (store original URL for later use)
            drawingNode = DrawingFrame.createDrawingFrame(type)[insertFunction](span);

            // apply the passed drawing attributes
            if (_.isObject(attributes)) { self.getDrawingStyles().setElementAttributes(drawingNode, attributes); }

            // applying all attributes (TODO: Also handling attributes from other layers)
            if (self.isImportFinished() && !insertIntoDrawingGroup) { self.getDrawingStyles().updateElementFormatting(drawingNode); }

            // saving the drawing attributes in drawing styles (ignoring the return value, no event is fired)
            if (target) { self.getDrawingStyles().savePlaceHolderAttributes(target, attrs, drawingNode); }

            // lastOperationEnd = Position.increaseLastIndex(start);

            return true;
        }

        // simplified version of implInsertDrawing, that allows only drawings inside slides

//        function implInsertDrawing(type, start, attrs, target) {
//
//            var // deep copy of attributes, because they are modified in webkit browsers
//                attributes = _.copy(attrs, true),
//                // new drawing node
//                drawingNode = null,
//                // determining the parent of the new drawing (this can only be a group or a slide, not a paragraph)
//                drawingParentPoint = Position.getDOMPosition(self.getRootNode(target), _.initial(start), true),
//                // the parent node
//                drawingParent = drawingParentPoint ? drawingParentPoint.node : null,
//                // the children of the drawing container
//                children = drawingParent ? $(drawingParent).children(DrawingFrame.NODE_SELECTOR) : null,
//                // the number of children in the drawing group
//                childrenCount = children ? children.length : 0,
//                // the function used to insert the new drawing frame
//                insertFunction = 'insertAfter',
//                // the element, relative to that the new drawing is inserted
//                insertElement = drawingParent;
//
//            // checking, if the drawing parent is a slide or a group container
//            if (!(DOM.isSlideNode(drawingParent) || DrawingFrame.isGroupDrawingFrame(drawingParent))) { return false; }
//
//            // checking, if the insert position is valid -> it is not possible to insert at position 2, if there is only
//            // one drawing inside slide or group container
//            if (_.last(start) > childrenCount) { return false; }
//
//            // check, if 'appendTo' must be used instead of 'insertAfter'
//            if (_.last(start) === 0) {
//                // inserting the new drawing as first child of the parent
//                insertFunction = 'prependTo';
//            } else {
//                // finding that drawing, after which the new drawing will be inserted
//                insertElement = children[_.last(start) - 1];
//            }
//
//            // insert the drawing with default settings between the two text nodes (store original URL for later use)
//            drawingNode = DrawingFrame.createDrawingFrame(type)[insertFunction](insertElement);
//
//            // apply the passed drawing attributes
//            if (_.isObject(attributes)) { drawingStyles.setElementAttributes(drawingNode, attributes); }
//
//            // saving the attributes in drawing styles (ignoring the return value, no event is fired)
//            if (target) { self.getDrawingStyles().savePlaceHolderAttributes(target, attrs); }
//
//            return true;
//        }

        /**
         * Inserts a table component into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new table.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied at the new table component, as map of
         *  attribute maps (name/value pairs), keyed by attribute family.
         *
         * @param {Object} [target]
         *  If exists, defines node, to which start position is related.
         *
         * @returns {Boolean}
         *  Whether the table has been inserted successfully.
         */
        function implInsertTable(start, attrs, target) {

            var // the new table node
                table = $('<table>').attr('role', 'grid').append($('<colgroup>')),
                // whether the table was inserted into the DOM
                inserted = self.insertContentNode(_.clone(start), table, (target ? target : undefined));

            // insertContentNode() writes warning to console
            if (!inserted) { return false; }

            // apply the passed table attributes
            if (attrs && attrs.table) { self.getTableStyles().setElementAttributes(table, attrs); }

            return true;
        }

        // public methods -----------------------------------------------------

        /**
         * Inserting an image into a slide.
         *
         * @param {Object} imageHolder
         *  The object containing the image attributes. This are 'url' or 'substring'.
         */
        this.insertImageURL = function (imageHolder) {

            var // the attributes for the image
                attrs = { drawing: self.getInsertDrawingAttibutes(), image: {}, line: { type: 'none' } },
                // the logical position of the drawing
                start = self.getNextAvailablePositionInActiveSlide();

            if (imageHolder.url) {
                attrs.image.imageUrl = imageHolder.url;
                attrs.drawing.name = imageHolder.name;
            } else if (imageHolder.substring(0, 10) === 'data:image') {
                attrs.image.imageData = imageHolder;
            } else {
                attrs.image.imageUrl = imageHolder;
            }

            return getImageSize(attrs.image.imageUrl || attrs.image.imageData).then(function (size) {

                var // the deferred for inserting the image
                    def = $.Deferred(),
                    // whether the image was imported successfully
                    result = false,
                    // created operation
                    newOperation = null;

                // exit silently if we lost the edit rights
                if (!self.getEditMode()) {
                    return def.resolve();
                }

                self.getUndoManager().enterUndoGroup(function () {

                    // expanding the drawing attributes with the default margins
                    _.extend(attrs.drawing, size, self.getDefaultDrawingMargins());

                    // creating the operation
                    newOperation = { name: Operations.DRAWING_INSERT, start: start, type: 'image', attrs: attrs };
                    self.extendPropertiesWithTarget(newOperation, self.getActiveTarget());

                    result = self.applyOperations(newOperation);

                }); // enterUndoGroup()

                // selecting the new inserted drawing
                self.getSelection().setTextSelection(start, Position.increaseLastIndex(_.clone(start)));

                return result ? def.resolve() : def.reject();
            })
            .fail(function () {
                self.getApp().rejectEditAttempt('image');
            });
        };

        /**
         * Inserting a text frame drawing node of type 'shape' at the end of the slide.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.insertTable=false]
         *   Whether a text frame containing a table shall be inserted.
         *  @param {Object} [options.size=null]
         *   The size of the table to be inserted. This is only evaluated, if the option
         *   'insertTable' is true. The size object must contain the two properties
         *   'width' and 'height'.
         */
        this.insertTextFrame = function (options) {

            // the undo manager returns the return value of the callback function
            return self.getUndoManager().enterUndoGroup(function () {

                // helper function, that generates the operations
                function doInsertTextFrame() {

                    var // the operations generator
                        generator = self.createOperationsGenerator(),
                        // the default width of the text frame in 1/100 mm
                        defaultWidth = 8000,
                        // the width of the parent paragraph
                        paraWidth = 0,
                        // whether a paragraph or a table shall be included into the text frame
                        insertTable = Utils.getBooleanOption(options, 'insertTable', false),
                        // the table size
                        size = Utils.getObjectOption(options, 'size', null),
                        // the current cursor position
                        start = self.getNextAvailablePositionInActiveSlide(),
                        // the default drawing attributes
                        drawingAttrs = Utils.extendOptions({ width: defaultWidth }, self.getInsertDrawingAttibutes()),
                        // the default border attributes
                        lineAttrs = { color: { type: 'rgb', value: '000000' }, style: 'single', type: 'none', width: Border.getWidthForPreset('thin') },
                        // the default fill color attributes
                        fillAttrs = { color: { type: 'auto' }, type: 'solid' },
                        // the default attributes
                        attrs = { drawing: drawingAttrs, shape: { autoResizeHeight: true }, line: lineAttrs, fill: fillAttrs },
                        // the position of the first paragraph inside the text frame
                        paraPos = _.clone(start),
                        // a required style ID for style handling
                        parentStyleId = null,
                        // the paragraph node, in which the text frame will be inserted
                        paraNode = null,
                        // the attributes of the inserted table
                        tableAttributes = null,
                        // default table style
                        tableStyleId = null,
                        // the text position
                        textPos = null,
                        // target for operation - if exists
                        target = self.getActiveTarget(),
                        // the drawing styles
                        drawingStyles = self.getDrawingStyles(),
                        // created operation
                        newOperation = null;

                    // checking the width of the parent paragraph (36836)
                    paraNode = Position.getParagraphElement(self.getNode(), _.initial(start));

                    if (paraNode) {
                        paraWidth = Utils.convertLengthToHmm($(paraNode).width(), 'px');
                        if (paraWidth < defaultWidth) {
                            attrs.drawing.width = paraWidth;
                        }
                    }

                    paraPos.push(0);

                    // Adding styleId (only for ODT)
                    if (self.getApp().isODF()) { attrs.styleId = 'Frame'; }

                    // handling the required styles for the text frame
                    if (_.isString(attrs.styleId) && drawingStyles.isDirty(attrs.styleId)) {

                        // checking the parent style of the specified style
                        parentStyleId = drawingStyles.getParentId(attrs.styleId);

                        if (_.isString(parentStyleId) && drawingStyles.isDirty(parentStyleId)) {
                            // inserting parent of text frame style to document
                            self.generateInsertStyleOp(generator, 'drawing', parentStyleId, true);
                            drawingStyles.setDirty(parentStyleId, false);
                        }

                        // insert text frame style to document
                        self.generateInsertStyleOp(generator, 'drawing', attrs.styleId);
                    }

                    // text frames are drawings of type shape
                    newOperation = { attrs: attrs, start: start, type: 'shape' };
                    self.extendPropertiesWithTarget(newOperation, target);
                    generator.generateOperation(Operations.DRAWING_INSERT, newOperation);

                    if (insertTable) {

                        tableAttributes = { table: { tableGrid: [], width: 'auto' } };
                        tableStyleId = self.getDefaultUITableStylesheet();

                        // prepare table column widths (values are relative to each other)
                        _(size.width).times(function () { tableAttributes.table.tableGrid.push(1000); });

                        // set default table style
                        if (_.isString(tableStyleId)) {
                            // add table style name to attributes
                            tableAttributes.styleId = tableStyleId;
                            // default: tables do not have last row, last column and vertical bands
                            tableAttributes.table.exclude = ['lastRow', 'lastCol', 'bandsVert'];
                        }

                        // insert the table, and add empty rows into the text frame
                        newOperation = { start: paraPos, attrs: tableAttributes };
                        self.extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.TABLE_INSERT, newOperation);

                        newOperation = { start: Position.appendNewIndex(_.clone(paraPos), 0), count: size.height, insertDefaultCells: true };
                        self.extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.ROWS_INSERT, newOperation);

                        textPos = _.clone(paraPos).concat([0, 0, 0, 0]);

                    } else {

                        // add a paragraph into the shape, so that the cursor can be set into the text frame
                        // -> an operation is required for this step, so that remote clients are also prepared (-> no implicit paragraph)
                        newOperation = { start: paraPos };
                        self.extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.PARA_INSERT, newOperation);

                        // reducing distance of paragraphs inside the text frame
                        newOperation = { start: paraPos, attrs: { paragraph: { marginBottom: 0 } } };
                        self.extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);

                        textPos = Position.appendNewIndex(_.clone(paraPos), 0);
                    }

                    // apply all collected operations
                    self.applyOperations(generator);

                    // setting cursor into text frame
                    self.getSelection().setTextSelection(textPos);
                }

                doInsertTextFrame();
                return $.when();

            }, this);  // enterUndoGroup()
        };

        // operation handler --------------------------------------------------

        /**
         * The handler for the insertDrawing operation.
         */
        this.insertDrawingHandler = function (operation) {

            var // the undo operation for the insertSlide operation
                undoOperation = null;

            if (!implInsertDrawing(operation.type, operation.start, operation.attrs, operation.target)) {
                return false;
            }

            if (self.getUndoManager().isUndoEnabled()) {
                // not registering for undo if this is a drawing inside a drawing group (36150)
                undoOperation = Position.isPositionInsideDrawingGroup(self.getNode(), operation.start) ? null : { name: Operations.DELETE, start: operation.start };
                if (undoOperation) { self.extendPropertiesWithTarget(undoOperation, operation.target); }
                self.getUndoManager().addUndo(undoOperation, operation);
            }

            return true;
        };

        /**
         * The handler for the insertTable operation.
         */
        this.insertTableHandler = function (operation) {

            var // the undo operation for the insertSlide operation
                undoOperation = null;

            if (!implInsertTable(operation.start, operation.attrs, operation.target)) {
                return false;
            }

            // generate undo/redo operations
            if (self.getUndoManager().isUndoEnabled()) {
                undoOperation = { name: Operations.DELETE, start: operation.start };
                self.extendPropertiesWithTarget(undoOperation, operation.target);
                self.getUndoManager().addUndo(undoOperation, operation);
            }

            return true;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    } // class ObjectOperationMixin

    // constants --------------------------------------------------------------

    // exports ================================================================

    return ObjectOperationMixin;

});
