/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/format/slidestyles', [
    'io.ox/office/tk/utils',
    'io.ox/office/drawinglayer/view/imageutil',
    'io.ox/office/editframework/model/stylecollection'
], function (Utils, Image, StyleCollection) {

    'use strict';

    var // definitions for slide attributes
        DEFINITIONS = {

            /**
             * Whether the drawings from layout or master slide shall be visible.
             */
            followMasterShapes: { def: true }
        };

    // private global functions ===============================================

    // class SlideStyles =======================================================

    /**
     * Contains the style sheets for page formatting attributes. The CSS
     * formatting will be read from and written to the slide container elements.
     *
     * @constructor
     *
     * @extends StyleCollection
     *
     * @param {TextModel} docModel
     *  The text document model containing instance.
     */
    function SlideStyles(docModel) {

        var // self reference
            // self = this,
            // collector for the place holder attributes in layout or master slides
            listStylesCollector = {};

        // base constructor ---------------------------------------------------

        StyleCollection.call(this, docModel, 'slide', {
            families: 'fill',
            formatHandler: updateSlideFormatting
        });

        // private functions --------------------------------------------------

        /**
         * Saving the list style attributes in a container, if the slide is a master
         * or layout slide.
         *
         * This function can be called directly from insertDrawing operation. In that
         * case the attributes already contain the full set of properties from the
         * families 'drawing' and 'presentation'.
         *
         * @param {String} id
         *  The id of the layout or master slide, that defines the attributes for
         *  the presentation place holder type.
         *
         * @param {Object} attrs
         *  The attributes object, that contains the attributes for the specified
         *  target and place holder type. This are the explicit attributes set at
         *  the drawing.
         *
         * @returns {Object|Null}
         *  If the attributes are registered, an object with the complete registration
         *  information is returned. The object's keys are: 'target', 'placeHolderType',
         *  'placeHolderIndex' and 'attrs'. If the attributes were not registered,
         *  because target is not set (no master or layout slide) or the attributes
         *  contain no place holder information (no place holder drawing), null is
         *  returned.
         */
        function saveListStylesAttributes(id, attrs) {

            var // the properties inside the listStyles object
                type = null;

            // id must be defined, it must be a layout or master slide id and the attributes
            // must have the property 'listStyles' specified.
            // -> the attributes are saved in the model.
            if (!id || !docModel.isLayoutOrMasterId(id) || !attrs || !attrs.listStyles) { return null; }

            // using type: 'title', 'body', 'other'
            for (type in attrs.listStyles) {
                if (_.contains(SlideStyles.LIST_STYLES_TYPES, type)) {
                    listStylesCollector[id] = listStylesCollector[id] || {};
                    listStylesCollector[id][type] = attrs.listStyles[type];
                }
            }

            // returning an object with the complete new registration information
            return { id: id, listStyles: attrs.listStyles };
        }

        /**
         * Updating the slide background node with the specified attributes of the family 'fill'.
         *
         * @param {jQuery} slide
         *  The slide element whose attributes have been changed, as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute family,
         *  containing the effective attribute values merged from style sheets and
         *  explicit attributes.
         */
        function updateBackground(slide, mergedAttributes) {

            var // the fill attributes of the slide
                fillAttrs = mergedAttributes.fill,
                // the CSS properties for the slide
                cssProps = {};

            // calculate the CSS fill attributes
            switch (fillAttrs.type) {
                case 'none':
                    // clear everything: color and bitmaps
                    cssProps.background = '';
                    break;

                case 'solid':
                    // use 'background' compound attribute to clear bitmaps
                    cssProps.background = docModel.getCssColor(fillAttrs.color, 'fill');
                    break;

                case 'bitmap':
                    // stretching the drawing, but keeping the ratio
                    docModel.getImageSize(fillAttrs.imageUrl).then(function (size) {

                        var // the ratio of width and height of the slide
                            slideRatio = docModel.getSlideRatio(),
                            // the ratio of width and height of the drawing
                            drawingRatio = Utils.round(size.width / size.height, 0.01),
                            // the background size value for the width
                            width = '100%',
                            // the background size value for the width
                            height = '100%';

                        if (slideRatio > drawingRatio) {
                            // the drawing needs to be cropped at top and bottom -> height need to be increased
                            height = (Utils.round(slideRatio / drawingRatio, 0.01) * 100) + '%';
                        } else {
                            // the drawing needs to be cropped at left and right sides -> width need to be increased
                            width = (Utils.round(drawingRatio / slideRatio, 0.01) * 100) + '%';
                        }

                        cssProps.background = 'url("' + Image.getFileUrl(docModel.getApp(), fillAttrs.imageUrl)  + '")';
                        cssProps['background-position'] = 'center';
                        cssProps['background-size'] = width + ' ' + height;
                        slide.css(cssProps);
                    });
                    break;

                default:
                    Utils.warn('SlideStyles.setSlideBackground(): unknown fill type "' + fillAttrs.type + '"');
            }

            // apply the fill attributes
            slide.css(cssProps);
        }

        /**
         * Will be called for every page whose attributes have been changed.
         *
         * @param {jQuery} slide
         *  The slide element whose attributes have been changed, as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute family,
         *  containing the effective attribute values merged from style sheets and
         *  explicit attributes.
         */
        function updateSlideFormatting(slide, mergedAttributes) {

            // updating the slide background
            updateBackground(slide, mergedAttributes);

            // switching to visible or non-visible drawings in master or layout slide
            slide.attr('followmastershapes', mergedAttributes.slide.followMasterShapes);

            // inform the view, that the merged slide attributes (including 'fill') need to be updated
            docModel.trigger('change:slideAttributes');
        }

        // public methods -----------------------------------------------------

        /**
         * Public helper function: Saving the list styles attributes in a container,
         * if the slide is a master or a layout slide.
         * See description at private function 'saveListStyleAttributes'.
         */
        this.saveListStylesAttributes = function (id, attrs) {
            return saveListStylesAttributes(id, attrs);
        };

        /**
         * Receiving the attributes that are necessary to remove a slide background.
         *
         * @returns {Object}
         *  The attributes required to remove the slide background.
         */
        this.getEmptySlideBackgroundAttributes = function () {
            return SlideStyles.NO_SLIDE_BACKGROUND;
        };

        /**
         * Receiving the list styles attributes defined at a specified slide. This is a layout or master slide.
         * If the list styles are defined, the attributes from the attributes from the parent layers can be used.
         *
         * @param {String} id
         *  The slide ID.
         *
         * @param {String} type
         *  The 'type' of element, for that the list styles are searched. This can be for example 'title'.
         *
         * @returns {Object|Null}
         *  The list styles attributes defined for the specified slide and the specified type.
         */
        this.getListStylesAttributes = function (id, type) {
            if (type && SlideStyles.LIST_TYPE_CONVERTER[type]) { type = SlideStyles.LIST_TYPE_CONVERTER[type]; }
            if (type && !_.contains(SlideStyles.LIST_STYLES_TYPES, type)) { type = SlideStyles.DEFAULT_LIST_STYLES_TYPE; }
            return (listStylesCollector && listStylesCollector[id] && listStylesCollector[id][type]) ? _.clone(listStylesCollector[id][type]) : null;
        };

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        docModel.registerAttributeDefinitions('slide', DEFINITIONS);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = null;
        });

    } // class SlideStyles

    // constants --------------------------------------------------------------

    /**
     * The allowed types for the slide's list styles.
     */
    SlideStyles.LIST_STYLES_TYPES = ['title', 'body', 'other'];

    /**
     * The default type for the slide's list styles.
     */
    SlideStyles.DEFAULT_LIST_STYLES_TYPE = 'other';

    /**
     * Converter for specific types for the slide's list styles.
     */
    SlideStyles.LIST_TYPE_CONVERTER = { ctrTitle: 'title' };

    /**
     * Default solid border line for paragraphs and tables.
     */
    SlideStyles.NO_SLIDE_BACKGROUND = { fill: { type: null, color: null, imageUrl: null } };

    // exports ================================================================

    // derive this class from class StyleCollection
    return StyleCollection.extend({ constructor: SlideStyles });

});
