/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

/* eslint no-new: 0 */

define('io.ox/office/portal/action/toolbaractionsimpl', [
    'io.ox/core/extensions',
    'io.ox/core/extPatterns/links',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/dialogs',
    'io.ox/office/tk/utils/driveutils',
    'io.ox/office/portal/portalutils',
    'settings!io.ox/office',
    'gettext!io.ox/office/portal/main'
], function (Ext, Links, Utils, Dialogs, DriveUtils, PortalUtils, Settings, gt) {

    'use strict';

    /**
     * Creates all extensions points and bind actions for the portal toolbar.
     *
     * @constructor
     *
     * @param {Object} options
     *  @param {String} [options.appBaseName]
     *  @param {String} [options.newDocText]
     *  @param {String} [options.openDocText]
     */
    function ToolbarActionsImpl(options) {

        var // the name of the application, used for filtering files in file picker
            appBaseName =  Utils.getStringOption(options, 'appBaseName'),
            // identifier for the appBaseName toolbar
            toolbarPointId = 'io.ox/office/portal/toolbar' + appBaseName,
            // deferred is set when "newdoc" & "loaddoc" is called, so we can allow only one new call at once
            newDeferred = null, loadDeferred = null;

        this.toolbarPoint = Ext.point(toolbarPointId);

        // create extension point for the 'new appBaseName' link
        Ext.point(toolbarPointId + '-newblank').extend(new Links.Link({
            id: 'new_' + appBaseName,
            index: 100,
            prio: 'hi',
            label: Utils.getStringOption(options, 'newDocText'),
            cssClasses: 'new-document',
            ref: toolbarPointId + '/actions/new/' + appBaseName
        }));

        // add link to the toolbar point
        this.toolbarPoint.extend({
            id: 'new-' + appBaseName + 'document',
            index: 100,
            draw: function (baton) {
                Ext.point(toolbarPointId + '-newblank').invoke('draw', this, baton);
            }
        });

        //add open document link to  the toolbar point
        this.toolbarPoint.extend(new Links.Link({
            id: 'open-' + appBaseName + 'document',
            index: 200,
            prio: 'hi',
            label: gt('Open document'),
            cssClasses: 'open-document',
            ref: 'io.ox/office/portal/open' + appBaseName
        }));

        // define action that will be fired on the 'new appBaseName', 'new text' and 'open document' link
        new Links.Action(toolbarPointId + '/actions/new/' + appBaseName, {
            capabilities: appBaseName,
            action: function () {
                if (newDeferred && newDeferred.state() === 'pending') {
                    Utils.warn('cant run new document, because old call is still pending');
                    return;
                }
                var moduleName = PortalUtils.getEditorModuleName(appBaseName);
                PortalUtils.trackEvent({ moduleName: moduleName, target: 'toolbar', action: 'newblank' });
                newDeferred = ox.launch(moduleName + '/main', PortalUtils.getNewLaunchOptions());
            }
        });

        new Links.Action('io.ox/office/portal/open' + appBaseName, {
            action: function () {
                require(['io.ox/files/filepicker', 'io.ox/office/baseframework/app/extensionregistry']).done(function (FilePicker, ExtensionRegistry) {
                    var preselectKey = 'recentOpenDocPath' + appBaseName;
                    var preselect = Settings.get(preselectKey) || DriveUtils.getStandardDocumentsFolderId();

                    var dialog = new FilePicker({
                        // filter files of disabled applications (capabilities)
                        filter: function (file) {
                            return ExtensionRegistry.isEditable(file.filename);
                        },
                        sorter: function (file) {
                            return (file.filename || file.title).toLowerCase();
                        },
                        primaryButtonText: gt('Open'),
                        header: gt('Open document'),
                        folder: preselect,
                        uploadButton: true,
                        multiselect: false,
                        width: Dialogs.getBestDialogWidth(800),
                        //uploadFolder: DriveUtils.getStandardDocumentsFolderId(),
                        hideTrashfolder: true,
                        acceptLocalFileType: ExtensionRegistry.getEditableTypes()
                    });
                    dialog.done(function (selectedFiles) {
                        var selectedFile = selectedFiles[0];
                        if (selectedFile) {
                            if (selectedFile.creation_date) {
                                require(['io.ox/office/baseframework/utils/infomessages',  'io.ox/core/notifications']).done(function (InfoMessages, Notifications) {
                                    DriveUtils.getPath(selectedFile.folder_id).done(function (paths) {
                                        var path = DriveUtils.preparePath(paths);
                                        var messageData = InfoMessages.getMessageData('INFO_DOC_SAVED_IN_FOLDER', { fullFileName: selectedFile.title, fullPathName: path, type: 'success', duration: 10000 });
                                        Notifications.yell(messageData.type, messageData.message);
                                    });
                                });

                            }

                            var actionModuleName = PortalUtils.getModuleNameForFile(selectedFile.filename);

                            if (loadDeferred && loadDeferred.state() === 'pending') {
                                Utils.warn('cant run load document, because old call is still pending');
                                return;
                            }

                            PortalUtils.trackEvent({ moduleName: appBaseName, actionModuleName: actionModuleName, target: 'toolbar', action: 'open' });
                            if (ExtensionRegistry.isNative(selectedFile.filename)) {
                                loadDeferred = ox.launch(actionModuleName + '/main', { action: 'load', file: selectedFile });
                            } else {
                                loadDeferred = ox.launch(actionModuleName + '/main', { action: 'convert', target_folder_id: selectedFile.folder_id, templateFile: selectedFile, preserveFileName: true });
                            }
                            Settings.set(preselectKey, selectedFile.folder_id).save();
                        }
                    });
                });
            }
        });

    } // class ToolbarActionsImpl

    return ToolbarActionsImpl;

});
