/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */

/* eslint no-new: 0 */

/**
 *  Defines extensions and actions of the recent documents pane
 */
define('io.ox/office/portal/action/recentsactions', [
    'io.ox/core/extensions',
    'io.ox/core/extPatterns/links',
    'io.ox/core/extPatterns/actions',
    'io.ox/office/portal/portalutils',
    'io.ox/office/tk/utils',
    'io.ox/office/portal/view/contextmenu',
    'io.ox/office/tk/io',
    'gettext!io.ox/office/portal/main',
    'io.ox/files/toolbar', // important for action links (download in contextmenu)
    'io.ox/mail/detail/links' /* this line is important for deep links usage (works with jq delegation) */

], function (Ext, Links, CoreActions, PortalUtils, Utils, ContextMenu, IO, gt) {

    'use strict';

    function RecentsExtensions() {

        var recentDocumentExtension = 'io.ox/portal/office/recents/document';

        this.documentPoint = Ext.point(recentDocumentExtension);

        new Links.Action(recentDocumentExtension + '/open', {

            action: function (baton) {
                var appModuleName = PortalUtils.getModuleNameForFile(baton.data.filename);

                PortalUtils.trackEvent({ moduleName: appModuleName, target: 'recentlist', action: 'edit' });

                ox.launch(appModuleName + '/main', { action: 'load', file: baton.data });
                //FIXME: TODO: combine me with normal file-actions
            }

        });

        this.documentPoint.extend({

            id: 'recent_document',

            draw: function (baton) {
                var fullFileName = baton.data['com.openexchange.file.sanitizedFilename'] || baton.data.filename;
                var escapedFilePath = '';
                var escapedFilePathinToolTip = '';
                var escapedFileName = Utils.escapeHTML(fullFileName);
                var escapedTitle = null;

                // build the filepath
                _.each(baton.data.path, function (folderInfo, index) {

                    // the path displayed in recent list: The css ellipsis function doesn't
                    // work right with :before, therefore the the divider icon
                    // (awesomfont - &#xf105; = fa-angle-right) needs to be embedded directly here.
                    var divider = (baton.data.path.length === (index + 1)) ? '' : '<span class = "divider">' + _.noI18n('&#xf105;') + '</span>'; // Don't insert the divider symbol for the last element.
                    escapedFilePath += _.noI18n(Utils.escapeHTML(folderInfo.filename)) + divider;

                    // the path displayed in the tool tip:
                    var dividerinToolTip = (baton.data.path.length === (index + 1)) ? '' : '<i class="fa breadcrumb-divider" aria-hidden="true"></i>'; // Don't insert the divider symbol for the last element.
                    escapedFilePathinToolTip += _.noI18n(Utils.escapeHTML(folderInfo.filename)) + dividerinToolTip;

                });

                // build the titel for the tooltip
                escapedTitle = _.noI18n(escapedFileName) + '<br> <span class ="tooltip-filepath">' + escapedFilePathinToolTip + '</span>';
                // iPad title problem: long tap on element -> ios popup appears -> html markup is visible
                if (Utils.COMPACT_DEVICE) {
                    escapedTitle = escapedFileName;
                }

                // build the HTML output. Important: the tooltip displays HTML,
                // so all used values must be escaped
                var link = $('<a>').addClass('document-link row')
                    .attr({
                        'data-toggle': 'tooltip',
                        'data-html': 'true',
                        'data-animation': 'false',
                        'data-delay': '{ "show": 1400 }',
                        'data-viewport': '{ "selector": "body", "padding": 43}',
                        title: escapedTitle,
                        tabindex: 0
                    }),
                    divMetadata = $('<div>').addClass('document-metadata'),
                    divName = $('<div>').addClass('document-name col-xs-9'),
                    divFilename = $('<div>').addClass('document-filename').text(_.noI18n(fullFileName)),
                    divPath = $('<div>').addClass('document-path').append(escapedFilePath),
                    divDate = $('<div>').addClass('document-date col-xs-3').text(_.noI18n(PortalUtils.formatDate(baton.data.last_modified ? baton.data.last_modified : baton.data.last_opened)));

                divName.append(PortalUtils.createDocumentIcon(fullFileName), divFilename, divPath);
                link.append(divMetadata.append(divName, divDate));

                link.click(function () {

                    // only invoke the action when no context menu is open
                    // use case: the user can click everywere on the screen to close the contextmenu
                    if (!ContextMenu.isOpen()) {

                        // provide feedback to the user which element will be opened
                        PortalUtils.markElement(link);

                        CoreActions.invoke(recentDocumentExtension + '/open', this, baton);
                    }

                });

                link.on('keydown', PortalUtils.keydownHandler);

                // handle context menu
                link.on('contextmenu', function (e) {
                    contextMenuHandler(e, link, baton);
                });

                // prevent focus change on click due to mousedown
                link.on('mousedown', function (e) {
                    e.preventDefault();
                });

                // iOS: check if there is a long tap to invoke a 'contextmenu' event
                if (Utils.IOS) {
                    link.on('touchstart', function (event) {
                        PortalUtils.iosContextEventHandler(event, link);
                    });
                }

                this.append(link);
            }
        });

        /*
         * A Handler function to generate the the context menu for recent list
         * itemns in the text/spreadsheet portal.
         *
         * @param {Event} event
         *  The event which triggered the context menu.
         *  The event contains the position were the context menu is opened.
         *
         * @param {JQuery} link
         *  The DOM element the context menu is opened for.
         *
         * @param {Object} baton
         *  The baton for the element the context menu is opened for.
         */
        function contextMenuHandler(event, link, baton) {

            // global flag that a context menu is open
            ContextMenu.setOpen(true);
            // create the context menu
            var cmenu = new ContextMenu(event);
            PortalUtils.markElement(link);

            // init the context menu GUI
            var editButton = cmenu.createItemNode('edit', { content: gt('Edit') });
            var editAsNewButton = cmenu.createItemNode('editasnew', { content: gt('Edit as new') });

            // download is not possible on iOS -> no download button in the context menu
            var downloadButton = null;
            if (!(Utils.IOS)) {
                downloadButton = cmenu.createItemNode('download', { content: gt('Download') });
            }

            var showInDriveButton = cmenu.createItemNode('showindrive', { content: gt('Show in Drive') });
            cmenu.createSectionNode('listActions');
            var removeFromRecentListButton = cmenu.createItemNode('removefromrecentlist', { section: 'listActions', content: /*#. context: remove the selected item from the 'recent documents' list in app portal */ gt('Remove from list') });
            var clearRecentListButton = cmenu.createItemNode('clearrecentlist ', { section: 'listActions', content: /*#. context: clear all items from the 'recent documents' list in app portal */ gt('Clear list') });

            // init event listener for the created buttons
            editButton.on('click', function () {
                cmenu.hide(); //hide the context menu after a button was clicked
                CoreActions.invoke(recentDocumentExtension + '/open', this, baton);
            });

            editAsNewButton.on('click', function () {
                cmenu.hide(); //hide the context menu after a button was clicked
                CoreActions.invoke(PortalUtils.getAppBaseName(baton.data.filename) + '-edit-asnew', this, baton); //action from filesaction.js
            });

            // download is not possible on iOS therefore no download button on iOS
            if (!(Utils.IOS)) {
                downloadButton.on('click', function () {
                    cmenu.hide(); //hide the context menu after a button was clicked
                    CoreActions.invoke('io.ox/files/actions/download', this, baton);
                });
            }

            showInDriveButton.on('click', function () {
                cmenu.hide(); //hide the context menu after a button was clicked

                PortalUtils.showInDrive(baton.data, true);
            });

            removeFromRecentListButton.on('click', function () {
                cmenu.hide(); //hide the context menu after a button was clicked
                IO.sendRequest(IO.FILTER_MODULE_NAME, { action: 'deletefilefromrecentlist', type: PortalUtils.getAppBaseName(baton.data.filename), remove_file_id: baton.data.id }).done(function () {

                    // Remove the item from the backbone collection which represents the recent list.
                    // This will fire an event ('update') which will re-render the recent list.
                    baton.model.collection.remove(baton.model);
                });
            });

            clearRecentListButton.on('click', function () {
                cmenu.hide(); //hide the context menu after a button was clicked
                IO.sendRequest(IO.FILTER_MODULE_NAME, { action: 'clearrecentlist', type: PortalUtils.getAppBaseName(baton.data.filename) }).done(function () {

                    // Remove all items from the backbone collection which represents the recent list.
                    // This will fire an event ('reset') which will re-render the recent list.
                    baton.model.collection.reset(null);
                });
            });

            // show the context menu
            cmenu.show();

            // remove the highlight and destory the context menu
            cmenu.on('popup:hide', function () {
                PortalUtils.removeMark(link);

                // After a click event the context menu is closed by the basemenu globalClickHandler().
                // When a context menu is open, clicking on an element should not invoke it's attached action, but just close the context menu.
                // As 'closing' and 'invocation' due to a click event happens at the same time, ContextMenu.setOpen(false) must be called delayed.
                window.setTimeout(function () { ContextMenu.setOpen(false); }, 300);

                // destroy the context menu
                cmenu = null;
            });
        }

    } // class RecentExtensions

    return new RecentsExtensions();

});
