/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/toppane', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/view/toolpane',
    'io.ox/office/baseframework/view/toolbar',
    'io.ox/office/editframework/view/editlabels',
    'io.ox/office/editframework/view/editcontrols',
    'gettext!io.ox/office/editframework/main'
], function (Utils, ToolPane, ToolBar, Labels, Controls, gt) {

    'use strict';

    // class TopPane ==========================================================

    /**
     * Represents the top-level view pane in OX Document applications,
     * containing the file name, the file drop-down menu, and additional global
     * controls such as the Close button and the Undo/Redo buttons.
     *
     * @constructor
     *
     * @extends ToolPane
     *
     * @param {EditView} docView
     *  The document view instance containing this view pane.
     *
     * @param {ToolBarTabCollection} toolBarTabs
     *  The collection of all registered tool bar tabs.
     */
    function TopPane(docView, toolBarTabs) {

        var // self reference
            self = this,

            // the tab buttons to activate the different edit tool bars
            tabGroup = null,

            // the tab buttons to activate the different edit tool bars, as drop-down list
            tabList = null,

            // the application status label
            statusLabel = null,

            // the 'View' menu group
            viewMenuGroup = null,

            // the containing nodes (lead, center, trail) of the TopPane (incl. content-nodes)
            leadWrapperNode = null,
            leadAreaNode = null,
            centerWrapperNode = null,
            centerAreaNode = null,
            trailWrapperNode = null,
            trailAreaNode = null,

            // shrink controls in multiple states
            shrinkState = 0;

        // base constructor ---------------------------------------------------

        ToolPane.call(this, docView, { position: 'top', classes: 'top-pane' });

        // private methods ----------------------------------------------------

        function paneLayoutHandler() {

            var // get all sizes of the containing divs in top-pane
                areaSizes           = self.getAreaSizes(),
                // total width of pane
                paneWidth           = areaSizes.total,

//                // lead, center and trailing width of the wrapped nodes
//                leadWrapperWidth    = leadWrapperNode.width(),
//                centerWrapperWidth  = centerWrapperNode.width(),
//                trailWrapperWidth   = trailWrapperNode.width(),

                // lead, center and trailing width of the included content
                leadWidth           = leadAreaNode.outerWidth(),
                centerWidth         = centerAreaNode.outerWidth(),
                trailWidth          = trailAreaNode.outerWidth();

            // set the new widths of the nodes (wrapper + content nodes)
            function widthUpdate() {
//                leadWrapperWidth    = leadWrapperNode.width();
//                centerWrapperWidth  = centerWrapperNode.width();
//                trailWrapperWidth   = trailWrapperNode.width();

                leadWidth           = leadAreaNode.outerWidth();
                centerWidth         = centerAreaNode.outerWidth();
                trailWidth          = trailAreaNode.outerWidth();
            }

            // get total needed width of all content-nodes
            function getTotalNeededWidth() {
                return (leadWidth + centerWidth + trailWidth);
            }

            // shrink controls in specific steps (self prio)
            function shrink() {
                switch (shrinkState) {

                    case 0:                 // on shrink state 0
                        // make the status-label smaller
                        statusLabel.activateSmallVersion();
                        shrinkState = 1;    // set new shrink state
                        break;

                    case 1:                 // on shrink state 1
                        // show the DropDown instead of the Tabs
                        tabGroup.hide();
                        tabList.show();
                        shrinkState = 2;    // set new shrink state
                        break;

                    case 2:                 // on shrink state 2
                        // make the tab-DropDown smaller
                        tabList.activateSmallVersion();
                        shrinkState = 3;    // set new shrink state
                        break;
                }
            }

            // unshrink controls in opposite direction as shrink does
            function unshrink() {
                switch (shrinkState) {

                    case 1:                 // on shrink state 1
                        // get the old styles of the status-label
                        statusLabel.deactivateSmallVersion();
                        shrinkState = 0;    // set new shrink state
                        break;

                    case 2:                 // on shrink state 2
                        // show the Tabs instead of the DropDown
                        tabGroup.show();
                        tabList.hide();
                        shrinkState = 1;    // set new shrink state
                        break;

                    case 3:                 // on shrink state 3
                        // show the normal version of the tab-DropDown
                        tabList.deactivateSmallVersion();
                        shrinkState = 2;    // set new shrink state
                        break;
                }
            }

            // NOT enough place
            if (getTotalNeededWidth() > paneWidth) {
                // reduce the place between the controls
                this.getNode().addClass('small-distance');
                // recalculate the widths
                widthUpdate();

                // if still not enough place
                if (getTotalNeededWidth() > paneWidth) {
                    // begin to shrink the controls
                    // as long as possible, or it fits in
                    shrink();
                }

            // FREE place available
            } else {
                // unshrink the current state
                unshrink();
                // recalculate the widths
                widthUpdate();

                // if we now don't fit in the pane
                if (getTotalNeededWidth() > paneWidth) {
                    // re-shrink the old state
                    shrink();

                // if we have still enough place
                } else {
                    // place the default padding between the controls
                    this.getNode().removeClass('small-distance');
                    // recalculate the widths
                    widthUpdate();

                    // if we now don't fit in the pane
                    if (getTotalNeededWidth() > paneWidth) {
                        // re-add the smaller padding
                        this.getNode().addClass('small-distance');
                    }
                }
            }

        }

        // public methods -----------------------------------------------------

        /**
         * Returns the 'View' drop-down menu group containing controls for view
         * settings of the application.
         *
         * @returns {CompoundButton}
         *  The 'View' drop-down menu group.
         */
        this.getViewMenuGroup = function () {
            return viewMenuGroup;
        };

        // initialization -----------------------------------------------------

        // initialize class members
        leadWrapperNode = this.getAreaWrapperNode('leading');
        leadAreaNode = leadWrapperNode.find('>.area-container');
        centerWrapperNode = this.getAreaWrapperNode('center');
        centerAreaNode = centerWrapperNode.find('>.area-container');
        trailWrapperNode = this.getAreaWrapperNode('trailing');
        trailAreaNode = trailWrapperNode.find('>.area-container');

        // create the tab button/list controls
        tabGroup = new Controls.ToolBarTabGroup(toolBarTabs);
        tabList = new Controls.ToolBarTabList(toolBarTabs);

        // create the tool bar containing the tab buttons for the edit tool pane
        this.addViewComponent(new ToolBar(docView)
            .addGroup('view/toolbars/tab', tabGroup)
            .addGroup('view/toolbars/tab', tabList.hide()),
        { targetArea: 'leading' });

        // create the tool bar containing the application status label
        statusLabel = new Controls.ApplicationStatusLabel(docView, { smallerVersion: { css: { width: 20 }, hideLabel: true } });
        this.addViewComponent(new ToolBar(docView).addGroup(null, statusLabel));

        // center status label in top bar when loading a document
        this.listenTo(docView.getApp(), 'docs:state', function (state) {
            centerWrapperNode.css('text-align', (state === 'preview' ? 'center' : ''));
        });

        // create the 'View' drop-down menu
        viewMenuGroup = new Controls.CompoundButton(docView, { label: Labels.VIEW_LABEL, anchorPadding: -3 });

        // create the tool bar with the standard controls
        this.addViewComponent(new ToolBar(docView)
            .addGroup('document/undo', new Controls.Button({ icon: 'docs-undo', tooltip: gt('Revert last operation') }), { visibleKey: 'document/editable' })
            .addGroup('document/redo', new Controls.Button({ icon: 'docs-redo', tooltip: gt('Restore last operation') }), { visibleKey: 'document/editable' })
            .addGap()
            .addGroup('view/searchgroup', new Controls.Button({ icon: 'fa-search', tooltip: gt('Toggle search'), toggle: true }), { visibleKey: 'app/valid' })
            .addGap()
            .addGroup(null, viewMenuGroup)
            .addSeparator()
            .addGroup('app/quit', new Controls.Button(Labels.QUIT_BUTTON_OPTIONS), { visibleKey: 'app/bundled' }),
        { targetArea: 'trailing' });

        // update view pane after it has been resized, or contents have been changed
        this.on('pane:layout', paneLayoutHandler);
        this.listenTo(toolBarTabs, 'tab:show', function () { self.updateAreaSizes(); });
        this.listenTo(toolBarTabs, 'tab:hide', function () { self.updateAreaSizes(); });

        // show tab buttons, hide drop-down list
        tabGroup.show();
        tabList.hide();

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = toolBarTabs = tabGroup = tabList = statusLabel = viewMenuGroup = null;
        });

    } // class TopPane

    // exports ================================================================

    // derive this class from class ToolPane
    return ToolPane.extend({ constructor: TopPane });

});
