/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/popup/attributestooltip', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/popup/tooltip',
    'less!io.ox/office/editframework/view/popup/attributestooltip'
], function (Utils, Forms, ToolTip) {

    'use strict';

    // class AttributesToolTip ================================================

    /**
     * A tooltip for formatting attribute sets shown in the operations pane in
     * debugging mode.
     *
     * @constructor
     *
     * @extends ToolTip
     */
    function AttributesToolTip(docView) {

        // the document model
        var docModel = docView.getDocModel();

        // the top-right close button
        var closeButton = $(Forms.createButtonMarkup({ content: Forms.createIconMarkup('fa-times') })).addClass('close-btn');

        // the registered value formatters
        var formatters = [];

        // base constructor ---------------------------------------------------

        ToolTip.call(this, {
            restrictSize: 'none', // override default of class ToolTip
            textSelect: true
        });

        // private methods ----------------------------------------------------

        /**
         * Generates the HTML mark-up for a color box.
         */
        function generateColorBoxMarkup(color, type) {
            var cssColor = docModel.getCssColor(color, type);
            return '<span style="display:inline-block;box-sizing:border-box;width:14px;height:14px;border:1px solid #ccc;background:' + cssColor + ';">&nbsp;</span>';
        }

        /**
         * Generates the HTML mark-up for the specified attribute value.
         */
        function generateAttributeMarkup(family, name, value) {

            // matches the passed string/RE selector against the test value
            function matchSelector(selector, test) {
                return (_.isString(selector) && (selector === test)) || (_.isRegExp(selector) && _.isString(test) && selector.test(test));
            }

            // try to find a matching custom formatter
            var formatterData = _.find(formatters, function (data) {

                // match family and/or attribute name
                if (_.isObject(data.selector)) {
                    if (('family' in data.selector) && !matchSelector(data.selector.family, family)) { return false; }
                    if (('name' in data.selector) && !matchSelector(data.selector.name, name)) { return false; }
                    if (('value' in data.selector) && !(matchSelector(data.selector.value, value) || (_.isFunction(data.selector.value) ? data.selector.value(value) : _.isEqual(data.selector.value, value)))) { return false; }
                    return true;
                }

                // string/RE shortcut for attribute name (ignoring family)
                return matchSelector(data.selector, name);
            });

            // the stringified HTML representation of the value
            var markup = Utils.escapeHTML(Utils.stringifyForDebug(value));

            // use the first matching formatter to format the value
            if (formatterData) { markup = formatterData.formatter(value, markup); }

            // return invalid marker, if formatter does not return a string
            return _.isString(markup) ? markup : '&lt;invalid&gt;';
        }

        // public methods -----------------------------------------------------

        /**
         * Registers a formatter callback function for specific attribute types
         * or values.
         *
         * @param {Object|String|RegExp} selector
         *  A selector for the attributes to be formatted with the callback
         *  function. If set to a string, or regular expression, the attribute
         *  name will be matched. Otherwise, must be an object with the
         *  following optional properties:
         *  - {String|RegExp} [family]
         *      If specified, the attribute family must match this value. By
         *      default, the attribute family will be ignored.
         *  - {String|RegExp} [name]
         *      If specified, the attribute name must match this value. By
         *      default, the attribute name will be ignored.
         *  - {Any} [value]
         *      If specified, the attribute value must match this value. If set
         *      to a regular expression, the attribute value must be a string
         *      that matches that expression. If set to a function, it will be
         *      invoked with the attribute value. If the return value is
         *      truthy, the attribute value matches. Otherwise, the attribute
         *      value must be exactly equal to the value (deep comparison of
         *      objects and arrays).
         *
         * @param {Function} formatter
         *  The callback function that generates HTML mark-up for the attribute
         *  values matching the specified selector. Receives the following
         *  parameters:
         *  (1) {Any} value
         *      The attribute value to be converted to HTML mark-up.
         *  (2) {String} markup
         *      The standard HTML mark-up for the passed attribute value (the
         *      string representation of the JSON data). Can be used to add
         *      some new mark-up to the original string representation, instead
         *      of replacing it with new mark-up.
         *  Must return the resulting HTML mark-up for the passed value.
         *
         * @returns {AttributesToolTip}
         *  A reference to this instance.
         */
        this.registerFormatter = function (selector, formatter) {
            formatters.unshift({ selector: selector, formatter: formatter });
            return this;
        };

        /**
         * Initializes the contents of this tool-tip from the settings of the
         * passed DOM element.
         *
         * @param {HTMLElement|jQuery} node
         *  The DOM element to be used to fill this tool-tip. The formatting
         *  attributes will be taken from a specific element attribute. See
         *  method AttributesToolTip.createAttributeMarkup() for details.
         *  Additionally, this tool-tip will be anchored to the DOM element.
         *
         * @returns {AttributesToolTip}
         *  A reference to this instance.
         */
        this.initialize = function (node) {

            // extract the attribute set from the passed DOM node
            var attributeSet = $(node).attr(AttributesToolTip.ATTRIBUTE_NAME);
            try {
                attributeSet = JSON.parse(attributeSet);
            } catch (error) {
                Utils.exception(error);
                attributeSet = null;
            }

            // an attribute set must be an object
            if (!_.isObject(attributeSet)) {
                return this.clearContents();
            }

            // the complete HTML mark-up for the tool-tip
            var markup = '<table><tr><th>Family</th><th>Attribute</th><th>Value</th></tr>';

            // generate the HTML mark-up for the style sheet identifier
            if ('styleId' in attributeSet) {
                var styleId = attributeSet.styleId;
                markup += '<tr><td colspan="2">style</td><td>' + (_.isString(styleId) ? Utils.escapeHTML(styleId) : '&lt;invalid&gt;') + '</td></tr>';
                delete attributeSet.styleId;
            }

            // generate the HTML mark-up for the attribute set
            var markupBlocks = _.map(attributeSet, function (attributes, family) {

                // the mark-up for all values, as objects with 'name' and 'markup' properties
                var markupLines = [];

                if (_.isObject(attributes)) {
                    _.each(attributes, function (value, name) {
                        markupLines.push({ name: name, markup: generateAttributeMarkup(family, name, value) });
                    });
                } else {
                    markupLines.push({ name: '<invalid>', markup: '' });
                }

                // convert the attribute lines to mark-up for table cells
                markupLines = _.sortBy(markupLines, 'name').map(function (line) {
                    return '<td>' + Utils.escapeHTML(line.name) + '</td><td>' + line.markup + '</td>';
                });

                // create the resulting mark-up (first cell with family name spans over all rows)
                var familyMarkup = '<tr><td rowspan="' + markupLines.length + '">' + Utils.escapeHTML(family) + '</td>' + markupLines.join('</tr></tr>') + '</tr>';
                return { family: family, markup: familyMarkup };
            });

            // finalize and set the HTML mark-up
            markup += _.pluck(_.sortBy(markupBlocks, 'family'), 'markup').join('');
            markup += '</table>';
            this.clearContents('table').prependContentNodes(markup);

            // set the passed DOM node as anchor for the tool-tip
            return this.setAnchor(node);
        };

        // initialization -----------------------------------------------------

        // add specific CSS class for content styling
        this.getNode().addClass('attributes-tooltip');

        // close the tool-tip with the button
        this.appendContentNodes(closeButton);
        closeButton.on('click', this.hide.bind(this));

        // register a formatter for colors (add a color box)
        this.registerFormatter({ name: /color/i, value: _.isObject }, function (color, markup) {
            return generateColorBoxMarkup(color, 'fill') + '&nbsp;' + markup;
        });

        // register a formatter for border lines (add a color box)
        this.registerFormatter({ name: /border/i, value: _.isObject }, function (border, markup) {
            return generateColorBoxMarkup(_.isObject(border.color) ? border.color : { type: 'auto' }, 'line') + '&nbsp;' + markup;
        });

        this.registerDestructor(function () {
            docModel = docView = closeButton = null;
        });

    } // class AttributesToolTip

    // constants --------------------------------------------------------------

    /**
     * The name of the HTML element attribute used to carry the attribute set
     * as stringified JSON data.
     *
     * @constant
     */
    AttributesToolTip.ATTRIBUTE_NAME = 'data-attributes-tooltip';

    /**
     * A CSS selector for HTML elements containing an attribute set, as used by
     * the class AttributesToolTip.
     *
     * @constant
     */
    AttributesToolTip.NODE_SELECTOR = '[' + AttributesToolTip.ATTRIBUTE_NAME + ']';

    // static methods ---------------------------------------------------------

    /**
     * Creates the HTML mark-up for an HTML element attribute used to carry the
     * passed set of formatting attributes, as used by the class
     * AttributesToolTip.
     *
     * @param {Object} attributeSet
     *  An incomplete or complete formatting attribute set, as used in various
     *  document operations.
     *
     * @returns {String}
     *  The HTML mark-up for an HTML element attribute carrying the passed set
     *  of formatting attributes.
     */
    AttributesToolTip.createAttributeMarkup = function (attributeSet) {
        return AttributesToolTip.ATTRIBUTE_NAME + '="' + Utils.escapeHTML(JSON.stringify(attributeSet)) + '"';
    };

    // exports ================================================================

    return ToolTip.extend({ constructor: AttributesToolTip });

});
