/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Mario Schroeder <mario.schroeder@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/clipboardpane', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/baseframework/view/pane'
], function (Utils, KeyCodes, Pane) {

    'use strict';

    // class ClipboardPane ====================================================

    /**
     * A view pane attached to the bottom border of the application view that
     * displays the HTML mark-up contained in the browser clipboard.
     *
     * @constructor
     *
     * @extends Pane
     *
     * @param {EditView} docView
     *  The document view containing this pane instance.
     */
    function ClipboardPane(docView) {

        var // the DOM node containing the clipboard data
            contentNode = null;

        // base constructor ---------------------------------------------------

        Pane.call(this, docView, {
            position: 'bottom',
            classes: 'clipboard-pane noI18n',
            size: 150,
            resizable: true,
            minSize: 100,
            maxSize: 400
        });

        // private methods ----------------------------------------------------

        function createOperationsMarkup(operations) {

            var markup = '<table><tbody><tr><th></th><th>Name</th><th>Parameters</th>';

            // add operations
            _.each(operations, function (operation, index) {
                var name = Utils.escapeHTML(operation.name || '');
                operation = _.clone(operation);
                delete operation.name;
                delete operation.osn;
                delete operation.opl;
                markup += '<tr><td>' + index + '</td><td>' + name + '</td><td>' + Utils.escapeHTML(Utils.stringifyForDebug(operation)) + '</td>';
            });

            markup += '</tbody></table>';
            return markup;
        }

        // methods ------------------------------------------------------------

        this.setClipboardContent = function (content) {

            var // the mark-up to be inserted into the clipboard node
                markup = null;

            if ((_.isArray(content)) && (content.length > 0)) {
                markup = createOperationsMarkup(content);
            } else if ((content instanceof $) && (content.contents().length > 0)) {
                markup = '<span>' + Utils.escapeHTML(content.html()) + '</span>';
            } else if ((_.isString(content)) && (content.length > 0)) {
                markup = '<span>' + Utils.escapeHTML(content) + '</span>';
            } else {
                markup = '<span>Clipboard contains no data</span>';
            }

            // clear old and set new clipboard content
            contentNode[0].innerHTML = markup;

            return this;
        };

        // initialization -----------------------------------------------------

        // initialize and insert the clipboard data node
        contentNode = $('<div class="clip-container" tabindex="1"><span>Clipboard debug window</span></div>');
        this.getNode().append(contentNode);

        // keyboard shortcuts
        contentNode.on('keydown', function (event) {
            if (KeyCodes.matchKeyCode(event, 'A', { ctrlOrMeta: true })) {
                var docRange = window.document.createRange();
                docRange.setStart(event.delegateTarget, 0);
                docRange.setEnd(event.delegateTarget, 1);
                window.getSelection().removeAllRanges();
                window.getSelection().addRange(docRange);
                return false;
            }
            if (KeyCodes.matchKeyCode(event, 'DELETE')) {
                contentNode.empty();
                return false;
            }
            if (event.keyCode === KeyCodes.ESCAPE) {
                docView.grabFocus();
                return false;
            }
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            contentNode.remove();
            contentNode = null;
        });

    } // class ClipboardPane

    // exports ================================================================

    // derive this class from class Pane
    return Pane.extend({ constructor: ClipboardPane });

});
