/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/utils/mixedborder', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border'
], function (Utils, Color, Border) {

    'use strict';

    var // the names of all border properties in cells, mapped by border mode properties
        CELL_BORDER_ATTRIBUTES = {
            left: 'borderLeft',
            right: 'borderRight',
            top: 'borderTop',
            bottom: 'borderBottom',
            insideh: 'borderInsideHor',
            insidev: 'borderInsideVert'
        },

        // the names of all border properties in paragraphs, mapped by border mode properties
        PARAGRAPH_BORDER_ATTRIBUTES = {
            left: 'borderLeft',
            right: 'borderRight',
            top: 'borderTop',
            bottom: 'borderBottom',
            insideh: 'borderInside'
        };

    // private static functions ===============================================

    /**
     * Returns a map with names of border attributes used in operations, mapped
     * by border mode keys as used in the UI.
     */
    function getBorderAttributeNames(options) {
        return Utils.getBooleanOption(options, 'paragraph', false) ? PARAGRAPH_BORDER_ATTRIBUTES : CELL_BORDER_ATTRIBUTES;
    }

    // static class MixedBorder ===============================================

    /**
     * Provides helper function to mix several border attributes, e.g. from a
     * selection containing multiple elements with borders, and to convert
     * between mixed borders and an abstract border mode used in GUI border
     * controls.
     */
    var MixedBorder = {};

    // public methods ---------------------------------------------------------

    /**
     * Returns whether the passed mixed border object is valid and visible
     * (line style ambiguous or different to 'none', and line width ambiguous
     * or greater than 0).
     *
     * @param {Object} [mixedBorder]
     *  The mixed border object, as returned for example by the method
     *  'MixedBorder.mixBorders()'. If the parameter is missing or not an
     *  object, it is considered invisible.
     *
     * @returns {Boolean}
     *  Whether the mixed border object represents a visible border style.
     */
    MixedBorder.isVisibleBorder = function (mixedBorder) {
        return _.isObject(mixedBorder) &&
            // in mixed state, there are always some borders visible
            (mixedBorder.mixed || (mixedBorder.style !== 'none')) &&
            // border width may be ambiguous, but must not be 0)
            (_.isNull(mixedBorder.width) || (_.isNumber(mixedBorder.width) && (mixedBorder.width > 0)));
    };

    /**
     * Mixes the passed border objects, and returns a mixed border object
     * representing the states of the single border properties.
     *
     * @param {Object|Array} [...]
     *  An unlimited number of parameters, each parameter can be a single mixed
     *  border objects and/or regular border attribute value, or an array of
     *  these objects.
     *
     * @returns {Object}
     *  A mixed border object representing the states of the properties of all
     *  passed borders. Contains the following properties:
     *  - {Boolean} mixed
     *      Whether the visibility state of the borders is ambiguous. If true,
     *      a few but not all borders are visible, and the other borders are
     *      invisible.
     *  - {String|Null} style
     *      The line style of all visible border lines. Will be null, if the
     *      line style is ambiguous. Will be 'none', if ALL borders are
     *      invisible.
     *  - {Number|Null} [width]
     *      The line width of all visible border lines, in 1/100 mm. Will be
     *      null, if the line width is ambiguous. Will be omitted, if the
     *      border is invisible.
     *  - {Object|Null} [color]
     *      The line color of all visible border lines. Will be null, if the
     *      line color is ambiguous. Will be omitted, if the border is
     *      invisible.
     */
    MixedBorder.mixBorders = function () {

        var // the resulting mixed border object
            mixedBorder = { mixed: false };

        // process all parameters
        _.each(arguments, function (arg) {
            _.each(_.getArray(arg), function (border) {

                var // whether the passed border is visible
                    visible = ('mixed' in border) ? MixedBorder.isVisibleBorder(border) : Border.isVisibleBorder(border);

                // set initial line style; or update the mixed visibility state
                if (!('style' in mixedBorder)) {
                    mixedBorder.style = border.style;
                } else if (!mixedBorder.mixed && _.isObject(border)) {
                    mixedBorder.mixed = (border.mixed === true) || (MixedBorder.isVisibleBorder(mixedBorder) !== visible);
                }

                // skip the other border line properties, if the border is invisible
                if (!visible) { return; }

                // mix line style (null represents mixed state of visible styles)
                if (mixedBorder.style === 'none') {
                    mixedBorder.style = border.style;
                } else if (mixedBorder.style !== border.style) {
                    mixedBorder.style = null;
                }

                // update line width (null represents mixed state)
                if (!('width' in mixedBorder)) {
                    mixedBorder.width = border.width;
                } else if (_.isNumber(mixedBorder.width) && (mixedBorder.width !== border.width)) {
                    mixedBorder.width = null;
                }

                // update line color (null represents mixed state)
                if (!('color' in mixedBorder)) {
                    mixedBorder.color = border.color;
                } else if (_.isObject(mixedBorder.color) && !Color.isEqual(mixedBorder.color, border.color)) {
                    mixedBorder.color = null;
                }
            });
        });

        // do not return unused attributes for invisible borders retrieved from input data
        if (mixedBorder.style === 'none') {
            return { mixed: false, style: 'none' };
        }

        // add style 'none', if style property is still missing (none of the borders is visible)
        return _.extend({ style: 'none' }, mixedBorder);
    };

    /**
     * Converts the passed mixed border attributes to an object describing the
     * visibility states of all borders.
     *
     * @param {Object} borderAttributes
     *  The attribute map containing mixed border objects, as returned for
     *  example by the method 'MixedBorder.mixBorders()', mapped by border
     *  attribute names (e.g. 'borderLeft', 'borderTop', etc.).
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.paragraph=false]
     *      If set to true, a single inner border attribute 'borderInside' will
     *      be used instead of the two attributes 'borderInsideHor' and
     *      'borderInsideVert' (for paragraphs which are stacked vertically and
     *      do not have vertical inner borders).
     *
     * @returns {Object}
     *  An object with position keys ('left', 'top', etc.), and Boolean values
     *  or Null values specifying whether the respective borders are visible.
     *  If the visibility state of a mixed border object in the passed
     *  attribute map is missing or ambiguous (property 'mixed' is true), the
     *  map value will be null. Otherwise, the Boolean value of the property
     *  states whether the border is visible.
     */
    MixedBorder.getBorderMode = function (borderAttributes, options) {

        var // the border attribute names
            attributeNames = getBorderAttributeNames(options),
            // result mapping border attribute names to boolean values
            borderMode = {};

        _.each(attributeNames, function (attrName, propName) {
            var border = borderAttributes[attrName];
            if (_.isObject(border)) {
                borderMode[propName] = (border.mixed === true) ? null : MixedBorder.isVisibleBorder(border);
            }
        });

        return borderMode;
    };

    /**
     * Converts the passed object describing the visibility states of cell
     * border attributes to a table attribute map containing all border
     * attributes.
     *
     * @param {Object|String} borderMode
     *  An object with position keys ('left', 'top', etc.), mapping Boolean
     *  values specifying whether the respective border is visible. Properties
     *  not contained in this object will not occur in the returned attribute
     *  map. Alternatively, the passed value can be a single position key as
     *  string, which causes to toggle the current visibility state of the
     *  respective border attribute.
     *
     * @param {Object} origAttributes
     *  The original attribute map of the target object (table, cell, ...).
     *
     * @param {Object} defaultBorder
     *  Default border style style used when making a border attribute visible
     *  without having a corresponding visible border value in the passed
     *  original border attributes.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.paragraph=false]
     *      If set to true, a single inner border attribute 'borderInside' will
     *      be used instead of the two attributes 'borderInsideHor' and
     *      'borderInsideVert' (for paragraphs which are stacked vertically and
     *      do not have vertical inner borders).
     *
     * @returns {Object}
     *  The table/cell attributes containing all border attributes that are
     *  mentioned in the passed border mode. In case the passed border mode is
     *  a string, the returned attributes will contain the corresponding border
     *  attribute only.
     */
    MixedBorder.getBorderAttributes = function (borderMode, origAttributes, defaultBorder, options) {

        var // the border attribute names
            attributeNames = getBorderAttributeNames(options),
            // the resulting border attributes
            borderAttributes = {};

        function addBorderAttribute(visible, propName) {

            // ignore invalid properties
            if (!(propName in attributeNames)) { return; }

            var // the real name of the border attribute
                borderAttrName = attributeNames[propName],
                // the original border attribute value or mixed border object
                origBorder = origAttributes[borderAttrName],
                // whether the original border is visible
                origVisible = MixedBorder.isVisibleBorder(origBorder);

            // toggle visibility if specified (but switch from 'mixed' state to visible)
            visible = _.isNull(visible) ? ((origVisible && origBorder.mixed) || !origVisible) : visible;

            // build the border attribute value according to visibility and original borders
            borderAttributes[borderAttrName] = _.copy(visible ? (origVisible ? origBorder : defaultBorder) : Border.NONE, true);
            delete borderAttributes[borderAttrName].mixed;
        }

        if (_.isObject(borderMode)) {
            // fill the resulting attribute map
            _.each(borderMode, addBorderAttribute);
        } else if (_.isString(borderMode)) {
            // toggle the visibility of a single border
            addBorderAttribute(null, borderMode);
        }

        return borderAttributes;
    };

    // exports ================================================================

    return MixedBorder;

});
