/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Oliver Specht <oliver.specht@open-xchange.com>
 */

define('io.ox/office/editframework/model/themecollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/model/modelobject'
], function (Utils, ModelObject) {

    'use strict';

    var // the attributes of the built-in default theme
        BUILTIN_THEME_ATTRS = {
            colorScheme: {
                accent1: '4f81bd',
                accent2: 'c0504d',
                accent3: '9bbb59',
                accent4: '8064a2',
                accent5: '4bacc6',
                accent6: 'f79646',
                text1: '000000',
                text2: '1f497d',
                light1: 'ffffff',
                light2: 'eeece1',
                dark1: '000000',
                dark2: '1f497d',
                background1: 'ffffff',
                background2: 'eeece1',
                hyperlink: '0000ff',
                followedHyperlink: '800080'
            }
        };

    // class Theme ============================================================

    /**
     * Contains the definitions of a single theme in a document.
     *
     * @constructor
     *
     * @param {Object} attributes
     *  The theme attributes as received from the 'insertTheme' operation.
     */
    function Theme(attributes) {

        var // the color scheme of this theme
            colorScheme = _.copy(Utils.getObjectOption(attributes, 'colorScheme', {}), true);

        // methods ------------------------------------------------------------

        /**
         * Returns whether this theme contains scheme color definitions.
         */
        this.hasSchemeColors = function () {
            return !_.isEmpty(colorScheme);
        };

        /**
         * Returns the RGB color value of the specified scheme color.
         *
         * @param {String} name
         *  The name of the scheme color.
         *
         * @returns {String|Null}
         *  The RGB value of the scheme color as hexadecimal string, if
         *  existing, otherwise null.
         */
        this.getSchemeColor = function (name) {
            var color = colorScheme[name];
            return (typeof color === 'string') ? color : null;
        };

    } // class Theme

    // class ThemeCollection ==================================================

    /**
     * Contains the definitions of all themes in a document.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {EditModel} docModel
     *  The document model containing this instance.
     */
    function ThemeCollection(docModel) {

        var // themes, mapped by identifier
            themes = {},

            // default theme (first inserted theme)
            defaultTheme = themes.Larissa = new Theme(BUILTIN_THEME_ATTRS),

            // whether the default theme is still the built-in theme
            isBuiltInTheme = true;

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // public methods -----------------------------------------------------

        /**
         * Adds a new theme to this container. An existing theme with the
         * specified name will be replaced.
         *
         * @param {String} name
         *  The name of of the new theme.
         *
         * @param {Object} attributes
         *  The formatting attributes of the theme.
         *
         * @returns {ThemeCollection}
         *  A reference to this instance.
         */
        this.insertTheme = function (name, attributes) {

            var // create a theme object with all passed attributes
                theme = themes[name] = new Theme(attributes);

            // set first inserted theme as new default theme
            if (isBuiltInTheme) {
                defaultTheme = theme;
                isBuiltInTheme = false;
            }

            // notify listeners
            this.trigger('insert:theme', name);

            return this;
        };

        /**
         * Gives access to a single theme.
         *
         * @param {String} [name]
         *  The name of the theme to be returned. If omitted or not existing,
         *  the 'current' theme will be returned.
         *
         * @returns {Theme}
         *  The specified theme, or the current theme. If no theme has been
         *  inserted, returns the built-in default theme.
         */
        this.getTheme = function (name) {
            return (name in themes) ? themes[name] : defaultTheme;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = themes = defaultTheme = null;
        });

    } // class ThemeCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: ThemeCollection });

});
