/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/operationsgenerator', [
    'io.ox/office/editframework/utils/operations'
], function (Operations) {

    'use strict';

    // class OperationsGenerator ==============================================

    /**
     * An instance of this class contains an array of operations and provides
     * methods to generate new operations.
     *
     * @constructor
     *
     * @param {EditModel} docModel
     *  The document model containing this generator.
     */
    function OperationsGenerator(docModel) {

        var // the operations buffer
            operations = [];

        // methods ------------------------------------------------------------

        /**
         * Returns the array with all operations that have been generated so
         * far. Does not remove the operations from this generator instance.
         *
         * @returns {Array}
         *  The array with all generated operations.
         */
        this.getOperations = function () {
            return operations;
        };

        /**
         * Returns the last generated operation, but does not remove that
         * operation from this generator instance.
         *
         * @returns {Object|Null}
         *  The last generated operation, if available; otherwise null.
         */
        this.getLastOperation = function () {
            return _.last(operations) || null;
        };

        /**
         * Reverses the entire operations array.
         *
         * @returns {OperationsGenerator}
         *  A reference to this instance.
         */
        this.reverseOperations = function () {
            operations.reverse();
            return this;
        };

        /**
         * Appends the passed JSON operations to the operations array.
         *
         * @param {Object|Array<Object>|OperationsGenerator} operations
         *  A single JSON operation, or an array of JSON operations, or an
         *  operations generator instance with operations to be appended to the
         *  operations array. Note that an operations generator passed to this
         *  method will NOT be cleared after its operations have been appended.
         *
         * @returns {OperationsGenerator}
         *  A reference to this instance.
         */
        this.appendOperations = function (ops) {
            if (ops instanceof OperationsGenerator) {
                operations = operations.concat(ops.getOperations());
            } else if (_.isArray(ops)) {
                operations = operations.concat(ops);
            } else {
                operations.push(ops);
            }
            return this;
        };

        /**
         * Creates and appends a new operation to the operations array.
         *
         * @param {String} name
         *  The name of the operation.
         *
         * @param {Object} [properties]
         *  Additional properties that will be stored in the JSON operation.
         *
         * @returns {Object}
         *  The created operation object.
         */
        this.generateOperation = function (name, properties) {
            var operation = _.extend({ name: name }, properties);
            operations.push(operation);
            return operation;
        };

        /**
         * Generates an 'insertStyleSheet' operation, if the specified style
         * sheet exists in the style sheet collection, and is marked dirty.
         *
         * @param {String} styleFamily
         *  The style family that specifies which style sheet collection will
         *  be checked for a dirty style sheet.
         *
         * @param {String} styleId
         *  The identifier of the style sheet.
         *
         * @returns {OperationsGenerator}
         *  A reference to this instance.
         */
        this.generateMissingStyleSheetOperation = function (styleFamily, styleId) {

            var // the style sheet container
                styleSheets = docModel.getStyleCollection(styleFamily),
                // additional properties for the operation
                operationOptions = null;

            // do not generate an operation for the default style
            if (styleSheets && (styleId !== styleSheets.getDefaultStyleId()) && styleSheets.isDirty(styleId)) {

                // create the properties for the 'insertStyleSheet' operation
                operationOptions = {
                    attrs: styleSheets.getStyleSheetAttributeMap(styleId),
                    type: styleFamily,
                    styleId: styleId,
                    styleName: styleSheets.getName(styleId),
                    parent: styleSheets.getParentId(styleId),
                    uiPriority: styleSheets.getUIPriority(styleId),
                    hidden: styleSheets.isHidden(styleId),
                    custom: styleSheets.isCustom(styleId)
                };
                if (!_.isString(operationOptions.parent)) { delete operationOptions.parent; }
                if (!_.isNumber(operationOptions.uiPriority)) { delete operationOptions.uiPriority; }

                // generate and store the 'insertStyleSheet' operation
                this.generateOperation(Operations.INSERT_STYLESHEET, operationOptions);

                // remove the dirty flag from the style sheet
                styleSheets.setDirty(styleId, false);
            }

            return this;
        };

    } // class OperationsGenerator

    // exports ================================================================

    return _.makeExtendable(OperationsGenerator);

});
