/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/attributedmodel', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/baseframework/model/modelobject'
], function (Utils, AttributeUtils, ModelObject) {

    'use strict';

    // class AttributedModel ==================================================

    /**
     * An abstract model object containing formatting attributes and the
     * reference to a style sheet of a specific attribute family.
     *
     * Triggers the following events:
     * - 'change:attributes'
     *      After the explicit attributes or the style sheet identifier have
     *      been changed, either directly by using the method
     *      AttributedModel.setAttributes(), or indirectly after the style
     *      sheet has been changed. The event handlers receive the following
     *      parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Object} newAttributes
     *          The current (new) merged attribute set.
     *      (3) {Object} oldAttributes
     *          The previous (old) merged attribute set.
     * No events will be generated while the application imports the document.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {EditModel} docModel
     *  The document model containing this attributed model object.
     *
     * @param {Object} [initAttributes]
     *  An attribute set with initial formatting attributes for the attributed
     *  model object.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all attributes also supported by the base
     *  class ModelObject, and the following additional options:
     *  @param {String} [initOptions.styleFamily]
     *      The attribute family of style sheets that can be referred by this
     *      attributed model object. If omitted, this object supports explicit
     *      attributes only. All attribute families that are supported by these
     *      style sheets will also be supported by this model object.
     *  @param {String} [initOptions.families]
     *      The attribute families of explicit attributes supported by this
     *      attributed model object (space-separated). The style family
     *      specified in the 'styleFamily' option, and all additional attribute
     *      families supported by the style sheet container (as specified in
     *      the constructor of the respective container) do not need to be
     *      repeated here.
     *  @param {AttributedModel} [initOptions.parentModel]
     *      A parent model object (instance of AttributedModel) that supports
     *      all or a few of the attribute families supported by this instance.
     *      The merged attributes of that parent model will be added to the own
     *      merged attribute set before adding the own style attributes and
     *      explicit attributes. This works also recursively (the parent model
     *      may define its own parent model). Attribute changes in the parent
     *      model will be forwarded to this instance (unless the option
     *      'listenToParent' has been set to false, see below).
     *  @param {Boolean} [initOptions.listenToParent=true]
     *      If set to false, this instance will not listen to change events of
     *      the parent model specified with the option 'parentModel'. This may
     *      be desired for performance reasons to reduce the number of event
     *      listeners in collections with many instances of this class.
     *  @param {Boolean} [initOptions.autoClear=false]
     *      If set to true, explicit attributes that are equal to the default
     *      attribute values (as specified in the attribute definitions of the
     *      respective attribute families), will not be inserted into the
     *      explicit attribute set, but the old explicit attribute will be
     *      removed from the explicit attribute set, when using the method
     *      AttributedModel.setAttributes(). This option has no effect, if this
     *      model instance uses style sheets (see option 'styleFamily'), or
     *      refers to a parent model object (see option 'parentModel').
     */
    function AttributedModel(docModel, initAttributes, initOptions) {

        var // self reference
            self = this,

            // the attribute family of the style sheets
            styleFamily = Utils.getStringOption(initOptions, 'styleFamily', null, true),

            // the parent model used to build the merged attribute set
            parentModel = Utils.getObjectOption(initOptions, 'parentModel'),

            // whether to remove explicit attributes equal to the attribute defaults
            autoClear = !styleFamily && !parentModel && Utils.getBooleanOption(initOptions, 'autoClear', false),

            // the collection of style sheets for the style family
            styleCollection = null,

            // all attribute families supported by instances of this class
            supportedFamilies = null,

            // the current auto style
            autoStyleId = null,

            // the (incomplete) explicit attributes of this model object
            explicitAttributeSet = {},

            // the (complete) cached merged attributes of this model object
            mergedAttributeSet = null;

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Returns the current effective style sheet identifier, if style
         * sheets are supported.
         */
        function getEffectiveStyleId() {
            return styleCollection ? styleCollection.getEffectiveStyleId(Utils.getStringOption(explicitAttributeSet, 'styleId', ''), true) : null;
        }

        /**
         * Adds or removes an attribute value in the specified attribute map.
         *
         * @param {Object} attributes
         *  A map with formatting attributes to be updated.
         *
         * @param {String} name
         *  The name of the formatting attribute to be changed.
         *
         * @param {Any} value
         *  The new value of the formatting attribute. If set to null, the
         *  attribute will be removed from the map.
         *
         * @param {Object} [definition]
         *  The definition descriptor of the attribute, if available. Used to
         *  check the new attribute value against the default value, if the
         *  option 'autoClear' has been passed to the constructor.
         */
        function updateAttributeValue(attributes, name, value, definition) {

            var // whether this attribute has changed
                attrChanged = false;

            if (_.isNull(value) || (autoClear && _.isObject(definition) && _.isEqual(value, definition.def))) {
                if (name in attributes) {
                    delete attributes[name];
                    attrChanged = true;
                }
            } else {
                if (!_.isEqual(attributes[name], value)) {
                    attributes[name] = _.copy(value, true);
                    attrChanged = true;
                }
            }

            // delete child attributes according to current attribute value
            if (attrChanged && autoClear && _.isObject(definition) && _.isArray(definition.childAttrs)) {
                if (_.isNull(value)) { value = definition.def; }
                _.each(definition.childAttrs, function (childAttr) {

                    var // the expected value of the parent attribute
                        expectedValue = childAttr.value,
                        // whether the current value of the primary attribute matches the expected value
                        valueMatches = _.isFunction(expectedValue) ? expectedValue.call(self, value) : _.isEqual(value, expectedValue);

                    // delete secondary attribute if expected value does not match
                    if (!valueMatches) { delete attributes[childAttr.name]; }
                });
            }

            return attrChanged;
        }

        /**
         * Recalculates the merged attribute set.
         */
        function calculateMergedAttributeSet(notify) {

            var // remember the old merged attributes
                oldMergedAttributeSet = mergedAttributeSet,
                // the identifier of the style sheet
                styleId = getEffectiveStyleId();

            // start with defaults and style sheet attributes
            mergedAttributeSet = docModel.getDefaultAttributes(supportedFamilies);

            // add merged attributes of parent model object
            if (parentModel) {
                docModel.extendAttributes(mergedAttributeSet, parentModel.getMergedAttributes());
                _.chain(mergedAttributeSet).keys().difference(supportedFamilies).each(function (family) {
                    delete mergedAttributeSet[family];
                });
            }

            // additional processing, if style sheet support is enabled
            if (styleCollection) {

                // set default auto-style identifier if existing
                if (!autoStyleId && styleCollection.isAuto(styleId)) {
                    autoStyleId = styleId;
                }

                // add auto style attributes to resulting merged attribute set
                if (_.isString(autoStyleId)) {
                    docModel.extendAttributes(mergedAttributeSet, styleCollection.getStyleAttributeSet(autoStyleId));
                    styleId = mergedAttributeSet.styleId;
                }

                // add style sheet attributes to resulting merged attribute set
                if (!autoStyleId || ('styleId' in explicitAttributeSet)) {
                    docModel.extendAttributes(mergedAttributeSet, styleCollection.getStyleAttributeSet(explicitAttributeSet.styleId || ''));
                    styleId = mergedAttributeSet.styleId;
                }
            }

            // add the passed explicit attributes
            docModel.extendAttributes(mergedAttributeSet, explicitAttributeSet);

            // restore the effective style sheet identifier
            if (styleCollection) {
                mergedAttributeSet.styleId = styleId;
            }

            // notify all listeners
            if (notify) {
                self.trigger('change:attributes', mergedAttributeSet, oldMergedAttributeSet);
            }
        }

        // protected methods --------------------------------------------------

        /**
         * Clones all contents from the passed attributed model into this model
         * instance.
         *
         * @internal
         *  Used during clone construction. DO NOT CALL from external code!
         *
         * @param {AttributedModel} sourceModel
         *  The source model instance whose attributes will be cloned into this
         *  model instance.
         *
         * @returns {AttributedModel}
         *  A reference to this instance.
         */
        this.cloneFrom = function (sourceModel) {

            // copy the auto style identifier of the source model
            autoStyleId = sourceModel.getAutoStyleId();

            // deep copy of all explicit attributes
            explicitAttributeSet = sourceModel.getExplicitAttributes();

            // recalculate the merged attributes (without notifications)
            calculateMergedAttributeSet(false);

            return this;
        };

        // public methods -----------------------------------------------------

        /**
         * Returns whether this model object contains a reference to an auto
         * style.
         *
         * @returns {Boolean}
         *  Whether this model object contains a reference to an auto style.
         */
        this.hasAutoStyle = function () {
            return _.isString(autoStyleId);
        };

        /**
         * Returns whether this model object contains any explicit attributes,
         * or a reference to a style sheet.
         *
         * @returns {Boolean}
         *  Whether this model object contains any explicit attributes, or a
         *  reference to a style sheet.
         */
        this.hasExplicitAttributes = function () {
            return !_.isEmpty(explicitAttributeSet);
        };

        /**
         * Returns whether this model object contains any explicit attributes,
         * a reference to a style sheet, or a reference to an auto style.
         *
         * @returns {Boolean}
         *  Whether this model object contains any explicit attributes, a
         *  reference to a style sheet, or a reference to an auto style.
         */
        this.isFormatted = function () {
            return this.hasAutoStyle() || this.hasExplicitAttributes();
        };

        /**
         * Returns the identifier of the auto style referred by this model.
         *
         * @returns {String|Null}
         *  The identifier of the auto style referred by this model, if
         *  available; otherwise null.
         */
        this.getAutoStyleId = function () {
            return autoStyleId;
        };

        /**
         * Returns the explicit attribute set of this model object.
         *
         * @param {Boolean} [direct=false]
         *  If set to true, the returned attribute set will be a reference to
         *  the original map stored in this instance, which MUST NOT be
         *  modified! By default, a deep clone of the attribute set will be
         *  returned that can be freely modified.
         *
         * @returns {Object}
         *  An (incomplete) attribute set (a map of attribute maps with
         *  name/value pairs, keyed by the attribute families, optionally
         *  containing the style sheet reference in the 'styleId' property),
         *  containing values for all attributes that have been explicitly set.
         */
        this.getExplicitAttributes = function (direct) {
            return (direct === true) ? explicitAttributeSet : _.copy(explicitAttributeSet, true);
        };

        /**
         * Returns the merged attribute set of this model object.
         *
         * @returns {Object}
         *  A complete attribute set (a map of attribute maps with name/value
         *  pairs, keyed by the attribute families, additionally containing the
         *  style sheet reference in the 'styleId' property), containing values
         *  for all known attributes, collected from the document default
         *  values, the style sheet currently referenced by this model object,
         *  and the explicit attributes of this model object. For efficiency,
         *  a reference to the cached attribute set will be returned. This
         *  object MUST NOT be changed!
         */
        this.getMergedAttributes = function () {
            return mergedAttributeSet;
        };

        /**
         * Returns whether this attributed model contains the same auto style
         * identifier and explicit attributes as the passed attributed model.
         *
         * @param {AttributedModel} attribModel
         *  The other attributed model to be compared to this model.
         *
         * @returns {Boolean}
         *  Whether both attributed models contain the same auto style
         *  identifier and explicit formatting attributes.
         */
        this.hasEqualAttributes = function (attribModel) {
            return (this === attribModel) ||
                ((autoStyleId === attribModel.getAutoStyleId()) && _.isEqual(explicitAttributeSet, attribModel.getExplicitAttributes(true)));
        };

        /**
         * Changes and/or removes specific explicit formatting attributes, or
         * the style sheet reference of this model object.
         *
         * @param {Object} attributeSet
         *  An (incomplete) attribute set (a map of attribute maps with
         *  name/value pairs, keyed by the attribute families, optionally
         *  containing the style sheet reference in the 'styleId' property)
         *  with all formatting attributes to be changed. To clear an explicit
         *  attribute value (thus defaulting to the current style sheet), the
         *  respective value in this attribute set has to be set explicitly to
         *  the null value. Attributes missing in this attribute set will not
         *  be modified.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.notify='auto']
         *      Specifies whether to trigger the event 'change:attributes'. If
         *      set to 'auto' or omitted, the event will be triggered if at
         *      least one formatting attribute has changed its value. If set to
         *      'always', the event will always be triggered without comparing
         *      the old and new attribute values. If set to 'never', no event
         *      will be triggered.
         *  @param {Boolean} [options.special=false]
         *      If set to true, 'special' attributes (marked with the 'special'
         *      flag in the attribute definitions denoting an attribute for
         *      internal use only, not for usage in document operations) can be
         *      changed too. Otherwise, special attributes will not be
         *      recognized by this method.
         *
         * @returns {Boolean}
         *  Whether the attributes of this model object have actually changed.
         */
        this.setAttributes = function (attributeSet, options) {

            var // whether to trigger the event
                notify = Utils.getStringOption(options, 'notify', 'auto'),
                // whether any explicit attribute or the style identifier has been changed
                changed = false;

            // set new style sheet or auto style
            if ('styleId' in attributeSet) {

                // style collection must exist (this model must support style sheets at all)
                if (!styleCollection) {
                    Utils.error('AttributedModel.setAttributes(): setting style sheets not supported');
                    return false;
                }

                var // the style sheet identifier from the passed attributes
                    styleId = attributeSet.styleId;

                // remove explicit reference to default style sheet (missing style
                // reference will automatically fall-back to the default style sheet)
                if (!_.isString(styleId) || (styleId === '') || (styleId === styleCollection.getDefaultStyleId())) {
                    styleId = null;
                }

                // special handling for auto styles
                if (_.isString(styleId) && styleCollection.isAuto(styleId)) {

                    // store the new auto style identifier in a special attribute
                    if (autoStyleId !== styleId) {
                        autoStyleId = styleId;
                        changed = true;
                    }

                    // remove explicit reference to a style sheet (will be resolved via auto style)
                    if ('styleId' in explicitAttributeSet) {
                        delete explicitAttributeSet.styleId;
                        changed = true;
                    }

                    // remove all explicit attributes supported by style sheets
                    styleCollection.getSupportedFamilies().forEach(function (family) {
                        if (family in explicitAttributeSet) {
                            delete explicitAttributeSet[family];
                            changed = true;
                        }
                    });

                } else {

                    // change the style sheet identifier in the internal map, update the resulting changed flag
                    if (updateAttributeValue(explicitAttributeSet, 'styleId', styleId)) {
                        changed = true;
                    }
                }
            }

            // update the remaining explicit attributes of all supported attribute families
            supportedFamilies.forEach(function (family) {

                // check if new attributes exist in the passed attribute set
                if (!_.isObject(attributeSet[family])) { return; }

                var // attribute definitions of the current family
                    definitions = docModel.getAttributeDefinitions(family),
                    // target attribute map for the current family
                    explicitAttributes = explicitAttributeSet[family] || (explicitAttributeSet[family] = {});

                // update the attribute map with the new attribute values
                _.each(attributeSet[family], function (value, name) {
                    if (AttributeUtils.isRegisteredAttribute(definitions, name, options)) {
                        if (updateAttributeValue(explicitAttributes, name, value, definitions[name])) {
                            changed = true;
                        }
                    }
                });

                // remove empty attribute value maps completely
                if (_.isEmpty(explicitAttributes)) {
                    delete explicitAttributeSet[family];
                }
            });

            // recalculate merged attributes and notify all listeners
            if (changed || !mergedAttributeSet) {
                calculateMergedAttributeSet((changed && (notify === 'auto')) || (notify === 'always'));
            }

            return changed;
        };

        /**
         * Updates the merged attribute set, according to the current defaults,
         * the style sheet and parent model referenced by this instance, and
         * its explicit attributes.
         *
         * @returns {AttributedModel}
         *  A reference to this instance.
         */
        this.refreshMergedAttributeSet = function () {
            calculateMergedAttributeSet(false);
            return this;
        };

        // initialization -----------------------------------------------------

        // initialize class members
        styleCollection = styleFamily ? docModel.getStyleCollection(styleFamily) : null;

        // get all supported attribute families
        supportedFamilies = Utils.getTokenListOption(initOptions, 'families', []);
        if (styleCollection) { supportedFamilies = supportedFamilies.concat(styleCollection.getSupportedFamilies()); }
        supportedFamilies = _.unique(supportedFamilies);

        // set the attributes passed to the constructor (filters by supported attributes)
        this.setAttributes(_.isObject(initAttributes) ? initAttributes : {}, { notify: 'never' });
        initAttributes = null;

        // listen to changes in the parent object, update the merged attributes and notify own listeners
        if (parentModel && Utils.getBooleanOption(initOptions, 'listenToParent', true)) {
            this.listenTo(parentModel, 'change:attributes', function () {
                calculateMergedAttributeSet(true);
            });
        }

        // listen to style sheet changes, update the merged attributes and notify own listeners
        if (styleCollection) {
            this.listenTo(styleCollection, 'triggered', function () {
                calculateMergedAttributeSet(true);
            });
        }

        // update merged attributes silently once after import (object may be destroyed before import finishes)
        this.waitForImportSuccess(function (alreadyFinished) {
            if (!alreadyFinished) { calculateMergedAttributeSet(false); }
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docModel = parentModel = styleCollection = null;
            explicitAttributeSet = mergedAttributeSet = null;
        });

    } // class AttributedModel

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: AttributedModel });

});
