/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/app/basecontroller', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/dialogs',
    'io.ox/office/tk/utils/driveutils',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/baseframework/app/appobjectmixin'
], function (Utils, KeyCodes, Dialogs, DriveUtils, BaseObject, TriggerObject, TimerMixin, AppObjectMixin) {

    'use strict';

    // class ItemState ========================================================

    /**
     * A descriptor for the complete state of a controller item.
     *
     * @constructor
     *
     * @property {String} key
     *  The key of the controller item.
     *
     * @property {Boolean} enabled
     *  The enabled state of the controller item.
     *
     * @property {Any} value
     *  The current value (the result of the getter callback) of the controller
     *  item.
     *
     * @property {Boolean} dirty
     *  Whether the item has been set to dirty state. A dirty item will always
     *  be updated, regardless if it has been changed since the last update.
     *
     * @property {Array<ItemState>} parentStates
     *  The item states of all parent items of the controller item, in order of
     *  the parent keys registered for the item. If the item does not contain
     *  any parent items, this property will be an empty array.
     *
     * @property {Array<Any>} parentValues
     *  The current values (results of all getter callbacks) of the parent
     *  items, in order of the parent keys registered for the item. This
     *  property is provided for convenience (the parent values are contained
     *  in the property 'parentStates' of this instance).
     *
     * @property {Boolean} parentsEnabled
     *  The accumulated enabled state of the parent items of the controller
     *  item. Will be true, if all parent items are enabled.
     */
    function ItemState(key, parentStates) {

        this.key = key;
        this.enabled = false;
        this.value = undefined;
        this.dirty = false;

        // parent states and values
        this.parentStates = parentStates;
        this.parentValues = _.pluck(parentStates, 'value');

        // all parents must be enabled for 'parentsEnabled' being true
        this.parentsEnabled = parentStates.every(function (state) { return state.enabled; });

    } // class ItemState

    // class BaseController ===================================================

    /**
     * A controller contains a collection of items, consisting of unique key
     * and value, and providing arbitrary getter and setter methods for their
     * values.
     *
     * Triggers the following events:
     * - 'change:items'
     *      After the current state (value and/or enabled state) of at least
     *      one controller item has been changed, after the controller has
     *      updated itself or the method BaseController.update() has been
     *      invoked. Event listeners receive the descriptors of the changed
     *      items in a map with item identifiers as keys, and instances of the
     *      class ItemState as values.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     * @extends AppObjectMixin
     *
     * @param {BaseApplication} app
     *  The application that has created this controller instance.
     *
     * @param {BaseModel} docModel
     *  The document model created by the passed application.
     *
     * @param {BaseView} docView
     *  The document view created by the passed application.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {Number} [initOptions.updateDelay=50]
     *      The delay for the debounced Controller.update() method, in
     *      milliseconds. Must not be less than 50.
     */
    function BaseController(app, docModel, docView, initOptions) {

        var // self reference
            self = this,

            // all registered controller items, mapped by item key
            items = {},

            // dummy item, used e.g. for undefined parent items
            dummyItem = null,

            // the delay time for the debounced BaseController.update() method
            // Bug 30479: must NOT be 0, to be able give some space for deferred
            // focus handling in form controls, for example in text fields which
            // want to commit their value before the controller will reset it.
            updateDelay = Utils.getIntegerOption(initOptions, 'updateDelay', 50, 50),

            // cached item states, mapped by item keys
            stateCache = {},

            // cached states of all controller items as notified the last time
            notifiedStateCache = {},

            // number of item setters currently running (recursion counter)
            runningSetters = 0,

            // shortcut definitions for 'keydown', mapped by key code (for performance)
            keyShortcuts = {},

            // shortcut definitions for 'keypress', mapped by char code (for performance)
            charShortcuts = {};

        // class Item ---------------------------------------------------------

        var Item = BaseObject.extend({ constructor: function (key, definition) {

            var // parent items whose values/states are needed to resolve the own value/state
                parentKeys = Utils.getOption(definition, 'parent'),
                // handler for enabled state (default to function that returns true)
                enableHandler = Utils.getFunctionOption(definition, 'enable', _.constant(true)),
                // handler for value getter (default to identity to forward first parent value)
                getHandler = Utils.getFunctionOption(definition, 'get', _.identity),
                // handler for value setter
                setHandler = Utils.getFunctionOption(definition, 'set'),
                // whether an asynchronous setter is cancelable
                cancelable = Utils.getBooleanOption(definition, 'cancel', false),
                // behavior for returning browser focus to application
                itemPreserveFocus = Utils.getBooleanOption(definition, 'preserveFocus', false),
                // custom focus target node or callback handler
                itemFocusTarget = Utils.getOption(definition, 'focusTarget');

            BaseObject.call(this);

            // private methods  - - - - - - - - - - - - - - - - - - - - - - - -

            /**
             * Returns the state descriptor of the specified parent item.
             */
            function getParentState(parentKey) {
                var parentItem = items[parentKey] || dummyItem;
                if (parentItem === dummyItem) {
                    Utils.warn('BaseController.Item.getParentState(): item "' + key + '" refers to unknown parent "' + parentKey + '"');
                }
                return parentItem.getState();
            }

            // methods  - - - - - - - - - - - - - - - - - - - - - - - - - - - -

            /**
             * Returns the unique key of this item.
             *
             * @returns {String}
             *  The unique key this item has been registered with.
             */
            this.getKey = function () {
                return key;
            };

            /**
             * Returns a descriptor for the current state of this item.
             *
             * @returns {ItemState}
             *  The state descriptor for this controller item.
             */
            this.getState = function () {

                // return cached state
                if (key in stateCache) { return stateCache[key]; }

                // create the state descriptor of this item
                var itemState = new ItemState(key, _.map(parentKeys, getParentState));

                // insert the state descriptor into the cache before resolving enabled state and value
                stateCache[key] = itemState;

                // resolve and store the value first (enable callback may want to query the own value)
                itemState.value = getHandler.apply(this, itemState.parentValues);

                // resolve and store the enabled state
                itemState.enabled = itemState.parentsEnabled && (enableHandler.apply(this, itemState.parentValues) === true);

                // copy dirty flag from previous state cache
                var oldItemState = notifiedStateCache[key];
                itemState.dirty = !oldItemState || oldItemState.dirty || itemState.parentStates.some(function (parent) { return parent.dirty; });

                // return the state descriptor
                return itemState;
            };

            /**
             * Returns whether all parent items are enabled.
             *
             * @returns {Boolean}
             *  False, if at least one parent item is disabled, otherwise true.
             */
            this.areParentsEnabled = function () {
                return this.getState().parentsEnabled;
            };

            /**
             * Returns whether this item is effectively enabled.
             *
             * @returns {Boolean}
             *  False, if at least one parent item is disabled, or if the
             *  enable handler of this item has returned false; otherwise true.
             */
            this.isEnabled = function () {
                return this.getState().enabled;
            };

            /**
             * Returns the current value of the specified parent item.
             *
             * @param {Number} [index=0]
             *  The index of the parent item (if multiple parent items have
             *  been specified for this item).
             */
            this.getParentValue = function (index) {
                var parentState = this.getState().parentStates[_.isNumber(index) ? index : 0];
                return parentState ? parentState.value : undefined;
            };

            /**
             * Returns the current value of this item.
             */
            this.getValue = function () {
                return this.getState().value;
            };

            /**
             * Executes the setter function of this item (passing in the new
             * value), and moves the browser focus back to the application
             * pane.
             *
             * @param {Any} value
             *  The new value of the item.
             *
             * @param {Object} [options]
             *  Optional parameters. See method BaseController.executeItem()
             *  for details.
             *
             * @returns {jQuery.Promise}
             *  A promise that will be resolved or rejected according to the
             *  result of the item setter function. If the item is disabled, a
             *  rejected promise will be returned. If the item setter returns a
             *  promise, that promise will be returned. Otherwise, the return
             *  value of the item setter function will be wrapped and returned
             *  in a resolved promise.
             */
            this.execute = function (value, options) {

                var // the current state and value of the item
                    state = this.getState(),
                    // the resulting promise
                    promise = null,
                    // whether the setter runs asynchronous code
                    pending = false,
                    // focus options after executing the item
                    focusOptions = _.extend({}, options);

                // final processing after the set handler has finished
                function finalize() {

                    // bug 40673: mark the item in the state cache as dirty, to force updating
                    // the item with the effective value as returned then by its getter
                    var oldState = notifiedStateCache[key];
                    if (oldState) { oldState.dirty = true; }

                    // update the controller items
                    self.update();
                }

                // mix item focus options and passed focus options
                focusOptions.preserveFocus = itemPreserveFocus || Utils.getBooleanOption(options, 'preserveFocus', false);
                focusOptions.focusTarget = Utils.getOption(options, 'focusTarget', itemFocusTarget);

                // bind focus target callback function to the item instance
                if (_.isFunction(focusOptions.focusTarget)) {
                    focusOptions.focusTarget = _.bind(focusOptions.focusTarget, this);
                }

                // do nothing if item is disabled
                if (state.enabled) {

                    // execute the set handler
                    runningSetters += 1;
                    if (_.isFunction(setHandler)) {
                        promise = setHandler.call(this, value);
                    }

                    // convert result of the setter callback to a promise
                    promise = $.when(promise);
                    promise.always(function () {
                        runningSetters -= 1;
                    });

                    // show busy screen if setter is running asynchronously
                    // (but not for modal dialogs which have their own blocker overlay)
                    pending = promise.state() === 'pending';
                    if (pending && !Dialogs.isModalDialogOpen()) {
                        var showCancel = cancelable && _.isFunction(promise.abort);
                        docView.enterBusy({ cancelHandler: showCancel ? function () { promise.abort(); } : null });
                    }
                } else {
                    // item is disabled
                    promise = $.Deferred().reject();
                }

                // focus back to application
                if (!focusOptions.preserveFocus) {
                    this.waitForAny(promise, function () {
                        if (pending) {
                            // execute in a timeout, needed for dialogs which are closed *after* resolve/reject
                            self.executeDelayed(function () { self.grabFocus(focusOptions); }, undefined, 'BaseController: grabFocus');
                        } else {
                            self.grabFocus(focusOptions);
                        }
                    });
                }

                // post-processing after the setter is finished
                Utils.logSelenium('time=' + _.now(), 'key=' + key, 'state=' + promise.state());
                if (pending) {
                    this.waitForAny(promise, function () {
                        docView.leaveBusy();
                        finalize();
                        Utils.logSelenium('time=' + _.now(), 'key=' + key, 'state=' + promise.state());
                    });
                } else {
                    finalize();
                }

                return promise;
            };

            // initialization - - - - - - - - - - - - - - - - - - - - - - - - -

            // always use an array of parent keys (as strings)
            parentKeys =
                _.isArray(parentKeys) ? _.filter(parentKeys, _.isString) :
                _.isString(parentKeys) ? [parentKeys] :
                null;

            // destroy all class members on destruction
            this.registerDestructor(function () {
                definition = parentKeys = null;
                enableHandler = getHandler = setHandler = null;
            });

        } }); // class Item

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);
        AppObjectMixin.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Handles 'keydown' and 'keypress' events and calls the setter of this
         * item, if it contains a matching keyboard shortcut definition.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object. If a matching shortcut definition has been
         *  found, propagation of the event will be stopped, and the browser
         *  default action will be suppressed.
         */
        function keyHandler(event) {

            var // whether to stop propagation and prevent the default action
                stopPropagation = false;

            // executes the item setter defined in the passed shortcut
            function executeItemForShortcut(shortcut) {

                var // the definition data of the shortcut
                    definition = shortcut.definition,
                    // the current value to be passed to the item setter
                    value = null;

                // check whether to actually process the key event according to the event target node
                if (_.isString(definition.selector) && ($(event.target).closest(definition.selector, app.getRootNode()[0]).length === 0)) {
                    return;
                }

                // bug 30096: do not allow to execute items while a blocking operation is still running
                if (runningSetters === 0) {
                    // call value resolver function, or take constant value
                    value = _.isFunction(definition.value) ? definition.value(self.getItemValue(shortcut.key)) : definition.value;
                    self.executeItem(shortcut.key, value, { sourceType: 'keyboard' });
                }

                // stop event propagation unless specified otherwise
                if (!Utils.getBooleanOption(definition, 'propagate', false)) {
                    stopPropagation = true;
                }
            }

            switch (event.type) {
                case 'keydown':
                    // process all shortcut definitions for the key code in the passed event
                    if (event.keyCode in keyShortcuts) {
                        _.each(keyShortcuts[event.keyCode], function (shortcut) {
                            // check if the additional modifier keys match the shortcut definition
                            if (KeyCodes.matchModifierKeys(event, shortcut.definition)) {
                                executeItemForShortcut(shortcut);
                            }
                        });
                    }
                    break;
                case 'keypress':
                    // process all shortcut definitions for the char code in the passed
                    // event, but ignore 'keypress' events in text fields
                    if ((event.charCode in charShortcuts) && !$(event.target).is('input,textarea')) {
                        _.each(charShortcuts[event.charCode], executeItemForShortcut);
                    }
                    break;
            }

            return stopPropagation ? false : undefined;
        }

        /**
         * Initializes the key handlers for all registered keyboard shortcuts.
         */
        function initializeKeyHandler() {

            var // the root node of the application window
                rootNode = app.getRootNode();

            // Bug 27528: IE does not allow to prevent default actions of some special key events
            // (especially CTRL+P which always opens the Print dialog, regardless whether the keydown
            // event has been canceled in the bubbling phase). The 'official' hack is to bind a
            // keydown event handler with IE's ancient 'attachEvent()' method, and to suppress the
            // default action by changing the key code in the event object (add smiley-with-big-eyes
            // here). Note that the 'addEventListener()' method does NOT work for this hack.
            // Bug 29544: IE 11 does not support the attachEvent() method anymore (hack for bug 27528 will fail).
            if (_.browser.IE && _.isFunction(rootNode[0].attachEvent)) {
                rootNode[0].attachEvent('onkeydown', function (event) {
                    var result = keyHandler(_.extend(event, { target: event.srcElement }));
                    // modify the key code to prevent the browser default action
                    if (result === false) { event.keyCode = 0; }
                    return result;
                });
            } else {
                rootNode.on('keydown', keyHandler);
            }

            // register 'keypress' event listener for shortcuts
            rootNode.on('keypress', keyHandler);

            // prevent that Ctrl+A shortcut selects the entire HTML document
            rootNode.on('keydown', function (event) {
                if (!$(event.target).is('input,textarea') && KeyCodes.matchKeyCode(event, 'A', { ctrlOrMeta: true })) {
                    return false;
                }
            });
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the application instance that owns this document controller.
         *
         * @returns {BaseApplication}
         *  The application instance that owns this document controller.
         */
        this.getApp = function () {
            return app;
        };

        /**
         * Adds the definition for a new item to this controller.
         *
         * @param {String} key
         *  The key of the new item.
         *
         * @param {Object} definition
         *  A map with callback functions defining the behavior of the item.
         *  All callback functions will be executed in the context of the Item
         *  class instance. The following attributes are supported:
         *  @param {String|Array<String>} [definition.parent]
         *      The key of an item that will be used to calculate intermediate
         *      results for the getter function and enabler function (see
         *      below), or an array of item keys used as parents for this item.
         *      The key feature of parent items is that if a controller enables
         *      or updates multiple items at once, the getter or enabler of the
         *      same parent item registered at multiple items will be executed
         *      exactly once before the first item getter or enabler is called,
         *      and its result will be cached and passed to all item getters or
         *      enablers that are using this parent item.
         *  @param {Function} [definition.enable]
         *      Predicate function returning true if the item is enabled, and
         *      false otherwise. If one or more parent items have been
         *      specified (see above) and at least one of them returned false,
         *      the item is in disabled state already, and this function will
         *      not be called anymore. Otherwise, the cached return values of
         *      the parent getters (see option 'get' below) will be passed to
         *      this function, in the same order as specified. If the item does
         *      not specify an enable handler, it defaults to the value true,
         *      so the enabled state is dependent on the parent items only.
         *  @param {Function} [definition.get]
         *      Getter function returning the current value of the item. Can be
         *      omitted for one-way action items (actions without a return
         *      value). If one ore more parent items have been specified (see
         *      above), the cached return values of their getters will be
         *      passed to this getter, in the same order as specified. May
         *      return null to indicate an ambiguous state. Defaults to a
         *      function that returns undefined; or, if parent items have been
         *      registered, that returns the value of the first parent.
         *  @param {Function} [definition.set]
         *      Setter function changing the value of an item to the first
         *      parameter of the setter. Can be omitted for read-only items.
         *      Defaults to an empty function. May return a promise to indicate
         *      that the setter executes asynchronous code. In this case, the
         *      application window will be blocked and a busy indicator will be
         *      shown while the setter is running.
         *  @param {Boolean} [definition.cancel=false]
         *      If set to true, and if the setter function of the item has
         *      returned a pending abortable promise, the busy blocker screen
         *      will show a Cancel button that will abort that promise. It is
         *      up to the implementation of the setter function to react
         *      properly on aborting the promise.
         *  @param {Object|Array} [definition.shortcut]
         *      One or multiple keyboard shortcut definitions. If the window
         *      root node of the application receives a 'keydown' or 'keypress'
         *      event that matches a shortcut definition, the setter function
         *      of this item will be executed. Can be as single shortcut
         *      definition, or an array of shortcut definitions. Each
         *      definition object supports the following attributes:
         *      - {Number|String} [shortcut.charCode]
         *          If specified, the shortcut definition will be matched
         *          against 'keypress' events, and the value represents the
         *          numeric code point of a Unicode character, or the Unicode
         *          character as a string.
         *      - {Number|String} [shortcut.keyCode]
         *          If specified, the shortcut definition will be matched
         *          against 'keydown' events, and the value represents the
         *          numeric key code, or the upper-case name of a key code, as
         *          defined in the static class KeyCodes. Be careful with digit
         *          keys, for example, the number 9 matches the TAB key
         *          (KeyCodes.TAB), but the string '9' matches the digit '9'
         *          key (KeyCodes['9'] with the key code 57).
         *      - {Boolean|Null} [shortcut.shift=false]
         *          If set to true, the SHIFT key must be pressed when the
         *          'keydown' events is received. If set to false (or omitted),
         *          the SHIFT key must not be pressed. If set to null, the
         *          current state of the SHIFT key will be ignored. Has no
         *          effect when evaluating 'keypress' events.
         *      - {Boolean|Null} [shortcut.alt=false]
         *          If set to true, the ALT key must be pressed when the
         *          'keydown' events is received. If set to false (or omitted),
         *          the ALT key must not be pressed. If set to null, the
         *          current state of the ALT key will be ignored. Has no effect
         *          when evaluating 'keypress' events.
         *      - {Boolean|Null} [shortcut.ctrl=false]
         *          If set to true, the CTRL key must be pressed when the
         *          'keydown' events is received. If set to false (or omitted),
         *          the CTRL key must not be pressed. If set to null, the
         *          current state of the CTRL key will be ignored. Has no
         *          effect when evaluating 'keypress' events.
         *      - {Boolean|Null} [shortcut.meta=false]
         *          If set to true, the META key must be pressed when the
         *          'keydown' events is received. If set to false (or omitted),
         *          the META key must not be pressed. If set to null, the
         *          current state of the META key will be ignored. Has no
         *          effect when evaluating 'keypress' events.
         *      - {Boolean} [shortcut.altOrMeta=false]
         *          Convenience option that if set to true, matches if either
         *          the ALT key, or the META key are pressed. Has the same
         *          effect as defining two separate shortcuts, one with the
         *          'shortcut.alt' option set to true, and one with the
         *          'shortcut.meta' option set to true, while keeping the other
         *          option false. Must not be used in a shortcut definition
         *          where these options are set explicitly.
         *      - {Boolean} [shortcut.ctrlOrMeta=false]
         *          Convenience option that if set to true, matches if either
         *          the CTRL key, or the META key are pressed. Has the same
         *          effect as defining two separate shortcuts, one with the
         *          'shortcut.ctrl' option set to true, and one with the
         *          'shortcut.meta' option set to true, while keeping the other
         *          option false. Must not be used in a shortcut definition
         *          where these options are set explicitly.
         *      - {Any|Function} [shortcut.value]
         *          The value that will be passed to the setter function of
         *          this item. If multiple shortcuts are defined for an item,
         *          each shortcut definition may define its own value. If this
         *          option contains a function, it will receive the current
         *          value of the controller item as first parameter, and its
         *          return value will be passed to the setter function.
         *      - {Boolean} [shortcut.propagate=false]
         *          If set to true, the event will propagate up to the DOM root
         *          element, and the browser will execute its default action.
         *          If omitted or set to false, the event will be cancelled
         *          immediately after calling the setter function.
         *      - {String} [shortcut.selector]
         *          A CSS selector used to restrict the keyboard shortcut to
         *          specific DOM nodes in the application DOM. The target node
         *          of the keyboard event has to match the selector, or has to
         *          be a descendant node of a DOM node matching the selector.
         *          Otherwise, the event will be ignored completely, and the
         *          browser will execute its default action.
         *  @param {Boolean} [definition.preserveFocus=false]
         *      Whether to return the browser focus to the application pane
         *      after executing the setter function of this item (the default
         *      behavior). If set to true, the current browser focus will not
         *      be changed. Otherwise, the focus will return to the application
         *      pane after the setter function has finished (will wait for
         *      asynchronous setters before returning the focus).
         *  @param {HTMLElement|jQuery|Object|Function} [options.focusTarget]
         *      A DOM node that will receive the browser focus after the item
         *      has been executed, an object that provides a grabFocus()
         *      method, or a callback function that returns a focus target
         *      dynamically. Default focus target is the view. Has no effect,
         *      if the option 'preserveFocus' is set to true.
         *
         * @returns {BaseController}
         *  A reference to this controller instance.
         */
        this.registerDefinition = function (key, definition) {

            // create a new item instance
            items[key] = new Item(key, definition);

            // special preparations for global keyboard shortcuts
            if (_.isObject(definition.shortcut)) {
                _.getArray(definition.shortcut).forEach(function (shortcut) {

                    var keyCode = Utils.getOption(shortcut, 'keyCode'),
                        charCode = Utils.getOption(shortcut, 'charCode');

                    if (_.isString(keyCode)) {
                        keyCode = KeyCodes[keyCode] || 0;
                    }
                    if (_.isNumber(keyCode) && (keyCode > 0)) {
                        (keyShortcuts[keyCode] || (keyShortcuts[keyCode] = [])).push({ key: key, definition: shortcut });
                    }
                    if (_.isString(charCode)) {
                        charCode = charCode.charCodeAt(0);
                    }
                    if (_.isNumber(charCode) && (charCode > 0)) {
                        (charShortcuts[charCode] || (charShortcuts[charCode] = [])).push({ key: key, definition: shortcut });
                    }
                });
            }

            return this;
        };

        /**
         * Adds definitions for multiple items to this controller.
         *
         * @param {Object} definitions
         *  A map of definitions for all new items, mapped by item key. Each
         *  new controller item will be defined by calling the method
         *  BaseController.registerDefinition(). See this method for more
         *  details.
         *
         * @returns {BaseController}
         *  A reference to this controller instance.
         */
        this.registerDefinitions = function (definitions) {
            _.each(definitions, function (definition, key) {
                this.registerDefinition(key, definition);
            }, this);
            return this;
        };

        /**
         * Refreshes the states and values of all registered items, and
         * triggers a 'change:items' event.
         *
         * @returns {BaseController}
         *  A reference to this controller.
         */
        this.update = (function () {

            // clear the entire cache state on every call of the update() method
            function invalidateStates() {
                stateCache = {};
            }

            // after a delay, collect all item states, and notify change listeners
            function updateStates() {

                // collect all item states missing in the state cache

                // IE workaround: early exit after 3 seconds (only important for view update),
                // but freezing too long can kill RT connection and the browser itself
                if (_.browser.IE) {
                    var t0 = _.now(), i = 0;
                    _.any(items, function (item) {
                        item.getState();
                        i = (i + 1) % 10;
                        if ((i === 0) && (_.now() - t0 > 3000)) {
                            Utils.error('BaseController.update(): performance bottleneck (early exit after 3s), please fix item getters!');
                            return true;
                        }
                    });
                } else {
                    _.invoke(items, 'getState');
                }

                // reduce to the properties that differ from previous state
                var changedStates = _.omit(stateCache, function (state, key) {
                    var prevState = notifiedStateCache[key];
                    return prevState && !state.dirty && (prevState.enabled === state.enabled) && _.isEqual(prevState.value, state.value);
                });

                // store the current cache for next update() call
                // bug 40770: create flat clones of all cache entries, to be able to change stateCache in-place later
                notifiedStateCache = _.mapObject(stateCache, function (state) { return _.clone(state); });

                // notify all listeners (with locked GUI refresh for performance,
                // view elements are listening to controller items to change their visibility)
                if (!_.isEmpty(changedStates)) {
                    docView.lockPaneLayout(function () {
                        self.trigger('change:items', changedStates);
                    });
                }
            }

            return self.createDebouncedMethod(invalidateStates, updateStates, { delay: updateDelay, infoString: 'BaseController: update' });
        }());

        /**
         * Returns the current enabled state of the specified item.
         *
         * @param {String} key
         *  The key of the item.
         *
         * @returns {Boolean}
         *  Whether the specified item exists and is enabled.
         */
        this.isItemEnabled = function (key) {
            return (key in items) && items[key].isEnabled();
        };

        /**
         * Returns the current value of the specified item.
         *
         * @param {String} key
         *  The key of the item.
         *
         * @returns {Any}
         *  The current value of the item, or undefined, if the item does not
         *  exist.
         */
        this.getItemValue = function (key) {
            return (key in items) ? items[key].getValue() : undefined;
        };

        /**
         * Triggers a controller item manually. Executes the setter function of
         * the item associated to the specified key, passing in the new value.
         *
         * @param {String} key
         *  The key of the item to be changed.
         *
         * @param {Any} [value]
         *  The new value of the item.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.sourceType='custom']
         *      The cause of the item invocation. Can be one of the following
         *      types:
         *      - 'click': The item has been executed due to a button click, or
         *          another comparable GUI action such as a tap on a touch
         *          device.
         *      - 'keyboard': The item has been executed due to a standard
         *          keyboard event, e.g. pressing the ENTER key on buttons or
         *          text fields, pressing the SPACE key on buttons, or pressing
         *          the TAB key in modified text fields; or a keyboard shortcut
         *          as defined for the respective controller item.
         *      - 'custom': (default) Any other cause.
         *  @param {Boolean} [options.preserveFocus=false]
         *      If set to true, the browser focus will not be modified after
         *      executing the item, regardless of the focus mode configured for
         *      that item.
         *  @param {HTMLElement|jQuery|Object|Function} [options.focusTarget]
         *      A DOM node that will receive the browser focus after the item
         *      has been executed, an object that provides a grabFocus()
         *      method, or a callback function that returns a focus target
         *      dynamically. Overrides the default focus target defined for the
         *      executed item (option 'focusTarget' in the item definition).
         *      Default focus target is the view. Has no effect, if the option
         *      'preserveFocus' is set to true.
         *  @param {jQuery.Event} [options.changeEvent]
         *      Set by the framework when executing a controller item after
         *      activating a GUI control element. Represents the event object
         *      of the 'group:change' event as triggered by instances of the
         *      Group class.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected according to the result
         *  of the item setter function. If the item is disabled or does not
         *  exist, a rejected promise will be returned. If the item setter
         *  returns a promise, that promise will be returned. Otherwise, the
         *  return value of the item setter function will be wrapped and
         *  returned in a resolved promise.
         */
        this.executeItem = function (key, value, options) {

            // execute setter of existing item
            if (key in items) {
                return items[key].execute(value, options);
            }

            // item does not exist: return focus to application
            this.grabFocus(options);
            return $.Deferred().reject();
        };

        /**
         * Moves the browser focus back to the application pane, or to a custom
         * focus target node, if specified in the passed options.
         *
         * @param {Object} [options]
         *  Optional parameters. See method BaseController.executeItem() for
         *  details.
         *
         * @returns {BaseController}
         *  A reference to this instance.
         */
        this.grabFocus = function (options) {

            var preserveFocus = Utils.getBooleanOption(options, 'preserveFocus', false),
                focusTarget = Utils.getOption(options, 'focusTarget'),
                changeEvent = Utils.getOption(options, 'changeEvent'),
                sourceNode = Utils.getOption(changeEvent, 'sourceNode');

            // keep current focus, if 'preserveFocus' is set
            // Bug 28214: Focus back to application if source GUI element is hidden
            // now. If the source GUI element is hidden now, IE has already changed
            // the 'document.activeElement' property, so it cannot be used to detect
            // visibility. Therefore, all group instances now insert the source DOM
            // node into the 'group:change' event.
            if (preserveFocus && (!sourceNode || $(sourceNode).is(':visible'))) {
                return this;
            }

            // resolve callback function to focus target
            if (_.isFunction(focusTarget)) {
                focusTarget = focusTarget.call(self, Utils.getStringOption(options, 'sourceType', 'custom'));
            }

            // move focus to specified target, or to the view
            if (_.isObject(focusTarget) && _.isFunction(focusTarget.grabFocus) && _.isFunction(focusTarget.isVisible) && focusTarget.isVisible()) {
                focusTarget.grabFocus(options);
            } else if (((focusTarget instanceof $) || (focusTarget instanceof HTMLElement)) && $(focusTarget).is(':visible')) {
                $(focusTarget).focus();
            } else {
                docView.grabFocus(options);
            }

            return this;
        };

        // initialization -----------------------------------------------------

        // dummy item singleton, used e.g. for undefined parent items
        dummyItem = new Item('\x00');

        // register all controller items
        this.registerDefinitions({

            // quit/destroy the application (do not return the promise,
            // otherwise this controller would wait for its own destruction)
            'app/quit': {
                set: function () { _.defer(function () { app.quit(); }); }
            },

            // disabled during import, enabled afterwards
            'app/imported': {
                enable: function () { return app.isImportFinished(); }
            },

            // disabled in stand-alone mode
            'app/bundled': {
                enable: function () { return !app.isStandalone(); }
            },

            // download the document: enabled in read-only mode, disabled during import
            'document/download': {
                parent: ['app/imported', 'system/filesystem'],
                set: function () { return app.download(); }
            },

            // print the document (download as PDF): enabled in read-only mode, disabled during import
            'document/print': {
                parent: 'app/imported',
                set: function () { return app.print(); },
                shortcut: { keyCode: 'P', ctrlOrMeta: true }
            },

            // compose new mail with this document's content being the mail-body's content:
            // enabled in read-only mode, disabled during import, disabled in stand-alone-mode
            'document/sendmail': {
                parent: ['app/imported', 'app/bundled'],
                enable: function () { return !DriveUtils.isGuest(); },
                set: function () { return app.sendMail(); }
            },
            'document/sendmail/inline-html': {
                parent: 'document/sendmail',
                enable: function () { return app.canSendMail(); },
                set: function (config) { return app.sendMail(config); }
            },

            // all supported operation systems have a file system where user can save in, but not on IOS
            'system/filesystem': {
                enable: function () { return !Utils.IOS; }
            }
        });

        // update once after successful import
        this.waitForImportSuccess(function () {
            initializeKeyHandler();
            this.listenTo(docView, 'refresh:layout', function () { self.update(); });
            this.update();
        }, this);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.invoke(items, 'destroy');
            dummyItem.destroy();
            self = docModel = docView = null;
            items = dummyItem = stateCache = keyShortcuts = charShortcuts = null;
        });

    } // class BaseController

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: BaseController });

});
