/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

(function (define, require) {

    // define.async() ---------------------------------------------------------

    function getLoader(name, deps, callback) {
        return function (n, req, onLoad) {
            // resolve module dependencies
            req(deps, function () {
                // get module (must return deferred object)
                var def = callback.apply(null, arguments);
                if (def && def.done) {
                    def.done(onLoad);
                } else if (console.error) {
                    console.error('Module "' + name + '" does not return a deferred object!');
                }
                name = deps = callback = null;
            });
        };
    }

    define.async = function (name, deps, callback) {
        // use loader plug-in to defer module definition
        define(name + ':init', { load: getLoader(name, deps, callback) });
        // define real module - will wait for promise
        define(name, [name + ':init!'], _.identity);
    };

    // require() with promises ------------------------------------------------

    function fallback(error) {
        console.error('require: Error in ' + error.requireModules, error.stack);
    }

    window.require = function (deps, success, fail) {

        if (_.isArray(deps)) {
            // use deferred object
            _(deps).each(function (name) {
                $(window).trigger('require:require', name);
            });
            var def = $.Deferred().done(success).fail(fail || fallback);
            require(deps, def.resolve, def.reject);
            return def.promise();
        }

        // bypass
        return require.apply(this, arguments);
    };

    _.extend(window.require, require);

    // text plug-ins ----------------------------------------------------------

    // loads a file via AJAX request
    function ajax(name, type, parentRequire, load, config) {
        return $.ajax({ url: config.baseUrl + name, dataType: type }).then(
            load,
            function fail() {
                // this simple line might save life time
                console.error('Cannot load ' + type + ' file', name, arguments);
                load.error.apply(load, arguments);
            }
        );
    }

    define('text', {
        load: function (name, parentRequire, load, config) {
            return ajax(name, 'text', parentRequire, load, config);
        }
    });

    define('json', {
        load: function (name, parentRequire, load, config) {
            return ajax(name + '.json', 'json', parentRequire, load, config);
        }
    });

    // LESS plug-in -----------------------------------------------------------

    var // Name of the current theme, or falsy before a theme is set.
        theme = '',
        // LessCSS files of the current theme.
        themeCommon = { name: 'common', selector: '#theme' },
        themeStyle = { name: 'style', selector: '#custom' },
        // List of LessCSS files to update for theme changes.
        lessFiles = [themeCommon, themeStyle];

    // loads a LESS file via text plug-in
    function insertLess(file) {
        return require(['text!themes/' + theme + '/' + file.name + '.css'], function (css) {
            file.node = insert(file.path, css, file.selector, file.node);
        });
    }

    define('less', {
        load: function (name, parentRequire, load, config) {
            var file = {
                path: config.baseUrl + name,
                name: name,
                selector: '#css'
            };
            lessFiles.push(file);
            if (theme) {
                insertLess(file).then(load, load.error);
            } else {
                load();
            }
        }
    });

}(window.define, window.require));
