/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/tk/object/timermixin'
], function (Utils, BaseObject, TimerMixin) {

    'use strict';

    // jQuery.Deferred base test ==============================================

    describe('class jQuery.Deferred', function () {
        it('should exist', function () {
            expect($.Deferred).to.be.a('function');
        });
        it('should create a pending deferred object', function () {
            var def = $.Deferred();
            expect(def).to.be.an('object');
            expect(def).to.respondTo('state');
            expect(def).to.respondTo('promise');
            expect(def).to.respondTo('resolve');
            expect(def).to.respondTo('reject');
            expect(def).to.respondTo('done');
            expect(def).to.respondTo('fail');
            expect(def).to.respondTo('then');
        });
        it('should have pending state', function () {
            expect($.Deferred().state()).to.equal('pending');
        });
        it('should resolve immediately', function () {
            var def = $.Deferred();
            expect(def.resolve()).to.equal(def);
            expect(def.state()).to.equal('resolved');
        });
        it('should reject immediately', function () {
            var def = $.Deferred();
            expect(def.reject()).to.equal(def);
            expect(def.state()).to.equal('rejected');
        });
    });

    // mix-in class TimerMixin ================================================

    describe('Toolkit mix-in class TimerMixin', function () {

        it('should exist', function () {
            expect(TimerMixin).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        function TimerClass() {
            BaseObject.call(this);
            TimerMixin.call(this);
        }

        var clock = null;
        before(function () { clock = sinon.useFakeTimers(); });
        after(function () { clock.restore(); });

        // public methods -----------------------------------------------------

        describe('method "createAbortablePromise"', function () {
            var obj = new TimerClass();
            it('should exist', function () {
                expect(obj).to.respondTo('createAbortablePromise');
            });
            it('should create an abortable promise', function () {
                var promise = obj.createAbortablePromise($.Deferred());
                expect(promise).to.be.an('object');
                expect(promise).to.respondTo('abort');
                expect(promise.state()).to.equal('pending');
                promise.abort();
                expect(promise.state()).to.equal('rejected');
            });
            it('should create an abortable promise of a resolved deferred object', function () {
                var promise = obj.createAbortablePromise($.Deferred().resolve());
                expect(promise).to.be.an('object');
                expect(promise).to.respondTo('abort');
                expect(promise.state()).to.equal('resolved');
                promise.abort();
                expect(promise.state()).to.equal('resolved');
            });
            it('should invoke the callback on abort', function () {
                var spy = sinon.spy(),
                    promise = obj.createAbortablePromise($.Deferred(), spy);
                expect(spy.called).to.equal(false);
                promise.abort();
                expect(spy.calledOnce).to.equal(true);
                expect(spy.calledOn(promise)).to.equal(true);
                expect(spy.calledWith('abort')).to.equal(true);
            });
            it('should abort the promise after a timeout', function () {
                var promise = obj.createAbortablePromise($.Deferred(), $.noop, 10);
                expect(promise.state()).to.equal('pending');
                clock.tick(11);
                expect(promise.state()).to.equal('rejected');
            });
            it('should override promise.then() to return an abortable promise', function () {
                var spy = sinon.spy(),
                    promise1 = obj.createAbortablePromise($.Deferred(), spy).fail(spy),
                    promise2 = promise1.then(null, spy, null, spy);
                expect(promise1).to.not.equal(promise2);
                expect(promise2).to.respondTo('abort');
                promise2.abort(42);
                expect(promise1.state()).to.equal('rejected');
                expect(promise2.state()).to.equal('rejected');
                expect(spy.callCount).to.equal(4);
                expect(spy.alwaysCalledWith(42)).to.equal(true);

            });
            it('should abort the promise on destruction', function () {
                var spy = sinon.spy(),
                    promise = obj.createAbortablePromise($.Deferred(), spy);
                expect(promise.state()).to.equal('pending');
                obj.destroy();
                expect(promise.state()).to.equal('rejected');
                expect(spy.calledWith('abort')).to.equal(true);
            });
        });

        describe('method "executeDelayed"', function () {
            var obj = new TimerClass();
            it('should exist', function () {
                expect(obj).to.respondTo('executeDelayed');
            });
            it('should invoke callback delayed', function () {
                var spy = sinon.spy(),
                    timer = obj.executeDelayed(spy);
                expect(spy.called).to.equal(false);
                expect(timer.state()).to.equal('pending');
                clock.tick(2);
                expect(timer.state()).to.equal('resolved');
                expect(spy.calledOnce).to.equal(true);
                expect(spy.calledOn(obj)).to.equal(true);
            });
            it('should invoke callbacks after specified delay time', function () {
                var spy = sinon.spy();
                obj.executeDelayed(spy, { delay: 10 });
                expect(spy.called).to.equal(false);
                clock.tick(9);
                expect(spy.called).to.equal(false);
                clock.tick(2);
                expect(spy.called).to.equal(true);
            });
            it('should accept delay time directly as numeric parameter', function () {
                var spy = sinon.spy();
                obj.executeDelayed(spy, 10);
                expect(spy.called).to.equal(false);
                clock.tick(9);
                expect(spy.called).to.equal(false);
                clock.tick(2);
                expect(spy.called).to.equal(true);
            });
            it('should invoke callbacks in correct order', function () {
                var spy1 = sinon.spy(), spy2 = sinon.spy();
                obj.executeDelayed(spy1, 10);
                obj.executeDelayed(spy2, 9);
                clock.tick(11);
                expect(spy1.called).to.equal(true);
                expect(spy2.called).to.equal(true);
                expect(spy1.calledAfter(spy2)).to.equal(true);
            });
            it('should allow to abort a timer', function () {
                var spy = sinon.spy(),
                    timer = obj.executeDelayed(spy, 10);
                clock.tick(9);
                timer.abort();
                clock.tick(1000);
                expect(spy.called).to.equal(false);
                expect(timer.state()).to.equal('rejected');
            });
            it('should forward the callback result', function () {
                var stub = sinon.stub().returns(42),
                    spy = sinon.spy(),
                    timer = obj.executeDelayed(stub, 10).done(spy);
                clock.tick(20);
                expect(stub.called).to.equal(true);
                expect(timer.state()).to.equal('resolved');
                expect(spy.calledWith(42)).to.equal(true);
            });
            it('should defer timer promise to callback promise', function () {
                var def = $.Deferred(),
                    stub = sinon.stub().returns(def.promise()),
                    spy = sinon.spy(),
                    timer = obj.executeDelayed(stub, 10).done(spy);
                clock.tick(20);
                expect(stub.called).to.equal(true);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(false);
                def.resolve(42);
                expect(timer.state()).to.equal('resolved');
                expect(spy.calledWith(42)).to.equal(true);
            });
            it('should not invoke callbacks after destruction', function () {
                var spy = sinon.spy(),
                    timer = obj.executeDelayed(spy, 10);
                clock.tick(9);
                obj.destroy();
                clock.tick(1000);
                expect(spy.called).to.equal(false);
                expect(timer.state()).to.equal('rejected');
            });
        });

        describe('method "repeatDelayed"', function () {
            var obj = new TimerClass();
            it('should exist', function () {
                expect(obj).to.respondTo('repeatDelayed');
            });
            it('should invoke callback delayed and stop on Utils.BREAK', function () {
                var stub = sinon.stub(),
                    timer = obj.repeatDelayed(stub);
                stub.onThirdCall().returns(Utils.BREAK);
                expect(stub.called).to.equal(false);
                clock.tick(10);
                expect(stub.calledThrice).to.equal(true);
                expect(stub.alwaysCalledOn(obj)).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should pass iteration index to callback', function () {
                var stub = sinon.stub(),
                    timer = obj.repeatDelayed(stub);
                stub.onThirdCall().returns(Utils.BREAK);
                clock.tick(10);
                expect(stub.getCall(0).calledWith(0)).to.equal(true);
                expect(stub.getCall(1).calledWith(1)).to.equal(true);
                expect(stub.getCall(2).calledWith(2)).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should invoke callbacks after specified delay time', function () {
                var stub = sinon.stub(),
                    timer = obj.repeatDelayed(stub, { delay: 10 });
                stub.onThirdCall().returns(Utils.BREAK);
                clock.tick(9);
                expect(stub.called).to.equal(false);
                clock.tick(2);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(8);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(2);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(8);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(2);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should accept delay time directly as numeric parameter', function () {
                var stub = sinon.stub(),
                    timer = obj.repeatDelayed(stub, 10);
                stub.onThirdCall().returns(Utils.BREAK);
                clock.tick(9);
                expect(stub.called).to.equal(false);
                clock.tick(2);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(10);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(10);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should run first iteration synchronously', function () {
                var stub = sinon.stub();
                stub.onThirdCall().returns(Utils.BREAK);
                var timer = obj.repeatDelayed(stub, { delay: 'immediate' });
                expect(stub.calledOnce).to.equal(true);
                clock.tick(10);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should use different repetition delay time', function () {
                var stub = sinon.stub(),
                    timer = obj.repeatDelayed(stub, { delay: 10, repeatDelay: 20 });
                stub.onThirdCall().returns(Utils.BREAK);
                clock.tick(9);
                expect(stub.called).to.equal(false);
                clock.tick(2);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(18);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(2);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(18);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(2);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should use specified number of cycles', function () {
                var stub = sinon.stub(),
                    timer = obj.repeatDelayed(stub, { delay: 10, cycles: 3 });
                clock.tick(9);
                expect(stub.called).to.equal(false);
                clock.tick(2);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(10);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(10);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should allow to abort a timer', function () {
                var spy = sinon.spy(),
                    timer = obj.repeatDelayed(spy, 10);
                clock.tick(25);
                expect(spy.calledTwice).to.equal(true);
                timer.abort();
                clock.tick(1000);
                expect(spy.calledTwice).to.equal(true);
                expect(timer.state()).to.equal('rejected');
            });
            it('should defer next iteration to callback promise', function () {
                var stub = sinon.stub(),
                    def = $.Deferred(),
                    timer = obj.repeatDelayed(stub, 10);
                stub.onSecondCall().returns(def);
                stub.onThirdCall().returns(Utils.BREAK);
                clock.tick(15);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(10);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(20);
                expect(stub.calledTwice).to.equal(true);
                def.resolve();
                clock.tick(9);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(2);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should abort on rejected callback promise', function () {
                var stub = sinon.stub(),
                    timer = obj.repeatDelayed(stub, 10);
                stub.onThirdCall().returns($.Deferred().reject());
                clock.tick(25);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(10);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('rejected');
            });
            it('should reduce delay time after a pending promise in fastAsync mode', function () {
                var stub = sinon.stub(),
                    def1 = $.Deferred(),
                    def2 = $.Deferred(),
                    timer = obj.repeatDelayed(stub, { delay: 10, fastAsync: true });
                stub.onFirstCall().returns(def1);
                stub.onSecondCall().returns(def2);
                stub.onThirdCall().returns(Utils.BREAK);
                clock.tick(11);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(4);
                def1.resolve();
                clock.tick(4);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(2);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(20);
                expect(stub.calledTwice).to.equal(true);
                def2.resolve();
                clock.tick(1);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should not invoke callbacks after destruction', function () {
                var spy = sinon.spy(),
                    timer = obj.repeatDelayed(spy, 10);
                clock.tick(25);
                expect(spy.calledTwice).to.equal(true);
                obj.destroy();
                clock.tick(1000);
                expect(spy.calledTwice).to.equal(true);
                expect(timer.state()).to.equal('rejected');
            });
        });

        describe('method "repeatSliced"', function () {
            var obj = new TimerClass();
            function worker(i) {
                clock.tick(150);
                if (i === 4) { return Utils.BREAK; }
            }
            it('should exist', function () {
                expect(obj).to.respondTo('repeatSliced');
            });
            it('should invoke callback in time slices and stop on Utils.BREAK', function () {
                var spy = sinon.spy(worker),
                    timer = obj.repeatSliced(spy);
                expect(spy.called).to.equal(false);
                clock.tick(301);
                expect(spy.callCount).to.equal(2);
                clock.tick(8);
                expect(spy.callCount).to.equal(2);
                clock.tick(302);
                expect(spy.callCount).to.equal(4);
                clock.tick(8);
                expect(spy.callCount).to.equal(4);
                clock.tick(152);
                expect(spy.callCount).to.equal(5);
                expect(spy.alwaysCalledOn(obj)).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should pass iteration index to callback', function () {
                var spy = sinon.spy(worker),
                    timer = obj.repeatSliced(spy);
                clock.tick(771);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
                expect(spy.getCall(0).calledWith(0)).to.equal(true);
                expect(spy.getCall(1).calledWith(1)).to.equal(true);
                expect(spy.getCall(2).calledWith(2)).to.equal(true);
                expect(spy.getCall(3).calledWith(3)).to.equal(true);
                expect(spy.getCall(4).calledWith(4)).to.equal(true);
            });
            it('should invoke callbacks after specified delay time', function () {
                var spy = sinon.spy(worker),
                    timer = obj.repeatSliced(spy, { delay: 10 });
                expect(spy.called).to.equal(false);
                clock.tick(9);
                expect(spy.called).to.equal(false);
                clock.tick(302);
                expect(spy.callCount).to.equal(2);
                clock.tick(310);
                expect(spy.callCount).to.equal(4);
                clock.tick(160);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should use specified slice time', function () {
                var spy = sinon.spy(worker),
                    timer = obj.repeatSliced(spy, { slice: 500 });
                expect(spy.called).to.equal(false);
                clock.tick(601);
                expect(spy.callCount).to.equal(4);
                clock.tick(8);
                expect(spy.callCount).to.equal(4);
                clock.tick(152);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should use specified interval delay time', function () {
                var spy = sinon.spy(worker),
                    timer = obj.repeatSliced(spy, { interval: 20 });
                expect(spy.called).to.equal(false);
                clock.tick(301);
                expect(spy.callCount).to.equal(2);
                clock.tick(18);
                expect(spy.callCount).to.equal(2);
                clock.tick(302);
                expect(spy.callCount).to.equal(4);
                clock.tick(18);
                expect(spy.callCount).to.equal(4);
                clock.tick(152);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should run first iteration synchronously', function () {
                var spy = sinon.spy(worker),
                    timer = obj.repeatSliced(spy, { delay: 'immediate' });
                expect(spy.callCount).to.equal(2);
                clock.tick(301);
                expect(spy.callCount).to.equal(4);
                clock.tick(160);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should allow to abort a timer', function () {
                var spy = sinon.spy(worker),
                    timer = obj.repeatSliced(spy);
                clock.tick(301);
                expect(spy.callCount).to.equal(2);
                timer.abort();
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
                expect(timer.state()).to.equal('rejected');
            });
            it('should defer next iteration to callback promise', function () {
                var def = $.Deferred();
                function worker(i) {
                    clock.tick(150);
                    return (i === 4) ? Utils.BREAK : def;
                }
                var spy = sinon.spy(worker),
                    timer = obj.repeatSliced(spy);
                clock.tick(151);
                expect(spy.callCount).to.equal(1);
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                def.resolve();
                clock.tick(2);
                expect(spy.callCount).to.equal(3);
                clock.tick(310);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should abort on rejected callback promise', function () {
                var def = $.Deferred();
                function worker(i) {
                    clock.tick(150);
                    return (i === 4) ? Utils.BREAK : def;
                }
                var spy = sinon.spy(worker),
                    timer = obj.repeatSliced(spy);
                clock.tick(151);
                expect(spy.callCount).to.equal(1);
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                def.reject();
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                expect(timer.state()).to.equal('rejected');
            });
            it('should not invoke callbacks after destruction', function () {
                var spy = sinon.spy(worker),
                    timer = obj.repeatSliced(spy);
                clock.tick(301);
                expect(spy.callCount).to.equal(2);
                obj.destroy();
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
                expect(timer.state()).to.equal('rejected');
            });
        });

        describe('method "iterateArraySliced"', function () {
            var ARRAY = ['a', 'b', 'c', 'd', 'e'],
                obj = new TimerClass();
            function worker() { clock.tick(150); }
            it('should exist', function () {
                expect(obj).to.respondTo('iterateArraySliced');
            });
            it('should return a resolved promise, if an empty array is given', function () {
                var spy = sinon.spy(),
                    timer = obj.iterateArraySliced([]);
                expect(timer).to.be.an('object');
                expect(timer.state()).to.equal('resolved');
                expect(spy.callCount).to.equal(0);
            });

            it('should invoke callback in time slices', function () {
                var spy = sinon.spy(worker),
                    timer = obj.iterateArraySliced(ARRAY, spy);
                expect(spy.called).to.equal(false);
                clock.tick(301);
                expect(spy.callCount).to.equal(2);
                clock.tick(8);
                expect(spy.callCount).to.equal(2);
                clock.tick(302);
                expect(spy.callCount).to.equal(4);
                clock.tick(8);
                expect(spy.callCount).to.equal(4);
                clock.tick(152);
                expect(spy.callCount).to.equal(5);
                expect(spy.alwaysCalledOn(obj)).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });

            it('should stop on Utils.BREAK', function () {
                var stub = sinon.stub();
                stub.onSecondCall().returns(Utils.BREAK);
                var timer = obj.iterateArraySliced(ARRAY, stub);
                expect(stub.called).to.equal(false);
                clock.tick(2);
                expect(stub.callCount).to.equal(2);
                expect(timer.state()).to.equal('resolved');
            });
            it('should pass array elements to callback', function () {
                var spy = sinon.spy(worker),
                    timer = obj.iterateArraySliced(ARRAY, spy);
                clock.tick(771);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
                expect(spy.getCall(0).calledWith('a', 0, ARRAY)).to.equal(true);
                expect(spy.getCall(1).calledWith('b', 1, ARRAY)).to.equal(true);
                expect(spy.getCall(2).calledWith('c', 2, ARRAY)).to.equal(true);
                expect(spy.getCall(3).calledWith('d', 3, ARRAY)).to.equal(true);
                expect(spy.getCall(4).calledWith('e', 4, ARRAY)).to.equal(true);
            });
            it('should process array elements in reversed order', function () {
                var spy = sinon.spy(worker),
                    timer = obj.iterateArraySliced(ARRAY, spy, { reverse: true });
                clock.tick(771);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
                expect(spy.getCall(0).calledWith('e', 4, ARRAY)).to.equal(true);
                expect(spy.getCall(1).calledWith('d', 3, ARRAY)).to.equal(true);
                expect(spy.getCall(2).calledWith('c', 2, ARRAY)).to.equal(true);
                expect(spy.getCall(3).calledWith('b', 1, ARRAY)).to.equal(true);
                expect(spy.getCall(4).calledWith('a', 0, ARRAY)).to.equal(true);
            });
            it('should work with array-like objects', function () {
                var spy = sinon.spy(worker),
                    timer = obj.iterateArraySliced('abcde', spy);
                clock.tick(771);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
                expect(spy.getCall(0).calledWith('a', 0, 'abcde')).to.equal(true);
                expect(spy.getCall(1).calledWith('b', 1, 'abcde')).to.equal(true);
                expect(spy.getCall(2).calledWith('c', 2, 'abcde')).to.equal(true);
                expect(spy.getCall(3).calledWith('d', 3, 'abcde')).to.equal(true);
                expect(spy.getCall(4).calledWith('e', 4, 'abcde')).to.equal(true);
            });
            it('should invoke callbacks after specified delay time', function () {
                var spy = sinon.spy(worker),
                    timer = obj.iterateArraySliced(ARRAY, spy, { delay: 10 });
                expect(spy.called).to.equal(false);
                clock.tick(9);
                expect(spy.called).to.equal(false);
                clock.tick(302);
                expect(spy.callCount).to.equal(2);
                clock.tick(310);
                expect(spy.callCount).to.equal(4);
                clock.tick(160);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should use specified slice time', function () {
                var spy = sinon.spy(worker),
                    timer = obj.iterateArraySliced(ARRAY, spy, { slice: 500 });
                expect(spy.called).to.equal(false);
                clock.tick(601);
                expect(spy.callCount).to.equal(4);
                clock.tick(8);
                expect(spy.callCount).to.equal(4);
                clock.tick(152);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should use specified interval delay time', function () {
                var spy = sinon.spy(worker),
                    timer = obj.iterateArraySliced(ARRAY, spy, { interval: 20 });
                expect(spy.called).to.equal(false);
                clock.tick(301);
                expect(spy.callCount).to.equal(2);
                clock.tick(18);
                expect(spy.callCount).to.equal(2);
                clock.tick(302);
                expect(spy.callCount).to.equal(4);
                clock.tick(18);
                expect(spy.callCount).to.equal(4);
                clock.tick(152);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should run first iteration synchronously', function () {
                var spy = sinon.spy(worker),
                    timer = obj.iterateArraySliced(ARRAY, spy, { delay: 'immediate' });
                expect(spy.callCount).to.equal(2);
                clock.tick(301);
                expect(spy.callCount).to.equal(4);
                clock.tick(160);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should allow to abort a timer', function () {
                var spy = sinon.spy(worker),
                    timer = obj.iterateArraySliced(ARRAY, spy);
                clock.tick(301);
                expect(spy.callCount).to.equal(2);
                timer.abort();
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
                expect(timer.state()).to.equal('rejected');
            });
            it('should defer next iteration to callback promise', function () {
                var def = $.Deferred();
                function worker() { clock.tick(150); return def; }
                var spy = sinon.spy(worker),
                    timer = obj.iterateArraySliced(ARRAY, spy);
                clock.tick(151);
                expect(spy.callCount).to.equal(1);
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                def.resolve();
                clock.tick(2);
                expect(spy.callCount).to.equal(3);
                clock.tick(310);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should abort on rejected callback promise', function () {
                var def = $.Deferred();
                function worker() { clock.tick(150); return def; }
                var spy = sinon.spy(worker),
                    timer = obj.iterateArraySliced(ARRAY, spy);
                clock.tick(151);
                expect(spy.callCount).to.equal(1);
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                def.reject();
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                expect(timer.state()).to.equal('rejected');
            });
            it('should not invoke callbacks after destruction', function () {
                var spy = sinon.spy(worker),
                    timer = obj.iterateArraySliced(ARRAY, spy);
                clock.tick(301);
                expect(spy.callCount).to.equal(2);
                obj.destroy();
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
                expect(timer.state()).to.equal('rejected');
            });
        });

        describe('method "iterateObjectSliced"', function () {
            var OBJECT = { a: 1, b: 2, c: 3, d: 4, e: 5 },
                obj = new TimerClass();
            function worker() { clock.tick(150); }
            it('should exist', function () {
                expect(obj).to.respondTo('iterateObjectSliced');
            });
            it('should invoke callback in time slices', function () {
                var spy = sinon.spy(worker),
                    timer = obj.iterateObjectSliced(OBJECT, spy);
                expect(spy.called).to.equal(false);
                clock.tick(301);
                expect(spy.callCount).to.equal(2);
                clock.tick(8);
                expect(spy.callCount).to.equal(2);
                clock.tick(302);
                expect(spy.callCount).to.equal(4);
                clock.tick(8);
                expect(spy.callCount).to.equal(4);
                clock.tick(152);
                expect(spy.callCount).to.equal(5);
                expect(spy.alwaysCalledOn(obj)).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should stop on Utils.BREAK', function () {
                var stub = sinon.stub();
                stub.onSecondCall().returns(Utils.BREAK);
                var timer = obj.iterateObjectSliced(OBJECT, stub);
                expect(stub.called).to.equal(false);
                clock.tick(2);
                expect(stub.callCount).to.equal(2);
                expect(timer.state()).to.equal('resolved');
            });
            it('should pass object properties to callback', function () {
                var spy = sinon.spy(worker),
                    timer = obj.iterateObjectSliced(OBJECT, spy);
                clock.tick(771);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
                expect(spy.getCall(0).calledWith(1, 'a', OBJECT)).to.equal(true);
                expect(spy.getCall(1).calledWith(2, 'b', OBJECT)).to.equal(true);
                expect(spy.getCall(2).calledWith(3, 'c', OBJECT)).to.equal(true);
                expect(spy.getCall(3).calledWith(4, 'd', OBJECT)).to.equal(true);
                expect(spy.getCall(4).calledWith(5, 'e', OBJECT)).to.equal(true);
            });
            it('should invoke callbacks after specified delay time', function () {
                var spy = sinon.spy(worker),
                    timer = obj.iterateObjectSliced(OBJECT, spy, { delay: 10 });
                expect(spy.called).to.equal(false);
                clock.tick(9);
                expect(spy.called).to.equal(false);
                clock.tick(302);
                expect(spy.callCount).to.equal(2);
                clock.tick(310);
                expect(spy.callCount).to.equal(4);
                clock.tick(160);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should use specified slice time', function () {
                var spy = sinon.spy(worker),
                    timer = obj.iterateObjectSliced(OBJECT, spy, { slice: 500 });
                expect(spy.called).to.equal(false);
                clock.tick(601);
                expect(spy.callCount).to.equal(4);
                clock.tick(8);
                expect(spy.callCount).to.equal(4);
                clock.tick(152);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should use specified interval delay time', function () {
                var spy = sinon.spy(worker),
                    timer = obj.iterateObjectSliced(OBJECT, spy, { interval: 20 });
                expect(spy.called).to.equal(false);
                clock.tick(301);
                expect(spy.callCount).to.equal(2);
                clock.tick(18);
                expect(spy.callCount).to.equal(2);
                clock.tick(302);
                expect(spy.callCount).to.equal(4);
                clock.tick(18);
                expect(spy.callCount).to.equal(4);
                clock.tick(152);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should run first iteration synchronously', function () {
                var spy = sinon.spy(worker),
                    timer = obj.iterateObjectSliced(OBJECT, spy, { delay: 'immediate' });
                expect(spy.callCount).to.equal(2);
                clock.tick(301);
                expect(spy.callCount).to.equal(4);
                clock.tick(160);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should allow to abort a timer', function () {
                var spy = sinon.spy(worker),
                    timer = obj.iterateObjectSliced(OBJECT, spy);
                clock.tick(301);
                expect(spy.callCount).to.equal(2);
                timer.abort();
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
                expect(timer.state()).to.equal('rejected');
            });
            it('should defer next iteration to callback promise', function () {
                var def = $.Deferred();
                function worker() { clock.tick(150); return def; }
                var spy = sinon.spy(worker),
                    timer = obj.iterateObjectSliced(OBJECT, spy);
                clock.tick(151);
                expect(spy.callCount).to.equal(1);
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                def.resolve();
                clock.tick(2);
                expect(spy.callCount).to.equal(3);
                clock.tick(310);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should abort on rejected callback promise', function () {
                var def = $.Deferred();
                function worker() { clock.tick(150); return def; }
                var spy = sinon.spy(worker),
                    timer = obj.iterateObjectSliced(OBJECT, spy);
                clock.tick(151);
                expect(spy.callCount).to.equal(1);
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                def.reject();
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                expect(timer.state()).to.equal('rejected');
            });
            it('should not invoke callbacks after destruction', function () {
                var spy = sinon.spy(worker),
                    timer = obj.iterateObjectSliced(OBJECT, spy);
                clock.tick(301);
                expect(spy.callCount).to.equal(2);
                obj.destroy();
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
                expect(timer.state()).to.equal('rejected');
            });
        });

        describe('method "createSynchronizedMethod"', function () {
            var obj = new TimerClass();
            it('should exist', function () {
                expect(obj).to.respondTo('createSynchronizedMethod');
            });
            it('should invoke callback serialized', function () {
                var defs = _.times(3, function () { return $.Deferred(); }),
                    stub = sinon.stub(),
                    method = obj.createSynchronizedMethod(stub);
                _.each(defs, function (def, index) { stub.onCall(index).returns(def); });
                expect(stub.callCount).to.equal(0);
                var promise0 = method();
                expect(stub.callCount).to.equal(1);
                expect(promise0.state()).to.equal('pending');
                var promise1 = method();
                expect(stub.callCount).to.equal(1);
                expect(promise1.state()).to.equal('pending');
                var promise2 = method();
                expect(stub.callCount).to.equal(1);
                expect(promise2.state()).to.equal('pending');
                clock.tick(10);
                defs[0].resolve();
                expect(stub.callCount).to.equal(1);
                clock.tick(2);
                expect(stub.callCount).to.equal(2);
                expect(promise0.state()).to.equal('resolved');
                expect(promise1.state()).to.equal('pending');
                clock.tick(10);
                defs[2].reject();
                expect(stub.callCount).to.equal(2);
                expect(promise1.state()).to.equal('pending');
                expect(promise2.state()).to.equal('pending');
                clock.tick(10);
                defs[1].resolve();
                expect(stub.callCount).to.equal(2);
                clock.tick(2);
                expect(stub.callCount).to.equal(3);
                expect(promise1.state()).to.equal('resolved');
                expect(promise2.state()).to.equal('rejected');
            });
            it('should pass parameters to callback', function () {
                var defs = _.times(3, function () { return $.Deferred(); }),
                    stub = sinon.stub(),
                    method = obj.createSynchronizedMethod(stub);
                _.each(defs, function (def, index) { stub.onCall(index).returns(def); });
                expect(stub.callCount).to.equal(0);
                method(1, 'a');
                method(2, 'b');
                method(3, 'c');
                clock.tick(2);
                defs[0].resolve();
                clock.tick(2);
                defs[1].resolve();
                clock.tick(2);
                defs[2].resolve();
                clock.tick(2);
                expect(stub.callCount).to.equal(3);
                expect(stub.getCall(0).calledWith(1, 'a')).to.equal(true);
                expect(stub.getCall(1).calledWith(2, 'b')).to.equal(true);
                expect(stub.getCall(2).calledWith(3, 'c')).to.equal(true);
            });
            it('should invoke callback with current context', function () {
                var spy = sinon.spy(),
                    method = obj.createSynchronizedMethod(spy);
                method.call(obj);
                expect(spy.called).to.equal(true);
                expect(spy.calledOn(obj)).to.equal(true);
            });
            it('should not invoke callbacks after destruction', function () {
                var defs = _.times(3, function () { return $.Deferred(); }),
                    stub = sinon.stub(),
                    method = obj.createSynchronizedMethod(stub);
                _.each(defs, function (def, index) { stub.onCall(index).returns(def); });
                var promises = _.times(defs.length, method);
                clock.tick(10);
                defs[0].resolve();
                clock.tick(2);
                expect(stub.callCount).to.equal(2);
                expect(promises[0].state()).to.equal('resolved');
                clock.tick(10);
                expect(promises[1].state()).to.equal('pending');
                expect(promises[2].state()).to.equal('pending');
                obj.destroy();
                defs[1].resolve();
                clock.tick(1000);
                defs[2].resolve();
                expect(stub.callCount).to.equal(2);
                expect(promises[1].state()).to.equal('rejected');
                expect(promises[2].state()).to.equal('rejected');
            });
        });

        describe('method "createDebouncedMethod"', function () {
            var obj = new TimerClass();
            it('should exist', function () {
                expect(obj).to.respondTo('createDebouncedMethod');
            });
            it('should invoke callback debounced', function () {
                var spy1 = sinon.spy(),
                    spy2 = sinon.spy(),
                    method = obj.createDebouncedMethod(spy1, spy2);
                expect(spy1.callCount).to.equal(0);
                expect(spy2.callCount).to.equal(0);
                method();
                expect(spy1.callCount).to.equal(1);
                expect(spy2.callCount).to.equal(0);
                method();
                expect(spy1.callCount).to.equal(2);
                expect(spy2.callCount).to.equal(0);
                method();
                expect(spy1.callCount).to.equal(3);
                expect(spy2.callCount).to.equal(0);
                clock.tick(10);
                expect(spy1.callCount).to.equal(3);
                expect(spy2.callCount).to.equal(1);
            });
            it('should pass parameters to callback', function () {
                var spy1 = sinon.spy(),
                    spy2 = sinon.spy(),
                    method = obj.createDebouncedMethod(spy1, spy2);
                method(1, 'a');
                method(2, 'b');
                method(3, 'c');
                clock.tick(10);
                expect(spy1.callCount).to.equal(3);
                expect(spy1.getCall(0).calledWith(1, 'a')).to.equal(true);
                expect(spy1.getCall(1).calledWith(2, 'b')).to.equal(true);
                expect(spy1.getCall(2).calledWith(3, 'c')).to.equal(true);
                expect(spy2.callCount).to.equal(1);
                expect(spy2.firstCall.args.length).to.equal(0);
            });
            it('should invoke callbacks with current context', function () {
                var spy = sinon.spy(),
                    method = obj.createDebouncedMethod(spy, spy);
                method.call(obj);
                clock.tick(10);
                expect(spy.calledTwice).to.equal(true);
                expect(spy.alwaysCalledOn(obj)).to.equal(true);
            });
            it('should use specified delay time', function () {
                var spy = sinon.spy(),
                    method = obj.createDebouncedMethod($.noop, spy, { delay: 10 });
                method();
                clock.tick(2);
                expect(spy.callCount).to.equal(0);
                method();
                clock.tick(9);
                expect(spy.callCount).to.equal(0);
                method();
                clock.tick(9);
                expect(spy.callCount).to.equal(0);
                method();
                clock.tick(9);
                expect(spy.callCount).to.equal(0);
                method();
                clock.tick(11);
                expect(spy.callCount).to.equal(1);
            });
            it('should use specified maximum delay time', function () {
                var spy = sinon.spy(),
                    method = obj.createDebouncedMethod($.noop, spy, { delay: 10, maxDelay: 30 });
                method();
                clock.tick(9);
                method();
                clock.tick(9);
                method();
                clock.tick(9);
                expect(spy.callCount).to.equal(0);
                method();
                clock.tick(4);
                expect(spy.callCount).to.equal(1);
                clock.tick(5);
                expect(spy.callCount).to.equal(1);
                method();
                clock.tick(9);
                expect(spy.callCount).to.equal(1);
                clock.tick(2);
                expect(spy.callCount).to.equal(2);
            });
            it('should not invoke callback after destruction', function () {
                var spy = sinon.spy(),
                    method = obj.createDebouncedMethod($.noop, spy);
                method();
                obj.destroy();
                clock.tick(1000);
                expect(spy.callCount).to.equal(0);
            });
        });
    });

    // ========================================================================
});
