/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/locale/parser'
], function (Parser) {

    'use strict';

    // static class Parser ====================================================

    describe('Toolkit module Parser', function () {

        it('should exist', function () {
            expect(Parser).to.be.an('object');
        });

        // static methods -----------------------------------------------------

        describe('method "numberToString"', function () {
            it('should exist', function () {
                expect(Parser).itself.to.respondTo('numberToString');
            });
            it('should convert numbers to string using local decimal separator', function () {
                expect(Parser.numberToString(0)).to.equal('0');
                expect(Parser.numberToString(42)).to.equal('42');
                expect(Parser.numberToString(-1)).to.equal('-1');
                expect(Parser.numberToString(0.5)).to.equal('0,5');
                expect(Parser.numberToString(-12.5)).to.equal('-12,5');
            });
        });

        describe('method "stringToNumber"', function () {
            it('should exist', function () {
                expect(Parser).itself.to.respondTo('stringToNumber');
            });
            it('should convert strings to number using local decimal separator', function () {
                expect(Parser.stringToNumber('0')).to.equal(0);
                expect(Parser.stringToNumber('42')).to.equal(42);
                expect(Parser.stringToNumber('-1')).to.equal(-1);
                expect(Parser.stringToNumber('0,5')).to.equal(0.5);
                expect(Parser.stringToNumber('-12,5')).to.equal(-12.5);
            });
            it('should return NaN for invalid strings', function () {
                expect(Parser.stringToNumber('')).to.be.nan;
                expect(Parser.stringToNumber('a')).to.be.nan;
            });
        });

        describe('method "parseLeadingNumber"', function () {
            it('should exist', function () {
                expect(Parser).itself.to.respondTo('parseLeadingNumber');
            });
            it('should convert strings to number using local decimal separator', function () {
                expect(Parser.parseLeadingNumber('0,5')).to.deep.equal({ number: 0.5, text: '0,5', sign: '', remaining: '' });
                expect(Parser.parseLeadingNumber('0,5', { complete: true })).to.deep.equal({ number: 0.5, text: '0,5', sign: '', remaining: '' });
                expect(Parser.parseLeadingNumber('0,5abc')).to.deep.equal({ number: 0.5, text: '0,5', sign: '', remaining: 'abc' });
                expect(Parser.parseLeadingNumber('0,5abc', { complete: true })).to.equal(null);
                expect(Parser.parseLeadingNumber('0#5')).to.deep.equal({ number: 0, text: '0', sign: '', remaining: '#5' });
                expect(Parser.parseLeadingNumber('0#5', { complete: true })).to.equal(null);
                expect(Parser.parseLeadingNumber('-12,5')).to.equal(null);
                expect(Parser.parseLeadingNumber('-12,5', { sign: true })).to.deep.equal({ number: -12.5, text: '-12,5', sign: '-', remaining: '' });
                expect(Parser.parseLeadingNumber('-12,5abc', { sign: true })).to.deep.equal({ number: -12.5, text: '-12,5', sign: '-', remaining: 'abc' });
                expect(Parser.parseLeadingNumber('-12,5abc', { sign: true, complete: true })).to.equal(null);
                expect(Parser.parseLeadingNumber('+0,5')).to.equal(null);
                expect(Parser.parseLeadingNumber('+0,5', { sign: true })).to.deep.equal({ number: 0.5, text: '+0,5', sign: '+', remaining: '' });
                expect(Parser.parseLeadingNumber('+0,5abc', { sign: true })).to.deep.equal({ number: 0.5, text: '+0,5', sign: '+', remaining: 'abc' });
                expect(Parser.parseLeadingNumber('+0,5abc', { sign: true, complete: true })).to.equal(null);
                expect(Parser.parseLeadingNumber('12,5E3')).to.deep.equal({ number: 12500, text: '12,5E3', sign: '', remaining: '' });
                expect(Parser.parseLeadingNumber('12,5e003')).to.deep.equal({ number: 12500, text: '12,5e003', sign: '', remaining: '' });
                expect(Parser.parseLeadingNumber('12,5E+003')).to.deep.equal({ number: 12500, text: '12,5E+003', sign: '', remaining: '' });
                expect(Parser.parseLeadingNumber('12,5E-003')).to.deep.equal({ number: 0.0125, text: '12,5E-003', sign: '', remaining: '' });
            });
            it('should convert strings to number using custom decimal separator', function () {
                expect(Parser.parseLeadingNumber('0#5', { sep: '#' })).to.deep.equal({ number: 0.5, text: '0#5', sign: '', remaining: '' });
                expect(Parser.parseLeadingNumber('0#5', { sep: '#', complete: true })).to.deep.equal({ number: 0.5, text: '0#5', sign: '', remaining: '' });
                expect(Parser.parseLeadingNumber('0#5abc', { sep: '#' })).to.deep.equal({ number: 0.5, text: '0#5', sign: '', remaining: 'abc' });
                expect(Parser.parseLeadingNumber('0#5abc', { sep: '#', complete: true })).to.equal(null);
                expect(Parser.parseLeadingNumber('0.5', { sep: '#' })).to.deep.equal({ number: 0, text: '0', sign: '', remaining: '.5' });
                expect(Parser.parseLeadingNumber('0.5', { sep: '#', complete: true })).to.equal(null);
                expect(Parser.parseLeadingNumber('0,5', { sep: '#' })).to.deep.equal({ number: 0, text: '0', sign: '', remaining: ',5' });
                expect(Parser.parseLeadingNumber('0,5', { sep: '#', complete: true })).to.equal(null);
                expect(Parser.parseLeadingNumber('-12#5', { sep: '#' })).to.equal(null);
                expect(Parser.parseLeadingNumber('-12#5', { sep: '#', sign: true })).to.deep.equal({ number: -12.5, text: '-12#5', sign: '-', remaining: '' });
                expect(Parser.parseLeadingNumber('-12#5abc', { sep: '#', sign: true })).to.deep.equal({ number: -12.5, text: '-12#5', sign: '-', remaining: 'abc' });
                expect(Parser.parseLeadingNumber('-12#5abc', { sep: '#', sign: true, complete: true })).to.equal(null);
                expect(Parser.parseLeadingNumber('+0#5', { sep: '#' })).to.equal(null);
                expect(Parser.parseLeadingNumber('+0#5', { sep: '#', sign: true })).to.deep.equal({ number: 0.5, text: '+0#5', sign: '+', remaining: '' });
                expect(Parser.parseLeadingNumber('+0#5abc', { sep: '#', sign: true })).to.deep.equal({ number: 0.5, text: '+0#5', sign: '+', remaining: 'abc' });
                expect(Parser.parseLeadingNumber('+0#5abc', { sep: '#', sign: true, complete: true })).to.equal(null);
                expect(Parser.parseLeadingNumber('12#5E3', { sep: '#' })).to.deep.equal({ number: 12500, text: '12#5E3', sign: '', remaining: '' });
                expect(Parser.parseLeadingNumber('12#5e003', { sep: '#' })).to.deep.equal({ number: 12500, text: '12#5e003', sign: '', remaining: '' });
                expect(Parser.parseLeadingNumber('12#5E+003', { sep: '#' })).to.deep.equal({ number: 12500, text: '12#5E+003', sign: '', remaining: '' });
                expect(Parser.parseLeadingNumber('12#5E-003', { sep: '#' })).to.deep.equal({ number: 0.0125, text: '12#5E-003', sign: '', remaining: '' });
            });
        });

        describe('method "parseFormatCode"', function () {
            it('should exist', function () {
                expect(Parser).itself.to.respondTo('parseFormatCode');
            });
            it('should cache the parsed format codes', function () {
                var format = Parser.parseFormatCode('General');
                expect(Parser.parseFormatCode('General')).to.equal(format);
            });
        });

        describe('method "generateCJKDateTimeFormat"', function () {
            it('should exist', function () {
                expect(Parser).itself.to.respondTo('generateCJKDateTimeFormat');
            });
            it('should convert the {H} meta token', function () {
                expect(Parser.generateCJKDateTimeFormat('{H}:mm:ss', 'traditional')).to.equal('h:mm:ss');
                expect(Parser.generateCJKDateTimeFormat('{H}:mm:ss', 'traditional', { ampm: true })).to.equal('AM/PMh:mm:ss');
                expect(Parser.generateCJKDateTimeFormat('{H}:mm:ss', 'traditional', { long: true })).to.equal('hh:mm:ss');
                expect(Parser.generateCJKDateTimeFormat('{H}:mm:ss', 'traditional', { ampm: true, long: true })).to.equal('AM/PMhh:mm:ss');
            });
            it('should convert the meta tokens for all scripts', function () {
                expect(Parser.generateCJKDateTimeFormat('YYYY{YEAR}MM{MONTH}DD{DAY} {H}{HOUR}:mm{MINUTE}:ss{SECOND}', 'traditional')).to.equal('YYYY\u5e74MM\u6708DD\u65e5 h\u6642:mm\u5206:ss\u79d2');
                expect(Parser.generateCJKDateTimeFormat('YYYY{YEAR}MM{MONTH}DD{DAY} {H}{HOUR}:mm{MINUTE}:ss{SECOND}', 'simplified')).to.equal('YYYY\u5e74MM\u6708DD\u65e5 h\u65f6:mm\u5206:ss\u79d2');
                expect(Parser.generateCJKDateTimeFormat('YYYY{YEAR}MM{MONTH}DD{DAY} {H}{HOUR}:mm{MINUTE}:ss{SECOND}', 'hangul')).to.equal('YYYY\ub144MM\uc6d4DD\uc77c h\uc2dc:mm\ubd84:ss\ucd08');
            });
        });
    });

    // ========================================================================
});
