/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/address',
    'io.ox/office/spreadsheet/utils/interval',
    'io.ox/office/spreadsheet/utils/range',
    'io.ox/office/spreadsheet/utils/range3d',
    'io.ox/office/spreadsheet/utils/addressarray',
    'io.ox/office/spreadsheet/utils/intervalarray',
    'io.ox/office/spreadsheet/utils/rangearray',
    'io.ox/office/spreadsheet/utils/range3darray',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (Address, Interval, Range, Range3D, AddressArray, IntervalArray, RangeArray, Range3DArray, SheetUtils) {

    'use strict';

    // static class SheetUtils ================================================

    describe('Spreadsheet module SheetUtils', function () {

        it('should exist', function () {
            expect(SheetUtils).to.be.an('object');
        });

        // helper classes -----------------------------------------------------

        describe('helper class Address', function () {
            it('should exist', function () {
                expect(SheetUtils).to.have.a.property('Address', Address);
            });
        });

        describe('helper class Interval', function () {
            it('should exist', function () {
                expect(SheetUtils).to.have.a.property('Interval', Interval);
            });
        });

        describe('helper class Range', function () {
            it('should exist', function () {
                expect(SheetUtils).to.have.a.property('Range', Range);
            });
        });

        describe('helper class Range3D', function () {
            it('should exist', function () {
                expect(SheetUtils).to.have.a.property('Range3D', Range3D);
            });
        });

        describe('helper class AddressArray', function () {
            it('should exist', function () {
                expect(SheetUtils).to.have.a.property('AddressArray', AddressArray);
            });
        });

        describe('helper class IntervalArray', function () {
            it('should exist', function () {
                expect(SheetUtils).to.have.a.property('IntervalArray', IntervalArray);
            });
        });

        describe('helper class RangeArray', function () {
            it('should exist', function () {
                expect(SheetUtils).to.have.a.property('RangeArray', RangeArray);
            });
        });

        describe('helper class RangeArray', function () {
            it('should exist', function () {
                expect(SheetUtils).to.have.a.property('Range3DArray', Range3DArray);
            });
        });

        // constants ----------------------------------------------------------

        describe('constant "MAX_NAME_LENGTH"', function () {
            it('should exist', function () {
                expect(SheetUtils.MAX_NAME_LENGTH).to.be.a('number').that.is.above(0);
            });
        });

        describe('constant "MAX_QUERY_CELL_COUNT"', function () {
            it('should exist', function () {
                expect(SheetUtils.MAX_QUERY_CELL_COUNT).to.be.a('number').that.is.above(0);
            });
        });

        describe('constant "MAX_FILL_CELL_COUNT"', function () {
            it('should exist', function () {
                expect(SheetUtils.MAX_FILL_CELL_COUNT).to.be.a('number').that.is.above(0);
            });
        });

        describe('constant "MAX_AUTOFILL_COL_ROW_COUNT"', function () {
            it('should exist', function () {
                expect(SheetUtils.MAX_AUTOFILL_COL_ROW_COUNT).to.be.a('number').that.is.above(0);
            });
        });

        describe('constant "MAX_MERGED_RANGES_COUNT"', function () {
            it('should exist', function () {
                expect(SheetUtils.MAX_MERGED_RANGES_COUNT).to.be.a('number').that.is.above(0);
            });
        });

        describe('constant "MAX_UNMERGE_CELL_COUNT"', function () {
            it('should exist', function () {
                expect(SheetUtils.MAX_UNMERGE_CELL_COUNT).to.be.a('number').that.is.above(0);
            });
        });

        describe('constant "MAX_CHANGE_ROWS_COUNT"', function () {
            it('should exist', function () {
                expect(SheetUtils.MAX_CHANGE_ROWS_COUNT).to.be.a('number').that.is.above(0);
            });
        });

        describe('constant "MAX_SELECTION_CELL_COUNT"', function () {
            it('should exist', function () {
                expect(SheetUtils.MAX_SELECTION_CELL_COUNT).to.be.a('number').that.is.above(0);
            });
        });

        describe('constant "MAX_SEARCH_RESULT_COUNT"', function () {
            it('should exist', function () {
                expect(SheetUtils.MAX_SEARCH_RESULT_COUNT).to.be.a('number').that.is.above(0);
            });
        });

        describe('constant "MAX_LENGTH_STANDARD_CELL"', function () {
            it('should exist', function () {
                expect(SheetUtils.MAX_LENGTH_STANDARD_CELL).to.be.a('number').that.is.above(0);
            });
        });

        describe('constant "MAX_LENGTH_STANDARD_EDIT"', function () {
            it('should exist', function () {
                expect(SheetUtils.MAX_LENGTH_STANDARD_EDIT).to.be.a('number').that.is.above(0);
            });
        });

        describe('constant "MAX_LENGTH_STANDARD_FORMULA"', function () {
            it('should exist', function () {
                expect(SheetUtils.MAX_LENGTH_STANDARD_FORMULA).to.be.a('number').that.is.above(0);
            });
        });

        describe('constant "MIN_CELL_SIZE"', function () {
            it('should exist', function () {
                expect(SheetUtils.MIN_CELL_SIZE).to.be.a('number').that.is.above(0);
            });
        });

        describe('constant "MIN_ZOOM"', function () {
            it('should exist', function () {
                expect(SheetUtils.MIN_ZOOM).to.be.a('number').that.is.above(0).and.below(1);
            });
        });

        describe('constant "MAX_ZOOM"', function () {
            it('should exist', function () {
                expect(SheetUtils.MAX_ZOOM).to.be.a('number').that.is.above(1);
            });
        });

        describe('constant "MULTI_SELECTION"', function () {
            it('should exist', function () {
                expect(SheetUtils.MULTI_SELECTION).to.be.a('boolean');
            });
        });

        describe('constant "ErrorCodes"', function () {
            it('should exist', function () {
                expect(SheetUtils.ErrorCodes).to.be.an('object');
            });
            it('should provide built-in error code #NULL!', function () {
                expect(SheetUtils.ErrorCodes.NULL).to.be.an('object');
                expect(SheetUtils.ErrorCodes.NULL).to.have.a.property('code', '#NULL!');
                expect(SheetUtils.ErrorCodes.NULL).to.have.a.property('num', 1);
            });
            it('should provide built-in error code #DIV/0!', function () {
                expect(SheetUtils.ErrorCodes.DIV0).to.be.an('object');
                expect(SheetUtils.ErrorCodes.DIV0).to.have.a.property('code', '#DIV/0!');
                expect(SheetUtils.ErrorCodes.DIV0).to.have.a.property('num', 2);
            });
            it('should provide built-in error code #VALUE!', function () {
                expect(SheetUtils.ErrorCodes.VALUE).to.be.an('object');
                expect(SheetUtils.ErrorCodes.VALUE).to.have.a.property('code', '#VALUE!');
                expect(SheetUtils.ErrorCodes.VALUE).to.have.a.property('num', 3);
            });
            it('should provide built-in error code #REF!', function () {
                expect(SheetUtils.ErrorCodes.REF).to.be.an('object');
                expect(SheetUtils.ErrorCodes.REF).to.have.a.property('code', '#REF!');
                expect(SheetUtils.ErrorCodes.REF).to.have.a.property('num', 4);
            });
            it('should provide built-in error code #NAME?', function () {
                expect(SheetUtils.ErrorCodes.NAME).to.be.an('object');
                expect(SheetUtils.ErrorCodes.NAME).to.have.a.property('code', '#NAME?');
                expect(SheetUtils.ErrorCodes.NAME).to.have.a.property('num', 5);
            });
            it('should provide built-in error code #NUM!', function () {
                expect(SheetUtils.ErrorCodes.NUM).to.be.an('object');
                expect(SheetUtils.ErrorCodes.NUM).to.have.a.property('code', '#NUM!');
                expect(SheetUtils.ErrorCodes.NUM).to.have.a.property('num', 6);
            });
            it('should provide built-in error code #N/A', function () {
                expect(SheetUtils.ErrorCodes.NA).to.be.an('object');
                expect(SheetUtils.ErrorCodes.NA).to.have.a.property('code', '#N/A');
                expect(SheetUtils.ErrorCodes.NA).to.have.a.property('num', 7);
            });
        });

        describe('class "ErrorCode"', function () {
            var NUM = SheetUtils.ErrorCodes.NUM;
            describe('method "equals"', function () {
                it('should exist', function () {
                    expect(NUM).to.respondTo('equals');
                });
                it('should return true for same error code', function () {
                    expect(NUM.equals(NUM)).to.equal(true);
                });
                it('should return false for other error code', function () {
                    expect(NUM.equals(SheetUtils.ErrorCodes.NULL)).to.equal(false);
                });
            });
            describe('method "toJSON"', function () {
                it('should exist', function () {
                    expect(NUM).to.respondTo('toJSON');
                });
                it('should convert error code to the string representation', function () {
                    expect(JSON.stringify(NUM)).to.equal('"' + NUM.code + '"');
                });
            });
        });

        // static methods -----------------------------------------------------

        describe('method "isErrorCode"', function () {
            it('should exist', function () {
                expect(SheetUtils).itself.to.respondTo('isErrorCode');
            });
            it('should return true for error codes', function () {
                expect(SheetUtils.isErrorCode(SheetUtils.ErrorCodes.NULL)).to.equal(true);
                expect(SheetUtils.isErrorCode(SheetUtils.ErrorCodes.NUM)).to.equal(true);
            });
            it('should return false for other values', function () {
                expect(SheetUtils.isErrorCode(0)).to.equal(false);
                expect(SheetUtils.isErrorCode('a')).to.equal(false);
                expect(SheetUtils.isErrorCode('#NULL!')).to.equal(false);
                expect(SheetUtils.isErrorCode('#NUM!')).to.equal(false);
                expect(SheetUtils.isErrorCode(true)).to.equal(false);
                expect(SheetUtils.isErrorCode(null)).to.equal(false);
            });
        });

        describe('method "makeErrorCode"', function () {
            it('should exist', function () {
                expect(SheetUtils).itself.to.respondTo('makeErrorCode');
            });
            var numError = SheetUtils.makeErrorCode('#NUM!');
            it('should return error code objects', function () {
                expect(numError).to.be.an('object');
                expect(numError).to.have.a.property('code', '#NUM!');
                expect(numError.toString()).to.equal('#NUM!');
                expect(numError).to.have.a.property('num', 6);
                expect(numError).to.respondTo('equals');
                expect(numError.equals(SheetUtils.ErrorCodes.NUM)).to.equal(true);
                expect(numError.equals(SheetUtils.ErrorCodes.NULL)).to.equal(false);
            });
            var unknownError = SheetUtils.makeErrorCode('UNKNOWN');
            it('should return NAN for unknown ErrorNumber', function () {
                expect(unknownError).to.be.an('object');
                expect(unknownError).to.have.a.property('code', 'UNKNOWN');
            });
        });

        describe('method "getSheetName"', function () {
            it('should exist', function () {
                expect(SheetUtils).itself.to.respondTo('getSheetName');
            });
            it('should return the English sheet name', function () {
                expect(SheetUtils.getSheetName(0)).to.equal('Sheet1');
                expect(SheetUtils.getSheetName(1)).to.equal('Sheet2');
                expect(SheetUtils.getSheetName(9)).to.equal('Sheet10');
            });
        });

        describe('method "getTableColName"', function () {
            it('should exist', function () {
                expect(SheetUtils).itself.to.respondTo('getTableColName');
            });
            it('should return the English column name', function () {
                expect(SheetUtils.getTableColName()).to.equal('Column');
            });
        });
    });

    // ========================================================================
});
