/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/model/formula/matrix',
    'io.ox/office/spreadsheet/model/formula/tokens'
], function (Matrix, Tokens) {

    'use strict';

    // global helpers =========================================================

    // a fake grammar configuration for native formulas
    var opConfig = {
        DEC: '.',
        SEP: ',',
        OPERATORS: { list: ',' },
        MAT_ROW: ';',
        MAT_COL: ','
    };

    // a fake grammar configuration for UI formulas
    var uiConfig = {
        DEC: ',',
        SEP: ';',
        OPERATORS: { list: '~' },
        MAT_ROW: '|',
        MAT_COL: ';'
    };

    // static class Tokens ====================================================

    describe('Spreadsheet module Tokens', function () {
        it('should exist', function () {
            expect(Tokens).to.be.an('object');
        });
    });

    // class FixedToken =======================================================

    describe('Spreadsheet class FixedToken', function () {
        it('should exist', function () {
            expect(Tokens.FixedToken).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var token = new Tokens.FixedToken('type', 'text');

        // public methods -----------------------------------------------------

        describe('method "getType"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('getType');
            });
            it('should return the token type', function () {
                expect(token.getType()).to.equal('type');
            });
        });

        describe('method "isType"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('isType');
            });
            it('should test the token type', function () {
                expect(token.isType('type')).to.equal(true);
                expect(token.isType('other')).to.equal(false);
            });
        });

        describe('method "matchesType"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('matchesType');
            });
            it('should match the token type', function () {
                expect(token.matchesType(/^t/)).to.equal(true);
                expect(token.matchesType(/^o/)).to.equal(false);
            });
        });

        describe('method "getText"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('getText');
            });
            it('should return the string representation', function () {
                expect(token.getText(opConfig)).to.equal('text');
                expect(token.getText(uiConfig)).to.equal('text');
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('toString');
            });
            it('should return the string representation', function () {
                expect(token.toString()).to.equal('type[text]');
            });
        });

        describe('method "getValue"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('getValue');
            });
            it('should return the value', function () {
                expect(token.getValue()).to.equal('text');
            });
        });

        describe('method "setValue"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('setValue');
            });
            it('should return whether the value changes', function () {
                expect(token.setValue('text')).to.equal(false);
                expect(token.getValue()).to.equal('text');
                expect(token.setValue('other')).to.equal(true);
                expect(token.getValue()).to.equal('other');
            });
            it('should return new value from method "getText"', function () {
                expect(token.getText(opConfig)).to.equal('other');
                expect(token.getText(uiConfig)).to.equal('other');
            });
        });

        describe('method "appendValue"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('appendValue');
            });
            it('should return whether the value changes', function () {
                expect(token.appendValue('')).to.equal(false);
                expect(token.getValue()).to.equal('other');
                expect(token.appendValue('text')).to.equal(true);
                expect(token.getValue()).to.equal('othertext');
            });
            it('should return new value from method "getText"', function () {
                expect(token.getText(opConfig)).to.equal('othertext');
                expect(token.getText(uiConfig)).to.equal('othertext');
            });
        });
    });

    // class OperatorToken ====================================================

    describe('Spreadsheet class OperatorToken', function () {
        it('should exist', function () {
            expect(Tokens.OperatorToken).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var token = new Tokens.OperatorToken('list');

        // public methods -----------------------------------------------------

        describe('method "getType"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('getType');
            });
            it('should return the token type', function () {
                expect(token.getType()).to.equal('op');
            });
        });

        describe('method "getText"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('getText');
            });
            it('should return the string representation', function () {
                expect(token.getText(opConfig)).to.equal(',');
                expect(token.getText(uiConfig)).to.equal('~');
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('toString');
            });
            it('should return the string representation', function () {
                expect(token.toString()).to.equal('op[list]');
            });
        });

        describe('method "getValue"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('getValue');
            });
            it('should return the value', function () {
                expect(token.getValue()).to.equal('list');
            });
        });
    });

    // class SeparatorToken ===================================================

    describe('Spreadsheet class SeparatorToken', function () {
        it('should exist', function () {
            expect(Tokens.SeparatorToken).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var token = new Tokens.SeparatorToken();

        // public methods -----------------------------------------------------

        describe('method "getType"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('getType');
            });
            it('should return the token type', function () {
                expect(token.getType()).to.equal('sep');
            });
        });

        describe('method "getText"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('getText');
            });
            it('should return the string representation', function () {
                expect(token.getText(opConfig)).to.equal(',');
                expect(token.getText(uiConfig)).to.equal(';');
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('toString');
            });
            it('should return the string representation', function () {
                expect(token.toString()).to.equal('sep[,]');
            });
        });
    });

    // class ParenthesisToken =================================================

    describe('Spreadsheet class ParenthesisToken', function () {
        it('should exist', function () {
            expect(Tokens.ParenthesisToken).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var openToken = new Tokens.ParenthesisToken(true),
            closeToken = new Tokens.ParenthesisToken(false);

        // public methods -----------------------------------------------------

        describe('method "getType"', function () {
            it('should exist', function () {
                expect(openToken).to.respondTo('getType');
            });
            it('should return the token type', function () {
                expect(openToken.getType()).to.equal('open');
                expect(closeToken.getType()).to.equal('close');
            });
        });

        describe('method "getText"', function () {
            it('should exist', function () {
                expect(openToken).to.respondTo('getText');
            });
            it('should return the string representation', function () {
                expect(openToken.getText(opConfig)).to.equal('(');
                expect(openToken.getText(uiConfig)).to.equal('(');
                expect(closeToken.getText(opConfig)).to.equal(')');
                expect(closeToken.getText(uiConfig)).to.equal(')');
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(openToken).to.respondTo('toString');
            });
            it('should return the string representation', function () {
                expect(openToken.toString()).to.equal('open[(]');
                expect(closeToken.toString()).to.equal('close[)]');
            });
        });
    });

    // class MatrixDelimiterToken =============================================

    describe('Spreadsheet class MatrixDelimiterToken', function () {
        it('should exist', function () {
            expect(Tokens.MatrixDelimiterToken).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var openToken = new Tokens.MatrixDelimiterToken('mat_open'),
            rowToken = new Tokens.MatrixDelimiterToken('mat_row'),
            colToken = new Tokens.MatrixDelimiterToken('mat_col'),
            closeToken = new Tokens.MatrixDelimiterToken('mat_close');

        // public methods -----------------------------------------------------

        describe('method "getType"', function () {
            it('should exist', function () {
                expect(openToken).to.respondTo('getType');
            });
            it('should return the token type', function () {
                expect(openToken.getType()).to.equal('mat_open');
                expect(rowToken.getType()).to.equal('mat_row');
                expect(colToken.getType()).to.equal('mat_col');
                expect(closeToken.getType()).to.equal('mat_close');
            });
        });

        describe('method "getText"', function () {
            it('should exist', function () {
                expect(openToken).to.respondTo('getText');
            });
            it('should return the string representation', function () {
                expect(openToken.getText(opConfig)).to.equal('{');
                expect(openToken.getText(uiConfig)).to.equal('{');
                expect(rowToken.getText(opConfig)).to.equal(';');
                expect(rowToken.getText(uiConfig)).to.equal('|');
                expect(colToken.getText(opConfig)).to.equal(',');
                expect(colToken.getText(uiConfig)).to.equal(';');
                expect(closeToken.getText(opConfig)).to.equal('}');
                expect(closeToken.getText(uiConfig)).to.equal('}');
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(openToken).to.respondTo('toString');
            });
            it('should return the string representation', function () {
                expect(openToken.toString()).to.equal('mat_open[{]');
                expect(rowToken.toString()).to.equal('mat_row[|]');
                expect(colToken.toString()).to.equal('mat_col[;]');
                expect(closeToken.toString()).to.equal('mat_close[}]');
            });
        });
    });

    // class LiteralToken =====================================================

    describe('Spreadsheet class LiteralToken', function () {
        it('should exist', function () {
            expect(Tokens.LiteralToken).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var token = new Tokens.LiteralToken(12.5);

        // public methods -----------------------------------------------------

        describe('method "getType"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('getType');
            });
            it('should return the token type', function () {
                expect(token.getType()).to.equal('lit');
            });
        });

        describe('method "getText"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('getText');
            });
            it('should return the string representation', function () {
                // TODO
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('toString');
            });
            it('should return the string representation', function () {
                expect(token.toString()).to.equal('lit[12.5]');
            });
        });

        describe('method "getValue"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('getValue');
            });
            it('should return the value', function () {
                expect(token.getValue()).to.equal(12.5);
            });
        });

        describe('method "setValue"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('setValue');
            });
            it('should return whether the value changes', function () {
                expect(token.setValue(12.5)).to.equal(false);
                expect(token.getValue()).to.equal(12.5);
                expect(token.setValue(1)).to.equal(true);
                expect(token.getValue()).to.equal(1);
                expect(token.setValue('1')).to.equal(true);
                expect(token.getValue()).to.equal('1');
                expect(token.setValue(true)).to.equal(true);
                expect(token.getValue()).to.equal(true);
                expect(token.setValue(null)).to.equal(true);
                expect(token.getValue()).to.equal(null);
            });
            it('should return new value from method "getText"', function () {
                // TODO
            });
        });
    });

    // class MatrixToken ======================================================

    describe('Spreadsheet class MatrixToken', function () {
        it('should exist', function () {
            expect(Tokens.MatrixToken).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var matrix = new Matrix([[12.5, '1'], [true, null]]);
        var token = new Tokens.MatrixToken(matrix);

        // public methods -----------------------------------------------------

        describe('method "getType"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('getType');
            });
            it('should return the token type', function () {
                expect(token.getType()).to.equal('mat');
            });
        });

        describe('method "getText"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('getText');
            });
            it('should return the string representation', function () {
                // TODO
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('toString');
            });
            it('should return the string representation', function () {
                expect(token.toString()).to.equal('mat[{12.5;"1"|TRUE;null}]');
            });
        });

        describe('method "getMatrix"', function () {
            it('should exist', function () {
                expect(token).to.respondTo('getMatrix');
            });
            it('should return the matrix', function () {
                expect(token.getMatrix()).to.equal(matrix);
            });
        });
    });

    // class ReferenceToken ===================================================

    describe('Spreadsheet class ReferenceToken', function () {
        it('should exist', function () {
            expect(Tokens.ReferenceToken).to.be.a('function');
        });
    });

    // class NameToken ========================================================

    describe('Spreadsheet class NameToken', function () {
        it('should exist', function () {
            expect(Tokens.NameToken).to.be.a('function');
        });
    });

    // ========================================================================
});
