/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/matrix',
    'io.ox/office/spreadsheet/model/formula/impl/logicalfuncs'
], function (SheetUtils, Matrix, LogicalFuncs) {

    'use strict';

    // module LogicalFuncs ====================================================

    describe('Spreadsheet module LogicalFuncs', function () {

        it('should exist', function () {
            expect(LogicalFuncs).to.be.an('object');
        });

        // private helpers ----------------------------------------------------

        var ErrorCodes = SheetUtils.ErrorCodes,
            Address = SheetUtils.Address,
            Range3D = SheetUtils.Range3D,
            Range3DArray = SheetUtils.Range3DArray;

        var mat = new Matrix([[1]]),
            ranges = new Range3DArray(new Range3D(0, 0, new Address(0, 0)));

        // simplified logical aggregator for testing purposes
        function aggregateBooleans(operands, initial, aggregate, finalize) {

            var // the intermediate result
                result = initial,
                // the count of all visited Boolean values in all operands
                count = 0;

            // process all function parameters
            _.each(operands, function (operand) {
                result = aggregate.call(this, result, operand);
                count += 1;
            }, this);

            // resolve the final result, throw error codes
            result = finalize.call(this, result, count);
            if (_.isBoolean(result)) { return result; }
            throw result;
        }

        var // a formula context stub for logical aggregation
            context = { aggregateBooleans: aggregateBooleans };

        // function implementations -------------------------------------------

        describe('function "AND"', function () {
            var AND = LogicalFuncs.AND;
            it('should exist', function () {
                expect(AND).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(AND).itself.to.respondTo('resolve');
            });
            it('should return the logical AND of the arguments', function () {
                expect(AND.resolve.call(context, false)).to.equal(false);
                expect(AND.resolve.call(context, true)).to.equal(true);
                expect(AND.resolve.call(context, false, false)).to.equal(false);
                expect(AND.resolve.call(context, false, true)).to.equal(false);
                expect(AND.resolve.call(context, true, false)).to.equal(false);
                expect(AND.resolve.call(context, true, true)).to.equal(true);
                expect(AND.resolve.call(context, false, false, false)).to.equal(false);
                expect(AND.resolve.call(context, false, false, true)).to.equal(false);
                expect(AND.resolve.call(context, false, true, false)).to.equal(false);
                expect(AND.resolve.call(context, false, true, true)).to.equal(false);
                expect(AND.resolve.call(context, true, false, false)).to.equal(false);
                expect(AND.resolve.call(context, true, false, true)).to.equal(false);
                expect(AND.resolve.call(context, true, true, false)).to.equal(false);
                expect(AND.resolve.call(context, true, true, true)).to.equal(true);
            });
        });

        describe('function "FALSE"', function () {
            var FALSE = LogicalFuncs.FALSE;
            it('should exist', function () {
                expect(FALSE).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(FALSE).itself.to.respondTo('resolve');
            });
            it('should return FALSE', function () {
                expect(FALSE.resolve()).to.equal(false);
            });
        });

        describe('function "IF"', function () {
            var IF = LogicalFuncs.IF;
            it('should exist', function () {
                expect(IF).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(IF).itself.to.respondTo('resolve');
            });
            it('should return first parameter, if condition is TRUE', function () {
                expect(IF.resolve(true, 2, 'x')).to.equal(2);
                expect(IF.resolve(true, 'a', 'x')).to.equal('a');
                expect(IF.resolve(true, false, 'x')).to.equal(false);
                expect(IF.resolve(true, null, 'x')).to.equal(null);
                expect(IF.resolve(true, mat, 'x')).to.equal(mat);
                expect(IF.resolve(true, ranges, 'x')).to.equal(ranges);
            });
            it('should return second parameter, if condition is FALSE', function () {
                expect(IF.resolve(false, 'x', 2)).to.equal(2);
                expect(IF.resolve(false, 'x', 'a')).to.equal('a');
                expect(IF.resolve(false, 'x', true)).to.equal(true);
                expect(IF.resolve(false, 'x', null)).to.equal(null);
                expect(IF.resolve(false, 'x', mat)).to.equal(mat);
                expect(IF.resolve(false, 'x', ranges)).to.equal(ranges);
            });
            it('should return FALSE, if second parameter is missing, and condition is FALSE', function () {
                expect(IF.resolve(true, 2)).to.equal(2);
                expect(IF.resolve(false, 2)).to.equal(false);
            });
        });

        describe('function "IFERROR"', function () {
            var IFERROR = LogicalFuncs.IFERROR;
            it('should exist', function () {
                expect(IFERROR).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(IFERROR).itself.to.respondTo('resolve');
            });
            it('should return the first parameter, if not an error code', function () {
                expect(IFERROR.resolve(2, 'x')).to.equal(2);
                expect(IFERROR.resolve('a', 'x')).to.equal('a');
                expect(IFERROR.resolve(false, 'x')).to.equal(false);
            });
            it('should return the second parameter, if first is an error code', function () {
                expect(IFERROR.resolve(ErrorCodes.DIV0, 2)).to.equal(2);
                expect(IFERROR.resolve(ErrorCodes.NA, 'a')).to.equal('a');
                expect(IFERROR.resolve(ErrorCodes.NAME, false)).to.equal(false);
                expect(IFERROR.resolve(ErrorCodes.NULL, 2)).to.equal(2);
                expect(IFERROR.resolve(ErrorCodes.NUM, 'a')).to.equal('a');
                expect(IFERROR.resolve(ErrorCodes.REF, false)).to.equal(false);
                expect(IFERROR.resolve(ErrorCodes.VALUE, 2)).to.equal(2);
                expect(IFERROR.resolve(ErrorCodes.DIV0, ErrorCodes.NAME)).to.equal(ErrorCodes.NAME);
            });
        });

        describe('function "IFNA"', function () {
            var IFNA = LogicalFuncs.IFNA;
            it('should exist', function () {
                expect(IFNA).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(IFNA).itself.to.respondTo('resolve');
            });
            it('should return the first parameter, if not the #N/A error code', function () {
                expect(IFNA.resolve(2, 'x')).to.equal(2);
                expect(IFNA.resolve('a', 'x')).to.equal('a');
                expect(IFNA.resolve(false, 'x')).to.equal(false);
                expect(IFNA.resolve(ErrorCodes.DIV0, 'x')).to.equal(ErrorCodes.DIV0);
                expect(IFNA.resolve(ErrorCodes.NAME, 'x')).to.equal(ErrorCodes.NAME);
                expect(IFNA.resolve(ErrorCodes.NULL, 'x')).to.equal(ErrorCodes.NULL);
                expect(IFNA.resolve(ErrorCodes.NUM, 'x')).to.equal(ErrorCodes.NUM);
                expect(IFNA.resolve(ErrorCodes.REF, 'x')).to.equal(ErrorCodes.REF);
                expect(IFNA.resolve(ErrorCodes.VALUE, 'x')).to.equal(ErrorCodes.VALUE);
            });
            it('should return the second parameter, if first is the #N/A error code', function () {
                expect(IFNA.resolve(ErrorCodes.NA, 2)).to.equal(2);
                expect(IFNA.resolve(ErrorCodes.NA, 'a')).to.equal('a');
                expect(IFNA.resolve(ErrorCodes.NA, false)).to.equal(false);
                expect(IFNA.resolve(ErrorCodes.NA, ErrorCodes.NAME)).to.equal(ErrorCodes.NAME);
            });
        });

        describe('function "NOT"', function () {
            var NOT = LogicalFuncs.NOT;
            it('should exist', function () {
                expect(NOT).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(NOT).itself.to.respondTo('resolve');
            });
            it('should return the negated argument', function () {
                expect(NOT.resolve(false)).to.equal(true);
                expect(NOT.resolve(true)).to.equal(false);
            });
        });

        describe('function "OR"', function () {
            var OR = LogicalFuncs.OR;
            it('should exist', function () {
                expect(OR).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(OR).itself.to.respondTo('resolve');
            });
            it('should return the logical OR of the arguments', function () {
                expect(OR.resolve.call(context, false)).to.equal(false);
                expect(OR.resolve.call(context, true)).to.equal(true);
                expect(OR.resolve.call(context, false, false)).to.equal(false);
                expect(OR.resolve.call(context, false, true)).to.equal(true);
                expect(OR.resolve.call(context, true, false)).to.equal(true);
                expect(OR.resolve.call(context, true, true)).to.equal(true);
                expect(OR.resolve.call(context, false, false, false)).to.equal(false);
                expect(OR.resolve.call(context, false, false, true)).to.equal(true);
                expect(OR.resolve.call(context, false, true, false)).to.equal(true);
                expect(OR.resolve.call(context, false, true, true)).to.equal(true);
                expect(OR.resolve.call(context, true, false, false)).to.equal(true);
                expect(OR.resolve.call(context, true, false, true)).to.equal(true);
                expect(OR.resolve.call(context, true, true, false)).to.equal(true);
                expect(OR.resolve.call(context, true, true, true)).to.equal(true);
            });
        });

        describe('function "TRUE"', function () {
            var TRUE = LogicalFuncs.TRUE;
            it('should exist', function () {
                expect(TRUE).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(TRUE).itself.to.respondTo('resolve');
            });
            it('should return TRUE', function () {
                expect(TRUE.resolve()).to.equal(true);
            });
        });

        describe('function "XOR"', function () {
            var XOR = LogicalFuncs.XOR;
            it('should exist', function () {
                expect(XOR).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(XOR).itself.to.respondTo('resolve');
            });
            it('should return the logical XOR of the arguments', function () {
                expect(XOR.resolve.call(context, false)).to.equal(false);
                expect(XOR.resolve.call(context, true)).to.equal(true);
                expect(XOR.resolve.call(context, false, false)).to.equal(false);
                expect(XOR.resolve.call(context, false, true)).to.equal(true);
                expect(XOR.resolve.call(context, true, false)).to.equal(true);
                expect(XOR.resolve.call(context, true, true)).to.equal(false);
                expect(XOR.resolve.call(context, false, false, false)).to.equal(false);
                expect(XOR.resolve.call(context, false, false, true)).to.equal(true);
                expect(XOR.resolve.call(context, false, true, false)).to.equal(true);
                expect(XOR.resolve.call(context, false, true, true)).to.equal(false);
                expect(XOR.resolve.call(context, true, false, false)).to.equal(true);
                expect(XOR.resolve.call(context, true, false, true)).to.equal(false);
                expect(XOR.resolve.call(context, true, true, false)).to.equal(false);
                expect(XOR.resolve.call(context, true, true, true)).to.equal(true);
            });
        });
    });

    // ========================================================================
});
