/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/cellcollection'
], function (SheetUtils, CellCollection) {

    'use strict';

    // class CellCollection ===================================================

    describe('Spreadsheet class CellCollection', function () {

        it('should exist', function () {
            expect(CellCollection).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var NUMBER_CELL_DATA = { result: 2, display: '2' },
            STRING_CELL_DATA = { result: 'a', display: 'a' },
            BOOLEAN_CELL_DATA = { result: true, display: 'TRUE' },
            ERROR_CELL_DATA = { result: SheetUtils.ErrorCodes.VALUE, display: '#VALUE!' },
            BLANK_CELL_DATA = { result: null, display: '' };

        // constants ----------------------------------------------------------

        describe('constant "DEFAULT_CELL_DATA"', function () {
            it('should exist', function () {
                expect(CellCollection.DEFAULT_CELL_DATA).to.be.an('object');
            });
        });

        describe('constant "PENDING_DISPLAY"', function () {
            it('should exist', function () {
                expect(CellCollection.PENDING_DISPLAY).to.be.a('string');
            });
        });

        // static methods -----------------------------------------------------

        describe('method "isBlank"', function () {
            it('should exist', function () {
                expect(CellCollection).itself.to.respondTo('isBlank');
            });
            it('should return false for non-blank cells', function () {
                expect(CellCollection.isBlank(NUMBER_CELL_DATA)).to.equal(false);
                expect(CellCollection.isBlank(STRING_CELL_DATA)).to.equal(false);
                expect(CellCollection.isBlank(BOOLEAN_CELL_DATA)).to.equal(false);
                expect(CellCollection.isBlank(ERROR_CELL_DATA)).to.equal(false);
            });
            it('should return true for blank cells and missing cell data', function () {
                expect(CellCollection.isBlank(BLANK_CELL_DATA)).to.equal(true);
                expect(CellCollection.isBlank(null)).to.equal(true);
            });
        });

        describe('method "isNumber"', function () {
            it('should exist', function () {
                expect(CellCollection).itself.to.respondTo('isNumber');
            });
            it('should return false for non-numeric cells and missing cell data', function () {
                expect(CellCollection.isNumber(STRING_CELL_DATA)).to.equal(false);
                expect(CellCollection.isNumber(BOOLEAN_CELL_DATA)).to.equal(false);
                expect(CellCollection.isNumber(ERROR_CELL_DATA)).to.equal(false);
                expect(CellCollection.isNumber(BLANK_CELL_DATA)).to.equal(false);
                expect(CellCollection.isNumber(null)).to.equal(false);
            });
            it('should return true for number cells', function () {
                expect(CellCollection.isNumber(NUMBER_CELL_DATA)).to.equal(true);
            });
        });

        describe('method "isBoolean"', function () {
            it('should exist', function () {
                expect(CellCollection).itself.to.respondTo('isBoolean');
            });
            it('should return false for non-boolean cells and missing cell data', function () {
                expect(CellCollection.isBoolean(NUMBER_CELL_DATA)).to.equal(false);
                expect(CellCollection.isBoolean(STRING_CELL_DATA)).to.equal(false);
                expect(CellCollection.isBoolean(ERROR_CELL_DATA)).to.equal(false);
                expect(CellCollection.isBoolean(BLANK_CELL_DATA)).to.equal(false);
                expect(CellCollection.isBoolean(null)).to.equal(false);
            });
            it('should return true for Boolean cells', function () {
                expect(CellCollection.isBoolean(BOOLEAN_CELL_DATA)).to.equal(true);
            });
        });

        describe('method "isError"', function () {
            it('should exist', function () {
                expect(CellCollection).itself.to.respondTo('isError');
            });
            it('should return false for non-error cells and missing cell data', function () {
                expect(CellCollection.isError(NUMBER_CELL_DATA)).to.equal(false);
                expect(CellCollection.isError(STRING_CELL_DATA)).to.equal(false);
                expect(CellCollection.isError(BOOLEAN_CELL_DATA)).to.equal(false);
                expect(CellCollection.isError(BLANK_CELL_DATA)).to.equal(false);
                expect(CellCollection.isError(null)).to.equal(false);
            });
            it('should return true for error cells', function () {
                expect(CellCollection.isError(ERROR_CELL_DATA)).to.equal(true);
            });
        });

        describe('method "isText"', function () {
            it('should exist', function () {
                expect(CellCollection).itself.to.respondTo('isText');
            });
            it('should return false for non-text cells and missing cell data', function () {
                expect(CellCollection.isText(NUMBER_CELL_DATA)).to.equal(false);
                expect(CellCollection.isText(BOOLEAN_CELL_DATA)).to.equal(false);
                expect(CellCollection.isText(ERROR_CELL_DATA)).to.equal(false);
                expect(CellCollection.isText(BLANK_CELL_DATA)).to.equal(false);
                expect(CellCollection.isText(null)).to.equal(false);
            });
            it('should return true for error cells', function () {
                expect(CellCollection.isText(STRING_CELL_DATA)).to.equal(true);
            });
        });

        describe('method "getTextOrientation"', function () {

            var LEFT_ALIGN_ATTRS = { attributes: { cell: { alignHor: 'left' } } },
                CENTER_ALIGN_ATTRS = { attributes: { cell: { alignHor: 'center' } } },
                RIGHT_ALIGN_ATTRS = { attributes: { cell: { alignHor: 'right' } } },
                JUSTIFY_ALIGN_ATTRS = { attributes: { cell: { alignHor: 'justify' } } },
                AUTO_ALIGN_ATTRS = { attributes: { cell: { alignHor: 'auto' } } };

            function getTextAlignment(cellData) {
                return CellCollection.getTextOrientation(cellData).cssTextAlign;
            }

            it('should exist', function () {
                expect(CellCollection).itself.to.respondTo('getTextOrientation');
            });
            it('should return the correct cell text alignment for number cells', function () {
                expect(getTextAlignment(_.extend({}, NUMBER_CELL_DATA, LEFT_ALIGN_ATTRS))).to.equal('left');
                expect(getTextAlignment(_.extend({}, NUMBER_CELL_DATA, CENTER_ALIGN_ATTRS))).to.equal('center');
                expect(getTextAlignment(_.extend({}, NUMBER_CELL_DATA, RIGHT_ALIGN_ATTRS))).to.equal('right');
                expect(getTextAlignment(_.extend({}, NUMBER_CELL_DATA, JUSTIFY_ALIGN_ATTRS))).to.equal('justify');
                expect(getTextAlignment(_.extend({}, NUMBER_CELL_DATA, AUTO_ALIGN_ATTRS))).to.equal('right');
                expect(getTextAlignment(_.extend({ formula: '=1' }, NUMBER_CELL_DATA, LEFT_ALIGN_ATTRS))).to.equal('left');
                expect(getTextAlignment(_.extend({ formula: '=1' }, NUMBER_CELL_DATA, CENTER_ALIGN_ATTRS))).to.equal('center');
                expect(getTextAlignment(_.extend({ formula: '=1' }, NUMBER_CELL_DATA, RIGHT_ALIGN_ATTRS))).to.equal('right');
                expect(getTextAlignment(_.extend({ formula: '=1' }, NUMBER_CELL_DATA, JUSTIFY_ALIGN_ATTRS))).to.equal('justify');
                expect(getTextAlignment(_.extend({ formula: '=1' }, NUMBER_CELL_DATA, AUTO_ALIGN_ATTRS))).to.equal('right');
            });
            it('should return the correct cell text alignment for string cells', function () {
                expect(getTextAlignment(_.extend({}, STRING_CELL_DATA, LEFT_ALIGN_ATTRS))).to.equal('left');
                expect(getTextAlignment(_.extend({}, STRING_CELL_DATA, CENTER_ALIGN_ATTRS))).to.equal('center');
                expect(getTextAlignment(_.extend({}, STRING_CELL_DATA, RIGHT_ALIGN_ATTRS))).to.equal('right');
                expect(getTextAlignment(_.extend({}, STRING_CELL_DATA, JUSTIFY_ALIGN_ATTRS))).to.equal('justify');
                expect(getTextAlignment(_.extend({}, STRING_CELL_DATA, AUTO_ALIGN_ATTRS))).to.equal('left');
            });
            it('should return the correct cell text alignment for Boolean cells', function () {
                expect(getTextAlignment(_.extend({}, BOOLEAN_CELL_DATA, LEFT_ALIGN_ATTRS))).to.equal('left');
                expect(getTextAlignment(_.extend({}, BOOLEAN_CELL_DATA, CENTER_ALIGN_ATTRS))).to.equal('center');
                expect(getTextAlignment(_.extend({}, BOOLEAN_CELL_DATA, RIGHT_ALIGN_ATTRS))).to.equal('right');
                expect(getTextAlignment(_.extend({}, BOOLEAN_CELL_DATA, JUSTIFY_ALIGN_ATTRS))).to.equal('justify');
                expect(getTextAlignment(_.extend({}, BOOLEAN_CELL_DATA, AUTO_ALIGN_ATTRS))).to.equal('center');
            });
            it('should return the correct cell text alignment for error cells', function () {
                expect(getTextAlignment(_.extend({}, ERROR_CELL_DATA, LEFT_ALIGN_ATTRS))).to.equal('left');
                expect(getTextAlignment(_.extend({}, ERROR_CELL_DATA, CENTER_ALIGN_ATTRS))).to.equal('center');
                expect(getTextAlignment(_.extend({}, ERROR_CELL_DATA, RIGHT_ALIGN_ATTRS))).to.equal('right');
                expect(getTextAlignment(_.extend({}, ERROR_CELL_DATA, JUSTIFY_ALIGN_ATTRS))).to.equal('justify');
                expect(getTextAlignment(_.extend({}, ERROR_CELL_DATA, AUTO_ALIGN_ATTRS))).to.equal('center');
            });
            it('should return the correct cell text alignment for blank cells', function () {
                expect(getTextAlignment(_.extend({}, BLANK_CELL_DATA, LEFT_ALIGN_ATTRS))).to.equal('left');
                expect(getTextAlignment(_.extend({}, BLANK_CELL_DATA, CENTER_ALIGN_ATTRS))).to.equal('center');
                expect(getTextAlignment(_.extend({}, BLANK_CELL_DATA, RIGHT_ALIGN_ATTRS))).to.equal('right');
                expect(getTextAlignment(_.extend({}, BLANK_CELL_DATA, JUSTIFY_ALIGN_ATTRS))).to.equal('justify');
                expect(getTextAlignment(_.extend({}, BLANK_CELL_DATA, AUTO_ALIGN_ATTRS))).to.equal('left');
            });
            it('should accept missing cell data', function () {
                expect(getTextAlignment(null)).to.match(/^(left|right)$/);
            });
        });

        describe('method "hasWrappingAttributes"', function () {
            it('should exist', function () {
                expect(CellCollection).itself.to.respondTo('hasWrappingAttributes');
            });
            it('should return true for existing "wrapText" attribute', function () {
                expect(CellCollection.hasWrappingAttributes({ attributes: { cell: { wrapText: false, alignHor: 'left', alignVert: 'top' } } })).to.equal(false);
                expect(CellCollection.hasWrappingAttributes({ attributes: { cell: { wrapText: true, alignHor: 'left', alignVert: 'top' } } })).to.equal(true);
            });
            it('should return true for horizontally justified alignment', function () {
                expect(CellCollection.hasWrappingAttributes({ attributes: { cell: { wrapText: false, alignHor: 'left', alignVert: 'top' } } })).to.equal(false);
                expect(CellCollection.hasWrappingAttributes({ attributes: { cell: { wrapText: false, alignHor: 'center', alignVert: 'top' } } })).to.equal(false);
                expect(CellCollection.hasWrappingAttributes({ attributes: { cell: { wrapText: false, alignHor: 'right', alignVert: 'top' } } })).to.equal(false);
                expect(CellCollection.hasWrappingAttributes({ attributes: { cell: { wrapText: false, alignHor: 'justify', alignVert: 'top' } } })).to.equal(true);
            });
            it('should return true for vertically justified alignment', function () {
                expect(CellCollection.hasWrappingAttributes({ attributes: { cell: { wrapText: false, alignHor: 'left', alignVert: 'top' } } })).to.equal(false);
                expect(CellCollection.hasWrappingAttributes({ attributes: { cell: { wrapText: false, alignHor: 'left', alignVert: 'middle' } } })).to.equal(false);
                expect(CellCollection.hasWrappingAttributes({ attributes: { cell: { wrapText: false, alignHor: 'left', alignVert: 'bottom' } } })).to.equal(false);
                expect(CellCollection.hasWrappingAttributes({ attributes: { cell: { wrapText: false, alignHor: 'left', alignVert: 'justify' } } })).to.equal(true);
            });
            it('should accept missing cell data', function () {
                expect(CellCollection.hasWrappingAttributes(null)).to.equal(false);
            });
        });

        // public methods -----------------------------------------------------

    });

    // ========================================================================
});
