/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/color'
], function (Utils, Color) {

    'use strict';

    // class Color ============================================================

    describe('EditFramework class Color', function () {

        it('should exist', function () {
            expect(Color).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // the default color scheme used in OX Documents
        var COLOR_SCHEME = {
                accent1: '4f81bd',
                accent2: 'c0504d',
                accent3: '9bbb59',
                accent4: '8064a2',
                accent5: '4bacc6',
                accent6: 'f79646',
                text1: '000000',
                text2: '1f497d',
                light1: 'ffffff',
                light2: 'eeece1',
                dark1: '000000',
                dark2: '1f497d',
                background1: 'ffffff',
                background2: 'eeece1',
                hyperlink: '0000ff',
                followedHyperlink: '800080'
            };

        // mock the getSchemeColor() method of an Office design theme
        var theme = { getSchemeColor: function (name) { return COLOR_SCHEME[name] || null; } };

        // verifies a color channel in the passed color model (interval [0;1])
        function verifyChannel(colorModel, channelName, exp) {
            expect(colorModel).to.have.a.property(channelName).that.is.closeTo(exp, 0.001);
        }

        // checks the 'rgb' property of the passed color descriptor
        function verifyRGB(colorDesc, expR, expG, expB, expA) {
            expect(colorDesc).to.have.a.property('rgb').that.is.an('object');
            verifyChannel(colorDesc.rgb, 'r', expR);
            verifyChannel(colorDesc.rgb, 'g', expG);
            verifyChannel(colorDesc.rgb, 'b', expB);
            verifyChannel(colorDesc.rgb, 'a', expA);
        }

        // checks the 'hsl' property of the passed color descriptor
        function verifyHSL(colorDesc, expH, expS, expL, expA) {
            expect(colorDesc).to.have.a.property('hsl').that.is.an('object');
            expect(colorDesc.hsl).to.have.a.property('h').that.is.closeTo(expH, 0.5);
            verifyChannel(colorDesc.hsl, 's', expS);
            verifyChannel(colorDesc.hsl, 'l', expL);
            verifyChannel(colorDesc.hsl, 'a', expA);
        }

        // constructs a color, and verifies the resolved color descriptor (no fall-backs, no transformations)
        function verifyColor(type, value, expR, expG, expB, expH, expS, expL, expA, expY, expHex, expCSS) {
            var colorDesc = new Color(type, value).resolve('', theme);
            verifyRGB(colorDesc, expR, expG, expB, expA);
            verifyHSL(colorDesc, expH, expS, expL, expA);
            verifyChannel(colorDesc, 'a', expA);
            verifyChannel(colorDesc, 'y', expY);
            expect(colorDesc).to.have.a.property('hex', expHex);
            expect(colorDesc).to.have.a.property('css', expCSS);
        }

        // constructs a color with transformations, and verifies the resolved color descriptor
        function verifyTransformedColor(css, type, value, expR, expG, expB, expH, expS, expL, expA, expY, expHex, expCSS) {
            var colorDesc = Color.parseCSS(css).transform(type, value).resolve('', theme);
            verifyRGB(colorDesc, expR, expG, expB, expA);
            verifyHSL(colorDesc, expH, expS, expL, expA);
            verifyChannel(colorDesc, 'a', expA);
            verifyChannel(colorDesc, 'y', expY);
            expect(colorDesc).to.have.a.property('hex', expHex);
            expect(colorDesc).to.have.a.property('css', expCSS);
        }

        // constants ----------------------------------------------------------

        describe('constant "AUTO"', function () {
            it('should exist', function () {
                expect(Color.AUTO).to.deep.equal({ type: 'auto' });
            });
        });

        describe('constant "HYPERLINK"', function () {
            it('should exist', function () {
                expect(Color.HYPERLINK).to.have.a.property('type');
            });
        });

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a color', function () {
                expect(new Color('auto')).to.be.an['instanceof'](Color);
            });
        });

        // method toJSON() needed in following tests
        describe('method "toJSON"', function () {
            it('should exist', function () {
                expect(new Color('auto')).to.respondTo('toJSON');
            });
            it('should return the JSON representation of auto color', function () {
                expect(new Color('auto').toJSON()).to.deep.equal({ type: 'auto' });
            });
            it('should return the JSON representation of RGB color', function () {
                expect(new Color('rgb', '4080c0').toJSON()).to.deep.equal({ type: 'rgb', value: '4080c0' });
            });
            it('should return the JSON representation of HSL color', function () {
                expect(new Color('hsl', { h: 0, s: 1, l: 0.5 }).toJSON()).to.deep.equal({ type: 'hsl', value: { h: 0, s: 1, l: 0.5 } });
            });
            it('should return the JSON representation of color with transformations', function () {
                expect(new Color('rgb', '4080c0', [{ type: 'tint', value: 25000 }]).toJSON()).to.deep.equal({ type: 'rgb', value: '4080c0', transformations: [{ type: 'tint', value: 25000 }] });
            });
            it('should return the JSON representation of color with fall-back RGB', function () {
                expect(new Color('scheme', 'accent1', null, '4080c0').toJSON()).to.deep.equal({ type: 'scheme', value: 'accent1', fallbackValue: '4080c0' });
            });
        });

        // static methods -----------------------------------------------------

        describe('method "parseJSON"', function () {
            it('should exist', function () {
                expect(Color).itself.to.respondTo('parseJSON');
            });
            it('should parse auto color', function () {
                var json = { type: 'auto' },
                    color = Color.parseJSON(json);
                expect(color).to.be.an['instanceof'](Color);
                expect(color.toJSON()).to.deep.equal(json);
            });
            it('should parse RGB color', function () {
                var json = { type: 'rgb', value: '4080c0' };
                expect(Color.parseJSON(json).toJSON()).to.deep.equal(json);
            });
            it('should parse HSL color', function () {
                var json = { type: 'hsl', value: { h: 0, s: 1, l: 0.5 } };
                expect(Color.parseJSON(json).toJSON()).to.deep.equal(json);
            });
            it('should parse transformations', function () {
                var json = { type: 'rgb', value: '4080c0', transformations: [{ type: 'tint', value: 25000 }] };
                expect(Color.parseJSON(json).toJSON()).to.deep.equal(json);
            });
            it('should parse fall-back RGB', function () {
                var json = { type: 'scheme', value: 'accent1', fallbackValue: '4080c0' };
                expect(Color.parseJSON(json).toJSON()).to.deep.equal(json);
            });
            it('should fall back to auto', function () {
                expect(Color.parseJSON({}).toJSON()).to.deep.equal({ type: 'auto' });
                expect(Color.parseJSON(null).toJSON()).to.deep.equal({ type: 'auto' });
            });
        });

        describe('method "parseCSS"', function () {
            it('should exist', function () {
                expect(Color).itself.to.respondTo('parseCSS');
            });
            it('should return null for missing or empty string', function () {
                expect(Color.parseCSS(null)).to.equal(null);
                expect(Color.parseCSS('')).to.equal(null);
            });
            it('should parse "transparent" keyword', function () {
                var color = Color.parseCSS('transparent');
                expect(color).to.be.an['instanceof'](Color);
                expect(color.toJSON()).to.deep.equal({ type: 'rgb', value: '000000', transformations: [{ type: 'alpha', value: 0 }] });
            });
            it('should return automatic color for "transparent" keyword in fill mode', function () {
                expect(Color.parseCSS('transparent', true).toJSON()).to.deep.equal({ type: 'auto' });
            });
            it('should parse color keywords', function () {
                expect(Color.parseCSS('black').toJSON()).to.deep.equal({ type: 'preset', value: 'black' });
                expect(Color.parseCSS('white').toJSON()).to.deep.equal({ type: 'preset', value: 'white' });
                expect(Color.parseCSS('red').toJSON()).to.deep.equal({ type: 'preset', value: 'red' });
                expect(Color.parseCSS('green').toJSON()).to.deep.equal({ type: 'preset', value: 'green' });
                expect(Color.parseCSS('blue').toJSON()).to.deep.equal({ type: 'preset', value: 'blue' });
                expect(Color.parseCSS('wrong_color_name')).to.equal(null);
            });
            it('should parse hexadecimal RGB color', function () {
                expect(Color.parseCSS('#4080c0').toJSON()).to.deep.equal({ type: 'rgb', value: '4080c0' });
                expect(Color.parseCSS('#4080C0').toJSON()).to.deep.equal({ type: 'rgb', value: '4080c0' });
                expect(Color.parseCSS('#48c').toJSON()).to.deep.equal({ type: 'rgb', value: '4488cc' });
                expect(Color.parseCSS('#')).to.equal(null);
                expect(Color.parseCSS('#4')).to.equal(null);
                expect(Color.parseCSS('#48')).to.equal(null);
                expect(Color.parseCSS('#4080')).to.equal(null);
                expect(Color.parseCSS('#4080c')).to.equal(null);
                expect(Color.parseCSS('#4080c04')).to.equal(null);
                expect(Color.parseCSS('#4080cg')).to.equal(null);
            });
            it('should parse function-style RGB(A) color', function () {
                expect(Color.parseCSS('rgb(64,128,192)').toJSON()).to.deep.equal({ type: 'rgb', value: '4080c0' });
                expect(Color.parseCSS('rgb( 0 , 127 , 255 )').toJSON()).to.deep.equal({ type: 'rgb', value: '007fff' });
                expect(Color.parseCSS('rgba(64,128,192,1)').toJSON()).to.deep.equal({ type: 'rgb', value: '4080c0' });
                expect(Color.parseCSS('rgba( 64 , 128 , 192 , 0.42 )').toJSON()).to.deep.equal({ type: 'rgb', value: '4080c0', transformations: [{ type: 'alpha', value: 42000 }] });
                expect(Color.parseCSS('rgba(64,128,192,0)').toJSON()).to.deep.equal({ type: 'rgb', value: '4080c0', transformations: [{ type: 'alpha', value: 0 }] });
                expect(Color.parseCSS('rgba(64,128,192,0)', true).toJSON()).to.deep.equal({ type: 'auto' });
                expect(Color.parseCSS('rgb(64,128)')).to.equal(null);
                expect(Color.parseCSS('rgb(64,128,192,1)')).to.equal(null);
                expect(Color.parseCSS('rgb(64,128,1000)').toJSON()).to.deep.equal({ type: 'rgb', value: '4080ff' });
                expect(Color.parseCSS('rgb(64,128,ff)')).to.equal(null);
                expect(Color.parseCSS('rgba(64,128,192)')).to.equal(null);
                expect(Color.parseCSS('rgba(64,128,192,1,1)')).to.equal(null);
                expect(Color.parseCSS('rgba(64,128,192,2)')).to.equal(null);
            });
            it('should parse function-style HSL(A) color', function () {
                expect(Color.parseCSS('hsl(60,100%,50%)').toJSON()).to.deep.equal({ type: 'hsl', value: { h: 60000, s: 100000, l: 50000 } });
                expect(Color.parseCSS('hsl( 0 , 0% , 0% )').toJSON()).to.deep.equal({ type: 'hsl', value: { h: 0, s: 0, l: 0 } });
                expect(Color.parseCSS('hsla(60,100%,50%,1)').toJSON()).to.deep.equal({ type: 'hsl', value: { h: 60000, s: 100000, l: 50000 } });
                expect(Color.parseCSS('hsla( 60 , 100% , 50% , 0.42 )').toJSON()).to.deep.equal({ type: 'hsl', value: { h: 60000, s: 100000, l: 50000 }, transformations: [{ type: 'alpha', value: 42000 }] });
                expect(Color.parseCSS('hsla(60,100%,50%,0)').toJSON()).to.deep.equal({ type: 'hsl', value: { h: 60000, s: 100000, l: 50000 }, transformations: [{ type: 'alpha', value: 0 }] });
                expect(Color.parseCSS('hsla(60,100%,50%,0)', true).toJSON()).to.deep.equal({ type: 'auto' });
                expect(Color.parseCSS('hsl(60,100%)')).to.equal(null);
                expect(Color.parseCSS('hsl(60,100%,50%,1)')).to.equal(null);
                expect(Color.parseCSS('hsl(60,100%,50)')).to.equal(null);
                expect(Color.parseCSS('hsl(60%,100%,50%)')).to.equal(null);
                expect(Color.parseCSS('hsl(420,200%,50%)').toJSON()).to.deep.equal({ type: 'hsl', value: { h: 60000, s: 100000, l: 50000 } });
                expect(Color.parseCSS('hsl(60,100%,ff%)')).to.equal(null);
                expect(Color.parseCSS('hsla(60,100%,50%)')).to.equal(null);
                expect(Color.parseCSS('hsla(60,100%,50%,1,1)')).to.equal(null);
                expect(Color.parseCSS('hsla(60,100%,50%,2)')).to.equal(null);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "getType"', function () {
            it('should exist', function () {
                expect(new Color('auto')).to.respondTo('getType');
            });
            it('should return the color type', function () {
                expect(new Color('auto').getType()).to.equal('auto');
                expect(new Color('rgb', '000000').getType()).to.equal('rgb');
                expect(new Color('hsl', { h: 0, s: 0, l: 0 }).getType()).to.equal('hsl');
                expect(new Color('crgb', { r: 0, g: 0, b: 0 }).getType()).to.equal('crgb');
                expect(new Color('system', 'menu').getType()).to.equal('system');
                expect(new Color('preset', 'black').getType()).to.equal('preset');
                expect(new Color('scheme', 'accent1').getType()).to.equal('scheme');
            });
        });

        describe('method "isAuto"', function () {
            var color = new Color('auto');
            it('should exist', function () {
                expect(color).to.respondTo('isAuto');
            });
            it('should return true for automatic color', function () {
                expect(color.isAuto()).to.equal(true);
            });
            it('should return false for all other colors', function () {
                expect(new Color('rgb', '000000').isAuto()).to.equal(false);
                expect(new Color('hsl', { h: 0, s: 0, l: 0 }).isAuto()).to.equal(false);
                expect(new Color('crgb', { r: 0, g: 0, b: 0 }).isAuto()).to.equal(false);
                expect(new Color('system', 'menu').isAuto()).to.equal(false);
                expect(new Color('preset', 'black').isAuto()).to.equal(false);
                expect(new Color('scheme', 'accent1').isAuto()).to.equal(false);
            });
        });

        describe('method "equals"', function () {
            it('should exist', function () {
                expect(new Color('auto')).to.respondTo('equals');
            });
            it('should return true for equal colors', function () {
                expect(new Color('auto').equals(new Color('auto', 'value'))).to.equal(true);
                expect(new Color('rgb', '4080c0').equals(new Color('rgb', '4080C0'))).to.equal(true);
                expect(new Color('hsl', { h: 25000, s: 50000, l: 75000 }).equals(new Color('hsl', { h: 25000, s: 50000, l: 75000 }))).to.equal(true);
                expect(new Color('crgb', { r: 25000, g: 50000, b: 75000 }).equals(new Color('crgb', { r: 25000, g: 50000, b: 75000 }))).to.equal(true);
                expect(new Color('system', 'menu').equals(new Color('system', 'Menu'))).to.equal(true);
                expect(new Color('preset', 'black').equals(new Color('preset', 'Black'))).to.equal(true);
                expect(new Color('scheme', 'accent1').equals(new Color('scheme', 'Accent1'))).to.equal(true);
                expect(new Color('rgb', '4080c0', [{ type: 'gray' }]).equals(new Color('rgb', '4080C0').transform('gray'))).to.equal(true);
            });
            it('should return false for different color types', function () {
                expect(new Color('auto', '4080c0').equals(new Color('rgb', '4080c0'))).to.equal(false);
                expect(new Color('rgb', '4080c0').equals(new Color('hsl', { h: 25000, s: 50000, l: 75000 }))).to.equal(false);
                expect(new Color('system', 'menu').equals(new Color('preset', 'black'))).to.equal(false);
            });
            it('should return false for different color values', function () {
                expect(new Color('rgb', '4080c0').equals(new Color('rgb', '4080c1'))).to.equal(false);
                expect(new Color('hsl', { h: 25000, s: 50000, l: 75000 }).equals(new Color('hsl', { h: 25001, s: 50000, l: 75000 }))).to.equal(false);
                expect(new Color('crgb', { r: 25000, g: 50000, b: 75000 }).equals(new Color('crgb', { r: 25001, g: 50000, b: 75000 }))).to.equal(false);
                expect(new Color('system', 'menu').equals(new Color('system', 'note'))).to.equal(false);
                expect(new Color('preset', 'black').equals(new Color('preset', 'white'))).to.equal(false);
                expect(new Color('scheme', 'accent1').equals(new Color('scheme', 'accent2'))).to.equal(false);
            });
            it('should return false for different color transformations', function () {
                expect(new Color('rgb', '4080c0').equals(new Color('rgb', '4080c0', [{ type: 'gray' }]))).to.equal(false);
                expect(new Color('rgb', '4080c0', [{ type: 'gray' }]).equals(new Color('rgb', '4080c0', [{ type: 'inv' }]))).to.equal(false);
                expect(new Color('rgb', '4080c0', [{ type: 'gray' }]).equals(new Color('rgb', '4080c0', [{ type: 'gray' }, { type: 'inv' }]))).to.equal(false);
                expect(new Color('rgb', '4080c0', [{ type: 'gray' }, { type: 'inv' }]).equals(new Color('rgb', '4080c0', [{ type: 'inv' }, { type: 'gray' }]))).to.equal(false);
                expect(new Color('rgb', '4080c0', [{ type: 'red', value: 25000 }]).equals(new Color('rgb', '4080c0', [{ type: 'green', value: 25000 }]))).to.equal(false);
                expect(new Color('rgb', '4080c0', [{ type: 'red', value: 25000 }]).equals(new Color('rgb', '4080c0', [{ type: 'red', value: 25001 }]))).to.equal(false);
            });
        });

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(new Color('auto')).to.respondTo('clone');
            });
            it('should create a clone', function () {
                expect(new Color('auto').clone()).to.be.an['instanceof'](Color);
                var color = new Color('rgb', '4080c0', [{ type: 'gray' }]);
                expect(color.clone().equals(color)).to.equal(true);
            });
        });

        describe('method "transform"', function () {
            it('should exist', function () {
                expect(new Color('auto')).to.respondTo('transform');
            });
            it('should store transformations passed as value', function () {
                var color = new Color('auto');
                expect(color.transform('alpha', 50000)).to.equal(color);
                color.transform('gray');
                expect(color.toJSON()).to.deep.equal({ type: 'auto', transformations: [{ type: 'alpha', value: 50000 }, { type: 'gray' }] });
            });
            it('should store transformations passed as object', function () {
                var color = new Color('auto');
                color.transform({ type: 'alpha', value: 50000 });
                color.transform({ type: 'gray' });
                expect(color.toJSON()).to.deep.equal({ type: 'auto', transformations: [{ type: 'alpha', value: 50000 }, { type: 'gray' }] });
            });
            it('should store transformations passed as array', function () {
                var color = new Color('auto');
                color.transform([{ type: 'alpha', value: 50000 }, { type: 'gray' }]);
                expect(color.toJSON()).to.deep.equal({ type: 'auto', transformations: [{ type: 'alpha', value: 50000 }, { type: 'gray' }] });
            });
        });

        describe('method "resolve"', function () {
            it('should exist', function () {
                expect(new Color('auto')).to.respondTo('resolve');
            });
            it('should use preset fall-back values for automatic color', function () {
                expect(new Color('auto').resolve('text')).to.have.a.property('css', '#000000');
                expect(new Color('auto').resolve('line')).to.have.a.property('css', '#000000');
                expect(new Color('auto').resolve('fill')).to.have.a.property('css', 'transparent');
            });
            it('should use fallback color for automatic color', function () {
                var desc = new Color('auto').resolve(new Color('rgb', '4080c0'));
                expect(desc).to.have.a.property('css', '#4080c0');
            });
            it('should return color descriptor for scheme colors', function () {
                verifyColor('scheme', 'accent1', 0.310, 0.506, 0.741, 213, 0.455, 0.525, 1, 0.481, '4f81bd', '#4f81bd');
                verifyColor('scheme', 'Accent1', 0.310, 0.506, 0.741, 213, 0.455, 0.525, 1, 0.481, '4f81bd', '#4f81bd');
                verifyColor('scheme', 'accent2', 0.753, 0.314, 0.302, 2, 0.477, 0.527, 1, 0.406, 'c0504d', '#c0504d');
                verifyColor('scheme', 'accent3', 0.608, 0.733, 0.349, 80, 0.419, 0.541, 1, 0.679, '9bbb59', '#9bbb59');
                verifyColor('scheme', 'accent4', 0.502, 0.392, 0.635, 267, 0.25, 0.514, 1, 0.433, '8064a2', '#8064a2');
                verifyColor('scheme', 'accent5', 0.294, 0.675, 0.776, 193, 0.519, 0.535, 1, 0.601, '4bacc6', '#4bacc6');
                verifyColor('scheme', 'accent6', 0.969, 0.588, 0.275, 27, 0.917, 0.622, 1, 0.646, 'f79646', '#f79646');
            });
            it('should return color descriptor for preset colors', function () {
                verifyColor('preset', 'black', 0, 0, 0, 0, 0, 0, 1, 0, '000000', '#000000');
                verifyColor('preset', 'Black', 0, 0, 0, 0, 0, 0, 1, 0, '000000', '#000000');
                verifyColor('preset', 'blue', 0, 0, 1, 240, 1, 0.5, 1, 0.072, '0000ff', '#0000ff');
                verifyColor('preset', 'brown', 0.647, 0.165, 0.165, 0, 0.594, 0.406, 1, 0.267, 'a52a2a', '#a52a2a');
                verifyColor('preset', 'cyan', 0, 1, 1, 180, 1, 0.5, 1, 0.787, '00ffff', '#00ffff');
                verifyColor('preset', 'gray', 0.502, 0.502, 0.502, 0, 0, 0.502, 1, 0.502, '808080', '#808080');
                verifyColor('preset', 'green', 0, 0.502, 0, 120, 1, 0.251, 1, 0.359, '008000', '#008000');
                verifyColor('preset', 'lime', 0, 1, 0, 120, 1, 0.5, 1, 0.715, '00ff00', '#00ff00');
                verifyColor('preset', 'magenta', 1, 0, 1, 300, 1, 0.5, 1, 0.285, 'ff00ff', '#ff00ff');
                verifyColor('preset', 'orange', 1, 0.647, 0, 39, 1, 0.5, 1, 0.675, 'ffa500', '#ffa500');
                verifyColor('preset', 'pink', 1, 0.753, 0.796, 350, 1, 0.876, 1, 0.809, 'ffc0cb', '#ffc0cb');
                verifyColor('preset', 'red', 1, 0, 0, 0, 1, 0.5, 1, 0.213, 'ff0000', '#ff0000');
                verifyColor('preset', 'turquoise', 0.251, 0.878, 0.816, 174, 0.721, 0.565, 1, 0.741, '40e0d0', '#40e0d0');
                verifyColor('preset', 'white', 1, 1, 1, 0, 0, 1, 1, 1, 'ffffff', '#ffffff');
                verifyColor('preset', 'yellow', 1, 1, 0, 60, 1, 0.5, 1, 0.928, 'ffff00', '#ffff00');
            });
            it('should return color descriptor for RGB colors', function () {
                verifyColor('rgb', '000000', 0, 0, 0, 0, 0, 0, 1, 0, '000000', '#000000');
                verifyColor('rgb', 'FFFFFF', 1, 1, 1, 0, 0, 1, 1, 1, 'ffffff', '#ffffff');
                verifyColor('rgb', 'FF0000', 1, 0, 0, 0, 1, 0.5, 1, 0.213, 'ff0000', '#ff0000');
                verifyColor('rgb', 'FFFF00', 1, 1, 0, 60, 1, 0.5, 1, 0.928, 'ffff00', '#ffff00');
                verifyColor('rgb', '00FF00', 0, 1, 0, 120, 1, 0.5, 1, 0.715, '00ff00', '#00ff00');
                verifyColor('rgb', '00FFFF', 0, 1, 1, 180, 1, 0.5, 1, 0.787, '00ffff', '#00ffff');
                verifyColor('rgb', '0000FF', 0, 0, 1, 240, 1, 0.5, 1, 0.072, '0000ff', '#0000ff');
                verifyColor('rgb', 'FF00FF', 1, 0, 1, 300, 1, 0.5, 1, 0.285, 'ff00ff', '#ff00ff');
                verifyColor('rgb', '808080', 0.502, 0.502, 0.502, 0, 0, 0.502, 1, 0.502, '808080', '#808080');
                verifyColor('rgb', '4080c0', 0.251, 0.502, 0.753, 210, 0.504, 0.502, 1, 0.467, '4080c0', '#4080c0');
            });
            it('should return color descriptor for CRGB colors', function () {
                verifyColor('crgb', { r: 0, g: 0, b: 0 }, 0, 0, 0, 0, 0, 0, 1, 0, '000000', '#000000');
                verifyColor('crgb', { r: 25000, g: 25000, b: 25000 }, 0.543, 0.543, 0.543, 0, 0, 0.543, 1, 0.543, '8a8a8a', '#8a8a8a');
                verifyColor('crgb', { r: 50000, g: 50000, b: 50000 }, 0.737, 0.737, 0.737, 0, 0, 0.737, 1, 0.737, 'bcbcbc', '#bcbcbc');
                verifyColor('crgb', { r: 75000, g: 75000, b: 75000 }, 0.881, 0.881, 0.881, 0, 0, 0.881, 1, 0.881, 'e1e1e1', '#e1e1e1');
                verifyColor('crgb', { r: 100000, g: 100000, b: 100000 }, 1, 1, 1, 0, 0, 1, 1, 1, 'ffffff', '#ffffff');
                verifyColor('crgb', { r: 25000, g: 50000, b: 75000 }, 0.543, 0.737, 0.881, 206, 0.587, 0.712, 1, 0.706, '8abce1', '#8abce1');
            });
            it('should return color descriptor for HSL colors', function () {
                verifyColor('hsl', { h: 0, s: 0, l: 0 }, 0, 0, 0, 0, 0, 0, 1, 0, '000000', '#000000');
                verifyColor('hsl', { h: 90000, s: 0, l: 0 }, 0, 0, 0, 90, 0, 0, 1, 0, '000000', '#000000');
                verifyColor('hsl', { h: 180000, s: 0, l: 0 }, 0, 0, 0, 180, 0, 0, 1, 0, '000000', '#000000');
                verifyColor('hsl', { h: 270000, s: 0, l: 0 }, 0, 0, 0, 270, 0, 0, 1, 0, '000000', '#000000');
                verifyColor('hsl', { h: 0, s: 25000, l: 0 }, 0, 0, 0, 0, 0.25, 0, 1, 0, '000000', '#000000');
                verifyColor('hsl', { h: 90000, s: 50000, l: 0 }, 0, 0, 0, 90, 0.5, 0, 1, 0, '000000', '#000000');
                verifyColor('hsl', { h: 180000, s: 75000, l: 0 }, 0, 0, 0, 180, 0.75, 0, 1, 0, '000000', '#000000');
                verifyColor('hsl', { h: 270000, s: 100000, l: 0 }, 0, 0, 0, 270, 1, 0, 1, 0, '000000', '#000000');
                verifyColor('hsl', { h: 0, s: 0, l: 25000 }, 0.25, 0.25, 0.25, 0, 0, 0.25, 1, 0.25, '404040', '#404040');
                verifyColor('hsl', { h: 90000, s: 0, l: 50000 }, 0.5, 0.5, 0.5, 90, 0, 0.5, 1, 0.5, '808080', '#808080');
                verifyColor('hsl', { h: 180000, s: 0, l: 75000 }, 0.75, 0.75, 0.75, 180, 0, 0.75, 1, 0.75, 'bfbfbf', '#bfbfbf');
                verifyColor('hsl', { h: 270000, s: 0, l: 100000 }, 1, 1, 1, 270, 0, 1, 1, 1, 'ffffff', '#ffffff');
                verifyColor('hsl', { h: 0, s: 50000, l: 100000 }, 1, 1, 1, 0, 0.5, 1, 1, 1, 'ffffff', '#ffffff');
                verifyColor('hsl', { h: 180000, s: 100000, l: 100000 }, 1, 1, 1, 180, 1, 1, 1, 1, 'ffffff', '#ffffff');
                verifyColor('hsl', { h: 10000, s: 20000, l: 20000 }, 0.24, 0.173, 0.16, 10, 0.2, 0.2, 1, 0.187, '3d2c29', '#3d2c29');
                verifyColor('hsl', { h: 50000, s: 20000, l: 50000 }, 0.6, 0.567, 0.4, 50, 0.2, 0.5, 1, 0.562, '999166', '#999166');
                verifyColor('hsl', { h: 90000, s: 20000, l: 80000 }, 0.8, 0.84, 0.76, 90, 0.2, 0.8, 1, 0.826, 'ccd6c2', '#ccd6c2');
                verifyColor('hsl', { h: 130000, s: 50000, l: 20000 }, 0.1, 0.3, 0.133, 130, 0.5, 0.2, 1, 0.245, '1a4d22', '#1a4d22');
                verifyColor('hsl', { h: 170000, s: 50000, l: 50000 }, 0.25, 0.75, 0.667, 170, 0.5, 0.5, 1, 0.638, '40bfaa', '#40bfaa');
                verifyColor('hsl', { h: 210000, s: 50000, l: 80000 }, 0.7, 0.8, 0.9, 210, 0.5, 0.8, 1, 0.786, 'b3cce6', '#b3cce6');
                verifyColor('hsl', { h: 250000, s: 80000, l: 20000 }, 0.093, 0.04, 0.36, 250, 0.8, 0.2, 1, 0.074, '180a5c', '#180a5c');
                verifyColor('hsl', { h: 290000, s: 80000, l: 50000 }, 0.767, 0.1, 0.9, 290, 0.8, 0.5, 1, 0.299, 'c41ae6', '#c41ae6');
                verifyColor('hsl', { h: 330000, s: 80000, l: 80000 }, 0.96, 0.64, 0.8, 330, 0.8, 0.8, 1, 0.72, 'f5a3cc', '#f5a3cc');
            });
            it('should return "dark" flag', function () {
                expect(new Color('preset', 'black').resolve('')).to.have.a.property('dark', true);
                expect(new Color('preset', 'white').resolve('')).to.have.a.property('dark', false);
            });
            it('should resolve color with "alpha" transformation', function () {
                verifyTransformedColor('rgba(255,0,0,0.5)', 'alpha', 0, 1, 0, 0, 0, 1, 0.5, 0, 0.213, 'ff0000', 'transparent');
                verifyTransformedColor('rgba(255,0,0,0.5)', 'alpha', 25000, 1, 0, 0, 0, 1, 0.5, 0.25, 0.213, 'ff0000', 'rgba(255,0,0,0.25)');
                verifyTransformedColor('rgba(255,0,0,0.5)', 'alpha', 50000, 1, 0, 0, 0, 1, 0.5, 0.5, 0.213, 'ff0000', 'rgba(255,0,0,0.5)');
                verifyTransformedColor('rgba(255,0,0,0.5)', 'alpha', 75000, 1, 0, 0, 0, 1, 0.5, 0.75, 0.213, 'ff0000', 'rgba(255,0,0,0.75)');
                verifyTransformedColor('rgba(255,0,0,0.5)', 'alpha', 100000, 1, 0, 0, 0, 1, 0.5, 1, 0.213, 'ff0000', '#ff0000');
            });
            it('should resolve color with "alphaMod" transformation', function () {
                verifyTransformedColor('rgba(255,0,0,0.5)', 'alphaMod', 0, 1, 0, 0, 0, 1, 0.5, 0, 0.213, 'ff0000', 'transparent');
                verifyTransformedColor('rgba(255,0,0,0.5)', 'alphaMod', 25000, 1, 0, 0, 0, 1, 0.5, 0.125, 0.213, 'ff0000', 'rgba(255,0,0,0.125)');
                verifyTransformedColor('rgba(255,0,0,0.5)', 'alphaMod', 50000, 1, 0, 0, 0, 1, 0.5, 0.25, 0.213, 'ff0000', 'rgba(255,0,0,0.25)');
                verifyTransformedColor('rgba(255,0,0,0.5)', 'alphaMod', 75000, 1, 0, 0, 0, 1, 0.5, 0.375, 0.213, 'ff0000', 'rgba(255,0,0,0.375)');
                verifyTransformedColor('rgba(255,0,0,0.5)', 'alphaMod', 100000, 1, 0, 0, 0, 1, 0.5, 0.5, 0.213, 'ff0000', 'rgba(255,0,0,0.5)');
                verifyTransformedColor('rgba(255,0,0,0.5)', 'alphaMod', 150000, 1, 0, 0, 0, 1, 0.5, 0.75, 0.213, 'ff0000', 'rgba(255,0,0,0.75)');
                verifyTransformedColor('rgba(255,0,0,0.5)', 'alphaMod', 200000, 1, 0, 0, 0, 1, 0.5, 1, 0.213, 'ff0000', '#ff0000');
                verifyTransformedColor('rgba(255,0,0,0.5)', 'alphaMod', 300000, 1, 0, 0, 0, 1, 0.5, 1, 0.213, 'ff0000', '#ff0000');
            });
            it('should resolve color with "alphaOff" transformation', function () {
                verifyTransformedColor('rgba(255,0,0,0.5)', 'alphaOff', -75000, 1, 0, 0, 0, 1, 0.5, 0, 0.213, 'ff0000', 'transparent');
                verifyTransformedColor('rgba(255,0,0,0.5)', 'alphaOff', -50000, 1, 0, 0, 0, 1, 0.5, 0, 0.213, 'ff0000', 'transparent');
                verifyTransformedColor('rgba(255,0,0,0.5)', 'alphaOff', -25000, 1, 0, 0, 0, 1, 0.5, 0.25, 0.213, 'ff0000', 'rgba(255,0,0,0.25)');
                verifyTransformedColor('rgba(255,0,0,0.5)', 'alphaOff', 0, 1, 0, 0, 0, 1, 0.5, 0.5, 0.213, 'ff0000', 'rgba(255,0,0,0.5)');
                verifyTransformedColor('rgba(255,0,0,0.5)', 'alphaOff', 25000, 1, 0, 0, 0, 1, 0.5, 0.75, 0.213, 'ff0000', 'rgba(255,0,0,0.75)');
                verifyTransformedColor('rgba(255,0,0,0.5)', 'alphaOff', 50000, 1, 0, 0, 0, 1, 0.5, 1, 0.213, 'ff0000', '#ff0000');
                verifyTransformedColor('rgba(255,0,0,0.5)', 'alphaOff', 750000, 1, 0, 0, 0, 1, 0.5, 1, 0.213, 'ff0000', '#ff0000');
            });
            it('should resolve color with "red" transformation', function () {
                verifyTransformedColor('#666666', 'red', 0, 0, 0.4, 0.4, 180, 1, 0.2, 1, 0.315, '006666', '#006666');
                verifyTransformedColor('#666666', 'red', 20000, 0.2, 0.4, 0.4, 180, 0.333, 0.3, 1, 0.357, '336666', '#336666');
                verifyTransformedColor('#666666', 'red', 40000, 0.4, 0.4, 0.4, 0, 0, 0.4, 1, 0.4, '666666', '#666666');
                verifyTransformedColor('#666666', 'red', 60000, 0.6, 0.4, 0.4, 0, 0.2, 0.5, 1, 0.443, '996666', '#996666');
                verifyTransformedColor('#666666', 'red', 80000, 0.8, 0.4, 0.4, 0, 0.5, 0.6, 1, 0.485, 'cc6666', '#cc6666');
                verifyTransformedColor('#666666', 'red', 100000, 1, 0.4, 0.4, 0, 1, 0.7, 1, 0.528, 'ff6666', '#ff6666');
            });
            it('should resolve color with "redMod" transformation', function () {
                verifyTransformedColor('#666666', 'redMod', 0, 0, 0.4, 0.4, 180, 1, 0.2, 1, 0.315, '006666', '#006666');
                verifyTransformedColor('#666666', 'redMod', 50000, 0.2, 0.4, 0.4, 180, 0.333, 0.3, 1, 0.357, '336666', '#336666');
                verifyTransformedColor('#666666', 'redMod', 100000, 0.4, 0.4, 0.4, 0, 0, 0.4, 1, 0.4, '666666', '#666666');
                verifyTransformedColor('#666666', 'redMod', 150000, 0.6, 0.4, 0.4, 0, 0.2, 0.5, 1, 0.443, '996666', '#996666');
                verifyTransformedColor('#666666', 'redMod', 200000, 0.8, 0.4, 0.4, 0, 0.5, 0.6, 1, 0.485, 'cc6666', '#cc6666');
                verifyTransformedColor('#666666', 'redMod', 250000, 1, 0.4, 0.4, 0, 1, 0.7, 1, 0.528, 'ff6666', '#ff6666');
                verifyTransformedColor('#666666', 'redMod', 300000, 1, 0.4, 0.4, 0, 1, 0.7, 1, 0.528, 'ff6666', '#ff6666');
            });
            it('should resolve color with "redOff" transformation', function () {
                verifyTransformedColor('#666666', 'redOff', -60000, 0, 0.4, 0.4, 180, 1, 0.2, 1, 0.315, '006666', '#006666');
                verifyTransformedColor('#666666', 'redOff', -40000, 0, 0.4, 0.4, 180, 1, 0.2, 1, 0.315, '006666', '#006666');
                verifyTransformedColor('#666666', 'redOff', -20000, 0.2, 0.4, 0.4, 180, 0.333, 0.3, 1, 0.357, '336666', '#336666');
                verifyTransformedColor('#666666', 'redOff', 0, 0.4, 0.4, 0.4, 0, 0, 0.4, 1, 0.4, '666666', '#666666');
                verifyTransformedColor('#666666', 'redOff', 20000, 0.6, 0.4, 0.4, 0, 0.2, 0.5, 1, 0.443, '996666', '#996666');
                verifyTransformedColor('#666666', 'redOff', 40000, 0.8, 0.4, 0.4, 0, 0.5, 0.6, 1, 0.485, 'cc6666', '#cc6666');
                verifyTransformedColor('#666666', 'redOff', 60000, 1, 0.4, 0.4, 0, 1, 0.7, 1, 0.528, 'ff6666', '#ff6666');
                verifyTransformedColor('#666666', 'redOff', 80000, 1, 0.4, 0.4, 0, 1, 0.7, 1, 0.528, 'ff6666', '#ff6666');
            });
            it('should resolve color with "green" transformation', function () {
                verifyTransformedColor('#666666', 'green', 0, 0.4, 0, 0.4, 300, 1, 0.2, 1, 0.114, '660066', '#660066');
                verifyTransformedColor('#666666', 'green', 20000, 0.4, 0.2, 0.4, 300, 0.333, 0.3, 1, 0.257, '663366', '#663366');
                verifyTransformedColor('#666666', 'green', 40000, 0.4, 0.4, 0.4, 0, 0, 0.4, 1, 0.4, '666666', '#666666');
                verifyTransformedColor('#666666', 'green', 60000, 0.4, 0.6, 0.4, 120, 0.2, 0.5, 1, 0.543, '669966', '#669966');
                verifyTransformedColor('#666666', 'green', 80000, 0.4, 0.8, 0.4, 120, 0.5, 0.6, 1, 0.686, '66cc66', '#66cc66');
                verifyTransformedColor('#666666', 'green', 100000, 0.4, 1, 0.4, 120, 1, 0.7, 1, 0.829, '66ff66', '#66ff66');
            });
            it('should resolve color with "greenMod" transformation', function () {
                verifyTransformedColor('#666666', 'greenMod', 0, 0.4, 0, 0.4, 300, 1, 0.2, 1, 0.114, '660066', '#660066');
                verifyTransformedColor('#666666', 'greenMod', 50000, 0.4, 0.2, 0.4, 300, 0.333, 0.3, 1, 0.257, '663366', '#663366');
                verifyTransformedColor('#666666', 'greenMod', 100000, 0.4, 0.4, 0.4, 0, 0, 0.4, 1, 0.4, '666666', '#666666');
                verifyTransformedColor('#666666', 'greenMod', 150000, 0.4, 0.6, 0.4, 120, 0.2, 0.5, 1, 0.543, '669966', '#669966');
                verifyTransformedColor('#666666', 'greenMod', 200000, 0.4, 0.8, 0.4, 120, 0.5, 0.6, 1, 0.686, '66cc66', '#66cc66');
                verifyTransformedColor('#666666', 'greenMod', 250000, 0.4, 1, 0.4, 120, 1, 0.7, 1, 0.829, '66ff66', '#66ff66');
                verifyTransformedColor('#666666', 'greenMod', 300000, 0.4, 1, 0.4, 120, 1, 0.7, 1, 0.829, '66ff66', '#66ff66');
            });
            it('should resolve color with "greenMod" transformation', function () {
                verifyTransformedColor('#666666', 'greenOff', -60000, 0.4, 0, 0.4, 300, 1, 0.2, 1, 0.114, '660066', '#660066');
                verifyTransformedColor('#666666', 'greenOff', -40000, 0.4, 0, 0.4, 300, 1, 0.2, 1, 0.114, '660066', '#660066');
                verifyTransformedColor('#666666', 'greenOff', -20000, 0.4, 0.2, 0.4, 300, 0.333, 0.3, 1, 0.257, '663366', '#663366');
                verifyTransformedColor('#666666', 'greenOff', 0, 0.4, 0.4, 0.4, 0, 0, 0.4, 1, 0.4, '666666', '#666666');
                verifyTransformedColor('#666666', 'greenOff', 20000, 0.4, 0.6, 0.4, 120, 0.2, 0.5, 1, 0.543, '669966', '#669966');
                verifyTransformedColor('#666666', 'greenOff', 40000, 0.4, 0.8, 0.4, 120, 0.5, 0.6, 1, 0.686, '66cc66', '#66cc66');
                verifyTransformedColor('#666666', 'greenOff', 60000, 0.4, 1, 0.4, 120, 1, 0.7, 1, 0.829, '66ff66', '#66ff66');
                verifyTransformedColor('#666666', 'greenOff', 80000, 0.4, 1, 0.4, 120, 1, 0.7, 1, 0.829, '66ff66', '#66ff66');
            });
            it('should resolve color with "blue" transformation', function () {
                verifyTransformedColor('#666666', 'blue', 0, 0.4, 0.4, 0, 60, 1, 0.2, 1, 0.371, '666600', '#666600');
                verifyTransformedColor('#666666', 'blue', 20000, 0.4, 0.4, 0.2, 60, 0.333, 0.3, 1, 0.386, '666633', '#666633');
                verifyTransformedColor('#666666', 'blue', 40000, 0.4, 0.4, 0.4, 0, 0, 0.4, 1, 0.4, '666666', '#666666');
                verifyTransformedColor('#666666', 'blue', 60000, 0.4, 0.4, 0.6, 240, 0.2, 0.5, 1, 0.414, '666699', '#666699');
                verifyTransformedColor('#666666', 'blue', 80000, 0.4, 0.4, 0.8, 240, 0.5, 0.6, 1, 0.429, '6666cc', '#6666cc');
                verifyTransformedColor('#666666', 'blue', 100000, 0.4, 0.4, 1, 240, 1, 0.7, 1, 0.443, '6666ff', '#6666ff');
            });
            it('should resolve color with "blueMod" transformation', function () {
                verifyTransformedColor('#666666', 'blueMod', 0, 0.4, 0.4, 0, 60, 1, 0.2, 1, 0.371, '666600', '#666600');
                verifyTransformedColor('#666666', 'blueMod', 50000, 0.4, 0.4, 0.2, 60, 0.333, 0.3, 1, 0.386, '666633', '#666633');
                verifyTransformedColor('#666666', 'blueMod', 100000, 0.4, 0.4, 0.4, 0, 0, 0.4, 1, 0.4, '666666', '#666666');
                verifyTransformedColor('#666666', 'blueMod', 150000, 0.4, 0.4, 0.6, 240, 0.2, 0.5, 1, 0.414, '666699', '#666699');
                verifyTransformedColor('#666666', 'blueMod', 200000, 0.4, 0.4, 0.8, 240, 0.5, 0.6, 1, 0.429, '6666cc', '#6666cc');
                verifyTransformedColor('#666666', 'blueMod', 250000, 0.4, 0.4, 1, 240, 1, 0.7, 1, 0.443, '6666ff', '#6666ff');
                verifyTransformedColor('#666666', 'blueMod', 300000, 0.4, 0.4, 1, 240, 1, 0.7, 1, 0.443, '6666ff', '#6666ff');
            });
            it('should resolve color with "blueOff" transformation', function () {
                verifyTransformedColor('#666666', 'blueOff', -60000, 0.4, 0.4, 0, 60, 1, 0.2, 1, 0.371, '666600', '#666600');
                verifyTransformedColor('#666666', 'blueOff', -40000, 0.4, 0.4, 0, 60, 1, 0.2, 1, 0.371, '666600', '#666600');
                verifyTransformedColor('#666666', 'blueOff', -20000, 0.4, 0.4, 0.2, 60, 0.333, 0.3, 1, 0.386, '666633', '#666633');
                verifyTransformedColor('#666666', 'blueOff', 0, 0.4, 0.4, 0.4, 0, 0, 0.4, 1, 0.4, '666666', '#666666');
                verifyTransformedColor('#666666', 'blueOff', 20000, 0.4, 0.4, 0.6, 240, 0.2, 0.5, 1, 0.414, '666699', '#666699');
                verifyTransformedColor('#666666', 'blueOff', 40000, 0.4, 0.4, 0.8, 240, 0.5, 0.6, 1, 0.429, '6666cc', '#6666cc');
                verifyTransformedColor('#666666', 'blueOff', 60000, 0.4, 0.4, 1, 240, 1, 0.7, 1, 0.443, '6666ff', '#6666ff');
                verifyTransformedColor('#666666', 'blueOff', 80000, 0.4, 0.4, 1, 240, 1, 0.7, 1, 0.443, '6666ff', '#6666ff');
            });
            it('should resolve color with "hue" transformation', function () {
                verifyTransformedColor('hsl(120,40%,40%)', 'hue', 0, 0.56, 0.24, 0.24, 0, 0.4, 0.4, 1, 0.308, '8f3d3d', '#8f3d3d');
                verifyTransformedColor('hsl(120,40%,40%)', 'hue', 60000, 0.56, 0.56, 0.24, 60, 0.4, 0.4, 1, 0.537, '8f8f3d', '#8f8f3d');
                verifyTransformedColor('hsl(120,40%,40%)', 'hue', 120000, 0.24, 0.56, 0.24, 120, 0.4, 0.4, 1, 0.469, '3d8f3d', '#3d8f3d');
                verifyTransformedColor('hsl(120,40%,40%)', 'hue', 180000, 0.24, 0.56, 0.56, 180, 0.4, 0.4, 1, 0.492, '3d8f8f', '#3d8f8f');
                verifyTransformedColor('hsl(120,40%,40%)', 'hue', 240000, 0.24, 0.24, 0.56, 240, 0.4, 0.4, 1, 0.263, '3d3d8f', '#3d3d8f');
                verifyTransformedColor('hsl(120,40%,40%)', 'hue', 300000, 0.56, 0.24, 0.56, 300, 0.4, 0.4, 1, 0.331, '8f3d8f', '#8f3d8f');
                verifyTransformedColor('hsl(120,40%,40%)', 'hue', 360000, 0.56, 0.24, 0.24, 0, 0.4, 0.4, 1, 0.308, '8f3d3d', '#8f3d3d');
            });
            it('should resolve color with "hueMod" transformation', function () {
                verifyTransformedColor('hsl(120,40%,40%)', 'hueMod', 0, 0.56, 0.24, 0.24, 0, 0.4, 0.4, 1, 0.308, '8f3d3d', '#8f3d3d');
                verifyTransformedColor('hsl(120,40%,40%)', 'hueMod', 50000, 0.56, 0.56, 0.24, 60, 0.4, 0.4, 1, 0.537, '8f8f3d', '#8f8f3d');
                verifyTransformedColor('hsl(120,40%,40%)', 'hueMod', 100000, 0.24, 0.56, 0.24, 120, 0.4, 0.4, 1, 0.469, '3d8f3d', '#3d8f3d');
                verifyTransformedColor('hsl(120,40%,40%)', 'hueMod', 150000, 0.24, 0.56, 0.56, 180, 0.4, 0.4, 1, 0.492, '3d8f8f', '#3d8f8f');
                verifyTransformedColor('hsl(120,40%,40%)', 'hueMod', 200000, 0.24, 0.24, 0.56, 240, 0.4, 0.4, 1, 0.263, '3d3d8f', '#3d3d8f');
                verifyTransformedColor('hsl(120,40%,40%)', 'hueMod', 250000, 0.56, 0.24, 0.56, 300, 0.4, 0.4, 1, 0.331, '8f3d8f', '#8f3d8f');
                verifyTransformedColor('hsl(120,40%,40%)', 'hueMod', 300000, 0.56, 0.24, 0.24, 0, 0.4, 0.4, 1, 0.308, '8f3d3d', '#8f3d3d');
                verifyTransformedColor('hsl(120,40%,40%)', 'hueMod', 350000, 0.56, 0.56, 0.24, 60, 0.4, 0.4, 1, 0.537, '8f8f3d', '#8f8f3d');
            });
            it('should resolve color with "hueOff" transformation', function () {
                verifyTransformedColor('hsl(120,40%,40%)', 'hueOff', -300000, 0.24, 0.56, 0.56, 180, 0.4, 0.4, 1, 0.492, '3d8f8f', '#3d8f8f');
                verifyTransformedColor('hsl(120,40%,40%)', 'hueOff', -240000, 0.24, 0.24, 0.56, 240, 0.4, 0.4, 1, 0.263, '3d3d8f', '#3d3d8f');
                verifyTransformedColor('hsl(120,40%,40%)', 'hueOff', -180000, 0.56, 0.24, 0.56, 300, 0.4, 0.4, 1, 0.331, '8f3d8f', '#8f3d8f');
                verifyTransformedColor('hsl(120,40%,40%)', 'hueOff', -120000, 0.56, 0.24, 0.24, 0, 0.4, 0.4, 1, 0.308, '8f3d3d', '#8f3d3d');
                verifyTransformedColor('hsl(120,40%,40%)', 'hueOff', -60000, 0.56, 0.56, 0.24, 60, 0.4, 0.4, 1, 0.537, '8f8f3d', '#8f8f3d');
                verifyTransformedColor('hsl(120,40%,40%)', 'hueOff', 0, 0.24, 0.56, 0.24, 120, 0.4, 0.4, 1, 0.469, '3d8f3d', '#3d8f3d');
                verifyTransformedColor('hsl(120,40%,40%)', 'hueOff', 60000, 0.24, 0.56, 0.56, 180, 0.4, 0.4, 1, 0.492, '3d8f8f', '#3d8f8f');
                verifyTransformedColor('hsl(120,40%,40%)', 'hueOff', 120000, 0.24, 0.24, 0.56, 240, 0.4, 0.4, 1, 0.263, '3d3d8f', '#3d3d8f');
                verifyTransformedColor('hsl(120,40%,40%)', 'hueOff', 180000, 0.56, 0.24, 0.56, 300, 0.4, 0.4, 1, 0.331, '8f3d8f', '#8f3d8f');
                verifyTransformedColor('hsl(120,40%,40%)', 'hueOff', 240000, 0.56, 0.24, 0.24, 0, 0.4, 0.4, 1, 0.308, '8f3d3d', '#8f3d3d');
                verifyTransformedColor('hsl(120,40%,40%)', 'hueOff', 300000, 0.56, 0.56, 0.24, 60, 0.4, 0.4, 1, 0.537, '8f8f3d', '#8f8f3d');
            });
            it('should resolve color with "sat" transformation', function () {
                verifyTransformedColor('hsl(120,40%,40%)', 'sat', 0, 0.4, 0.4, 0.4, 120, 0, 0.4, 1, 0.4, '666666', '#666666');
                verifyTransformedColor('hsl(120,40%,40%)', 'sat', 20000, 0.32, 0.48, 0.32, 120, 0.2, 0.4, 1, 0.434, '527a52', '#527a52');
                verifyTransformedColor('hsl(120,40%,40%)', 'sat', 40000, 0.24, 0.56, 0.24, 120, 0.4, 0.4, 1, 0.469, '3d8f3d', '#3d8f3d');
                verifyTransformedColor('hsl(120,40%,40%)', 'sat', 60000, 0.16, 0.64, 0.16, 120, 0.6, 0.4, 1, 0.503, '29a329', '#29a329');
                verifyTransformedColor('hsl(120,40%,40%)', 'sat', 80000, 0.08, 0.72, 0.08, 120, 0.8, 0.4, 1, 0.538, '14b814', '#14b814');
                verifyTransformedColor('hsl(120,40%,40%)', 'sat', 100000, 0, 0.8, 0, 120, 1, 0.4, 1, 0.572, '00cc00', '#00cc00');
            });
            it('should resolve color with "satMod" transformation', function () {
                verifyTransformedColor('hsl(120,40%,40%)', 'satMod', 0, 0.4, 0.4, 0.4, 120, 0, 0.4, 1, 0.4, '666666', '#666666');
                verifyTransformedColor('hsl(120,40%,40%)', 'satMod', 50000, 0.32, 0.48, 0.32, 120, 0.2, 0.4, 1, 0.434, '527a52', '#527a52');
                verifyTransformedColor('hsl(120,40%,40%)', 'satMod', 100000, 0.24, 0.56, 0.24, 120, 0.4, 0.4, 1, 0.469, '3d8f3d', '#3d8f3d');
                verifyTransformedColor('hsl(120,40%,40%)', 'satMod', 150000, 0.16, 0.64, 0.16, 120, 0.6, 0.4, 1, 0.503, '29a329', '#29a329');
                verifyTransformedColor('hsl(120,40%,40%)', 'satMod', 200000, 0.08, 0.72, 0.08, 120, 0.8, 0.4, 1, 0.538, '14b814', '#14b814');
                verifyTransformedColor('hsl(120,40%,40%)', 'satMod', 250000, 0, 0.8, 0, 120, 1, 0.4, 1, 0.572, '00cc00', '#00cc00');
                verifyTransformedColor('hsl(120,40%,40%)', 'satMod', 300000, 0, 0.8, 0, 120, 1, 0.4, 1, 0.572, '00cc00', '#00cc00');
            });
            it('should resolve color with "satOff" transformation', function () {
                verifyTransformedColor('hsl(120,40%,40%)', 'satOff', -60000, 0.4, 0.4, 0.4, 120, 0, 0.4, 1, 0.4, '666666', '#666666');
                verifyTransformedColor('hsl(120,40%,40%)', 'satOff', -40000, 0.4, 0.4, 0.4, 120, 0, 0.4, 1, 0.4, '666666', '#666666');
                verifyTransformedColor('hsl(120,40%,40%)', 'satOff', -20000, 0.32, 0.48, 0.32, 120, 0.2, 0.4, 1, 0.434, '527a52', '#527a52');
                verifyTransformedColor('hsl(120,40%,40%)', 'satOff', 0, 0.24, 0.56, 0.24, 120, 0.4, 0.4, 1, 0.469, '3d8f3d', '#3d8f3d');
                verifyTransformedColor('hsl(120,40%,40%)', 'satOff', 20000, 0.16, 0.64, 0.16, 120, 0.6, 0.4, 1, 0.503, '29a329', '#29a329');
                verifyTransformedColor('hsl(120,40%,40%)', 'satOff', 40000, 0.08, 0.72, 0.08, 120, 0.8, 0.4, 1, 0.538, '14b814', '#14b814');
                verifyTransformedColor('hsl(120,40%,40%)', 'satOff', 60000, 0, 0.8, 0, 120, 1, 0.4, 1, 0.572, '00cc00', '#00cc00');
                verifyTransformedColor('hsl(120,40%,40%)', 'satOff', 80000, 0, 0.8, 0, 120, 1, 0.4, 1, 0.572, '00cc00', '#00cc00');
            });
            it('should resolve color with "lum" transformation', function () {
                verifyTransformedColor('hsl(120,40%,40%)', 'lum', 0, 0, 0, 0, 120, 0.4, 0, 1, 0, '000000', '#000000');
                verifyTransformedColor('hsl(120,40%,40%)', 'lum', 20000, 0.12, 0.28, 0.12, 120, 0.4, 0.2, 1, 0.234, '1f471f', '#1f471f');
                verifyTransformedColor('hsl(120,40%,40%)', 'lum', 40000, 0.24, 0.56, 0.24, 120, 0.4, 0.4, 1, 0.469, '3d8f3d', '#3d8f3d');
                verifyTransformedColor('hsl(120,40%,40%)', 'lum', 60000, 0.44, 0.76, 0.44, 120, 0.4, 0.6, 1, 0.669, '70c270', '#70c270');
                verifyTransformedColor('hsl(120,40%,40%)', 'lum', 80000, 0.72, 0.88, 0.72, 120, 0.4, 0.8, 1, 0.834, 'b8e0b8', '#b8e0b8');
                verifyTransformedColor('hsl(120,40%,40%)', 'lum', 100000, 1, 1, 1, 120, 0.4, 1, 1, 1, 'ffffff', '#ffffff');
            });
            it('should resolve color with "lumMod" transformation', function () {
                verifyTransformedColor('hsl(120,40%,40%)', 'lumMod', 0, 0, 0, 0, 120, 0.4, 0, 1, 0, '000000', '#000000');
                verifyTransformedColor('hsl(120,40%,40%)', 'lumMod', 50000, 0.12, 0.28, 0.12, 120, 0.4, 0.2, 1, 0.234, '1f471f', '#1f471f');
                verifyTransformedColor('hsl(120,40%,40%)', 'lumMod', 100000, 0.24, 0.56, 0.24, 120, 0.4, 0.4, 1, 0.469, '3d8f3d', '#3d8f3d');
                verifyTransformedColor('hsl(120,40%,40%)', 'lumMod', 150000, 0.44, 0.76, 0.44, 120, 0.4, 0.6, 1, 0.669, '70c270', '#70c270');
                verifyTransformedColor('hsl(120,40%,40%)', 'lumMod', 200000, 0.72, 0.88, 0.72, 120, 0.4, 0.8, 1, 0.834, 'b8e0b8', '#b8e0b8');
                verifyTransformedColor('hsl(120,40%,40%)', 'lumMod', 250000, 1, 1, 1, 120, 0.4, 1, 1, 1, 'ffffff', '#ffffff');
                verifyTransformedColor('hsl(120,40%,40%)', 'lumMod', 300000, 1, 1, 1, 120, 0.4, 1, 1, 1, 'ffffff', '#ffffff');
            });
            it('should resolve color with "lumOff" transformation', function () {
                verifyTransformedColor('hsl(120,40%,40%)', 'lumOff', -60000, 0, 0, 0, 120, 0.4, 0, 1, 0, '000000', '#000000');
                verifyTransformedColor('hsl(120,40%,40%)', 'lumOff', -40000, 0, 0, 0, 120, 0.4, 0, 1, 0, '000000', '#000000');
                verifyTransformedColor('hsl(120,40%,40%)', 'lumOff', -20000, 0.12, 0.28, 0.12, 120, 0.4, 0.2, 1, 0.234, '1f471f', '#1f471f');
                verifyTransformedColor('hsl(120,40%,40%)', 'lumOff', 0, 0.24, 0.56, 0.24, 120, 0.4, 0.4, 1, 0.469, '3d8f3d', '#3d8f3d');
                verifyTransformedColor('hsl(120,40%,40%)', 'lumOff', 20000, 0.44, 0.76, 0.44, 120, 0.4, 0.6, 1, 0.669, '70c270', '#70c270');
                verifyTransformedColor('hsl(120,40%,40%)', 'lumOff', 40000, 0.72, 0.88, 0.72, 120, 0.4, 0.8, 1, 0.834, 'b8e0b8', '#b8e0b8');
                verifyTransformedColor('hsl(120,40%,40%)', 'lumOff', 60000, 1, 1, 1, 120, 0.4, 1, 1, 1, 'ffffff', '#ffffff');
                verifyTransformedColor('hsl(120,40%,40%)', 'lumOff', 80000, 1, 1, 1, 120, 0.4, 1, 1, 1, 'ffffff', '#ffffff');
            });
            it('should resolve color with "shade" transformation', function () {
                verifyTransformedColor('hsl(120,40%,40%)', 'shade', 0, 0, 0, 0, 120, 0.4, 0, 1, 0, '000000', '#000000');
                verifyTransformedColor('hsl(120,40%,40%)', 'shade', 25000, 0.06, 0.14, 0.06, 120, 0.4, 0.1, 1, 0.117, '0f240f', '#0f240f');
                verifyTransformedColor('hsl(120,40%,40%)', 'shade', 50000, 0.12, 0.28, 0.12, 120, 0.4, 0.2, 1, 0.234, '1f471f', '#1f471f');
                verifyTransformedColor('hsl(120,40%,40%)', 'shade', 75000, 0.18, 0.42, 0.18, 120, 0.4, 0.3, 1, 0.352, '2e6b2e', '#2e6b2e');
                verifyTransformedColor('hsl(120,40%,40%)', 'shade', 100000, 0.24, 0.56, 0.24, 120, 0.4, 0.4, 1, 0.469, '3d8f3d', '#3d8f3d');
                verifyTransformedColor('orange', 'shade', 50000, 0.5, 0.324, 0, 39, 1, 0.25, 1, 0.338, '805300', '#805300');
            });
            it('should resolve color with "tint" transformation', function () {
                verifyTransformedColor('hsl(120,40%,40%)', 'tint', 0, 1, 1, 1, 120, 0.4, 1, 1, 1, 'ffffff', '#ffffff');
                verifyTransformedColor('hsl(120,40%,40%)', 'tint', 25000, 0.79, 0.91, 0.79, 120, 0.4, 0.85, 1, 0.876, 'c9e8c9', '#c9e8c9');
                verifyTransformedColor('hsl(120,40%,40%)', 'tint', 50000, 0.58, 0.82, 0.58, 120, 0.4, 0.7, 1, 0.752, '94d194', '#94d194');
                verifyTransformedColor('hsl(120,40%,40%)', 'tint', 75000, 0.37, 0.73, 0.37, 120, 0.4, 0.55, 1, 0.627, '5eba5e', '#5eba5e');
                verifyTransformedColor('hsl(120,40%,40%)', 'tint', 100000, 0.24, 0.56, 0.24, 120, 0.4, 0.4, 1, 0.469, '3d8f3d', '#3d8f3d');
                verifyTransformedColor('orange', 'tint', 50000, 1, 0.824, 0.5, 39, 1, 0.75, 1, 0.838, 'ffd280', '#ffd280');
            });
            it('should resolve color with "gamma" transformation', function () {
                verifyTransformedColor('rgb(200,100,50)', 'gamma', null, 0.899, 0.662, 0.488, 25, 0.669, 0.693, 1, 0.7, 'e5a97c', '#e5a97c');
                verifyTransformedColor('hsl(135,40%,40%)', 'gamma', null, 0.533, 0.775, 0.605, 138, 0.349, 0.654, 1, 0.711, '88c69a', '#88c69a');
            });
            it('should resolve color with "invGamma" transformation', function () {
                verifyTransformedColor('rgb(200,100,50)', 'invGamma', null, 0.576, 0.119, 0.025, 10, 0.918, 0.3, 1, 0.21, '931e06', '#931e06');
                verifyTransformedColor('hsl(135,40%,40%)', 'invGamma', null, 0.039, 0.268, 0.075, 129, 0.745, 0.154, 1, 0.206, '0a4413', '#0a4413');
            });
            it('should resolve color with "comp" transformation', function () {
                verifyTransformedColor('rgb(200,100,50)', 'comp', null, 0.196, 0.588, 0.784, 200, 0.6, 0.49, 1, 0.519, '3296c8', '#3296c8');
                verifyTransformedColor('hsl(135,40%,40%)', 'comp', null, 0.56, 0.24, 0.48, 315, 0.4, 0.4, 1, 0.325, '8f3d7a', '#8f3d7a');
            });
            it('should resolve color with "inv" transformation', function () {
                verifyTransformedColor('rgb(200,100,50)', 'inv', null, 0.216, 0.608, 0.804, 200, 0.6, 0.51, 1, 0.539, '379bcd', '#379bcd');
                verifyTransformedColor('hsl(135,40%,40%)', 'inv', null, 0.76, 0.44, 0.68, 315, 0.4, 0.6, 1, 0.525, 'c270ad', '#c270ad');
            });
            it('should resolve color with "gray" transformation', function () {
                verifyTransformedColor('rgb(200,100,50)', 'gray', null, 0.461, 0.461, 0.461, 0, 0, 0.461, 1, 0.461, '767676', '#767676');
                verifyTransformedColor('hsl(135,40%,40%)', 'gray', null, 0.475, 0.475, 0.475, 0, 0, 0.475, 1, 0.475, '797979', '#797979');
            });
            it('should resolve multiple transformations in order', function () {
                expect(new Color('rgb', '00ff00').transform('red', 100000).transform('gray').resolve('').css).to.equal('#ededed');
                expect(new Color('rgb', '00ff00').transform('gray').transform('red', 100000).resolve('').css).to.equal('#ffb6b6');
            });
        });
    });

    // ========================================================================
});
