/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/border'
], function (Utils, Border) {

    'use strict';

    // static class Border ====================================================

    describe('EditFramework module Border', function () {

        it('should exist', function () {
            expect(Border).to.be.an('object');
        });

        // private helpers ----------------------------------------------------

        function Theme(attributes) {
            var colorScheme = _.copy(Utils.getObjectOption(attributes, 'colorScheme', {}), true);

            this.hasSchemeColors = function () {
                return !_.isEmpty(colorScheme);
            };
            this.getSchemeColor = function (name) {
                return _.isString(colorScheme[name]) ? colorScheme[name] : null;
            };

        }

        var AUTO = { type: 'auto' },

            border = { style: 'solid', width: 1, color: AUTO },
            noBorder = { style: 'none', width: 1, color: AUTO },
            doubleBorder = { style: 'double', width: 73, color: AUTO },
            dashedBorder = { style: 'dashed', width: 23, color: AUTO },
            hairBorder = { style: 'dashed', width: 1, color: AUTO },

            theme = new Theme({ colorScheme: {
                accent1: '4f81bd',
                accent2: 'c0504d',
                accent3: '9bbb59',
                accent4: '8064a2',
                accent5: '4bacc6',
                accent6: 'f79646',
                text1: '000000',
                text2: '1f497d',
                light1: 'ffffff',
                light2: 'eeece1',
                dark1: '000000',
                dark2: '1f497d',
                background1: 'ffffff',
                background2: 'eeece1',
                hyperlink: '0000ff',
                followedHyperlink: '800080'
            }});

        // static methods -----------------------------------------------------

        describe('method "getCssBorderStyle"', function () {
            it('should exist', function () {
                expect(Border).itself.to.respondTo('getCssBorderStyle');
            });
            it('should return true for check if paragraph is a registered attribute', function () {
                expect(Border.getCssBorderStyle('dotted')).to.equal('dotted');
            });
            it('should return true for check if paragraph is a registered attribute', function () {
                expect(Border.getCssBorderStyle('undefinedString')).to.equal('solid');
            });
            it('should return false for check if table is a registered attribute', function () {
                expect(Border.getCssBorderStyle('dotted')).not.to.equal('double');
            });
        });

        describe('method "getBorderPattern"', function () {
            it('should exist', function () {
                expect(Border).itself.to.respondTo('getBorderPattern');
            });
            it('should return [2,2] for dotted lines', function () {
                expect(Border.getBorderPattern('dotted', 1)).to.deep.equal([2, 2]);
            });
            it('should return [2,2] for dotted lines', function () {
                expect(Border.getBorderPattern('dotted', 1)).to.deep.equal([2, 2]);
                expect(Border.getBorderPattern('dotted', 23)).to.deep.equal([24, 24]);
            });
            it('should return ??? for solid lines', function () {
                expect(Border.getBorderPattern('dashed', 1)).to.deep.equal([6, 2]);
            });
            it('should return ??? for unknown lines', function () {
                expect(Border.getBorderPattern('unknown', 1)).to.equal(null);
            });
        });

        describe('method "getCssBorderAttributes"', function () {
            it('should exist', function () {
                expect(Border).itself.to.respondTo('getCssBorderAttributes');
            });
            it('should return a map with css border attributes', function () {
                expect(Border.getCssBorderAttributes(border, theme)).to.deep.equal({ style: 'solid', width: 1, color: '#000000' });
                expect(Border.getCssBorderAttributes(border, theme, { preview: true })).to.deep.equal({ style: 'solid', width: 1, color: '#000000' });
                expect(Border.getCssBorderAttributes(noBorder, theme)).to.deep.equal({ style: 'none', width: 0, color: '#000000' });
                expect(Border.getCssBorderAttributes(doubleBorder, theme)).to.deep.equal({ style: 'double', width: 3, color: '#000000' });
                expect(Border.getCssBorderAttributes(doubleBorder, theme, { preview: true })).to.deep.equal({ style: 'double', width: 3, color: '#000000' });
                expect(Border.getCssBorderAttributes(dashedBorder, theme)).to.deep.equal({ style: 'dashed', width: 1, color: '#000000' });
            });
            it('should return a map with CSS border attributes for a hair line', function () {
                var attrs = Border.getCssBorderAttributes(hairBorder, theme, { transparentHair: true });
                expect(attrs).to.have.a.property('style', 'dashed');
                expect(attrs).to.have.a.property('width', 1);
                expect(attrs).to.have.a.property('color', 'rgba(0,0,0,0.5)');
            });
        });

        describe('method "getCssBorder"', function () {
            it('should exist', function () {
                expect(Border).itself.to.respondTo('getCssBorder');
            });
            it('should return a map with css border attributes', function () {
                expect(Border.getCssBorder(border, theme)).to.equal('solid 1px #000000');
                expect(Border.getCssBorder(noBorder, theme)).to.equal('none');
                expect(Border.getCssBorder(noBorder, theme, { clearNone: true })).to.equal('');
            });
        });

        describe('method "isEqual"', function () {
            it('should exist', function () {
                expect(Border).itself.to.respondTo('isEqual');
            });
            it('should return a boolean whether the two given border object are equal', function () {
                expect(Border.isEqual(border, _.clone(border))).to.equal(true);
                expect(Border.isEqual(border, noBorder)).to.equal(false);
                expect(Border.isEqual(border, {})).to.equal(false);
                expect(Border.isEqual({}, {})).to.equal(true);
                expect(Border.isEqual(noBorder, noBorder)).to.equal(true);
            });
        });
    });

    // ========================================================================
});
