/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

'use strict';

module.exports = function (grunt) {

    // imports ================================================================

    var fs = require('fs');
    var PNGImage = require('node-png').PNG;

    // configuration ==========================================================

    // Root source directory.
    var SRC_ROOT = 'icons';

    // Path to the icons definition file.
    var ICONS_JSON_PATH = SRC_ROOT + '/icons.json';

    // Source directory of all images.
    var IMAGES_ROOT = SRC_ROOT + '/images';

    // Target directory name for all generated files.
    var APPS_ICONS_DIR = 'apps/io.ox/office/tk/icons';

    // Root destination directory for all generated files.
    var DEST_ROOT = 'build/' + APPS_ICONS_DIR;

    // Path to the generated LESS file with a LESS mix-in class for defining bitmap URLs.
    var DEFINITIONS_LESS_PATH = DEST_ROOT + '/definitions.less';

    // Path to the generated LESS file with icon class names and positions.
    var DOCS_ICONS_LESS_PATH = DEST_ROOT + '/docs-icons.less';

    // Base height of all icons, for scaling factor 1.
    var ICON_HEIGHT = 16;

    // Base width of an icon set, for scaling factor 1.
    var ICONSET_WIDTH = 720;

    // Definitions for all supported icon set pixel resolutions.
    var ICONSET_RESOLUTIONS = [
        { name: 'desktop', scale: 1 },
        { name: 'retina', scale: 2 }
    ];

    // Definition for all colors of the target icon sets.
    var ICONSET_COLORS = [
        { name: 'black', color: [0x33, 0x33, 0x33] },
        { name: 'white', color: [0xff, 0xff, 0xff] },
        { name: 'blue', color: [0x3c, 0x73, 0xaa] }
    ];

    // task configuration =====================================================

    // specify a specific destination file for 'newer' plug-in (output directory by itself does not work)
    grunt.config('iconsets', {
        all: {
            src: ['grunt/tasks/iconsets.js', ICONS_JSON_PATH, IMAGES_ROOT + '/**/*.png'],
            dest: DOCS_ICONS_LESS_PATH // will be generated as last file, after the icon sets
        }
    });

    // class CountDown ========================================================

    /**
     * Simple helper class that will invoke the registered callback functions,
     * after the end of a count-down has been reached.
     *
     * @constructor
     *
     * @param {Number} count
     *  The count-down size.
     */
    function CountDown(count) {
        this._steps = count;
        this._callbacks = [];
    }

    /**
     * Registers a callback function that will be executed after the end of the
     * count-down has been reached.
     *
     * @param {Function} callback
     *  The callback function to be registered.
     */
    CountDown.prototype.done = function (callback) {
        this._callbacks.push(callback.bind(this));
    };

    /**
     * Moves the count-down one step closer to the end. After this method has
     * been invoked the number of times specified in the constructor of this
     * instance, all registered callback functions will be invoked.
     */
    CountDown.prototype.step = function () {
        this._steps -= 1;
        if (this._steps === 0) {
            process.nextTick(function () {
                this._callbacks.forEach(function (callback) {
                    callback();
                });
            }.bind(this));
        }
    };

    // helper functions =======================================================

    /**
     * Checks that the passed value is a non-empty string, exits on error.
     */
    function ensureString(value, msg) {
        if ((typeof value !== 'string') || (value.length === 0)) {
            grunt.fatal(msg);
        }
    }

    /**
     * Checks that the passed value is an object, exits on error.
     */
    function ensureObject(value, msg) {
        if (grunt.util.kindOf(value) !== 'object') {
            grunt.fatal(msg);
        }
    }

    /**
     * Checks that the passed value is a non-empty array, exits on error.
     */
    function ensureArray(value, msg) {
        if ((grunt.util.kindOf(value) !== 'array') || (value.length === 0)) {
            grunt.fatal(msg);
        }
    }

    /**
     * Returns the file name of an icon set bitmap file.
     *
     * @param {String} type
     *  The type of the icon set ('desktop', or 'retina').
     *
     * @param {String} color
     *  The color name to be inserted into the file name.
     *
     * @returns {String}
     *  The file name of the icon set bitmap file, without path, but with file
     *  extension).
     */
    function getIconSetFileName(type, color) {
        return 'docs-icons-' + type + '-' + color + '.png';
    }

    /**
     * Writes a log message for a new file.
     */
    function logFileCreated(path) {
        grunt.log.writeln('File ' + path.cyan + ' created.');
    }

    /**
     * Writes a new text file.
     *
     * @param {String} path
     *  The full file name to be written.
     *
     * @param {Array<String>} contents
     *  The text contents to be written to the text file, as array of text
     *  lines.
     */
    function writeTextLines(path, lines) {
        grunt.file.write(path, lines.join('\n') + '\n');
        logFileCreated(path);
    }

    // task implementation ====================================================

    grunt.registerTask('iconsets', 'Assembles the icon sets from single image files.', function () {

        // conversion runs asynchronously
        var resolveTask = this.async();

        // ensure existing destination directory
        grunt.file.mkdir(DEST_ROOT);

        // read and validate the icon definitions
        var definitions = grunt.file.readJSON(ICONS_JSON_PATH);
        ensureArray(definitions, 'invalid icon definitions: expecting JSON array');

        // the path to the bitmap files, with a unique prefix for the bitmap URLs needed for cache busting
        var ICONSET_URL_PATH = 'v=' + grunt.config('pkg.version') + '.' + grunt.template.date(new Date(), 'yyyymmdd.HHMMss') + '/' + APPS_ICONS_DIR;

        // the contents of the resulting LESS file with the LESS mix-in class to define bitmap URLs
        var defLessLines = [
            '.docs-icons-url(@color-id) {',
            '    i[class^="docs-"] {',
            '        background-image: ~"url(' + ICONSET_URL_PATH + '/' + getIconSetFileName('desktop', '@{color-id}') + ')";',
            '        &.retina { background-image: ~"url(' + ICONSET_URL_PATH + '/' + getIconSetFileName('retina', '@{color-id}') + ')"; }',
            '    }',
            '}'
        ];

        // the contents of the resulting LESS file with icon definitions
        var iconLessLines = [];

        // create count-down triggers
        var imageCountDown = new CountDown(definitions.length * ICONSET_RESOLUTIONS.length);
        var iconSetCountDown = new CountDown(ICONSET_RESOLUTIONS.length * ICONSET_COLORS.length);

        // load all source images, store them in the entries of the 'definitions' array
        definitions.forEach(function (def) {

            // validate array element
            ensureObject(def, 'invalid icon definition: expecting objects as array elements');
            ensureString(def.path, 'invalid icon definition: missing path to source image');

            // validate classes property (may be a plain string)
            if (typeof def.classes === 'string') { def.classes = [def.classes]; }
            ensureArray(def.classes, 'invalid icon definition: expecting string or array as "classes" property');
            def.classes.forEach(function (cls) { ensureString(cls, 'invalid icon definition: expecting strings in "classes" property'); });

            // validate locales property (may be a plain string)
            if ('locales' in def) {
                if (typeof def.locales === 'string') { def.locales = [def.locales]; }
                ensureArray(def.locales, 'invalid icon definition: expecting string or array as "locales" property');
                def.locales.forEach(function (lc) { ensureString(lc, 'invalid icon definition: expecting strings in "locales" property'); });
            }

            // the source PNG images, mapped by icon set name ('desktop', 'retina')
            def.images = {};

            // the base size of the icon, for scaling factor 1 (width will be determined dynamically, see below)
            def.width = null;
            def.height = ICON_HEIGHT;

            // load the images for all scale factors, store them in the 'images' map
            ICONSET_RESOLUTIONS.forEach(function (resolutionData) {

                // current icon set name, scaling factor, and expected height for the scaling factor
                var resName = resolutionData.name;
                var resScale = resolutionData.scale;
                var height = def.height * resScale;

                // read the source image file into a binary buffer
                var path = IMAGES_ROOT + '/' + def.path + '_' + height + '.png';
                var buffer = grunt.file.read(path, { encoding: null });

                // parse the source image file
                var image = new PNGImage();
                image.parse(buffer, function (err) {
                    if (err) { grunt.fatal('invalid image ' + path, err); }

                    // validate the image height
                    if (image.height !== height) {
                        grunt.fatal('wrong image height in file ' + path + ' (expected ' + height + 'px but got ' + image.height + 'px)');
                    }

                    // store width for first imported image, check width of images with other scaling factor
                    if (typeof def.width !== 'number') {
                        def.width = image.width / resScale;
                    } else if (image.width !== def.width * resScale) {
                        grunt.fatal('wrong image width in file ' + path + ' (expected ' + (def.width * resScale) + 'px but got ' + image.width + 'px)');
                    }

                    // store the image object, continue after all images have been loaded
                    def.images[resName] = image;
                    imageCountDown.step();
                });
            });
        });

        // wait for all source images to be loaded
        imageCountDown.done(function () {

            // the height of the icon occupied in the icon set (one pixel padding above and below)
            var height = ICON_HEIGHT + 2;

            // initialize the LESS file containing the icon definitions
            iconLessLines.push(
                '.io-ox-office-main i[class^="docs-"] {',
                '    width: ' + height + 'px;', // default width, may be overridden by some icons
                '    height: ' + height + 'px;'
            );

            // calculate the positions of all icons in the final icon set bitmaps
            var offsetx = 0, offsety = 0;
            definitions.forEach(function (def) {

                // the width of the icon occupied in the icon set (one pixel padding left and right)
                var width = def.width + 2;

                // start a new line in the icon set on demand
                if (offsetx + width > ICONSET_WIDTH) {
                    offsetx = 0;
                    offsety += height;
                }

                // store the position in the 'def' array element
                def.x = offsetx;
                def.y = offsety;

                // generate the text line for the current icon in the LESS file
                var line = def.classes.map(function (cls) { return '&.docs-' + cls; }).join(', ');
                if (def.locales) { line += ' { ' + def.locales.map(function (lc) { return '&.lc-' + lc; }).join(', '); }
                line += ' { background-position: ' + -offsetx + 'px ' + -offsety + 'px;';
                if (def.width !== ICON_HEIGHT) { line += ' width: ' + (def.width + 2) + 'px;'; }
                line += ' }';
                if (def.locales) { line += ' }'; }
                iconLessLines.push(line);

                // go to next offset position
                offsetx += width;
            });

            // the resulting height of an icon set, for scaling factor 1
            var ICONSET_HEIGHT = offsety + height;

            // finalize the text lines of the LESS file
            iconLessLines.push(
                '    background-size: ' + ICONSET_WIDTH + 'px ' + ICONSET_HEIGHT + 'px;',
                '}'
            );

            // create and write an icon set for each combination of scaling factor and output color
            ICONSET_RESOLUTIONS.forEach(function (resolutionData) {

                // current icon set name, and scaling factor
                var resName = resolutionData.name;
                var resScale = resolutionData.scale;

                // process all target colors
                ICONSET_COLORS.forEach(function (colorData) {

                    // new image data is not clean out-of-the-box, explicitly fill with zeros
                    var image = new PNGImage({ width: ICONSET_WIDTH * resScale, height: ICONSET_HEIGHT * resScale });
                    image.data.fill(0);

                    // insert all source images into the icon set
                    definitions.forEach(function (def) {
                        def.images[resName].bitblt(image, 0, 0, def.width * resScale, def.height * resScale, (def.x + 1) * resScale, (def.y + 1) * resScale);
                    });

                    // redye all pixels in the image set
                    var color = colorData.color;
                    for (var d = image.data, i = 0, l = d.length; i < l; i += 4) {
                        for (var j = 0; j < 3; j += 1) { d[i + j] = color[j]; }
                    }

                    // create the output stream for the icon set
                    var path = DEST_ROOT + '/' + getIconSetFileName(resName, colorData.name);
                    var strm = fs.createWriteStream(path).on('error', function () {
                        grunt.fatal('cannot create icon set file ' + path);
                    });

                    // stream the image to the output file
                    strm = image.pack().pipe(strm);
                    strm.on('error', function () { grunt.fatal('cannot write image file ' + path); });

                    // continue after all icon sets have been written
                    strm.on('finish', function () {
                        logFileCreated(path);
                        iconSetCountDown.step();
                    });
                });
            });
        });

        // finalize and write the LESS files (as last step, after generating the icon sets)
        iconSetCountDown.done(function () {
            writeTextLines(DEFINITIONS_LESS_PATH, defLessLines);
            writeTextLines(DOCS_ICONS_LESS_PATH, iconLessLines);
            grunt.log.ok('Icon sets created successfully.');
            resolveTask();
        });
    });
};
