/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/tk/utils/dateutils', [
    'io.ox/office/tk/config',
    'io.ox/office/tk/utils'
], function (Config, Utils) {

    'use strict';

    // static class DateUtils =================================================

    //most short dates we know from spreadsheet
    var NO_YEARS = {
        cs: 'D.MMM',
        de: 'DD.MMM',
        en: 'MMM DD',
        es: 'DD/MMM',
        fr: 'DD/MMM',
        hu: 'DD.MMM',
        it: 'DD/MMM',
        ja: 'M\u6708D\u65E5',
        lv: 'MMM DD',
        nl: 'DD-MMM',
        pl: 'DD MMM',
        sk: 'MMM DD',
        zh: 'D-MMM'
    };

    var SHORT_MONTHS = null;

    /**
     * lazy init of short months list
     *
     * moment short months are not limited to 3 signs, and most have a dot in the string.
     * both is not good to be compatible with other office programms, so we prepare the list!
     *
     */
    function getShortMonths() {
        if (!SHORT_MONTHS) {
            SHORT_MONTHS = [];
            _.each(moment.localeData()._monthsShort, function (month) {
                SHORT_MONTHS.push(month.replace('.', '').substring(0, 3));
            });
        }
        return SHORT_MONTHS;
    }

    // the exported DateUtils class
    var DateUtils = {};

    // methods ----------------------------------------------------------------

    /**
     * @param {Number} timestamp
     *
     * @return {Date} date with assigned timestamp
     */
    DateUtils.getLocalDate = function (timestamp) {
        if (_.isUndefined(timestamp)) {
            return new Date();
        } else {
            return new Date(timestamp);
        }
    };

    /**
     * @param {Number} timestamp
     *
     * @return {Date} UTC-date with assigned timestamp
     */
    DateUtils.getUTCDate = function (timestamp) {
        var offset = moment().utcOffset() * 60 * 1000;
        if (_.isUndefined(timestamp)) {
            return new Date(new Date().getTime() - offset);
        } else {
            return new Date(timestamp - offset);
        }
    };

    /**
     * @return {Number} the time of a week in millieseconds
     */
    DateUtils.getWeekInMS = function () {
        return 604800000;
    };

    /**
     * formats assigned date with assigned format
     *
     * @param {Date} date
     *
     * @param {String} format
     *
     * @return {String} formated date
     */
    DateUtils.format = function (date, format) {
        var result = moment(date.getTime()).format(format);
        return result;
    };

    /**
     * @return {String} dateformat, day month and year
     */
    DateUtils.getDateFormat = function () {
        return moment.localeData()._longDateFormat.L;
    };

    /**
     * @return {String} timeformat, normally hour and minutes
     */
    DateUtils.getTimeFormat = function () {
        return moment.localeData()._longDateFormat.LT;
    };

    /**
     * @return {String} dateformat for complete weekdays
     */
    DateUtils.getWeekDayFormat = function () {
        return 'dddd';
    };

    /**
     * @return {String} dateformat, day month without year,
     *  but not implemented at the moment!!!
     */
    DateUtils.getNoYearFormat = function () {
        return NO_YEARS[Config.LANGUAGE] || NO_YEARS.en;
    };

    // locale arrays

    /**
     * @return {Array} of {String} months of moment.js
     */
    DateUtils.getLocaleMonths = function () {
        return moment.localeData()._months;
    };

    /**
     * @return {Array} of {String} short months,
     *  max three signs without dots
     */
    DateUtils.getLocaleMonthsShort = function () {
        return getShortMonths();
    };

    /**
     * @return {Array} of {String} weekDays of moment.js
     */
    DateUtils.getLocaleDays = function () {
        return moment.localeData()._weekdays;
    };

    /**
     * @return {Array} of {String} short weekDays of moment.js,
     *  means two or three sign without dots
     */
    DateUtils.getLocaleDaysShort = function () {
        return moment.localeData()._weekdaysMin;
    };

    // locale getter

    /**
     * @param {Number} month as nr, by default January is 0
     *  and Devember is 11
     *
     * @return {String} months of moment.js
     */
    DateUtils.getLocaleMonth = function (month) {
        return moment.months(month);
    };

    /**
     * @param {Number} month as nr, by default January (Jan) is 0
     *  and Devember (Dec) is 11
     *
     * @return {String} months,
     *  means max three signs without dots
     */
    DateUtils.getLocaleMonthShort = function (month) {
        return getShortMonths()[month];
    };

    /**
     * @param {Number} day as nr, by default Sunday is 0
     *  and Saturday is 6
     *
     * @return {String} weekDays of moment.js
     */
    DateUtils.getLocaleDay = function (day) {
        return moment.weekdays(day);
    };

    /**
     * @param {Number} day as nr, by default Sunday (Sun) is 0
     *  and Saturday (Sat) is 6
     *
     * @return {String} weekDaysMin of moment.js,
     *  means two or three sign without dots
     */
    DateUtils.getLocaleDayShort = function (day) {
        return moment.weekdaysMin(day);
    };

    DateUtils.isDateFormat = function (format) {
        return (/(D+|MM+|Y+)/).test(format);
    };

    DateUtils.isTimeFormat = function (format) {
        return (/(h+|m+|s+)/).test(format);
    };

    /**
     * Converts the UTC date time string to a date string in local time
     * (keeping the final letter 'Z', that indicates UTC).
     *
     * @param {String} utcString
     *  The UTC date string in ISO 8601 format.
     *
     * @param {Object} [options]
     *  A map of options to control the properties of the returned string.
     *  @param {Boolean} [options.useMilliSeconds=false]
     *      Whether the milliseconds shall be part of the returned string.
     *      Default is, that the string is returned without milliseconds.
     *
     * @returns {String}
     *  The current local date in an ISO string representation (including ending
     *  'Z', which is not quite correct).
     */
    DateUtils.convertUTCDateToLocalDate = function (utcString, options) {

        var // whether the seconds can be set to '00' in the returned string
            useMilliSeconds = Utils.getBooleanOption(options, 'useMilliSeconds', false),
            // the local date object
            localDate = new Date(utcString),
            // the local date string
            localDateString = null,
            // the offset in hours between local time and UTC time
            offset = localDate.getTimezoneOffset() / 60,
            // the hours of the local date
            hours = localDate.getHours();

        // hours outside the range 0..23 will affect the date automatically
        localDate.setHours(hours - offset);

        localDateString = localDate.toISOString();

        if (!useMilliSeconds) {
            // removing all digits after the final '.' together with this '.'
            localDateString = localDateString.replace(/\.\d{3}Z$/, 'Z');
        }

        return localDateString;
    };

    /**
     * Returns the current UTC time ISO date format. ISO date format:
     * '2012-04-17T13:38:00Z' with 'Z' at the end for UTC (+00:00)
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.useSeconds=true]
     *      Whether the seconds can be set to '00' in the returned string.
     *      Default is, that the string is returned with seconds and milliseconds.
     *
     * @returns {String}
     *  The current UTC date in an ISO string representation.
     */
    DateUtils.getIsoDateString = function (options) {

        var // the current date in UTC time zone
            d = new Date().toISOString(),
            // whether the seconds can be set to '00' in the returned string
            useSeconds = Utils.getBooleanOption(options, 'useSeconds', true);

        if (!useSeconds) {
            // replacing all digits before the trailing Z and after the final ':' with 0
            d = d.replace(/:[\d\.]+?Z$/, ':00Z');
        }

        return d;
    };

    /**
     * Returning the current local time faked as UTC time in ISO date format.
     * This is necessary, because the round trip with MS Office requires this.
     * So local time '2012-04-17' and '13:38:00' results in a string in
     * ISO date format: '2012-04-17T13:38:00Z'
     * 'Z' at the end for UTC (+00:00)
     * So string contains always a shifted time, except in UTC time zone.
     *
     * This functions shifts the offset in the milliseconds number and then
     * converts to date object, so that there are no problems with offsets
     * in hours, where the day, month or even year might be adapted.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.useSeconds=true]
     *      Whether the seconds can be set to '00' in the returned string.
     *      Default is, that the string is returned with seconds and milliseconds.
     *
     * @returns {String}
     *  The current local time as UTC date in an ISO string representation.
     */
    DateUtils.getMSFormatIsoDateString = function (options) {

        var // the local date object
            localDate = new Date(),
            // the local date time in milliseconds
            localTime = localDate.getTime(),
            // the offset in milliseconds between local time and UTC time
            offset = localDate.getTimezoneOffset() * 60 * 1000,
            // the shifted local time in milliseconds
            shiftedTime = localTime - offset,
            // the date string in ISO format
            isoDateString = new Date(shiftedTime).toISOString();

        if (!Utils.getBooleanOption(options, 'useSeconds', true)) {
            // replacing all digits before the trailing Z and after the final ':' with 0
            isoDateString = isoDateString.replace(/:[\d\.]+?Z$/, ':00Z');
        }

        return isoDateString;
    };

    // exports ================================================================

    return DateUtils;
});
