/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/locale/parser', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/arraytemplate',
    'io.ox/office/tk/locale/localedata'
], function (Utils, ArrayTemplate, LocaleData) {

    'use strict';

    var // all available text colors in format codes, as name to RGB map
        TEXT_COLORS = {
            black: '000000',
            blue: '0000FF',
            cyan: '00FFFF',
            green: '00FF00',
            magenta: 'FF00FF',
            red: 'FF0000',
            white: 'FFFFFF',
            yellow: 'FFFF00'
        },

        // a regular expression matching a text color in brackets at the beginning of a string
        TEXT_COLOR_TOKEN_RE = new RegExp('^\\[(' + _.keys(TEXT_COLORS).join('|') + ')\\]', 'i'),

        // characters for CJK date/time format code templates
        CJK_DATE_TIME_CHARACTERS = {
            traditional: { Y: '\u5e74', M: '\u6708', D: '\u65e5', h: '\u6642', m: '\u5206', s: '\u79d2' },
            simplified:  { Y: '\u5e74', M: '\u6708', D: '\u65e5', h: '\u65f6', m: '\u5206', s: '\u79d2' },
            hangul:      { Y: '\ub144', M: '\uc6d4', D: '\uc77c', h: '\uc2dc', m: '\ubd84', s: '\ucd08' }
        };

    // private global methods =================================================

    /**
     * Returns the regular expression used to match a floating-point number at
     * the beginning of a string. The regular expression will support decimal
     * and scientific notation, and will contain two matching groups:
     * - Group 1: The leading sign character (may be an empty string).
     * - Group 2: The floating-point number (decimal or scientific), without
     *      the sign character.
     *
     * @param {String} sep
     *  The decimal separator inserted into the regular expression.
     *
     * @returns {RegExp}
     *  The regular expression matching a floating-point number at the
     *  beginning of a string.
     */
    var getNumberRegExp = _.memoize(function (sep) {
        sep = _.escapeRegExp(sep);
        return new RegExp('^([-+]?)((?:\\d*' + sep + '\\d+|\\d+' + sep + '?)(?:E[-+]?\\d+)?)', 'i');
    });

    // class FormatToken ======================================================

    /**
     * A parsed token from a number format code.
     *
     * @constructor
     *
     * @property {String} type
     *  The type of this format token.
     *
     * @property {String} text
     *  The text represented by this format token, e.g. for literal text, or
     *  for specific delimiter tokens for numbers or dates.
     *
     * @property {Number} length
     *  The target length of this format token, e.g. for date/time tokens
     *  specifying the type or length of the date/time component.
     */
    function FormatToken(type, text, length) {

        this.type = type;
        this.text = text;
        this.length = length;

    } // class FormatToken

    // class FormatTokenArray =================================================

    /**
     * A typed array with instances of the class FormatToken.
     *
     * @constructor
     */
    var FormatTokenArray = ArrayTemplate.create(FormatToken);

    // class FormatSection ====================================================

    /**
     * Contains the parsed information of a single section in a format code.
     * Sections will be separated with semicolons in the format code and can be
     * used to define different number formats for specific intervals of
     * numbers, e.g. for positive and negative numbers.
     *
     * @constructor
     *
     * @property {FormatTokenArray} tokens
     *  An array with the parsed tokens of this section.
     *
     * @property {Boolean} dynamic
     *  Whether a value formatted according to this format section results in
     *  different text, depending on the available space, e.g. floating-point
     *  numbers formatted with the 'General' token that will be rounded to fit
     *  into the available space.
     *
     * @property {String|Null} colorName
     *  The lower-case name of the text color to be used, if this format
     *  section is used to format a value. The value null does not change the
     *  original text color of the value.
     *
     * @property {String|Null} rgbColor
     *  The RGB value of the text color to be used, if this format section is
     *  used to format a value. The value null does not change the original
     *  text color of the value.
     *
     * @property {String|Null} operator
     *  The operator used to define the number interval covered by this code
     *  section, together with the property 'boundary'. Must be one of '<'
     *  (less than), '<=' (less than or equal), '>' (greater than), '>='
     *  (greater than or equal), '=' (equal), or '<>' (not equal). The value
     *  null represents a missing operator (this format section will match all
     *  numbers).
     *
     * @property {Number} boundary
     *  The boundary value used to define the number interval covered by this
     *  code section, together with the property 'operator'.
     *
     * @property {Boolean} autoMinus
     *  Whether this code section will cause to automatically add a leading
     *  minus sign when formatting negative numbers. Happens if the interval of
     *  this code section covers at least negative numbers, and non-negative
     *  numbers (zero and/or positive numbers).
     *
     * @property {Number} percent
     *  The number of percent tokens contained in this section.
     *
     * @property {Boolean} scientific
     *  Whether this section contains an exponent token resulting in scientific
     *  notation.
     *
     * @property {Boolean} fractional
     *  Whether this section contains parsed tokens for the fractional part of
     *  a number.
     *
     * @property {Boolean} currency
     *  Whether this section contains a currency symbol.
     *
     * @property {Boolean} dateTime
     *  Whether this section contains any date/time tokens.
     *
     * @property {Boolean} hasAmPm
     *  Whether this section contains an AM/PM token that switches the time to
     *  12-hours format.
     *
     * @property {Boolean} text
     *  Whether this section contains a text token.
     *
     * @property {Boolean} fill
     *  Whether this section contains a fill token (a character that fills the
     *  remaining space available for the formatted text).
     */
    function FormatSection() {

        this.tokens = new FormatTokenArray();
        this.dynamic = false;

        this.colorName = null;
        this.rgbColor = null;

        this.operator = null;
        this.boundary = 0;
        this.autoMinus = true;

        this.percent = 0;
        this.scientific = false;
        this.fractional = false;
        this.currency = false;
        this.dateTime = false;
        this.hasAmPm = false;
        this.text = false;
        this.fill = false;

    } // class FormatSection

    // public methods ---------------------------------------------------------

    /**
     * Adds a new format token without literal text, and without a specific
     * length to this code section.
     *
     * @param {String} type
     *  The type of the new format token.
     */
    FormatSection.prototype.pushToken = function (type) {
        this.tokens.push(new FormatToken(type, '', 0));
    };

    /**
     * Adds a new format token with literal text to this code section, or
     * extends the text of the last token, if it has the same type.
     *
     * @param {String} type
     *  The type of the new format token.
     *
     * @param {String} text
     *  The text represented by the new format token, e.g. for literal text, or
     *  for specific delimiter tokens for numbers or dates.
     */
    FormatSection.prototype.pushTextToken = function (type, text) {
        var token = this.tokens.last();
        if (token && (token.type === type)) {
            token.text += text;
        } else {
            this.tokens.push(new FormatToken(type, text, 0));
        }
    };

    /**
     * Adds a new literal text token to this code section.
     *
     * @param {String} text
     *  The text literal represented by the new token.
     */
    FormatSection.prototype.pushLiteralToken = function (text) {
        this.pushTextToken('lit', text);
    };

    /**
     * Adds a new date/time token to this code section.
     *
     * @param {String} type
     *  The exact type of the date/time token. Must be one of 'year', 'month',
     *  'day', 'hour', 'minute', 'second', 'hours', 'minutes', or 'seconds'
     *  (the latter three types are for total number of hours, minutes, or
     *  seconds contained in a date).
     *
     * @param {Number} length
     *  The length of the date token. The meaning of this value depends on the
     *  type:
     *  - 'year': 2 for short years, 4 for full years.
     *  - 'month': 1 for short month number, 2 for month number with leading
     *      zero, 3 for abbreviated month name, 4 for full month name, or 5 for
     *      the first character of the month name.
     *  - 'day': 1 for short day number, 2 for day number with leading zero, 3
     *      for abbreviated name of week day, or 4 for full name of week day.
     *  - 'hour', 'minute', 'second': The length of the time token: 1 for short
     *      numbers, 2 for numbers with leading zero.
     *  - 'hours', 'minutes', 'seconds': The minimum length of the number, to
     *      be filled with leading zeros.
     */
    FormatSection.prototype.pushDateTimeToken = function (type, length) {
        this.tokens.push(new FormatToken(type, '', length));
        this.dateTime = true;
    };

    /**
     * Adds a new AM/PM token to this code section.
     *
     * @param {Number} length
     *  1, if the token represents the short variant ('a' or 'p'), or 2 for the
     *  long variant ('am' or 'pm').
     */
    FormatSection.prototype.pushAmPmToken = function (length) {
        this.pushDateTimeToken('ampm', length);
        this.hasAmPm = true;
    };

    /**
     * Sets the text color to be used to render a value formatted with this
     * format section.
     *
     * @param {String} colorName
     *  A valid predefined name of a text color.
     */
    FormatSection.prototype.setColor = function (colorName) {
        if (colorName in TEXT_COLORS) {
            this.colorName = colorName;
            this.rgbColor = TEXT_COLORS[colorName];
        } else {
            Utils.error('FormatSection.setColor(): invalid color "' + colorName + '"');
        }
    };

    /**
     * Changes the interval covered by this code section.
     *
     * @param {String} operator
     *  The operator used to define the number interval covered by this code
     *  section. See description of the class property 'operator' for details.
     *
     * @param {Number} boundary
     *  The boundary value used to define the number interval covered by this
     *  code section. See description of the class property 'boundary' for
     *  details.
     *
     * @param {Boolean} [def=false]
     *  Whether this invocation wants to initialize the default interval of
     *  this code section. If the section already contains a custom interval,
     *  it will not be changed.
     */
    FormatSection.prototype.setInterval = function (operator, boundary, def) {

        // do not change custom intervals, if default flag has been passed
        if (def && this.operator) { return; }

        // initialize 'autoMinus' flag, and check validity of the operator
        switch (operator) {
        case '<':  this.autoMinus = boundary > 0;  break;
        case '<=': this.autoMinus = boundary >= 0; break;
        case '>': // same auto-minus for both operators (e.g.: [>-1] and [>=-1] cover negative and zero, but [>0] and [>=0] do not)
        case '>=': this.autoMinus = boundary < 0;  break;
        case '=':  this.autoMinus = false;         break;
        case '<>': this.autoMinus = true;          break;
        default:
            Utils.error('FormatSection.setInterval(): invalid interval operator "' + operator + '"');
            return;
        }

        // store passed values, if operator is valid
        this.operator = operator;
        this.boundary = boundary;
    };

    /**
     * Returns whether this code section can be used to format the passed
     * number, according to the properties 'operator' and 'boundary'.
     *
     * @param {Number} number
     *  The number to be checked.
     *
     * @returns {Boolean}
     *  Whether this code section can be used to format the passed number.
     */
    FormatSection.prototype.contains = function (number) {
        switch (this.operator) {
        case '<':  return number < this.boundary;
        case '<=': return number <= this.boundary;
        case '>':  return number > this.boundary;
        case '>=': return number >= this.boundary;
        case '=':  return number === this.boundary;
        case '<>': return number !== this.boundary;
        }
        return true; // default for single code section covering all numbers
    };

    // static methods ---------------------------------------------------------

    /**
     * Parses the leading code section contained in the passed format code, and
     * creates a new instance of the class FormatSection if possible.
     *
     * @param {String} formatCode
     *  The number format code to be parsed.
     *
     * @returns {Object}
     *  The parser result, with the following properties:
     *  - {FormatSection|Null} section
     *      The new format section instance; or null, if the passed format code
     *      contains an unrecoverable syntax error.
     *  - {String|Null} remainder
     *      The remaining format code without the parsed code section; or null,
     *      if the entire format code has been parsed successfully.
     */
    FormatSection.parse = function (formatCode) {

        var // the current code section
            section = new FormatSection(),
            // the matches of the regular expressions
            matches = null,
            // whether the parser has encountered a syntax error
            parseError = false,
            // whether the parser has found a section separator
            hasSeparator = false,
            // whether last date/time token was of type hour
            lastWasHour = false;

        // stop if a code section, or the entire format code has been consumed
        while (!parseError && !hasSeparator && (formatCode.length > 0)) {

            // try/finally to be able to continue the while-loop at any place
            try {

                // code section separator
                if ((matches = /^;/.exec(formatCode))) {
                    hasSeparator = true;
                    continue;
                }

                // 'General' format token
                if ((matches = /^GENERAL/i.exec(formatCode))) {
                    section.pushToken('general');
                    section.dynamic = true;
                    continue;
                }

                // decimal fractional part
                if ((matches = /^\.[0?#]+/i.exec(formatCode))) {
                    section.pushTextToken('sign', matches[0]);
                    section.fractional = true;
                    continue;
                }

                // fractional notation
                if ((matches = /^[0?#]+\/[0?#]+/i.exec(formatCode))) {
                    section.pushTextToken('frac', matches[0]);
                    section.fractional = true;
                    continue;
                }

                // integral digits
                if ((matches = /^,?[0#?]+,?/.exec(formatCode))) {
                    section.pushTextToken('nr', matches[0]);
                    continue;
                }

                // scientific notation
                if ((matches = /^E[-+]([0?#]+)/i.exec(formatCode))) {
                    section.pushTextToken('scien', matches[1]);
                    section.scientific = true;
                    continue;
                }

                // date/time: 12 hours, AM/PM marker
                if ((matches = /^AM\/PM/i.exec(formatCode))) {
                    section.pushAmPmToken(2);
                    continue;
                }

                // date/time: 12 hours, A/P marker
                if ((matches = /^A\/P/i.exec(formatCode))) {
                    section.pushAmPmToken(1);
                    continue;
                }

                // date/time category: year
                if ((matches = /^Y+/i.exec(formatCode))) {
                    section.pushDateTimeToken('year', Math.min(4, Utils.roundUp(matches[0].length, 2))); // only YY and YYYY allowed
                    lastWasHour = false;
                    continue;
                }

                // date/time category: month
                if ((matches = /^M+/.exec(formatCode))) {
                    section.pushDateTimeToken('month', Math.min(5, matches[0].length)); // M to MMMMM allowed
                    lastWasHour = false;
                    continue;
                }

                // date/time category: day/weekday
                if ((matches = /^D+/i.exec(formatCode))) {
                    section.pushDateTimeToken('day', Math.min(4, matches[0].length)); // D to DDDD allowed
                    lastWasHour = false;
                    continue;
                }

                // date/time category: total number of hours
                if ((matches = /^\[(h+)\]/i.exec(formatCode))) {
                    section.pushDateTimeToken('hours', matches[1].length); // [h], [hh], [hhh], ... (unlimited length)
                    lastWasHour = true;
                    continue;
                }

                // date/time category: total number of minutes
                if ((matches = /^\[(m+)\]/.exec(formatCode))) {
                    section.pushDateTimeToken('minutes', matches[1].length); // [m], [mm], [mmm], ... (unlimited length)
                    continue;
                }

                // date/time category: total number of seconds
                if ((matches = /^\[(s+)\]/i.exec(formatCode))) {
                    section.pushDateTimeToken('seconds', matches[1].length); // [s], [ss], [sss], ... (unlimited length)
                    lastWasHour = false;
                    continue;
                }

                // date/time category: hour
                if ((matches = /^h+/i.exec(formatCode))) {
                    section.pushDateTimeToken('hour', Math.min(2, matches[0].length)); // h or hh allowed
                    lastWasHour = true;
                    continue;
                }

                // date/time category: minute together with seconds
                if ((matches = /^(m+)(?=\W+s)/i.exec(formatCode))) {
                    section.pushDateTimeToken('minute', Math.min(2, matches[1].length)); // m or mm allowed
                    lastWasHour = false;
                    continue;
                }

                // date/time category: minute
                if ((matches = /^m+/.exec(formatCode))) {
                    if (lastWasHour) {
                        section.pushDateTimeToken('minute', Math.min(2, matches[0].length)); // m or mm allowed
                    } else {
                        section.pushDateTimeToken('month', Math.min(5, matches[0].length)); // M to MMMMM allowed
                    }
                    lastWasHour = false;
                    continue;
                }

                // date/time category: second
                if ((matches = /^s+/i.exec(formatCode))) {
                    section.pushDateTimeToken('second', Math.min(2, matches[0].length)); // s or ss allowed
                    lastWasHour = false;
                    continue;
                }

                // percent sign
                if ((matches = /^%/.exec(formatCode))) {
                    section.pushLiteralToken('%');
                    section.percent += 1;
                    continue;
                }

                // string literal, enclosed in double-quotes
                if ((matches = /^"([^"]*)"/.exec(formatCode))) {
                    section.pushLiteralToken(matches[1]);
                    continue;
                }

                // character literal, preceded by backspace
                if ((matches = /^\\(.)/.exec(formatCode))) {
                    section.pushLiteralToken(matches[1]);
                    continue;
                }

                // whitespace (TODO: store and handle exact character width)
                if ((matches = /^_./.exec(formatCode))) {
                    section.pushLiteralToken(' ');
                    continue;
                }

                // repeated fill character
                if ((matches = /^\*(.)/.exec(formatCode))) {
                    // subsequent fill characters will be inserted once literally
                    // (only the first fill character in the section will be expanded)
                    if (section.fill) {
                        section.pushLiteralToken(matches[1]);
                    } else {
                        section.pushTextToken('fill', matches[1]);
                        section.dynamic = section.fill = true;
                    }
                    continue;
                }

                // number interval
                if ((matches = /^\[(<=?|>=?|<>|=)([^[\];]+)\]/.exec(formatCode))) {
                    var parseResult = Parser.parseLeadingNumber(matches[2], { dec: '.', sign: true, complete: true });
                    if (parseResult && (parseResult.remaining.length === 0)) {
                        section.setInterval(matches[1], parseResult.number);
                    } else {
                        parseError = true;
                    }
                    continue;
                }

                // text color
                if ((matches = TEXT_COLOR_TOKEN_RE.exec(formatCode))) {
                    section.setColor(matches[1].toLowerCase());
                    continue;
                }

                // locale data with hexadecimal LCID and calendar information
                if ((matches = /^\[\$-([0-9a-f]+)\]/i.exec(formatCode))) {
                    continue;
                }

                // currency symbol with hexadecimal LCID
                if ((matches = /^\[\$([^[\];-]+)-([0-9a-f]+)\]/i.exec(formatCode))) {
                    section.pushLiteralToken(matches[1]);
                    section.currency = true;
                    continue;
                }

                // ignore other unsupported specifiers
                if ((matches = /^(\[[^[\];]*\])/.exec(formatCode))) {
                    Utils.warn('FormatSection.parse(): unsupported section specifier "' + matches[1] + '"');
                    continue;
                }

                // placeholder for text value
                if ((matches = /^@/.exec(formatCode))) {
                    section.pushToken('text');
                    section.text = true;
                    continue;
                }

                // unmatched double-quote, unmatched opening bracket, or incomplete escape sequence: syntax error
                if (/^["\[\\_*]/.test(formatCode)) {
                    parseError = true;
                    continue;
                }

                // any non-matching token is considered a character literal
                matches = [formatCode[0]];
                section.pushLiteralToken(matches[0]);
            }
            // remove the current match from the formula string
            finally {
                formatCode = formatCode.slice(matches[0].length);
            }
        }

        // create and return the parser result
        return {
            section: parseError ? null : section,
            remainder: hasSeparator ? formatCode : null
        };
    };

    // class FormatSectionArray ===============================================

    /**
     * A typed array with instances of the class FormatSection.
     *
     * @constructor
     */
    var FormatSectionArray = ArrayTemplate.create(FormatSection);

    // class ParsedFormat =====================================================

    /**
     * Represents a parsed number format code.
     *
     * @constructor
     *
     * @property {FormatSectionArray} numberSections
     *  An array of format code sections for different number intervals.
     *
     * @property {FormatSection} textSection
     *  A dedicated code section to format string values.
     *
     * @property {Boolean} syntaxError
     *  Whether the format code has has been used to create this instance
     *  contains a syntax error.
     */
    function ParsedFormat() {

        this.numberSections = new FormatSectionArray();
        this.textSection = null;
        this.syntaxError = false;

    } // class ParsedFormat

    // public methods ---------------------------------------------------------

    /**
     * Parses the leading code section contained in the passed format code, and
     * inserts a new instance of the class FormatSection into this object if
     * possible. If the passed format code contains a syntax error, the public
     * flag 'syntaxError' will be set to true.
     *
     * @param {String} formatCode
     *  The number format code to be parsed.
     *
     * @returns {String|Null}
     *  The remaining format code without the parsed code section; or null, if
     *  the entire format code has been parsed successfully.
     */
    ParsedFormat.prototype.parseSection = function (formatCode) {

        var // result of the section parser (new code section, and remaining format code)
            parseResult = FormatSection.parse(formatCode);

        // missing code section indicates syntax error
        if (!parseResult.section) {
            this.syntaxError = true;
            return null;
        }

        // there can only be a single text sections at the end of the format code
        if (this.textSection) {
            this.syntaxError = true;
            return null;
        }

        // store the new code section in this instance
        if (parseResult.section.text) {
            this.textSection = parseResult.section;
        } else {
            this.numberSections.push(parseResult.section);
        }

        // return the remaining format code
        return parseResult.remainder;
    };

    /**
     * Returns an existing code section, according to the passed value.
     *
     * @param {Any} value
     *  The value used to pick a code section. For finite numbers, one of the
     *  numeric sections will be returned. For strings, the text section will
     *  be returned. For all other values, null will be returned.
     *
     * @returns {FormatSection|Null}
     *  The format code section for the passed value, if available; otherwise
     *  null.
     */
    ParsedFormat.prototype.getSection = function (value) {

        // do not return sections, if the format code contains a syntax error
        if (this.syntaxError) { return null; }

        // return text section for strings
        if (_.isString(value)) { return this.textSection; }

        // reject other values than finite numbers
        if (!_.isNumber(value) || !isFinite(value)) { return null; }

        // pick first matching section for numbers
        return this.numberSections.find(function (s) { return s.contains(value); });
    };

    // static class Parser ====================================================

    /**
     * A static class providing methods to parse numbers, dates, format codes,
     * and other locale-dependent source data.
     */
    var Parser = {};

    // static methods ---------------------------------------------------------

    /**
     * Converts the passed number to a string, using the decimal separator of
     * the current UI locale. This is a simple method that uses JavaScript's
     * conversion from numbers to strings.
     *
     * @param {Number} number
     *  The number to be converted.
     *
     * @returns {String}
     *  The passed number, converted to string, with the decimal separator of
     *  the current UI locale.
     */
    Parser.numberToString = function (number) {
        return String(number).replace(/\./g, LocaleData.DEC);
    };

    /**
     * Converts the passed string to a number, using the decimal separator of
     * the current UI locale. This is a simple method that uses JavaScript's
     * parseFloat() conversion function.
     *
     * @param {String} text
     *  The text to be converted to a number.
     *
     * @returns {Number}
     *  The passed number, converted to string. If the passed text does not
     *  represent a floating-point number, NaN will be returned. If the text
     *  consists of a number with following garbage (e.g. '123abc'), the
     *  leading number will be returned (same behavior as JavaScript's native
     *  parseFloat() function).
     */
    Parser.stringToNumber = function (text) {
        return parseFloat(text.replace(LocaleData.DEC, '.'));
    };

    /**
     * Tries to parse a floating-point number from the leading part of the
     * passed string.
     *
     * @param {String} text
     *  The text containing a floating-point number.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.sep]
     *      If specified, a custom decimal separator character. By default, the
     *      decimal separator of the current UI language will be used.
     *  @param {Boolean} [options.sign=false]
     *      If set to true, the passed string may contain a leading sign
     *      character ('-' or '+').
     *  @param {Boolean} [options.complete=false]
     *      If set to true, the entire passed string must represent a valid
     *      floating-point number. No remaining 'garbage' text will be
     *      accepted.
     *
     * @returns {Object|Null}
     *  A result descriptor, if the passed string starts with a valid floating-
     *  point number (according to the passed options); otherwise null. The
     *  result descriptor contains the following properties:
     *  - {Number} number
     *      The parsed floating-point number.
     *  - {String} text
     *      The leading text from the passed string that has been parsed to the
     *      floating-point number.
     *  - {String} sign
     *      The sign character. Will be the empty string, if no sign exists in
     *      the passed text, or if the option 'sign' has not been set
     *      (see above).
     *  - {String} remaining
     *      The remaining characters from the passed string following the
     *      parsed floating-point number. Will be the empty string, if the
     *      option 'complete' has been set (see above).
     */
    Parser.parseLeadingNumber = function (text, options) {

        var // the decimal separator used to parse the number
            sep = Utils.getStringOption(options, 'sep', LocaleData.DEC),
            // whether to accept a leading sign
            sign = Utils.getBooleanOption(options, 'sign', false),
            // whether to reject trailing garbage
            complete = Utils.getBooleanOption(options, 'complete', false),
            // try to parse the passed string for a leading floating-point number
            matches = getNumberRegExp(sep).exec(text),
            // the parsed floating-point number
            number = null;

        // reject leading sign unless specified in the passed options
        if (!matches || (!sign && matches[1].length > 0)) { return null; }

        // reject trailing garbage if specified in the passed options
        if (complete && (matches[0].length < text.length)) { return null; }

        // parse the floating-point number
        number = parseFloat(matches[2].replace(sep, '.'));
        if (!_.isFinite(number)) { return null; }
        if (matches[1] === '-') { number = -number; }

        // build the result descriptor
        return { number: number, text: matches[0], sign: matches[1], remaining: text.substr(matches[0].length) };
    };

    /**
     * Parses a format code, and returns an instance of the class ParsedFormat
     * containing all properties of the format code needed to format numbers or
     * strings.
     *
     * @param {String} formatCode
     *  The format code to be parsed.
     *
     * @returns {ParsedFormat}
     *  An instance of the class ParsedFormat containing all properties of the
     *  format code needed to format numbers or strings.
     */
    Parser.parseFormatCode = _.memoize(function (formatCode) {

        // the resulting parsed format code
        var parsedFormat = new ParsedFormat();

        // empty format code string is considered invalid
        parsedFormat.syntaxError = formatCode.length === 0;

        // stop if the entire format code has been consumed, or a syntax error was found
        while (!parsedFormat.syntaxError && _.isString(formatCode)) {
            formatCode = parsedFormat.parseSection(formatCode);
        }

        // initialize default intervals of numeric code sections
        switch (parsedFormat.numberSections.length) {
        case 0:
            // text format only: format numbers as 'General' number format
            parsedFormat.numberSections.push(new FormatSection());
            parsedFormat.numberSections[0].pushToken('general');
            break;
        case 1:
            // one number section for all numbers
            break;
        case 2:
            // two number sections: positive and zero, negative
            parsedFormat.numberSections[0].setInterval('>=', 0, true);
            parsedFormat.numberSections[1].setInterval('<', 0, true);
            break;
        case 3:
            // three number sections: positive, negative, zero
            parsedFormat.numberSections[0].setInterval('>', 0, true);
            parsedFormat.numberSections[1].setInterval('<', 0, true);
            parsedFormat.numberSections[2].setInterval('=', 0, true);
            break;
        default:
            // invalid number of number sections
            parsedFormat.syntaxError = true;
        }

        // initialize missing text section
        if (!parsedFormat.textSection) {
            parsedFormat.textSection = new FormatSection();
            parsedFormat.textSection.pushToken('text');
            var numSection = parsedFormat.numberSections[0];
            // use color of first number section, but only if it is the 'General' format code
            if (numSection.colorName && (numSection.tokens.length === 1) && (numSection.tokens[0].type === 'general')) {
                parsedFormat.textSection.setColor(parsedFormat.numberSections[0].colorName);
            }
        } else if (parsedFormat.textSection.operator) {
            // text section with number interval is invalid
            parsedFormat.syntaxError = true;
        }

        return parsedFormat;
    });

    /**
     * Generates a date/time format code for CJK locales based on the passed
     * format code template.
     *
     * @param {String} formatCodeTemplate
     *  The format code template used to generate the date/time format code.
     *  May contain the meta tokens {YEAR}, {MONTH}, {DAY}, {HOUR}, {MINUTE},
     *  and {SECOND}, which will be replaced by the respective CJK ideographs
     *  according to the passed CJK script type (language); and the meta token
     *  {H} that will be replaced by an hour token according to the passed
     *  options.
     *
     * @param {String} script
     *  The CJK script type used to replace the meta tokens in the passed
     *  format code template. MUST be one of:
     *  - 'traditional': CJK Unified Ideographs (Traditional Chinese, Japanese)
     *  - 'simplified': CJK Unified Ideographs (Simplified Chinese)
     *  - 'hangul': Hangul alphabet (Korean).
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.ampm=false]
     *      If set to true, the token 'AM/PM' will be added in front of the
     *      hour token inserted for the meta token {H}.
     *  @param {Boolean} [options.long=false]
     *      If set to true, the hour token inserted for the meta token {H} will
     *      be 'hh', otherwise 'h'.
     *
     * @returns {String}
     *  The format code generated from the passed format code template.
     */
    Parser.generateCJKDateTimeFormat = function (formatCodeTemplate, script, options) {
        var chars = CJK_DATE_TIME_CHARACTERS[script];
        return formatCodeTemplate
            .replace(/{H}/g, ((options && options.ampm) ? 'AM/PM' : '') + ((options && options.long) ? 'hh' : 'h'))
            .replace(/{YEAR}/g, chars.Y)
            .replace(/{MONTH}/g, chars.M)
            .replace(/{DAY}/g, chars.D)
            .replace(/{HOUR}/g, chars.h)
            .replace(/{MINUTE}/g, chars.m)
            .replace(/{SECOND}/g, chars.s);
    };

    // exports ================================================================

    return Parser;

});
