/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/tk/control/unitfield', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/parser',
    'io.ox/office/settings/units',
    'io.ox/office/tk/control/spinfield'
], function (Utils, Parser, Units, SpinField) {

    'use strict';

    // class UnitValidator ====================================================

    /**
     * A validator for unit fields that restricts the allowed values to
     * floating-point numbers and optionally an additional unit. Supported
     * values for the unit are 'px', 'pc', 'pt', 'in', 'cm' and 'mm'. If the
     * unit cannot be matched with one unit of this list, 'mm' is used as
     * fall-back unit.
     *
     * @constructor
     *
     * @extends SpinField.NumberValidator
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options that are supported by the
     *  base class NumberValidator.
     */
    var UnitValidator = SpinField.NumberValidator.extend({ constructor: function (initOptions) {

        var // the regular expression for the unit field
            regex = null;

        // base constructor ---------------------------------------------------

        SpinField.NumberValidator.call(this, initOptions);

        // methods ------------------------------------------------------------

        /**
         * Converts the passed value to a display string with measurement unit.
         *
         * @param {Number} value
         *  The length in 1/100 mm.
         *
         * @returns {String}
         *  The display string for the passed value, with measurement unit. If
         *  the value is invalid, an empty string will be returned.
         */
        this.valueToText = function (value) {
            if (_.isFinite(value)) {
                var mm = Utils.round(value, this.getPrecision()) / 100;
                var unit = Units.getStandardUnit();
                var result = Utils.convertLength(mm, 'mm', unit, Units.getStandardUnitStep());
                return Parser.numberToString(result) + ' ' + unit;
            }
            return '';
        };

        /**
         * Converts the passed display string of a length with measurement unit
         * to a number. Supported unit strings are 'px', 'pc', 'pt', 'in', 'cm'
         * and 'mm'. If this unit string is not defined or cannot be resolved
         * to one of the listed units, 'mm' is used as default, to calculate
         * the length in 1/100 mm.
         *
         * @param {String} text
         *  The display string of a length with measurement unit.
         *
         * @returns {Number|Null}
         *  The length in 1/100 mm or null, if the passed text is invalid.
         */
        this.textToValue = function (text) {

            var matches = regex.exec(text) || [],
                value = matches[1] || '',
                unit = (matches[3] || '').toLowerCase();

            value = Parser.stringToNumber(value);
            if (_.isNumber(value) && _.isFinite(value)) {
                unit = /^(px|pc|pt|in|cm|mm)$/.test(unit) ? unit : 'mm';
                return this.restrictValue(Utils.convertLengthToHmm(value, unit));
            }

            return null;
        };

        /**
         * The validation functions that controls the input into the unit field
         * corresponding to the defined regular expression.
         *
         * @returns {Boolean}
         *  Whether the input into the unit field matches the defined regular
         *  expression.
         */
        this.validate = function (text) {
            return regex.test(text);
        };

        // initialization -----------------------------------------------------

        // the regular expression for the unit field
        regex = new RegExp('^(' + ((this.getMin() < 0) ? '[+-]?' : '\\+?') + '[0-9]*([.,][0-9]*)?)\\s*([a-zA-Z]*)$');

    }}); // class UnitValidator

    // class UnitField ========================================================

    /**
     * A specialized spin field used to represent and input a length value
     * together with a measurement unit.
     *
     * @constructor
     *
     * @extends SpinField
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the SpinField base class
     *  and the UnitValidator class (a sub class of TextField.NumberValidator),
     *  except the option 'validator' which will be set to an instance of
     *  UnitValidator.
     */
    function UnitField(initOptions) {

        // base constructor ---------------------------------------------------

        // bug 31765: force keyboard type to 'text', Android does not like characters in numeric text fields
        SpinField.call(this, Utils.extendOptions(initOptions, { keyboard: 'text', validator: new UnitValidator(initOptions) }));

    } // class UnitField

    // exports ================================================================

    // derive this class from class SpinField
    return SpinField.extend({ constructor: UnitField });

});
