/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/text/view/popup/universalcontextmenu', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/text/dom',
    'io.ox/office/text/position',
    'io.ox/office/text/view/controls',
    'io.ox/office/text/view/labels',
    'io.ox/office/baseframework/view/popup/contextmenu',
    'io.ox/office/editframework/view/control/languagepicker',
    'io.ox/office/drawinglayer/view/drawingframe',
    'gettext!io.ox/office/text/main'
], function (Utils, AttributeUtils, Dom, Position, Controls, Labels, ContextMenu, LanguagePicker, DrawingFrame, gt) {

    'use strict';

    // class UniversalContextMenu ================================================

    /**
     * A context menu for normal text. Provides menu actions to manipulate
     * entire text.
     *
     * @constructor
     *
     * @extends ContextMenu
     */
    function UniversalContextMenu(docView, initOptions) {

        var // self reference
            self = this,

            // document model
            docModel = docView.getDocModel(),
            editDiv = docModel.getNode(),

            // selection
            selection = docModel.getSelection(),
            // spell checker
            spellChecker = docModel.getSpellChecker(),

            // field manager instance
            fieldManager = docModel.getFieldManager(),

            // the clicked node
            node = null,
            // the type of the clicked node (drawing, table, ...)
            eleType = null,

            // is a range is selected or not
            isRangeSelected = false,
            isDrawingSelected = false,
            // xox position of the cursor
            resetSelectionTo = null,

            // attributes of the current text selection (to detect hyperlinks aso.)
            characterAttrs = null,

            // needs to use special behavior for osx chrome
            chromeOnOSX = (_.browser.MacOS && _.browser.Chrome);

        // base constructor ---------------------------------------------------

        ContextMenu.call(this, docView, editDiv, Utils.extendOptions({
            enableKey: Utils.getArrayOption(initOptions, 'enableKey', ['document/editable']),
            delay: Utils.getNumberOption(initOptions, 'delay', 0)
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * A handler that should be invoked on contextmenu event (see in global 'contextmenu.js')
         */
        function contextMenuEventHander(event) {
            if (chromeOnOSX) {
                var oxoPosition = Position.getOxoPositionFromPixelPosition(editDiv, event.pageX, event.pageY),
                    selectionStartPos = selection.getStartPosition(),
                    selectionEndPos = selection.getEndPosition();

                if (!_.isNull(oxoPosition)) {
                    // click into table
                    if (Position.isPositionInTable(editDiv, oxoPosition.start)) {

                        var reset = false;

                        if (Dom.isParagraphNode(event.target) && Dom.isEmptyParagraph(event.target)) { reset = true; }
                        if (Dom.isSpan(event.target) && Dom.isEmptySpan(event.target)) { reset = true; }
                        if (Dom.isCellNode(event.target)) { reset = true; oxoPosition.start.push(0); }

                        if (reset) {
                            resetSelectionTo = oxoPosition.start;
                            selection.setTextSelection(resetSelectionTo);
                            return;
                        }
                    }

                    // click into another paragraph
                    if (_.first(oxoPosition.start) !== _.first(selectionStartPos) && _.first(oxoPosition.start) !== _.first(selectionEndPos)) {
                        resetSelectionTo = oxoPosition.start;
                        selection.setTextSelection(resetSelectionTo);

                    // click into the same paragraph
                    } else {
                        // don't click on the selection
                        if (!Position.isNodePositionInsideRange(oxoPosition.start, _.last(selection.getStartPosition()), _.last(selection.getEndPosition()))) {
                            resetSelectionTo = oxoPosition.start;
                            selection.setTextSelection(resetSelectionTo);
                        }
                    }
                }
            }
        }

        /**
         * A handler that should be invoked on detect (see in global 'contextmenu.js')
         */
        function detectHandler() {
            isRangeSelected = !selection.isTextCursor();
        }

        /**
         * A handler that should be invoked on reset (see in global 'contextmenu.js')
         * The following options are supported:
         * @param {Object} [options]
         *  @param {Boolean} [options.resetTextSelection=false]
         *      Resets the text selection, if the resetSelectionTo variable
         *      isn't null
         */
        function resetHandler(options) {
            if (Utils.getBooleanOption(options, 'resetTextSelection', false) && !_.isNull(resetSelectionTo)) {
                selection.setTextSelection(resetSelectionTo);
            }
            if (!_.isNull(resetSelectionTo)) {
                resetSelectionTo = null;
            }
        }

        function initMenu() {
            // Is Drawing
            if (Dom.isDrawingFrame(eleType)) {
                if (!Dom.isInCommentNode(node)) {
                    prepareDrawingMenu();
                    return;
                }

            // Is Table
            } else if (Dom.isTableNode(eleType) || Dom.isCellNode(eleType)) {
                if (Dom.isCellNode(eleType)) {
                    if (Dom.isSpan(node) && !Dom.isEmptyCell(Dom.getCellContentNode($(node).parent()))) {
                        prepareTextMenu();
                        return;
                    }
                }
                prepareTableMenu();
                return;

            // Is Text
            } else {
                // Is TextFrame
                if (selection.isAdditionalTextframeSelection() && !Dom.isTextSpan(node)) {
                    prepareDrawingMenu();
                    return;
                }
                prepareTextMenu();
                return;
            }
        }

        // preparation of context menus ---------------------------------------
        /**
         * Prepares the context menu for text
         */
        function prepareTextMenu() {
            if (Dom.isChangeTrackNode(node) && !isRangeSelected) {
                self.addGroup('acceptSelectedChangeTracking', new Controls.Button(Labels.ACCEPT_CURRENT_OPTIONS))
                    .addGroup('rejectSelectedChangeTracking', new Controls.Button(Labels.REJECT_CURRENT_OPTIONS))
                    .addGroup('acceptChangeTracking', new Controls.Button(Labels.ACCEPT_ALL_OPTIONS))
                    .addGroup('rejectChangeTracking', new Controls.Button(Labels.REJECT_ALL_OPTIONS));

            } else if (fieldManager.isHighlightState()) {
                self.addGroup('updateField', new Controls.Button(Labels.UPDATE_FIELD))
                    .addGroup('updateAllFields', new Controls.Button(Labels.UPDATE_ALL_FIELDS));

            } else {
                characterAttrs = AttributeUtils.getExplicitAttributes(node, { family: 'character', direct: true });

                // Spellcheck
                if (characterAttrs.spellerror && !isRangeSelected) {
                    var spellResult = spellChecker.getSpellErrorWord();
                    if (spellResult.replacements && spellResult.replacements.length > 0) {
                        _.each(spellResult.replacements, function (replace) {
                            self.addGroup('document/spelling/replace', new Controls.Button({ label: replace, value: replace }));
                        });
                    }
                    self.addSeparator()
                        .addGroup('character/language', new LanguagePicker({ autoCloseParent: false, autoHideGroups: true, updateCaptionMode: 'none', label: gt('Language'), icon: 'fa-chevron-right', iconPos: 'trailing', caret: false, showListIcons: false, anchorBorder: 'right left' }));

                // Hyperlink
                } else if (characterAttrs.url && !isRangeSelected) {
                    self.addGroup('character/hyperlink/dialog', new Controls.Button({ label: Labels.EDIT_HYPERLINK_LABEL }))
                        .addGroup('character/hyperlink/remove', new Controls.Button({ label: Labels.REMOVE_HYPERLINK_LABEL }))
                        .addGroup(null, new Controls.Button({ label: Labels.OPEN_HYPERLINK_LABEL, href: characterAttrs.url }));

                    //AppTooltipMixin
                    docView.getApp().enableTooltipOnShow(self, characterAttrs.url);

                // Normal Text
                } else {
                    self.addGroup(null, new Controls.CompoundButton(docView, { autoCloseParent: false, autoHideGroups: true, label: Labels.INSERT_HEADER_LABEL, anchorBorder: 'right left', icon: 'fa-chevron-right', iconPos: 'trailing', caret: false })
                            .addGroup('image/insert/dialog', new Controls.Button(Labels.INSERT_IMAGE_OPTIONS))
                            .addGroup('textframe/insert', new Controls.Button(Labels.INSERT_TEXTFRAME_OPTIONS))
                            .addGroup('comment/insert', new Controls.Button(Labels.INSERT_COMMENT_OPTIONS))
                            .addGroup('character/hyperlink/dialog', new Controls.Button(Labels.INSERT_HYPERLINK_OPTIONS))
                        );
                    if (!Utils.SMALL_DEVICE) {
                        self.addGroup(null, new Controls.InTextStyleContextSubMenu(docView, { autoCloseParent: false, autoHideGroups: true, anchorBorder: 'right left', icon: 'fa-chevron-right', iconPos: 'trailing', caret: false }));
                    }
                    self.addSeparator()
                        .addGroup('character/language', new LanguagePicker({ autoCloseParent: false, autoHideGroups: true, updateCaptionMode: 'none', label: gt('Language'), icon: 'fa-chevron-right', iconPos: 'trailing', caret: false, showListIcons: false, anchorBorder: 'right left' }));
                }
            }

            self.refreshImmediately();
        }

        /**
         * Prepares the context menu for drawings
         */
        function prepareDrawingMenu() {
            self.addGroup('drawing/delete', new Controls.Button({ label: Labels.DELETE_DRAWING_LABEL, tooltip: Labels.DELETE_DRAWING_TOOLTIP }));
        }

        /**
         * Prepares the context menu for tables
         */
        function prepareTableMenu() {
            self.addGroup('table/insert/row', new Controls.Button({ label: gt('Insert row') }))
                .addGroup('table/delete/row', new Controls.Button({ label: gt('Delete row') }))
                .addGroup('table/insert/column', new Controls.Button({ label: gt('Insert column') }))
                .addGroup('table/delete/column', new Controls.Button({ label: gt('Delete column') }))
                .addGroup('table/delete', new Controls.Button({ label: gt('Delete table') }))
                .addSeparator()
                .addGroup(null, new Controls.CompoundButton(docView, { autoCloseParent: false, autoHideGroups: true, label: Labels.INSERT_HEADER_LABEL, anchorBorder: 'right left', icon: 'fa-chevron-right', iconPos: 'trailing', caret: false })
                    .addGroup('image/insert/dialog', new Controls.Button(Labels.INSERT_IMAGE_OPTIONS))
                    .addGroup('textframe/insert', new Controls.Button(Labels.INSERT_TEXTFRAME_OPTIONS))
                    .addGroup('comment/insert', new Controls.Button(Labels.INSERT_COMMENT_OPTIONS))
                    .addGroup('character/hyperlink/dialog', new Controls.Button(Labels.INSERT_HYPERLINK_OPTIONS))
                );
        }

        /**
         * Prepares the context menu.
         * Checks if a table, drawing or something else was clicked
         * and initialize the correct menu
         */
        function contextMenuPrepareHandler(event) {
            var sourceEvent     = event.sourceEvent,
                sourceTarget    = (sourceEvent) ? sourceEvent.target : null;

            selection           = docModel.getSelection();
            isDrawingSelected   = docModel.isDrawingSelected();
            isRangeSelected     = (isDrawingSelected) ? false : !selection.isTextCursor();

            if (!sourceTarget && !isDrawingSelected) { return; }

            if (isDrawingSelected) {
                node = selection.getSelectedDrawing()[0];
            } else {
                node = (_.browser.IE && $(sourceTarget).is('.page')) ? $(window.getSelection().focusNode.parentNode) : sourceTarget;
            }

            var arrDomTree = Dom.getDomTree(node, {
                    endNodes: [
                        DrawingFrame.isTextFrameNode,
                        Dom.isDrawingFrame,
                        Dom.isCellNode,
                        Dom.isTableNode
                    ]
                });

            eleType = _.last(arrDomTree);

            self.destroyAllGroups();
        }

        // public methods ----------------------------------------------------
        /**
         * Returns whether a range is selected or not
         * @return {Boolean}
         *   Is a range selected or not
         */
        this.isRangeSelected = function () {
            return isRangeSelected;
        };

        this.isDrawingSelected = function () {
            return isDrawingSelected;
        };

        this.getXY = function (event) {
            if (isDrawingSelected) {
                return {
                    pageX: event.pageX,
                    pageY: event.pageY
                };

            } else if (_.browser.IE && $(event.target).is('.page')) {
                var focusNode = $(window.getSelection().focusNode.parentNode),
                    xy = focusNode.offset();

                return {
                    pageX: xy.left,
                    pageY: xy.top
                };

            } else {
                return (event.sourceEvent) ? {
                    pageX: event.sourceEvent.pageX,
                    pageY: event.sourceEvent.pageY
                } : null;
            }
        };

        // initialization -----------------------------------------------------

        // register some special handler for text
        this.registerContextMenuHandler(contextMenuEventHander);
        this.registerDetectHandler(detectHandler);
        this.registerResetHandler(resetHandler);

        // preprocessing before the context menu will be shown
        this.on('contextmenu:prepare', contextMenuPrepareHandler);

        this.on('popup:beforeshow', initMenu);

        this.listenTo(docView.getContentRootNode(), 'scroll', function () { self.hide(); });

        // destroy all class members
        this.registerDestructor(function () {
            self = docView = docModel = editDiv = initOptions = null;
            selection = spellChecker = fieldManager = null;
        });

    } // class UniversalContextMenu

    // exports ================================================================

    // derive this class from class ContextMenu
    return ContextMenu.extend({ constructor: UniversalContextMenu });

});
