/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/operationsgenerator', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/model/operationsgenerator',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/text/utils/operations',
    'io.ox/office/text/dom',
    'io.ox/office/text/position',
    'io.ox/office/drawinglayer/view/imageutil'
], function (Utils, AttributeUtils, OperationsGenerator, DrawingFrame, Operations, DOM, Position, Image) {

    'use strict';

    // class TextOperationsGenerator ==========================================

    /**
     * An instance of this class contains an operations array and provides
     * methods to generate operations for various element nodes.
     *
     * @constructor
     *
     * @extends OperationsGenerator
     *
     * @param {TextModel} app
     *  The document model containing this generator.
     */
    function TextOperationsGenerator(docModel) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        OperationsGenerator.call(this, docModel);

        // methods ------------------------------------------------------------

        /**
         * Creates and appends a new operation to the operations array. Adds
         * explicit attributes of the passed node to the 'attrs' option of the
         * new operation.
         *
         * @param {HTMLElement|jQuery} element
         *  The element node that may contain explicit formatting attributes.
         *  If this object is a jQuery collection, uses the first node it
         *  contains.
         *
         * @param {String} name
         *  The name of the operation.
         *
         * @param {Object} [operationOptions]
         *  Additional options that will be stored in the operation.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.ignoreFamily]
         *      If specified, this property will be ignored in the generated
         *      operation.
         *  @param {String} [options.target]
         *      If specified, this property will be send as target with operation.
         *
         * @returns {Object}
         *  The created operation object.
         */
        this.generateOperationWithAttributes = function (element, name, operationOptions, options) {

            var // explicit attributes of the passed node
                attributes = AttributeUtils.getExplicitAttributes(element),
                // a specified property, that shall be ignored in the setAttributes operation
                ignoreFamily = Utils.getStringOption(options, 'ignoreFamily', ''),
                // a specified property, that shall send target string with operation
                target = Utils.getStringOption(options, 'target', '');

            // deleting the specified property from the attributes
            if (attributes && ignoreFamily.length > 0 && attributes[ignoreFamily]) { delete attributes[ignoreFamily]; }

            // add the 'attrs' entry if there are attributes
            if (!_.isEmpty(attributes)) {
                operationOptions = _.extend({ attrs: attributes }, operationOptions);
            }

            // push the operation
            return this.generateOperation(name, extendPropertiesWithTarget(operationOptions, target));
        };

        /**
         * Generates the 'setAttributes' operation needed to set the explicit
         * formatting attributes of the passed element node. If the passed node
         * does not contain any explicit attributes, no operation will be
         * generated.
         *
         * @param {HTMLElement|jQuery} element
         *  The element node whose formatting attributes will be converted to
         *  an operation. If this object is a jQuery collection, uses the first
         *  node it contains.
         *
         * @param {Object} position
         *  All operation attributes describing the logical position of the
         *  passed element in the document model. All attributes will be added
         *  to the generated operation.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.clearFamily]
         *      If specified, a style family for which additional formatting
         *      attributes with null values will be inserted into the
         *      'setAttributes' operation.
         *  @param {String} [options.ignoreFamily]
         *      If specified, this property will be ignored in the generated
         *      'setAttributes' operation.
         *  @param {String} [options.target]
         *      If specified, this property will be send as target with operation.
         *
         * @returns {OperationsGenerator}
         *  A reference to this instance.
         */
        this.generateSetAttributesOperation = function (element, position, options) {

            var // explicit attributes of the passed node
                elementAttributes = AttributeUtils.getExplicitAttributes(element),
                // the operation options
                operationOptions = _.clone(position),
                // the style families for generated null attributes
                clearFamily = Utils.getStringOption(options, 'clearFamily', ''),
                // a specified property, that shall be ignored in the setAttributes operation
                ignoreFamily = Utils.getStringOption(options, 'ignoreFamily', ''),
                // a specified property, that shall send target string with operation
                target = Utils.getStringOption(options, 'target', '');

            // deleting the specified property from the attributes
            if (elementAttributes && ignoreFamily.length > 0 && elementAttributes[ignoreFamily]) { delete elementAttributes[ignoreFamily]; }

            // insert null values for all attributes registered for the specified style family
            if (clearFamily.length > 0) {
                operationOptions.attrs = docModel.getStyleCollection(clearFamily).buildNullAttributes();
            } else {
                operationOptions.attrs = {};
            }

            // merge the explicit attributes of the passed element
            docModel.extendAttributes(operationOptions.attrs, elementAttributes);

            // no attributes, no operation
            if (!_.isEmpty(operationOptions.attrs)) {
                this.generateOperation(Operations.SET_ATTRIBUTES, extendPropertiesWithTarget(operationOptions, target));
            }

            return this;
        };

        /**
         * Generates all operations needed to recreate the child nodes of the
         * passed paragraph.
         *
         * @param {HTMLElement|jQuery} paragraph
         *  The paragraph element whose content nodes will be converted to
         *  operations. If this object is a jQuery collection, uses the first
         *  node it contains.
         *
         * @param {Number[]} position
         *  The logical position of the passed paragraph node. The generated
         *  operations will contain positions starting with this address.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.start]
         *      The logical index of the first character to be included into
         *      the generated operations. By default, operations will include
         *      all contents from the beginning of the paragraph.
         *  @param {Number} [options.end]
         *      The logical index of the last character to be included into the
         *      generated operations (closed range). By default, operations
         *      will include all contents up to the end of the paragraph.
         *  @param {Boolean} [options.clear=false]
         *      If set to true, a 'setAttributes' operation will be generated
         *      for the first 'insertText' operation that clears all character
         *      attributes of the inserted text. This prevents that applying
         *      the operations at another place in the document clones the
         *      character formatting of the target position.
         *  @param {String} [options.ignoreFamily]
         *      If specified, this property will be ignored.
         *  @param {Number} [options.targetOffset]
         *      If set to a number, the logical positions in the operations
         *      generated for the child nodes will start at this offset. If
         *      omitted, the original node offset will be used in the logical
         *      positions.
         *  @param {String} [options.target]
         *      If specified, this property will be send as target with operation.
         *
         * @returns {TextOperationsGenerator}
         *  A reference to this instance.
         */
        this.generateParagraphChildOperations = function (paragraph, position, options) {

            var // start of text range to be included in the operations
                rangeStart = Utils.getIntegerOption(options, 'start'),
                // end of text range to be included in the operations
                rangeEnd = Utils.getIntegerOption(options, 'end'),
                // start position of text nodes in the generated operations
                targetOffset = Utils.getIntegerOption(options, 'targetOffset'),
                // used to merge several text portions into the same operation
                lastTextOperation = null,
                // a specified property, that shall be ignored in the setAttributes operation
                ignoreFamily = Utils.getStringOption(options, 'ignoreFamily', ''),
                // a specified property, that shall send target string with operation
                target = Utils.getStringOption(options, 'target', ''),
                // formatting ranges for text portions, must be applied after the contents
                attributeRanges = [],
                // attributes passed to the first insert operation to clear all formatting
                clearAttributes = null;

            // generates the specified operation, adds that attributes in clearAttributes on first call
            function generateOperationWithClearAttributes(name, operationOptions) {

                // add the character attributes that will be cleared on first insertion operation
                if (_.isObject(clearAttributes)) {
                    operationOptions.attrs = clearAttributes;
                    clearAttributes = null;
                }

                return self.generateOperation(name, operationOptions);
            }

            // clear all attributes of the first inserted text span
            if (Utils.getBooleanOption(options, 'clear', false)) {
                clearAttributes = docModel.getStyleCollection('character').buildNullAttributes();
            }

            // process all content nodes in the paragraph and create operations
            Position.iterateParagraphChildNodes(paragraph, function (node, nodeStart, nodeLength, offsetStart, offsetLength) {

                var // logical start index of the covered part of the child node
                    startIndex = _.isNumber(targetOffset) ? targetOffset : (nodeStart + offsetStart),
                    // logical end index of the covered part of the child node (closed range)
                    endIndex = startIndex + offsetLength - 1,
                    // logical start position of the covered part of the child node
                    startPosition = Position.appendNewIndex(position, startIndex),
                    // logical end position of the covered part of the child node
                    endPosition = Position.appendNewIndex(position, endIndex),
                    // text of a portion span
                    text = null,
                    // type of the drawing or the hard break or the field or the range marker
                    type = null,
                    // id and position of a text marker range
                    rangeId = null, rangePos = null,
                    // the field attributes
                    attrs = {};

                // operation to create a (non-empty) generic text portion
                if (DOM.isTextSpan(node)) {

                    // extract the text covered by the specified range
                    text = node.firstChild.nodeValue.substr(offsetStart, offsetLength);
                    // append text portions to the last 'insertText' operation
                    if (lastTextOperation) {
                        lastTextOperation.text += text;
                    } else {
                        if (text.length > 0) {
                            lastTextOperation = generateOperationWithClearAttributes(Operations.TEXT_INSERT, extendPropertiesWithTarget({ start: startPosition, text: text }, target));
                        }
                    }
                    attributeRanges.push({ node: node, position: _.extend({ start: startPosition }, (text.length > 1) ? { end: endPosition } : undefined) });

                } else {

                    // anything else than plain text will be inserted, forget last text operation
                    lastTextOperation = null;

                    // operation to create a text field
                    if (DOM.isFieldNode(node)) {
                        // extract text of all embedded spans representing the field
                        text = $(node).text();
                        attrs.field = {};
                        $.each($(node).data(), function (name, value) {
                            if (name !== 'type') {
                                attrs.field[name] = value;
                            } else {
                                type = value;
                            }
                        });
                        self.generateOperation(Operations.FIELD_INSERT, extendPropertiesWithTarget({ start: startPosition, representation: text, type: type, attrs: attrs }, target));
                        // attributes are contained in the embedded span elements
                        attributeRanges.push({ node: node.firstChild, position: { start: startPosition } });

                    // operation to create a tabulator
                    } else if (DOM.isTabNode(node)) {
                        generateOperationWithClearAttributes(Operations.TAB_INSERT, extendPropertiesWithTarget({ start: startPosition }, options.target));
                        // attributes are contained in the embedded span elements
                        attributeRanges.push({ node: node.firstChild, position: { start: startPosition } });

                    // operation to create a hard break
                    } else if (DOM.isHardBreakNode(node)) {
                        // reading type of hard break from the node
                        type = $(node).data('type');
                        generateOperationWithClearAttributes(Operations.HARDBREAK_INSERT, extendPropertiesWithTarget({ start: startPosition, type: type }, target));
                        // attributes are contained in the embedded span elements
                        attributeRanges.push({ node: node.firstChild, position: { start: startPosition } });

                    // operation to create a drawing (including its attributes)
                    } else if (DrawingFrame.isDrawingFrame(node) || DOM.isDrawingPlaceHolderNode(node)) {
                        // switching to node in drawing layer
                        if (DOM.isDrawingPlaceHolderNode(node)) { node = DOM.getDrawingPlaceHolderNode(node); }
                        // skip drawing nodes that cannot be restored (inserting one space because of position counting inside paragraph)
                        if (DOM.isUnrestorableDrawingNode(node)) {
                            generateOperationWithClearAttributes(Operations.TEXT_INSERT, extendPropertiesWithTarget({ start: startPosition, text: ' ' }, target));
                            return;
                        }
                        // generate operations for the drawing
                        this.generateDrawingOperations(node, startPosition, { ignoreFamily: ignoreFamily, target: options.target });

                    } else if (DOM.isCommentPlaceHolderNode(node)) {
                        // switching to node in comment layer
                        node = DOM.getCommentPlaceHolderNode(node);

                        // generate operations for the drawing
                        this.generateCommentOperations(node, startPosition, { ignoreFamily: ignoreFamily, target: options.target });

                    } else if (DOM.isRangeMarkerNode(node)) {

                        rangeId = DOM.getRangeMarkerId(node);
                        rangePos = DOM.isRangeMarkerStartNode(node) ? 'start' : (DOM.isRangeMarkerEndNode(node) ? 'end' : undefined);
                        type = DOM.getRangeMarkerType(node);
                        self.generateOperation(Operations.RANGE_INSERT, extendPropertiesWithTarget({ start: startPosition, id: rangeId, type: type, position: rangePos }, options.target));

                    } else if (DOM.isComplexFieldNode(node)) {

                        self.generateOperation(Operations.COMPLEXFIELD_INSERT, extendPropertiesWithTarget({ start: startPosition, id: DOM.getComplexFieldId(node), instruction: DOM.getComplexFieldInstruction(node) }, options.target));

                    } else {
                        Utils.error('TextOperationsGenerator.generateParagraphChildOperations(): unknown content node');
                        return Utils.BREAK;
                    }
                }

                // custom target offset: advance offset by covered node length
                if (_.isNumber(targetOffset)) {
                    targetOffset += offsetLength;
                }

            }, this, { start: rangeStart, end: rangeEnd });

            // Generate 'setAttribute' operations after all contents have been
            // created via 'insertText', 'insertField', etc. Otherwise, these
            // operations would clone the attributes of the last text portion
            // instead of creating a clean text node as expected in this case.
            _(attributeRanges).each(function (range) {
                this.generateSetAttributesOperation(range.node, range.position, { ignoreFamily: ignoreFamily, target: target });
            }, this);

            return this;
        };

        /**
         * Generates all operations needed to recreate the passed paragraph.
         *
         * @param {HTMLElement|jQuery} paragraph
         *  The paragraph element whose contents will be converted to
         *  operations. If this object is a jQuery collection, uses the first
         *  node it contains.
         *
         * @param {Number[]} position
         *  The logical position of the passed paragraph node. The generated
         *  operations will contain positions starting with this address.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.ignoreFamily]
         *      If specified, this property will be ignored.
         *  @param {String} [options.target]
         *      If specified, this property will be send as target with operation.
         *
         * @returns {TextOperationsGenerator}
         *  A reference to this instance.
         */
        this.generateParagraphOperations = function (paragraph, position, options) {

            // operations to create the paragraph element and formatting
            this.generateOperationWithAttributes(paragraph, Operations.PARA_INSERT, { start: position }, options);

            // process all content nodes in the paragraph and create operations
            return this.generateParagraphChildOperations(paragraph, position, options);
        };

        /**
         * Generates all operations needed to recreate the passed table cell.
         *
         * @param {HTMLTableCellElement|jQuery} cellNode
         *  The table cell element that will be converted to operations. If
         *  this object is a jQuery collection, uses the first node it
         *  contains.
         *
         * @param {Number[]} position
         *  The logical position of the passed table cell. The generated
         *  operations will contain positions starting with this address.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.ignoreFamily]
         *      If specified, this property will be ignored.
         *  @param {String} [options.target]
         *      If specified, this property will be send as target with operation.
         *
         * @returns {TextOperationsGenerator}
         *  A reference to this instance.
         */
        this.generateTableCellOperations = function (cellNode, position, options) {

            // operation to create the table cell element
            this.generateOperationWithAttributes(cellNode, Operations.CELLS_INSERT, { start: position, count: 1 }, options);

            // generate operations for the contents of the cell
            return this.generateContentOperations(cellNode, position, options);
        };

        /**
         * Generates all operations needed to recreate the passed table row.
         *
         * @param {HTMLTableRowElement|jQuery} rowNode
         *  The table row element that will be converted to operations. If this
         *  object is a jQuery collection, uses the first node it contains.
         *
         * @param {Number[]} position
         *  The logical position of the passed table row. The generated
         *  operations will contain positions starting with this address.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.ignoreFamily]
         *      If specified, this property will be ignored.
         *  @param {String} [options.target]
         *      If specified, this property will be send as target with operation.
         *
         * @returns {TextOperationsGenerator}
         *  A reference to this instance.
         */
        this.generateTableRowOperations = function (rowNode, position, options) {

            // operation to create the table row element
            this.generateOperationWithAttributes(rowNode, Operations.ROWS_INSERT, { start: position }, options);

            // generate operations for all cells
            position = Position.appendNewIndex(position);
            Utils.iterateSelectedDescendantNodes(rowNode, 'td', function (cellNode) {
                this.generateTableCellOperations(cellNode, position, options);
                position = Position.increaseLastIndex(position);
            }, this, { children: true });

            return this;
        };

        /**
         * Generates all operations needed to recreate the passed table.
         *
         * @param {HTMLTableElement|jQuery} tableNode
         *  The table element that will be converted to operations. If this
         *  object is a jQuery collection, uses the first node it contains.
         *
         * @param {Number[]} position
         *  The logical position of the passed table node. The generated
         *  operations will contain positions starting with this address.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.ignoreFamily]
         *      If specified, this property will be ignored.
         *  @param {String} [options.target]
         *      If specified, this property will be send as target with operation.
         *
         * @returns {TextOperationsGenerator}
         *  A reference to this instance.
         */
        this.generateTableOperations = function (tableNode, position, options) {

            // operation to create the table element
            this.generateOperationWithAttributes(tableNode, Operations.TABLE_INSERT, { start: position }, options);

            // generate operations for all rows
            position = Position.appendNewIndex(position);
            DOM.getTableRows(tableNode).each(function () {
                self.generateTableRowOperations(this, position, options);
                position = Position.increaseLastIndex(position);
            });

            return this;
        };

        /**
         * Generates all operations needed to recreate the passed drawing group.
         *
         * @param {HTMLElement|jQuery} drawingNode
         *  The drawing element whose contents will be converted to operations.
         *  If this object is a jQuery collection, uses the first node it contains.
         *
         * @param {Number[]} position
         *  The logical position of the passed drawing node. The generated
         *  operations will contain positions starting with this address.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.ignoreFamily]
         *      If specified, this property will be ignored.
         *  @param {String} [options.target]
         *      If specified, this property will be send as target with operation.
         *
         * @returns {TextOperationsGenerator}
         *  A reference to this instance.
         */
        this.generateDrawingOperations = function (drawingNode, position, options) {

            var // the type of the drawing
                type = DrawingFrame.getDrawingType(drawingNode);

            if (type === 'image') {
                // special image handling: We need to take care that images are always readable for the target instance
                this.generateImageOperationWithAttributes(drawingNode, { start: position }, options);
            } else {

                this.generateOperationWithAttributes(drawingNode, Operations.DRAWING_INSERT, { start: position, type: type }, options);

                if (type === 'group') {
                    // generate operations for the contents of the text frame
                    this.generateDrawingGroupChildOperations(drawingNode, position, options);
                } else if (type === 'shape' && DrawingFrame.isTextFrameShapeDrawingFrame(drawingNode)) {
                    // generate operations for the contents of the text frame
                    this.generateContentOperations(drawingNode, position, options);
                }
            }

            return this;
        };

        /**
         * Generates all operations needed to recreate the passed comment.
         *
         * @param {HTMLElement|jQuery} commentNode
         *  The comment element whose contents will be converted to operations.
         *  If this object is a jQuery collection, uses the first node it contains.
         *
         * @param {Number[]} position
         *  The logical position of the passed drawing node. The generated
         *  operations will contain positions starting with this address.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.ignoreFamily]
         *      If specified, this property will be ignored.
         *  @param {String} [options.target]
         *      If specified, this property will be send as target with operation.
         *
         * @returns {TextOperationsGenerator}
         *  A reference to this instance.
         */
        this.generateCommentOperations = function (commentNode, position, options) {

            // restoring author, date and id
            var id = DOM.getTargetContainerId(commentNode),
                author = DOM.getCommentAuthor(commentNode),
                uid = DOM.getCommentAuthorUid(commentNode),
                date = DOM.getCommentDate(commentNode);

            this.generateOperationWithAttributes(commentNode, Operations.COMMENT_INSERT, { start: position, id: id, author: author, uid: uid, date: date }, options);

            // using the target for the created operations
            var localOptions = options || {};
            localOptions.target = id;

            // generate operations for the contents of the text frame
            this.generateContentOperations(commentNode, [], localOptions);

            return this;
        };

        /**
         * Generates a usable image operation for an internal & external URL.
         *
         * @param {HTMLElement|jQuery} imageNode
         *  The image drawing element. If this object is a jQuery collection,
         *  uses the first node it contains.
         *
         * @param {Object} [operationOptions]
         *  Additional options that will be stored in the operation.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.ignoreFamily]
         *      If specified, this property will be ignored in the generated
         *      operation.
         *  @param {String} [options.target]
         *      If specified, this property will be send as target with operation.
         *
         * @returns {Object}
         *  The created operation object.
         */
        this.generateImageOperationWithAttributes = function (imageNode, operationOptions, options) {

            var // the new operation
                operation = self.generateOperationWithAttributes(imageNode, Operations.DRAWING_INSERT, _.extend({ type: 'image' }, operationOptions), options),
                // the image URL from the attributes map
                imageUrl = operation.attrs && operation.attrs.image && operation.attrs.image.imageUrl;

            // set image attributes so they can be used in the same instance or between different instances
            if (imageUrl && DOM.isDocumentImageNode(imageNode)) {
                _(operation.attrs.image).extend({
                    imageData: DOM.getBase64FromImageNode(imageNode, Image.getMimeTypeFromImageUri(imageUrl)),
                    sessionId: DOM.getUrlParamFromImageNode(imageNode, 'session'),
                    fileId: DOM.getUrlParamFromImageNode(imageNode, 'id')
                });
            }

            return operation;
        };

        /**
         * Generates all operations needed to recreate the children of the passed
         * drawing group.
         *
         * @param {HTMLElement|jQuery} drawingGroupNode
         *  The drawing group element whose contents will be converted to
         *  operations. If this object is a jQuery collection, uses the first
         *  node it contains.
         *
         * @param {Number[]} position
         *  The logical position of the passed drawing group node. The generated
         *  operations will contain positions starting with this address.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.ignoreFamily]
         *      If specified, this property will be ignored.
         *  @param {String} [options.target]
         *      If specified, this property will be send as target with operation.
         *
         * @returns {TextOperationsGenerator}
         *  A reference to this instance.
         */
        this.generateDrawingGroupChildOperations = function (drawingGroupNode, position, options) {

            // generate operations for all drawings inside the drawing group
            position = Position.appendNewIndex(position);
            Utils.iterateSelectedDescendantNodes($(drawingGroupNode).children().first(), '.drawing', function (drawingNode) {
                this.generateDrawingOperations(drawingNode, position, options);
                position = Position.increaseLastIndex(position);
            }, this, { children: true });

            return this;
        };

        /**
         * Generates all operations needed to recreate the contents of the
         * passed root node. Root nodes are container elements for text
         * paragraphs and other first-level content nodes (e.g. tables).
         * Examples for root nodes are the entire document root node, table
         * cells, or text shapes. Note that the operation to create the root
         * node itself will NOT be generated.
         *
         * @param {HTMLElement|jQuery} rootNode
         *  The root node containing the content nodes that will be converted
         *  to operations. The passed root node may contain an embedded node
         *  that serves as parent for all content nodes.  If this object is a
         *  jQuery collection, uses the first node it contains.
         *
         * @param {Number[]} position
         *  The logical position of the passed node. The generated operations
         *  will contain positions starting with this address.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.ignoreFamily]
         *      If specified, this property will be ignored.
         *  @param {String} [options.target]
         *      If specified, this property will be send as target with operation.
         *
         * @returns {TextOperationsGenerator}
         *  A reference to this instance.
         */
        this.generateContentOperations = function (rootNode, position, options) {

            var // the container node (direct parent of the target content nodes)
                containerNode = DOM.getChildContainerNode(rootNode);

            // iterate all child elements of the root node and create operations
            position = Position.appendNewIndex(position);
            Utils.iterateDescendantNodes(containerNode, function (node) {

                if (DOM.isParagraphNode(node)) {
                    // skip implicit paragraph nodes without increasing the position
                    if (DOM.isImplicitParagraphNode(node)) { return; }
                    // operations to create a paragraph
                    this.generateParagraphOperations(node, position, options);
                } else if (DOM.isTableNode(node)) {
                    // skip table nodes with exceeded size without increasing the position
                    if (DOM.isExceededSizeTableNode(node)) { return; }
                    // operations to create a table with its structure and contents
                    this.generateTableOperations(node, position, options);
                } else if (DOM.isMarginalDrawingLayerNode(node)) {
                    // skip over text drawing layer in header/footer
                } else {
                    Utils.error('TextOperationsGenerator.generateContentOperations(): unexpected node "' + Utils.getNodeName(node) + '" at position ' + JSON.stringify(position) + '.');
                    // continue with next child node (do not increase position)
                    return;
                }

                // increase last element of the logical position (returns a clone)
                position = Position.increaseLastIndex(position);

            }, this, { children: true });

            return this;
        };

        /**
         * Utility method for extending object with target property,
         * but only if its defined.
         *
         * @param{Object} object
         *   object that is being extended
         * @param{String[]} target
         *   Id referencing to header or footer node
         *
         * @returns {Object} enhanced object
         *
         */
        function extendPropertiesWithTarget (object, target) {
            if (target) {
                object.target = target;
            }
            return object;
        }

    } // class TextOperationsGenerator

    // exports ================================================================

    // derive this class from class OperationsGenerator
    return OperationsGenerator.extend({ constructor: TextOperationsGenerator });

});
