/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Malte Timmermann <malte.timmermann@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 * @author Oliver Specht <oliver.specht@open-xchange.com>
 */

define('io.ox/office/text/editor', [
    'io.ox/office/text/utils/textutils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/io',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/model/editmodel',
    'io.ox/office/drawinglayer/utils/drawingutils',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/drawinglayer/view/imageutil',
    'io.ox/office/text/utils/config',
    'io.ox/office/text/utils/operations',
    'io.ox/office/text/dom',
    'io.ox/office/text/selection',
    'io.ox/office/text/remoteselection',
    'io.ox/office/text/table',
    'io.ox/office/text/hyperlink',
    'io.ox/office/editframework/utils/hyperlinkutils',
    'io.ox/office/text/operationsgenerator',
    'io.ox/office/text/position',
    'io.ox/office/text/rangeMarker',
    'io.ox/office/text/changeTrack',
    'io.ox/office/text/commentLayer',
    'io.ox/office/text/components/field/fieldmanager',
    'io.ox/office/text/drawingLayer',
    'io.ox/office/text/drawingResize',
    'io.ox/office/text/tableResize',
    'io.ox/office/text/utils/snapshot',
    'io.ox/office/text/model/modelattributesmixin',
    'io.ox/office/text/format/tablestyles',
    'io.ox/office/text/format/listcollection',
    'io.ox/office/text/format/stylesheetmixin',
    'io.ox/office/text/export',
    'io.ox/office/text/clipboardmixin',
    'io.ox/office/text/pageLayout',
    'io.ox/office/text/plaintext/searchhandler',
    'io.ox/office/text/plaintext/spellchecker',
    'io.ox/office/text/model/numberFormatter',
    'io.ox/office/text/view/textdialogs',
    'gettext!io.ox/office/text/main'
], function (Utils, KeyCodes, IO, Forms, Tracking, Color, Border, AttributeUtils, EditModel, DrawingUtils, DrawingFrame, Image, Config, Operations, DOM, Selection, RemoteSelection, Table, Hyperlink, HyperlinkUtils, TextOperationsGenerator, Position, RangeMarker, ChangeTrack, CommentLayer, FieldManager, DrawingLayer, DrawingResize, TableResize, Snapshot, ModelAttributesMixin, TableStyles, ListCollection, StylesheetMixin, Export, ClipboardMixin, PageLayout, SearchHandler, SpellChecker, NumberFormatter, Dialogs, gt) {

    'use strict';

    var // style attributes for heading 1 -6 based on latent styles
        HEADINGS_CHARATTRIBUTES = [
            { color: { type: 'scheme', value: 'accent1', transformations: [{ type: 'shade', value: 74902 }], fallbackValue: '376092' }, bold: true, fontSize: 14 },
            { color: { type: 'scheme', value: 'accent1', fallbackValue: '4F81BD' }, bold: true, fontSize: 13 },
            { color: { type: 'scheme', value: 'accent1', fallbackValue: '4F81BD' }, bold: true },
            { color: { type: 'scheme', value: 'accent1', fallbackValue: '4F81BD' }, bold: true, italic: true },
            { color: { type: 'scheme', value: 'accent1', transformations: [{ type: 'shade', value: 49804 }], fallbackValue: '244061' } },
            { color: { type: 'scheme', value: 'accent1', transformations: [{ type: 'shade', value: 49804 }], fallbackValue: '244061' }, italic: true }
        ],

        DEFAULT_PARAGRAPH_DEFINTIONS = { 'default': true, styleId: 'Standard', styleName: 'Normal' },

        // style attributes for lateral table style
        DEFAULT_LATERAL_TABLE_DEFINITIONS = { 'default': true, styleId: 'TableGrid', styleName: 'Table Grid', uiPriority: 59 },
        DEFAULT_LATERAL_TABLE_ATTRIBUTES = {
            wholeTable: {
                paragraph: { lineHeight: { type: 'percent', value: 100 }, marginBottom: 0 },
                table: {
                    borderTop:        { color: Color.AUTO, width: 17, style: 'single' },
                    borderBottom:     { color: Color.AUTO, width: 17, style: 'single' },
                    borderInsideHor:  { color: Color.AUTO, width: 17, style: 'single' },
                    borderInsideVert: { color: Color.AUTO, width: 17, style: 'single' },
                    borderLeft:       { color: Color.AUTO, width: 17, style: 'single' },
                    borderRight:      { color: Color.AUTO, width: 17, style: 'single' },
                    paddingBottom: 0,
                    paddingTop: 0,
                    paddingLeft: 190,
                    paddingRight: 190
                }
            }
        },

        // style attributes for lateral comment style
        DEFAULT_LATERAL_COMMENT_DEFINITIONS = { 'default': false, styleId: 'annotation text', styleName: 'annotation text', parent: 'Standard', uiPriority: 99 },
        DEFAULT_LATERAL_COMMENT_ATTRIBUTES = { character: { fontSize: 10 }, paragraph: { lineHeight: { type: 'percent', value: 100 }, marginBottom: 0 }},

        // style attributes for lateral header style
        DEFAULT_LATERAL_HEADER_DEFINITIONS = { 'default': false, styleId: 'Header', styleName: 'Header', parent: 'HeaderFooter', uiPriority: 99 },

        // style attributes for lateral footer style
        DEFAULT_LATERAL_FOOTER_DEFINITIONS = { 'default': false, styleId: 'Footer', styleName: 'Footer', parent: 'HeaderFooter', uiPriority: 99 },

        DEFAULT_HYPERLINK_DEFINTIONS = { 'default': false, styleId: 'Hyperlink', styleName: 'Hyperlink', uiPriority: 99 },
        DEFAULT_HYPERLINK_CHARATTRIBUTES = { color: _.extend({ fallbackValue: '0080C0' }, Color.HYPERLINK), underline: true },

        DEFAULT_DRAWING_MARGINS = { marginTop: 317, marginLeft: 317, marginBottom: 317, marginRight: 317 },

        DEFAULT_DRAWING_DEFINITION = { 'default': true, styleId: 'default_drawing_style', styleName: 'Default Graphic Style', uiPriority: 50 },
        DEFAULT_DRAWING_ATTRS = { line: { color: { type: 'rgb', value: '3465a4' } } },

        DEFAULT_DRAWING_TEXTFRAME_DEFINITION = { 'default': false, styleId: 'Frame', uiPriority: 60 },
        DEFAULT_DRAWING_TEXTFRAME_ATTRS = {
            line: { style: 'solid', type: 'solid', width: 2, color: { type: 'rgb', value: '000000' } },
            shape: { paddingLeft: 150, paddingTop: 150, paddingRight: 150, paddingBottom: 150 },
            drawing: { anchorVertAlign: 'top', marginLeft: 201, indentLeft: 201, marginTop: 201, marginRight: 201, indentRight: 201, marginBottom: 201, textWrapMode: 'square', textWrapSide: 'both', anchorHorAlign: 'center', anchorHorBase: 'column', anchorVertBase: 'paragraph' }
        },

        // internal clipboard
        clipboardOperations = [],
        clipboardId = '',

        // if set to true, events caught by the editor will be logged to console
        LOG_EVENTS = false,

        // Performance: List of operation names, that cache the paragraph in the selection
        PARAGRAPH_CACHE_OPERATIONS = {},

        // Load performance: List of operation names for operations that need to be executed, even if the document is loaded
        // from local storage. These operations do not modify the DOM, but register additional document information, that is
        // not stored in the DOM.
        REQUIRED_LOAD_OPERATIONS = {};

    // Fill performance objects with key value pairs
    (function () {
        PARAGRAPH_CACHE_OPERATIONS[Operations.TEXT_INSERT] = true;
        PARAGRAPH_CACHE_OPERATIONS[Operations.PARA_INSERT] = true;
        PARAGRAPH_CACHE_OPERATIONS[Operations.PARA_SPLIT] = true;
        PARAGRAPH_CACHE_OPERATIONS[Operations.SET_ATTRIBUTES] = true;

        REQUIRED_LOAD_OPERATIONS[Operations.SET_DOCUMENT_ATTRIBUTES] = true;
        REQUIRED_LOAD_OPERATIONS[Operations.INSERT_THEME] = true;
        REQUIRED_LOAD_OPERATIONS[Operations.INSERT_FONT_DESCRIPTION] = true;
        REQUIRED_LOAD_OPERATIONS[Operations.INSERT_STYLESHEET] = true;
        REQUIRED_LOAD_OPERATIONS[Operations.INSERT_LIST] = true;
    })();

    // private global functions ===============================================

    /**
     * Returns true, if the passed keyboard event is ctrl+v, meta+v or shift+insert.
     * Attention: Comparison with KeyCodes.INSERT only allowed in processKeyDown, not in processKeyPressed
     * Info: Use keyCode only keyDown, charCode only in keyPress
     *
     * @param event
     *  A jQuery keyboard event object.
     */
    function isPasteKeyEventKeyDown(event) {
        return KeyCodes.matchKeyCode(event, 'V', { ctrlOrMeta: true }) || KeyCodes.matchKeyCode(event, 'INSERT', { shift: true });
    }

    /**
     * Returns true, if the passed keyboard event is ctrl+c, meta+c or ctrl+insert.
     * Attention: Comparison with KeyCodes.INSERT only allowed in processKeyDown, not in processKeyPressed
     *
     * @param event
     *  A jQuery keyboard event object.
     */
    function isCopyKeyEventKeyDown(event) {
        return KeyCodes.matchKeyCode(event, 'C', { ctrlOrMeta: true }) || KeyCodes.matchKeyCode(event, 'INSERT', { ctrlOrMeta: true });
    }

    /**
     * Returns true, if the passed keyboard event is ctrl+x, meta+x or shift+delete.
     * Attention: Comparison with KeyCodes.DELETE only allowed in processKeyDown, not in processKeyPressed
     *
     * @param event
     *  A jQuery keyboard event object.
     */
    function isCutKeyEventKeyDown(event) {
        return KeyCodes.matchKeyCode(event, 'X', { ctrlOrMeta: true }) || KeyCodes.matchKeyCode(event, 'DELETE', { shift: true });
    }

    /**
     * Returns true, if the passed keyboard event is the global F6 accessibility key event
     */
    function isF6AcessibilityKeyEvent(event) {
        // ignore all modifier keys
        return event.keyCode === KeyCodes.F6;
    }

    /**
     * Returns true, if the passed keyboard event is ctrl+f, meta+f, ctrl+g, meta+g, ctrl+shift+g or meta+shift+g.
     *
     * @param event
     *  A jQuery keyboard event object.
     */
    function isSearchKeyEvent(event) {
        return (KeyCodes.matchModifierKeys(event, { ctrlOrMeta: true, shift: null }) && (event.charCode === 102 || event.keyCode === KeyCodes.F || event.charCode === 103 || event.keyCode === KeyCodes.G));
    }

    /**
     * Returns true, if the passed keyboard event is a common browser keyboard shortcut
     * that should be handled by the browser itself.
     *
     * @param {jQuery.event} event
     *  A jQuery keyboard event object.
     *
     * @returns {Boolean}
     *  Whether the passed event is a browser shortcut.
     */
    function isBrowserShortcutKeyEvent(event) {

        var // Ctrl
            ctrl = KeyCodes.matchModifierKeys(event, { ctrl: true }),
            // Ctrl and Shift
            ctrlShift = KeyCodes.matchModifierKeys(event, { ctrl: true, shift: true }),
            // Meta and Alt
            metaAlt = KeyCodes.matchModifierKeys(event, { meta: true, alt: true }),
            // Meta and Shift
            metaShift = KeyCodes.matchModifierKeys(event, { meta: true, shift: true }),
            // Ctrl and Meta
            ctrlMeta = KeyCodes.matchModifierKeys(event, { ctrl: true, meta: true }),
            // Ctrl or Meta
            ctrlOrMeta = KeyCodes.matchModifierKeys(event, { ctrlOrMeta: true }),
            // Ctrl or Meta and Shift
            ctrlOrMetaShift = KeyCodes.matchModifierKeys(event, { ctrlOrMeta: true, shift: true }),
            // Ctrl or Meta and Alt
            ctrlOrMetaAlt = KeyCodes.matchModifierKeys(event, { ctrlOrMeta: true, alt: true });

        if (event.type === 'keypress') {
            // check charCode for keypress event

            // Switch to the specified/last tab - Ctrl + 1...8, Cmd + 1...8 / Ctrl + 9, Cmd + 9
            if (ctrlOrMeta && event.charCode >= 49 && event.charCode <= 57) {
                return true;
            }

            // Open a new tab             - Ctrl + T, Cmd + T
            // Reopen the last closed tab - Ctrl + Shift + T, Cmd + Shift + T
            if ((ctrlOrMeta || ctrlOrMetaShift) && (event.charCode === 116 || event.charCode === 84)) {
                return true;
            }

            // Close the current tab     - Ctrl + W, Ctrl + F4, Cmd + W
            // Open a new browser window - Ctrl + N, Cmd + N
            if (ctrlOrMeta && (event.charCode === 119 || event.charCode === 110)) {
                return true;
            }

            // Close the current window - Ctrl + Shift + W, Cmd + Shift + W
            if (ctrlOrMetaShift && (event.charCode === 87 || event.charCode === 119)) {
                return true;
            }

            // Open a new window in incognito mode - Ctrl + Shift + N, Cmd + Shift + N, Ctrl + Shift + P, Cmd + Shift + P
            if (ctrlOrMetaShift && (event.charCode === 78 || event.charCode === 80)) {
                return true;
            }

            // Minimize the current window - Ctrl + M, Cmd + M
            if (ctrlOrMeta && event.charCode === 109) {
                return true;
            }

            // Hide browser - Ctrl + H, Cmd + H
            // Hide all other windows - Ctrl + Alt + H, Cmd + Alt + H
            if ((ctrlOrMeta || ctrlOrMetaAlt) && (event.charCode === 104 || event.charCode === 170)) {
                return true;
            }

            // Quit browser - Ctrl + Q, Cmd + Q
            if (ctrlOrMeta && event.charCode === 113) {
                return true;
            }

            // Zoom in on the page - Ctrl + '+', Cmd + '+'
            // Zoom out on the page - Ctrl + '-', Cmd + '-'
            // Reset zoom level - Ctrl + 0, Cmd + 0
            if (ctrlOrMeta && (event.charCode === 43 || event.charCode === 45 || event.charCode === 48)) {
                return true;
            }

            // Full-screen mode - Ctrl + Cmd + F
            if (ctrlMeta && (event.charCode === 102 || event.charCode === 6)) { //6 is for safari
                return true;
            }

            // Open the browsing history - Ctrl + H, Cmd + Shift + H
            if ((ctrl || metaShift) && (event.charCode === 72 || event.charCode === 104)) {
                return true;
            }

            // Open the download history - Ctrl + J, Cmd + J, Cmd + Shift + J, Cmd + Alt + 2
            if (((ctrlOrMeta || metaShift) && (event.charCode === 106 || event.charCode === 74)) || (metaAlt && event.charCode === 8220)) {
                return true;
            }

            // Bookmark the current web site - Ctrl + D, Cmd + D
            if (ctrlOrMeta && event.charCode === 100) {
                return true;
            }

            // Toggles the bookmarks bar                                               - Ctrl + Shift + B, Cmd + Shift + B
            // Save all open pages in your current window as bookmarks in a new folder - Ctrl + Shift + D, Cmd + Shift + D
            if (ctrlOrMetaShift && (event.charCode === 66 || event.charCode === 68)) {
                return true;
            }

            // Open the Bookmarks window - Ctrl + Shift + B, Cmd + Shift + B, Cmd + Alt + B
            if ((ctrlOrMetaShift || metaAlt) && (event.charCode === 66 || event.charCode === 8747)) {
                return true;
            }

            // Open Developer Tools - Ctrl + Shift + (I,J,K,S), Cmd + Alt + (I,J,K,S)
            if ((ctrlShift || metaAlt) && (event.charCode === 8260 || event.charCode === 73 ||
                                           event.charCode === 186  || event.charCode === 74 ||
                                           event.charCode === 8710 || event.charCode === 75 ||
                                           event.charCode === 8218 || event.charCode === 83)) {
                return true;
            }

            // For function keys Firefox sends a keydown event with corresponding keyCode as expected,
            // but also sends a keypress event with charCode 0 and a corresponding keyCode.
            if (_.browser.Firefox && (event.charCode === 0)) {

                // Full-screen mode - F11, Open Firebug - F12
                if (event.keyCode === KeyCodes.F11 || event.keyCode === KeyCodes.F12) {
                    return true;
                }
            }

        } else {
            // check keyCode for keyup and keydown events

            // Switch to the specified/last tab - Ctrl + 1...8, Cmd + 1...8 / Ctrl + 9, Cmd + 9
            if (ctrlOrMeta && ((event.keyCode >= KeyCodes['1'] && event.keyCode <= KeyCodes['9']))) {
                return true;
            }

            // Switch to the next/previous tab - Ctrl + Tab / Ctrl + Shift + Tab
            if ((KeyCodes.matchKeyCode(event, KeyCodes.TAB, { ctrl: true }) || KeyCodes.matchKeyCode(event, KeyCodes.TAB, { ctrl: true, shift: true }))) {
                return true;
            }

            // Open a new tab             - Ctrl + T, Cmd + T
            // Reopen the last closed tab - Ctrl + Shift + T, Cmd + Shift + T
            if ((ctrlOrMeta || ctrlOrMetaShift) && event.keyCode === KeyCodes.T) {
                return true;
            }

            // Close the current tab     - Ctrl + W, Ctrl + F4, Cmd + W
            // Open a new browser window - Ctrl + N, Cmd + N
            if (ctrlOrMeta && (event.keyCode === KeyCodes.W || event.keyCode === KeyCodes.F4 || event.keyCode === KeyCodes.N)) {
                return true;
            }

            // Close the current window - Alt + F4, Ctrl + Shift + W, Cmd + Shift + W
            if ((KeyCodes.matchModifierKeys(event, { alt: true }) && event.keyCode === KeyCodes.F4) || (ctrlOrMetaShift && event.keyCode === KeyCodes.W)) {
                return true;
            }

            // Open a new window in incognito mode - Ctrl + Shift + N, Cmd + Shift + N, Ctrl + Shift + P, Cmd + Shift + P
            if (ctrlOrMetaShift && (event.keyCode === KeyCodes.N || event.keyCode === KeyCodes.P)) {
                return true;
            }

            // Minimize the current window - Ctrl + M, Cmd + M
            if (ctrlOrMeta && event.keyCode === KeyCodes.M) {
                return true;
            }

            // Hide browser - Ctrl + H, Cmd + H
            // Hide all other windows - Ctrl + Alt + H, Cmd + Alt + H
            if ((ctrlOrMeta || ctrlOrMetaAlt) && event.keyCode === KeyCodes.H) {
                return true;
            }

            // Quit browser - Ctrl + Q, Cmd + Q
            if (ctrlOrMeta && event.keyCode === KeyCodes.Q) {
                return true;
            }

            // Zoom in on the page - Ctrl + '+', Cmd + '+'
            // Zoom out on the page - Ctrl + '-', Cmd + '-'
            // Reset zoom level - Ctrl + 0, Cmd + 0
            if (ctrlOrMeta && (event.keyCode === KeyCodes.NUM_PLUS  || event.keyCode === 187 /*Safari*/ || event.keyCode === 171 /*Firefox*/ ||
                               event.keyCode === KeyCodes.NUM_MINUS || event.keyCode === 189 /*Safari*/ || event.keyCode === 173 /*Firefox*/ ||
                               event.keyCode === KeyCodes.NUM_0     || event.keyCode === KeyCodes['0'])) {
                return true;
            }

            // Full-screen mode - F11, Ctrl + Cmd + F
            if ((event.keyCode === KeyCodes.F11) || (ctrlMeta && event.keyCode === KeyCodes.F)) {
                return true;
            }

            // Open the browsing history - Ctrl + H, Cmd + Shift + H
            if ((ctrl || metaShift) && event.keyCode === KeyCodes.H) {
                return true;
            }

            // Open the download history - Ctrl + J, Cmd + J, Cmd + Shift + J, Cmd + Alt + 2
            if (((ctrlOrMeta || metaShift) && event.keyCode === KeyCodes.J) || (metaAlt && event.keyCode === KeyCodes['2'])) {
                return true;
            }

            // Bookmark the current web site - Ctrl + D, Cmd + D
            if (ctrlOrMeta && event.keyCode === KeyCodes.D) {
                return true;
            }

            // Toggles the bookmarks bar                                               - Ctrl + Shift + B, Cmd + Shift + B
            // Save all open pages in your current window as bookmarks in a new folder - Ctrl + Shift + D, Cmd + Shift + D
            if (ctrlOrMetaShift && (event.keyCode === KeyCodes.B || event.keyCode === KeyCodes.D)) {
                return true;
            }

            // Open the Bookmarks window - Ctrl + Shift + B, Cmd + Shift + B, Cmd + Alt + B
            if ((ctrlOrMetaShift || metaAlt) && event.keyCode === KeyCodes.B) {
                return true;
            }

            // Open Firebug - F12
            if (event.keyCode === KeyCodes.F12) {
                return true;
            }

            // Open Developer Tools - Ctrl + Shift + (I,J,K,S), Cmd + Alt + (I,J,K,S)
            if ((ctrlShift || metaAlt) && (event.keyCode === KeyCodes.I ||
                                           event.keyCode === KeyCodes.J ||
                                           event.keyCode === KeyCodes.K ||
                                           event.keyCode === KeyCodes.S)) {
                return true;
            }
        }

        return false;
    }

    function getPrintableCharFromCharCode(charCode) {
        return (_.isNumber(charCode) && (charCode >= 32)) ? String.fromCharCode(charCode) : undefined;
    }

    function getPrintableChar(event) {
        return getPrintableCharFromCharCode(event.charCode) || getPrintableCharFromCharCode(event.which) || '';
    }

    // undo/redo --------------------------------------------------------------

    /**
     * Returns whether the passed undo action is an 'insertText' action (an
     * action with a 'insertText' redo operation and a 'deleteText' undo
     * operation).
     *
     * @param {UndoAction} action
     *  The action to be tested.
     *
     * @param {Boolean} single
     *  If true, the action must contain exactly one 'deleteText' undo
     *  operation and exactly one 'insertText' redo operation. Otherwise, the
     *  last operations of each of the arrays are checked, and the arrays may
     *  contain other operations.
     */
    function isInsertTextAction(action, single) {
        return (single ? (action.undoOperations.length === 1) : (action.undoOperations.length >= 1)) && (_.first(action.undoOperations).name === Operations.DELETE) &&
            (single ? (action.redoOperations.length === 1) : (action.redoOperations.length >= 1)) && (_.last(action.redoOperations).name === Operations.TEXT_INSERT);
    }

    /**
     * Checks, whether two directly following insertText operations can be merged. This merging
     * happens for two reasons:
     * 1. Create undo operations, that remove complete words, not only characters.
     * 2. Reduce the number of operations before sending them to the server.
     *
     * The merge is possible under the following circumstances:
     * 1. The next operation has no attributes AND the previous operation is NOT a
     *    change tracked operation
     * OR
     * 2. The next operation has only the 'changes' attribute (no other attributes
     *    allowed) and the previous operation has the same 'changes' attribute.
     *    In this comparison not the complete 'changes' attribute is compared, but
     *    only the 'author' property of the 'inserted' object. The date is ignored.
     *
     * @param {UndoAction} lastOperation
     *  The preceeding operation that will be extended if possible.
     *
     * @param {UndoAction} nextOperation
     *  The following operation that will be tried to merge into the preceeding operation.
     *
     * @returns {Boolean}
     *  Whether the two (insertText) operations can be merged successfully .
     */
    function mergeNeighboringOperations(lastOperation, nextOperation) {

        // Checking, if two following operations can be merged, although the next operation
        // contains an attribute. Merging is possible, if the attribute object only contains
        // the 'changes' object added by change tracking. If the previous operation contains
        // the same changes object, a merge is possible.
        function isValidNextChangesAttribute() {

            // checking, if the changes attribute is the only one of the next operation and if the author is defined
            if (nextOperation.attrs && nextOperation.attrs.changes && (_.keys(nextOperation.attrs).length === 1) && nextOperation.attrs.changes.inserted && nextOperation.attrs.changes.inserted.author) {
                // checking if the previous operation has the same changes attribute author
                if (lastOperation.attrs && lastOperation.attrs.changes && lastOperation.attrs.changes.inserted && lastOperation.attrs.changes.inserted.author) {
                    // if the authors are identical, the operations can be merged (ignoring the time)
                    if (nextOperation.attrs.changes.inserted.author === lastOperation.attrs.changes.inserted.author) {
                        return true;
                    }
                }
            }

            return false;
        }

        // Checking, if a specified operation is a change track operation.
        function isChangeTrackInsertOperation(operation) {
            return (operation.attrs && operation.attrs.changes);
        }

        // 1. There are no attributes in the next operation AND the previous operation was not a change track operation OR
        // 2. Both are change track operations and have valid change track attributes
        return ((!('attrs' in nextOperation) && !isChangeTrackInsertOperation(lastOperation)) || isValidNextChangesAttribute());
    }

    /**
     * Tries to merge the passed undo actions. Merging works only if both
     * actions represent a single 'insertText' operation, and the passed action
     * appends a single character directly after the text of this action.
     *
     * @param {UndoAction} lastAction
     *  The existing action that will be extended if possible.
     *
     * @param {UndoAction} nextAction
     *  The new action that will be tried to merge into the existing action.
     *
     * @returns {Boolean}
     *  Whether the passed new undo action has been merged successfully into
     *  the existing undo action.
     */
    function mergeUndoActionHandler(lastAction, nextAction) {

        var // check if this and the passed action is an 'insertText' action
            validActions = isInsertTextAction(lastAction, false) && isInsertTextAction(nextAction, true),

            // the redo operation of this action and the passed action
            lastRedo = _.last(lastAction.redoOperations),
            nextRedo = nextAction.redoOperations[0];

        // check that the operations are valid for merging the actions
        if (validActions && (nextRedo.text.length >= 1) && Position.hasSameParentComponent(lastRedo.start, nextRedo.start) && Position.hasSameTargetComponent(lastRedo.target, nextRedo.target)) {

            // check that the new action adds the character directly after the text of this action and does not change the attributes
            // check that the last character of this action is not a space character (merge actions word by word)
            // check that the operations can be merged corresponding to their attributes
            if ((_.last(lastRedo.start) + lastRedo.text.length === _.last(nextRedo.start)) && (lastRedo.text.substr(-1) !== ' ') && mergeNeighboringOperations(lastRedo, nextRedo)) {
                // merge undo operation (delete one more character)
                lastAction.undoOperations[0].end[lastAction.undoOperations[0].end.length - 1] += nextRedo.text.length;
                // merge redo operation (add the character)
                lastRedo.text += nextRedo.text;
                return true;
            }
        }
        return false;
    }

    // class Editor ===========================================================

    /**
     * The text editor model. Contains and manages the entire DOM of the edited
     * text document. Implements execution of all supported operations.
     *
     * Triggers the events supported by the base class EditModel, and the
     * following additional events:
     * - 'selection': When the editor selection has been changed. Event
     *      handlers receive a reference to the selection object (instance of
     *      class Selection).
     * - 'changeTrack:stateInfo': When the change tracking state of the document
     *      is modified. This event is triggered with the option 'state: true' after
     *      loading the document and the document contains change tracked elements
     *      or after receiving an operation with change track attributes.
     *      The event is triggered with 'state: false', if no change tracked element was
     *      found inside the document.
     * - 'drawingHeight:update': When the height of a drawing is changed. This can
     *      be used to update the borders of the drawing and surrounding groups.
     * - 'update:absoluteElements': When the document is modified in that way, that
     *      it is necessary to update all absolute positioned elements.
     * - 'cacheBuffer:flush': When edit rights are transfered, and there are still
     *      not registerd operations in buffer, like for update of date fields after load.
     * - 'change:pageSettings': This event is triggered after a change of the documents
     *      page settings.
     *
     * @constructor
     *
     * @extends EditModel
     * @extends ModelAttributesMixin
     * @extends ClipboardMixin
     *
     * @param {EditApplication} app
     *  The application containing this editor instance.
     */
    function Editor(app) {

        var // self reference for local functions
            self = this,

            // the root element for the document contents
            editdiv = DOM.createPageNode().attr({ contenteditable: true, tabindex: 1, 'data-focus-role': 'page', role: 'main' }).addClass('user-select-text noI18n f6-target'),

            // the undo manager of this document
            undoManager = null,

            // the logical selection, synchronizes with browser DOM selection
            selection = null,

            // the remote selection handling collaborative overlays
            remoteSelection = null,

            // the change track handler
            changeTrack = null,

            // the handler object for fields
            fieldManager = null,

            // the drawing layer for all drawings assigned to the page
            drawingLayer = null,

            // the comment layer for all comments
            commentLayer = null,

            // the handler for all range markers in the document
            rangeMarker = null,

            // instance handling visual page layout, like page breaks, headers/footers, etc.
            pageLayout = null,

            // instance handling search functions
            searchHandler = null,

            //
            spellChecker = null,

            // instance of text's NumberFormatter class
            numberFormatter = null,

            // shortcuts for style sheet containers
            characterStyles = null,
            paragraphStyles = null,
            tableStyles = null,
            tableRowStyles = null,
            tableCellStyles = null,
            drawingStyles = null,
            pageStyles = null,

            // shortcuts for other format containers
            listCollection = null,

            // values needed for pagebreaks calculus
            pageAttributes,
            pageMaxHeight,
            pagePaddingLeft,
            pagePaddingTop,
            pagePaddingBottom,
            pageWidth,
            pbState = true,
            quitFromPageBreak = false,
            blockOnInsertPageBreaks = false,
            draftModeState = false,

            // attributes that were set without a selection and are only set for a single character
            preselectedAttributes = null,

            // whether the preselected attributes must not be set to null (for example after splitParagraph)
            keepPreselectedAttributes = false,

            // whether mouse down occured in the editdiv (and no mouse up followed)
            activeMouseDownEvent = false,

            // the period of time, that a text input is deferred, waiting for a further text input (in ms).
            inputTextTimeout = 2,

            // the maximum number of characters that are buffered before an insertText operation is created.
            maxTextInputChars = 3,

            // whether a group of operations requires, that the undo stack must be deleted
            deleteUndoStack = false,

            // whether an operation is part of an undo group
            isInUndoGroup = false,

            undoRedoRunning = false,

            // whether the user accepted, that an undo is not possible
            userAcceptedMissingUndo = false,

            // the promise that controls an already active (deferred) text input
            inputTextPromise = null,

            // a deferred that is resolved, after the text is inserted. This is required for the undo group and must be global.
            insertTextDef = null,

            // the content, that will be printed in the current active text input process
            activeInputText = null,

            // a paragraph node, that is implicit and located behind a table, might temporarely get an increased height
            increasedParagraphNode = null,

            // all paragraphs (and their background-color) that are artificially highlighted
            highlightedParagraphs = [],

            // the osn of the document during loading
            docLoadOSN = null,

            // whether an update of element formatting is required
            requiresElementFormattingUpdate = true,

            // whether an operation was triggered by a GUI action (for table operations)
            guiTriggeredOperation = false,

            // whether the keyboard events need to be blocked, for example during pasting
            // of internal clipboard
            blockKeyboardEvent = false,

            // whether an internal or external clipboard paste is currently running
            pasteInProgress = false,

            // the current event from the keyDown handler
            lastKeyDownEvent,

            // the logical position where the operation ends, can be used for following cursor positioning
            lastOperationEnd,     // Need to decide: Should the last operation modify this member, or should the selection be passed up the whole call chain?!

            // whether an undo shall set the selection range after applying the operation
            useSelectionRangeInUndo = false,

            // indicates an active ime session
            imeActive = false,
            // ime objects now stored in a queue as we can have more than
            // one ime processing at a time. due to asynchronous processing
            imeStateQueue = [],
            // ime update cache
            imeUpdateText = null,
            // true if a compositionend event has been received
            imeCompositionEndReceived = false,

            // mutation observer
            editDivObserver = null,
            // iPad read-only paragraph
            roIOSParagraph = null,

            // operations recording
            recordOperations = true,

            // collects information for debounced list update
            updateListsDebouncedOptions = {},

            // collects information for debounced page break insertion
            currentProcessingNode = null,

            // collects information for debounced automatic resizing of text frame nodes
            currentProcessingTextFrameNode = null,

            // collects information for debounced target node update
            targetNodeForUpdate = null,

            // whether the paragraph cache can be used (only for internal operations)
            useParagraphCache = false,

            // Performance: Caching the synchronously used paragraph element
            paragraphCache = null,

            // handling double click and triple click in IE
            // whether there is a double click event waiting for a third click
            doubleClickEventWaiting = false,
            // whether the third click occured after a double click, activating the tripleclick event
            tripleClickActive = false,
            // the period of time, that a double click event 'waits' for a further click (in ms).
            doubleClickTimeOut = 60,

            //Android Chrome Code
            androidTimeouter = null,
            androidTyping = false,

            // global var holding info are header and footer in state of editing
            headerFooterEditState = false,

            // global var holding container root node for header and footer state
            $headFootRootNode = $(),

            // reference to parent node of header footer root node
            $parentOfHeadFootRootNode = $(),

            // reference to index of header footer root node, if parent is detached
            rootNodeIndex = null,

            // string holding current editable element's target id
            activeTarget = '',

            // whether the document was loaded with in 'fastLoad' process
            fastLoadImport = false,

            // whether the document was loaded with in 'localStorage' process
            localStorageImport = false,

            // whether a current change of selection is triggered by a remote client
            updatingRemoteSelection = false,

            // public listeners for IPAD workaround, all important listeners on editdiv are save, that we can reuse them
            listenerList = null,

            // the maximum number of top level nodes that defines a large selection (that need to be handled asynchronously)
            MAX_TOP_LEVEL_NODES = 100;

        // base constructor ---------------------------------------------------

        EditModel.call(this, app, { mergeUndoActionHandler: mergeUndoActionHandler, generatorClass: TextOperationsGenerator, operationsFinalizer: finalizeOperations });
        ModelAttributesMixin.call(this);
        ClipboardMixin.call(this, app);

        StylesheetMixin.call(this, app);

        // public methods -----------------------------------------------------

        /**
         * Returns the root DOM element representing this editor.
         *
         * @returns {jQuery}
         *  The root node containing the document model and representation, as
         *  jQuery object.
         */
        this.getNode = function () {
            return editdiv;
        };

        /**
         * Selects the entire document.
         *
         * @returns {Editor}
         *  A reference to this instance.
         */
        this.selectAll = function () {
            selection.selectAll();
            return this;
        };

        /**
         * Returning the drawing layer, that handles all drawings
         * located inside the drawing layer node.
         *
         * @returns {Object}
         *  A reference to the drawing layer object.
         */
        this.getDrawingLayer = function () {
            return drawingLayer;
        };

        /**
         * Returning the comment layer object, that handles all comments
         * in the document.
         *
         * @returns {Object}
         *  A reference to the change track object.
         */
        this.getCommentLayer = function () {
            return commentLayer;
        };

        /**
         * Returning the range marker object, that handles all range marker
         * nodes in the document.
         *
         * @returns {Object}
         *  A reference to the range marker object.
         */
        this.getRangeMarker = function () {
            return rangeMarker;
        };

        /**
         * Returning the complex field object, that handles all complex fields
         * in the document.
         *
         * @returns {Object}
         *  A reference to the complex field object.
         */
        this.getFieldManager = function () {
            return fieldManager;
        };

        /**
         * Returning the change track object, that handles all change
         * tracks in the document.
         *
         * @returns {Object}
         *  A reference to the change track object.
         */
        this.getChangeTrack = function () {
            return changeTrack;
        };

        /**
         * Getter for pageLayout object instance.
         *
         * @returns{Object}
         * A reference to the page layout object.
         *
         */
        this.getPageLayout = function () {
            return pageLayout;
        };

        /**
         * Returns instance of NumberFormatter class.
         *
         * @returns {Object}
         */
        this.getNumberFormatter = function () {
            return numberFormatter;
        };

        /**
         * Whether the document was loaded from local storage.
         *
         * @returns {Boolean}
         *  Returns whether the document was loaded from local storage.
         *
         */
        this.isLocalStorageImport = function () {
            return localStorageImport;
        };

        /**
         * Whether the document was loaded with 'fast load' process.
         *
         * @returns {Boolean}
         *  Returns whether the document was loaded with 'fast load' process.
         *
         */
        this.isFastLoadImport = function () {
            return fastLoadImport;
        };

        /**
         * Whether undo or redo operations are currently running.
         *
         * @returns {Boolean}
         *  Returns whether undo or redo operations are currently running.
         *
         */
        this.isUndoRedoRunning = function () {
            return undoRedoRunning;
        };

        /**
         * Setting the global variable lastOperationEnd. This is necessary,
         * because more and more operation handlers are handled in external
         * modules and they need to set the end of their operation.
         *
         * @param {Number[]} position
         *  The logical position.
         */
        this.setLastOperationEnd = function (pos) {
            lastOperationEnd = pos;
        };

        /**
         * Getting the global variable lastOperationEnd. This is necessary,
         * because more and more operation are generated in external
         * modules and they need to set the selection after applying the
         * operations.
         *
         * @returns {Number[]}
         *  The logical position.
         */
        this.getLastOperationEnd = function () {
            return lastOperationEnd;
        };

        /**
         * Creates and returns an implicit paragraph node, that can be directly
         * included into the dom (validated and updated).
         *
         * @returns {HTMLElement}
         *  The implicit paragraph
         */
        this.getValidImplicitParagraphNode = function () {
            var paragraph = DOM.createImplicitParagraphNode();
            validateParagraphNode(paragraph);
            paragraphStyles.updateElementFormatting(paragraph);
            return paragraph;
        };

        /**
         * Prepares a 'real' paragraph after insertion of text, tab, drawing, ...
         * by exchanging an 'implicit' paragraph (in empty documents, empty cells,
         * behind tables, ...) with the help of an operation. Therefore the server is
         * always informed about creation and removal of paragraphs and implicit
         * paragraphs are only required for user input in the browser.
         *
         * @param {Number[]} position
         *  The logical text position.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.ignoreLength=false]
         *      If set to true, no check for the paragraph length is done. Normally
         *      a implicit paragraph must be empty. In special cases this check
         *      must be omitted.
         */
        this.doCheckImplicitParagraph = function (position, options) {
            handleImplicitParagraph(position, options);
        };

        /**
         * Public API function for the private function with the same name.
         * Shows a warning dialog with Yes/No buttons before deleting document
         * contents that cannot be restored.
         *
         * @param {String} title
         *  The title of the dialog box.
         *
         * @param {String} message
         *  The message text shown in the dialog box.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved if the Yes
         *  button has been pressed, or rejected if the No button has been pressed.
         */
        this.showDeleteWarningDialog = function (title, message) {
            return showDeleteWarningDialog(title, message);
        };

        // operations API -----------------------------------------------------

        /**
         * Copies the current selection into the internal clipboard and deletes
         * the selection.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  dialog has been closed with the default action; or rejected, if the
         *  dialog has been canceled. If no dialog is shown, the deferred is
         *  resolved immediately.
         */
        this.cut = function (event) {

            var // the clipboard event data
                clipboardData = Utils.getClipboardData(event);

            // set the internal clipboard data and
            // add the data to the clipboard event if the browser supports the clipboard api
            this.copy(event);

            // if the browser supports the clipboard api, use the copy function to add data to the clipboard
            // the iPad supports the api, but cut doesn't work
            if (clipboardData && !Utils.IOS) {

                // prevent default cut handling for desktop browsers, but not for touch devices
                if (!Utils.TOUCHDEVICE) {
                    event.preventDefault();
                }

                // delete current selection
                return self.deleteSelected().done(function () {
                    selection.setTextSelection(selection.getStartPosition()); // setting the cursor position
                });

            }

            return this.executeDelayed(function () {
                // focus and restore browser selection
                selection.restoreBrowserSelection();
                // delete restored selection
                return self.deleteSelected().done(function () {
                    selection.setTextSelection(selection.getStartPosition()); // setting the cursor position
                });
            }, undefined, 'Text: cut');
        };

        /**
         * Generates operations needed to copy the current text selection to
         * the internal clipboard.
         *
         * @returns {Array}
         *  The operations array that represents the current selection.
         */
        function copyTextSelection() {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // zero-based index of the current content node
                targetPosition = 0,
                // result of the iteration process
                result = null,
                // OX Text default paragraph list style sheet
                listParaStyleId = self.getDefaultUIParagraphListStylesheet(),
                // indicates if we had to add a paragraph style
                listParaStyleInserted = false,
                // the applied list style ids
                listStyleIds = [],
                // the attribute property for the change track
                changesFamily = 'changes',
                // a logical helper start position, if a text frame is selected
                startPosition = null,
                // attributes of the contentNode
                attributes,
                // a new created operation
                newOperation = null;

            // in the case of a text cursor selection inside a selected text frame, the text frame itself can be used for copying
            if (selection.getSelectionType() === 'text' && !self.hasSelectedRange() && selection.isAdditionalTextframeSelection()) {
                startPosition = Position.getOxoPosition(self.getNode(), selection.getSelectedTextFrameDrawing(), 0);
                selection.setTextSelection(startPosition, Position.increaseLastIndex(startPosition));
            }

            // ignoring empty selections
            if (selection.isTextCursor()) { return []; }

            // in case we are in header/footer and we have a selection that contains special page fields, don't copy
            if (fieldManager.checkIfSpecialFieldsSelected()) {
                return [];
            }

            // visit the paragraphs and tables covered by the text selection
            result = selection.iterateContentNodes(function (contentNode, position, startOffset, endOffset) {

                // paragraphs may be covered partly
                if (DOM.isParagraphNode(contentNode)) {

                    // if we have a list add the list paragraph style and the list style
                    if (paragraphStyles.containsStyleSheet(listParaStyleId)) {

                        if (!listParaStyleInserted) {
                            generator.generateMissingStyleSheetOperation('paragraph', listParaStyleId);
                            listParaStyleInserted = true;
                        }

                        attributes = paragraphStyles.getElementAttributes(contentNode);
                        if (attributes.paragraph && attributes.paragraph.listStyleId && !_.contains(listStyleIds, attributes.paragraph.listStyleId)) {
                            newOperation = listCollection.getListOperationFromListStyleId(attributes.paragraph.listStyleId);
                            if (newOperation) {  // check for valid operation (38007)
                                generator.generateOperation(Operations.INSERT_LIST, newOperation);
                                listStyleIds.push(attributes.paragraph.listStyleId);
                            }
                        }

                    }

                    // first or last paragraph: generate operations for covered text components
                    if (_.isNumber(startOffset) || _.isNumber(endOffset)) {

                        // some selections might cause invalid logical positions (38095)
                        if (_.isNumber(startOffset) && _.isNumber(endOffset) && (endOffset < startOffset)) { return Utils.BREAK; }

                        // special handling for Firefox and IE when selecting list paragraphs.
                        // if the selection includes the end position of a non list paragraph
                        // above the list, we don't add an empty paragraph to the document
                        // fix for bug 29752
                        if ((_.browser.Firefox || _.browser.IE) &&
                                Position.getParagraphLength(editdiv, position) === startOffset &&
                                !DOM.isListLabelNode(contentNode.firstElementChild) &&
                                Utils.findNextNode(editdiv, contentNode, DOM.PARAGRAPH_NODE_SELECTOR) &&
                                DOM.isListLabelNode(Utils.findNextNode(editdiv, contentNode, DOM.PARAGRAPH_NODE_SELECTOR).firstElementChild)) {
                            return;
                        }

                        // generate a splitParagraph and setAttributes operation for
                        // contents of first paragraph (but for multiple-paragraph
                        // selections only)
                        if (!_.isNumber(endOffset)) {
                            generator.generateOperation(Operations.PARA_SPLIT, { start: [targetPosition, 0] });
                        }

                        // handling for all paragraphs, also the final (36825)
                        generator.generateSetAttributesOperation(contentNode, { start: [targetPosition] }, { clearFamily: 'paragraph', ignoreFamily: changesFamily });

                        // operations for the text contents covered by the selection
                        generator.generateParagraphChildOperations(contentNode, [targetPosition], { start: startOffset, end: endOffset, targetOffset: 0, clear: true, ignoreFamily: changesFamily });

                    } else {

                        // skip embedded implicit paragraphs
                        if (DOM.isImplicitParagraphNode(contentNode)) { return; }

                        // generate operations for entire paragraph
                        generator.generateParagraphOperations(contentNode, [targetPosition], { ignoreFamily: changesFamily });
                    }

                // entire table: generate complete operations array for the table (if it is not an exceeded-size table)
                } else if (DOM.isTableNode(contentNode)) {

                    // skip embedded oversized tables
                    if (DOM.isExceededSizeTableNode(contentNode)) { return; }

                    // generate operations for entire table
                    generator.generateTableOperations(contentNode, [targetPosition], { ignoreFamily: changesFamily });

                } else {
                    Utils.error('Editor.copyTextSelection(): unknown content node "' + Utils.getNodeName(contentNode) + '" at position ' + JSON.stringify(position));
                    return Utils.BREAK;
                }

                targetPosition += 1;

            }, this, { shortestPath: true });

            // return operations, if iteration has not stopped on error
            return (result === Utils.BREAK) ? [] : generator.getOperations();
        }

        /**
         * Sets a paragraph to contentEditable = false while removing it
         * from an optional previous remembered paragraph.
         *
         * paragraph {HTMLElement|null}
         *  The paragraph where contentEditable should be set to false.
         */
        function setIOSROParagraph(paragraph) {
            if (roIOSParagraph !== null) {
                $(roIOSParagraph).removeAttr('contentEditable');
                roIOSParagraph = null;
            }
            if (paragraph && paragraph.isContentEditable) {
                paragraph.contentEditable = 'false';
                roIOSParagraph = paragraph;
            }
        }

        /**
         * Generates operations needed to copy the current cell range selection
         * to the internal clipboard.
         *
         * @returns {Array}
         *  The operations array that represents the current selection.
         */
        function copyCellRangeSelection() {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // information about the cell range
                cellRangeInfo = selection.getSelectedCellRange(),
                // merged attributes of the old table
                oldTableAttributes = null,
                // explicit attributes for the new table
                newTableAttributes = null,
                // all rows in the table
                tableRowNodes = null,
                // relative row offset of last visited cell
                lastRow = -1,
                // the attribute property for the change track
                changesFamily = 'changes',
                // result of the iteration process
                result = null;

            // generates operations for missing rows and cells, according to lastRow/lastCol
            function generateMissingRowsAndCells(row/*, col*/) {

                // generate new rows (repeatedly, a row may be covered completely by merged cells)
                while (lastRow < row) {
                    lastRow += 1;
                    generator.generateOperationWithAttributes(tableRowNodes[lastRow], Operations.ROWS_INSERT, { start: [1, lastRow], count: 1, insertDefaultCells: false }, { ignoreFamily: changesFamily });
                }

                // TODO: detect missing cells, which are covered by merged cells outside of the cell range
                // (but do not generate cells covered by merged cells INSIDE the cell range)
            }

            if (!cellRangeInfo) {
                Utils.error('Editor.copyCellRangeSelection(): invalid cell range selection');
                return [];
            }

            // split the paragraph to insert the new table between the text portions
            generator.generateOperation(Operations.PARA_SPLIT, { start: [0, 0] });

            // generate the operation to create the new table
            oldTableAttributes = tableStyles.getElementAttributes(cellRangeInfo.tableNode);
            newTableAttributes = AttributeUtils.getExplicitAttributes(cellRangeInfo.tableNode);
            newTableAttributes.table = newTableAttributes.table || {};
            newTableAttributes.table.tableGrid = oldTableAttributes.table.tableGrid.slice(cellRangeInfo.firstCellPosition[1], cellRangeInfo.lastCellPosition[1] + 1);
            if (newTableAttributes && newTableAttributes[changesFamily]) { delete newTableAttributes[changesFamily]; }
            generator.generateOperation(Operations.TABLE_INSERT, { start: [1], attrs: newTableAttributes });

            // all covered rows in the table
            tableRowNodes = DOM.getTableRows(cellRangeInfo.tableNode).slice(cellRangeInfo.firstCellPosition[0], cellRangeInfo.lastCellPosition[0] + 1);

            // visit the cell nodes covered by the selection
            result = selection.iterateTableCells(function (cellNode, position, row, col) {

                // generate operations for new rows, and for cells covered by merged cells outside the range
                generateMissingRowsAndCells(row, col);

                // generate operations for the cell
                generator.generateTableCellOperations(cellNode, [1, row, col], { ignoreFamily: changesFamily });
            });

            // missing rows at bottom of range, covered completely by merged cells (using relative cellRangeInfo, task 30839)
            generateMissingRowsAndCells(cellRangeInfo.lastCellPosition[0] - cellRangeInfo.firstCellPosition[0], cellRangeInfo.lastCellPosition[1] - cellRangeInfo.firstCellPosition[1] + 1);

            // return operations, if iteration has not stopped on error
            return (result === Utils.BREAK) ? [] : generator.getOperations();
        }

        /**
         * collects all attrs for assigned styleId out of family-stylecollection
         * and writes all data into the assigned generator
         */
        this.generateInsertStyleOp = function (generator, family, styleId, setDefault) {
            var styleCollection = this.getStyleCollection(family),
                param = {
                    attrs: styleCollection.getStyleSheetAttributeMap(styleId),
                    type: family,
                    styleId: styleId,
                    styleName: styleCollection.getName(styleId),
                    parent: styleCollection.getParentId(styleId),
                    uiPriority: styleCollection.getUIPriority(styleId),
                    hidden: styleCollection.isHidden(styleId)
                };

            if (!_.isUndefined(setDefault)) { param['default'] = setDefault; }

            // parent is an optional value, should not be send as 'null'
            if (param.parent === null) { delete param.parent; }

            generator.generateOperation(Operations.INSERT_STYLESHEET, param);
            styleCollection.setDirty(styleId, false);
        };

        /**
         * Copies the current selection into the internal clipboard and
         * attaches the clipboard data to the copy event if the browser
         * supports the clipboard api.
         */
        this.copy = function (event) {

            var // the clipboard div
                clipboard,
                // the clipboard event data
                clipboardData = Utils.getClipboardData(event),
                // html clipboard data cleaned up for export
                htmlExportData,
                // start and end of current selection
                start = selection.getStartPosition(),
                end = selection.getEndPosition(),
                // information about the cell range and containing table
                cellRangeInfo,
                // is the cell defined by the start position the first cell in the row
                isFirstCell,
                // is the cell defined by the end position the last cell in the row
                isLastCell;

            if (_.browser.IE && (end[0] - start[0]) > 200) {
                // bigger selection take more time to copy,
                // in IE it can take more than 10seconds,
                // so we warn the user
                if (!window.confirm(gt('The selected text is very long. It will take some time to be copied. Do you really want to continue?'))) { return; }
            }

            // generate a new unique id to identify the clipboard operations
            clipboardId = ox.session + ':' + _.uniqueId();

            switch (selection.getSelectionType()) {

            case 'text':
                cellRangeInfo = selection.getSelectedCellRange();
                isFirstCell = $(Position.getLastNodeFromPositionByNodeName(editdiv, start, DOM.TABLE_CELLNODE_SELECTOR)).prev().length === 0;
                isLastCell = $(Position.getLastNodeFromPositionByNodeName(editdiv, end, DOM.TABLE_CELLNODE_SELECTOR)).next().length === 0;

                // if the selected range is inside the same table or parent table and
                // the start position is the first cell in the start row and the end position
                // is the last cell in the end row use table selection otherwise, use text selection.
                if (cellRangeInfo && isFirstCell && isLastCell && !_.isEqual(cellRangeInfo.firstCellPosition, cellRangeInfo.lastCellPosition)) {
                    clipboardOperations = copyCellRangeSelection();
                } else {
                    clipboardOperations = copyTextSelection();
                }
                break;

            case 'drawing':
                clipboardOperations = copyTextSelection();
                break;

            case 'cell':
                clipboardOperations = copyCellRangeSelection();
                break;

            default:
                clipboardId = '';
                Utils.error('Editor.copy(): unsupported selection type: ' + selection.getSelectionType());
            }

            htmlExportData = Export.getHTMLFromSelection(self, { clipboardId: clipboardId, clipboardOperations: clipboardOperations });
            // set clipboard debug pane content
            this.trigger('debug:clipboard', htmlExportData);

            // if browser supports clipboard api add data to the event
            // chrome say it supports clipboard api, but it does not!!!
            // Bug 39428
            if (clipboardData) {
                // add operation data
                clipboardData.setData('text/ox-operations', JSON.stringify(clipboardOperations));
                // add plain text and html of the current browser selection
                clipboardData.setData('text/plain', selection.getTextFromBrowserSelection());
                clipboardData.setData('text/html', htmlExportData);

                // prevent default copy handling for desktop browsers, but not for touch devices
                event.preventDefault();
            } else {

                // copy the currently selected nodes to the clipboard div and append it to the body
                clipboard = app.getView().createClipboardNode();

                clipboard.append(htmlExportData);

                // focus the clipboard node and select all of it's child nodes
                selection.setBrowserSelectionToContents(clipboard);

                this.executeDelayed(function () {
                    // set the focus back
                    app.getView().grabFocus();
                    // remove the clipboard node
                    clipboard.remove();
                }, undefined, 'Text: copy');
            }
        };

        /**
         * Returns whether the internal clipboard contains operations.
         */
        this.hasInternalClipboard = function () {
            return clipboardOperations.length > 0;
        };

        /**
         * Deletes the current selection and pastes the internal clipboard to
         * the resulting cursor position.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  dialog has been closed with the default action; or rejected, if the
         *  dialog has been canceled. If no dialog is shown, the deferred is
         *  resolved immediately.
         */
        this.pasteInternalClipboard = function (dropPosition) {

            // check if clipboard contains something
            if (!this.hasInternalClipboard()) { return $.when(); }

            // make sure that only one paste call is processed at the same time
            if (checkSetClipboardPasteInProgress()) { return $.when(); }

            // Group all executed operations into a single undo action.
            // The undo manager returns the return value of the callback function.
            return undoManager.enterUndoGroup(function () {

                var // the deferred to keep the undo group open until it is resolved or rejected
                    undoDef = $.Deferred(),
                    // target position to paste the clipboard contents to
                    anchorPosition = null,
                    // the generated paste operations with transformed positions
                    operations = null,
                    // operation generator for additional insert style sheets operations
                    generator = this.createOperationsGenerator(),
                    // the next free list style number part
                    listStyleNumber = 1,
                    // the list style map
                    listStyleMap = {},
                    // a snapshot object
                    snapshot = null,
                    // whether the user aborted the pasting action
                    userAbort = false,
                    // whether there is a selection range before pasting
                    hasRange = selection.hasRange(),
                    // the logical range position
                    rangeStart = null, rangeEnd = null,
                    // whether a operation was removed from the list of operations
                    operationRemoved = false;

                // builds the list style map from insertListStyle operations, mapping the list style ids
                // of the source document to the list style ids of the destination document
                function createListStyleMap(operation) {

                    var listStyleId,
                        listLevel = null,
                        isKnownListStyle = false;

                    function getFreeListStyleId() {

                        var sFreeId = 'L';

                        while (listCollection.hasListStyleId('L' + listStyleNumber)) {
                            listStyleNumber++;
                        }
                        sFreeId += listStyleNumber;
                        listStyleNumber++;
                        return sFreeId;
                    }

                    if ((operation.name === Operations.INSERT_LIST) && _.isObject(operation.listDefinition)) {

                        // deep copy the operation before changing
                        operation = _.copy(operation, true);

                        // check if we already have a list style with the list definition and if not create a new id
                        listStyleId = listCollection.getListStyleIdForListDefinition(operation.listDefinition);

                        if (listStyleId) {
                            isKnownListStyle = true;
                        } else {
                            listStyleId = getFreeListStyleId();
                        }

                        listStyleMap[operation.listStyleId] = listStyleId;
                        operation.listStyleId = listStyleId;

                        // handling the base style id (that cannot be sent to the server)
                        if (operation.baseStyleId) {

                             // if the base style Id is the same, do not send any operation (filter cannot handle base style id)
                            if (isKnownListStyle && operation.baseStyleId === listCollection.getBaseStyleIdFromListStyle(operation.listStyleId)) {
                                operationRemoved = true;  // needs to be marked for cleanup
                                return null;
                            }

                            // but never send baseStyleId information to the server because filter cannot handle it correctly
                            delete operation.baseStyleId;
                        }

                        // make sure every list level has a listStartValue set
                        for (listLevel in operation.listDefinition) {
                            if (operation.listDefinition.hasOwnProperty(listLevel) && !operation.listDefinition[listLevel].hasOwnProperty('listStartValue')) {
                                operation.listDefinition[listLevel].listStartValue = 1;
                            }
                        }
                    }

                    return operation;
                }

                // transforms a position being relative to [0,0] to a position relative to anchorPosition
                function transformPosition(position) {

                    var // the resulting position
                        resultPosition = null;

                    if ((position[0] === 0) && (position.length > 1)) {
                        // adjust text/drawing offset for first paragraph
                        resultPosition = anchorPosition.slice(0, -1);
                        resultPosition.push(anchorPosition[anchorPosition.length - 1] + position[1]);
                        resultPosition = resultPosition.concat(position.slice(2));
                    } else {
                        // adjust paragraph offset for following paragraphs
                        resultPosition = anchorPosition.slice(0, -2);
                        resultPosition.push(anchorPosition[anchorPosition.length - 2] + position[0]);
                        resultPosition = resultPosition.concat(position.slice(1));
                    }

                    return resultPosition;
                }

                // if the operation references a style try to add the style sheet to the document
                function addMissingStyleSheet(operation) {
                    if ((operation.name === Operations.TABLE_INSERT) && _.isObject(operation.attrs) && _.isString(operation.attrs.styleId)) {
                        // generate operation to insert a dirty table style into the document
                        generator.generateMissingStyleSheetOperation('table', operation.attrs.styleId);
                    }
                }

                // apply list style mapping to setAttributes and insertParagraph operations
                function mapListStyles(operation) {

                    if ((operation.name === Operations.SET_ATTRIBUTES || operation.name === Operations.PARA_INSERT) &&
                            operation.attrs && operation.attrs.paragraph && _.isString(operation.attrs.paragraph.listStyleId) &&
                        listStyleMap[operation.attrs.paragraph.listStyleId]) {

                        // apply the list style id from the list style map to the operation
                        operation.attrs.paragraph.listStyleId = listStyleMap[operation.attrs.paragraph.listStyleId];
                    }
                }

                // changes all drawing from floating to inline
                function mapDrawing(operation) {
                    if (operation.name === Operations.DRAWING_INSERT && operation.attrs.drawing && !operation.attrs.drawing.inline) {
                        _.extend(operation.attrs.drawing, { inline: true, anchorHorBase: null, anchorHorAlign: null, anchorHorOffset: null, anchorVertBase: null, anchorVertAlign: null, anchorVertOffset: null, textWrapMode: null, textWrapSide: null });
                    }
                }

                // transforms the passed operation relative to anchorPosition
                function transformOperation(operation) {

                    // clone the operation to transform the positions (no deep clone,
                    // as the position arrays will be recreated, not modified inplace)
                    operation = _.clone(operation);

                    // transform position of operation (but not, if a target is defined (for example in comments)
                    if (_.isArray(operation.start) && !operation.target) {
                        // start may exist but is relative to position then
                        operation.start = transformPosition(operation.start);
                        // attribute 'end' only with attribute 'start'
                        if (_.isArray(operation.end)) {
                            operation.end = transformPosition(operation.end);
                        }
                        addMissingStyleSheet(operation);
                    }

                    // map list style ids from source to destination document
                    mapListStyles(operation);
                    // change drawing from floating to inline - fix for bug #32873
                    mapDrawing(operation);

                    var text = '  name="' + operation.name + '", attrs=';
                    var op = _.clone(operation);
                    delete op.name;
                    Utils.log(text + JSON.stringify(op));

                    return operation;
                }

                // helper function to check, if the operations can be pasted to the anchor position
                function isForbiddenPasting(anchorPosition, clipboardOperations) {

                    // helper function to find insertDrawing operations with type 'shape'
                    function containsInsertShapeDrawing(operations) {
                        return _.find(operations, function (operation) {
                            return operation.name && operation.name === 'insertDrawing' && operation.type && operation.type === 'shape';
                        });
                    }

                    // helper function to find insertComment operations
                    function containsInsertComment(operations) {
                        return _.find(operations, function (operation) {
                            return operation.name && operation.name === 'insertComment';
                        });
                    }

                    // helper function to find specified operations in the list of operations
                    function containsSpecifiedOperations(operations, operationList) {
                        return _.find(operations, function (operation) {
                            return operation.name && _.contains(operationList, operation.name);
                        });
                    }

                    // pasting text frames into text frames or comments is not supported
                    if (containsInsertShapeDrawing(clipboardOperations)) {
                        if (Position.isPositionInsideTextframe(self.getNode(), anchorPosition)) {
                            // inform the user, that this pasting is not allowed
                            app.getView().yell({ type: 'info', message: gt('Pasting shapes into text frames is not supported.') });
                            return true;
                        } else if (self.isCommentFunctionality()) {
                            // inform the user, that this pasting is not allowed
                            app.getView().yell({ type: 'info', message: gt('Pasting shapes into comments is not supported.') });
                            return true;
                        }
                    }

                    // pasting comments is only allowed into the main document (target must not be defined)
                    if (containsInsertComment(clipboardOperations) && self.getActiveTarget()) {
                        // inform the user, that this pasting is not allowed
                        app.getView().yell({ type: 'info', message: gt('Pasting content with comments is not allowed at target position.') });
                        return true;
                    }

                    // pasting tables or drawings into 'shapes with text content' in odf or into comments in odf is not supported
                    if (containsSpecifiedOperations(clipboardOperations, ['insertDrawing', 'insertTable']) && (self.isReducedOdfTextframeFunctionality() || self.isOdfCommentFunctionality())) {
                        // inform the user, that this pasting is not allowed
                        app.getView().yell({ type: 'info', message: gt('Pasting content into this object is not supported.') });
                        return true;
                    }

                    return false;
                }

                // finding a valid text cursor position after pasting
                function getFinalCursorPosition(defaultPosition, operations) {

                    var // the determined cursor position
                        cursorPosition = defaultPosition,
                        // the searched split paragraph operation
                        searchOperation = null,
                        // searching for a splitting of paragraph
                        searchOperationName = Operations.PARA_SPLIT,
                        // the position after splitting a paragraph
                        splitPosition = null;

                    if (operations && operations.length > 0) {
                        // setting the cursor after pasting.
                        // Also taking care of splitted paragraphs (36471) -> search last split paragraph operation
                        searchOperation = _.find(operations.reverse(), function (operation) {
                            return operation.name && operation.name === searchOperationName;
                        });
                    }

                    if (searchOperation) {
                        // is this position after split behind the defaultPosition? Then is should be used
                        splitPosition = _.clone(searchOperation.start);
                        splitPosition.pop();  // the paragraph position
                        splitPosition = Position.increaseLastIndex(splitPosition);
                        splitPosition.push(0);  // setting cursor to start of second part of splitted paragraph
                        if (Position.isValidPositionOrder(cursorPosition, splitPosition)) { cursorPosition = splitPosition; }
                    }

                    return cursorPosition;
                }

                // applying the paste operations
                function doPasteInternalClipboard() {

                    var // the apply actions deferred
                        applyDef = null,
                        // the newly created operations
                        newOperations = null,
                        // the current target node
                        target = self.getActiveTarget();

                    // init the next free list style number part
                    listStyleNumber = parseInt(listCollection.getFreeListId().slice(1, listCollection.getFreeListId().length), 10);
                    if (!_.isNumber(listStyleNumber)) {
                        listStyleNumber = 1;
                    }

                    // paste clipboard to current cursor position
                    anchorPosition = dropPosition ? dropPosition.start : selection.getStartPosition();
                    if (anchorPosition.length >= 2) {

                        // doing some checks, if the content can be pasted into the destination element.
                        if (isForbiddenPasting(anchorPosition, clipboardOperations)) {
                            return $.when();
                        }

                        Utils.info('Editor.pasteInternalClipboard()');
                        self.setBlockKeyboardEvent(true);

                        // creating a snapshot
                        if (!snapshot) { snapshot = new Snapshot(app); }

                        // show a nice message with cancel button
                        app.getView().enterBusy({
                            cancelHandler: function () {
                                userAbort = true;  // user aborted the pasting process
                                // restoring the document state
                                snapshot.apply();
                                // calling abort function for operation promise
                                app.enterBlockOperationsMode(function () { if (applyDef && applyDef.abort) { applyDef.abort(); } });
                            },
                            warningLabel: gt('Sorry, pasting from clipboard will take some time.')
                        });

                        // map the operations
                        operations = _(clipboardOperations).map(createListStyleMap);
                        if (operationRemoved) { operations = Utils.removeFalsyItemsInArray(operations); }
                        operations = _(operations).map(transformOperation);

                        // concat with newly generated operations
                        newOperations = generator.getOperations();
                        if (newOperations.length) {
                            operations = newOperations.concat(operations);
                        }

                        // adding change track information, if required
                        if (changeTrack.isActiveChangeTracking()) {
                            changeTrack.handleChangeTrackingDuringPaste(operations);
                        } else {
                            changeTrack.removeChangeTrackInfoAfterSplitInPaste(operations);
                        }

                        // handling target positions (for example inside comments)
                        commentLayer.handlePasteOperationTarget(operations);
                        fieldManager.handlePasteOperationTarget(operations);

                        // check if destination paragraph for paste already has pageBreakBefore attribute
                        pageLayout.handleManualPageBreakDuringPaste(operations);

                        handleImplicitParagraph(anchorPosition);

                        // if pasting into header or footer or comment, extend operations with target
                        if (target) {
                            _.each(operations, function (operation) {
                                if (operation.name !== Operations.INSERT_STYLESHEET) {
                                    operation.target = target;
                                }
                            });
                        }

                        // set clipboard debug pane content
                        self.trigger('debug:clipboard', operations);

                        // apply operations
                        applyDef = self.applyOperations(operations, { async: true })
                        .progress(function (progress) {
                            app.getView().updateBusyProgress(progress);
                        });

                    } else {
                        Utils.warn('Editor.pasteInternalClipboard(): invalid cursor position');
                        applyDef = $.Deferred().reject();
                    }

                    return applyDef.promise();
                }

                // creating a snapshot
                if (hasRange) {
                    snapshot = new Snapshot(app);
                    rangeStart = _.clone(selection.getStartPosition());
                    rangeEnd = _.clone(selection.getEndPosition());
                }

                // delete current selection
                self.deleteSelected({ alreadyPasteInProgress: true, snapshot: snapshot })
                .then(doPasteInternalClipboard)
                .always(function () {
                    // no longer blocking page break calculations (40107)
                    self.setBlockOnInsertPageBreaks(false);
                    // leaving busy mode
                    leaveAsyncBusy();
                    // close undo group
                    undoDef.resolve();
                    // deleting the snapshot
                    if (snapshot) { snapshot.destroy(); }
                    // setting the cursor range after cancel by user
                    if (userAbort && hasRange) { selection.setTextSelection(rangeStart, rangeEnd); }
                }).done(function () {
                    selection.setTextSelection(getFinalCursorPosition(lastOperationEnd, operations));
                });

                return undoDef.promise();

            }, this); // enterUndoGroup()
        };

        this.paste = function (event) {

            var // the clipboard div
                clipboard,
                // the clipboard event data
                clipboardData = Utils.getClipboardData(event),
                // the list items of the clipboard event data
                items = clipboardData && clipboardData.items,
                // the list of mime types of the clipboard event data
                types = clipboardData && clipboardData.types,
                // the operation data from the internal clipboard
                eventData,
                // the file reader
                reader,
                // the list item of type text/html
                htmlEventItem = null,
                // the list item of type text/plain
                textEventItem = null,
                // the file list item
                fileEventItem = null,
                // the event URL data
                urlEventData = null;

            if (!self.getEditMode()) {
                //paste via burger-menu in FF and Chrome must be handled
                app.rejectEditAttempt();
                event.preventDefault();
                return;
            }

            // handles the result of reading file data from the file blob received from the clipboard data api
            function onLoadHandler(evt) {
                var data = evt && evt.target && evt.target.result;

                if (data && data.substring(0, 10) === 'data:image') {
                    createOperationsFromExternalClipboard([{ operation: Operations.DRAWING_INSERT, data: data, depth: 0 }]);
                } else {
                    app.rejectEditAttempt('image');
                }
            }

            // returns true if the html clipboard has a matching clipboard id set
            function isHtmlClipboardIdMatching(html) {
                return ($(html).find('#ox-clipboard-data').attr('data-ox-clipboard-id') === clipboardId);
            }

            // returns the operations attached to the html clipboard, or null
            function getHtmlAttachedOperations(html) {
                var operations;

                try {
                    operations = JSON.parse($(html).find('#ox-clipboard-data').attr('data-ox-operations') || '{}');
                } catch (e) {
                    Utils.warn('getHtmlAttachedOperations', e);
                    operations = null;
                }

                return operations;
            }

            // if the browser supports the clipboard api, look for operation data
            // from the internal clipboard to handle as internal paste.
            if (clipboardData) {
                eventData = clipboardData.getData('text/ox-operations');
                if (eventData) {
                    // prevent default paste handling for desktop browsers, but not for touch devices
                    if (!Utils.TOUCHDEVICE) {
                        event.preventDefault();
                    }

                    // set the operations from the event to be used for the paste
                    clipboardOperations = (eventData.length > 0) ? JSON.parse(eventData) : [];
                    self.pasteInternalClipboard();
                    return;
                }

                // check if clipboardData contains a html item
                htmlEventItem = _.find(items, function (item) { return item.type.toLowerCase() === 'text/html'; });

                // Chrome doesn't paste images into a (content editable) div, check if clipboardData contains an image item
                fileEventItem = _.find(items, function (item) { return item.type.toLowerCase().indexOf('image') !== -1; });

                // check if we have a mime type to get an URL from
                urlEventData = clipboardData.getData(_.find(types, function (type) { return type.toLowerCase().indexOf('text/uri-list') !== -1; }));

                // check if clipboardData contains a plain item
                textEventItem = _.find(items, function (item) { return item.type.toLowerCase() === 'text/plain'; });

                if (htmlEventItem || textEventItem) {
                    (htmlEventItem || textEventItem).getAsString(function (content) {
                        var div, ops;

                        // set clipboard debug pane content
                        self.trigger('debug:clipboard', content);

                        // clean up html to put only harmless html into the <div>
                        div = $('<div>').html(Utils.parseAndSanitizeHTML(content));

                        if (htmlEventItem) {
                            if (isHtmlClipboardIdMatching(div)) {
                                // if the clipboard id matches it's an internal copy & paste
                                self.pasteInternalClipboard();

                            } else {
                                ops = getHtmlAttachedOperations(div);
                                if (_.isArray(ops)) {
                                    // it's not an internal copy & paste, but we have clipboard operations piggy backed to use for clipboardOperations
                                    clipboardOperations = ops;
                                    self.pasteInternalClipboard();

                                } else {
                                    // use html clipboard
                                    ops = self.parseClipboard(div);
                                    createOperationsFromExternalClipboard(ops);
                                }
                            }
                        } else {
                            //text only
                            ops = self.parseClipboardText(div.text());
                            createOperationsFromExternalClipboard(ops);
                        }
                    });

                } else if (fileEventItem) {
                    reader = new window.FileReader();
                    reader.onload = onLoadHandler;
                    reader.readAsDataURL(fileEventItem.getAsFile());

                } else if (urlEventData && Image.hasUrlImageExtension(urlEventData)) {
                    createOperationsFromExternalClipboard([{ operation: Operations.DRAWING_INSERT, data: urlEventData, depth: 0 }]);
                }

                if (htmlEventItem || fileEventItem || urlEventData || textEventItem) {
                    // prevent default paste handling of the browser
                    event.preventDefault();
                    return;
                }
            }

            // append the clipboard div to the body and place the cursor into it
            clipboard = app.getView().createClipboardNode();

            // focus and select the clipboard container node
            selection.setBrowserSelectionToContents(clipboard);

            // read pasted data
            this.executeDelayed(function () {

                var clipboardData,
                    operations;

                // set the focus back
                app.getView().grabFocus();

                if (isHtmlClipboardIdMatching(clipboard)) {
                    // if the clipboard id matches it's an internal copy & paste
                    self.pasteInternalClipboard();

                } else {
                    // look for clipboard operations
                    operations = getHtmlAttachedOperations(clipboard);
                    if (_.isArray(operations)) {
                        // it's not an internal copy & paste, but we have clipboard operations piggy backed to use for clipboardOperations
                        clipboardOperations = operations;
                        self.pasteInternalClipboard();

                    } else {
                        // set clipboard debug pane content
                        self.trigger('debug:clipboard', clipboard);
                        // use html clipboard
                        clipboardData = self.parseClipboard(clipboard);
                        createOperationsFromExternalClipboard(clipboardData);
                    }
                }

                // remove the clipboard node
                clipboard.remove();
            }, undefined, 'Text: paste');
        };

        /**
         * Sets the state wether a cipboard paste is in progress
         *
         * @param {Boolean} state
         *  The clipboard pasting state
         */
        function setClipboardPasteInProgress(state) {
            pasteInProgress = state;
        }

        /**
         * Returns true if a clipboard paste is still in progress, otherwise false.
         * And then sets the state to true.
         * and the processing of this paste
         *
         * @returns {Boolean}
         *  Wether a clipboard paste is in progress
         */
        function checkSetClipboardPasteInProgress() {
            var state = pasteInProgress;
            pasteInProgress = true;
            return state;
        }

        /**
         * Replays the operations provided.
         *
         * @param {Array|String} operations
         *  An array of operations to be replayed (processed) or a string
         *  of operations in JSON format to be replayed.
         *
         * @param {Number} [maxOSN]
         *  An optional maximum number, until that the operations will be
         *  executed.
         *
         * @param {Boolean} external
         *  Specifies if the operations are interal or external.
         */
        this.replayOperations = function (operations, maxOSN) {
            var // operations to replay
                operationsToReplay = null,
                // delay for operations replay
                delay = 10;

            function replayOperationNext(operationArray) {
                var // the next operation if available
                    op = (operationArray.length > 0) ? operationArray.shift() : null,
                    // the result of apply operation
                    result;

                if (op && (!maxOSN || !op.osn || (op.osn && op.osn <= maxOSN))) {
                    result = self.applyOperations(op);
                    if (result) {
                        self.executeDelayed(function () {
                            replayOperationNext(operationArray);
                        }, delay, 'Text: replayOperations');
                    }
                }
            }

            if (operations && operations.length > 0) {
                if (_.isString(operations)) {
                    try {
                        operationsToReplay = JSON.parse(operations);
                    } catch (e) {
                        // nothing to do
                    }
                } else if (_.isArray(operations)) {
                    operationsToReplay = operations;
                }

                if (operationsToReplay) {
                    // now try to process the operations
                    replayOperationNext(operationsToReplay);
                }
            } else {
                // This is only a debug/developer feature therefore no need
                // to translate this message!
                app.getView().yell({ type: 'error', message: 'Recorder cannot play operations. Format unknown or not supported.' });
            }
        };

        /**
         * Returning the explicit attributes from a node, that is specified
         * by a logical position.
         *
         * @param {Number[]} pos
         *  The logical position of the target element.
         *
         * @returns {String}
         *  The explicit attributes of the node. If there are no explicit
         *  attributes, an empty string is returned.
         */
        this.getExplicitAttributesString = function (pos) {

            if (!pos) { return ''; }

            var // the node info at the specified position
                element = Position.getDOMPosition(self.getCurrentRootNode(), pos, true),
                // the node at the specified position
                node = null,
                // the node attributes
                attrs = null;

            if (!element || !element.node) { return ''; }
            node = element.node;
            attrs = AttributeUtils.getExplicitAttributes(node);
            return attrs ? JSON.stringify(attrs) : '';
        };

        /**
         * Helper function for generating a logging string, that contains
         * all strings of an array in unique, sorted form together with
         * the number of occurences in the array.
         * Input:
         * ['abc', 'def', 'abc', 'ghi', 'def']
         * Output:
         * 'abc (2), def (2), ghi'
         *
         * @param {Array} arr
         *  An array with strings.
         *
         * @returns {String}
         *  The string constructed from the array content.
         */
        function getSortedArrayString(arr) {

            var counter = {},
                arrayString = '';

            _.countBy(arr, function (elem) {
                counter[elem] = counter[elem] ? counter[elem] + 1 : 1;
            });

            _.each(_.keys(counter).sort(), function (elem, index) {
                var newString = elem + ' (' + counter[elem] + ')';
                newString += ((index + 1) < _.keys(counter).length) ? ', ' : '';
                arrayString += newString;
            });

            return arrayString;
        }

        /**
         * Returning a list of objects describing the extensions that are necessary
         * to load a document from the local storage.
         * One object can contain the following properties:
         *  extension: This is the required string extension, that is used to save
         *      one layer (page content, drawing layer, ...) in the local
         *      storage.
         *  optional:  Whether this is an optional value from local storage (like
         *      drawing layer) or a required value (like page content).
         *  additionalOptions: Some additional options that can be set to influence
         *      the loading process.
         *
         * @returns {Object[]}
         *  An array with objects describing the strings that are used in the local
         *  storage.
         */
        this.getSupportedStorageExtensions = function () {

            var // the list of all supported extensions for the local registry
                allExtensions = [];

            // the page content layer
            allExtensions.push({ extension: '', optional: 'false' });
            // the drawing layer
            allExtensions.push({ extension: '_DL', optional: 'true', additionalOptions: { drawingLayer: true } });
            // header&footer layer
            allExtensions.push({ extension: '_HFP', optional: 'true', additionalOptions: { headerFooterLayer: true } });
            // the comment layer
            allExtensions.push({ extension: '_CL', optional: 'true', additionalOptions: { commentLayer: true } });

            return allExtensions;
        };

        /**
         * Generating a document string from the editdiv element. jQuery data
         * elements are saved at the sub elements with the attribute 'jquerydata'
         * assigned to these elements. Information about empty paragraphs
         * is stored in the attribute 'nonjquerydata'.
         *
         * @param {jQuery.Deferred} [maindef]
         *  The deferred used to update the progress.
         *
         * @param {Number} [startProgress]
         *  The start value of the notification progress.
         *
         * @param {Number} [endProgress]
         *  The end value of the notification progress.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected if the document strings
         *  could be generated successfully. All document strings are packed into
         *  an array. Several document strings are generated for each layer, that
         *  needs to be stored in the local storage. This are the page content,
         *  the drawing layer, ... .
         */
        this.getFullModelDescription = function (maindef, startProgress, endProgress) {

            var // collecting all keys from data object for debug reasons
                dataKeys = [],
                // collector for all different saved nodes (page content, drawing layer , ...)
                allNodesCollector = [],
                // collector for all html strings and the extension in the registry for every layer (page content, drawing layer , ...)
                allHtmlStrings = [],
                // the number of chunks, that will be evaluated
                chunkNumber = 5,
                // the current notification progress
                currentNotify = startProgress,
                // the notify difference for one chunk
                notifydiff = Utils.round((endProgress - startProgress) / chunkNumber, 0.01),
                // a selected drawing
                selectedDrawing = null,
                // the padding-bottom value saved at the pagecontent node
                paddingBottomInfo = null;

            // converting all editdiv information into a string, saving especially the data elements
            // finding all elements, that have a 'data' jQuery object set.
            // Helper function to split the array of document nodes into smaller parts.
            function prepareNodeForStorage(node) {

                var // the jQuery data object of the node
                    dataObject = null,
                    // the return promise
                    def = $.Deferred();

                // updating progress bar
                if (maindef) {
                    currentNotify += Utils.round(notifydiff, 0.01);
                    maindef.notify(currentNotify); // updating corresponding to the length inside each -> smaller snippets deferred.
                }

                try {

                    // the current node, as jQuery object
                    node = $(node);

                    // simplify restauration of empty text nodes
                    if ((node.is('span')) && (DOM.isEmptySpan(node))) {
                        dataKeys.push('isempty');
                        // using nonjquerydata instead of jquerydata (32286)
                        node.attr('nonjquerydata', JSON.stringify({ isempty: true }));
                    }

                    // removing additional tracking handler options, that are specific for OX Text (Chrome only)
                    if (_.browser.WebKit && DOM.isDrawingFrame(node)) {
                        node.removeData('trackingOptions');
                        node.removeData('tracking-options');
                    }

                    // removing spell check information at node
                    spellChecker.clearSpellcheckHighlighting(node);

                    // handling jQuery data objects
                    dataObject = node.data();
                    if (_.isObject(dataObject) && !_.isEmpty(dataObject)) {
                        if (dataObject[DOM.DRAWINGPLACEHOLDER_LINK]) { delete dataObject[DOM.DRAWINGPLACEHOLDER_LINK]; }  // no link to other drawing allowed
                        if (dataObject[DOM.DRAWING_SPACEMAKER_LINK]) { delete dataObject[DOM.DRAWING_SPACEMAKER_LINK]; }  // no link to other drawing allowed
                        if (dataObject[DOM.COMMENTPLACEHOLDER_LINK]) { delete dataObject[DOM.COMMENTPLACEHOLDER_LINK]; }  // no link to other comment allowed
                        dataKeys = dataKeys.concat(_.keys(dataObject));
                        node.attr('jquerydata', JSON.stringify(dataObject));  // different key -> not overwriting isempty value
                        // Utils.log('Saving data: ' + this.nodeName + ' ' + this.className + ' : ' + JSON.stringify($(this).data()));

                        // handling images that have the session used in the src attribute
                        // Avoiding error 403 during saving the document in the console (Bug 37640)
                        if (DOM.isImageNode(node)) {
                            var imgNode = node.find('img'),
                                srcAttr = imgNode.attr('src');
                            if (_.isString(srcAttr) && (srcAttr.length > 0)) {
                                imgNode.attr('src', null);
                                imgNode.attr(DOM.LOCALSTORAGE_SRC_OVERRIDE, srcAttr.replace(/\bsession=(\w+)\b/, 'session=_REPLACE_SESSION_ID_'));
                            }
                        }
                    }

                    def.resolve();

                } catch (ex) {
                    // dataError = true;
                    Utils.info('quitHandler, failed to save document in local storage (3): ' + ex.message);
                    def.reject();
                }

                return def.promise();
            }  // end of 'prepareNodeForStorage'

            // removing drawing selections
            if (selection.getSelectionType() === 'drawing') {
                selectedDrawing = selection.getSelectedDrawing();
                DrawingFrame.clearSelection(selectedDrawing);
                // removing additional tracking handler options, that are specific for OX Text (Chrome only)
                selectedDrawing.removeData('trackingOptions');
                selectedDrawing.removeData('tracking-options');
            }

            // removing selections around 'additional' text frame selections
            if (selection.isAdditionalTextframeSelection()) {
                selectedDrawing = selection.getSelectedTextFrameDrawing();
                DrawingFrame.clearSelection(selectedDrawing);
                // removing additional tracking handler options, that are specific for OX Text (Chrome only)
                selectedDrawing.removeData('trackingOptions');
                selectedDrawing.removeData('tracking-options');
            }

            // removing artificial selection of empty paragraphs
            if (!_.isEmpty(highlightedParagraphs)) { removeArtificalHighlighting(); }

            // removing a change track selection, if it exists
            if (changeTrack.getChangeTrackSelection()) { changeTrack.clearChangeTrackSelection(); }

            // removing classes at a highlighted comment thread (because comment thread is active or hovered), if necessary
            commentLayer.clearCommentThreadHighlighting();

            // removing an optionally set comment author filter
            commentLayer.removeAuthorFilter();

            // removing active header/footer context menu, if exists
            if (self.isHeaderFooterEditState()) {
                pageLayout.leaveHeaderFooterEditMode();
            }

            // removing a simulated cursor
            if (selection.isSimulatedCursorActive()) { selection.clearSimulatedCursor(); }

            // removing search highlighting
            searchHandler.clearHighlighting();

            // removing artifical increased size of an implicit paragraph
            if (increasedParagraphNode) {
                $(increasedParagraphNode).css('height', 0);
                increasedParagraphNode = null;
            }

            // First layer: Collecting all nodes of the page content
            allNodesCollector.push({ extension: '', selector: DOM.PAGECONTENT_NODE_SELECTOR, allNodes: self.getNode().children(DOM.PAGECONTENT_NODE_SELECTOR).find('*') });

            // saving specific page content node styles at its first child (style 'padding-bottom')
            paddingBottomInfo = self.getNode().children(DOM.PAGECONTENT_NODE_SELECTOR).css('padding-bottom');
            self.getNode().children(DOM.PAGECONTENT_NODE_SELECTOR).children(':first').attr('pagecontentattr', JSON.stringify({ 'padding-bottom': paddingBottomInfo }));

            // Second layer: Handling the drawing layer (taking care of order in allNodesCollector)
            if (drawingLayer.containsDrawingsInPageContentNode()) { allNodesCollector.push({ extension: '_DL', selector: DOM.DRAWINGLAYER_NODE_SELECTOR, allNodes: self.getNode().children(DOM.DRAWINGLAYER_NODE_SELECTOR).find('*') }); }

            // Third layer: Handling the header/footer placeholder layer (taking care of order in allNodesCollector)
            if (pageLayout.hasContentHeaderFooterPlaceHolder()) { allNodesCollector.push({ extension: '_HFP', selector: DOM.HEADER_FOOTER_PLACEHOLDER_SELECTOR, allNodes: self.getNode().children(DOM.HEADER_FOOTER_PLACEHOLDER_SELECTOR).find('*') }); }

            // Fourth layer: Handling the comment layer (taking care of order in allNodesCollector)
            if (!commentLayer.isEmpty()) { allNodesCollector.push({ extension: '_CL', selector: DOM.COMMENTLAYER_NODE_SELECTOR, allNodes: self.getNode().children(DOM.COMMENTLAYER_NODE_SELECTOR).find('*') }); }

            // iterating over all nodes (first page content, then drawing layer, ...)
            return self.iterateArraySliced(allNodesCollector, function (oneLayer) {

                return self.iterateArraySliced(oneLayer.allNodes, prepareNodeForStorage, { delay: 'immediate', infoString: 'Text: getFullModelDescription: inner' })
                .done(function () {
                    // collecting the html strings for each layer
                    allHtmlStrings.push({ extension: oneLayer.extension, htmlString: self.getNode().children(oneLayer.selector).html() });
                });
            }, { infoString: 'Text: getFullModelDescription: outer' })
            .then(function () {
                return allHtmlStrings;
            });

        };

        /**
         * Generating the editdiv element from an html string. jQuery data
         * elements are restored at the sub elements, if the attribute 'jquerydata'
         * is assigned to these elements.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.usedLocalStorage=false]
         *      If set to true, the specified html string was loaded from the
         *      local storage. In this case, no browser specific modifications
         *      need to be done on the string.
         *  @param {Boolean} [options.drawingLayer=false]
         *      If set to true, the specified html string is the drawing layer.
         *  @param {Boolean} [options.commentLayer=false]
         *      If set to true, the specified html string is the comment layer.
         *  @param {Boolean} [options.headerFooterLayer=false]
         *      If set to true, the specified html string is the header/footer placeholder layer.
         *
         * @param {String} htmlString
         *  The html string for the editdiv element.
         */
        this.setFullModelNode = function (htmlString, options) {

            var // collecting all keys assigned to the data objects
                dataKeys = [],
                // the data objects saved in the string for the nodes
                dataObject = null,
                // whether the html string was read from the local storage
                usedLocalStorage = Utils.getBooleanOption(options, 'usedLocalStorage', false),
                // whether the html string describes the comment layer (loaded from local storage)
                isCommentLayer = Utils.getBooleanOption(options, 'commentLayer', false),
                // whether the html string describes the drawing layer (loaded from local storage)
                isDrawingLayer = Utils.getBooleanOption(options, 'drawingLayer', false),
                // whether the html string describes the header/footer layer (loaded from local storage)
                isHeaderFooterLayer = Utils.getBooleanOption(options, 'headerFooterLayer', false),
                // eather the page content node, parent of the editdiv, or headerFooterContainer node
                contentNode = null;

            // helper function to get the node in which the html string needs to be inserted
            function getStringParentNode() {

                var // the parent node for each layer
                    parentNode = null;

                if (isDrawingLayer) {
                    parentNode = drawingLayer.getDrawingLayerNode();
                } else if (isCommentLayer) {
                    parentNode = commentLayer.getOrCreateCommentLayerNode({ visible: false });
                } else if (isHeaderFooterLayer) {
                    parentNode = pageLayout.getHeaderFooterPlaceHolder();
                } else {
                    parentNode = self.getNode().children(DOM.PAGECONTENT_NODE_SELECTOR);
                }

                return parentNode;
            }

            // registering import type
            if (!app.isImportFinished()) {
                if (usedLocalStorage) {
                    localStorageImport = true;
                } else {
                    fastLoadImport = true;
                }
            }

            if (htmlString) {

                // Avoiding error 403 during loading the document in the console (35405)
                // -> replacing the '_REPLACE_SESSION_ID_' before it is inserted into the DOM
                // -> this is a problem, if the user really uses this replacement string in the dom
                htmlString = htmlString.replace(/\bsession=_REPLACE_SESSION_ID_\b/g, 'session=' + ox.session);

                // setting the content node for each layer
                contentNode = getStringParentNode();

                // setting new string to editdiv node
                contentNode
                    .html(htmlString)
                    .find('[jquerydata], [nonjquerydata]')
                    .each(function () {
                        var node = $(this),
                            imgNode = null,
                            srcAttr = null;

                        if (node.attr('nonjquerydata')) {
                            try {
                                dataObject = JSON.parse(node.attr('nonjquerydata'));
                                for (var key in dataObject) {
                                    if (key === 'isempty') {  // the only supported key for nonjquerydata
                                        DOM.ensureExistingTextNode(this);
                                        dataKeys.push(key);
                                    }
                                }
                                node.removeAttr('nonjquerydata');
                            } catch (ex) {
                                Utils.error('Failed to parse attributes: ' + node.attr('jquerydata') + ' Exception message: ' + ex.message);
                            }
                        }

                        if (node.attr('jquerydata')) {
                            // Utils.log('Restoring style at: ' + this.nodeName + ' ' + this.className + ' : ' + $(this).attr('jquerydata'));
                            try {
                                dataObject = JSON.parse(node.attr('jquerydata'));

                                // No list style in comments in odt: 38829 (simply removing the list style id)
                                if (app.isODF() && !usedLocalStorage && isCommentLayer && DOM.isParagraphNode(node) && dataObject.attributes && dataObject.attributes.paragraph && dataObject.attributes.paragraph.listStyleId) { delete dataObject.attributes.paragraph.listStyleId; }

                                for (var key in dataObject) {
                                    node.data(key, dataObject[key]);
                                    dataKeys.push(key);
                                }

                                node.removeAttr('jquerydata');
                            } catch (ex) {
                                Utils.error('Failed to parse attributes: ' + node.attr('jquerydata') + ' Exception message: ' + ex.message);
                            }
                        }

                        if (DOM.isImageNode(node)) {
                            imgNode = node.find('img');
                            srcAttr = imgNode.attr(DOM.LOCALSTORAGE_SRC_OVERRIDE);
                            if (_.isString(srcAttr) && (srcAttr.length > 0)) {
                                imgNode.attr('src', srcAttr);
                                imgNode.attr(DOM.LOCALSTORAGE_SRC_OVERRIDE, null);
                            }
                        }
                    });

                contentNode.find('table[size-exceed-values]').each(function () {
                    var // the values provided by the backend which specifies
                        // what part of the table exceeds the size
                        values = $(this).attr('size-exceed-values'),
                        // the separate parts
                        parts = null;

                    if (values) {
                        parts = values.split(',');
                        if (parts && parts.length === 3) {
                            DOM.makeExceededSizeTable($(this), parts[0], parts[1], parts[2]);
                        }
                    }
                });

                // setting table cell attribute 'contenteditable' to true for non-MSIE browsers (task 33642)
                // -> this is not necessary, if the document was loaded from the local storage
                if (!usedLocalStorage && !_.browser.IE) { contentNode.find('div.cell').attr('contenteditable', true); }

                // assigning page content attributes, that are saved at its first child
                if (contentNode.children(':first').attr('pagecontentattr')) {
                    dataObject = JSON.parse(contentNode.children(':first').attr('pagecontentattr'));
                    if (dataObject['padding-bottom']) {
                        contentNode.css('padding-bottom', dataObject['padding-bottom']);
                    }
                    contentNode.children(':first').removeAttr('pagecontentattr');
                }

                if (!_.isEmpty(dataKeys)) { Utils.info('Setting data keys: ' + getSortedArrayString(dataKeys)); }

            }
        };

        /**
         * Debugging code for performance measurements. html code is the parameter for
         * this function. This code is sent to the server, returned again and included
         * into the DOM at the global editdiv element.
         *
         * @param {String} htmlCode
         *  The html code for the global editdiv element.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  formatting has been updated successfully, or rejected when an error
         *  has occurred.
         */
        this.applyPageHtmlCode = function (htmlCode) {

            var // parameters for the server request
                params = { action: 'gethtmlcode', htmlcode: htmlCode };

            return app.sendFileRequest(IO.FILTER_MODULE_NAME, params, {
                method: 'POST',
                resultFilter: function (data) {
                    // returning undefined rejects the entire request
                    return Utils.getStringOption(data, 'htmlcode');
                }
            })
            .done(function (markup) {
                editdiv.html(markup);
            });
        };

        this.setRecordingOperations = function (state) {
            recordOperations = state;
        };

        this.isRecordingOperations = function () {
            return recordOperations;
        };

        function initDocument() {

            var // container for the top-level paragraphs
                pageContentNode = DOM.getPageContentNode(editdiv),
                // the initial paragraph node in an empty document
                paragraph = DOM.createImplicitParagraphNode();

            // create empty page with single paragraph
            pageContentNode.empty().append(paragraph);
            validateParagraphNode(paragraph);

            // in Internet Explorer it is necessary to add new empty text nodes in paragraph again
            // newly, also in other browsers (new jQuery version?)
            repairEmptyTextNodes(paragraph);

            // initialize default page formatting
            pageStyles.updateElementFormatting(editdiv);

            // register keyboard event handlers on the clipboard node to handle key strokes when
            // a drawing is selected in which case the browser selection is inside the clipboard node
            selection.getClipboardNode().on({
                keydown: processKeyDown,
                keypress: processKeyPressed,
                cut: _.bind(self.cut, self),
                copy: _.bind(self.copy, self)
            });
        }

        // ==================================================================
        // HIGH LEVEL EDITOR API which finally results in Operations
        // and creates Undo Actions.
        // Public functions, that are called from outside the editor
        // and generate operations.
        // ==================================================================

        /**
         * special flag for the very slow android chrome
         */
        this.isAndroidTyping = function () {
            return androidTyping;
        };

        /**
         *
         */
        this.handleAndroidTyping = function () {
            androidTyping = true;
            if (androidTimeouter) {
                window.clearTimeout(androidTimeouter);
            }
            androidTimeouter = window.setTimeout(function () {
                androidTyping = false;
            }, 1000);
        };

        /**
         * Generates the operations that will delete the current selection, and
         * executes the operations.
         * Info: With the introduction of 'rangeMarker.handlePartlyDeletedRanges' it is
         * now possible, that not only the end position of the current selection is
         * modified within 'deleteSelected', but also the start position. This happens
         * for example, if the selection includes an end range marker. In this case the
         * corresponding start range marker is also removed. If this is located in the
         * same paragraph as the current selection start position, this position will
         * also be updated. Therefore after executing 'deleteSelected', it is necessary
         * that the start position is read from the selection again without relying on
         * the value before executing 'deleteSelected'.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.alreadyPasteInProgress=false]
         *      Whether deleteSelected was triggered from the pasting of clipboard. In
         *      this case the check function 'checkSetClipboardPasteInProgress()' does
         *      not need to be called, because blocking is already handled by the
         *      clipboard function.
         *  @param {Object} [options.snapshot]
         *      The snapshot that can be used, if the user cancels a long running
         *      delete action. Typically this is created inside this function, but if
         *      it was already created by the calling 'paste' function, it can be reused
         *      here.

         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  dialog has been closed with the default action; or rejected, if the
         *  dialog has been canceled. If no dialog is shown, the deferred is
         *  resolved immediately.
         */
        this.deleteSelected = function (options) {

            var // the operations generator
                generator = this.createOperationsGenerator(),
                // the logical position of the first and current partially covered paragraph
                firstParaPosition = null, currentParaPosition = null,
                // an optional new start position that can be caused by additionally removed start range nodes outside the selection range
                newStartPosition = null,
                // a helper paragraph for setting attributes
                paragraph = null,
                // whether the selection is a rectangular cell selection
                isCellSelection = selection.getSelectionType() === 'cell',
                // whether it is necessary to ask the user before deleting content
                askUser = false,
                // whether an additional merge operation is required
                doMergeParagraph = false,
                // the paragraph at the end of the selection
                selectionEndParagraph = null,
                // the promise for generating the operations
                operationGeneratorPromise = null,
                // the promise for asking the user (if required)
                askUserPromise = null,
                // the promise for the asychronous execution of operations
                operationsPromise = null,
                // a node counter for the iterator function
                counter = 0,
                // the first and last node of iteration
                firstNode = null, lastNode = null, currentNode = null,
                // whehter a node is the first node  of a selection
                isFirstNode = false,
                // wheter the first or last paragraph of a selection are only partly removed
                firstParaRemovedPartly = false, lastParaRemovedPartly = false,
                // whether it is necessary to check, if the last paragraph of the selection need to be merged
                checkMergeOfLastParagraph = false,
                // changeTrack: Whether the content was really deleted (inserted by same user) or only attributes were set
                changeTrackDeletedContent = false,
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // current element container node
                rootNode = self.getCurrentRootNode(target),
                // whether it is required to check for range markers inside the selection
                isRangeCheckRequired = !selection.isAllSelected() && !rangeMarker.isEmpty(),
                // all classes used to identify the range markers
                allRangeMarkerSelector = isRangeCheckRequired ? DOM.RANGEMARKERNODE_SELECTOR : null,
                // the collector for all range marker nodes (including comments)
                allRangeMarkerNodes = $(),
                // created operation
                newOperation = null,
                // whether a progress bar shall be made visible (this might already be done by an outer function)
                alreadyPasteInProgress = Utils.getBooleanOption(options, 'alreadyPasteInProgress', false),
                // whether a valid snapshot was already created by the caller of this function (typically a 'paste' action over a selection range)
                snapshot = Utils.getObjectOption(options, 'snapshot', null),
                // whether the snapshot was created inside this function
                snapshotCreated = false,
                // whether the deleting will be done asynchronous (in large selections)
                asyncDelete = false,
                // the ratio of operation generation to applying of operation
                operationRatio = 0.3,
                // an optional array with selection ranges for large selections
                splittedSelectionRange = null,
                // whether the user aborted the delete process
                userAbort = false;

            // helper function, that is necessary because the user might be asked,
            // if he really wants to delete the content.
            function doDeleteSelected() {

                // clean up function after the operations were applied
                function postDeleteOperations() {

                    guiTriggeredOperation = false;

                    if (firstParaPosition) {
                        // set paragraph attributes immediately, if paragraph is without content
                        paragraph = Position.getParagraphElement(rootNode, firstParaPosition);
                        if ((paragraph) && ($(paragraph).text().length === 0)) {
                            validateParagraphNode(paragraph);
                        }
                    }

                    if (!isCellSelection) {
                        if (currentParaPosition) {
                            paragraph = Position.getParagraphElement(rootNode, currentParaPosition);
                            if ((paragraph) && ($(paragraph).text().length === 0)) {
                                validateParagraphNode(paragraph);
                            }
                        } else {
                            Utils.warn('Editor.doDeleteSelected(): currentParaPosition is missing!');
                        }

                        // special handling for additionally removed range start handlers
                        if (newStartPosition) { lastOperationEnd = newStartPosition; }

                        if (changeTrack.isActiveChangeTracking() && !changeTrackDeletedContent) {
                            // if change tracking is active, and the content was not deleted (inserted by the same user before?),
                            // then the old selection can be reused.
                            lastOperationEnd = selection.getStartPosition();  // setting cursor before(!) removed selection
                        }
                    } else {
                        // cell selection
                        if (firstParaPosition) {   // not complete table selected
                            lastOperationEnd = Position.getFirstPositionInCurrentCell(rootNode, firstParaPosition);
                        }
                    }

                    // collapse selection (but not after canceling delete process)
                    if (!userAbort) {
                        selection.setTextSelection(lastOperationEnd);
                    }
                }

                if (asyncDelete) {

                    // apply the operations (undo group is created automatically)
                    guiTriggeredOperation = true;  // Fix for 30597

                    // applying all operations asynchronously
                    operationsPromise = applyTextOperationsAsync(generator, null, { showProgress: false, leaveOnSuccess: true, progressStart: operationRatio });

                    operationsPromise.always(function () {
                        postDeleteOperations();
                    });

                } else {

                    // apply the operations (undo group is created automatically)
                    guiTriggeredOperation = true;  // Fix for 30597

                    // applying all operations synchronously
                    self.applyOperations(generator);

                    postDeleteOperations();

                    operationsPromise = $.when();
                }

                return operationsPromise;
            }

            // helper function to generate all delete operations
            function generateAllDeleteOperations(selectionRange) {

                var // the logical start position, if specified
                    localStartPos = (selectionRange && selectionRange[0]) || null,
                    // the logical end position, if specified
                    localEndPos = (selectionRange && selectionRange[1]) || null;

                // visit all content nodes (tables, paragraphs) in the selection
                selection.iterateContentNodes(function (node, position, startOffset, endOffset, parentCovered) {

                    var // whether the node is the last child of its parent
                        isLastChild = node === node.parentNode.lastChild,
                        // whether a paragraph is selected completely
                        paragraphSelected = false,
                        // a collector of all rows of a table
                        allRows = null;

                    counter++;  // counting the nodes
                    if (counter === 1) {
                        isFirstNode = true;
                        firstNode = node;
                    } else {
                        isFirstNode = false;
                        lastNode = node;
                        lastParaRemovedPartly = false;
                    }

                    // saving the current node
                    currentNode = node;

                    if (DOM.isParagraphNode(node)) {

                        // remember first and last paragraph
                        if (!firstParaPosition) { firstParaPosition = position; }
                        currentParaPosition = position;

                        if (_.isNumber(startOffset) && !_.isNumber(endOffset)) {
                            if (startOffset === 0) {
                                paragraphSelected = true;  // the full paragraph can be delete
                            } else {
                                checkMergeOfLastParagraph = true;  // for the final paragraph, a merge-check is necessary after iteration
                            }
                        }

                        // checking if a paragraph is selected completely and requires merge
                        paragraphSelected = _.isNumber(startOffset) && (startOffset === 0) && !_.isNumber(endOffset);

                        if (changeTrack.isActiveChangeTracking()) { // complete separation of code for change tracking

                            // do not delete the paragraph node, if it is only covered partially;
                            // or if it is the last paragraph when the parent container is cleared completely
                            if (parentCovered ? isLastChild : (_.isNumber(startOffset) || _.isNumber(endOffset))) {

                                // 'deleteText' operation needs valid start and end position
                                startOffset = _.isNumber(startOffset) ? startOffset : 0;
                                endOffset = _.isNumber(endOffset) ? endOffset : (Position.getParagraphLength(rootNode, position) - 1);

                                // delete the covered part of the paragraph
                                if (isCellSelection) {
                                    // TODO
                                } else if (paragraphSelected || ((startOffset === 0) && (endOffset === 0) && (Position.getParagraphLength(rootNode, position) === 0))) {
                                    if (changeTrack.isActiveChangeTracking()) {
                                        if (!DOM.isImplicitParagraphNode(node)) { // Checking if this is an implicit paragraph
                                            newOperation = { start: position, attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } } };
                                            extendPropertiesWithTarget(newOperation, target);
                                            generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                                        }
                                    }
                                } else if (startOffset <= endOffset) {
                                    // Will be handled during iteration over all paragraph children
                                }
                            } else {
                                newOperation = { start: position, attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } } };
                                extendPropertiesWithTarget(newOperation, target);
                                generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                            }

                            // if change tracking is active, three different cases need to be disinguished:
                            // - the node is already marked for removal -> no operation needs to be created
                            // - the node is marked for insertion of new text -> text can be removed with delete operation
                            // - if this is a text span without change track attribute -> a setAttributes operation is created
                            // - other content is simply removed, no change tracking implemented yet

                            // iterating over all nodes of the paragraph is required
                            Position.iterateParagraphChildNodes(node, function (subnode, nodestart, nodelength, offsetstart, offsetlength) {

                                var // whether a delete operation needs to be created
                                    createDeleteOperation = true,
                                    // whether a delete operation needs to be created
                                    createSetAttributesOperation = false,
                                    // whether the node is a change track insert node
                                    isChangeTrackInsertNode = false;

                                if (DOM.isTextSpan(subnode) || DOM.isTextComponentNode(subnode) || DrawingFrame.isDrawingFrame(subnode)) {
                                    if (changeTrack.isInsertNodeByCurrentAuthor(subnode)) {
                                        // the content was inserted during active change track
                                        createDeleteOperation = true;
                                        createSetAttributesOperation = false;
                                        isChangeTrackInsertNode = true;
                                    } else if (DOM.isChangeTrackRemoveNode(subnode)) {
                                        // the content was already removed during active change track
                                        createDeleteOperation = false;
                                        createSetAttributesOperation = false;
                                    } else {
                                        // a set attributes operation is required for text spans without change track attribute
                                        createDeleteOperation = false;
                                        createSetAttributesOperation = true;
                                    }
                                }

                                if (createSetAttributesOperation) {
                                    newOperation = { start: position.concat([nodestart + offsetstart]), end: position.concat([nodestart + offsetstart + offsetlength - 1]), attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } } };
                                    extendPropertiesWithTarget(newOperation, target);
                                    generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                                }

                                if (createDeleteOperation) {
                                    newOperation = { start: position.concat([nodestart + offsetstart]), end: position.concat([nodestart + offsetstart + offsetlength - 1]) };
                                    extendPropertiesWithTarget(newOperation, target);
                                    generator.generateOperation(Operations.DELETE, newOperation);

                                    // the 'inserted' is no longer valid at a completely removed node
                                    // Info: The order is reversed -> 1. step: Setting attributes, 2. step: Removing all content
                                    if (isChangeTrackInsertNode && (offsetstart === 0) && (offsetlength === nodelength)) {
                                        newOperation = { start: position.concat([nodestart]), end: position.concat([nodestart + offsetlength - 1]), attrs: { changes: { mode: null }} };
                                        extendPropertiesWithTarget(newOperation, target);
                                        generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                                    }

                                }

                            }, undefined, { start: startOffset, end: endOffset });

                        } else {  // no change tracking active

                            if (!DOM.isImplicitParagraphNode(node)) {

                                // do not delete the paragraph node, if it is only covered partially;
                                // or if it is the last paragraph when the parent container is cleared completely
                                if (parentCovered ? isLastChild : (_.isNumber(startOffset) || _.isNumber(endOffset))) {

                                    // 'deleteText' operation needs valid start and end position
                                    startOffset = _.isNumber(startOffset) ? startOffset : 0;
                                    endOffset = _.isNumber(endOffset) ? endOffset : (Position.getParagraphLength(rootNode, position) - 1);

                                    // delete the covered part of the paragraph
                                    if (isCellSelection) {
                                        if (containsUnrestorableElements(node)) { askUser = true; }
                                        if (isRangeCheckRequired) { allRangeMarkerNodes = allRangeMarkerNodes.add(Position.collectElementsBySelector(node, allRangeMarkerSelector)); }
                                        newOperation = { start: position };
                                        extendPropertiesWithTarget(newOperation, target);
                                        generator.generateOperation(Operations.DELETE, newOperation);
                                    } else if (paragraphSelected || ((startOffset === 0) && (endOffset === 0) && (Position.getParagraphLength(rootNode, position) === 0))) {  // -> do not delete from [1,0] to [1,0]
                                        if (isRangeCheckRequired && paragraphSelected) { allRangeMarkerNodes = allRangeMarkerNodes.add(Position.collectElementsBySelector(node, allRangeMarkerSelector)); }
                                        newOperation = { start: position };
                                        extendPropertiesWithTarget(newOperation, target);
                                        generator.generateOperation(Operations.DELETE, newOperation);
                                        handleTriggeringListUpdate(node);
                                    } else if (startOffset <= endOffset) {
                                        if (isFirstNode) {
                                            firstParaRemovedPartly = true;
                                        } else {
                                            lastParaRemovedPartly = true;
                                        }
                                        if (containsUnrestorableElements(node, startOffset, endOffset)) { askUser = true; }
                                        if (isRangeCheckRequired) { allRangeMarkerNodes = allRangeMarkerNodes.add(Position.collectElementsBySelector(node, allRangeMarkerSelector, startOffset, endOffset)); }
                                        newOperation = { start: position.concat([startOffset]), end: position.concat([endOffset]) };
                                        extendPropertiesWithTarget(newOperation, target);
                                        generator.generateOperation(Operations.DELETE, newOperation);
                                    }
                                } else {
                                    if (containsUnrestorableElements(node)) { askUser = true; }
                                    if (isRangeCheckRequired) { allRangeMarkerNodes = allRangeMarkerNodes.add(Position.collectElementsBySelector(node, allRangeMarkerSelector)); }
                                    newOperation = { start: position };
                                    extendPropertiesWithTarget(newOperation, target);
                                    generator.generateOperation(Operations.DELETE, newOperation);
                                }
                            }
                        }

                    } else if (DOM.isTableNode(node)) {

                        allRows = DOM.getTableRows(node);

                        if (changeTrack.isActiveChangeTracking() && !changeTrack.isInsertNodeByCurrentAuthor(node) && !changeTrack.allAreInsertedNodesByCurrentAuthor(allRows)) {
                            // Table is not (necessarily) marked with 'inserted', but all rows
                            // -> setting changes attribute at all rows, not at the table
                            _.each(allRows, function (row, index) {
                                var rowPosition = _.clone(position);
                                rowPosition.push(index);
                                newOperation = { start: rowPosition, attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } } };
                                extendPropertiesWithTarget(newOperation, target);
                                generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                            });
                        } else {
                            // delete entire table
                            newOperation = { start: position };
                            extendPropertiesWithTarget(newOperation, target);
                            generator.generateOperation(Operations.DELETE, newOperation);
                            // checking, if this is a table with exceeded size
                            if (DOM.isExceededSizeTableNode(node) || containsUnrestorableElements(node)) { askUser = true; }
                            if (isRangeCheckRequired) { allRangeMarkerNodes = allRangeMarkerNodes.add(Position.collectElementsBySelector(node, allRangeMarkerSelector)); }
                            // important for the new position: Although change track might be activated, the content was really removed
                            changeTrackDeletedContent = true;
                        }
                    } else {
                        Utils.error('Editor.deleteSelected(): unsupported content node');
                        return Utils.BREAK;
                    }

                }, this, { shortestPath: true, startPos: localStartPos, endPos: localEndPos });

            }  // end of generateAllDeleteOperations

            // helper function to resort all delete operations and optionally add further delete operations
            // for example for range markers
            function resortDeleteOperations() {

                // operations MUST be executed in reverse order to preserve the positions
                generator.reverseOperations();

                // Merging paragraphs additionally is required, if:
                // - firstNode and lastNode of selection are (different) paragraphs
                // - firstNode and lastNode are not removed completely
                // - firstNode and lastNode have the same parent
                //
                // In Firefox tables there is a cell selection. In this case all paragraphs
                // are deleted completely, so that no merge is required.
                // Merging is also not allowed, if change tracking is active.

                doMergeParagraph = firstParaPosition && firstParaRemovedPartly && lastParaRemovedPartly && (firstNode.parentNode === lastNode.parentNode);

                // it is additionally necessary to merge paragraphs, if the selection ends at position 0 inside a paragraph.
                // In this case the 'end-paragraph' will not be iterated in 'selection.iterateContentNodes'. So this need
                // to be checked manually.
                if (!doMergeParagraph && checkMergeOfLastParagraph && (_.last(selection.getEndPosition()) === 0)) {
                    // there is another chance to merge paragraph, if mouse selection goes to start of following paragraph
                    // using currentParaPosition and end of selection and lastNode
                    // Additionally the first paragraph and the selectionEndParagraph need to have the same parent
                    selectionEndParagraph = Position.getDOMPosition(rootNode, _.initial(selection.getEndPosition())).node;
                    if (currentNode && currentNode.nextSibling && selectionEndParagraph && (currentNode.nextSibling === selectionEndParagraph) && (firstNode.parentNode === selectionEndParagraph.parentNode)) {
                        doMergeParagraph = true;
                    }
                }

                if (doMergeParagraph) {
                    if (changeTrack.isActiveChangeTracking()) {
                        newOperation = { start: firstParaPosition, attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } } };
                        extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                    } else {
                        newOperation = { start: firstParaPosition };
                        extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.PARA_MERGE, newOperation);
                    }
                }

                // handling for all collected ranges (if content is removed before the current start of the selection
                // (for example range start nodes), if is necessary to know the new position of the current start
                // position.
                if (allRangeMarkerNodes.length > 0) {
                    newStartPosition = rangeMarker.handlePartlyDeletedRanges(allRangeMarkerNodes, generator, selection.getStartPosition());
                }

            }  // end of resortDeleteOperations

            // helper function for asking the user, if unrestorable content shall be deleted
            function askUserHandler() {

                // always reset user value, if there is unrestorable content
                userAcceptedMissingUndo = false;

                // Asking the user, if he really wants to remove the content, if it contains
                // a table with exceeded size or other unrestorable elements. This cannot be undone.
                if (askUser) {
                    askUserPromise = showDeleteWarningDialog(
                        gt('Delete Contents'),
                        gt('Deleting the selected elements cannot be undone. Do you want to continue?')
                    );
                    // return focus to editor after 'No' button
                    askUserPromise.then(function () {
                        userAcceptedMissingUndo = true;
                    }, function () {
                        if (asyncDelete) {
                            // user abort during visible dialog -> no undo required -> leaving busy mode immediately
                            leaveAsyncBusy();
                        }
                        app.getView().grabFocus();
                    });
                } else {
                    askUserPromise = $.when();
                }

                return askUserPromise;
            }

            if (!selection.hasRange()) { return $.when(); }

            // Search not only for range markers, but also for comments, if required
            if (allRangeMarkerSelector && !commentLayer.isEmpty()) { allRangeMarkerSelector += ', ' + DOM.COMMENTPLACEHOLDER_NODE_SELECTOR; }

            // checking selection size -> make array with splitted selection [0,100], [101, 200], ... (39061)
            splittedSelectionRange = Position.splitLargeSelection(self.getCurrentRootNode(), selection.getStartPosition(), selection.getEndPosition(), MAX_TOP_LEVEL_NODES);

            if (splittedSelectionRange && splittedSelectionRange.length > 1) {

                asyncDelete = true;

                // make sure that only one asynchronous call is processed at the same time
                if (!alreadyPasteInProgress && checkSetClipboardPasteInProgress()) { return $.when(); }

                // blocking keyboard input during generation and applying of operations
                self.setBlockKeyboardEvent(true);

                // creating a snapshot (but not, if it was already created by the caller of this function)
                if (!snapshot) {
                    snapshot = new Snapshot(app);
                    snapshotCreated = true;
                }

                // show a message with cancel button
                // -> immediately grabbing the focus, after calling enterBusy. This guarantees, that the
                // keyboard blocker works. Otherwise the keyboard events will not be catched by the page.
                app.getView().enterBusy({
                    cancelHandler: function () {
                        userAbort = true;  // user aborted the process
                        if (operationsPromise && operationsPromise.abort) { // order is important, the latter has to win
                            // restoring the old document state
                            snapshot.apply();
                            // calling abort function for operation promise
                            app.enterBlockOperationsMode(function () { operationsPromise.abort(); });
                        } else if (operationGeneratorPromise && operationGeneratorPromise.abort) {
                            // cancel during creation of operation -> no undo required
                            operationGeneratorPromise.abort();
                        }
                    },
                    immediate: true,
                    warningLabel: gt('Sorry, deleting content will take some time.')
                }).grabFocus();

                // generate operations asynchronously
                operationGeneratorPromise = self.iterateArraySliced(splittedSelectionRange, function (oneRange) {
                    // calling function to generate operations synchronously with reduced selection range
                    generateAllDeleteOperations(oneRange);
                }, { delay: 'immediate', infoString: 'Text: generateAllDeleteOperations' })
                    .then(function () {

                        // synchronous resorting of delete operations
                        resortDeleteOperations();

                        // Asking the user, if he really wants to remove the content, if it contains
                        // a table with exceeded size or other unrestorable elements. This cannot be undone.
                        // return askUserHandler();
                    }, function () {
                        // user abort during creation of operation -> no undo required
                        // -> leaving busy mode immediately
                        leaveAsyncBusy();
                    })
                    // add progress handling
                    .progress(function (progress) {
                        // update the progress bar according to progress of the operations promise
                        app.getView().updateBusyProgress(operationRatio * progress);
                    });

            } else {

                // synchronous handling for small selections
                asyncDelete = false;

                // calling function to generate operations synchronously
                generateAllDeleteOperations();

                // collecting further delete operations
                resortDeleteOperations();

                // in synchronous process the promise for operation generation is already resolved
                operationGeneratorPromise = $.when();
            }

            // delete contents on resolved promise
            return operationGeneratorPromise.then(askUserHandler).then(doDeleteSelected).always(function () {
                if (snapshotCreated) { snapshot.destroy(); }
            });
        };

        /**
         * Editor API function to generate 'delete' operations.
         * This is a generic function, that can be used to delete any component (text, paragraph,
         * cell, row, table, ...). Deleting columns is not supported, because columns cannot
         * be described with a logical position.
         * The parameter 'start' and 'end' are used to specify the position of the components that
         * shall be deleted. For all components except 'text' the 'end' position will be ignored.
         * For paragraphs, cells, ... only one specific component can be deleted within this
         * operation. Only on text level a complete range can be deleted.
         *
         * @param {Number[]} start
         *  The logical start position.
         *
         * @param {Number[]} [end]
         *  The logical end position (optional). This can be different from 'start' only for text ranges
         *  inside one paragraph. A text range can include characters, fields, and drawing objects,
         *  but must be contained in a single paragraph.
         *
         * @param {Object} [options]
         *  Additional optional options, that can be used for performance reasons.
         *  @param {Boolean} [options.setTextSelection=true]
         *      If set to false, the text selection will not be set within this
         *      function. This is useful, if the caller takes care of setting the
         *      cursor after the operation. This is the case for 'Backspace' and
         *      'Delete' operations. It is a performance issue, that the text
         *      selection is not set twice.
         *  @param {Boolean} [options.handleUnrestorableContent=false]
         *      If set to true, it will be checked if the range, that shall be
         *      deleted, contains unrestorable content. In this case a dialog appears,
         *      in which the user is asked, if he really wants to delete this content.
         *      This is the case for 'Backspace' and 'Delete' operations.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  dialog has been closed with the default action; or rejected, if the
         *  dialog has been canceled. If no dialog is shown, the deferred is
         *  resolved immediately.
         */
        this.deleteRange = function (start, end, options) {
            var // a helper dom position element
                domPos = null,
                // whether the cursor shall be set in this function call
                setTextSelection = Utils.getBooleanOption(options, 'setTextSelection', true),
                // whether start and end position are different
                handleUnrestorableContent = Utils.getBooleanOption(options, 'handleUnrestorableContent', false),
                // whether start and end logical position are different
                isSimplePosition = !_.isArray(end) || _.isEqual(start, end),
                // a paragraph node
                paragraph = null,
                // whether it is necessary to ask the user before deleting range
                askUser = false,
                // the resulting promise
                promise = null,
                // target ID of currently active root node, if existing
                target = self.getActiveTarget(),
                // currently active root node
                activeRootNode = self.getCurrentRootNode(target),
                // created operation
                newOperation = null;

            // helper function, that is necessary because the user might be asked,
            // if he really wants to delete the range.
            function doDeleteRange() {

                var // whether a delete operation needs to be created
                    createDeleteOperation = true,
                    // whether a delete operation needs to be created
                    createSetAttributesOperation = false,
                    // attribute object for changes attribute
                    attrs,
                    // the dom point at text level position
                    element;

                // special handling for active change tracking
                if (changeTrack.isActiveChangeTracking()) {

                    // TODO: Also checking the end position?!
                    // -> not required, if this function is only called from deleting with 'backspace' or 'delete',
                    // that call first of all 'deleteSelected'.
                    // -> otherwise it might happen during active change tracking, that more than one operation
                    // needs to be generated.
                    element = Position.getDOMPosition(paragraph, [_.last(start)], true);

                    if (element && element.node && (DOM.isTextSpan(element.node) || DOM.isInlineComponentNode(element.node))) {
                        if (changeTrack.isInsertNodeByCurrentAuthor(element.node)) {
                            // the content was inserted during active change track (by the same user)
                            createDeleteOperation = true;
                            createSetAttributesOperation = false;
                        } else if (DOM.isChangeTrackRemoveNode(element.node)) {
                            // the content was already removed during active change track
                            createDeleteOperation = false;
                            createSetAttributesOperation = false;
                        } else {
                            // a set attributes operation is required for text spans without change track attribute
                            // or for spans inserted by another user
                            createDeleteOperation = false;
                            createSetAttributesOperation = true;
                        }
                    }

                    if (createSetAttributesOperation) {
                        createDeleteOperation = false;
                        // adding changes attributes
                        attrs = { changes: { removed: changeTrack.getChangeTrackInfo() } };
                        newOperation = { name: Operations.SET_ATTRIBUTES, start: _.clone(start), end: _.clone(end), attrs: attrs };
                        extendPropertiesWithTarget(newOperation, target);
                        self.applyOperations(newOperation);
                    }

                    // Setting the cursor to the correct position, if no character was deleted
                    if (!createDeleteOperation) {
                        lastOperationEnd = _.clone(end);
                        lastOperationEnd[lastOperationEnd.length - 1] += 1;
                    }

                }

                if (createDeleteOperation) {
                    // Using end as it is, not subtracting '1' like in 'deleteText'
                    guiTriggeredOperation = true;  // Fix for 30597
                    newOperation = { name: Operations.DELETE, start: _.clone(start), end: _.clone(end) };
                    extendPropertiesWithTarget(newOperation, target);
                    self.applyOperations(newOperation);
                    guiTriggeredOperation = false;

                    // Setting paragraph attributes immediately, if paragraph is without content.
                    // This avoids jumping of following paragraphs.
                    if ((_.last(start) === 0) && (_.last(end) === 0)) {
                        domPos = Position.getDOMPosition(activeRootNode, _.chain(start).clone().initial().value(), true);
                        if ((domPos) && (domPos.node) && ($(domPos.node).is(DOM.PARAGRAPH_NODE_SELECTOR)) && ($(domPos.node).text().length === 0)) {
                            validateParagraphNode(domPos.node);
                        }
                    }

                    // setting the cursor position
                    if (setTextSelection) {
                        selection.setTextSelection(lastOperationEnd);
                    }
                }
            }

            end = end || _.clone(start);

            // checking, if an unrestorable
            if (handleUnrestorableContent) {
                paragraph = Position.getParagraphElement(activeRootNode, _.initial(start));
                if (paragraph) {
                    if (isSimplePosition && containsUnrestorableElements(paragraph, _.last(start), _.last(start))) {
                        askUser = true;
                    } else {
                        // if node contains special page number fields in header and footer restore original state before delete
                        fieldManager.checkRestoringSpecialFields(paragraph, _.last(start), _.last(end));
                    }
                }
            }

            // Asking the user, if he really wants to remove the content, if it contains
            // a table with exceeded size or other unrestorable elements. This cannot be undone.
            if (askUser) {
                promise = showDeleteWarningDialog(
                    gt('Delete Contents'),
                    gt('Deleting the selected elements cannot be undone. Do you want to continue?')
                );
                // return focus to editor after 'No' button
                promise.fail(function () { app.getView().grabFocus(); });
            } else {
                promise = $.when();
            }

            // delete contents on resolved promise
            return promise.done(doDeleteRange);
        };

        /**
         * Editor API function to generate 'delete' operation for the complete table.
         * This method use the 'deleteRows' method with a predefined start and end index.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the dialog has been closed with
         *  the default action; or rejected, if the dialog has been canceled. If no dialog is shown,
         *  the deferred is resolved immediately.
         */
        this.deleteTable = function () {
            var // the start position of the selection
                position = selection.getStartPosition(),
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // current element container node
                rootNode = self.getCurrentRootNode(target),
                // the index of the row inside the table, in which the selection ends
                endIndex = Position.getLastRowIndexInTable(rootNode, position);

            return self.deleteRows({ start: 0, end: endIndex });
        };

        /**
         * Editor API function to generate 'delete' operations for rows or  a complete table. This
         * function is triggered by an user event. It evaluates the current selection. If the
         * selection includes all rows of the table, the table is removed completely within one
         * operation. If the row or table contains unrestorable content, the user is asked, whether
         * he really wants to delete the rows.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the dialog has been closed with
         *  the default action; or rejected, if the dialog has been canceled. If no dialog is shown,
         *  the deferred is resolved immediately.
         */
        this.deleteRows = function (options) {

            var // the operations generator
                generator = this.createOperationsGenerator(),
                // the start position of the selection
                position = selection.getStartPosition(),
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // current element container node
                rootNode = self.getCurrentRootNode(target),
                // the index of the row inside the table, in which the selection starts
                start = Utils.getOption(options, 'start', Position.getRowIndexInTable(rootNode, position)),
                // the index of the row inside the table, in which the selection ends
                end = Utils.getOption(options, 'end', selection.hasRange() ? Position.getRowIndexInTable(rootNode, selection.getEndPosition()) : start),
                // the logical position of the table
                tablePos = Position.getLastPositionFromPositionByNodeName(rootNode, position, DOM.TABLE_NODE_SELECTOR),
                // the index of the last row in the table
                lastRow = Position.getLastRowIndexInTable(rootNode, position),
                // whether the complete table shall be deleted
                isCompleteTable = ((start === 0) && (end === lastRow)) ? true : false,
                // logical row position
                rowPosition = null,
                // the table node containing the row
                tableNode = null,
                // the row nodes to be deleted
                rowNode = null,
                // the collector of all table rows
                allRows = null,
                // loop counter
                i = 0,
                // whether it is necessary to ask the user before removing row or table
                askUser = false,
                // the resulting promise
                promise = null,
                // whether it is required to check for range markers inside the selection
                isRangeCheckRequired = !selection.isAllSelected() && !rangeMarker.isEmpty(),
                // all classes used to identify the range markers
                allRangeMarkerSelector = isRangeCheckRequired ? DOM.RANGEMARKERNODE_SELECTOR : null,
                // the collector for all range marker nodes (including comments)
                allRangeMarkerNodes = $(),
                // created operation
                newOperation = null;

            // helper function, that is necessary because the user might be asked,
            // if he really wants to delete the selected rows.
            function doDeleteRows() {

                guiTriggeredOperation = true;
                self.applyOperations(generator);
                guiTriggeredOperation = false;

                // setting the cursor position after deleting content
                selection.setTextSelection(lastOperationEnd);
            }

            // helper function to determine a valid new start position for range markers. This is necessary, if
            // from an existing range, only the start marker is removed, but not the end marker.
            function getValidInsertStartRangePosition () {

                var // the logical position for inserting range markers
                    insertStartRangePosition = null,
                    // whether the last row was deleted
                    lastRowDeleted = (end === lastRow);

                if (!lastRowDeleted) {
                    // a following remaining row comes to the current cursor position, so that it is still valid
                    insertStartRangePosition = selection.getStartPosition();
                    insertStartRangePosition[insertStartRangePosition.length - 1] = 0;
                } else {
                    // Find the first valid position behind the table
                    insertStartRangePosition = Position.getFirstPositionInParagraph(self.getCurrentRootNode(), Position.increaseLastIndex(tablePos));
                    if (isCompleteTable) {
                        // because the table is removed completely, the position needs to be reduced by 1
                        insertStartRangePosition[insertStartRangePosition.length - 2] -= 1;
                    }
                }

                return insertStartRangePosition;
            }

            // Search not only for range markers, but also for comments, if required
            if (allRangeMarkerSelector && !commentLayer.isEmpty()) { allRangeMarkerSelector += ', ' + DOM.COMMENTPLACEHOLDER_NODE_SELECTOR; }

            // Generating only one operation, if the complete table is removed.
            // Otherwise sending one operation for removing each row.
            if (isCompleteTable) {
                tableNode = Position.getTableElement(rootNode, tablePos);
                allRows = DOM.getTableRows(tableNode);
                // not setting attributes, if this table was inserted with change tracking by the current user
                if (changeTrack.isActiveChangeTracking() && !changeTrack.isInsertNodeByCurrentAuthor(tableNode) && !changeTrack.allAreInsertedNodesByCurrentAuthor(allRows)) {
                    // Table is not (necessarily) marked with 'inserted', but all rows
                    // -> setting changes attribute at all rows, not at the table
                    _.each(allRows, function (row, index) {
                        rowPosition = _.clone(tablePos);
                        rowPosition.push(index);
                        newOperation = { start: rowPosition, attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } } };
                        extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                    });
                } else {
                    newOperation = { start: _.copy(tablePos, true) };
                    extendPropertiesWithTarget(newOperation, target);
                    generator.generateOperation(Operations.DELETE, newOperation);
                    if (containsUnrestorableElements(tablePos)) { askUser = true; }
                    if (isRangeCheckRequired) { allRangeMarkerNodes = allRangeMarkerNodes.add(Position.collectElementsBySelector(tableNode, allRangeMarkerSelector)); }
                }
            } else {
                for (i = end; i >= start; i--) {
                    rowPosition = _.clone(tablePos);
                    rowPosition.push(i);
                    rowNode = Position.getTableRowElement(rootNode, rowPosition);
                    if (changeTrack.isActiveChangeTracking() && !changeTrack.isInsertNodeByCurrentAuthor(rowNode)) {
                        newOperation = { start: rowPosition, attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } } };
                        extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                    } else {
                        newOperation = { start: rowPosition };
                        extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.DELETE, newOperation);
                        if (containsUnrestorableElements(rowPosition)) { askUser = true; }
                        if (isRangeCheckRequired) { allRangeMarkerNodes = allRangeMarkerNodes.add(Position.collectElementsBySelector(rowNode, allRangeMarkerSelector)); }
                    }
                }
            }

            // handling for all collected ranges
            if (allRangeMarkerNodes.length > 0) {
                rangeMarker.handlePartlyDeletedRanges(allRangeMarkerNodes, generator, getValidInsertStartRangePosition());
            }

            // Asking the user, if he really wants to remove the content, if it contains
            // at least one row with exceeded size or other unrestorable elements. This cannot be undone.
            if (askUser) {
                promise = showDeleteWarningDialog(
                    gt('Delete Rows'),
                    gt('Deleting the selected rows cannot be undone. Do you want to continue?')
                );
            } else {
                promise = $.when();
            }

            // delete rows on resolved promise
            return promise.done(doDeleteRows);
        };

        this.deleteCells = function () {

            var isCellSelection = selection.getSelectionType() === 'cell',
                startPos = selection.getStartPosition(),
                endPos = selection.getEndPosition(),
                localPos = null,
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // current element container node
                rootNode = self.getCurrentRootNode(target),
                // created operation
                newOperation = null;

            startPos.pop();  // removing character position and paragraph
            startPos.pop();
            endPos.pop();
            endPos.pop();

            var startCol = startPos.pop(),
                endCol = endPos.pop(),
                startRow = startPos.pop(),
                endRow = endPos.pop(),
                tablePos = _.clone(startPos),
                operations = [];

            for (var i = endRow; i >= startRow; i--) {

                var rowPosition = Position.appendNewIndex(tablePos, i),
                    localStartCol = startCol,
                    localEndCol = endCol;

                if (!isCellSelection && (i < endRow) && (i > startCol)) {
                    // removing complete rows
                    localStartCol = 0;
                    localEndCol = Position.getLastColumnIndexInRow(rootNode, rowPosition);
                }

                for (var j = localEndCol; j >= localStartCol; j--) {
                    localPos = Position.appendNewIndex(rowPosition, j);
                    newOperation = { name: Operations.DELETE, start: localPos };
                    extendPropertiesWithTarget(newOperation, target);
                    operations.push(newOperation);
                }

                // removing empty row
                var rowNode = Position.getDOMPosition(rootNode, rowPosition).node;
                if ($(rowNode).children().length === 0) {
                    localPos = Position.appendNewIndex(tablePos, i);
                    newOperation = { name: Operations.DELETE, start: localPos };
                    extendPropertiesWithTarget(newOperation, target);
                    operations.push(newOperation);
                }

                // checking if the table is empty
                var tableNode = Position.getDOMPosition(rootNode, tablePos).node;
                if (Table.getRowCount(tableNode) === 0) {
                    newOperation = { name: Operations.DELETE, start: _.clone(tablePos) };
                    extendPropertiesWithTarget(newOperation, target);
                    operations.push(newOperation);
                }
            }

            // apply the operations (undo group is created automatically)
            this.applyOperations(operations);

            // setting the cursor position
            selection.setTextSelection(lastOperationEnd);
        };

        this.mergeCells = function () {

            var isCellSelection = selection.getSelectionType() === 'cell',
                startPos = selection.getStartPosition(),
                endPos = selection.getEndPosition(),
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // current root container node for element
                rootNode = self.getCurrentRootNode(target),
                // created operation
                newOperation = null;

            startPos.pop();  // removing character position and paragraph
            startPos.pop();
            endPos.pop();
            endPos.pop();

            var startCol = startPos.pop(),
                endCol = endPos.pop(),
                startRow = startPos.pop(),
                endRow = endPos.pop(),
                tablePos = _.clone(startPos),
                endPosition = null,
                operations = [];

            if (endCol > startCol) {

                for (var i = endRow; i >= startRow; i--) {  // merging for each row

                    var rowPosition = Position.appendNewIndex(tablePos, i);

                    var localStartCol = startCol,
                        localEndCol = endCol;

                    if (!isCellSelection && (i < endRow) && (i > startCol)) {
                        // merging complete rows
                        localStartCol = 0;
                        localEndCol = Position.getLastColumnIndexInRow(rootNode, rowPosition);
                    }

                    var count = localEndCol - localStartCol,
                        cellPosition = Position.appendNewIndex(rowPosition, localStartCol);

                    newOperation = { name: Operations.CELL_MERGE, start: cellPosition, count: count };
                    extendPropertiesWithTarget(newOperation, target);
                    operations.push(newOperation);

                    endPosition = _.clone(cellPosition);
                }

                // apply the operations (undo group is created automatically)
                this.applyOperations(operations);

                endPosition.push(0);
                endPosition.push(0);

                // setting the cursor position
                selection.setTextSelection(endPosition);
            }
        };

        this.insertCell = function () {

            var isCellSelection = selection.getSelectionType() === 'cell',
                startPos = selection.getStartPosition(),
                endPos = selection.getEndPosition(),
                count = 1,  // default, adding one cell in each row
                endPosition = null,
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // current root container node for element
                rootNode = self.getCurrentRootNode(target),
                // created operation
                newOperation = null;

            startPos.pop();  // removing character position and paragraph
            startPos.pop();
            endPos.pop();
            endPos.pop();

            var startCol = startPos.pop(),
                endCol = endPos.pop(),
                startRow = startPos.pop(),
                endRow = endPos.pop(),
                tablePos = _.copy(startPos, true),
                attrs = { cell: {} },
                operations = [];

            for (var i = endRow; i >= startRow; i--) {

                var rowPosition = Position.appendNewIndex(tablePos, i),
                    localEndCol = endCol;

                if (!isCellSelection && (i < endRow) && (i > startCol)) {
                    // removing complete rows
                    localEndCol = Position.getLastColumnIndexInRow(rootNode, rowPosition);
                }

                localEndCol++;  // adding new cell behind existing cell
                var cellPosition = Position.appendNewIndex(rowPosition, localEndCol);
                attrs.cell.gridSpan = 1;  // only 1 grid for the new cell

                newOperation = { name: Operations.CELLS_INSERT, start: cellPosition, count: count, attrs: attrs };
                extendPropertiesWithTarget(newOperation, target);
                operations.push(newOperation);

                // Applying new tableGrid, if the current tableGrid is not sufficient
                var tableDomPoint = Position.getDOMPosition(rootNode, tablePos),
                    rowDomPoint = Position.getDOMPosition(rootNode, rowPosition);

                if (tableDomPoint && DOM.isTableNode(tableDomPoint.node)) {

                    var tableGridCount = tableStyles.getElementAttributes(tableDomPoint.node).table.tableGrid.length,
                        rowGridCount = Table.getColSpanSum($(rowDomPoint.node).children());

                    if (rowGridCount > tableGridCount) {

                        localEndCol--;  // behind is evaluated in getTableGridWithNewColumn
                        var insertmode = 'behind',
                            tableGrid = Table.getTableGridWithNewColumn(rootNode, tablePos, localEndCol, insertmode);

                        // Setting new table grid attribute to table
                        newOperation = { name: Operations.SET_ATTRIBUTES, attrs: { table: { tableGrid: tableGrid } }, start: _.clone(tablePos) };
                        extendPropertiesWithTarget(newOperation, target);
                        operations.push(newOperation);
                        requiresElementFormattingUpdate = false;  // no call of implTableChanged -> attributes are already set in implSetAttributes
                    }

                }

                endPosition = _.clone(cellPosition);
            }

            guiTriggeredOperation = true;

            // apply the operations (undo group is created automatically)
            this.applyOperations(operations);

            guiTriggeredOperation = false;
            requiresElementFormattingUpdate = true;

            endPosition.push(0);
            endPosition.push(0);

            // setting the cursor position
            selection.setTextSelection(endPosition);
        };

        /**
         * Editor API function to generate 'delete' operations for columns or a complete table. This
         * function is triggered by an user event. It evaluates the current selection. If the
         * selection includes all columns of the table, the table is removed completely within one
         * operation. If the column or table contains unrestorable content, the user is asked, whether
         * he really wants to delete the columns.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the dialog has been closed with
         *  the default action; or rejected, if the dialog has been canceled. If no dialog is shown,
         *  the deferred is resolved immediately.
         */
        this.deleteColumns = function () {

            var // the operations generator
                generator = this.createOperationsGenerator(),
                // the start position of the selection
                position = selection.getStartPosition(),
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // current root container node for element
                rootNode = self.getCurrentRootNode(),
                // the logical position of the table
                tablePos = Position.getLastPositionFromPositionByNodeName(rootNode, position, DOM.TABLE_NODE_SELECTOR),
                // the table node
                tableNode = Position.getDOMPosition(rootNode, tablePos).node,
                // the maximum grid number of the table
                maxGrid = Table.getColumnCount(tableNode) - 1,
                // the row node at the logical start position of the selection
                rowNode = Position.getLastNodeFromPositionByNodeName(rootNode, position, 'tr'),
                // the index of the column at the logical start position of the selection
                startColIndex = Position.getColumnIndexInRow(rootNode, position),
                // the index of the column at the logical end position of the selection
                endColIndex = selection.hasRange() ? Position.getColumnIndexInRow(rootNode, selection.getEndPosition()) : startColIndex,
                // a helper object for calculating grid positions in the table
                returnObj = Table.getGridPositionFromCellPosition(rowNode, startColIndex),
                // the start grid position in the table
                startGrid = returnObj.start,
                // the end grid position in the table
                endGrid = selection.hasRange() ? Table.getGridPositionFromCellPosition(rowNode, endColIndex).end : returnObj.end,
                // whether the complete table shall be deleted
                isCompleteTable = ((startGrid === 0) && (endGrid === maxGrid)) ? true : false,
                // logical row position
                rowPos = null,
                // logical cell position
                cellPos = null,
                // the index of the last row in the table
                maxRow = null,
                // whether all rows will be deleted
                deletedAllRows = false,
                // a jQuery collection containing all rows of the specified table
                allRows = null,
                // An array, that contains for each row an array with two integer values
                // for start and end position of the cells.
                allCellRemovePositions = null,
                // a row node
                currentRowNode = null,
                // a cell node
                cellNode = null,
                // Two integer values for start and end position of the cells in the row.
                oneRowCellArray = null,
                // helper numbers for cell descriptions inside a row
                start = 0, end = 0,
                // the table grid of a table
                tableGrid = null,
                // loop counter
                i = 0, j = 0,
                // whether it is necessary to ask the user before removing row or table
                askUser = false,
                // the resulting promise
                promise = null,
                // whether it is necessary to set the table grid again, after removing cells
                refreshTableGrid = false,
                // whether the delete column operation is required with activated change tracking
                // This delete is required, if the column was inserted with change tracking
                deleteForChangeTrackRequired = false,
                // whether it is required to check for range markers inside the selection
                isRangeCheckRequired = !selection.isAllSelected() && !rangeMarker.isEmpty(),
                // all classes used to identify the range markers
                allRangeMarkerSelector = isRangeCheckRequired ? DOM.RANGEMARKERNODE_SELECTOR : null,
                // the collector for all range marker nodes (including comments)
                allRangeMarkerNodes = $(),
                // created operation
                newOperation = null;

            // helper function, that is necessary because the user might be asked,
            // if he really wants to delete the selected rows.
            function doDeleteColumns() {

                // apply the operations (undo group is created automatically)
                guiTriggeredOperation = true;
                self.applyOperations(generator);
                guiTriggeredOperation = false;
                requiresElementFormattingUpdate = true;

                // setting the cursor position after deleting the content
                selection.setTextSelection(lastOperationEnd);
            }

            // helper function to determine a valid new start position for range markers. This is necessary, if
            // from an existing range, only the start marker is removed, but not the end marker.
            function getValidInsertStartRangePosition () {

                var // the logical position for inserting range markers
                    insertStartRangePosition = null;

                if (isCompleteTable || deletedAllRows) {
                    // Find the first valid position behind the table
                    insertStartRangePosition = Position.getFirstPositionInParagraph(self.getCurrentRootNode(), Position.increaseLastIndex(tablePos));
                    // because the table is removed completely, the position needs to be reduced by 1
                    insertStartRangePosition[insertStartRangePosition.length - 2] -= 1;
                }

                return insertStartRangePosition;
            }

            // Search not only for range markers, but also for comments, if required
            if (allRangeMarkerSelector && !commentLayer.isEmpty()) { allRangeMarkerSelector += ', ' + DOM.COMMENTPLACEHOLDER_NODE_SELECTOR; }

            // Generating only one operation, if the complete table is removed.
            // Otherwise sending operations for removing each columns, maybe for rows and
            // for setting new table attributes.
            if (isCompleteTable) {
                allRows = DOM.getTableRows(tableNode);
                // not setting attributes, if this table was inserted with change tracking by the current user
                if (changeTrack.isActiveChangeTracking() && !changeTrack.isInsertNodeByCurrentAuthor(tableNode) && !changeTrack.allAreInsertedNodesByCurrentAuthor(allRows)) {
                    // Table is not (necessarily) marked with 'inserted', but all rows
                    // -> setting changes attribute at all rows, not at the table
                    _.each(allRows, function (row, index) {
                        var rowPosition = _.clone(tablePos);
                        rowPosition.push(index);
                        newOperation = { start: rowPosition, attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } } };
                        extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                    });
                } else {
                    newOperation = { start: _.copy(tablePos, true) };
                    extendPropertiesWithTarget(newOperation, target);
                    generator.generateOperation(Operations.DELETE, newOperation);
                    if (containsUnrestorableElements(tablePos)) { askUser = true; }
                    if (isRangeCheckRequired) { allRangeMarkerNodes = allRangeMarkerNodes.add(Position.collectElementsBySelector(tableNode, allRangeMarkerSelector)); }
                }
            } else {
                // generating delete columns operation, but further operations might be necessary
                // -> if this is necessary for change tracking, this need to be evaluated using deleteForChangeTrackRequired
                if (!changeTrack.isActiveChangeTracking()) {
                    newOperation = { start: tablePos, startGrid: startGrid, endGrid: endGrid };
                    extendPropertiesWithTarget(newOperation, target);
                    generator.generateOperation(Operations.COLUMNS_DELETE, newOperation);
                }

                // Checking, if there will be rows without cells after the columns are deleted
                maxRow = Table.getRowCount(tableNode) - 1;
                deletedAllRows = true;
                allRows = DOM.getTableRows(tableNode);
                allCellRemovePositions = Table.getAllRemovePositions(allRows, startGrid, endGrid);

                for (i = maxRow; i >= 0; i--) {
                    rowPos = _.clone(tablePos);
                    rowPos.push(i);
                    currentRowNode = Position.getDOMPosition(rootNode, rowPos).node;
                    oneRowCellArray =  allCellRemovePositions[i];
                    end = oneRowCellArray.pop();
                    start = oneRowCellArray.pop();

                    if ($(currentRowNode).children().length === (end - start + 1)) {
                        // checking row attributes
                        if (changeTrack.isActiveChangeTracking() && !changeTrack.isInsertNodeByCurrentAuthor(currentRowNode)) { // not setting attributes, if this row was inserted with change tracking
                            newOperation = { start: _.copy(rowPos, true), attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } } };
                            extendPropertiesWithTarget(newOperation, target);
                            generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                        } else {
                            newOperation = { start: rowPos };
                            extendPropertiesWithTarget(newOperation, target);
                            generator.generateOperation(Operations.DELETE, newOperation);
                            if (containsUnrestorableElements(rowPos)) { askUser = true; }
                            if (isRangeCheckRequired) { allRangeMarkerNodes = allRangeMarkerNodes.add(Position.collectElementsBySelector(currentRowNode, allRangeMarkerSelector)); }
                            refreshTableGrid = true;
                            deleteForChangeTrackRequired = true;
                        }
                    } else {
                        deletedAllRows = false;
                        // checking unrestorable content in all cells, that will be removed
                        for (j = start; j <= end; j++) {
                            cellPos = _.clone(rowPos);
                            cellPos.push(j);

                            // marking all cells with change track attribute (TODO: Also necessary to check the parent nodes?)
                            cellNode = Position.getSelectedTableCellElement(rootNode, cellPos);
                            if (changeTrack.isActiveChangeTracking() && !changeTrack.isInsertNodeByCurrentAuthor(cellNode)) { // not setting attributes, if this cell was inserted with change tracking
                                newOperation = { start: _.copy(cellPos, true), attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } } };
                                extendPropertiesWithTarget(newOperation, target);
                                generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                            } else {
                                if (containsUnrestorableElements(cellPos)) { askUser = true; }
                                if (isRangeCheckRequired) { allRangeMarkerNodes = allRangeMarkerNodes.add(Position.collectElementsBySelector(cellNode, allRangeMarkerSelector)); }
                                refreshTableGrid = true;
                                deleteForChangeTrackRequired = true;
                            }
                        }
                    }
                }

                if (changeTrack.isActiveChangeTracking() && deleteForChangeTrackRequired) {
                    newOperation = { start: tablePos, startGrid: startGrid, endGrid: endGrid };
                    extendPropertiesWithTarget(newOperation, target);
                    generator.generateOperation(Operations.COLUMNS_DELETE, newOperation);
                }

                // Deleting the table explicitely, if all its content was removed
                if (deletedAllRows) {
                    // not setting attributes, if this table was inserted with change tracking by the current user
                    if (changeTrack.isActiveChangeTracking() && !changeTrack.isInsertNodeByCurrentAuthor(tableNode) && !changeTrack.allAreInsertedNodesByCurrentAuthor(allRows)) {
                        // Table is not (necessarily) marked with 'inserted', but all rows
                        // -> setting changes attribute at all rows, not at the table
                        _.each(allRows, function (row, index) {
                            var rowPosition = _.clone(tablePos);
                            rowPosition.push(index);
                            newOperation = { start: rowPosition, attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } } };
                            extendPropertiesWithTarget(newOperation, target);
                            generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                        });
                    } else {
                        newOperation = { start: _.clone(tablePos) };
                        extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.DELETE, newOperation);
                        if (containsUnrestorableElements(tablePos)) { askUser = true; }
                        if (isRangeCheckRequired) { allRangeMarkerNodes = allRangeMarkerNodes.add(Position.collectElementsBySelector(tableNode, allRangeMarkerSelector)); }
                    }
                } else {
                    if (refreshTableGrid) {
                        // Setting new table grid attribute to table (but not, if change tracking is activated
                        tableGrid = _.clone(tableStyles.getElementAttributes(tableNode).table.tableGrid);
                        tableGrid.splice(startGrid, endGrid - startGrid + 1);  // removing column(s) in tableGrid (automatically updated in table node)
                        newOperation = { attrs: { table: { tableGrid: tableGrid } }, start: _.clone(tablePos) };
                        extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                        requiresElementFormattingUpdate = false;   // no call of implTableChanged -> attributes are already set in implSetAttributes
                    }
                }
            }

            // handling for all collected ranges
            if (allRangeMarkerNodes.length > 0) {
                rangeMarker.handlePartlyDeletedRanges(allRangeMarkerNodes, generator, getValidInsertStartRangePosition());
            }

            // Asking the user, if he really wants to remove the columns, if it contains
            // at least one cell with exceeded size or other unrestorable elements. This cannot be undone.
            if (askUser) {
                promise = showDeleteWarningDialog(
                    gt('Delete Columns'),
                    gt('Deleting the selected columns cannot be undone. Do you want to continue?')
                );
            } else {
                promise = $.when();
            }

            // delete columns on resolved promise
            return promise.done(doDeleteColumns);
        };

        this.insertRow = function () {

            if (!this.isRowAddable()) {
                app.getView().rejectEditTextAttempt('tablesizerow'); // checking table size (26809)
                return;
            }

            undoManager.enterUndoGroup(function () {

                var // inserting only one row
                    count = 1,
                    // whether default cells shall be inserted into the new row
                    insertDefaultCells = false,
                    // the current logical position of the selection
                    position = selection.getEndPosition(),
                    // target for operation - if exists, it's for ex. header or footer
                    target = self.getActiveTarget(),
                    // current root container node for element
                    rootNode = self.getCurrentRootNode(target),
                    // the row number from the logical position
                    referenceRow = null,
                    // the html dom node of the row
                    rowNode = null,
                    // the logical position of the row
                    rowPos = Position.getLastPositionFromPositionByNodeName(rootNode, position, 'tr'),
                    // the row attributes
                    rowAttrs = {},
                    // operations generator
                    generator = this.createOperationsGenerator(),
                    // created operation
                    newOperation = null;

                if (rowPos !== null) {

                    rowNode = Position.getTableRowElement(rootNode, rowPos);

                    referenceRow = _.last(rowPos);

                    rowPos[rowPos.length - 1] += 1;

                    // modifying the attributes, if changeTracking is activated
                    if (changeTrack.isActiveChangeTracking()) {
                        rowAttrs.changes = { inserted: changeTrack.getChangeTrackInfo(), removed: null };
                    } else if (DOM.isChangeTrackNode(rowNode)) {
                        // change tracking is not active, but reference row has change track attribute -> this attribute needs to be removed
                        rowAttrs.changes = { inserted: null, removed: null, modified: null };
                    }

                    newOperation = { start: rowPos, count: count, insertDefaultCells: insertDefaultCells, referenceRow: referenceRow, attrs: rowAttrs };
                    extendPropertiesWithTarget(newOperation, target);
                    generator.generateOperation(Operations.ROWS_INSERT, newOperation);

                    // cells with list in the first paragraph, should automatically receive this list information, too
                    if (rowNode) {
                        // checking the cell contents in the cloned row
                        $(rowNode).children('td').each(function () {
                            var // the container for the content nodes
                                container = DOM.getCellContentNode(this),
                                // the first paragraph in the cell determines the attributes for the new cell
                                paragraph = container[0].firstChild,
                                // the paragraph attributes
                                paraAttributes = null,
                                // the logical position of the new paragraph
                                paraPos = _.clone(rowPos);

                            if (DOM.isParagraphNode(paragraph)) {
                                paraAttributes = AttributeUtils.getExplicitAttributes(paragraph);

                                if (!_.isEmpty(paraAttributes)) {
                                    paraPos.push($(this).prevAll().length);
                                    paraPos.push(0);
                                    // generating an operation for setting paragraph attributes in the new row
                                    // -> no implicit paragraph in these cells

                                    newOperation = { start: paraPos, attrs: paraAttributes };
                                    extendPropertiesWithTarget(newOperation, target);
                                    generator.generateOperation(Operations.PARA_INSERT, newOperation);
                                }
                            }
                        });
                    }

                    guiTriggeredOperation = true;
                    // apply all collected operations
                    this.applyOperations(generator);
                    guiTriggeredOperation = false;
                }

                // setting the cursor position
                selection.setTextSelection(lastOperationEnd);

            }, this); // enterUndoGroup()
        };

        this.insertColumn = function () {

            if (!this.isColumnAddable()) {
                app.getView().rejectEditTextAttempt('tablesizecolumn'); // checking table size (26809)
                return;
            }

            undoManager.enterUndoGroup(function () {

                var position = selection.getEndPosition(),
                    // target for operation - if exists, it's for ex. header or footer
                    target = self.getActiveTarget(),
                    rootNode = self.getCurrentRootNode(target),
                    cellPosition = Position.getColumnIndexInRow(rootNode, position),
                    tablePos = Position.getLastPositionFromPositionByNodeName(rootNode, position, DOM.TABLE_NODE_SELECTOR),
                    rowNode = Position.getLastNodeFromPositionByNodeName(rootNode, position, 'tr'),
                    insertMode = 'behind',
                    gridPosition = Table.getGridPositionFromCellPosition(rowNode, cellPosition).start,
                    tableGrid = Table.getTableGridWithNewColumn(rootNode, tablePos, gridPosition, insertMode),
                    // table node element
                    table = Position.getTableElement(rootNode, tablePos),
                    // all rows in the table
                    allRows = DOM.getTableRows(table),
                    // operations generator
                    generator = this.createOperationsGenerator(),
                    // the attributes assigned to the cell(s)
                    cellAttrs = null,
                    // created operation
                    newOperation = null;

                newOperation = { start: tablePos, tableGrid: tableGrid, gridPosition: gridPosition, insertMode: insertMode };
                extendPropertiesWithTarget(newOperation, target);
                generator.generateOperation(Operations.COLUMN_INSERT, newOperation);

                // Setting new table grid attribute to table
                newOperation = { attrs: { table: { tableGrid: tableGrid } }, start: _.clone(tablePos) };
                extendPropertiesWithTarget(newOperation, target);
                generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);

                // Iterating over all rows to keep paragraph attributes of first paragraph in each new cell
                // -> these paragraphs must not be explicit
                if (allRows) {
                    allRows.each(function (i, row) {
                        var // the current logical cell position
                            cellPosition = Table.getCellPositionFromGridPosition(row, gridPosition),
                            // the current cell node
                            cellClone = $(row).children(DOM.TABLE_CELLNODE_SELECTOR).slice(cellPosition, cellPosition + 1),
                            // the container for the content nodes
                            container = null,
                            // the first paragraph in the cell determines the attributes for the new cell
                            paragraph = null,
                            // whether the current or the following cell is used to find paragraph attributes
                            currentCellUsed = true,
                            // the paragraph attributes
                            paraAttributes = null,
                            // the position of the cell inside its row
                            cellNumber = null,
                            // the logical position of the row
                            rowPos = Position.getOxoPosition(rootNode, row, 0),
                            // the logical position of the new cell
                            cellPos = _.clone(rowPos),
                            // the logical position of the new paragraph
                            paraPos = _.clone(rowPos);

                        // using the current cell only, if it is the last cell in the row. Otherwise the attributes from the
                        // following cell need to be examined.
                        if (cellClone.next().length > 0) {
                            cellClone = cellClone.next();
                            currentCellUsed = false;
                        }

                        container = DOM.getCellContentNode(cellClone);
                        paragraph = container[0].firstChild;

                        if (DOM.isParagraphNode(paragraph)) {
                            paraAttributes = AttributeUtils.getExplicitAttributes(paragraph);

                            if (!_.isEmpty(paraAttributes)) {
                                cellNumber = $(cellClone).prevAll().length;
                                if (currentCellUsed) { cellNumber++; }
                                paraPos.push(cellNumber);
                                paraPos.push(0);
                                // generating an operation for setting paragraph attributes in the new row
                                // -> no implicit paragraph in these cells
                                newOperation = { start: paraPos, attrs: paraAttributes };
                                extendPropertiesWithTarget(newOperation, target);
                                generator.generateOperation(Operations.PARA_INSERT, newOperation);
                            }
                        }

                        // modifying the cell attributes, if changeTracking is activated or if change track attributes need to be removed
                        cellAttrs = null;
                        if (changeTrack.isActiveChangeTracking()) {
                            cellAttrs = { changes: { inserted: changeTrack.getChangeTrackInfo(), removed: null } };
                        } else if (DOM.isChangeTrackNode(cellClone)) {
                            // change tracking is not active, but reference cell has change track attribute -> this attribute needs to be removed
                            cellAttrs = { changes: { inserted: null, removed: null, modified: null } };
                        }

                        if (cellAttrs !== null) {
                            cellPos = _.clone(rowPos);
                            cellNumber = cellNumber !== null ? cellNumber : $(cellClone).prevAll().length;
                            cellPos.push(cellNumber);
                            newOperation = { attrs: cellAttrs, start: _.clone(cellPos) };
                            extendPropertiesWithTarget(newOperation, target);
                            generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                        }

                    });
                }

                // no call of implTableChanged in implInsertColumn
                requiresElementFormattingUpdate = false;  // no call of implTableChanged -> attributes are already set in implSetAttributes
                guiTriggeredOperation = true;

                // apply all collected operations
                this.applyOperations(generator);

                requiresElementFormattingUpdate = true;
                guiTriggeredOperation = false;

                // setting the cursor position
                selection.setTextSelection(lastOperationEnd);

            }, this); // enterUndoGroup()
        };

        /**
         * Inserting a paragraph into the document.
         *
         * @param {Number[]} start
         *  The logical position of the paragraph to be inserted.
         */
        this.insertParagraph = function (start) {

            var // the attributes of the inserted paragraph
                attrs = {},
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // created operation
                newOperation = null;

            // handling change tracking
            if (changeTrack.isActiveChangeTracking()) { attrs.changes = { inserted: changeTrack.getChangeTrackInfo() }; }

            newOperation = { name: Operations.PARA_INSERT, start: _.clone(start), attrs: attrs };
            extendPropertiesWithTarget(newOperation, target);

            // applying operation
            this.applyOperations(newOperation);
        };

        /**
         * Inserting a table into the document.
         * The undo manager returns the return value of the callback function.
         * For inserting a table it is important, that there is always a
         * paragraph between the new table and the neighboring table. This is
         * important for merging of tables, where the removal of this paragraph
         * leads to an automatic merge.
         *
         * @param {Object} size
         *  An object containing the properties 'width' and 'height' for the
         *  column and row count of the table.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  dialog has been closed with the default action; or rejected, if the
         *  dialog has been canceled. If no dialog is shown, the deferred is
         *  resolved immediately.
         *  The undo manager returns the return value of the callback function.
         */
        this.insertTable = function (size) {

            return undoManager.enterUndoGroup(function () {

                var // cursor position used to split the paragraph
                    startPosition = null,
                    // paragraph to be split for the new table, and its position
                    paragraph = null, position = null,
                    // text offset in paragraph, first and last text position in paragraph
                    offset = 0, startOffset = 0, endOffset = 0,
                    // table attributes
                    attributes = { table: { tableGrid: [], width: 'auto' } },
                    // table row and paragraph attributes
                    rowAttributes = null, rowOperationAttrs = null, paraOperationAttrs = {},
                    // default table style
                    tableStyleId = self.getDefaultUITableStylesheet(),
                    // a new implicit paragraph
                    newParagraph = null,
                    // operations generator
                    generator = this.createOperationsGenerator(),
                    // whether an additional paragraph is required behind or before the table
                    insertAdditionalParagraph = false,
                    // the logical position of the additional paragraph
                    paraPosition = null,
                    // target for operation - if exists, it's for ex. header or footer
                    target = self.getActiveTarget(),
                    // container root node of table
                    rootNode = self.getCurrentRootNode(),
                    // properties to be passed to generator for insert table operation
                    generatorProperties = {};

                function doInsertTable() {
                    startPosition = selection.getStartPosition();
                    position = startPosition.slice(0, -1);
                    paragraph = Position.getParagraphElement(rootNode, position);
                    if (!paragraph) { return; }

                    if (!DOM.isImplicitParagraphNode(paragraph)) {
                        // split paragraph, if the cursor is between two characters,
                        // or if the paragraph is the very last in the container node
                        offset = _.last(startPosition);
                        startOffset = Position.getFirstTextNodePositionInParagraph(paragraph);
                        endOffset = Position.getLastTextNodePositionInParagraph(paragraph);

                        if ((!paragraph.nextSibling) && (offset === endOffset)) {
                            // create a new empty implicit paragraph behind the table
                            newParagraph = DOM.createImplicitParagraphNode();
                            validateParagraphNode(newParagraph);
                            $(paragraph).after(newParagraph);
                            implParagraphChanged(newParagraph);
                            position = Position.increaseLastIndex(position);
                            // in Internet Explorer it is necessary to add new empty text nodes in paragraph again
                            // newly, also in other browsers (new jQuery version?)
                            repairEmptyTextNodes(newParagraph);
                        } else if ((startOffset < offset) && (offset < endOffset)) {
                            self.splitParagraph(startPosition);
                            position = Position.increaseLastIndex(position);
                        } else if ((!paragraph.nextSibling) && (offset > 0)) {
                            self.splitParagraph(startPosition);
                            position = Position.increaseLastIndex(position);
                        } else if (offset === endOffset) {
                            // cursor at the end of the paragraph: insert before next content node
                            position = Position.increaseLastIndex(position);
                            // additionally adding an empty paragraph behind the table, if the following
                            // node is a table. Therefore there is always one paragraph between two tables.
                            // If this paragraph is removed, this leads to a merge of the tables (if this
                            // is possible).
                            if (paragraph.nextSibling && DOM.isTableNode(paragraph.nextSibling)) {
                                insertAdditionalParagraph = true;
                                paraPosition = _.clone(position);
                                paraPosition = Position.increaseLastIndex(paraPosition);
                            }
                        }
                    } else {
                        // if this is an explicit paragraph and the previous node is a table,
                        // there must be a new paragraph before the new table.
                        if (paragraph.previousSibling && DOM.isTableNode(paragraph.previousSibling)) {
                            insertAdditionalParagraph = true;
                            paraPosition = _.clone(position);
                        }
                    }

                    // prepare table column widths (values are relative to each other)
                    _(size.width).times(function () { attributes.table.tableGrid.push(1000); });

                    // set default table style
                    if (_.isString(tableStyleId)) {

                        // insert a pending table style if needed
                        Table.checkForLateralTableStyle(generator, self, tableStyleId);

                        // add table style name to attributes
                        attributes.styleId = tableStyleId;

                        // default: tables do not have last row, last column and vertical bands
                        attributes.table.exclude = ['lastRow', 'lastCol', 'bandsVert'];
                    }

                    // modifying the attributes, if changeTracking is activated
                    if (changeTrack.isActiveChangeTracking()) {
                        attributes.changes = { inserted: changeTrack.getChangeTrackInfo() };
                        rowAttributes = { changes: { inserted: changeTrack.getChangeTrackInfo() }};
                    }

                    // insert the table, and add empty rows

                    generatorProperties = { start: _.clone(position), attrs: attributes };
                    extendPropertiesWithTarget(generatorProperties, target);
                    generator.generateOperation(Operations.TABLE_INSERT, generatorProperties);

                    rowOperationAttrs = { start: Position.appendNewIndex(position, 0), count: size.height, insertDefaultCells: true };
                    extendPropertiesWithTarget(rowOperationAttrs, target);
                    if (rowAttributes) { rowOperationAttrs.attrs = rowAttributes; }
                    generator.generateOperation(Operations.ROWS_INSERT, rowOperationAttrs);

                    // also adding a paragraph behind the table, if there is a following table
                    if (insertAdditionalParagraph) {
                        // modifying the attributes, if changeTracking is activated
                        if (changeTrack.isActiveChangeTracking()) {
                            paraOperationAttrs.changes = { inserted: changeTrack.getChangeTrackInfo() };
                        }
                        generatorProperties = { start: paraPosition, attrs: paraOperationAttrs };
                        extendPropertiesWithTarget(generatorProperties, target);
                        generator.generateOperation(Operations.PARA_INSERT, generatorProperties);
                    }

                    // apply all collected operations
                    self.applyOperations(generator);

                    // set the cursor to first paragraph in first table cell
                    selection.setTextSelection(position.concat([0, 0, 0, 0]));
                }

                if (selection.hasRange()) {
                    return self.deleteSelected()
                    .done(function () {
                        doInsertTable();
                    });
                }

                doInsertTable();
                return $.when();

            }, this); // enterUndoGroup()

        };

        /**
         * Inserting a drawing group node into the document. This group can be
         * used as container for drawing frames
         * TODO: This code cannot be triggered via GUI yet. It is currently ony
         * available for testing reasons.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the dialog
         *  asking the user, if he wants to delete the selected content (if any)
         *  has been closed with the default action; or rejected, if the dialog
         *  has been canceled. If no dialog is shown, the deferred is  resolved
         *  immediately. The undo manager returns the return value of the callback
         *  function.
         */
        this.insertDrawingGroup = function () {

            var // whether the text frame is included into a drawing group node
                insertIntoGroup = false,
                // the position of the text frame inside a group node
                groupPosition = 0,
                // helper nodes
                startNodeInfo = null, contentNode = null;

            return undoManager.enterUndoGroup(function () {

                // helper function, that generates the operations
                function doInsertDrawingGroup () {

                    var // the operations generator
                        generator = self.createOperationsGenerator(),
                        start = selection.getStartPosition(),
                        attrs = { drawing: { width: 6000, height: 3000 } };

                    // special code for inserting into group
                    if (insertIntoGroup) {
                        start.push(groupPosition);  // adding the position inside the group -> TODO: Remove asap
                    } else {
                        handleImplicitParagraph(start);
                    }

                    // modifying the attributes, if changeTracking is activated
                    if (changeTrack.isActiveChangeTracking()) {
                        attrs.changes = { inserted: changeTrack.getChangeTrackInfo(), removed: null };
                    }

                    // text frames are drawings of type shape
                    generator.generateOperation(Operations.DRAWING_INSERT, { attrs: attrs, start: start, type: 'group' });

                    // apply all collected operations
                    self.applyOperations(generator);
                }

                // -> integrating a text frame into a drawing frame of type 'group' by selecting the drawing group
                if (selection.hasRange() && selection.isDrawingFrameSelection()) {
                    startNodeInfo = Position.getDOMPosition(editdiv, selection.getStartPosition(), true);
                    if (startNodeInfo && startNodeInfo.node && DrawingFrame.isGroupDrawingFrame(startNodeInfo.node)) {
                        contentNode = $(startNodeInfo.node).children().first();
                        // checking the number of children inside the content node
                        groupPosition = DrawingFrame.isPlaceHolderNode(contentNode) ? 0 : contentNode.children().length;
                        insertIntoGroup = true;
                    }
                }

                if (!insertIntoGroup && selection.hasRange()) {
                    return self.deleteSelected()
                    .done(function () {
                        doInsertDrawingGroup();
                    });
                }

                doInsertDrawingGroup();
                return $.when();

            }, this);  // enterUndoGroup()
        };

        /**
         * Handling the property 'autoResizeHeight' of a selected text frame node.
         *
         * @param {Boolean} state
         *  Whether the property 'autoResizeHeight' of a selected text frame shall be
         *  enabled or disabled.
         */
        this.handleTextFrameAutoFit = function (state) {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // the options for the setAttributes operation
                operationOptions = {},
                // a selected text frame node or the text frame containing the selection
                textFrame = selection.getAnyTextFrameDrawing({ forceTextFrame: true });

            // collecting the attributes for the operation
            operationOptions.attrs =  {};
            operationOptions.attrs.shape = { autoResizeHeight: state };

            // if auto fit is disabled, the current height must be set explicitely
            if (!state) {
                if (textFrame && textFrame.length > 0) {
                    operationOptions.attrs.drawing = { height: Utils.convertLengthToHmm(textFrame.height(), 'px') };
                }
            }

            if (selection.isAdditionalTextframeSelection()) {
                operationOptions.start = Position.getOxoPosition(self.getCurrentRootNode(), textFrame, 0);
            } else {
                operationOptions.start = selection.getStartPosition();
            }

            // generate the 'setAttributes' operation
            generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);

            // apply all collected operations
            this.applyOperations(generator);
        };

        /**
         * Checking whether a currently selected text frame node has 'autoResizeHeight' enabled or not.
         *
         * @returns {Boolean}
         *  Whether a a currently selected text frame node has 'autoResizeHeight' enabled. If no text frame
         *  is selected, false is returned.
         */
        this.isAutoResizableTextFrame = function () {

            var // the selected text frame node (also finding text frames inside groups)
                textFrame = selection.getAnyTextFrameDrawing({ forceTextFrame: true });

            return textFrame && textFrame.length > 0 && DrawingFrame.isAutoResizeHeightDrawingFrame(textFrame) || false;
        };

        /**
         * Inserting a text frame drawing node of type 'shape' into the document.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the dialog
         *  asking the user, if he wants to delete the selected content (if any)
         *  has been closed with the default action; or rejected, if the dialog
         *  has been canceled. If no dialog is shown, the deferred is  resolved
         *  immediately. The undo manager returns the return value of the callback
         *  function.
         */
        this.insertTextFrame = function () {

            return undoManager.enterUndoGroup(function () {

                // helper function, that generates the operations
                function doInsertTextFrame () {

                    var // the operations generator
                        generator = self.createOperationsGenerator(),
                        // the default width of the textframe in hmm
                        defaultWidth = 5000,
                        // the width of the parent paragraph
                        paraWidth = 0,
                        // the current cursor position
                        start = selection.getStartPosition(),
                        // the default border attributes
                        lineAttrs = { color: { type: 'rgb', value: '000000' }, style: 'single', type: 'solid', width: Border.getWidthForPreset('thin') },
                        // the default fill color attributes
                        fillAttrs = { color: { type: 'auto' }, type: 'solid' },
                        // the default attributes
                        attrs = { drawing: { width: defaultWidth }, shape: { autoResizeHeight: true }, line: lineAttrs, fill: fillAttrs },
                        // the position of the first paragraph inside the text frame
                        paraPos = _.clone(start),
                        // a required style ID for style handling
                        parentStyleId = null,
                        // the paragraph node, in which the text frame will be inserted
                        paraNode = null,
                        // target for operation - if exists, it's for ex. header or footer
                        target = self.getActiveTarget(),
                        // created operation
                        newOperation = null;

                    handleImplicitParagraph(start);

                    // checking the width of the parent paragraph (36836)
                    paraNode = Position.getParagraphElement(self.getNode(), _.initial(start));

                    if (paraNode) {
                        paraWidth = Utils.convertLengthToHmm($(paraNode).width(), 'px');
                        if (paraWidth < defaultWidth) {
                            attrs.drawing.width = paraWidth;
                        }
                    }

                    paraPos.push(0);

                    // modifying the attributes, if changeTracking is activated
                    if (changeTrack.isActiveChangeTracking()) {
                        attrs.changes = { inserted: changeTrack.getChangeTrackInfo(), removed: null };
                    }

                    // Adding styleId (only for ODT)
                    if (app.isODF()) { attrs.styleId = 'Frame'; }

                    // handling the required styles for the text frame
                    if (_.isString(attrs.styleId) && drawingStyles.isDirty(attrs.styleId)) {

                        // checking the parent style of the specified style
                        parentStyleId = drawingStyles.getParentId(attrs.styleId);

                        if (_.isString(parentStyleId) && drawingStyles.isDirty(parentStyleId)) {
                            // inserting parent of text frame style to document
                            self.generateInsertStyleOp(generator, 'drawing', parentStyleId, true);
                            drawingStyles.setDirty(parentStyleId, false);
                        }

                        // insert text frame style to document
                        self.generateInsertStyleOp(generator, 'drawing', attrs.styleId);
                    }

                    // text frames are drawings of type shape
                    newOperation = { attrs: attrs, start: start, type: 'shape' };
                    extendPropertiesWithTarget(newOperation, target);
                    generator.generateOperation(Operations.DRAWING_INSERT, newOperation);

                    // add a paragraph into the shape, so that the cursor can be set into the text frame
                    // -> an operation is required for this step, so that remote clients are also prepared (-> no implicit paragraph)
                    // TODO: Change tracking of this paragraph?
                    newOperation = { start: paraPos };
                    extendPropertiesWithTarget(newOperation, target);
                    generator.generateOperation(Operations.PARA_INSERT, newOperation);

                    // reducing distance of paragraphs inside the text frame
                    newOperation = { start: paraPos, attrs: { paragraph: { marginBottom: 0 }}};
                    extendPropertiesWithTarget(newOperation, target);
                    generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);

                    // apply all collected operations
                    self.applyOperations(generator);

                    // setting cursor into text frame
                    selection.setTextSelection(lastOperationEnd);
                }

                if (selection.hasRange()) {
                    return self.deleteSelected()
                    .done(function () {
                        doInsertTextFrame();
                    });
                }

                doInsertTextFrame();
                return $.when();

            }, this);  // enterUndoGroup()
        };

        /**
         * Generating an operation to assign a new anchor to a drawing. Supported anchor
         * values are 'inline', 'paragraph' and 'page'.
         *
         * @param {String} anchor
         *  A string describing the new anchor position of the drawing.
         */
        this.anchorDrawingTo = function (anchor) {

            var // the drawing position
                start = null,
                // the pixel position of the drawing relative to the page node
                pixelPos = null,
                // the drawing node info object
                startNodeInfo = null,
                // an optional footer offset in px
                footerOffset = 0,
                // the page number of the drawing
                pageNumber = 1,
                // an options object
                options = {
                    anchorHorBase: null,
                    anchorHorAlign: null,
                    anchorVertBase: null,
                    anchorVertAlign: null
                },
                // currently active root node
                activeRootNode = self.getCurrentRootNode();

            if (self.isDrawingSelected()) {
                start = selection.getStartPosition();
            } else if (self.getSelection().isAdditionalTextframeSelection()) {
                start = Position.getOxoPosition(activeRootNode, self.getSelection().getSelectedTextFrameDrawing(), 0);
            } else {
                return;
            }

            startNodeInfo = Position.getDOMPosition(activeRootNode, start, true);

            if (startNodeInfo && startNodeInfo.node && DrawingFrame.isDrawingFrame(startNodeInfo.node)) {

                // set Anchor to page or paragraph
                if (anchor === 'page' || anchor === 'paragraph') {

                    if (anchor === 'page') {

                        pixelPos = Position.getPixelPositionToRootNodeOffset(activeRootNode, startNodeInfo.node, app.getView().getZoomFactor());

                        if (DOM.isHeaderNode(activeRootNode)) {
                            pageNumber = 1;  // in header or footer the page number is not required
                        } else if (DOM.isFooterNode(activeRootNode)) {
                            pageNumber = 1;  // in header or footer the page number is not required
                            // if this is a footer, the height between page start and footer need to be added
                            footerOffset = Utils.convertHmmToLength(pageLayout.getPageAttribute('height'), 'px', 1) - Utils.round($(activeRootNode).outerHeight(true), 1);
                            pixelPos.y += footerOffset;
                        } else {
                            pageNumber = pageLayout.getPageNumber(startNodeInfo.node); // maybe this drawing is not on page 1
                            if (pageNumber > 1) { pixelPos.y = pixelPos.y - Position.getVerticalPagePixelPosition(activeRootNode, pageLayout, pageNumber, app.getView().getZoomFactor()); }
                        }

                        options.anchorHorBase = 'page';
                        options.anchorVertBase = 'page';

                        options.anchorHorOffset = Utils.convertLengthToHmm(pixelPos.x, 'px');
                        options.anchorVertOffset = Utils.convertLengthToHmm(pixelPos.y, 'px');

                    } else {
                        options.anchorHorBase = 'column';
                        options.anchorVertBase = 'paragraph';

                        options.anchorHorOffset = 0;
                        options.anchorVertOffset = 0;
                    }

                    options.inline = false;
                    options.anchorHorAlign = 'offset';
                    options.anchorVertAlign = 'offset';

                // set anchor inline into the text
                } else {
                    options.inline = true;
                }

                this.setAttributes('drawing', { drawing: options });

                // the selection needs to be repainted. Otherwise it is not possible
                // to move the drawing to the page borders.
                DrawingFrame.clearSelection(startNodeInfo.node);
                DrawingResize.drawDrawingSelection(app, startNodeInfo.node);
            }

        };

        /**
         * Generating an operation to assign a border to a drawing.
         *
         * @param {String} preset
         *  A string describing the new border of the drawing.
         */
        this.setDrawingBorder = function (preset) {

            // there must be a drawing selection
            if (!self.getSelection().isAnyDrawingSelection()) { return; }

            if (!self.isDrawingSelected() && !self.getSelection().isAdditionalTextframeSelection()) {
                return;
            }

            var // old attributes
                oldLineAttrs = self.getAttributes('drawing').line,
                // new line attributes
                lineAttrs = DrawingUtils.resolvePresetBorder(preset);

            // if no border color is available, push default black
            if ((lineAttrs.type !== 'none') && (!oldLineAttrs || !oldLineAttrs.color || (oldLineAttrs.color.type === 'auto'))) {
                lineAttrs.color = { type: 'rgb', value: '000000' };
            }
            this.setAttributes('drawing', { line: lineAttrs });
        };

        /**
         * Generating an operation to assign a color to a drawing border.
         *
         * @param {Object} color
         *  An object describing the new border color of the drawing.
         */
        this.setDrawingBorderColor = function (color) {

            // there must be a drawing selection
            if (!self.getSelection().isAnyDrawingSelection()) { return; }

            if (!self.isDrawingSelected() && !self.getSelection().isAdditionalTextframeSelection()) {
                return;
            }

            if (color.type === 'auto') {
                self.setAttributes('drawing', { line: { type: 'none' } });
            } else {
                self.setAttributes('drawing', { line: { type: 'solid', color: color } });
            }
        };

        /**
         * Generating an operation to assign a fill color to a drawing.
         *
         * @param {Object} color
         *  An object describing the new fill color of the drawing.
         */
        this.setDrawingFillColor = function (color) {
            // there must be a drawing selection
            if (!self.getSelection().isAnyDrawingSelection()) { return; }

            var // the fill attributes for the drawing
                fillAttrs = {};

            if (color.type === 'auto') {
                fillAttrs.type = 'none';
                fillAttrs.color = { type: 'auto' };
            } else {
                fillAttrs.color = color;
                fillAttrs.type = 'solid';
            }

            self.setAttributes('drawing', { fill: fillAttrs });
        };

        this.showInsertImageDialog = function () {
            var def = Image.showInsertImageDialog(app);
            def = def.then(function (imageFragment) {
                return self.insertImageURL(imageFragment);
            });

            return def;
        };

        this.insertImageURL = function (imageHolder, dropPosition) {

            var attrs = { drawing: {}, image: {}, line: { type: 'none' } };
            if (imageHolder.url) {
                attrs.image.imageUrl = imageHolder.url;
                attrs.drawing.name = imageHolder.name;
            } else if (imageHolder.substring(0, 10) === 'data:image') {
                attrs.image.imageData = imageHolder;
            } else {
                attrs.image.imageUrl = imageHolder;
            }

            return getImageSize(attrs.image.imageUrl || attrs.image.imageData).then(function (size) {

                var def = $.Deferred(),
                    result = false,
                    // created operation
                    newOperation = null;

                // exit silently if we lost the edit rights
                if (!self.getEditMode()) {
                    return def.resolve();
                }

                undoManager.enterUndoGroup(function () {

                    var start = dropPosition ? dropPosition.start : selection.getStartPosition();

                    _.extend(attrs.drawing, size, DEFAULT_DRAWING_MARGINS);

                    handleImplicitParagraph(start);

                    // modifying the attributes, if changeTracking is activated
                    if (changeTrack.isActiveChangeTracking()) {
                        attrs.changes = { inserted: changeTrack.getChangeTrackInfo(), removed: null };
                    }

                    newOperation = {
                        name: Operations.DRAWING_INSERT,
                        start: start,
                        type: 'image',
                        attrs: attrs
                    };
                    extendPropertiesWithTarget(newOperation, self.getActiveTarget());

                    result = self.applyOperations(newOperation);
                }); // enterUndoGroup()

                // setting the cursor position
                selection.setTextSelection(lastOperationEnd);

                return result ? def.resolve() : def.reject();
            })
            .fail(function () {
                app.rejectEditAttempt('image');
            });
        };

        this.insertHyperlinkDirect = function (url, text, dropPosition) {

            var generator = this.createOperationsGenerator(),
                hyperlinkStyleId = self.getDefaultUIHyperlinkStylesheet();

            if (url && url.length > 0) {

                undoManager.enterUndoGroup(function () {

                    // helper function to insert hyper link after selected content is removed
                    function doInsertHyperlink() {

                        var newText = text || url,
                            // created operation
                            newOperation = null,
                            // target string property of operation
                            target = self.getActiveTarget(),
                            // the logical start and end positions
                            start = null, end = null;

                        // reading start and end position (after calling deleteSelected())
                        start = dropPosition ? dropPosition.start : selection.getStartPosition();

                        // insert new text
                        handleImplicitParagraph(start);
                        newOperation = { text: newText, start: _.clone(start) };
                        extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.TEXT_INSERT, newOperation);

                        // Calculate end position of new text
                        // will be used for setAttributes operation
                        end = Position.increaseLastIndex(start, newText.length);

                        if (characterStyles.isDirty(hyperlinkStyleId)) {
                            // insert hyperlink style to document
                            self.generateInsertStyleOp(generator, 'character', hyperlinkStyleId);
                        }

                        newOperation = {
                            attrs: { styleId: hyperlinkStyleId, character: { url: url } },
                            start: _.clone(start),
                            end: _.clone(end)
                        };
                        extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);

                        // apply all collected operations
                        self.applyOperations(generator);
                    }

                    // delete selected range (this creates new operations)
                    if (selection.hasRange()) {
                        return self.deleteSelected()
                        .done(function () {
                            doInsertHyperlink();
                        });
                    }

                    doInsertHyperlink();
                    return $.when();
                });
            }
        };

        /**
         * Shows the hyperlink dialog for the current selection, and returns a
         * Deferred object that allows to wait for the dialog.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  dialog has been closed with the default action; or rejected, if the
         *  dialog has been canceled.
         */
        this.insertHyperlinkDialog = function () {

            var generator = this.createOperationsGenerator(),
                text = '', url = '',
                startPos = null,
                start = selection.getStartPosition(),
                end = selection.getEndPosition(),
                // target string for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // currenty active root node
                activeRootNode = self.getCurrentRootNode(target),
                // created operation
                newOperation = null;

            if (!selection.hasRange()) {
                var newSelection = Hyperlink.findSelectionRange(this, selection);
                if (newSelection.start !== null && newSelection.end !== null) {
                    startPos = selection.getStartPosition();
                    start[start.length - 1] = newSelection.start;
                    end[end.length - 1] = newSelection.end;
                    selection.setTextSelection(start, end);
                }
            }

            if (!this.hasEnclosingParagraph()) {
                return $.Deferred().reject();
            }

            // use range to retrieve text and possible url
            if (selection.hasRange()) {

                // Find out the text/url of the selected text to provide them to the
                // hyperlink dialog
                selection.iterateNodes(function (node, pos, start, length) {
                    if ((start >= 0) && (length >= 0) && DOM.isTextSpan(node)) {
                        var nodeText = $(node).text();
                        if (nodeText) {
                            text = text.concat(nodeText.slice(start, start + length));
                        }
                        if (url.length === 0) {
                            var charAttributes = characterStyles.getElementAttributes(node).character;
                            if (charAttributes.url && charAttributes.url.length > 0) {
                                url = charAttributes.url;
                            }
                        }
                    }
                });
            }

            // show hyperlink dialog
            var promise = new Dialogs.HyperlinkDialog(app.getView(), url, text).show();
            return promise.done(function (data) {
                // set url to selected text
                var hyperlinkStyleId = self.getDefaultUIHyperlinkStylesheet(),
                    url = data.url,
                    oldAttrs = null,
                    attrs = null,
                    hyperlinkNode = null,
                    insertedLink = false;

                undoManager.enterUndoGroup(function () {

                    if (data.url === null && data.text === null) {
                        // remove hyperlink
                        // setAttribute uses a closed range therefore -1
                        attrs = Hyperlink.CLEAR_ATTRIBUTES;

                        if (changeTrack.isActiveChangeTracking()) {
                            hyperlinkNode = Position.getDOMPosition(activeRootNode, selection.getEndPosition()).node;
                            if (hyperlinkNode && (hyperlinkNode.nodeType === 3)) { hyperlinkNode = hyperlinkNode.parentNode; }
                            // Expanding operation for change tracking with old explicit attributes
                            oldAttrs = changeTrack.getOldNodeAttributes(hyperlinkNode);
                            // adding the old attributes, author and date for change tracking
                            if (oldAttrs) {
                                oldAttrs = _.extend(oldAttrs, changeTrack.getChangeTrackInfo());
                                attrs.changes = { modified: oldAttrs };
                            }
                        }
                    } else {

                        // if the data.text is a emtpy string, a default value is set (fix for Bug 40313)
                        if (data.text === '') {
                            data.text = data.url;
                        }

                        // insert/change hyperlink
                        if (data.text !== text) {

                            // text has been changed
                            if (selection.hasRange()) {
                                self.deleteSelected();
                                start = selection.getStartPosition();
                            }

                            if (changeTrack.isActiveChangeTracking()) {
                                attrs = attrs || {};
                                attrs.changes = { inserted: changeTrack.getChangeTrackInfo() };
                                insertedLink = true;
                            }

                            // insert new text
                            handleImplicitParagraph(start);
                            newOperation = { text: data.text, start: _.clone(start), attrs: attrs };
                            extendPropertiesWithTarget(newOperation, target);
                            generator.generateOperation(Operations.TEXT_INSERT, newOperation);

                            // Calculate end position of new text, will be used for setAttributes operation
                            end = Position.increaseLastIndex(start, data.text.length);
                        }

                        if (characterStyles.isDirty(hyperlinkStyleId)) {
                            // insert hyperlink style to document
                            self.generateInsertStyleOp(generator, 'character', hyperlinkStyleId);
                        }

                        attrs = { styleId: hyperlinkStyleId, character: { url: url } };

                        if (changeTrack.isActiveChangeTracking() && !insertedLink) {
                            hyperlinkNode = Position.getDOMPosition(activeRootNode, end).node;
                            if (hyperlinkNode && (hyperlinkNode.nodeType === 3)) { hyperlinkNode = hyperlinkNode.parentNode; }
                            // Expanding operation for change tracking with old explicit attributes
                            oldAttrs = changeTrack.getOldNodeAttributes(hyperlinkNode);
                            // adding the old attributes, author and date for change tracking
                            if (oldAttrs) {
                                oldAttrs = _.extend(oldAttrs, changeTrack.getChangeTrackInfo());
                                attrs.changes = { modified: oldAttrs };
                            }
                        }
                    }

                    end[end.length - 1] -= 1;
                    newOperation = {
                        attrs: attrs,
                        start: _.clone(start),
                        end: _.clone(end)
                    };
                    extendPropertiesWithTarget(newOperation, target);
                    generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);

                    // apply all collected operations
                    self.applyOperations(generator);

                }, self); // enterUndoGroup()

            }).always(function () {
                if (startPos) {
                    selection.setTextSelection(startPos);
                }
            });
        };

        this.removeHyperlink = function () {

            var generator = this.createOperationsGenerator(),
                startPos = null,
                start = selection.getStartPosition(),
                end = selection.getEndPosition(),
                oldAttrs = null,
                attrs = null,
                hyperlinkNode = null,
                // target string for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // currently active root node
                activeRootNode = self.getCurrentRootNode(target),
                // created operation
                newOperation = null;

            if (!selection.hasRange()) {
                var newSelection = Hyperlink.findSelectionRange(this, selection);
                if (newSelection.start !== null && newSelection.end !== null) {
                    startPos = selection.getStartPosition();
                    start[start.length - 1] = newSelection.start;
                    end[end.length - 1] = newSelection.end;
                    selection.setTextSelection(start, end);
                }
            }

            if (selection.hasRange() && this.hasEnclosingParagraph()) {

                attrs = Hyperlink.CLEAR_ATTRIBUTES;

                if (changeTrack.isActiveChangeTracking()) {
                    hyperlinkNode = Position.getDOMPosition(activeRootNode, selection.getEndPosition()).node;
                    if (hyperlinkNode && (hyperlinkNode.nodeType === 3)) { hyperlinkNode = hyperlinkNode.parentNode; }
                    // Expanding operation for change tracking with old explicit attributes
                    oldAttrs = changeTrack.getOldNodeAttributes(hyperlinkNode);
                    // adding the old attributes, author and date for change tracking
                    if (oldAttrs) {
                        oldAttrs = _.extend(oldAttrs, changeTrack.getChangeTrackInfo());
                        attrs.changes = { modified: oldAttrs };
                    }
                }

                // remove hyperlink
                // setAttribute uses a closed range therefore -1
                end[end.length - 1] -= 1;

                newOperation = {
                    attrs: attrs,
                    start: _.clone(start),
                    end: _.clone(end)
                };
                extendPropertiesWithTarget(newOperation, target);
                generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);

                // apply the operations (undo group is created automatically)
                this.applyOperations(generator);

                this.executeDelayed(function () {
                    app.getView().grabFocus();
                    if (startPos) {
                        selection.setTextSelection(startPos);
                    }
                }, undefined, 'Text: removeHyperLink');
            }
        };

        /**
         * Inserting a tab stopp.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  dialog has been closed with the default action; or rejected, if the
         *  dialog has been canceled. If no dialog is shown, the deferred is
         *  resolved immediately.
         *  The undo manager returns the return value of the callback function.
         */
        this.insertTab = function () {

            return undoManager.enterUndoGroup(function () {

                function doInsertTabAndSetCursor() {
                    var start = selection.getStartPosition(),
                        operation = { name: Operations.TAB_INSERT, start: start },
                        // target for operation - if exists, it's for ex. header or footer
                        target = self.getActiveTarget();

                    extendPropertiesWithTarget(operation, target);
                    handleImplicitParagraph(start);
                    if (_.isObject(preselectedAttributes)) { operation.attrs = preselectedAttributes; }
                    // modifying the attributes, if changeTracking is activated
                    if (changeTrack.isActiveChangeTracking()) {
                        operation.attrs = operation.attrs || {};
                        operation.attrs.changes = { inserted: changeTrack.getChangeTrackInfo(), removed: null };
                    }
                    self.applyOperations(operation);

                    if (Utils.IOS && self.isHeaderFooterEditState()) {
                        //header footer workaround (Bug 39976)
                        self.applyOperations({ name: Operations.TEXT_INSERT, text: ' ', start: Position.increaseLastIndex(start) });
                        self.deleteRange(Position.increaseLastIndex(start));
                    } else {
                        selection.setTextSelection(lastOperationEnd);
                    }
                }

                if (selection.hasRange()) {
                    return self.deleteSelected()
                    .done(function () {
                        doInsertTabAndSetCursor();
                    });
                }

                doInsertTabAndSetCursor();

                return $.when();

            }, this);

        };

        /**
         * Inserting a hard line break, that leads to a new line inside a paragraph. This
         * function can be called from the side pane with a button or with 'Shift+Enter' from
         * within the document.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  dialog has been closed with the default action; or rejected, if the
         *  dialog has been canceled. If no dialog is shown, the deferred is
         *  resolved immediately.
         *  The undo manager returns the return value of the callback function.
         */
        this.insertHardBreak = function () {

            return undoManager.enterUndoGroup(function () {

                function doInsertHardBreak() {
                    var start = selection.getStartPosition(),
                        operation = { name: Operations.HARDBREAK_INSERT, start: start },
                        // target for operation - if exists, it's for ex. header or footer
                        target = self.getActiveTarget();

                    extendPropertiesWithTarget(operation, target);
                    handleImplicitParagraph(start);
                    // modifying the attributes, if changeTracking is activated
                    if (changeTrack.isActiveChangeTracking()) {
                        operation.attrs = operation.attrs || {};
                        operation.attrs.changes = { inserted: changeTrack.getChangeTrackInfo(), removed: null };
                    }
                    self.applyOperations(operation);
                }

                if (selection.hasRange()) {
                    return self.deleteSelected()
                    .done(function () {
                        doInsertHardBreak();
                        selection.setTextSelection(lastOperationEnd); // finally setting the cursor position
                    });
                }

                doInsertHardBreak();
                selection.setTextSelection(lastOperationEnd); // finally setting the cursor position
                return $.when();

            }, this);

        };

        /**
         * Inserting a maunal page break, that leads to a new page inside a document. This
         * function can be called from the toolbar with a button or with 'CTRL+Enter' from
         * within the document.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  dialog has been closed with the default action; or rejected, if the
         *  dialog has been canceled. If no dialog is shown, the deferred is
         *  resolved immediately.
         *  The undo manager returns the return value of the callback function.
         */
        this.insertManualPageBreak = function () {

            if (self.isHeaderFooterEditState()) {
                return;
            }

            return undoManager.enterUndoGroup(function () {
                var // start position of selection
                    start = selection.getStartPosition(),
                    // copy of start position, prepared for modifying
                    positionOfCursor = _.clone(start),
                    // last index in positionOfCursor array, used for incrementing cursor position
                    lastIndex = positionOfCursor.length - 1,
                    // first level element, paragraph or table
                    firstLevelNode = Position.getContentNodeElement(editdiv, start.slice(0, 1)),
                    // object passed to generateOperation method
                    operationOptions,
                    // operations generator
                    generator = this.createOperationsGenerator();

                function doInsertManualPageBreak() {

                    var // the attributes for the paragraph between the tables
                        paraAttrs = {};

                    // refreshing start position, that might be modified by deleteSelected()
                    start = selection.getStartPosition();

                    // different behavior for inserting page break inside paragraphs and tables
                    if (DOM.isParagraphNode(firstLevelNode)) {

                        // first split paragraph
                        generator.generateOperation(Operations.PARA_SPLIT, { start: start });
                        handleImplicitParagraph(start);
                        //update cursor position value after paragraph split
                        positionOfCursor[lastIndex - 1] += 1;
                        positionOfCursor[lastIndex] = 0;
                        // set paragraph attribute page break
                        operationOptions = { start: positionOfCursor.slice(0, -1), attrs: { paragraph: { pageBreakBefore: true } } };

                        if (changeTrack.isActiveChangeTracking()) {
                            operationOptions.attrs.changes = { inserted: changeTrack.getChangeTrackInfo(), removed: null };
                        }
                        generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);

                    } else if (DOM.isTableNode(firstLevelNode)) {
                        var firstParagraphPosInFirstCell,
                            firstParagraphPosHelper = positionOfCursor.slice(0, 3);

                        // we always apply pageBreakBefore attribute to the first paragraph in table - because of compatibility with MS Word
                        firstParagraphPosHelper[2] = 0;
                        firstParagraphPosInFirstCell = Position.getFirstPositionInCurrentCell(editdiv, firstParagraphPosHelper);
                        // before applying paragraph attribute, we need to check if it's implicit paragraph, #34624
                        // #35328 - table is not yet split, and we dont check impl paragraph for cursor position, but for first par in row, where we apply pageBreakBefore attr
                        handleImplicitParagraph(firstParagraphPosInFirstCell);

                        // different behavior for odt and msword page breaks in table
                        if (app.isODF()) {
                            if (start[0] === 0 || (DOM.isManualPageBreakNode(firstLevelNode))) {
                                // if table is at first position in document, or there is already page break applied to table, ODF ignores operation
                                return;
                            }

                            // #35446 - odf applies pageBreakBefore attr always on first cell in table, and doesn't split table
                            firstParagraphPosInFirstCell = Position.getFirstPositionInParagraph(editdiv, positionOfCursor.slice(0, 1));
                            handleImplicitParagraph(firstParagraphPosInFirstCell);
                            // odf specific, set attr to table, too, #35585
                            operationOptions = { start: firstParagraphPosInFirstCell.slice(0, -1), attrs: { paragraph: { pageBreakBefore: true } } };
                            generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);

                        } else {
                            // split table with page break, but only if it's not first row. Otherwise, just insert new paragraph before table
                            if (positionOfCursor[1] !== 0) {
                                generator.generateOperation(Operations.TABLE_SPLIT, { start: positionOfCursor.slice(0, 2) }); // passing row position of table closest to pagecontent node (top level table)

                                firstParagraphPosInFirstCell[0] += 1;
                                firstParagraphPosInFirstCell[1] = 0;

                                operationOptions = { start: firstParagraphPosInFirstCell.slice(0, -1), attrs: { paragraph: { pageBreakBefore: true } } };
                                generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);

                                // Insert empty paragraph in between split tables.
                                // Adding change track information only to the inserted paragraph and only, if
                                // change tracking is active.
                                // -> the splitted table cannot be marked as inserted. Rejecting this change track
                                // requires automatic merge of the splitted table, if possible.
                                if (changeTrack.isActiveChangeTracking()) {
                                    paraAttrs.changes = { inserted: changeTrack.getChangeTrackInfo(), removed: null };
                                }

                                // update position of cursor
                                positionOfCursor[0] += 1;

                                // #36130 - avoid error in console: Selection.restoreBrowserSelection(): missing text selection range
                                selection.setTextSelection(Position.getFirstPositionInParagraph(editdiv, start.slice(0, 1)));

                                // insert empty paragraph in between split tables
                                generator.generateOperation(Operations.PARA_INSERT, { start: positionOfCursor.slice(0, 1), attrs: paraAttrs });

                                // update position of cursor
                                positionOfCursor[0] += 1;
                                positionOfCursor[1] = 0;

                            } else {
                                operationOptions = { start: firstParagraphPosInFirstCell.slice(0, -1), attrs: { paragraph: { pageBreakBefore: true } } };
                                generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);

                                // if we apply page break to element that already has that property, set to paragraph that is inserted before it
                                if (DOM.isManualPageBreakNode(firstLevelNode)) {
                                    paraAttrs.paragraph = { pageBreakBefore: true };
                                }

                                // Insert empty paragraph in between split tables.
                                // Adding change track information only to the inserted paragraph and only, if
                                // change tracking is active.
                                // -> the splitted table cannot be marked as inserted. Rejecting this change track
                                // requires automatic merge of the splitted table, if possible.
                                if (changeTrack.isActiveChangeTracking()) {
                                    paraAttrs.changes = { inserted: changeTrack.getChangeTrackInfo(), removed: null };
                                }

                                // insert empty paragraph in between split tables
                                generator.generateOperation(Operations.PARA_INSERT, { start: positionOfCursor.slice(0, 1), attrs: paraAttrs });

                                // update position of cursor
                                positionOfCursor[0] += 1;

                                // #36130 - avoid error in console: Selection.restoreBrowserSelection(): missing text selection range
                                selection.setTextSelection(Position.getFirstPositionInParagraph(editdiv, positionOfCursor.slice(0, 1)));
                            }
                        }
                    }

                    self.applyOperations(generator);
                }

                if (selection.hasRange() && !DOM.isTableNode(firstLevelNode)) { // #35346 don't delete if table is selected
                    return self.deleteSelected()
                    .done(function () {
                        doInsertManualPageBreak();
                        selection.setTextSelection(positionOfCursor); // finally setting the cursor position
                    });
                }

                doInsertManualPageBreak();
                selection.setTextSelection(positionOfCursor); // finally setting the cursor position
                return $.when();

            }, this);

        };

        /**
         * Creating the operation for splitting a paragraph. A check is necessary, if the paragraph
         * contains leading floated drawings. In this case it might be necessary to update the
         * specified split position. For example the specified split position is [0,1]. If there
         * is a floated drawing at position [0,0], it is necessary to split the paragraph at
         * the position [0,0], so that the floated drawing is also moved into the following
         * paragraph.
         *
         * @param {Number[]} position
         *  The logical split position.
         *
         * @param {Node|jQuery|Null} [para]
         *  The paragraph that will be splitted. If this object is a jQuery collection, uses
         *  the first DOM node it contains. If missing, it will be determined from the specified
         *  logical position. This parameter can be used for performance reasons.
         */
        this.splitParagraph = function (position, para) {

            var
                // currently active root node
                activeRootNode = self.getCurrentRootNode(),
                // the paragraph node, if not specified as parameter
                paragraph = para || Position.getParagraphElement(activeRootNode, position.slice(0, -1)),
                // whether the paragraph contains floated children
                hasFloatedChildren = DOM.containsFloatingDrawingNode(paragraph),
                // the last position of the specified logical split position
                offset = _.last(position),
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                operationOptions = {};

            // move leading floating drawings to the new paragraph, if passed position points inside floating drawings
            // or directly after the last floating drawing
            if ((hasFloatedChildren) && (offset > 0) && (offset <= Position.getLeadingFloatingDrawingCount(paragraph))) {
                position[position.length - 1] = 0;
            }
            operationOptions = { name: Operations.PARA_SPLIT, start: _.clone(position) };
            extendPropertiesWithTarget(operationOptions, target);
            this.applyOperations(operationOptions);
        };

        this.mergeParagraph = function (position) {
            var
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                operationOptions = { name: Operations.PARA_MERGE, start: _.clone(position) };

            extendPropertiesWithTarget(operationOptions, target);
            this.applyOperations(operationOptions);
        };

        /**
         * Creates operation to split given table in two tables. Split point is row where the cursor is positioned.
         * If cursor is placed in first row, only paragraph is inserted, and whole table shifted for one position down.
         * Row where the cursor is always goes with second newly created table, being the first row in that table.
         * In between two newly created tables, new paragraph is inserted.
         * If there is table in table(s), and cursor is in that table, only that table is split.
         * ODF doesnt support table split.
         */
        this.splitTable = function () {
            return undoManager.enterUndoGroup(function () {
                var // operations generator
                    generator = this.createOperationsGenerator(),
                    //position of the cursor
                    position = selection.getStartPosition(),
                    // target for operation - if exists, it's for ex. header or footer
                    target = self.getActiveTarget(),
                    // currently active root node
                    activeRootNode = self.getCurrentRootNode(target),
                    // element
                    elementNode = Position.getContentNodeElement(activeRootNode, position.slice(0, -1)),
                    // table node from position
                    tableNode = $(elementNode).closest('table'),
                    // the position of the 'old' table
                    tablePosition = Position.getOxoPosition(activeRootNode, tableNode),
                    // the position of table and row
                    tableAndRowPosition = position.slice(0, tablePosition.length + 1),
                    // cursor position in new table after splitting
                    newPosition = _.clone(tableAndRowPosition),
                    // attributes for the split table operation
                    attrs = null,
                    operationOptions = {};

                if (DOM.isTableNode(tableNode) && !DOM.isExceededSizeTableNode(tableNode)) { // proceed only if it is table node
                    if (_.last(tableAndRowPosition) !== 0) { // split if its not first row, otherwise just insert new paragraph before
                        operationOptions = { start: tableAndRowPosition };
                        extendPropertiesWithTarget(operationOptions, target);
                        generator.generateOperation(Operations.TABLE_SPLIT, operationOptions);

                        // update cursor position
                        newPosition[newPosition.length - 2] += 1;
                        newPosition[newPosition.length - 1] = 0;
                    }
                    // Adding change track information only to the inserted paragraph and only, if
                    // change tracking is active.
                    // -> the splitted table cannot be marked as inserted. Rejecting this change track
                    // requires automatic merge of the splitted table, if possible.
                    if (changeTrack.isActiveChangeTracking()) {
                        attrs = attrs || {};
                        attrs.changes = { inserted: changeTrack.getChangeTrackInfo(), removed: null };
                    }

                    // #36130 - avoid error in console: Selection.restoreBrowserSelection(): missing text selection range
                    selection.setTextSelection(Position.getFirstPositionInParagraph(activeRootNode, newPosition.slice(0, -1)));

                    operationOptions = { start: newPosition.slice(0, -1), attrs: attrs };
                    extendPropertiesWithTarget(operationOptions, target);
                    generator.generateOperation(Operations.PARA_INSERT, operationOptions);

                    // apply all collected operations
                    self.applyOperations(generator);
                    // set cursor in new paragraph between split tables
                    selection.setTextSelection(newPosition, null, { simpleTextSelection: false, splitOperation: false });
                }
            }, this);
        };

        /**
         * Creates one table from two neighbour tables. Conditions are that there is no other element between them,
         * both tables have same table style, and same number of columns (table grids). Condition that must be fulfilled,
         * is that union of this two tables doesn't exceed defined number of columns, row, or cells.
         * ODF doesn't support this feature.
         *
         * @param {Number[]} position
         *  OXO position of first table we try to merge
         * @param {Object} [options]
         * Optional parameters:
         *      @param {Boolean} [options.next]
         *      If set to false, we try to merge passed table position with previous element.
         *      Otherwise with next.
         *
         */
        this.mergeTable = function (position, options) {

            return undoManager.enterUndoGroup(function () {

                var // merging is designed to try to merge with next element, so if we are already at next element jump back to previous
                    next = Utils.getBooleanOption(options, 'next', true),
                    // target for operation - if exists, it's for ex. header or footer
                    target = self.getActiveTarget(),
                    // currently active root node
                    activeRootNode = self.getCurrentRootNode(target),
                    // element
                    elementNode = Position.getContentNodeElement(activeRootNode, position),
                    // the 'old' table node
                    tableNode = $(elementNode).closest('table'),
                    // oxo position of table
                    tablePosition = Position.getOxoPosition(activeRootNode, tableNode),
                    // second table node, which we try to merge with first, old one
                    secondTableNode,
                    // operations generator
                    generator = this.createOperationsGenerator(),
                    operationOptions = {};

                // merging is designed to try to merge with next element, so if we are already at next element jump back to previous
                if (!next) {
                    secondTableNode = DOM.getAllowedNeighboringNode(tableNode, { next: false });
                    if (tablePosition[0] !== 0) {
                        tablePosition[0] -= 1;
                    }
                } else {
                    secondTableNode = DOM.getAllowedNeighboringNode(tableNode, { next: true });
                }

                if (DOM.isTableNode(secondTableNode) && Table.mergeableTables(tableNode, secondTableNode)) {
                    // merging is posible only if two tables have the same number of cols and style id
                    operationOptions = { start: tablePosition };
                    extendPropertiesWithTarget(operationOptions, target);
                    generator.generateOperation(Operations.TABLE_MERGE, operationOptions);
                    // apply all collected operations
                    self.applyOperations(generator);

                    return true;
                }

            }, this);
        };

        this.insertText = function (text, position, attrs) {

            var
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // current element container node
                rootNode = self.getCurrentRootNode(target);

            // adding changes attributes -> but for performance reasons only, if the change tracking is active!
            if (changeTrack.isActiveChangeTracking()) {
                attrs = attrs || {};
                attrs.changes = { inserted: changeTrack.getChangeTrackInfo(), removed: null };
            }

            undoManager.enterUndoGroup(function () {

                var operation = { name: Operations.TEXT_INSERT, text: text, start: _.clone(position) };

                handleImplicitParagraph(position);
                if (_.isObject(attrs)) { operation.attrs = _.copy(attrs, true); }
                guiTriggeredOperation = true;  // Fix for 30587, remote client requires synchronous attribute setting
                this.applyOperations(operation);
                guiTriggeredOperation = false;

                // Bug 25077: Chrome cannot set curser after a trailing SPACE
                // character, need to validate the paragraph node immediately
                // (which converts trailing SPACE characters to NBSP)
                // -> only do this, if it is really necessary (performance)
                if (_.browser.WebKit && (text === ' ')) {
                    validateParagraphNode(Position.getParagraphElement(rootNode, position.slice(0, -1)));
                }

            }, this);
        };

        /**
         * creates a default list either with decimal numbers or bullets
         * @param type {String} 'numbering' or 'bullet'
         */
        this.createDefaultList = function (type) {

            var defListStyleId = listCollection.getDefaultNumId(type);

            undoManager.enterUndoGroup(function () {

                if (defListStyleId === undefined) {
                    var listOperation = listCollection.getDefaultListOperation(type);
                    this.applyOperations(listOperation);
                    defListStyleId = listOperation.listStyleId;
                }
                setListStyle(defListStyleId, 0);
            }, this);
        };

        this.createSelectedListStyle = function (listStyleId, listLevel) {

            undoManager.enterUndoGroup(function () {
                var listOperation,
                    listOperationAndStyle,
                    savedSelection = _.copy(selection.getStartPosition()),
                    _start = selection.getStartPosition(),
                    currentParaPosition,
                    para,
                    paraAttributes,
                    prevParagraph, prevAttributes,
                    nextParagraph, nextAttributes;

                _start.pop();
                para = Position.getParagraphElement(editdiv, _start);
                paraAttributes = paragraphStyles.getElementAttributes(para).paragraph;

                if (!selection.hasRange() && paraAttributes.listStyleId !== '') {
                    //merge level from new listlevel with used list level and apply that new list to all consecutive paragraphs using the old list
                    if (listStyleId !== paraAttributes.listStyleId) {

                        listOperationAndStyle = listCollection.mergeListStyle(paraAttributes.listStyleId, listStyleId, paraAttributes.listLevel);
                        if (listOperationAndStyle.listOperation) {
                            this.applyOperations(listOperationAndStyle.listOperation);
                        }
                        prevParagraph = Utils.findPreviousNode(editdiv, para, DOM.PARAGRAPH_NODE_SELECTOR);
                        while (prevParagraph) {
                            prevAttributes = paragraphStyles.getElementAttributes(prevParagraph).paragraph;
                            if (prevAttributes.listStyleId === paraAttributes.listStyleId) {
                                currentParaPosition = Position.getOxoPosition(editdiv, prevParagraph, 0);
                                currentParaPosition.push(0);
                                selection.setTextSelection(currentParaPosition);
                                setListStyle(listOperationAndStyle.listStyleId);
                                prevParagraph = Utils.findPreviousNode(editdiv, prevParagraph, DOM.PARAGRAPH_NODE_SELECTOR);
                            } else {
                                prevParagraph = null;
                            }
                        }

                        nextParagraph = Utils.findNextNode(editdiv, para, DOM.PARAGRAPH_NODE_SELECTOR);
                        while (nextParagraph) {
                            nextAttributes = paragraphStyles.getElementAttributes(nextParagraph).paragraph;
                            if (nextAttributes.listStyleId === paraAttributes.listStyleId) {
                                currentParaPosition = Position.getOxoPosition(editdiv, nextParagraph, 0);
                                currentParaPosition.push(0);
                                selection.setTextSelection(currentParaPosition);
                                setListStyle(listOperationAndStyle.listStyleId);
                                nextParagraph = Utils.findNextNode(editdiv, nextParagraph, DOM.PARAGRAPH_NODE_SELECTOR);
                            } else {
                                nextParagraph = null;
                            }
                        }

                        selection.setTextSelection(savedSelection);
                        setListStyle(listOperationAndStyle.listStyleId);
                    }

                } else {
                    listOperation = listCollection.getSelectedListStyleOperation(listStyleId);
                    if (listOperation) {
                        this.applyOperations(listOperation);
                    }
                    listLevel = Math.max(0, listLevel || 0);  // setting listLevel for paragraphs, that were not in lists before

                    setListStyle(listStyleId, listLevel);
                }

            }, this); // enterUndoGroup
        };

        this.createList = function (type, options) {

            var defListStyleId = (!options || (!options.symbol && !options.listStartValue)) ? listCollection.getDefaultNumId(type) : undefined,
                // the attributes object for the operation
                allAttrs = null,
                // the old paragraph attributes
                oldAttrs = null,
                // the paragraph dom node
                paraNode = null,
                // the operation object
                newOperation = null;

            if (defListStyleId === undefined) {
                var listOperation = listCollection.getDefaultListOperation(type, options);
                this.applyOperations(listOperation);
                defListStyleId = listOperation.listStyleId;
            }
            if (options && options.startPosition) {
                var start = _.clone(options.startPosition),
                    listParaStyleId = this.getDefaultUIParagraphListStylesheet(),
                    insertStyleOperation = null;

                start.pop();
                // register pending style sheet via 'insertStyleSheet' operation
                if (_.isString(listParaStyleId) && paragraphStyles.isDirty(listParaStyleId)) {

                    var styleSheetAttributes = {
                        attrs: paragraphStyles.getStyleSheetAttributeMap(listParaStyleId),
                        type: 'paragraph',
                        styleId: listParaStyleId,
                        styleName: paragraphStyles.getName(listParaStyleId),
                        parent: paragraphStyles.getParentId(listParaStyleId),
                        uiPriority: paragraphStyles.getUIPriority(listParaStyleId)
                    };

                    // parent is an optional value, should not be send as 'null'
                    if (styleSheetAttributes.parent === null) { delete styleSheetAttributes.parent; }

                    insertStyleOperation = _.extend({ name: Operations.INSERT_STYLESHEET }, styleSheetAttributes);
                    this.applyOperations(insertStyleOperation);

                    // remove the dirty flag
                    paragraphStyles.setDirty(listParaStyleId, false);
                }

                // the attributes object for the operation
                allAttrs = { styleId: listParaStyleId, paragraph: { listStyleId: defListStyleId, listLevel: 0 } };

                // handling change track informations for set attributes operations (36259)
                if (changeTrack.isActiveChangeTracking()) {
                    // Creating one setAttribute operation for each span inside the paragraph, because of old attributes!
                    oldAttrs = {};
                    paraNode = Position.getParagraphElement(editdiv, start);

                    if (DOM.isParagraphNode(paraNode)) {
                        oldAttrs = changeTrack.getOldNodeAttributes(paraNode);
                    }

                    // adding the old attributes, author and date for change tracking
                    if (oldAttrs) {
                        oldAttrs = _.extend(oldAttrs, changeTrack.getChangeTrackInfo());
                        allAttrs.changes = { modified: oldAttrs };
                    }
                }

                newOperation = {
                    name: Operations.SET_ATTRIBUTES,
                    attrs: allAttrs,
                    start: start
                };
                this.applyOperations(newOperation);
                newOperation = _.copy(newOperation);
                newOperation.start = Position.increaseLastIndex(newOperation.start);
                this.applyOperations(newOperation);
            } else {
                this.setAttributes('paragraph', { styleId: this.getDefaultUIParagraphListStylesheet(), paragraph: { listStyleId: defListStyleId, listLevel: 0 } });
            }

        };

        /**
         * Removes bullet/numbered list formatting from the selected
         * paragraphs.
         */
        this.removeListAttributes = function () {

            var paragraph = Position.getLastNodeFromPositionByNodeName(self.getCurrentRootNode(), selection.getStartPosition(), DOM.PARAGRAPH_NODE_SELECTOR),
                prevPara = Utils.findPreviousNode(self.getCurrentRootNode(), paragraph, DOM.PARAGRAPH_NODE_SELECTOR),
                attrs = self.getAttributes('paragraph'),
                newAttrs = { paragraph: { listStyleId: null, listLevel: -1 } };

            if (!attrs.styleId || attrs.styleId === self.getDefaultUIParagraphListStylesheet()) {
                //set list style only, if there is no special para style chosen
                newAttrs.styleId = self.getDefaultUIParagraphStylesheet();
            }

            self.setAttributes('paragraph', newAttrs);
            if (prevPara) {
                paragraphStyles.updateElementFormatting(prevPara);
            }
        };

        // style sheets and formatting attributes -----------------------------

        /**
         * Returns the collection of all list definitions.
         */
        this.getListCollection = function () {
            return listCollection;
        };

        /**
         * Adds the passed attributes to the set of preselected attributes that
         * will be applied to the next inserted text contents without moving
         * the text cursor or executing any other action.
         *
         * @param {Object} attributes
         *  A character attribute set that will be added to the set of
         *  preselected attributes.
         *
         * @returns {Editor}
         *  A reference to this instance.
         */
        this.addPreselectedAttributes = function (attributes) {
            preselectedAttributes = preselectedAttributes || {};
            this.extendAttributes(preselectedAttributes, attributes);
            return this;
        };

        /**
         * Returns the values of all formatting attributes of the elements in
         * the current selection associated to the specified attribute family.
         *
         * @param {String} family
         *  The name of the attribute family used to select specific elements
         *  in the current selection:
         *  - 'character': all text spans (text portions, text components),
         *  - 'paragraph': all paragraph nodes,
         *  - 'table': all table nodes,
         *  - 'drawing': all drawing object nodes.
         *
         * @returns {Object}
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute family.
         */
        this.getAttributes = function (family, options) {

            var // whether the selection is a simple cursor
                isCursor = selection.isTextCursor(),
                // table or drawing element contained by the selection
                element = null,
                // resulting merged attributes
                mergedAttributes = null,
                // max iterations
                maxIterations = Utils.getOption(options, 'maxIterations'),
                // get only attributes of the group
                groupOnly = Utils.getBooleanOption(options, 'groupOnly', false);

            // merges the passed element attributes into the resulting attributes
            function mergeElementAttributes(elementAttributes) {

                var // whether any attribute is still unambiguous
                    hasNonNull = false;

                // merges the passed attribute value into the attributes map
                function mergeAttribute(attributes, name, value) {
                    if (!(name in attributes)) {
                        // initial iteration: store value
                        attributes[name] = value;
                    } else if (!_.isEqual(value, attributes[name])) {
                        // value differs from previous value: ambiguous state
                        attributes[name] = null;
                    }
                    hasNonNull = hasNonNull || !_.isNull(attributes[name]);
                }

                // initial iteration: store attributes and return
                if (!mergedAttributes) {
                    mergedAttributes = elementAttributes;
                    return;
                }

                // process all passed element attributes
                _(elementAttributes).each(function (attributeValues, subFamily) {
                    if (subFamily === 'styleId') {
                        mergeAttribute(mergedAttributes, 'styleId', attributeValues);
                    } else {
                        var mergedAttributeValues = mergedAttributes[subFamily];
                        _(attributeValues).each(function (value, name) {
                            mergeAttribute(mergedAttributeValues, name, value);
                        });
                    }
                });

                // stop iteration, if all attributes are ambiguous
                return hasNonNull ? undefined : Utils.BREAK;
            }

            switch (family) {

            case 'character':
                selection.iterateNodes(function (node) {
                    return DOM.iterateTextSpans(node, function (span) {
                        // ignore empty text spans (they cannot be formatted via operations),
                        // but get formatting of an empty span selected by a text cursor
                        if (isCursor || (span.firstChild.nodeValue.length > 0)) {
                            return mergeElementAttributes(characterStyles.getElementAttributes(span));
                        }
                    });
                }, null, { maxIterations: maxIterations });
                if (isCursor && preselectedAttributes) {
                    // add preselected attributes (text cursor selection cannot result in ambiguous attributes)
                    self.extendAttributes(mergedAttributes, preselectedAttributes);
                }
                break;

            case 'paragraph':
                selection.iterateContentNodes(function (paragraph) {
                    return mergeElementAttributes(paragraphStyles.getElementAttributes(paragraph));
                }, null, { maxIterations: maxIterations });
                break;

            case 'cell':
                selection.iterateTableCells(function (cell) {
                    return mergeElementAttributes(tableCellStyles.getElementAttributes(cell));
                });
                break;

            case 'table':
                if ((element = selection.getEnclosingTable())) {
                    mergeElementAttributes(tableStyles.getElementAttributes(element));
                }
                break;

            case 'drawing':
                var doMergeElementAttributes = false;
                // TODO: needs change when multiple drawings can be selected
                if ((element = selection.getSelectedDrawing()[0]) && DrawingFrame.isDrawingFrame(element)) {
                    doMergeElementAttributes = true;
                } else if ((element = selection.getSelectedTextFrameDrawing()[0]) && DrawingFrame.isDrawingFrame(element)) {
                    doMergeElementAttributes = true;
                }

                if (doMergeElementAttributes) {
                    if (DrawingFrame.isGroupDrawingFrame(element) && groupOnly !== true) {
                        _.each(DrawingFrame.getAllGroupDrawingChildren(element), function (drawing) {
                            mergeElementAttributes(drawingStyles.getElementAttributes(drawing));
                        });
                    } else {
                        mergeElementAttributes(drawingStyles.getElementAttributes(element));
                    }
                }
                break;

            default:
                Utils.error('Editor.getAttributes(): missing implementation for family "' + family + '"');
            }

            return mergedAttributes || {};
        };

        /**
         * Changes a single attribute of the specified attribute family in the
         * current selection.
         *
         * @param {String} family
         *  The name of the attribute family containing the specified
         *  attribute.
         *
         * @param {String} name
         *  the key of the attribute in the assigned family
         *
         * @param {Object} value
         *  the new attributes
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the selection, before applying the new attributes.
         */
        this.setAttribute = function (family, name, value, options) {
            this.setAttributes(family, Utils.makeSimpleObject(family, Utils.makeSimpleObject(name, value)), options);
        };

        /**
         * Changes multiple attributes of the specified attribute family in the
         * current selection.
         *
         * @param {String} family
         *  The name of the attribute family containing the passed attributes.
         *
         * @param {Object} attributes
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute families.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the selection while applying the new attributes.
         */
        this.setAttributes = function (family, attributes, options) {

            // Create an undo group that collects all undo operations generated
            // in the local setAttributes() method (it calls itself recursively
            // with smaller parts of the current selection).
            undoManager.enterUndoGroup(function () {

                var // table or drawing element contained by the selection
                    element = null,
                    // operations generator
                    generator = this.createOperationsGenerator(),
                    // the style sheet container
                    styleSheets = this.getStyleCollection(family),
                    // logical position
                    localPosition = null,
                    // another logical position
                    localDestPosition = null,
                    // the length of the paragraph
                    paragraphLength = null,
                    // whether after assigning cell attributes, it is still necessary to assign table attributes
                    createTableOperation = true,
                    // paragraph helper position and helper node
                    paraPos = null, paraNode = null,
                    // the old attributes, required for change tracking
                    oldAttrs = null,
                    // whether the modification can be change tracked
                    isTrackableChange = true,
                    // currently active root node
                    activeRootNode = self.getCurrentRootNode(),
                    // if we apply new style, do not remove manual pagebreaks
                    pageBreakBeforeAttr = false,
                    pageBreakAfterAttr = false,
                    // the old attributes, required for pageBreak paragraph style
                    oldParAttr = null,
                    // the promise for the asychronous execution of operations
                    operationsPromise = null,
                    // the promise for generating the operations
                    operationGeneratorPromise = null,
                    // the complete promise for generation and applying of operations
                    setAttributesPromise = null,
                    // the ratio of operation generation to applying of operation
                    operationRatio = 0.3,
                    // an optional array with selection ranges for large selections
                    splittedSelectionRange = null,
                    // a snapshot object
                    snapshot = null;

                /**
                 * Helper function to generate operations. This is done synchronously. But for asynchronous usage,
                 * the selection range can be restricted. This is especially useful for large selections.
                 *
                 * @param {Number[][]} [selectionRange]
                 *  An array with two logical positions for the start and the end position of a selection range.
                 */
                function generateAllSetAttributeOperations(selectionRange) {

                    var // the logical start position, if specified
                        localStartPos = (selectionRange && selectionRange[0]) || null,
                        // the logical end position, if specified
                        localEndPos = (selectionRange && selectionRange[1]) || null;

                    // generates a 'setAttributes' operation with the correct attributes
                    function generateSetAttributeOperation(startPosition, endPosition, oldAttrs) {

                        var // the options for the operation
                            operationOptions = { start: startPosition, attrs: _.clone(attributes) };

                        // adding the old attributes, author and date for change tracking
                        if (oldAttrs) {
                            oldAttrs = _.extend(oldAttrs, changeTrack.getChangeTrackInfo());
                            operationOptions.attrs.changes = { modified: oldAttrs };
                        }

                        // add end position if specified
                        if (_.isArray(endPosition)) {
                            operationOptions.end = endPosition;
                        }
                        // paragraph's style manual page break before
                        if (pageBreakBeforeAttr) {
                            operationOptions.attrs.paragraph.pageBreakBefore = true;
                            pageBreakBeforeAttr = false;
                        }
                        // paragraph's style manual page break after
                        if (pageBreakAfterAttr) {
                            operationOptions.attrs.paragraph.pageBreakAfter = true;
                            pageBreakAfterAttr = false;
                        }

                        // generate the 'setAttributes' operation
                        generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);
                    }

                    // setting a specified selection, if defined
                    // if (selectionRange) { selection.setTextSelection.apply(selection, selectionRange); }

                    // generate 'setAttribute' operations
                    switch (family) {

                    case 'character':
                        if (selection.hasRange()) {
                            selection.iterateContentNodes(function (paragraph, position, startOffset, endOffset) {

                                // validate start offset (iterator passes 'undefined' for fully covered paragraphs)
                                if (!_.isNumber(startOffset)) {
                                    startOffset = 0;
                                }
                                // validate end offset (iterator passes 'undefined' for fully covered paragraphs)
                                if (!_.isNumber(endOffset)) {
                                    endOffset = Position.getParagraphNodeLength(paragraph) - 1;
                                }

                                if (changeTrack.isActiveChangeTracking()) {

                                    // iterating over all children to send the old attributes in the operation
                                    // remove all empty text spans which have sibling text spans, and collect
                                    // sequences of sibling text spans (needed for white-space handling)
                                    Position.iterateParagraphChildNodes(paragraph, function (node, nodestart, nodelength, offsetstart, offsetlength) {

                                        // evaluate only text spans that are partly or completely covered by selection
                                        if (DOM.isTextSpan(node)) {
                                            // one operation for each span
                                            generateSetAttributeOperation(position.concat([nodestart + offsetstart]), position.concat([nodestart + offsetstart + offsetlength - 1]), changeTrack.getOldNodeAttributes(node));
                                        }

                                    }, undefined, { start: startOffset, end: endOffset });

                                } else {

                                    // set the attributes at the covered text range
                                    if (startOffset === 0 && endOffset <= 0 && DOM.isImplicitParagraphNode(paragraph)) {
                                        generator.generateOperation(Operations.PARA_INSERT, { start: position });  // generating paragraph
                                        generateSetAttributeOperation(position);  // assigning attribute to new paragraph
                                    } else if (startOffset === 0 && endOffset <= 0 && Position.getParagraphNodeLength(paragraph) === 0) {
                                        generateSetAttributeOperation(position);  // assigning attribute to existing empty paragraph
                                    } else if (startOffset <= endOffset) {
                                        generateSetAttributeOperation(position.concat([startOffset]), position.concat([endOffset]));
                                    }

                                }
                                spellChecker.reset(attributes, paragraph);
                            }, null, { startPos: localStartPos, endPos: localEndPos });
                        } else {  // selection has no range
                            if (Position.getParagraphLength(activeRootNode, _.clone(selection.getStartPosition())) === 0) {
                                // Task 28187: Setting character attributes at empty paragraphs
                                // In implSetAttributes it is additionally necessary to remove the character attributes
                                // from text span so that character styles at paragraph become visible

                                paraPos = _.clone(selection.getStartPosition());
                                paraPos.pop();
                                paraNode = Position.getParagraphElement(activeRootNode, paraPos);

                                // Defining the oldAttrs, if change tracking is active
                                if (changeTrack.isActiveChangeTracking()) {
                                    // Creating one setAttribute operation for each span inside the paragraph, because of old attributes!
                                    oldAttrs = {};
                                    if (DOM.isParagraphNode(paraNode)) { oldAttrs = changeTrack.getOldNodeAttributes(paraNode); }
                                }

                                if (DOM.isImplicitParagraphNode(paraNode)) { generator.generateOperation(Operations.PARA_INSERT, { start: paraPos }); }
                                // generating setAttributes operation
                                generateSetAttributeOperation(_.initial(_.clone(selection.getStartPosition())), undefined, oldAttrs);
                            } else {
                                // using preselected attributes for non-empty paragraphs
                                self.addPreselectedAttributes(attributes);
                            }
                        }
                        break;

                    case 'paragraph':
                        // deleted all changes for bug #26454#, because it did the opposite of word behavior, and that is not what the user expects! @see Bug 37174

                        selection.iterateContentNodes(function (paragraph, position) {
                            // Defining the oldAttrs, if change tracking is active
                            if (changeTrack.isActiveChangeTracking()) {
                                // Expanding operation for change tracking with old explicit attributes
                                oldAttrs = changeTrack.getOldNodeAttributes(paragraph);
                            }
                            // Preserve pageBreakBefore and/or pageBreakAfter attribute after setting new paragraph style
                            if (DOM.isManualPageBreakNode(paragraph)) {
                                oldParAttr = AttributeUtils.getExplicitAttributes(paragraph);
                                if (_.isObject(oldParAttr) && oldParAttr.paragraph) {
                                    if (oldParAttr.paragraph.pageBreakBefore === true) {
                                        pageBreakBeforeAttr = true;
                                    }
                                    if (oldParAttr.paragraph.pageBreakAfter === true) {
                                        pageBreakAfterAttr = true;
                                    }
                                }

                            }

                            // generating a new paragraph, if it is implicit
                            if (DOM.isImplicitParagraphNode(paragraph)) { generator.generateOperation(Operations.PARA_INSERT, { start: position }); }
                            // generating setAttributes operation
                            generateSetAttributeOperation(position, undefined, oldAttrs);
                        }, null, { startPos: localStartPos, endPos: localEndPos });
                        break;

                    case 'cell':
                        selection.iterateTableCells(function (cell, position) {
                            // Defining the oldAttrs, if change tracking is active
                            if (changeTrack.isActiveChangeTracking()) {
                                // Expanding operation for change tracking with old explicit attributes
                                oldAttrs = changeTrack.getOldNodeAttributes(cell);
                            }
                            generateSetAttributeOperation(position, undefined, oldAttrs);
                        });
                        break;

                    case 'table':
                        if ((element = selection.getEnclosingTable())) {

                            localPosition = Position.getOxoPosition(activeRootNode, element, 0);  // the logical position of the table

                            if (Utils.getBooleanOption(options, 'clear', false)) {
                                // removing hard attributes at tables and cells, so that the table style will be visible
                                Table.removeTableAttributes(self, activeRootNode, element, localPosition, generator);
                            }

                            if (Utils.getBooleanOption(options, 'onlyVisibleBorders', false)) {
                                // setting border width directly at cells -> overwriting values of table style
                                Table.setBorderWidthToVisibleCells(element, attributes, tableCellStyles, activeRootNode, generator, self);
                                createTableOperation = false;
                            }

                            if (Utils.getBooleanOption(options, 'cellSpecificTableAttribute', false)) {
                                // setting border mode directly at cells -> overwriting values of table style
                                Table.setBorderModeToCells(element, attributes, TableStyles.getBorderStyleFromAttributes(self.getAttributes('cell').cell || {}), activeRootNode, generator, self);
                            }

                            // setting attributes to the table element
                            // -> also setting table attributes, if attributes are already assigned to table cells, to
                            // keep the getter functions simple and performant
                            if (createTableOperation) {
                                // Defining the oldAttrs, if change tracking is active
                                if (changeTrack.isActiveChangeTracking()) {
                                    // Expanding operation for change tracking with old explicit attributes
                                    oldAttrs = changeTrack.getOldNodeAttributes(element);
                                }
                                generateSetAttributeOperation(localPosition, undefined, oldAttrs);
                            }
                        }
                        break;

                    case 'drawing':
                        if (self.isDrawingSelected()) {
                            element = selection.getSelectedDrawing()[0];
                        } else if (self.getSelection().isAdditionalTextframeSelection()) {
                            element = selection.getSelectedTextFrameDrawing()[0];
                        } else {
                            break;
                        }
                        // TODO: needs change when multiple drawings can be selected
                        // TODO: this fails if a drawing style sheet changes the inline/floating mode instead of explicit attributes
                        if (DrawingFrame.isDrawingFrame(element)) {

                            localPosition = Position.getOxoPosition(activeRootNode, element, 0);

                            if (_.isObject(attributes.drawing)) {

                                // when switching from inline to floated, saving current position in the drawing, so that it can
                                // be set correctly when switching back to inline.
                                // This is only necessary, if the drawing was moved in that way, that implMove needed to be called.
                                if ((attributes.drawing.inline === false) && (DOM.isInlineDrawingNode(element))) {
                                    $(element).data('inlinePosition', localPosition[localPosition.length - 1]);
                                    // Fixing vertical offset, if drawing is set from inline to floated (31722)
                                    if ((attributes.drawing.anchorVertBase) && (attributes.drawing.anchorVertBase === 'paragraph') &&
                                        (_.isNumber(attributes.drawing.anchorVertOffset)) && (attributes.drawing.anchorVertOffset === 0) &&
                                        (($(element).offset().top - $(element).closest(DOM.PARAGRAPH_NODE_SELECTOR).offset().top > 0))) {
                                        attributes.drawing.anchorVertOffset = Utils.convertLengthToHmm($(element).offset().top - $(element).closest(DOM.PARAGRAPH_NODE_SELECTOR).offset().top, 'px');
                                    }
                                }

                                // when switching from floated to inline, a move of the drawing might be necessary
                                if ((attributes.drawing.inline === true) && (DOM.isFloatingDrawingNode(element)) && ($(element).data('inlinePosition'))) {

                                    localDestPosition = _.clone(localPosition);
                                    paragraphLength = Position.getParagraphLength(activeRootNode, localDestPosition);
                                    if ((paragraphLength - 1) < $(element).data('inlinePosition')) {  // -> is this position still valid?
                                        localDestPosition[localDestPosition.length - 1] = paragraphLength - 1;
                                    } else {
                                        localDestPosition[localDestPosition.length - 1] = $(element).data('inlinePosition');
                                    }
                                    if (!_.isEqual(localPosition, localDestPosition)) {
                                        undoManager.enterUndoGroup(function () {

                                            var // the logical position of the paragraph
                                                paraPos = _.clone(localDestPosition),
                                                // the paragraph node
                                                paraNode = null;

                                            paraPos.pop();
                                            paraNode = Position.getParagraphElement(activeRootNode, paraPos);

                                            if (DOM.isImplicitParagraphNode(paraNode)) { generator.generateOperation(Operations.PARA_INSERT, { start: paraPos }); }
                                            generator.generateOperation(Operations.MOVE, { start: localPosition, end: localPosition, to: localDestPosition });

                                            localPosition = _.clone(localDestPosition);
                                        }, this);
                                    }
                                }

                                // setting new position (inline, paragraph, ...) is not change tracked
                                if (!changeTrack.ctSupportsDrawingPosition()) { isTrackableChange = false; }
                            }

                            // setting border and fillcolor of drawing is not change tracked
                            if (_.isObject(attributes.fill) || _.isObject(attributes.line)) { isTrackableChange = false; }

                            // Defining the oldAttrs, if change tracking is active
                            // -> and if change track supports changing of drawing position
                            if (changeTrack.isActiveChangeTracking() && isTrackableChange) {
                                // Expanding operation for change tracking with old explicit attributes
                                oldAttrs = changeTrack.getOldNodeAttributes(element);
                            }

                            // by setting fill- or line-attributes on a group
                            if (DrawingFrame.isGroupDrawingFrame(element) && (_.isObject(attributes.fill) || _.isObject(attributes.line))) {
                                // generate operations for all children
                                _.each(DrawingFrame.getAllGroupDrawingChildren(element), function (ele) {
                                    generateSetAttributeOperation(Position.getOxoPosition(activeRootNode, ele, 0));
                                });
                            // otherwise
                            } else {
                                // set attributes on the group
                                generateSetAttributeOperation(localPosition, undefined, oldAttrs);
                            }

                            $(element).closest(DOM.PARAGRAPH_NODE_SELECTOR).removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated

                        }
                        break;

                    default:
                        Utils.error('Editor.setAttributes(): missing implementation for family "' + family + '"');
                    }
                }

                /**
                 * Helper function to apply the generated operations asynchronously.
                 *
                 * @returns {jQuery.Promise}
                 *  A promise that will be resolved when the operations have been applied.
                 */
                function doSetAttributesAsync() {

                    operationsPromise = applyTextOperationsAsync(generator, null, { showProgress: false, leaveOnSuccess: true, progressStart: operationRatio });

                    // restore the original selection after successful apply of operations
                    operationsPromise
                    .done(function () {
                        selection.restoreBrowserSelection();
                    });

                    return operationsPromise;
                }

                // add all attributes to be cleared
                if (Utils.getBooleanOption(options, 'clear', false)) {
                    if (family !== 'table') {  // special behaviour for tables follows below
                        attributes = self.extendAttributes(styleSheets.buildNullAttributes(), attributes);
                    }
                }

                // nothig to do if no attributes will be changed
                if (_.isEmpty(attributes)) { return; }

                // register pending style sheet via 'insertStyleSheet' operation
                if (_.isString(attributes.styleId) && styleSheets.isDirty(attributes.styleId)) {
                    self.generateInsertStyleOp(generator, family, attributes.styleId);
                }

                // checking selection size -> make array with splitted selection [0,100], [101, 200], ... (39061)
                splittedSelectionRange = Position.splitLargeSelection(self.getCurrentRootNode(), selection.getStartPosition(), selection.getEndPosition(), MAX_TOP_LEVEL_NODES);

                if (splittedSelectionRange && splittedSelectionRange.length > 1) {

                    // make sure that only one asynchronous call is processed at the same time
                    if (checkSetClipboardPasteInProgress()) { return $.when(); }

                    // blocking keyboard input during generation and applying of operations
                    self.setBlockKeyboardEvent(true);

                    // creating a snapshot
                    snapshot = new Snapshot(app);

                    // show a message with cancel button
                    // -> immediately grabbing the focus, after calling enterBusy. This guarantees, that the
                    // keyboard blocker works. Otherwise the keyboard events will not be catched by the page.
                    app.getView().enterBusy({
                        cancelHandler: function () {
                            if (operationsPromise && operationsPromise.abort) { // order is important, the latter has to win
                                // restoring the old document state
                                snapshot.apply();
                                // calling abort function for operation promise
                                app.enterBlockOperationsMode(function () { operationsPromise.abort(); });
                            } else if (operationGeneratorPromise && operationGeneratorPromise.abort) {
                                operationGeneratorPromise.abort();  // no undo of changes required
                            }
                        },
                        immediate: true,
                        warningLabel: gt('Sorry, formatting content will take some time.')
                    }).grabFocus();

                    // generate operations asynchronously
                    operationGeneratorPromise = self.iterateArraySliced(splittedSelectionRange, function (oneRange) {
                        // calling function to generate operations synchronously with reduced selection range
                        generateAllSetAttributeOperations(oneRange);
                    }, { delay: 'immediate', infoString: 'Text: generateAllSetAttributeOperations' })
                        // add progress handling
                        .progress(function (progress) {
                            // update the progress bar according to progress of the operations promise
                            app.getView().updateBusyProgress(operationRatio * progress);
                        })
                        .fail(function () {
                            // leaving the busy mode during creation of operations -> no undo required
                            leaveAsyncBusy();
                        });

                    setAttributesPromise = operationGeneratorPromise.then(doSetAttributesAsync).always(function () {
                        if (snapshot) { snapshot.destroy(); }
                    });

                } else {
                    // synchronous handling for small selections

                    // calling function to generate operations synchronously
                    generateAllSetAttributeOperations();

                    // apply all collected operations
                    self.applyOperations(generator);

                    // applying operations synchronously
                    setAttributesPromise = $.when();
                }

                return setAttributesPromise;

            }, this); // enterUndoGroup();
        };

        /**
         * Changes multiple attributes of the specified attribute family for
         * the passed table node. Different from setAttributes in a way that uses
         * passed node instead of selected node with selection.
         *
         * @param {Node|jQuery} tableNode
         *  Table node to which attributes are applied
         *
         * @param {String} family
         *  The name of the attribute family containing the passed attributes.
         *
         * @param {Object} attributes
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute families.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the selection while applying the new attributes.
         *  @param {Boolean} [options.onlyVisibleBorders=false]
         *      Setting border width directly at cells -> overwriting values of table style.
         *  @param {Boolean} [options.cellSpecificTableAttribute=false]
         *      Setting border mode directly at cells -> overwriting values of table style.
         *
         */
        this.setAttributesToPassedTableNode = function (tableNode, family, attributes, options) {

            // Create an undo group that collects all undo operations generated
            // in the local setAttributes() method (it calls itself recursively
            // with smaller parts of the current selection).
            undoManager.enterUndoGroup(function () {

                var
                    // logical position
                    localPosition = null,
                    // operations generator
                    generator = this.createOperationsGenerator(),
                    // whether after assigning cell attributes, it is still necessary to assign table attributes
                    createTableOperation = true,
                    // the old attributes, required for change tracking
                    oldAttrs = null;

                // generates a 'setAttributes' operation with the correct attributes
                function generateSetAttributeOperation(startPosition, endPosition, oldAttrs) {

                    var // the options for the operation
                        operationOptions = { start: startPosition, attrs: _.clone(attributes) };

                    // adding the old attributes, author and date for change tracking
                    if (oldAttrs) {
                        oldAttrs = _.extend(oldAttrs, changeTrack.getChangeTrackInfo());
                        operationOptions.attrs.changes = { modified: oldAttrs };
                    }

                    // add end position if specified
                    if (_.isArray(endPosition)) {
                        operationOptions.end = endPosition;
                    }

                    // generate the 'setAttributes' operation
                    generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);
                }

                // nothig to do if no attributes will be changed
                if (_.isEmpty(attributes)) { return; }

                if (tableNode) {

                    localPosition = Position.getOxoPosition(editdiv, tableNode, 0);  // the logical position of the table

                    if (Utils.getBooleanOption(options, 'clear', false)) {
                        // removing hard attributes at tables and cells, so that the table style will be visible
                        Table.removeTableAttributes(self, editdiv, tableNode, localPosition, generator);
                    }

                    if (Utils.getBooleanOption(options, 'onlyVisibleBorders', false)) {
                        // setting border width directly at cells -> overwriting values of table style
                        Table.setBorderWidthToVisibleCells(tableNode, attributes, tableCellStyles, editdiv, generator, self);
                        createTableOperation = false;
                    }

                    if (Utils.getBooleanOption(options, 'cellSpecificTableAttribute', false)) {
                        // setting border mode directly at cells -> overwriting values of table style
                        Table.setBorderModeToCells(tableNode, attributes, TableStyles.getBorderStyleFromAttributes(self.getAttributes('cell').cell || {}), editdiv, generator, self);
                    }

                    // setting attributes to the table element
                    // -> also setting table attributes, if attributes are already assigned to table cells, to
                    // keep the getter functions simple and performant
                    if (createTableOperation) {
                        // Defining the oldAttrs, if change tracking is active
                        if (changeTrack.isActiveChangeTracking()) {
                            // Expanding operation for change tracking with old explicit attributes
                            oldAttrs = changeTrack.getOldNodeAttributes(tableNode);
                        }
                        generateSetAttributeOperation(localPosition, undefined, oldAttrs);
                    }
                }

                // apply all collected operations
                this.applyOperations(generator);

            }, this); // end of enterUndoGroup
        };

        /**
         * Removes all hard-set attributes depending on the current selection.
         * A selection range clears only hard-set character attributes and without
         * a range the hard-set paragraph attributes are cleared.
         *
         */
        this.resetAttributes = function () {

            function clearParagraphAttributes() {
                var paraAttributes = paragraphStyles.buildNullAttributes();
                delete paraAttributes.styleId;
                self.setAttributes('paragraph', paraAttributes);
            }

            function clearCharacterAttributes() {
                var charAttributes = characterStyles.buildNullAttributes();
                // don't reset hyperlink attribute
                delete charAttributes.styleId;
                delete charAttributes.character.url;
                self.setAttributes('character', charAttributes);
            }

            if (!selection.hasRange()) {
                undoManager.enterUndoGroup(function () {

                    var start = selection.getStartPosition(),
                        end = selection.getEndPosition(),
                        activeRootNode = self.getCurrentRootNode(),
                        firstParagraph = Position.getLastNodeFromPositionByNodeName(activeRootNode, start, DOM.PARAGRAPH_NODE_SELECTOR),
                        lastParagraph = Position.getLastNodeFromPositionByNodeName(activeRootNode, end, DOM.PARAGRAPH_NODE_SELECTOR),
                        prevPara,
                        nextPara;

                    clearParagraphAttributes();
                    prevPara = Utils.findPreviousNode(activeRootNode, firstParagraph, DOM.PARAGRAPH_NODE_SELECTOR);
                    if (prevPara) {
                        paragraphStyles.updateElementFormatting(prevPara);
                    }
                    nextPara = Utils.findNextNode(activeRootNode, lastParagraph, DOM.PARAGRAPH_NODE_SELECTOR);
                    if (nextPara) {
                        paragraphStyles.updateElementFormatting(nextPara);
                    }
                    self.addPreselectedAttributes(characterStyles.buildNullAttributes());
                }, this);
            } else {
                clearCharacterAttributes();
            }
        };

        /**
         * Returns the current selection.
         */
        this.getSelection = function () {
            return selection;
        };

        /**
         * Returns the remote selection.
         */
        this.getRemoteSelection = function () {
            return remoteSelection;
        };

        /**
         * Returns whether the current selection selects any text. This
         * includes the rectangular table cell selection mode.
         */
        this.isTextSelected = function () {
            return selection.getSelectionType() !== 'drawing';
        };

        /**
         * Returns whether the editor contains a selection range (text,
         * drawings, or table cells) instead of a simple text cursor.
         */
        this.hasSelectedRange = function () {
            return !selection.isTextCursor();
        };

        /**
         * Returns whether the editor contains a selection within a
         * single paragraph or not.
         */
        this.hasEnclosingParagraph = function () {
            return selection.getEnclosingParagraph() !== null;
        };

        // PUBLIC TABLE METHODS

        /**
         * Returns whether the editor contains a selection within a
         * table or not.
         */
        this.isPositionInTable = function () {
            return !_.isNull(selection.getEnclosingTable());
        };

        /**
         * Returns whether the editor contains a selection over one or multiple
         * cells within a table
         */
        this.isCellRangeSelected = function () {
            // Firefox has beautiful cell-range-selection
            if (_.browser.Firefox) {
                var currentSelection = selection.getBrowserSelection();
                return DOM.isCellRangeSelected(currentSelection.active.start.node, currentSelection.active.end.node);

            // every other browser has to check the selection
            } else {
                return Position.positionsInTowCellsInSameTable(self.getCurrentRootNode(), selection.getStartPosition(), selection.getEndPosition());
            }
        };

        /**
         * Returns the number of columns, if the editor contains a selection within a
         * table.
         */
        this.getNumberOfColumns = function () {
            var table = selection.getEnclosingTable();
            return table && Table.getColumnCount(table);
        };

        /**
         * Returns the number of rows, if the editor contains a selection within a
         * table.
         */
        this.getNumberOfRows = function () {
            var table = selection.getEnclosingTable();
            return table && Table.getRowCount(table);
        };

        /**
         * Returns whether a further row can be added, if the editor contains a
         * selection within a table. The maximum number of rows and cells can
         * be defined in the configuration.
         */
        this.isRowAddable = function () {

            var // the number of rows in the current table
                rowCount = this.getNumberOfRows(),
                // the number of columns in the current table
                colCount = this.getNumberOfColumns();

            // check that cursor is located in a table, check maximum row count and maximum cell count
            return _.isNumber(rowCount) && _.isNumber(colCount) &&
                (rowCount < Config.MAX_TABLE_ROWS) &&
                ((rowCount + 1) * colCount <= Config.MAX_TABLE_CELLS);
        };

        /**
         * Returns whether a further column can be added, if the editor contains a selection within a
         * table. The maximum number of columns and cells can be defined in the configuration.
         */
        this.isColumnAddable = function () {

            var // the number of rows in the current table
                rowCount = this.getNumberOfRows(),
                // the number of columns in the current table
                colCount = this.getNumberOfColumns();

            // check that cursor is located in a table, check maximum column count and maximum cell count
            return _.isNumber(rowCount) && _.isNumber(colCount) &&
                (colCount < Config.MAX_TABLE_COLUMNS) &&
                ((colCount + 1) * rowCount <= Config.MAX_TABLE_CELLS);
        };

        // PUBLIC DRAWING METHODS

        /**
         * Returns whether the current selection selects one or more drawings.
         */
        this.isDrawingSelected = function () {
            return selection.getSelectionType() === 'drawing';
        };

        /**
         * Returns whether the current selection is a text selection inside a shape
         * in an odf document. These 'shapes with text' need to be distinguished from
         * 'classical' text frames in odf format, because much less functionality is
         * available.
         */
        this.isReducedOdfTextframeFunctionality = function () {
            return app.isODF() && selection.isAdditionalTextframeSelection() && DrawingFrame.isReducedOdfTextframeNode(selection.getSelectedTextFrameDrawing());
        };

        /**
         * Returns whether the current selection is a text selection inside a 'classic'
         * text frame in an odf document. These 'classic' text frames need to be
         * distinguished from 'shapes with text', because more functionality is
         * available.
         */
        this.isFullOdfTextframeFunctionality = function () {
            return app.isODF() && selection.isAdditionalTextframeSelection() && DrawingFrame.isFullOdfTextframeNode(selection.getSelectedTextFrameDrawing());
        };

        /**
         * Returns whether the current selection is a selection inside a comment
         * in an odf document.
         */
        this.isOdfCommentFunctionality = function () {
            return app.isODF() && self.isCommentFunctionality();
        };

        /**
         * Returns whether the current selection is a selection inside a comment.
         */
        this.isCommentFunctionality = function () {
            return self.getActiveTarget() && commentLayer.getCommentRootNode(self.getActiveTarget());
        };

        /**
         * Returns whether the current selection is a selection inside a comment or a header.
         */
        this.isTargetFunctionality = function () {
            return self.getActiveTarget() && (commentLayer.getCommentRootNode(self.getActiveTarget()) || self.isHeaderFooterEditState());
        };

        /**
         * Returns whether the current selection selects text, not cells and
         * not drawings.
         */
        this.isTextOnlySelected = function () {
            return selection.getSelectionType() === 'text';
        };

        /**
         * Returns whether clipboard paste is in progress.
         *
         * @returns {Boolean}
         */
        this.isClipboardPasteInProgress = function () {
            return pasteInProgress;
        };

        /**
         * Returns the default lateral heading character styles
         */
        this.getDefaultHeadingCharacterStyles = function () {
            return HEADINGS_CHARATTRIBUTES;
        };

        /**
         * Returns the default lateral paragraph style
         */
        this.getDefaultParagraphStyleDefinition = function () {
            return DEFAULT_PARAGRAPH_DEFINTIONS;
        };

        /**
         * Returns the default lateral table definiton
         */
        this.getDefaultLateralTableDefinition = function () {
            return DEFAULT_LATERAL_TABLE_DEFINITIONS;
        };

        /**
         * Returns the default lateral table attributes
         */
        this.getDefaultLateralTableAttributes = function () {
            return DEFAULT_LATERAL_TABLE_ATTRIBUTES;
        };

        this.getDefaultLateralHyperlinkDefinition = function () {
            return DEFAULT_HYPERLINK_DEFINTIONS;
        };

        this.getDefaultLateralHyperlinkAttributes = function () {
            return DEFAULT_HYPERLINK_CHARATTRIBUTES;
        };

        /**
         * Returns the default lateral drawing attributes
         */
        this.getDefaultDrawingDefintion = function () {
            return DEFAULT_DRAWING_DEFINITION;
        };

        this.getDefaultDrawingAttributes = function () {
            return DEFAULT_DRAWING_ATTRS;
        };

        this.getDefaultDrawingTextFrameDefintion = function () {
            return DEFAULT_DRAWING_TEXTFRAME_DEFINITION;
        };

        this.getDefaultDrawingTextFrameAttributes = function () {
            return DEFAULT_DRAWING_TEXTFRAME_ATTRS;
        };

        this.getDefaultCommentTextDefintion = function () {
            return DEFAULT_LATERAL_COMMENT_DEFINITIONS;
        };

        this.getDefaultHeaderTextDefinition = function () {
            return DEFAULT_LATERAL_HEADER_DEFINITIONS;
        };

        this.getDefaultFooterTextDefinition = function () {
            return DEFAULT_LATERAL_FOOTER_DEFINITIONS;
        };

        this.getDefaultCommentTextAttributes = function () {
            return DEFAULT_LATERAL_COMMENT_ATTRIBUTES;
        };

        /**
         * Returns the document default paragraph stylesheet id
         */
        this.getDefaultUIParagraphStylesheet = function () {
            var styleNames = [],
                styleId = null;

            if (app.isODF()) {
                styleNames = paragraphStyles.getStyleSheetNames();
                _(styleNames).each(function (name, id) {
                    var lowerId = id.toLowerCase();
                    if (lowerId === 'standard' || lowerId === 'normal') {
                        styleId = id;
                    }
                });

                if (!styleId) {
                    styleId = paragraphStyles.getDefaultStyleId();
                }
            } else {
                styleId = paragraphStyles.getDefaultStyleId();
            }

            return styleId;
        };

        /**
         * Returns the document default table stylesheet id
         * which can be used to set attributes for a new
         * table.
         */
        this.getDefaultUITableStylesheet = function () {
            var styleNames = tableStyles.getStyleSheetNames(),
                highestUIPriority = 99,
                tableStyleId = null;

            _(styleNames).each(function (name, id) {
                var uiPriority = tableStyles.getUIPriority(id);

                if (uiPriority && (uiPriority < highestUIPriority)) {
                    tableStyleId = id;
                    highestUIPriority = uiPriority;
                }
            });

            return tableStyleId;
        };

        /**
         * Returns the document default hyperlink stylesheet id
         */
        this.getDefaultUIHyperlinkStylesheet = function () {
            var styleNames = characterStyles.getStyleSheetNames(),
                hyperlinkId = null;

            _(styleNames).find(function (name, id) {
                var lowerName = name.toLowerCase();
                if (lowerName.indexOf('hyperlink') >= 0) {
                    hyperlinkId = id;
                    return true;
                }
                return false;
            });

            return hyperlinkId;
        };

        /**
         * Returns the document default paragraph list stylesheet id
         */
        this.getDefaultUIParagraphListStylesheet = function () {
            var styleNames = paragraphStyles.getStyleSheetNames(),
                 paragraphListId = null;

            _(styleNames).find(function (name, id) {
                var lowerName = name.toLowerCase();
                if (lowerName.indexOf('list paragraph') === 0 || lowerName.indexOf('listparagraph') === 0) {
                    paragraphListId = id;
                    return true;
                }
            });

            return paragraphListId;
        };

        /**
         * Optimizing the actions before they are sent to the server. Especially
         * TEXT_INSERT operations can be merged. This function is a callback
         * that is started from 'sendActions' in the application.
         *
         * @param {Array} actions
         *  An array containing all actions that are investigated for optimization.
         *
         * @returns {Array}
         *  An array with optimized actions.
         */
        this.optimizeActions = function (actions) {

            var newActions = [];

            _.each(actions, function (next) {

                var insertNextOp = true,
                    last = _.last(newActions);

                if ((last) && (last.operations) && (last.operations.length === 1) && (next) && (next.operations) && (next.operations.length === 1)) {

                    if ((last.operations[0].name === Operations.TEXT_INSERT) && (next.operations[0].name === Operations.TEXT_INSERT)) {

                        // trying to merge two following text insert operations
                        if (Position.hasSameParentComponent(last.operations[0].start, next.operations[0].start) && Position.hasSameTargetComponent(last.operations[0].target, next.operations[0].target)) {

                            // check that the operations are valid for merging. This is the case, if the next operation has no attributes and the previous
                            // operation is not a change track operation, or if the following two operations both are change track operations.
                            if (mergeNeighboringOperations(last.operations[0], next.operations[0])) {
                                // check that the 'next' action adds the character directly after the 'last' action
                                if ((_.last(last.operations[0].start) + last.operations[0].text.length === _.last(next.operations[0].start))) {
                                    // merge insert operation (add the characters)
                                    last.operations[0].text += next.operations[0].text;
                                    // increasing the operation length
                                    if ((last.operations[0].opl) && (next.operations[0].opl)) {
                                        last.operations[0].opl += next.operations[0].opl;
                                    }
                                    insertNextOp = false;
                                } else if ((_.last(last.operations[0].start) <= _.last(next.operations[0].start)) && (_.last(next.operations[0].start) <= _.last(last.operations[0].start) + last.operations[0].text.length)) {
                                    // the new text is inserted into an already existing text
                                    last.operations[0].text = last.operations[0].text.slice(0, _.last(next.operations[0].start) - _.last(last.operations[0].start)) + next.operations[0].text + last.operations[0].text.slice(_.last(next.operations[0].start) - _.last(last.operations[0].start));
                                    // increasing the operation length
                                    if ((last.operations[0].opl) && (next.operations[0].opl)) {
                                        last.operations[0].opl += next.operations[0].opl;
                                    }
                                    insertNextOp = false;
                                }
                            }
                        }
                    }
                }

                if (insertNextOp) {
                    newActions.push(next);
                }

            });

            return newActions;
        };

        /**
         * Central place to make changes to all sychnronously applied operations. Before the generated operations
         * are applied to the operation handler, this is the point, where global changes for all operations can be
         * made.
         *
         * @param {Array} operations
         *  An array containing all operations that are investigated to be finalized.
         */
        function finalizeOperations(operations) {

            _.each(operations, function (operation) {

                // adding the target to each operation, if it is not already added to the operation
                // and if it is a non-empty string
                if (operation.name && operation.start && !operation.target && activeTarget && !undoRedoRunning) {
                    operation.target = activeTarget;
                }
            });
        }

        /**
         * Load performance: Setting the document operation state number of a document that is currently
         * loaded using the local storage.
         *
         * @param {Number} osn
         *  The document operation state number of the currently loaded document.
         */
        this.setLoadDocumentOsn = function (osn) {
            docLoadOSN = osn;
        };

        /**
         * Setting the global property 'guiTriggeredOperation'.
         *
         * @param {Boolean} state
         *  Whether the running process was triggered locally via GUI.
         */
        this.setGUITriggeredOperation = function (state) {
            guiTriggeredOperation = state;
        };

        /**
         * Load performance: Filtering operations corresponding to the operation name.
         *
         * @param {Object} operation
         *  An operation object.
         *
         * @returns {Boolean}
         *  Whether the specified operation is listed in the array of operations, that are required, even
         *  if the document is loaded from the local storage. Furthermore all operations must be executed,
         *  if their operation state number is larger the operation state number of the saved document.
         *  If 'true' is returned, the operation must be executed. If this is not necessary, 'false' is returned.
         */
        this.operationFilter = function (operation) {
            var forceOperationExecution = (operation && operation.osn && docLoadOSN && (operation.osn >= docLoadOSN));
            return (forceOperationExecution || (operation && operation.name && REQUIRED_LOAD_OPERATIONS[operation.name]));
        };

        /**
         * Handler for loading empty document with fast load. It applies html string, then actions,
         * formats document synchronosly, calculates page breaks, and at the end, leaves busy mode.
         *
         * @param {String} markup
         *  The HTML mark-up to be shown as initial document contents.
         *
         * @param {Array} actions
         *  The operation actions to be applied to finalize the fast import.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the actions have been applied.
         */
        this.fastEmptyLoadHandler = function (markup, actions) {
            // markup is in form of object, parse and get data from mainDocument
            markup = JSON.parse(markup);
            self.setFullModelNode(markup.mainDocument);

            return self.applyActions(actions, { external: true, useStorageData: false }).done(function () {
                self.firstPartOfDocumentFormatting();
                pageLayout.callInitialPageBreaks();
            });
        };

        /**
         * Load performance: Execute post process activities after loading the document from the
         * local storage.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  formatting has been updated successfully, or rejected when an error
         *  has occurred.
         */
        this.updateDocumentFormattingStorage = function () {

            // initialize default page formatting
            pageStyles.updateElementFormatting(editdiv);

            // handling page breaks correctly
            pageAttributes = pageStyles.getElementAttributes(editdiv);
            pageMaxHeight = Utils.convertHmmToLength(pageAttributes.page.height - pageAttributes.page.marginTop - pageAttributes.page.marginBottom, 'px', 1);
            pagePaddingLeft = Utils.convertHmmToLength(pageAttributes.page.marginLeft, 'px', 1);
            pagePaddingTop = Utils.convertHmmToLength(pageAttributes.page.marginTop, 'px', 1);
            pagePaddingBottom = Utils.convertHmmToLength(pageAttributes.page.marginBottom, 'px', 1);
            pageWidth = Utils.convertHmmToLength(pageAttributes.page.width, 'px', 1);

            // receiving list of all change track authors of the document
            this.triggerExtendAuthors(this.getDocumentAttributes().changeTrackAuthors);

            if (editdiv.find(DrawingFrame.BORDER_NODE_SELECTOR).length > 0) {
                this.trigger('drawingHeight:update', this.getNode().find(DrawingFrame.BORDER_NODE_SELECTOR));
            }

            if (pageLayout.hasContentHeaderFooterPlaceHolder() || !drawingLayer.isEmpty()) {
                pageLayout.updateStartHeaderFooterStyle();
                pageLayout.headerFooterCollectionUpdate();
                pageLayout.callInitialPageBreaks();
            }

            return $.when();
        };

        /**
         * Updates the formatting of all elements, after the import operations
         * of the document have been applied.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  formatting has been updated successfully, or rejected when an error
         *  has occurred.
         */
        this.updateDocumentFormatting = function () {

            var // the result Deferred object
                def = $.Deferred(),
                // the content node of the page
                pageContentNode = DOM.getPageContentNode(editdiv),
                // the comment layer node of the page
                commentLayerNode = DOM.getCommentLayerNode(editdiv),
                // all top-level paragraphs, all tables (also embedded), and all drawings
                formattingNodes = pageContentNode.find('> ' + DOM.PARAGRAPH_NODE_SELECTOR + ', ' + DrawingFrame.NODE_SELECTOR + ', ' + DOM.TABLE_NODE_SELECTOR),
                // all paragraphs, tables and drawings inside comments
                commentContentNodes = commentLayerNode.children(DOM.COMMENTTHREADNODE_SELECTOR).children(DOM.COMMENTNODE_SELECTOR).find(DOM.PARAGRAPH_NODE_SELECTOR + ', ' + DrawingFrame.NODE_SELECTOR + ', ' + DOM.TABLE_NODE_SELECTOR),
                // number of progress ticks for tables (1 tick per paragraph/drawing)
                TABLE_PROGRESS_TICKS = 20,
                // total number of progress ticks
                totalProgressTicks = formattingNodes.length + (TABLE_PROGRESS_TICKS - 1) * formattingNodes.filter(DOM.TABLE_NODE_SELECTOR).length,
                // current progress for formatting process
                currentProgressTicks = 0,
                // update progress bar at most 250 times
                progressBlockSize = totalProgressTicks / 250,
                // index of progress block from last progress update
                lastProgressBlock = -1,
                // marker for geting nodes for displaying imediately on document loading
                splitPoint,
                // header and footer container nodes needs to be updated also, if there is content inside them
                headerFooterFormattingNodes = pageLayout.getHeaderFooterPlaceHolder().find(DOM.PARAGRAPH_NODE_SELECTOR + ', ' + DrawingFrame.NODE_SELECTOR + ', ' + DOM.TABLE_NODE_SELECTOR),
                // fast display first page -> initialPageBreaks is called 2x, abort first promise when second is triggered
                initialPbPromise,
                //
                splitListNodes,
                indexSplitPoint;

            pageAttributes = pageStyles.getElementAttributes(self.getNode());
            pageMaxHeight = Utils.convertHmmToLength(pageAttributes.page.height - pageAttributes.page.marginTop - pageAttributes.page.marginBottom, 'px', 1);
            pagePaddingLeft = Utils.convertHmmToLength(pageAttributes.page.marginLeft, 'px', 1);
            pagePaddingTop = Utils.convertHmmToLength(pageAttributes.page.marginTop, 'px', 1);
            pagePaddingBottom = Utils.convertHmmToLength(pageAttributes.page.marginBottom, 'px', 1);
            pageWidth = Utils.convertHmmToLength(pageAttributes.page.width, 'px', 1);

            // #27818 - On loading document with table, table style is missing for proper formatting
            if (app.isODF()) {
                insertMissingTableStyles();
            }

            // add content of comment nodes
            formattingNodes = formattingNodes.add(commentContentNodes);

            // first 90% for formatting (last 10% for updating lists)
            function updateFormatProgress(progressTicks) {
                var progressBlock = Math.floor(progressTicks / progressBlockSize);
                if (lastProgressBlock < progressBlock) {
                    def.notify(0.9 * progressTicks / totalProgressTicks);
                    lastProgressBlock = progressBlock;
                }
            }

            // updates the formatting of the passed element, returns a promise
            function updateElementFormatting(element) {

                var // the Promise for asynchronous element formatting
                    promise = null,
                    // whether the element is a direct child of the page content node
                    topLevel = !element.parentNode;

                // convert element to jQuery object
                element = $(element);

                // insert the detached element into the page content node
                // (performance optimization for large documents, see below)
                if (topLevel) { pageContentNode.append(element); }

                // determine the type of the passed element
                if (DOM.isParagraphNode(element)) {
                    // validate DOM contents AND update formatting of the paragraph
                    validateParagraphNode(element);
                    paragraphStyles.updateElementFormatting(element);
                    updateFormatProgress(currentProgressTicks += 1);

                } else if (DrawingFrame.isDrawingFrame(element)) {

                    // TODO: validate DOM of paragraphs embedded in text frames
                    drawingStyles.updateElementFormatting(element);
                    updateFormatProgress(currentProgressTicks += 1);
                    if (Utils.SMALL_DEVICE && $(element).hasClass('float')) {
                        //verticalOffsetNode.remove();
                        $(element).parent().find('.float.offset').remove();

                        $(element).removeClass('float left right').addClass('inline').css('margin', '0 1mm');
                        $(element).css({ width: 'auto' });
                    }

                    // also updating all paragraphs inside text frames (not searching twice for the drawings)
                    implParagraphChangedSync($(element).find(DOM.PARAGRAPH_NODE_SELECTOR));

                    // trigger update of parent paragraph
                    self.trigger('paragraphUpdate:after', $(element).closest(DOM.PARAGRAPH_NODE_SELECTOR));

                } else if (DOM.isTableNode(element) && !DOM.isExceededSizeTableNode(element)) {

                    // Bug 28409: Validate DOM contents of embedded paragraphs without
                    // formatting (done by the following table formatting). Finds and
                    // processes all cell paragraphs contained in a top-level table
                    // (also from embedded tables).
                    if (topLevel) {
                        element.find(DOM.CELLCONTENT_NODE_SELECTOR + ' > ' + DOM.PARAGRAPH_NODE_SELECTOR).each(function () {
                            validateParagraphNode(this);
                        });
                    }
                    // update table formatting asynchronous to prevent browser alerts
                    promise = tableStyles.updateElementFormatting(element, { async: true })
                        .progress(function (localProgress) {
                            updateFormatProgress(currentProgressTicks + TABLE_PROGRESS_TICKS * localProgress);
                        })
                        .always(function () {
                            currentProgressTicks += TABLE_PROGRESS_TICKS;
                        });
                }

                return promise;
            }

            // updates the formatting of all lists, forwards the progress, returns a promise
            function updateListFormatting() {
                return updateLists({ async: true }).progress(function (progress) {
                    def.notify(0.9 + 0.1 * progress);  // last 10% for updating lists
                });
            }

            // dumps profiling information to browser console
            function dumpProfilingInformation() {
                var tableCount,
                    cellCount,
                    paragraphCount,
                    spanCount,
                    drawingCount;

                function dumpUpdateFormattingCount(styleSheets, elementCount, elementName) {
                    var updateCount = styleSheets.DBG_COUNT || 0,
                        perElementCount = (elementCount === 0) ? 0 : Utils.round(updateCount / elementCount, 0.01);
                    if ((elementCount > 0) || (updateCount > 0)) {
                        Utils.info('Editor.updateDocumentFormatting(): ' + elementCount + ' ' + elementName + ' updated ' + updateCount + ' times (' + perElementCount + ' per element)');
                    }
                }

                function dumpParagraphCount(methodName, debugCount) {
                    var updateCount = debugCount || 0,
                        perElementCount = (paragraphCount === 0) ? 0 : Utils.round(updateCount / paragraphCount, 0.01);
                    Utils.info('Editor.' + methodName + '(): called ' + updateCount + ' times (' + perElementCount + ' per paragraph)');
                }

                if (!Config.DEBUG) { return $.noop; }

                tableCount = editdiv.find(DOM.TABLE_NODE_SELECTOR).length;
                cellCount = editdiv.find('td').length;
                paragraphCount = editdiv.find(DOM.PARAGRAPH_NODE_SELECTOR).length;
                spanCount = editdiv.find('span').filter(function () { return DOM.isPortionSpan(this) || DOM.isTextComponentNode(this.parentNode); }).length;
                drawingCount = editdiv.find(DrawingFrame.NODE_SELECTOR).length;

                dumpUpdateFormattingCount(characterStyles, spanCount, 'text spans');
                dumpUpdateFormattingCount(paragraphStyles, paragraphCount, 'paragraphs');
                dumpUpdateFormattingCount(tableCellStyles, cellCount, 'table cells');
                dumpUpdateFormattingCount(tableStyles, tableCount, 'tables');
                dumpUpdateFormattingCount(drawingStyles, drawingCount, 'drawing objects');
                dumpParagraphCount('validateParagraphNode', validateParagraphNode.DBG_COUNT);
            }

            // resetting an existing selection, after all operations are applied
            selection.resetSelection();
            lastOperationEnd = null;

            // receiving list of all change track authors of the document
            self.triggerExtendAuthors(self.getDocumentAttributes().changeTrackAuthors);

            //US 74030826 - fetch first part of elements for fast displaying of first page
            splitPoint = pageContentNode.children('.splitpoint');
            if (splitPoint.length > 0) {
                indexSplitPoint = formattingNodes.index(splitPoint);
                splitListNodes = formattingNodes.slice(0, indexSplitPoint);
                formattingNodes = formattingNodes.slice(indexSplitPoint);
            }

            // detach all child nodes of the page content node, this improves
            // performance in very large documents notably
            formattingNodes.filter(function () { return this.parentNode === pageContentNode[0]; }).detach();

            // first, update the root page node
            pageStyles.updateElementFormatting(editdiv);

            // assign style to header footer placeholder node
            pageLayout.updateStartHeaderFooterStyle();

            if (splitPoint.length > 0) {
                splitListNodes = splitListNodes.add(headerFooterFormattingNodes);
                self.firstPartOfDocumentFormatting(splitListNodes);
                initialPbPromise = pageLayout.callInitialPageBreaks();
                app.leaveBusyDuringImport();
            } else {
                // add header and footer to formatting nodes
                formattingNodes = formattingNodes.add(headerFooterFormattingNodes);
            }

            // abort first promise before calling function second time (avoid duplicate calls if first is not finnished)
            if (initialPbPromise) {
                initialPbPromise.abort();
            }
            this.iterateArraySliced(formattingNodes, updateElementFormatting, { delay: 'immediate', infoString: 'Text: updateElementFormatting' })
                // .then(updateDrawingGroups)
                .then(updateListFormatting)
                .then(function () { pageLayout.callInitialPageBreaks({ triggerEvent: true }); })
                .done(function () { dumpProfilingInformation(); def.resolve(); })
                .fail(function () { def.reject(); });

            return def.promise();
        };

        /**
         * Synchronous formatting of first part of document (1-2 pages) for fast display.
         *
         * @param {jQuery} [splitListNodes]
         *  If exists, collected nodes for synchronous formatting.
         */
        this.firstPartOfDocumentFormatting = function (splitListNodes) {
            var formattingNodes;

            if (splitListNodes) {
                formattingNodes = splitListNodes;
            } else {
                // first, update the root page node
                pageStyles.updateElementFormatting(editdiv);
                // assign style to header footer placeholder node
                pageLayout.updateStartHeaderFooterStyle();

                formattingNodes = DOM.getPageContentNode(editdiv).find('> ' + DOM.PARAGRAPH_NODE_SELECTOR + ', ' + DrawingFrame.NODE_SELECTOR + ', ' + DOM.TABLE_NODE_SELECTOR);
            }

            _.each(formattingNodes, function (element) {
                var // whether the element is a direct child of the page content node
                    topLevel = !element.parentNode;

                element = $(element);
                // determine the type of the passed element
                if (DOM.isParagraphNode(element)) {
                    // validate DOM contents AND update formatting of the paragraph
                    validateParagraphNode(element);
                    paragraphStyles.updateElementFormatting(element);
                } else if (DrawingFrame.isDrawingFrame(element)) {
                    // TODO: validate DOM of paragraphs embedded in text frames
                    drawingStyles.updateElementFormatting(element);
                } else if (DOM.isTableNode(element) && !DOM.isExceededSizeTableNode(element)) {
                    // Bug 28409: Validate DOM contents of embedded paragraphs without
                    // formatting (done by the following table formatting). Finds and
                    // processes all cell paragraphs contained in a top-level table
                    // (also from embedded tables).
                    if (topLevel) {
                        element.find(DOM.CELLCONTENT_NODE_SELECTOR + ' > ' + DOM.PARAGRAPH_NODE_SELECTOR).each(function () {
                            validateParagraphNode(this);
                        });
                    }
                    // update table formatting asynchronous to prevent browser alerts
                    tableStyles.updateElementFormatting(element);
                }
            });
        };

        /**
         * returns the preselected attributed, needed for external call of "interText"
         */
        this.getPreselectedAttributes = function () {
            return preselectedAttributes;
        };

        /**
         * Returns wheter editor is in page breaks mode, or without them.
         *
         * @returns {Boolean}
         *
         */
        this.isPageBreakMode = function () {
            return pbState;
        };

        /**
         * Returns wheter editor is in draft mode (limited functionality mode for small devices).
         *
         * @returns {Boolean}
         *
         */
        this.isDraftMode = function () {
            return draftModeState;
        };

        /**
         * Method that toggles and simulates Draftmode in web browser.
         *
         * @param {Boolean} state
         *  New state. If false, toggles draftmode off.
         *
         * @returns {Editor}
         *  A reference to this instance.
         *
         */
        this.toggleDraftMode = function (state) {
            var pageContentNode = DOM.getPageContentNode(editdiv);

            if (draftModeState === state) { return this; }
            draftModeState = state;
            if (draftModeState) {
                /*
                _.each(DOM.getPageContentNode(editdiv).find(DrawingFrame.NODE_SELECTOR), function (drawing) {
                    if ($(drawing).hasClass('float')) {
                        //verticalOffsetNode.remove();
                        $(drawing).parent().find('.float.offset').remove();

                        $(drawing).removeClass('float left right').addClass('inline').css('margin', '0 1mm');
                        $(drawing).css({ width: 'auto' });
                    }
                });
                */
                //clear zoom level that might interfere with normal mode
                app.getView().increaseZoomLevel(100, true);
                app.getView().getContentRootNode().addClass('draft-mode');
                this.togglePageBreakMode(false);
            } else {
                //return drawings style to float
                app.getView().getContentRootNode().removeClass('draft-mode');
                //clear zoom level from draft mode, which is different from zoom in normal mode
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform', 'scale(1)');
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform-origin', '');
                pageContentNode.css({ width: '100%', backgroundColor: '' });
                app.getView().increaseZoomLevel(100);
                this.togglePageBreakMode(true);
            }
            return this;
        };

        /**
         * Turn page breaks on or off. Removes them from document, and makes cleanup.
         *
         * @param {Boolean} state
         *   New state. If false, toggles page breaks off.
         *
         * @returns {Editor}
         *   A reference to this instance.
         */
        this.togglePageBreakMode = function (state) {
            var
                splittedParagraphs,
                pageContentNode = DOM.getPageContentNode(editdiv),
                spans;

            if (pbState === state) { return this; }
            pbState = state;

            if (!pbState) {
                pageContentNode.find('.page-break').remove();
                pageContentNode.find('.pb-row').remove();
                pageContentNode.find('.tb-split-nb').removeClass('tb-split-nb');
                pageContentNode.find('.break-above-tr').removeClass('break-above-tr');
                pageContentNode.find('.break-above').removeClass('break-above');
                pageContentNode.find('.last-on-page').removeClass('last-on-page');
                editdiv.children('.header-wrapper, .footer-wrapper').addClass('hiddenVisibility');

                splittedParagraphs = pageContentNode.find('.contains-pagebreak');
                if (splittedParagraphs.length > 0) {
                    _.each(splittedParagraphs, function (splittedItem) {
                        spans = $(splittedItem).find('.break-above-span').removeClass('break-above-span');
                        $(splittedItem).children('.break-above-drawing').removeClass('break-above-drawing');
                        validateParagraphNode(splittedItem);
                        if (spans.length > 0) {
                            _.each(spans, function (span) {
                                Utils.mergeSiblingTextSpans(span);
                                Utils.mergeSiblingTextSpans(span, true);
                            });
                        }
                        $(splittedItem).removeClass('contains-pagebreak');
                    });
                }
                pageContentNode.css({ paddingBottom: '' });
            } else {
                editdiv.children('.header-wrapper, .footer-wrapper').removeClass('hiddenVisibility');
                pageLayout.callInitialPageBreaks();
            }
            app.getView().recalculateDocumentMargin();
            return this;
        };

        /**
         * Setter for global state - if headers and footers are currently edited
         *
         *  @param {Boolean} state
         */
        this.setHeaderFooterEditState = function (state) {
            if (headerFooterEditState !== state) {
                headerFooterEditState = state;
            }
        };

        /**
         * Getter for global state of editing or not headers and footers
         *
         *  @returns {Boolean} headerFooterEditState
         */
        this.isHeaderFooterEditState = function () {
            return headerFooterEditState;
        };

        /**
         * Setter for header/footer container root node
         *
         *  @param {jQuery} $node
         */
        this.setHeaderFooterRootNode = function ($node) {
            $headFootRootNode = $node;
            rootNodeIndex = editdiv.find('.header, .footer').filter('[data-container-id="' + $headFootRootNode.attr('data-container-id') + '"]').index($headFootRootNode);
            $parentOfHeadFootRootNode = $headFootRootNode.parent();
            selection.setNewRootNode($node);
        };

        /**
         * Getter for header/footer container root node
         *
         * @param {String[]} [target]
         *  if exists, find node by target id
         *
         *  @returns {jQuery} $headFootRootNode
         */
        this.getHeaderFooterRootNode = function (target) {
            // if node is detached from DOM, fetch from parent reference in DOM
            if (target || !$headFootRootNode.parent().parent().length) {
                // if it comes from operation, it will have target argument
                if (target) {
                    if (target === $headFootRootNode.attr('data-container-id')) { // if target is same as cached node reference, use this node
                        if (self.isHeaderFooterEditState()) {
                            return $headFootRootNode;
                        }
                        $headFootRootNode = editdiv.find('.header, .footer').filter('[data-container-id="' + target + '"]').eq(rootNodeIndex);
                    } else { // otherwise use first node with that target

                        // leaving current edit state, because another header/footer is selected
                        pageLayout.leaveHeaderFooterEditMode($headFootRootNode, $headFootRootNode.parent());

                        // finding new header/footer
                        $headFootRootNode = editdiv.find('.header, .footer').filter('[data-container-id="' + target + '"]').eq(0);
                    }
                } else {
                    if (self.getEditMode()) {
                        if (!self.isHeaderFooterEditState()) {
                            Utils.warn('Editor.getHeaderFooterRootNode failed to fetch valid node!');
                            return editdiv;
                        }
                        $headFootRootNode = editdiv.find('.header, .footer').filter('[data-container-id="' + $headFootRootNode.attr('data-container-id') + '"]').eq(rootNodeIndex);
                    } else {
                        // remote clients need to fetch template node
                        $headFootRootNode = pageLayout.getHeaderFooterPlaceHolder().children('[data-container-id="' + target + '"]').first();
                    }
                }
                if (!$headFootRootNode.length) {
                    Utils.warn('Editor.getHeaderFooterRootNode not found!');
                    return editdiv;
                }

                pageLayout.enterHeaderFooterEditMode($headFootRootNode);

                //selection.resetSelection();
                selection.setTextSelection(selection.getFirstDocumentPosition());
            }

            return $headFootRootNode;
        };

        /**
         * Getter method for currently active root container node.
         *
         * @param {String[]} [target]
         *  ID of root container node
         *
         * @returns {jQuery}
         *  Found node
         *
         */
        this.getCurrentRootNode = function (target) {
            // target for operation - if exists, it's for ex. header or footer
            var currTarget = target || self.getActiveTarget();

            // First checking, if the target is a target for a comment. Then this is handled by the comment layer.
            if (currTarget && commentLayer.getCommentRootNode(currTarget)) { return commentLayer.getCommentRootNode(currTarget); }

            // container root node of header/footer
            return currTarget ? (app.isImportFinished() ? self.getHeaderFooterRootNode(target) : pageLayout.getHeaderFooterPlaceHolder().children('[data-container-id="' + target + '"]')) : editdiv;
        };

        /**
         * Getter method for root container node with passed target,
         * or if target is not passed, default editdiv node.
         *
         * @param {String[]} [target]
         *  ID of root container node
         *
         * @param {Number} [index]
         *  Cardinal number of target node in the document from top to bottom.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.allowMarginalUpdate=true]
         *      If set to false, the call of 'replaceAllTypesOfHeaderFootersDebounced'
         *      is not allowed. This is necessary for task 40199.
         *      TODO: This option and the complete call of 'replaceAllTypesOfHeaderFootersDebounced'
         *      must be removed from this getter function.
         *
         * @returns {jQuery}
         *  Found node
         */
        this.getRootNode = function (target, index, options) {
            var $node;

            if (target) {
                if (commentLayer.getCommentRootNode(target)) {
                    $node = commentLayer.getCommentRootNode(target);
                } else {
                    // special handling for headers&footers
                    $node = pageLayout.getRootNodeTypeHeaderFooter(target, index);
                    if (app.isImportFinished() && !self.getEditMode() && Utils.getBooleanOption(options, 'allowMarginalUpdate', true)) {
                        replaceAllTypesOfHeaderFootersDebounced();  // TODO: This needs to be removed from this getter function
                    }
                }
            } else {
                $node = editdiv;
            }

            return $node;
        };

        /**
         * Utility method for extending object with target property,
         * but only if its defined.
         *
         * @param {Object} object
         *  object that is being extended
         *
         * @param {String[]} target
         *  Id referencing to header or footer node
         */
        function extendPropertiesWithTarget(object, target) {
            if (target) {
                object.target = target;
            }
        }

        /**
         * Public method interface for private function extendPropertiesWithTarget.
         *
         * @param{Object} object
         *   object that is being extended
         * @param{String[]} target
         *   Id referencing to header or footer node
         *
         */
        this.extendPropertiesWithTarget = function (object, target) {
            extendPropertiesWithTarget(object, target);
        };

        /**
         * This function is also available as private function. Check is
         * necessary, if this really must be a public function. Using
         * revealing pattern here.
         *
         * Prepares the text span at the specified logical position for
         * insertion of a new text component or character. Splits the text span
         * at the position, if splitting is required. Always splits the span,
         * if the position points between two characters of the span.
         * Additionally splits the span, if there is no previous sibling text
         * span while the position points to the beginning of the span, or if
         * there is no next text span while the position points to the end of
         * the span.
         *
         * @param {Number[]} position
         *  The logical text position.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.isInsertText=false]
         *      If set to true, this function was called from implInsertText.
         *  @param {Boolean} [options.useCache=false]
         *      If set to true, the paragraph element saved in the selection can
         *      be reused.
         *  @param {Boolean} [options.allowDrawingGroup=false]
         *      If set to true, the element can also be inserted into a drawing
         *      frame of type 'group'.
         *
         * @returns {HTMLSpanElement|Null}
         *  The text span that precedes the passed offset. Will be the leading
         *  part of the original text span addressed by the passed position, if
         *  it has been split, or the previous sibling text span, if the passed
         *  position points to the beginning of the span, or the entire text
         *  span, if the passed position points to the end of a text span and
         *  there is a following text span available or if there is no following
         *  sibling at all.
         *  Returns null, if the passed logical position is invalid.
         */
        this.prepareTextSpanForInsertion = function (position, options, target) {
            return prepareTextSpanForInsertion(position, options, target);
        };

        /**
         * Called when all initial document operations have been processed.
         * Can be used to start post-processing tasks which need a fully
         * processed document.
         */
        function documentLoaded() {

            if (changeTrack.updateSideBar()) { self.trigger('changeTrack:stateInfo', { state: true }); }
            insertCollaborativeOverlay();
            insertMissingCharacterStyles();
            insertMissingParagraphStyles();
            insertMissingTableStyles();
            if (app.isODF()) { insertMissingDrawingStyles(); }

            // Special observer for iPad/Safari mobile. Due to missing events to detect
            // and process DOM changes we have to add special code to handle these cases.
            if (Utils.IOS) {
                var MutationObserver = window.MutationObserver || window.WebKitMutationObserver || window.MozMutationObserver;

                editDivObserver = new MutationObserver(function (mutations) {
                    if (!self) {
                        editDivObserver.disconnect();
                        return;
                    }
                    if (!self.isProcessingOperations()) {
                        // Optimization: we only check DOM mutations when we don't process our own
                        // operations.

                        mutations.forEach(function (mutation) {
                            var i = 0;

                            if (mutation.addedNodes.length > 0) {

                                for (i = 0; i < mutation.addedNodes.length; i++) {
                                    var node = mutation.addedNodes[i];

                                    if (node.nodeType === 1 && node.tagName === 'IMG' && node.className === '-webkit-dictation-result-placeholder') {
                                        // We have detected a Siri input. Switch model to read-only/disconnect and disconnect observer
                                        app.rejectEditAttempt('siri');
                                        editDivObserver.disconnect();
                                    }
                                }
                            }
                        });
                    }
                });

                editDivObserver.observe(editdiv[0], { childList: true, subtree: true });

                // Prevent text suggestions which are available via context menu on iPad/iPhone.
                selection.on('change', function (event, options) {
                    var paragraph, text;

                    // Performance: Ignoring handler for 'simpleTextSelection' operations
                    if (!Utils.getBooleanOption(options, 'simpleTextSelection', false)) {
                        if (selection.hasRange()) {
                            if (selection.getSelectionType() === 'text' && self.getEditMode() === true) {
                                paragraph = selection.getEnclosingParagraph();
                                text = null;

                                if (paragraph) {
                                    text = selection.getSelectedText();
                                    if (text && text.length > 0 && text.indexOf(' ') === -1) {
                                        setIOSROParagraph(paragraph);
                                    } else {
                                        setIOSROParagraph(null);
                                    }
                                }
                            }
                        } else {
                            // in case the text cursor is set to a different position
                            setIOSROParagraph(null);
                        }
                    }
                });
            }
        }

        /**
         * Method to check whether page break calculation are blocked and returned from it.
         *
         * @returns {Boolean}
         *  True, if calculation should stop running.
         *
         */
        this.checkQuitFromPageBreaks = function () {
            return quitFromPageBreak;
        };

        /**
         * Set block on running loops inside page breaks calculation, and immediately returns from it.
         * It is called from functions that have bigger priority than page breaks: such as text insert and delete.
         *
         * @param {Boolean} state
         *  New state
         *
         */
        this.setQuitFromPageBreaks = function (state) {
            quitFromPageBreak = state;
        };

        /**
         * Check if debounced method should skip insertPageBreaks function, but run other piggybacked, like updateAbsolutePositionedDrawings.
         * This is different from checkQuitFromPageBreaks, as it block page breaks inside debounced method.
         *
         * @returns {Boolean}
         *  True, if debounced method should skip call of insertPageBreaks.
         *
         */
        this.getBlockOnInsertPageBreaks = function () {
            return blockOnInsertPageBreaks;
        };

        /**
         * Sets block on pageLayout.insertPageBreak function, that is evaluated inside insertPageBreaksDebounced.
         * This is different from setQuitFromPageBreaks, as it block page breaks inside debounced method,
         * not inside insertPageBreaks function.
         *
         * @param {Boolean} state
         *  New state
         *
         */
        this.setBlockOnInsertPageBreaks = function (state) {
            if (app.isImportFinished()) {
                blockOnInsertPageBreaks = state;
            }
        };

        /**
         * Getter to fetch global variable node currentProcessingNode,
         * which is set during direct call of inserting page breaks, registerPageBreaksInsert.
         *
         * @returns {jQuery|Node}
         *
         */
        this.getCurrentProcessingNode = function () {
            return currentProcessingNode;
        };

        /**
         * Public method to get access to debounced insertPageBreaks call.
         *
         * @param {jQuery|Node} [node]
         *  Node from where to start page breaks calculation (top-down direction).
         *
         */
        this.insertPageBreaks = function (node) {
            insertPageBreaksDebounced(node);
        };

        /**
         * Getter for argument node shared between direct and callback methods of updateEditingHeaderFooterDebounced
         *
         * @returns {jQuery} target node
         */
        this.getTargetNodeForUpdate = function () {
            return targetNodeForUpdate;
        };

        /**
         * Public method to call validateParagraphNode outside model
         *
         * @param {jQuery|Node} element
         *
         */
        this.validateParagraphElement = function (element) {
            validateParagraphNode(element);
        };

        /**
         * Getter for global variable containing target of operation (header or footer target id).
         *
         * @returns {jQuery} activeTarget
         *
         */
        this.getActiveTarget = function () {
            return activeTarget;
        };

        /**
         * Setter for target of operation (header or footer target id).
         *
         * @param{String} value
         *  sets passed value to activeTarget
         *
         */
        this.setActiveTarget = function (value) {
            activeTarget = value || '';
        };

        /**
         * Getter for the keyboard block property.
         */
        this.getBlockKeyboardEvent = function () {
            return blockKeyboardEvent;
        };

        /**
         * Setting a blocker for the incoming keyboard events. This is necessary
         * for long running processes.
         */
        this.setBlockKeyboardEvent = function (value) {
            blockKeyboardEvent = value;
        };

        /**
         * public listeners for IPAD workaround:
         * so we can unlisten all function on editdiv
         * and assign them new on editdivs parent
         */
        this.getListenerList = function () {
            return listenerList;
        };

        /**
         * Check, whether the start position of the current selection
         * is inside the specified (paragraph) node. In this case the
         * start position has to start with the logical position of the
         * node.
         *
         * @param {Node|jQuery} [node]
         *  The DOM node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains.
         *
         * @returns {Boolean}
         *  Whether the start of the selection is located inside the
         *  specified node.
         */
        this.selectionInNode = function (node) {

            var // the logical position of the specified node
                pos = Position.getOxoPosition(self.getNode(), node);

            return Utils.compareNumberArrays(pos, selection.getStartPosition(), pos.length) === 0;
        };

        /**
         * Public method to access updateEditingHeaderFooterDebounced function outside of Editor class.
         */
        this.updateEditingHeaderFooterDebounced = function () {
            return updateEditingHeaderFooterDebounced();
        };

        /**
         * Public method to access searchHandler outside of Editor class.
         */
        this.getSearchHandler = function () {
            return searchHandler;
        };

        /**
         * Public method to access spellchecker outside of Editor class.
         */
        this.getSpellChecker = function () {
            return spellChecker;
        };

        /**
         * Public model method that triggers editapplication's function to extend document's unique list of authors.
         *
         * @param {Array} authors
         *  List of authors in document.
         */
        this.triggerExtendAuthors = function (authors) {
            self.trigger('extendauthors', authors);
        };

        /**
         * Updating the models for drawings in drawing layer, comments, complex fields or range
         * markers. If a node like a paragraph, a table row, a table cell, a complete table or a
         * header or footer is removed, it is necessary to check, if the collections of drawing
         * layer, comments, range markers or complex fields are affected.
         * Additionally the drawings and comments inside their layers need to be removed, if a
         * placeholder is removed.
         *
         * @param {HTMLElement|jQuery} parentNode
         *  The node (paragraph, table, row, cell, ...) that will be checked for specific elements
         *  that are stored in models.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.marginal=false]
         *      If set to true, this updating of models was triggered by a change of header or footer.
         */
        this.updateCollectionModels = function (parentNode, options) {

            var // whehter this is a marginal model update
                marginal = Utils.getBooleanOption(options, 'marginal', false);

            // checking, if the paragraph contains comment place holder nodes. In this case the comment
            // needs to be removed from comment layer, too.
            if (!marginal || app.isODF()) { commentLayer.removeAllInsertedCommentsFromCommentLayer(parentNode); }

            // checking, if the paragraph contains drawing place holder nodes. In this case the drawings
            // needs to be removed from drawing layer, too.
            if (!marginal) { drawingLayer.removeAllInsertedDrawingsFromDrawingLayer(parentNode); }

            // checking, if the paragraph contains additional range marker nodes, that need to be removed, too.
            rangeMarker.removeAllInsertedRangeMarker(parentNode);

            // checking, if the paragraph contains additional field nodes, that need to be removed.
            fieldManager.removeAllFieldsInNode(parentNode);
        };

        // ==================================================================
        // END of Editor API
        // ==================================================================

        // ==================================================================
        // Private functions for document post-processing
        // ==================================================================

        /**
         * Checks stored character styles of a document and adds 'missing'
         * character styles (e.g. hyperlink).
         */
        function insertMissingCharacterStyles() {
            var styleNames = characterStyles.getStyleSheetNames(),
                parentId = characterStyles.getDefaultStyleId(),
                hyperlinkMissing = true;

            _(styleNames).find(function (name) {
                var lowerName = name.toLowerCase();
                if (lowerName.indexOf('hyperlink') >= 0) {
                    hyperlinkMissing = false;
                    return true;
                }
                return false;
            });

            if (hyperlinkMissing) {
                var hyperlinkAttr = { character: self.getDefaultLateralHyperlinkAttributes() },
                    hyperlinkDef = self.getDefaultLateralHyperlinkDefinition();
                characterStyles.insertStyleSheet(
                        hyperlinkDef.styleId, hyperlinkDef.styleName,
                        parentId, hyperlinkAttr,
                        { hidden: false, priority: hyperlinkDef.uiPriority, defStyle: false, dirty: true });
            }
        }

        /**
         * Check the stored paragraph styles of a document and adds 'missing'
         * heading / default and other paragraph styles.
         */
        function insertMissingParagraphStyles() {

            var // the number extensions for the supported headings
                headings = [0, 1, 2, 3, 4, 5],
                // the paragraph style names
                styleNames = paragraphStyles.getStyleSheetNames(),
                // the default paragraph style id
                parentId = paragraphStyles.getDefaultStyleId(),
                // whether a default style is already defined after loading
                hasDefaultStyle = _.isString(parentId) && (parentId.length > 0),
                // the id of the paragraph list style
                paragraphListStyleId = null,
                // the id of the paragraph comment style
                commentStyleId = null,
                // the id of the paragraph header style
                headerStyleId = null,
                // the id of the paragraph footer style
                footerStyleId = null,
                // the default style id of a visible paragraph
                visibleParagraphDefaultStyleId = null,
                // the default style id of a hidden paragraph
                hiddenDefaultStyle = false,
                // the default paragraph definition
                defParaDef,
                // the comment paragraph definition
                commentDef = null,
                // the comment paragraph attributes
                commentAttrs = null,
                // the header paragraph definition
                headerDef = null,
                // the footer paragraph definition
                footerDef = null,
                // header/footer style tab position right
                rightTabPos = pageLayout.getPageAttribute('width') - pageLayout.getPageAttribute('marginLeft') - pageLayout.getPageAttribute('marginRight'),
                // header/footer style tab position center
                centerTabPos = rightTabPos / 2,
                // the header paragraph attributes
                headerAttrs = null,
                // the footer paragraph attributes
                footerAttrs = null;

            if (!hasDefaultStyle) {
                // add a missing default paragraph style
                defParaDef = self.getDefaultParagraphStyleDefinition();
                paragraphStyles.insertStyleSheet(defParaDef.styleId, defParaDef.styleName, null, null,
                        { hidden: false, priority: 1, defStyle: defParaDef['default'], dirty: true });
                parentId = defParaDef.styleId;
                visibleParagraphDefaultStyleId = parentId;
            } else {
                hiddenDefaultStyle = paragraphStyles.isHidden(parentId);
                if (!hiddenDefaultStyle) {
                    visibleParagraphDefaultStyleId = parentId;
                }
            }

            // Search through all paragraph styles to find out what styles
            // are missing.
            _(styleNames).each(function (name, id) {
                var styleAttributes = paragraphStyles.getStyleAttributeSet(id).paragraph,
                    outlineLvl = styleAttributes.outlineLevel,
                    lowerName = name.toLowerCase(),
                    lowerId = id.toLowerCase();

                if (_.isNumber(outlineLvl) && (outlineLvl >= 0 && outlineLvl < 6)) {
                    headings = _(headings).without(outlineLvl);
                }

                if (!paragraphListStyleId) {
                    if (lowerName.indexOf('list paragraph') === 0 || lowerName.indexOf('listparagraph') === 0) {
                        paragraphListStyleId = id;
                    }
                }

                if (!commentStyleId) {
                    if (lowerName.indexOf('annotation text') === 0) {
                        commentStyleId = id;
                    }
                }

                if (!headerStyleId) {
                    if (lowerName.indexOf('header') === 0) {
                        headerStyleId = id;
                    }
                }

                if (!footerStyleId) {
                    if (lowerName.indexOf('footer') === 0) {
                        footerStyleId = id;
                    }
                }

                // Check and store the headings style which is used
                // by odf documents as seed for other heading styles.
                if (hiddenDefaultStyle && !visibleParagraphDefaultStyleId) {
                    if (lowerId === 'standard' || lowerId === 'normal') {
                        visibleParagraphDefaultStyleId = id || parentId;
                    }
                }
            });

            // add the missing paragraph heading styles using predefined values
            if (headings.length > 0) {
                var defaultCharStyles = self.getDefaultHeadingCharacterStyles(),
                    headingsParentId = hiddenDefaultStyle ? visibleParagraphDefaultStyleId : parentId,
                    headingsNextId = hiddenDefaultStyle ? visibleParagraphDefaultStyleId : parentId;

                _(headings).each(function (level) {
                    var attr = {},
                        charAttr = defaultCharStyles[level];
                    attr.character = charAttr;
                    attr.paragraph = { outlineLevel: level, nextStyleId: headingsNextId };
                    paragraphStyles.insertStyleSheet('heading' + (level + 1), 'heading ' + (level + 1),
                            headingsParentId, attr, { hidden: false, priority: 9, defStyle: false, dirty: true });
                });
            }

            // add missing paragraph list style
            if (!paragraphListStyleId) {
                paragraphStyles.insertStyleSheet(
                    'ListParagraph',
                    'List Paragraph', hiddenDefaultStyle ? (visibleParagraphDefaultStyleId || parentId) : parentId,
                    { paragraph: { indentLeft: 1270, contextualSpacing: true, nextStyleId: 'ListParagraph' }},
                    { hidden: false, priority: 34, defStyle: false, dirty: true });
            }

            // add missing comment style
            if (!commentStyleId) {
                commentDef = self.getDefaultCommentTextDefintion();
                commentAttrs = self.getDefaultCommentTextAttributes();
                paragraphStyles.insertStyleSheet(commentDef.styleId, commentDef.styleName, commentDef.parent, commentAttrs, { priority: commentDef.uiPriority, defStyle: commentDef['default'], dirty: true, hidden: false });
            }

            // add missing header style
            if (!headerStyleId) {
                headerDef = self.getDefaultHeaderTextDefinition();
                headerAttrs = { paragraph: { tabStops: [{ value: 'center', pos: centerTabPos }, { value: 'right', pos: rightTabPos }] } };
                paragraphStyles.insertStyleSheet(headerDef.styleId, headerDef.styleName, headerDef.parent, headerAttrs, { priority: headerDef.uiPriority, defStyle: headerDef['default'], dirty: true, hidden: false });
            }

            // add missing footer style
            if (!footerStyleId) {
                footerDef = self.getDefaultFooterTextDefinition();
                footerAttrs = { paragraph: { tabStops: [{ value: 'center', pos: centerTabPos }, { value: 'right', pos: rightTabPos }] } };
                paragraphStyles.insertStyleSheet(footerDef.styleId, footerDef.styleName, footerDef.parent, footerAttrs, { priority: footerDef.uiPriority, defStyle: footerDef['default'], dirty: true, hidden: false });
            }
        }

        /**
         * Check the stored drawing styles of a document and adds 'missing' style. This is
         * currently only used for odt documents, where the insertion of a text frame requires
         * a corresponding drawing style.
         */
        function insertMissingDrawingStyles() {

            var // the names of the drawing styles
                styleNames = drawingStyles.getStyleSheetNames(),
                // the default style Id
                parentId = drawingStyles.getDefaultStyleId(),
                // whether a default style is already defined
                hasDefaultStyle = _.isString(parentId) && (parentId.length > 0),
                // whether the style for the text frames is missing
                textframestyleMissing = true,
                // the drawing style definition
                drawingDef = null,
                // the drawing style attributes
                drawingAttrs = null;

            // adding the default style, if required
            if (!hasDefaultStyle) {
                drawingDef = self.getDefaultDrawingDefintion();
                drawingAttrs = self.getDefaultDrawingAttributes();
                drawingStyles.insertStyleSheet(drawingDef.styleId, drawingDef.styleName, null, drawingAttrs, { priority: drawingDef.uiPriority, defStyle: drawingDef['default'], dirty: true });
                parentId = drawingDef.styleId;
            }

            // checking the style used for text frames
            _(styleNames).find(function (name) {
                var lowerName = name.toLowerCase();
                if (lowerName.indexOf('frame') >= 0) {
                    textframestyleMissing = false;
                    return true;
                }
                return false;
            });

            if (textframestyleMissing) {
                drawingDef = self.getDefaultDrawingTextFrameDefintion();
                drawingAttrs = self.getDefaultDrawingTextFrameAttributes();
                // Info: Setting option hidden to true. Otherwise this style will replace the style 'default_drawing_style', so that for example line colors are modified.
                drawingStyles.insertStyleSheet(drawingDef.styleId, null, parentId, drawingAttrs, { priority: drawingDef.uiPriority, defStyle: drawingDef['default'], dirty: true, hidden: true });
            }

        }

        /**
         * Check the stored table styles of a document and adds a 'missing'
         * default table style. This ensures that we can insert tables that
         * are based on a reasonable default style.
         */
        function insertMissingTableStyles() {

            var styleNames = tableStyles.getStyleSheetNames(),
                parentId = tableStyles.getDefaultStyleId(),
                hasDefaultStyle = _.isString(parentId) && (parentId.length > 0),
                defTableDef = self.getDefaultLateralTableDefinition(),
                defTableAttr = self.getDefaultLateralTableAttributes();

            // inserts the table style sheet described by the passed parameters, if it does not exist yet, and marks it as dirty
            function insertMissingTableStyle(styleId, styleName, uiPriority, category, attrs) {
                // check by style sheet name (they are always loaded in English from Word, while ste style IDs are localized)
                if (!tableStyles.containsStyleSheetByName(styleName)) {
                    tableStyles.insertStyleSheet(styleId, styleName, null, attrs, { priority: uiPriority, dirty: true, category: category });
                } else {
                    tableStyles.setStyleOptions(styleId, { category: category });
                }
            }

            // inserts a group of table style definitions for the primary text color and all six accent colors
            function insertMissingTableStyleGroup(baseStyleId, baseStyleName, uiPriority, generateAttributesFunc) {
                insertMissingTableStyle(baseStyleId, baseStyleName, uiPriority, baseStyleName, generateAttributesFunc('text1'));
                for (var index = 1; index <= 6; index += 1) {
                    insertMissingTableStyle(baseStyleId + '-Accent' + index, baseStyleName + ' Accent ' + index, uiPriority, baseStyleName, generateAttributesFunc('accent' + index));
                }
            }

            if (!hasDefaultStyle) {
                // Add a missing default table style
                tableStyles.insertStyleSheet(defTableDef.styleId, defTableDef.styleName, null, defTableAttr, { priority: 59, defStyle: defTableDef['default'], dirty: true });
            } else {
                // Search for a style defined in the document that can be used for tables
                // If we cannot find it we have to add it.
                var lowestUIPriority = 99,
                    tableStyleId = null;
                _(styleNames).each(function (name, id) {
                    var uiPriority = tableStyles.getUIPriority(id);

                    if (uiPriority && (uiPriority < lowestUIPriority)) {
                        tableStyleId = id;
                        lowestUIPriority = uiPriority;
                    }
                });

                if ((!tableStyleId) || ((tableStyleId === tableStyles.getDefaultStyleId()) && (lowestUIPriority === 99))) {
                    // OOXML uses a default table style which contains no border
                    // definitions. Therfore we add our own default table style
                    // if we only find the default style with uiPriority 99
                    tableStyles.insertStyleSheet(defTableDef.styleId, defTableDef.styleName, parentId, defTableAttr, { priority: 59, dirty: true });
                }
            }

            // adding further table styles, if not already available
            insertMissingTableStyleGroup('LightShading',   'Light Shading',    60, TableStyles.getLightShadingTableStyleAttributes);
            insertMissingTableStyleGroup('MediumShading1', 'Medium Shading 1', 63, TableStyles.getMediumShading1TableStyleAttributes);
            insertMissingTableStyleGroup('MediumShading2', 'Medium Shading 2', 64, TableStyles.getMediumShading2TableStyleAttributes);
            insertMissingTableStyleGroup('MediumGrid1',    'Medium Grid 1',    67, TableStyles.getMediumGrid1TableStyleAttributes);
        }

        // ====================================================================
        // Private functions for the hybrid edit mode
        // ====================================================================

        function processMouseDown(event) {

            var // whether the user has privileges to edit the document
                readOnly = self.getEditMode() !== true,
                // jQuerified target of the event
                $eventTarget = $(event.target),
                // mouse click on a table
                tableCell = $eventTarget.closest(DOM.TABLE_CELLNODE_SELECTOR),
                tableCellPoint = null,
                // drawing start and end position for selection
                startPosition = null, endPosition = null,
                // mouse click on a drawing node
                drawingNode = null,
                // mouse click into a comment node
                commentNode = null,
                // mouse click on a tabulator node
                tabNode = null,
                // move handler node for drawings
                moveNode = null,
                // mouse click on a table resize node
                resizerNode = null,
                // mouse click on field node
                field = null,
                // the id of an optionally activated comment node
                activatedCommentId = '',
                // change track popup
                changeTrackPopup = app.getView().getChangeTrackPopup(),
                // the currently activated node
                activeRootNode = self.getCurrentRootNode(),
                // current page number
                pageNum = null,

                // delay of opening the changeTrack-popup
                changeTrackPopupTimeout = app.getView().getChangeTrackPopupTimeout(),
                // the changeTrack-popup
                changeTrakkPopup = app.getView().getChangeTrackPopup(),

                // whether the right mouse-button was clicked, or not
                rightClick = (event.button === 2);

            // fix for not clearing anoying selection range in Chrome on click
            if (_.browser.Chrome && selection.hasRange() && !rightClick) {
                window.getSelection().empty();
            }

            // expanding a waiting double click to a triple click
            if (doubleClickEventWaiting) {
                tripleClickActive = true;
                event.preventDefault();
                return;
            }

            // do not show the changeTrack-popup, if the right mousebutton was cliked
            // or hide it, if it's already open
            if (rightClick) {
                if (changeTrackPopupTimeout) { changeTrackPopupTimeout.abort(); }
                if (changeTrakkPopup.isVisible()) { changeTrakkPopup.hide(); }
            }

            activeMouseDownEvent = true;

            // make sure that the clickable parts in the hyperlink popup are
            // processed by the browser
            if (Hyperlink.isClickableNode($eventTarget, readOnly)) {
                return;
            } else if (Hyperlink.isPopupOrChildNode($eventTarget)) {
                // don't call selection.processBrowserEvent as we don't have
                // a valid position in the document if we process mouse event
                // on the popup
                event.preventDefault();
                return;
            }

            // also dont do processBrowserEvent if the change track popup is clicked
            if (changeTrackPopup.getNode().find(event.target).length > 0) {
                event.preventDefault();
                return;
            }

            // not a link!!! for halo view fix for Bug 39056
            // clicking into a comment meta info node handler or in IE on the comment border (thread itself or text frame content node) (37672)
            if ((event.type !== 'touchstart' || !$eventTarget.is('a')) && (DOM.isNodeInsideCommentMetaInfo(event.target) || (_.browser.IE && (DOM.isCommentThreadNode(event.target) || (DrawingFrame.isTextFrameContentNode(event.target) && DOM.isCommentNode(event.target.parentNode)))))) {
                event.preventDefault();
                return;
            }

            // Fix for 29257 and 29380 and 31623
            if ((_.browser.IE === 11) && ($eventTarget.is('table') ||
                    DOM.isListLabelNode(event.target) || (event.target.parentNode && DOM.isListLabelNode(event.target.parentNode)) ||
                    DOM.isPlaceholderNode(event.target) || (event.target.parentNode && DOM.isPlaceholderNode(event.target.parentNode)) ||
                    (event.target.parentNode && event.target.parentNode.parentNode && DOM.isPlaceholderNode(event.target.parentNode.parentNode)))) {
                return;
            }

            // Fix for 29409: preventing default to avoid grabbers in table cells or inline components
            if ((_.browser.IE === 11) &&
                ($eventTarget.is('div.cell') ||
                DOM.isInlineComponentNode(event.target) ||
                (event.target.parentNode && DOM.isInlineComponentNode(event.target.parentNode)))) {
                event.preventDefault(); // -> not returning !
            }

            if (Utils.SMALL_DEVICE) { // in draft mode, allow only text selection
                if (activeTarget) { commentLayer.handleCommentSelectionOnSmallDevice(event, activeTarget); }
                selection.processBrowserEvent(event);
                return;
            }

            if (self.isHeaderFooterEditState()) {
                if ((_.browser.Firefox || _.browser.IE) && $eventTarget.is('.cover-overlay, .inactive-selection, .page-break')) {
                    event.preventDefault();
                    return; //#37151
                }
                if (DOM.isMarginalContextItem(event.target)) {
                    event.preventDefault();
                    if ($eventTarget.hasClass('marginal-menu-goto')) {
                        pageNum = pageLayout.getPageNumber(self.getCurrentRootNode());
                        pageLayout.leaveHeaderFooterAndSetCursor(self.getCurrentRootNode());
                        if ($eventTarget.parent().hasClass('above-node')) {
                            pageLayout.jumpToHeaderOnCurrentPage(pageNum);
                        } else {
                            pageLayout.jumpToFooterOnCurrentPage(pageNum);
                        }
                    } else {
                        pageLayout.leaveHeaderFooterAndSetCursor(self.getCurrentRootNode());
                    }
                    return;
                } else if ($eventTarget.hasClass('marginal-menu-type') || $eventTarget.hasClass('marginal-label') || $eventTarget.hasClass('fa-caret-down')) {
                    event.preventDefault();
                    pageLayout.toggleMarginalMenuDropdown(event.target);
                    return;
                } else if ($eventTarget.hasClass('marginal-none')) {
                    event.preventDefault();
                    pageLayout.setHeaderFooterTypeInDoc('none');
                    return;
                } else if ($eventTarget.hasClass('marginal-same')) {
                    event.preventDefault();
                    pageLayout.setHeaderFooterTypeInDoc('default');
                    return;
                } else if ($eventTarget.hasClass('marginal-first')) {
                    event.preventDefault();
                    pageLayout.setHeaderFooterTypeInDoc('first');
                    return;
                } else if ($eventTarget.hasClass('marginal-even')) {
                    event.preventDefault();
                    pageLayout.setHeaderFooterTypeInDoc('evenodd');
                    return;
                } else if ($eventTarget.hasClass('marginal-first-even')) {
                    event.preventDefault();
                    pageLayout.setHeaderFooterTypeInDoc('all');
                    return;
                } else if ($eventTarget.hasClass('marginal-separator-line') || $eventTarget.hasClass('marginal-container')) {
                    event.preventDefault();
                    return;
                }
            } else {
                if (_.browser.IE && !rightClick && $eventTarget.is('.inactive-selection, .cover-overlay')) { // #35678 prevent resize rectangles on headers in IE
                    event.preventDefault();
                    selection.updateIESelectionAfterMarginalSelection(activeRootNode, $eventTarget);
                    return;
                }
            }

            if (_.browser.IE && $eventTarget.hasClass('inner-pb-line')) {
                event.preventDefault();
                return;
            }

            if (tableCell.length > 0) {
                // saving initial cell as anchor cell for table cell selection
                // -> this has to work also in read-only mode
                tableCellPoint = DOM.Point.createPointForNode(tableCell);
                selection.setAnchorCellRange(new DOM.Range(tableCellPoint, new DOM.Point(tableCellPoint.node, tableCellPoint.node + 1)));
            }

            // tabulator handler
            tabNode = $eventTarget.closest(DOM.TAB_NODE_SELECTOR);
            if (tabNode.length > 0) {
                // handle in processMouseUp event
                return;
            }

            // checking for a selection on a drawing node. But if this is a text frame drawing node, this is not a drawing
            // selection, if the click happened on the text frame sub element.
            drawingNode = $eventTarget.closest(DrawingFrame.NODE_SELECTOR);

            if (drawingNode.length > 0) {
                if (DrawingFrame.isOnlyBorderMoveableDrawing(drawingNode) && $eventTarget.closest(DrawingFrame.TEXTFRAME_NODE_SELECTOR).length > 0) {
                    drawingNode = $();
                } else if (DOM.isNodeInsideComment(drawingNode.parent())) {
                    commentNode = drawingNode.closest(DOM.COMMENTNODE_SELECTOR);
                    activatedCommentId = commentLayer.activateCommentNode(commentNode, selection, { deactivate: true }); // happens not, if the node is already active
                    activeRootNode = commentNode; // selecting a drawing inside a comment
                } else if (DOM.isCommentNode(drawingNode)) {
                    // checking, if this was a click inside a comment
                    activatedCommentId = commentLayer.activateCommentNode(drawingNode, selection, { deactivate: true }); // happens not, if the node is already active
                    activeRootNode = drawingNode;
                    drawingNode = $();
                }
            }

            // deactivating comment node, but not if it was already deactivated in 'activateCommentNode' before
            if (activeTarget !== '' && !activatedCommentId && !self.isHeaderFooterEditState()) {
                //  leaving comment, restore original rootNode (but not if this mouse down happens in the same comment)
                commentLayer.deActivateCommentNode(activeTarget, selection);
            }

            // checking for selection on a field
            field = $eventTarget.closest(DOM.FIELD_NODE_SELECTOR);

            if (field.length > 0) {
                // prevent default click handling of the browser
                event.preventDefault();

                // set focus to the document container (may be located in GUI edit fields)
                app.getView().grabFocus();

                // select the field
                startPosition = Position.getOxoPosition(activeRootNode, field[0], 0);
                endPosition = startPosition;
                selection.setTextSelection(startPosition, endPosition, { event: event });
                return;
            }

            // in read only mode allow text selection only (and drawing selection, task 30041)
            if (readOnly && drawingNode.length === 0) {
                selection.processBrowserEvent(event);
                return;
            }

            // Leaving Header-footer edit state
            if (self.isHeaderFooterEditState() && !DOM.isHeaderOrFooter(event.target) && DOM.getMarginalTargetNode(self.getNode(), event.target).length === 0) {
                pageLayout.leaveHeaderFooterAndSetCursor($parentOfHeadFootRootNode.find('.active-selection'), $parentOfHeadFootRootNode);
                updateEditingHeaderFooterDebounced(); // this call is important for repaiting layout after leaving h/f edit mode
            }

            // checking for a selection on a resize node
            resizerNode = $eventTarget.closest(DOM.RESIZE_NODE_SELECTOR);

            if (drawingNode.length > 0) {

                if (event.type !== 'touchstart') {
                    // prevent default click handling of the browser
                    event.preventDefault();
                }

                // set focus to the document container (may be located in GUI edit fields)
                app.getView().grabFocus();

                // do nothing if the drawing is already selected (prevent endless
                // recursion when re-triggering the event, to be able to start
                // moving the drawing immediately, see below)
                // Also create a drawing selection, if there is an additional text frame selection,
                // that also has a text selection. In this case the text selection is removed.
                if (!DrawingFrame.isSelected(drawingNode) || selection.isAdditionalTextframeSelection()) {

                    // select the drawing
                    startPosition = Position.getOxoPosition(activeRootNode, drawingNode[0], 0);
                    endPosition = Position.increaseLastIndex(startPosition);
                    selection.setTextSelection(startPosition, endPosition, { event: event });

                    // The following triggering of the event is necessary to move a drawing without a further mousedown
                    // or touchstart event. Without this additional trigger the first mousedown/touchstart would only select
                    // the drawing. For moving it, a further mousedown/touchstart is requrired. By triggering this event
                    // again, it is possible to move a drawing already with the first mousedown/touchstart.
                    // For moving a drawing, a tracker node was created and registered for 'tracking:start' in the
                    // preceeding DrawingResize.drawDrawingSelection() call. This additional mousedown/touchstart
                    // generates a 'tracking:start' event, that will be handled by the tracker node.
                    // The event target for this event in 'processMouseDown' could be the 'img' or the 'div.drawing'. But
                    // it has to be sent to the tracker. Therefore it is necessary to adapt the event target, before
                    // triggering it again.
                    moveNode = drawingNode.find('.tracker');
                    if (moveNode.length > 0) {
                        event.target = moveNode[0];
                        moveNode.trigger(event);
                    }
                }

                return (event.type === 'touchstart') ? true : false;

            } else if (resizerNode.length > 0) {

                if (!DOM.isActiveResizeNode(resizerNode)) {
                    // Because the event is triggered again to avoid a second click, it is necessary to mark the resize node
                    // as 'active'. Otherwise the event would be triggered endlessly.

                    // prevent default click handling of the browser
                    event.preventDefault();
                    // set focus to the document container (may be located in GUI edit fields)
                    app.getView().grabFocus();
                    // draw the resizer
                    TableResize.drawTableCellResizeSelection(self, app.getView(), app.getWindowNode(), resizerNode);

                    // send initial mouse down event to the registered handlers
                    // The following triggering of the event is necessary to resize the table without a further mousedown
                    // or touchstart event. Without this additional trigger the first mousedown/touchstart would only select
                    // the resize node. For moving it, a further mousedown/touchstart is requrired. By triggering this event
                    // again, it is possible to move a resize node already with the first mousedown/touchstart.
                    resizerNode.trigger(event);
                }
            } else {

                // clicked somewhere else: calculate logical selection from browser selection,
                // after browser has processed the mouse event
                selection.processBrowserEvent(event);

            }
        }

        /**
         * Handler that is called, if the application loses the focus.
         */
        function processBlur() {

            // remove an existing blinking cursor simulation (26292)
            if (selection.isSimulatedCursorActive()) {
                // Not using Position.getDOMPosition to find blinking node at saved
                // position, but simply remove it completely by finding all elements
                // in the document.
                selection.clearSimulatedCursor();
            }

        }

        function processMouseUp(event) {

            var // whether the document is opened in read-only mode
                readOnly = self.getEditMode() !== true,
                // a drawing node and a tab node
                drawing = null, drawingNode = null, tabNode = null,
                // logical positions
                startPosition = null, oxoPosition = null,
                // whether this event was triggered by a right click
                rightClick = (event.button === 2);

            activeMouseDownEvent = false;

            // make sure that the clickable parts in the hyperlink popup are
            // processed by the browser but we don't call selection.processBrowserEvent
            // as otherwise we have a bad browser selection which produce assertions
            // and have other strange effects (especially for IE)
            if (Hyperlink.isClickableNode($(event.target), readOnly)) {
                return;
            }

            // handle mouse events in edit mode only
            if (readOnly) {
                selection.processBrowserEvent(event);
                return;
            }

            tabNode = $(event.target).closest(DOM.TAB_NODE_SELECTOR);
            if (tabNode.length) {
                // tabulator handler to decide if cursor goes before or after tab
                if (tabNode.offset().left + tabNode.width() / 2 < event.clientX) {
                    startPosition = Position.getOxoPosition(self.getCurrentRootNode(), tabNode.next(), 0);
                    selection.setTextSelection(startPosition);
                    return;
                } else {
                    startPosition = Position.getOxoPosition(self.getCurrentRootNode(), tabNode, 0);
                    selection.setTextSelection(startPosition);
                    return;
                }
            }

            if (selection.getSelectionType() === 'drawing') {
                drawingNode = $(event.target).closest(DrawingFrame.NODE_SELECTOR);

                if (rightClick && drawingNode.length === 0) {
                    oxoPosition = Position.getOxoPositionFromPixelPosition(editdiv, event.pageX, event.pageY);
                    selection.setTextSelection(oxoPosition.start);

                } else {
                    // mouse up while drawing selected: selection does not change,
                    // but scroll drawing completely into the visible area
                    drawing = selection.getSelectedDrawing();
                    app.getView().scrollToChildNode(drawing);
                }

                if (_.browser.IE < 11 || Utils.IOS) {
                    // in IE (< 11) processBrowserEvent was not called in mouseDown. Therefore
                    // it is now necessary to call it in mouseup, even if drawings are
                    // selected. (Fix for 29382: Excluding IE 11)

                    // on the iPad when deselecting an image,
                    // processBrowserEvent is called in mouseDown, but applyBrowserSelection is not.
                    // so we need to call processBrowserEvent also in mouseUp to finally call applyBrowserSelection.
                    selection.processBrowserEvent(event);
                } else if (_.browser.IE === 11) {
                    // Avoiding windows frame around the drawing by repainting OX Text drawing frame
                    // TODO: Is this still necessary?
                    // -> removing this for text frames, because text selection will be disturbed
                    if (!DrawingFrame.isTextFrameShapeDrawingFrame(drawing)) {
                        DrawingFrame.clearSelection(drawing);
                        DrawingResize.drawDrawingSelection(app, drawing);
                    }
                }

            } else {
                if (rightClick && !selection.hasRange()) {
                    oxoPosition = Position.getOxoPositionFromPixelPosition(editdiv, event.pageX, event.pageY);
                    if (oxoPosition) { selection.setTextSelection(oxoPosition.start); }

                } else {
                    // calculate logical selection from browser selection, after
                    // browser has processed the mouse event
                    selection.processBrowserEvent(event);
                }
            }
        }

        /**
         * Event handler for scrolling tables horizontaly in draft mode
         */
        var processTouchEventsForTableDraftMode = (function () {
            var currentTable, // target table
                startXpos = 0, // initial pageX value on touchstart for calculating event offset
                currentXpos = 0, // temp pageX value got on touchmove
                tableCssLeft = 0,
                pageContentWidth = 0,
                currentTableWidth = 0;

            return function (event) {
                pageContentWidth = editdiv.find(DOM.PAGECONTENT_NODE_SELECTOR).width();
                currentTable = $(event.target).parents(DOM.TABLE_NODE_SELECTOR).last();
                currentTableWidth = currentTable.width();

                switch (event.type) {
                case 'touchstart':
                    if (event.originalEvent && event.originalEvent.changedTouches) {
                        startXpos = event.originalEvent.changedTouches[0].pageX;
                    }
                    tableCssLeft = parseInt(currentTable.css('left'), 10);
                    break;
                case 'touchmove':
                    if (event.originalEvent && event.originalEvent.changedTouches) {
                        currentXpos = (event.originalEvent.changedTouches[0].pageX - startXpos);
                    }
                    if (Math.abs(currentXpos) < pageContentWidth / 2) {
                        currentTable.css({ left: tableCssLeft + currentXpos });
                    } else {
                        currentTable.css({ left: (currentXpos > 0) ? 0 : (pageContentWidth - currentTableWidth) });
                    }
                    break;
                case 'touchend':
                    tableCssLeft = parseInt(currentTable.css('left'), 10);
                    if (tableCssLeft > 0) {
                        currentTable.css({ left: 0 });
                    } else if (tableCssLeft < pageContentWidth - currentTableWidth) {
                        currentTable.css({ left: pageContentWidth - currentTableWidth });
                    }
                    currentXpos = 0;
                    break;
                }
            };
        }());

        function processHeaderFooterEdit(event) {
            if (self.getEditMode() !== true || !app.isImportFinished()) {
                return;
            }
            event.stopPropagation();

            var $currTarget = $(event.currentTarget),
                $refToParentNode = $();

            if ($currTarget.hasClass('cover-overlay')) {
                $currTarget = $currTarget.parents('.header, .footer');
            }

            if (self.isHeaderFooterEditState()) {
                if (self.getHeaderFooterRootNode()[0] === $currTarget[0]) {
                    return;
                } else {
                    pageLayout.leaveHeaderFooterEditMode(self.getHeaderFooterRootNode());

                    // store parent element as a reference (element itself is going to be replaced)
                    $refToParentNode = $currTarget.parent();

                    // must be direct call
                    pageLayout.updateEditingHeaderFooter();
                }
            }
            if (DOM.isHeaderOrFooter($currTarget)) {

                if ($refToParentNode.length > 0) {
                    $currTarget = $refToParentNode.children('[data-container-id="' + $currTarget.attr('data-container-id') + '"]');
                }
                if (!$currTarget.length) {
                    // from edit mode of one h/f we doubleclicked on other, which is not created yet
                    $currTarget = $(event.currentTarget);
                    if ($currTarget.hasClass('cover-overlay')) {
                        $currTarget = $currTarget.parents('.header, .footer');
                    }
                }
                if (DOM.isHeaderOrFooter($currTarget)) {
                    pageLayout.enterHeaderFooterEditMode($currTarget);

                    selection.setNewRootNode($currTarget);
                    selection.setTextSelection(selection.getFirstDocumentPosition());
                }
            } else {
                Utils.warn('Editor.processHeaderFooterEdit(): Failed to fetch header or footer');
            }
        }

        function processMouseUpOnDocument(event) {

            if (activeMouseDownEvent) {
                activeMouseDownEvent = false;
                processMouseUp(event);
            }
        }

        // Registering keypress handler at the document for Chrome
        // to avoid problems with direct keypress to document
        // -> This handler is triggered once without event, therefore
        // the check for the event is required.
        function processKeypressOnDocument(event) {

            // Fix for the following scenario:
            // - local client uses Chrome browser
            // - local client selects drawing, without have edit privileges
            // - local client removes the focus from the Chrome browser
            // - remote client removes the selected drawing
            // - local client set browser focus by clicking on top of browser (or side pane)
            // - local client can type a letter direct with keypress into the document
            //   -> the letter is inserted into the document, no operation, no read-only warning
            // After removing the drawing in the remote client, the focus cannot be set correctly
            // in the local client, because the browser does not have the focus. Therefore the focus
            // remains in the clipboard and is then switched to the body element.

            var readOnly = self.getEditMode() !== true;

            // special handling for the following scenario:
            // - there is an event (once this handler is called without event)
            // - the target of the event was the body. So we cannot conflict with events triggered in the side pane, that bubble up to the body.
            // - the document is in read-only mode -> this might be removed in the future, if more than one user is able to edit the document.
            // - this edit application is visible
            // - the event is not triggered a second time -> this event is branded with the property '_triggeredAgain' to avoid endless loop.
            // - the currently focused element is the body element (which is a second test similar to the check of the event target).
            if (event && $(event.target).is('body') && readOnly && app.isActive() && !event._triggeredAgain && $(window.document.activeElement).is('body')) {
                event.preventDefault();
                // Trigger the same event again at the div.page once more (-> read-only warning will be displayed)
                event.target = editdiv[0];
                event._triggeredAgain = true;
                editdiv.trigger(event);
            }
        }

        // Fix for 29751: IE supports double click for word selection
        function processDoubleClick(event) {

            var // the boundaries of the double-clicked word
                wordBoundaries = null;

            if (_.browser.IE) {

                if (DOM.isTextSpan(event.target)) {
                    doubleClickEventWaiting = true;
                    // Executing the code deferred, so that a triple click becomes possible.
                    self.executeDelayed(function () {
                        // modifying the double click to a triple click, if a further mousedown happened
                        if (tripleClickActive) { event.type = 'tripleclick'; }
                        doubleClickEventWaiting = false;
                        tripleClickActive = false;
                        selection.processBrowserEvent(event);
                    }, doubleClickTimeOut, 'Text: processDoubleClick');

                }
            } else if (_.browser.WebKit) { // 39357
                if (DOM.isTextSpan(event.target) && DOM.isListParagraphNode(event.target.parentNode) && _.last(selection.getStartPosition()) > 0) {
                    wordBoundaries = Position.getWordBoundaries(self.getCurrentRootNode(), selection.getStartPosition(), { addFinalSpaces: true });
                    if (_.last(wordBoundaries[0]) === 0) { selection.setTextSelection.apply(selection, wordBoundaries); }
                }
            }
        }

        function processKeyDown(event) {

            var readOnly = self.getEditMode() !== true,
                isCellSelection = false,
                currentBrowserSelection = null,
                checkPosition = null,
                returnObj = null,
                startPosition,
                endPosition,
                paraLen,
                activeRootNode = self.getCurrentRootNode(),
                // keys combination used to jump into header/footer on diff OS
                keyCombo = null;

            // saving the keyDown event for later usage in processKeyPressed
            lastKeyDownEvent = event;

            // open the change track popup, when available, with 'ALT' + 'DOWN_ARROW' combination
            if (KeyCodes.matchKeyCode(event, 'DOWN_ARROW', { alt: true })) {
                app.getView().showChangeTrackPopup();
                return;
            }

            // close the change track popup, when available, with 'ALT' + 'UP_ARROW' combination
            if (KeyCodes.matchKeyCode(event, 'UP_ARROW', { alt: true })) {
                var changeTrackPopup = app.getView().getChangeTrackPopup();
                if (changeTrackPopup) { changeTrackPopup.hide(); }
                return;
            }

            if (roIOSParagraph) {
                if (self.getEditMode() === true) {
                    $(roIOSParagraph).removeAttr('contenteditable');
                }
                roIOSParagraph = null;
            }

            // Fix for 32910: Client reports internal error if user types as soon as possible after pasting lengthly content with lists
            if (self.getBlockKeyboardEvent()) {
                event.preventDefault();
                return false;
            }

            // special handling for IME input on IE
            // having a browser selection that spans up ak range,
            // the IME input session start event (compositionstart) is omitted.
            // Normally we have to call deleteSelected() to remove the selection within
            // compositionstart. Unfortunately IE is NOT able to work correctly if the
            // browser selection is changed during a IME session. Therefore this code
            // cancels the IME session whenever we detect a selected range to preserve
            // a consistent document state.
            // TODO: We have to find a better solution to support IME and selections.
            if (_.browser.IE && (event.keyCode === KeyCodes.IME_INPUT) && selection.hasRange()) {
                event.preventDefault();
                return false;
            }

            if (DOM.isIgnorableKey(event.keyCode) || isBrowserShortcutKeyEvent(event)) {
                return;
            }

            if (Hyperlink.isClickableNode($(event.target), readOnly)) {
                // We need the keyboard events for the popup which normally contains
                // an external URL which must be opened by the browser.
                return;
            }

            dumpEventObject(event);

            // 'escape' converts a cell selection into a text selection
            if ((event.keyCode === KeyCodes.ESCAPE) && (selection.getSelectionType() === 'cell')) {
                Table.resolveCellSelection(selection, activeRootNode);
                event.preventDefault();
                return false;
            }

            // resetting values for table cell selection, if no 'shift' is pressed
            if ((selection.getAnchorCellRange() !== null) && !event.shiftKey) {
                selection.setAnchorCellRange(null);
            }

            if (DOM.isCursorKey(event.keyCode)) {

                // deferring all cursor position calculations, because insertText is also deferred (39358)
                self.executeDelayed(function () {

                    isCellSelection = (selection.getSelectionType() === 'cell');

                    if (isCellSelection) {
                        currentBrowserSelection = selection.getBrowserSelection();
                    }

                    // Setting cursor up and down in tables on MS IE or Webkit
                    if ((event.keyCode === KeyCodes.UP_ARROW || event.keyCode === KeyCodes.DOWN_ARROW) && (!event.shiftKey) && (!event.ctrlKey)) {
                        // Setting cursor up and down in tables on MS IE or Webkit
                        if ((_.browser.IE || _.browser.WebKit) && self.isPositionInTable()) {
                            if (Table.updownCursorTravel(event, selection, activeRootNode, { up: event.keyCode === KeyCodes.UP_ARROW, down: event.keyCode === KeyCodes.DOWN_ARROW, readonly: readOnly })) {
                                return;
                            }
                        }
                        // Setting cursor up and down before and behind exceeded-size tables in Firefox (27395)
                        if (_.browser.Firefox) {
                            if (Table.updownExceededSizeTable(event, selection, activeRootNode, { up: event.keyCode === KeyCodes.UP_ARROW, down: event.keyCode === KeyCodes.DOWN_ARROW })) {
                                return;
                            }
                        }
                    }

                    // Changing table cell selections in FireFox when shift is used with arrow keys.
                    // Switching from text selection to cell selection already happened in 'selection.moveTextCursor()'.
                    if (_.browser.Firefox && DOM.isArrowCursorKey(event.keyCode) && event.shiftKey && isCellSelection && self.isPositionInTable()) {

                        event.preventDefault();

                        if ((event.keyCode === KeyCodes.UP_ARROW) || (event.keyCode === KeyCodes.DOWN_ARROW)) {
                            Table.changeCellSelectionVert(app, selection, currentBrowserSelection, { backwards: event.keyCode === KeyCodes.UP_ARROW });
                        }

                        if ((event.keyCode === KeyCodes.LEFT_ARROW) || (event.keyCode === KeyCodes.RIGHT_ARROW)) {
                            Table.changeCellSelectionHorz(app, selection, currentBrowserSelection, { backwards: event.keyCode === KeyCodes.LEFT_ARROW });
                        }

                        return;
                    }

                    if (_.browser.IE && DOM.isArrowCursorKey(event.keyCode) && event.shiftKey &&
                        (self.isPositionInTable() || Position.isPositionInTable(activeRootNode, selection.getEndPosition()) || Position.isPositionInTable(activeRootNode, selection.getStartPosition()))) {

                        if ((event.keyCode === KeyCodes.LEFT_ARROW) || (event.keyCode === KeyCodes.RIGHT_ARROW)) {
                            if (Table.changeCellSelectionHorzIE(app, activeRootNode, selection, currentBrowserSelection, { backwards: event.keyCode === KeyCodes.LEFT_ARROW })) {
                                event.preventDefault();
                                return;
                            }
                        }

                    }
                    // jumping to header/footer on Mac and other OS
                    keyCombo = _.browser.MacOS ? { ctrl: true, meta: true } : { ctrl: true, alt: true };

                    if (KeyCodes.matchKeyCode(event, 'PAGE_UP', keyCombo)) { // Go to header (if exists, if not, create it first) on the page where cursor is
                        event.preventDefault();
                        if (!self.isHeaderFooterEditState() && !readOnly) {
                            pageLayout.jumpToHeaderOnCurrentPage();
                            return;
                        }
                    } else if (KeyCodes.matchKeyCode(event, 'PAGE_DOWN', keyCombo)) { // Go to footer (if exists, if not, create it first) on the page where cursor is
                        event.preventDefault();
                        if (!self.isHeaderFooterEditState() && !readOnly) {
                            pageLayout.jumpToFooterOnCurrentPage();
                            return;
                        }
                    } else if (KeyCodes.matchKeyCode(event, 'END', { ctrl: true })) {
                        event.preventDefault();
                        selection.setTextSelection(selection.getLastDocumentPosition());
                    }

                    // any navigation key: change drawing selection to text selection before
                    selection.selectDrawingAsText();

                    // let browser process the key event (this may create a drawing selection)
                    selection.processBrowserEvent(event, { readonly: readOnly }); // processing is dependent from read-only mode

                }, inputTextTimeout, 'Text: processKeyDown: isCursorKey');

                return;
            }

            // handle just cursor, copy, search and the global F6 accessibility events if in read only mode
            if (readOnly && !isCopyKeyEventKeyDown(event) && !isF6AcessibilityKeyEvent(event) && !KeyCodes.matchKeyCode(event, 'TAB', { shift: null }) && !KeyCodes.matchKeyCode(event, 'ESCAPE', { shift: null }) && !isSearchKeyEvent(event)) {
                if (!app.isImportFinished()) {
                    app.rejectEditAttempt('loadingInProgress');
                } else {
                    app.rejectEditAttempt();
                }
                event.preventDefault();
                return;
            }

            if (event.keyCode === KeyCodes.DELETE && !event.shiftKey) {

                event.preventDefault();

                // if there is an active 'inputTextPromise', it has to be set to null now, so that
                // following text input is included into a new deferred
                inputTextPromise = null;

                // Executing the code for 'DELETE' deferred. This is necessary because of deferred input of text. Defer time is 'inputTextTimeout'
                self.executeDelayed(function () {

                    startPosition = selection.getStartPosition();

                    if (event.ctrlKey) {
                        var storedEndPos = Position.getRangeForDeleteWithCtrl(self.getCurrentRootNode(), startPosition, false);

                        selection.setTextSelection(startPosition, storedEndPos, { simpleTextSelection: true });
                    }

                    if (selection.hasRange()) {
                        self.deleteSelected()
                        .done(function () {
                            startPosition = selection.getStartPosition(); // refresh required after deleteSelected()
                            selection.setTextSelection((lastOperationEnd && (startPosition.length === lastOperationEnd.length)) ? startPosition : lastOperationEnd, null, { simpleTextSelection: true });
                        });
                    } else {
                        endPosition = selection.getEndPosition();

                        returnObj = Position.skipDrawingsAndTables(_.clone(startPosition), _.clone(endPosition), activeRootNode, { backwards: false });
                        paraLen = returnObj.paraLen;
                        startPosition = returnObj.start;
                        endPosition = returnObj.end;

                        if (startPosition[startPosition.length - 1] < paraLen) {
                            self.deleteRange(startPosition, endPosition, { setTextSelection: false, handleUnrestorableContent: true })
                            .always(function () {
                                // If change tracking is active, lastOperationEnd is set inside deleteRange
                                if (changeTrack.isActiveChangeTracking() && _.isArray(lastOperationEnd) && !_.isEqual(lastOperationEnd, startPosition)) {
                                    selection.setTextSelection(lastOperationEnd, null, { simpleTextSelection: true });
                                } else {
                                    selection.setTextSelection(startPosition, null, { simpleTextSelection: true });
                                }
                            });

                        } else {
                            var mergeselection = _.clone(startPosition),
                                characterPos = mergeselection.pop(),
                                nextParagraphPosition = _.copy(mergeselection);

                            nextParagraphPosition[nextParagraphPosition.length - 1] += 1;

                            var domPos = Position.getDOMPosition(activeRootNode, nextParagraphPosition),
                                nextIsTable = false,
                                isLastParagraph = false;

                            if (domPos) {
                                if (DOM.isTableNode(domPos.node)) {
                                    nextIsTable = true;
                                }
                            } else {
                                nextParagraphPosition[nextParagraphPosition.length - 1] -= 1;
                                isLastParagraph = true;
                            }

                            if (DOM.isMergeableParagraph(Position.getParagraphElement(activeRootNode, mergeselection))) {
                                if (changeTrack.isActiveChangeTracking() && !changeTrack.isInsertNodeByCurrentAuthor(domPos.node)) {  // not really deleting paragraph, but marking for deletion
                                    self.applyOperations({
                                        name: Operations.SET_ATTRIBUTES,
                                        start: nextParagraphPosition,  // the latter paragraph needs to be marked for removal
                                        attrs: { changes: { removed: changeTrack.getChangeTrackInfo() }}
                                    });
                                    // Setting the cursor to the correct position, if no character was deleted
                                    lastOperationEnd = _.clone(mergeselection);
                                    lastOperationEnd[lastOperationEnd.length - 1] += 1;
                                    lastOperationEnd = Position.appendNewIndex(lastOperationEnd);

                                } else {
                                    if (paraLen === 0 && !DOM.isManualPageBreakNode(domPos.node)) {  // Simply remove an empty paragraph
                                        self.deleteRange(_.initial(_.clone(startPosition)));
                                        // updating lists, if required. This is the case, if the deleted paragraph is part of a list
                                        handleTriggeringListUpdate(domPos.node.previousSibling);
                                    } else { // Merging two paragraphs
                                        self.mergeParagraph(mergeselection);
                                    }
                                }
                            }

                            if (nextIsTable) {
                                if (characterPos === 0) {
                                    // removing empty paragraph
                                    var localPos = _.clone(startPosition);
                                    localPos.pop();
                                    if (changeTrack.isActiveChangeTracking() && !changeTrack.isInsertNodeByCurrentAuthor(domPos.node)) {  // not really deleting paragraph, but marking for deletion
                                        self.applyOperations({
                                            name: Operations.SET_ATTRIBUTES,
                                            start: localPos,  // the latter paragraph needs to be marked for removal
                                            attrs: { changes: { removed: changeTrack.getChangeTrackInfo() }}
                                        });
                                        // Setting the cursor to the correct position, if no character was deleted
                                        lastOperationEnd = _.clone(localPos);
                                        lastOperationEnd[lastOperationEnd.length - 1] += 1;
                                        lastOperationEnd = Position.getFirstPositionInParagraph(activeRootNode, lastOperationEnd);

                                    } else {
                                        undoManager.enterUndoGroup(function () {

                                            var // the current paragraph
                                                currentParagraph = Position.getContentNodeElement(activeRootNode, startPosition.slice(0, -1)),
                                                // the previous allowed neighbour of the paragraph
                                                prevNeighbour = DOM.getAllowedNeighboringNode(currentParagraph, { next: false }),
                                                // the next allowed neighbour of the paragraph
                                                nextNeighbour = DOM.getAllowedNeighboringNode(currentParagraph),
                                                // the logical position of the previous neighbour node
                                                prevNeighbourPos = null;

                                            // remove paragraph explicitely
                                            if (!DOM.isImplicitParagraphNode(currentParagraph)) { // first handle if implicit paragraph at that position
                                                self.deleteRange(localPos);
                                                nextParagraphPosition[nextParagraphPosition.length - 1] -= 1;
                                            } else {
                                                $(currentParagraph).remove();
                                            }

                                            if (nextNeighbour && DOM.isTableNode(nextNeighbour)) {

                                                // check if table has paragraph with manual page break set, and remove it
                                                if (DOM.isManualPageBreakNode(nextNeighbour)) {
                                                    removePageBreakBeforeAttribute(nextNeighbour);
                                                }

                                                // if both neighbours are tables, and it's possible, merge them
                                                if (prevNeighbour && DOM.isTableNode(prevNeighbour) && Table.mergeableTables(prevNeighbour, nextNeighbour) && !app.isODF()) {
                                                    prevNeighbourPos = Position.getOxoPosition(activeRootNode, prevNeighbour);
                                                    self.mergeTable(prevNeighbourPos, { next: true });
                                                }
                                            }

                                            return $.when();
                                        }, this); // end of enterUndoGroup()
                                    }
                                }
                                startPosition = Position.getFirstPositionInParagraph(activeRootNode, nextParagraphPosition);
                            } else if (isLastParagraph) {

                                if (Position.isPositionInTable(activeRootNode, nextParagraphPosition)) {

                                    returnObj = Position.getFirstPositionInNextCell(activeRootNode, nextParagraphPosition);
                                    startPosition = returnObj.position;
                                    var endOfTable = returnObj.endOfTable;
                                    if (endOfTable) {
                                        startPosition[startPosition.length - 1] += 1;
                                        startPosition = Position.getFirstPositionInParagraph(activeRootNode, startPosition);
                                    }
                                }
                            }

                            if (changeTrack.isActiveChangeTracking() && _.isArray(lastOperationEnd) && !_.isEqual(lastOperationEnd, startPosition)) {
                                selection.setTextSelection(lastOperationEnd, lastOperationEnd, { simpleTextSelection: true });
                            } else {
                                selection.setTextSelection(startPosition, null, { simpleTextSelection: true });
                            }
                        }
                    }

                }, inputTextTimeout, 'Text: processKeyDown: delete');

            } else if (event.keyCode === KeyCodes.BACKSPACE && !event.altKey) {

                event.preventDefault();

                // if there is an active 'inputTextPromise', it has to be set to null now, so that
                // following text input is included into a new deferred
                inputTextPromise = null;

                // Executing the code for 'BACKSPACE' deferred. This is necessary because of deferred input of text. Defer time is 'inputTextTimeout'
                var handleBack = function () {

                    startPosition = selection.getStartPosition();

                    if (event.ctrlKey) {
                        var storedEndPos = Position.getRangeForDeleteWithCtrl(self.getCurrentRootNode(), startPosition, true);

                        selection.setTextSelection(storedEndPos, startPosition, { simpleTextSelection: true });
                        startPosition = storedEndPos;
                    }

                    if (selection.hasRange()) {
                        self.deleteSelected()
                        .done(function () {
                            startPosition = selection.getStartPosition(); // refresh required after deleteSelected()
                            selection.setTextSelection(startPosition, null, { simpleTextSelection: true });
                        });
                    } else {
                        returnObj = Position.skipDrawingsAndTables(_.clone(startPosition), _.clone(selection.getEndPosition()), activeRootNode, { backwards: true });
                        if (returnObj.start === null || returnObj.end === null) { // #33154 backspace on table with position [0, 0] and exceeded size limit
                            return;
                        }

                        startPosition = returnObj.start;
                        endPosition = returnObj.end;

                        if (startPosition[startPosition.length - 1] > 0) {

                            startPosition[startPosition.length - 1] -= 1;
                            endPosition[endPosition.length - 1] -= 1;
                            self.deleteRange(startPosition, endPosition, { setTextSelection: false, handleUnrestorableContent: true })
                            .fail(function () {
                                // keeping the current selection
                                startPosition[startPosition.length - 1] += 1;
                            })
                            .always(function () {
                                selection.setTextSelection(startPosition, null, { simpleTextSelection: true });
                            });

                        } else if (startPosition[startPosition.length - 2] >= 0) {

                            var paragraph = Position.getLastNodeFromPositionByNodeName(activeRootNode, startPosition, DOM.PARAGRAPH_NODE_SELECTOR),
                                listLevel = -1,
                                atParaStart = startPosition[startPosition.length - 1] === 0,
                                styleId = null,
                                elementAttributes = null,
                                styleAttributes = null,
                                paraAttributes = null;

                            if (!_(startPosition).all(function (value) { return (value === 0); })) {

                                startPosition[startPosition.length - 2] -= 1;
                                startPosition.pop();  // -> removing last value from startPosition !

                                var length = Position.getParagraphLength(activeRootNode, startPosition),
                                    domPos = Position.getDOMPosition(activeRootNode, startPosition),
                                    prevIsTable = false,
                                    paraBehindTable = null,
                                    localStartPos = null,
                                    fromStyle = false,
                                    prevPara;

                                if ((domPos) && (DOM.isTableNode(domPos.node))) {
                                    prevIsTable = true;
                                }

                                if (atParaStart) {

                                    elementAttributes = paragraphStyles.getElementAttributes(paragraph);
                                    styleId = elementAttributes.styleId;
                                    paraAttributes = elementAttributes.paragraph;
                                    listLevel = paraAttributes.listLevel;
                                    styleAttributes = paragraphStyles.getStyleAttributeSet(styleId).paragraph;

                                    if (paraAttributes.listStyleId !== '') {
                                        fromStyle = paraAttributes.listStyleId === styleAttributes.listStyleId;
                                    }

                                    if (fromStyle) {
                                        // works in OOX as listStyle 0 is a non-numbering style
                                        // will not work in OpenDocument
                                        self.setAttribute('paragraph', 'listStyleId', 'L0');
                                        return;
                                    } else if (listLevel >= 0) {
                                        implModifyListLevel(listLevel, false, -1);
                                        return;
                                    } else if (styleId === self.getDefaultUIParagraphListStylesheet()) {
                                        self.setAttributes('paragraph', { styleId: self.getDefaultUIParagraphStylesheet() });
                                        // invalidate previous paragraph to recalculate bottom distance
                                        prevPara = Utils.findPreviousNode(activeRootNode, paragraph, DOM.PARAGRAPH_NODE_SELECTOR);
                                        if (prevPara) {
                                            paragraphStyles.updateElementFormatting(prevPara);
                                        }
                                        return;
                                    }
                                }

                                if (startPosition[startPosition.length - 1] >= 0) {
                                    if (!prevIsTable) {
                                        if (changeTrack.isActiveChangeTracking() && !changeTrack.isInsertNodeByCurrentAuthor(paragraph)) {
                                            localStartPos = _.clone(startPosition);
                                            localStartPos[localStartPos.length - 1] += 1;
                                            self.applyOperations({
                                                name: Operations.SET_ATTRIBUTES,
                                                start: localStartPos,  // the latter paragraph is marked for removal
                                                attrs: { changes: { removed: changeTrack.getChangeTrackInfo() }}
                                            });
                                        } else {
                                            if (Position.getParagraphLength(activeRootNode, startPosition) === 0 && !DOM.isManualPageBreakNode(paragraph)) {
                                                // simply remove first paragraph, if it is empty
                                                self.deleteRange(startPosition);
                                            } else {
                                                // merge two following paragraphs, if both are not empty
                                                self.mergeParagraph(startPosition);
                                            }
                                        }
                                    } else {

                                        checkPosition = _.clone(startPosition);
                                        checkPosition[checkPosition.length - 1] += 1; // increasing paragraph again
                                        //checkPosition.push(0);

                                        // if it is not an implicit paragraph now, it can be removed via operation
                                        paraBehindTable = Position.getParagraphElement(activeRootNode, checkPosition);
                                        if ((Position.getParagraphNodeLength(paraBehindTable) === 0) && !DOM.isImplicitParagraphNode(paraBehindTable)) {
                                            // remove paragraph explicitely or mark it as removed
                                            if (changeTrack.isActiveChangeTracking() && !changeTrack.isInsertNodeByCurrentAuthor(paragraph)) {
                                                self.applyOperations({
                                                    name: Operations.SET_ATTRIBUTES,
                                                    start: checkPosition,  // the latter paragraph is marked for removal
                                                    attrs: { changes: { removed: changeTrack.getChangeTrackInfo() }}
                                                });
                                            } else {
                                                undoManager.enterUndoGroup(function () {
                                                    var prevNeighbour = DOM.getAllowedNeighboringNode(paraBehindTable, { next: false }),
                                                        nextNeighbour = DOM.getAllowedNeighboringNode(paraBehindTable),
                                                        prevNeighbourPos = Position.getOxoPosition(activeRootNode, prevNeighbour);
                                                    // remove paragraph explicitely
                                                    self.applyOperations({ name: Operations.DELETE, start: checkPosition });
                                                    // check if table has paragraph with manual page break set, and remove it
                                                    if (DOM.isTableNode(nextNeighbour) && DOM.isManualPageBreakNode(nextNeighbour)) {
                                                        removePageBreakBeforeAttribute(nextNeighbour);
                                                    }
                                                    // if both neighbours are tables, and it's possible, merge them
                                                    if (DOM.isTableNode(prevNeighbour) && DOM.isTableNode(nextNeighbour) && Table.mergeableTables(prevNeighbour, nextNeighbour) && !app.isODF()) {
                                                        self.mergeTable(prevNeighbourPos, { next: true });
                                                    }
                                                    return $.when();
                                                }, this); // end of enterUndoGroup()
                                            }
                                        }
                                    }
                                }

                                if (prevIsTable) {
                                    startPosition = Position.getLastPositionInParagraph(activeRootNode, startPosition, { ignoreImplicitParagraphs: true });
                                } else {
                                    var isFirstPosition = (startPosition[startPosition.length - 1] < 0) ? true : false;
                                    if (isFirstPosition) {
                                        if (DOM.isFirstContentNodeInTextframe(paragraph)) {
                                            return; // simply ignore 'backspace' at start position of text frame
                                        } else if (Position.isPositionInTable(activeRootNode, startPosition)) {

                                            returnObj = Position.getLastPositionInPrevCell(activeRootNode, startPosition, { ignoreImplicitParagraphs: true });
                                            startPosition = returnObj.position;
                                            var beginOfTable = returnObj.beginOfTable;
                                            if (beginOfTable) {
                                                var table = Position.getContentNodeElement(activeRootNode, startPosition);
                                                if (DOM.isFirstContentNodeInTextframe(table)) {
                                                    return; // simply ignore 'backspace' at start position of text frame
                                                }
                                                // check if table has paragraph with manual page break set, and remove it
                                                if (DOM.isManualPageBreakNode(table)) {
                                                    removePageBreakBeforeAttribute(table);
                                                }
                                                // update position of cursor
                                                startPosition[startPosition.length - 1] -= 1;
                                                startPosition = Position.getLastPositionInParagraph(activeRootNode, startPosition, { ignoreImplicitParagraphs: true });
                                            }
                                        } else {
                                            startPosition.push(length);
                                        }
                                    } else {
                                        startPosition.push(length);
                                    }
                                }
                            } else {
                                // The position contains only '0's -> Backspace at the beginning of the document -> handle paragraphs with list style.
                                elementAttributes = paragraphStyles.getElementAttributes(paragraph);
                                styleId = elementAttributes.styleId;
                                paraAttributes = elementAttributes.paragraph;
                                listLevel = paraAttributes.listLevel;
                                styleAttributes = paragraphStyles.getStyleAttributeSet(styleId).paragraph;

                                if ((paraAttributes.listStyleId !== '') && (paraAttributes.listStyleId === styleAttributes.listStyleId)) {
                                    // works in OOX as listStyle 0 is a non-numbering style will not work in OpenDocument
                                    self.setAttribute('paragraph', 'listStyleId', 'L0');
                                } else if (listLevel >= 0) {
                                    implModifyListLevel(listLevel, false, -1);
                                } else if (styleId === self.getDefaultUIParagraphListStylesheet()) {
                                    self.setAttributes('paragraph', { styleId: self.getDefaultUIParagraphStylesheet() });
                                }
                            }

                            selection.setTextSelection(startPosition, null, { simpleTextSelection: true });
                        }
                    }

                };
                if (_.browser.Android) {
                    //it has to be syncron on android, otherwise it can overjump some letters
                    handleBack();
                } else {
                    self.executeDelayed(handleBack, inputTextTimeout, 'Text: processKeyDown: backSpace');
                }

            } else if (Utils.boolXor(event.metaKey, event.ctrlKey) && !event.altKey) {

                if (event.keyCode === KeyCodes.ENTER) {
                    // insert manual page break, but not inside text frames
                    if (selection.isAdditionalTextframeSelection()) {
                        event.preventDefault();
                        return;
                    }
                    self.insertManualPageBreak();
                }

                // prevent browser from evaluating the key event, but allow cut, copy and paste events
                if (!isPasteKeyEventKeyDown(event) && !isCopyKeyEventKeyDown(event) && !isCutKeyEventKeyDown(event)) {
                    event.preventDefault();
                }

                //self.clearPreselectedAttributes();
            } else if (KeyCodes.matchKeyCode(event, 'TAB', { shift: null })) {

                event.preventDefault();

                if (self.isPositionInTable()) {

                    // if there is an active 'inputTextPromise', it has to be set to null now, so that
                    // following text input is included into a new deferred
                    inputTextPromise = null;

                    // Executing the code for 'TAB' deferred. This is necessary because of deferred input of text.
                    // Defer time is 'inputTextTimeout'
                    self.executeDelayed(function () {

                        if (event.shiftKey) {
                            // Jump into first position of previous cell. Do not jump, if there is no previous cell
                            returnObj = Position.getFirstPositionInPreviousCell(activeRootNode, selection.getStartPosition());
                            if (!returnObj.beginOfTable) {
                                selection.setTextSelection(returnObj.position);
                            }
                        } else {
                            // Jumping into the next table cell or, if this is already the last cell of a table, insert a new row.
                            returnObj = Position.getFirstPositionInNextCell(activeRootNode, selection.getStartPosition());

                            if (returnObj.endOfTable) {

                                if (readOnly) {
                                    app.rejectEditAttempt();
                                    event.preventDefault();
                                    return;
                                }

                                if (self.isRowAddable()) {
                                    self.insertRow();
                                    selection.setTextSelection(lastOperationEnd);
                                } else {
                                    app.getView().rejectEditTextAttempt('tablesizerow'); // checking table size (26809)
                                    return;
                                }
                            } else {
                                selection.setTextSelection(returnObj.position);
                            }
                        }

                    }, inputTextTimeout, 'Text: processKeyDown: tab');

                // select next/prev drawing
                } else if (self.isDrawingSelected()) {
                    app.getView().executeControllerItem('document/selectDrawing', { backwards: KeyCodes.matchKeyCode(event, 'TAB', { shift: true }) });
                    return;

                } else {

                    if (readOnly) {
                        app.rejectEditAttempt();
                        event.preventDefault();
                        return;
                    }

                    // (shift)Tab: Change list indent (if in list) when selection is at first position in paragraph
                    var paragraph = Position.getLastNodeFromPositionByNodeName(activeRootNode, selection.getStartPosition(), DOM.PARAGRAPH_NODE_SELECTOR),
                        mustInsertTab = true;  // always true, independent from shift-key
                    if (!selection.hasRange() &&
                            _.last(selection.getStartPosition()) === Position.getFirstTextNodePositionInParagraph(paragraph)) {
                        var elementAttributes = paragraphStyles.getElementAttributes(paragraph),
                            styleId = elementAttributes.styleId,
                            paraAttributes = elementAttributes.paragraph,
                            listLevel = paraAttributes.listLevel,
                            styleAttributes = paragraphStyles.getStyleAttributeSet(styleId).paragraph;

                        if (paraAttributes.listStyleId !== '') {
                            mustInsertTab = false;
                            var fromStyle = listLevel === -1 || paraAttributes.listStyleId === styleAttributes.listStyleId;
                            if (listLevel === -1) {
                                listLevel = paraAttributes.listLevel;
                            }
                            if (listLevel !== -1) {

                                if (!fromStyle) {
                                    implModifyListLevel(listLevel, !event.shiftKey, 0);
                                } else {
                                    // numbering via paragraph style (e.g. outline numbering)
                                    var newStyleId = listCollection.findPrevNextStyle(paraAttributes.listStyleId, styleId, event.shiftKey);
                                    if (newStyleId) {
                                        self.setAttributes('paragraph', { styleId: newStyleId });
                                    }
                                }
                            }
                        }
                    }
                    if (mustInsertTab) {
                        // if there is an active 'inputTextPromise', it has to be set to null now, so that
                        // following text input is included into a new deferred
                        inputTextPromise = null;

                        // Executing the code for 'TAB' deferred. This is necessary because of deferred input of text.
                        // Defer time is 'inputTextTimeout'
                        self.executeDelayed(function () {
                            self.insertTab();
                        }, inputTextTimeout, 'Text: processKeyDown: insertTab');
                    }
                }
            } else if (event.keyCode === KeyCodes.ESCAPE) {

                // tracking of drawing node or table resizers canceled: do nothing else
                if (Tracking.cancelTracking()) { return false; }

                // deselect drawing node before the search bar
                if (self.isDrawingSelected()) {
                    selection.setTextSelection(selection.getEndPosition());
                    return false;
                }

                // deselect the paragraph in the text frame. Set the selection to the text frame itself.
                if (selection.isAdditionalTextframeSelection()) {
                    var start = Position.getOxoPosition(self.getCurrentRootNode(), selection.getSelectedTextFrameDrawing(), 0),
                        end = Position.increaseLastIndex(start);

                    // select drawing
                    selection.setTextSelection(start, end);
                    return false;
                }

                if (self.isHeaderFooterEditState()) {
                    // leave edit state
                    pageLayout.leaveHeaderFooterAndSetCursor(activeRootNode, activeRootNode.parent());
                    return false;
                }
                // else: let ESCAPE key bubble up the DOM tree
            } else if (KeyCodes.matchKeyCode(event, 'ENTER') && selection.getSelectionType() === 'drawing') {

                if (DrawingFrame.isTextFrameShapeDrawingFrame(selection.getSelectedDrawing())) {
                    event.preventDefault();
                    app.getView().executeControllerItem('document/setCursorIntoTextframe');
                }
            } else if (KeyCodes.matchKeyCode(event, 'F9')) {
                if (selection.hasRange()) {
                    fieldManager.updateHighlightedFieldSelection();
                } else if (fieldManager.isHighlightState()) {
                    fieldManager.updateHighlightedField();
                }
            }

        }

        /**
         * Handler for the keyPressed event, following the keyDown event.
         *
         * Info: Do not use comparisons with event.keyCode inside processKeyPressed!
         *  The event from processKeyDown, that was executed before this processKeyPressed
         *  is stored in the variable 'lastKeyDownEvent'. Please use this, if you want
         *  to make comparisons with keyCodes.
         *
         * Addition: Try to use this event ONLY for information about character
         *  that is beeing pressed on keyboard. For getting information about physical key being pressed,
         *  please use function processKeyDown! For combination of modifier keys and characters,
         *  use lastKeyDownEvent for modifier, and event for chars!
         *
         * @param event
         *  A jQuery keyboard event object.
         */
        function processKeyPressed(event) {

            var // Editor mode
                readOnly = self.getEditMode() !== true,
                // the char element
                c = null,
                hyperlinkSelection = null,
                // currently active root node
                activeRootNode = self.getCurrentRootNode();

            // Android Chrome Code
            if (_.browser.Android && event.keyCode === 0) { return; }
            if (event.keyCode === KeyCodes.ENTER && _.browser.Android) { event.preventDefault(); }

            dumpEventObject(event);

            if (lastKeyDownEvent && DOM.isIgnorableKey(lastKeyDownEvent.keyCode)) {
                return;
            }

            // Fix for 32910: Client reports internal error if user types as soon as possible after pasting lengthly content with lists
            if (self.getBlockKeyboardEvent()) {
                event.preventDefault();
                return false;
            }

            if (lastKeyDownEvent && DOM.isCursorKey(lastKeyDownEvent.keyCode)) {
                // Fix for 32008: CTRL+POS1/END create special characters on Mac/Chrome
                if (_.browser.WebKit && _.browser.MacOS && _.browser.Chrome && lastKeyDownEvent.ctrlKey) {
                    if ((lastKeyDownEvent.keyCode === 35) || (lastKeyDownEvent.keyCode === 36)) {
                        event.preventDefault();
                    }
                }
                return;
            }

            // needs to be checked on both keydown and keypress events
            if (isBrowserShortcutKeyEvent(event)) {
                return;
            }

            if (Hyperlink.isClickableNode($(event.target), readOnly)) {
                // We need the keyboard events for the popup which normally contains
                // an external URL which must be opened by the browser.
                return;
            }

            // TODO: Check if unnecessary! In most browsers returned already at processKeyDown!
            // handle just cursor, copy, escape, search and the global F6 accessibility events if in read only mode
            if (readOnly && !isCopyKeyEventKeyDown(lastKeyDownEvent) && !isF6AcessibilityKeyEvent(lastKeyDownEvent) && !KeyCodes.matchKeyCode(lastKeyDownEvent, 'TAB', { shift: null }) && !KeyCodes.matchKeyCode(lastKeyDownEvent, 'ESCAPE', { shift: null }) && !isSearchKeyEvent(event)) {
                if (!app.isImportFinished()) {
                    app.rejectEditAttempt('loadingInProgress');
                } else {
                    app.rejectEditAttempt();
                }
                event.preventDefault();
                return;
            }

            // Special behavior for the iPad! Due to strange event order in mobile Safari we have to
            // pass the keyPressed event to the browser. We use the textInput event to cancel the DOM
            // modification instead. The composition system needs the keyPressed event to start a
            // a new composition.
            if (!Utils.IOS) {
                // prevent browser from evaluating the key event, but allow cut, copy and paste events
                if (!isPasteKeyEventKeyDown(lastKeyDownEvent) && !isCopyKeyEventKeyDown(lastKeyDownEvent) && !isCutKeyEventKeyDown(lastKeyDownEvent)) {
                    event.preventDefault();
                }
            } else {
                // We still get keydown/keypressed events during a composition session in mobile Safari.
                // According to the W3C specification this shouldn't happen. Therefore we have to prevent
                // to call our normal keyPressed code.
                if (imeActive) {
                    return;
                }
            }

            c = getPrintableChar(event);

            // TODO
            // For now (the prototype), only accept single chars, but let the browser process, so we don't need to care about DOM stuff
            // TODO: But we at least need to check if there is a selection!!!

            if ((!lastKeyDownEvent.ctrlKey || (lastKeyDownEvent.ctrlKey && lastKeyDownEvent.altKey && !lastKeyDownEvent.shiftKey)) && !lastKeyDownEvent.metaKey && (c.length === 1)) {

                undoManager.enterUndoGroup(function () {

                    var // the logical start position of the selection
                        startPosition = null,
                        // the string to be inserted with insert text operation
                        insertText = null,
                        // whether the insertTextDef was created
                        createdInsertTextDef = false;

                    // Helper function for collecting and inserting characters. This function is typically executed synchronously,
                    // but in special cases with a selection containing unrestorable content, it is executed deferred.
                    function doInsertCharacter() {

                        // merging input text, if there is already a promise and there are not already 5 characters deferred
                        if ((inputTextPromise) && (activeInputText) && (activeInputText.length < maxTextInputChars)) {
                            // deleting existing deferred for insert text and creating a new deferred
                            // containing the new and the old text
                            inputTextPromise.abort();
                            activeInputText += c;
                        } else {
                            // creating a completely new deferred for insert text
                            inputTextPromise = null;
                            activeInputText = c;
                        }

                        insertText = activeInputText;
                        // Calling insertText deferred. Restarting, if there is further text input in the next 'inputTextTimeout' ms.
                        inputTextPromise = self.executeDelayed(function () {
                            startPosition = selection.getStartPosition();

                            self.insertText(insertText, startPosition, preselectedAttributes);

                            // set cursor behind character
                            selection.setTextSelection(Position.increaseLastIndex(startPosition, insertText.length), null, { simpleTextSelection: true, insertOperation: true });

                            // Setting to null, so that new timeouts can be started.
                            // If there is already a running new inputTextPromise (fast typing), then this is also set to null, so that it cannot
                            // add further characters. But this is not problematic. Example: '123 456' -> After typing space, the inputTextPromise
                            // is set to null. Then it can happen that '4' and '5' are typed, and then this deferred function is executed. In this
                            // case the inputTextPromise for the '45' is terminated, even though there would be sufficient time to add the '6' too.
                            // But this is not important, because this concatenation is only required for faster type feeling.
                            inputTextPromise = null;
                            // Resolving the insertTextDef for the undoGroup. This is very important, because otherwise there could be
                            // pending actions that will never be sent to the server (33226). Every new character in 'processKeyPressed'
                            // needs the same 'insertTextDef', because this is registered for the undoGroup with the return value of
                            // enterUndoGroup. A first character registers the promise of the 'insertTextDef' and a second very fast
                            // following character must use the same (already created) insertTextDef. It is finally resolved
                            // during the execution of this deferred function. What happens, if there is already a new inputTextPromise?
                            // Again the example '123 456'. After typing '123 ' the inputTextPromise is set to null. Then '456' is typed
                            // and collected. Then this deferred function is executed for '123 ', deletes the inputTextPromise (so that
                            // '456' is also completed) and resolves the insertTextDef, so that it will be set to null (via the 'always'
                            // function). This makes also the '456' a complete block with resolved undoGroup. If later this function is
                            // executed for '456', the insertTextDef might already be set to null. But in this case it simply does not
                            // need to be resolved, because this already happened at the previous run of the function with '123 '.
                            // The correct undoGroups are anyhow created by merging following insertText operations (mergeUndoActionHandler).
                            if (insertTextDef) {  // -> the insertTextDef might be already resolved in previous call of this function
                                insertTextDef.resolve();  // resolving the deferred for the undo group
                            }
                        }, inputTextTimeout, 'Text: insertText');

                        // After a space, always a new text input promise is required -> text will be written.
                        // Setting inputTextPromise to null, makes it unreachable for abort() .
                        if (c === ' ') { inputTextPromise = null; }
                    }

                    // Setting insertTextDef, if it is still null.
                    // Because of the undoGroup this insertTextDef must be global (33226). It cannot be local and be created
                    // in every run inside this 'undoManager.enterUndoGroup'.

                    if (!insertTextDef) {
                        createdInsertTextDef = true;
                        insertTextDef = $.Deferred()
                        .always(function () {
                            insertTextDef = null;
                        });
                    }

                    if (selection.hasRange()) {

                        self.deleteSelected()
                        .done(function () {
                            // inserting the character with taking care of existing selection range
                            // -> this might lead to a deferred character insertion
                            doInsertCharacter();
                        })
                        .fail(function () {
                            if (insertTextDef) {
                                insertTextDef.resolve();
                            }
                        });
                    } else {

                        // check left text to support hyperlink auto correction
                        if (event.charCode === KeyCodes.SPACE) {
                            self.executeDelayed(function () {
                                startPosition = selection.getStartPosition();
                                hyperlinkSelection = Hyperlink.checkForHyperlinkText(self, selection.getEnclosingParagraph(), startPosition);

                                if (hyperlinkSelection !== null) {
                                    Hyperlink.insertHyperlink(self,
                                                              hyperlinkSelection.start,
                                                              hyperlinkSelection.end,
                                                              (hyperlinkSelection.url === null) ? hyperlinkSelection.text : hyperlinkSelection.url,
                                                              self.getActiveTarget());
                                    self.addPreselectedAttributes(Hyperlink.CLEAR_ATTRIBUTES);
                                }
                            }, inputTextTimeout, 'Text: handleHyperLink');
                        }

                        // inserting the character without taking care of existing selection range
                        doInsertCharacter();
                    }

                    // returning a promise for the undo group
                    return createdInsertTextDef ? insertTextDef.promise() : undefined;

                }); // enterUndoGroup
            } else if (c.length > 1) {
                // TODO?
            } else {

                if (KeyCodes.matchKeyCode(lastKeyDownEvent, 'ENTER', { shift: null })) {

                    // if there is an active 'inputTextPromise', it has to be set to null now, so that
                    // following text input is included into a new deferred
                    inputTextPromise = null;

                    // 'Enter' key is possible with and without shift-key. With shift-key a hard break is inserted.
                    if (!lastKeyDownEvent.shiftKey) {

                        // Executing the code for 'ENTER' deferred. This is necessary because of deferred input of text.
                        // Defer time is 'inputTextTimeout'
                        self.executeDelayed(function () {

                            var // whether the selection has a range (saving state, because the selection will be removed by deleteSelected())
                                hasSelection = selection.hasRange(),
                                // a deferred object required for the undo group
                                enterDef = $.Deferred();

                            // Helper function for executing the 'Enter' key event. This function is typically executed synchronously,
                            // but in special cases with a selection containing unrestorable content, it is executed deferred.
                            function doHandleEnterKey() {

                                var // the logical start position of the selection
                                    startPosition = selection.getStartPosition(),
                                    // the index of the last value in the logical start position
                                    lastValue = startPosition.length - 1,
                                    // a new logical position
                                    newPosition = _.clone(startPosition),
                                    // an object containing the start and end position of a hyperlink
                                    hyperlinkSelection = null,
                                    // a logical table position
                                    localTablePos = null,
                                    // whether the current position is the start position in a paragraph
                                    isFirstPositionInParagraph = (startPosition[lastValue] === 0),
                                    // whether the current position is the start position of the first paragraph in the document or table cell
                                    isFirstPositionOfFirstParagraph = (isFirstPositionInParagraph && (startPosition[lastValue - 1] === 0)),
                                    // the paragraph element addressed by the passed logical position
                                    paragraph = Position.getLastNodeFromPositionByNodeName(activeRootNode, startPosition, DOM.PARAGRAPH_NODE_SELECTOR),
                                    // Performance: Whether a simplified selection can be used
                                    isSimpleTextSelection = true, isSplitOperation = true,
                                    listLevel,
                                    paragraphLength,
                                    endOfParagraph,
                                    split,
                                    numAutoCorrect,
                                    paraText,
                                    labelText;

                                // Local helper function, to check if cursor is currently placed inside complex field node,
                                // so the list formatting can be ignored if true
                                function isCursorInsideCxField() {
                                    var domPos = Position.getDOMPosition(selection.getRootNode(), selection.getStartPosition()),
                                        node = null;
                                    if (domPos && domPos.node) {
                                        node = domPos.node;
                                        if (node.nodeType === 3) {
                                            node = node.parentNode;
                                        }
                                    }
                                    return DOM.isInsideComplexFieldRange(node);
                                }

                                // check for a possible hyperlink text
                                hyperlinkSelection = Hyperlink.checkForHyperlinkText(self, paragraph, startPosition);
                                // set hyperlink style and url attribute
                                if (hyperlinkSelection !== null) {
                                    Hyperlink.insertHyperlink(self, hyperlinkSelection.start, hyperlinkSelection.end, hyperlinkSelection.url, self.getActiveTarget());
                                }

                                if ((isFirstPositionInParagraph) && (isFirstPositionOfFirstParagraph) && (lastValue >= 4) &&
                                    (Position.isPositionInTable(activeRootNode, [0])) &&
                                    _(startPosition).all(function (value) { return (value === 0); })) {
                                    //at first check if a paragraph has to be inserted before the current table
                                    self.insertParagraph([0]);
                                    // Setting attributes to new paragraph immediately (task 25670)
                                    paragraphStyles.updateElementFormatting(Position.getParagraphElement(activeRootNode, [0]));
                                    newPosition = [0, 0];
                                } else if ((isFirstPositionInParagraph) && (isFirstPositionOfFirstParagraph) && (lastValue >= 4) &&
                                           (localTablePos = Position.getTableBehindTablePosition(activeRootNode, startPosition)) || // Inserting an empty paragraph between to tables
                                           (localTablePos = Position.getTableAtCellBeginning(activeRootNode, startPosition))) { // Inserting an empty paragraph at beginning of table cell
                                    // Inserting an empty paragraph between to tables
                                    self.insertParagraph(localTablePos);
                                    // Setting attributes to new paragraph immediately (task 25670)
                                    paragraphStyles.updateElementFormatting(Position.getParagraphElement(activeRootNode, localTablePos));
                                    newPosition = Position.appendNewIndex(localTablePos);
                                } else {
                                    // demote or end numbering instead of creating a new paragraph
                                    listLevel = paragraphStyles.getElementAttributes(paragraph).paragraph.listLevel;
                                    paragraphLength = Position.getParagraphNodeLength(paragraph);
                                    endOfParagraph = paragraphLength === _.last(startPosition);
                                    split = true;
                                    numAutoCorrect = {};

                                    if (!hasSelection && listLevel >= 0 && paragraphLength === 0) {
                                        listLevel--;
                                        undoManager.enterUndoGroup(function () {
                                            if (listLevel < 0) {
                                                //remove list label and update paragraph
                                                $(paragraph).children(DOM.LIST_LABEL_NODE_SELECTOR).remove();
                                                self.setAttributes('paragraph', { styleId: self.getDefaultUIParagraphStylesheet(), paragraph: { listStyleId: null, listLevel: -1 } });
                                                implParagraphChanged(paragraph);
                                            } else {
                                                self.setAttribute('paragraph', 'listLevel', listLevel);
                                            }
                                        });
                                        split = false;
                                    }

                                    if (!hasSelection && listLevel < 0 && paragraphLength > 2 && !isCursorInsideCxField()) {
                                        // detect Numbering/Bullet labels at paragraph start

                                        if (paragraph !== undefined) {
                                            paraText = paragraph.textContent;
                                            // Task 30826 -> No automatic list generation, if split happens inside the first '. ' substring
                                            if (_.last(startPosition) > ((paraText.indexOf('. ') + 1))) {
                                                if (((paraText.indexOf('. ') >= 0) && (paragraphLength > 3) && (paraText.indexOf('. ') < (paraText.length - 2))) ||
                                                    ((paraText.indexOf(' ') >= 0) && (paragraphLength > 2) && (paraText.indexOf('* ') === 0 || paraText.indexOf('- ') === 0))) {
                                                    // Fix for 29508: Not detecting '123. ' or ' xmv. ' as list
                                                    // Fix for 29732: Detecting '- ' and '* ' automatically as list
                                                    labelText = paraText.split(' ')[0];
                                                    numAutoCorrect.listDetection = listCollection.detectListSymbol(labelText);
                                                    if (numAutoCorrect.listDetection.numberFormat !== undefined) {
                                                        numAutoCorrect.startPosition = _.clone(startPosition);
                                                        numAutoCorrect.startPosition[numAutoCorrect.startPosition.length - 1] = 0;
                                                        numAutoCorrect.endPosition = selection.getEndPosition();
                                                        numAutoCorrect.endPosition[numAutoCorrect.endPosition.length - 1] = labelText.length;
                                                    }
                                                }
                                            }
                                        }
                                    }

                                    if (split === true) {
                                        newPosition[lastValue - 1] += 1;
                                        newPosition[lastValue] = 0;

                                        undoManager.enterUndoGroup(function () {

                                            var localParaPos = _.clone(startPosition),
                                                charStyles = characterStyles.getElementAttributes(paragraph.lastChild).character,
                                                attrs = {},
                                                newParagraph = null,
                                                paraAttrs = null,
                                                charAttrs = null,
                                                paraStyleId = null,
                                                isListParagraph = false;

                                            if (paragraphLength > 0 && endOfParagraph && charStyles && charStyles.url) {

                                                // Special handling: Inserting new paragraph after return at end of paragraph after hyperlink (task 30742)
                                                paraAttrs = AttributeUtils.getExplicitAttributes(paragraph);
                                                charAttrs = AttributeUtils.getExplicitAttributes(paragraph.lastChild);
                                                paraStyleId = AttributeUtils.getElementStyleId(paragraph);
                                                isListParagraph = isListStyleParagraph(null, paraAttrs);

                                                if (charAttrs && charAttrs.character && charAttrs.character.url) { delete charAttrs.character.url; }
                                                if (paraStyleId) { attrs.styleId = paraStyleId; }
                                                if (paraAttrs.paragraph && !_.isEmpty(paraAttrs.paragraph)) { attrs.paragraph = paraAttrs.paragraph; }
                                                if (charAttrs.character && !_.isEmpty(charAttrs.character)) { attrs.character = charAttrs.character; }
                                                // use only paragraph attributes in list paragraphs as character attributes, otherwise merge character attributes, task 30794
                                                if (paraAttrs.character && !_.isEmpty(paraAttrs.character)) {
                                                    if (isListParagraph) {
                                                        attrs.character = paraAttrs.character;
                                                    } else {
                                                        attrs.character = _.extend(paraAttrs.character, attrs.character || {});
                                                    }
                                                }

                                                // checking for 'nextStyleId'
                                                styleAttributes = paragraphStyles.getStyleSheetAttributeMap(paraStyleId);
                                                if (styleAttributes.paragraph && styleAttributes.paragraph.nextStyleId) { attrs.styleId = styleAttributes.paragraph.nextStyleId; }

                                                // modifying the attributes, if changeTracking is activated
                                                if (changeTrack.isActiveChangeTracking()) {
                                                    attrs.changes = { inserted: changeTrack.getChangeTrackInfo() };
                                                }

                                                self.applyOperations({ name: Operations.PARA_INSERT, start: newPosition.slice(0, -1), attrs: attrs });

                                                newParagraph = Position.getParagraphElement(activeRootNode, newPosition.slice(0, -1));
                                                implParagraphChangedSync($(newParagraph));

                                                selection.setTextSelection(newPosition, null, { simpleTextSelection: false, splitOperation: false });
                                                isSplitOperation = false;

                                                // updating lists, if required
                                                handleTriggeringListUpdate(paragraph);

                                                // in Internet Explorer it is necessary to add new empty text nodes in paragraphs again
                                                // newly, also in other browsers (new jQuery version?)
                                                repairEmptyTextNodes(newParagraph);

                                            } else {

                                                keepPreselectedAttributes = true;  // preselected attributes must also be valid in following paragraph (26459)
                                                useParagraphCache = true;
                                                paragraphCache = paragraph;  // Performance: Caching the currently used paragraph for temporary usage
                                                handleImplicitParagraph(startPosition);
                                                guiTriggeredOperation = true; // Fix for 30597
                                                self.splitParagraph(startPosition, paragraph);
                                                guiTriggeredOperation = false;
                                                paragraphCache = null;  // Performance: Deleting the currently used paragraph cache
                                                useParagraphCache = false;

                                                // Special behaviour for splitting an empty paragraph
                                                if (_.last(localParaPos) === 0) { //  || (iPad)) {
                                                    // Fix for 28568: Splitted empty paragraph needs immediately content -> calling validateParagraphNode
                                                    validateParagraphNode(paragraph);
                                                }

                                                // Avoiding cursor jumping after 'Enter' at the end of a paragraph, because new paragraph is empty.
                                                if (paragraphLength > 0 && endOfParagraph) {
                                                    implParagraphChangedSync($(paragraph.nextSibling));
                                                }

                                                // modifying the attributes, if changeTracking is activated
                                                if (changeTrack.isActiveChangeTracking()) {
                                                    self.applyOperations({
                                                        name: Operations.SET_ATTRIBUTES,
                                                        start: newPosition.slice(0, -1),
                                                        attrs: { changes: { inserted: changeTrack.getChangeTrackInfo(), removed: null }}
                                                    });
                                                } else {
                                                    if (DOM.isChangeTrackNode(paragraph)) {
                                                        self.applyOperations({
                                                            name: Operations.SET_ATTRIBUTES,
                                                            start: newPosition.slice(0, -1),
                                                            attrs: { changes: { inserted: null, removed: null, modified: null }}
                                                        });
                                                    }
                                                }

                                                // checking 'nextStyleId' at paragraphs
                                                if (endOfParagraph) {

                                                    var styleId = AttributeUtils.getElementStyleId(paragraph),
                                                        styleName = paragraphStyles.getName(styleId),
                                                        styleAttributes = paragraphStyles.getStyleSheetAttributeMap(styleId);

                                                    if (styleAttributes.paragraph && styleAttributes.paragraph.nextStyleId) {
                                                        var nextStyleId = styleAttributes.paragraph.nextStyleId;

                                                        if (nextStyleId === styleId || nextStyleId === styleName) {
                                                            //next style is same style
                                                        } else {
                                                            //listStyleId = null because header want normal text after them
                                                            self.applyOperations({
                                                                name: Operations.SET_ATTRIBUTES,
                                                                start: newPosition.slice(0, -1),
                                                                attrs: { styleId: nextStyleId, paragraph: { listStyleId: null, listLevel: -1 } }
                                                            });
                                                        }
                                                    }
                                                }
                                            }
                                        });

                                        // now apply 'AutoCorrection'
                                        if (numAutoCorrect.listDetection && numAutoCorrect.listDetection.numberFormat !== undefined) {
                                            undoManager.enterUndoGroup(function () {
                                                self.deleteRange(numAutoCorrect.startPosition, numAutoCorrect.endPosition);
                                                self.createList((numAutoCorrect.listDetection.numberFormat === 'bullet') ? 'bullet' : 'numbering', {
                                                    listStartValue: numAutoCorrect.listDetection.listStartValue,
                                                    symbol: numAutoCorrect.listDetection.symbol,
                                                    startPosition: numAutoCorrect.startPosition,
                                                    numberFormat: numAutoCorrect.listDetection.numberFormat
                                                });
                                            });
                                        }

                                    }
                                }

                                selection.setTextSelection(newPosition, null, { simpleTextSelection: isSimpleTextSelection, splitOperation: isSplitOperation });

                                keepPreselectedAttributes = false;

                            }  // function doHandleEnterKey

                            undoManager.enterUndoGroup(function () {

                                if (selection.hasRange()) {
                                    self.deleteSelected()
                                    .done(function () {
                                        doHandleEnterKey();
                                    })
                                    .always(function () {
                                        enterDef.resolve();   // closing the undo group
                                    });
                                    return enterDef.promise();  // for the undo group
                                } else {
                                    doHandleEnterKey();
                                }

                                return $.when();
                            });  // undo group

                        }, inputTextTimeout, 'Text: handleEnterKey');

                    } else if (lastKeyDownEvent.shiftKey) {
                        // insert a hard break
                        self.insertHardBreak();
                    }
                }
            } // end of else
        }

        /**
         * Handler for the keyUp event.
         *
         * @param event
         *  A jQuery keyboard event object.
         */
        function processKeyUp() {
            // to be able to distinguish tap on ipad's suggestion word, which triggers only textInput event,
            // from tap on normal letter from virtual keyboard, clear cached event value on key up event.
            lastKeyDownEvent = null;
        }

        /**
         * Handler for the 'compositionstart' event.
         * The event is dispatched before a text composition system begins a new
         * composition session and before the DOM is modified due to the composition
         * process.
         *
         * @param event {jQuery.event}
         *  The 'compositionstart' event
         */
        function processCompositionStart() {
            //Utils.log('processCompositionStart');

            var // the span to be used for special webkit treatment
                insertSpan,
                // ime state object
                imeState = {};

            if (_.browser.Android) { return; }

            if (imeStateQueue.length) {
                //fix for Bug 35543 - 3-set korean on chrome, compositionStart comes directly after compositionEnd,
                //but we have deffered the postProcessCompositionEnd, so we have to abort it and call it directly
                var last = _.last(imeStateQueue);
                var delayed = last.delayed;
                if (delayed) {
                    delayed.abort();
                    postProcessCompositionEnd();
                }
            }

            // delete current selection first
            self.deleteSelected();

            imeActive = true;
            // determin the span where the IME text will be inserted into
            imeState.imeStartPos = selection.getStartPosition();
            insertSpan = $(Position.getLastNodeFromPositionByNodeName(editdiv, imeState.imeStartPos, 'span'));

            // if the span is empty webkit browsers replace the content of the paragraph node
            // to avoid this we add a non-breaking space charater here which will be removed in processCompositionEnd()
            // this workaround is not needed for Firefox and IE and would lead to runtime errors
            if (_.browser.WebKit && (insertSpan.text().length === 0)) {
                imeState.imeAdditionalChar = true;
                insertSpan.text('\xa0');
                selection.setBrowserSelection(DOM.Range.createRange(insertSpan, 1, insertSpan, 1));
            } else {
                imeState.imeAdditionalChar = false;
            }

            // Special code for iPad. When we detect a compositionstart we have to
            // remove the last char from the activeInputText. Unfortunately we only
            // know now that the char should be processed by the composition system (IME).
            // Therefore we have to remove this char from the active input text.
            if (_.browser.iOS && _.browser.WebKit) {
                // check and abort running input text promise
                if (inputTextPromise) {
                    inputTextPromise.abort();
                }
                // remove latest character from active input text
                if (activeInputText) {
                    activeInputText = activeInputText.slice(0, -1);
                }
            }
            // now store the current state into our queue
            imeStateQueue.push(imeState);
        }

        /**
         * Handler for the 'compositionupdate' event.
         * The event is dispatched during a composition session when a text composition
         * system updates its active text passage with a new character.
         *
         * @param event {jQuery.event}
         *  The 'compositionupdate' event
         */
        function processCompositionUpdate(event) {
            if (_.browser.Android) { return; }

            //Utils.log('processCompositionUpdate');
            // Fix for 34726: Store the latest update tex for later use.
            imeUpdateText = event.originalEvent.data;
        }

        /**
         * Handler for the 'compositionend' event.
         * The event is dispatched after the text composition system completes or cancels
         * the current composition session (e.g., the IME is closed, minimized, switched
         * out of focus, or otherwise dismissed, and the focus switched back to the user
         * agent).
         * Be careful: There are browsers which violate the w3c specification (e.g. Chrome
         * and Safari) and send the compositionend event NOT when the composition system
         * completes, but at a moment where DOM manipulations are NOT allowed.
         *
         * @param event {jQuery.event}
         *  The 'compositionend' event
         */
        function processCompositionEnd(event) {
            if (_.browser.Android) { return; }

            //Utils.log('processCompositionEnd');

            var // current ime state
                imeState = _.last(imeStateQueue);

            // store the original composition end event
            imeState.event = event;
            imeState.imeUpdateText = imeUpdateText;

            // According to the w3c 'http://www.w3.org/TR/DOM-Level-3-Events/#event-type-compositionend'.
            // the 'compositionend' event is dispatched immediately after the text composition system completes the composition session.
            // webkit desktop browsers call 'compositionend' before writing back the IME data to the DOM.
            // therefore we need to defer the deletion of the IME inserted chars and call of insertText operation.
            if (_.browser.Chrome && _.device('macos')) {
                // For Chrome/MacOS we use the "input" event as "workaround". So we can
                // synchronously process the event, although later. This doesn't work
                // with Safari when using the Pinyin - Traditionell IME.
                imeCompositionEndReceived = true;
            } else if ((_.browser.Safari && _.device('macos')) ||
                       (_.browser.Chrome)) {
                // For Safari we have to use the old, bad way to delay the process
                // the composition end event. This is also necessary for Windows where
                // some IME implementations send two or more input events which are
                // not always behind the DOM manipulation
                imeState.delayed = self.executeDelayed(postProcessCompositionEnd, undefined, 'Text: postProcessCompositionEnd');
            } else {
                // all other desktop browser and the iPad can process 'compositionend' synchronously.
                postProcessCompositionEnd();
            }

            imeActive = false;
        }

        function postProcessCompositionEnd() {
            //Utils.log('postProcessCompositionEnd');

            var // pull the first state from the queue
                imeState = imeStateQueue.shift(),
                // retrieve the IME text from the compositonend event
                imeText = imeState.event.originalEvent.data,
                // determine the start position stored in the imeState
                startPosition = imeState.imeStartPos ? _.clone(imeState.imeStartPos) : selection.getStartPosition(),
                // the end position of the ime inserted text
                endPosition;

            if (_.browser.WebKit) {
                // Fix for 34726: This is a workaround for a special behaviour of
                // MacOS X. The keyboard switches to composition, if the user presses
                // the '´' key. If this key is pressed twice, the composition is started,
                // ended and started again. This collides with the asynchronous processing
                // of the composition end due to broken 'compositionend' notification of
                // webkit/blink based browers. So the asynchronous code is executed while
                // the browser started a new composition. Manipulating the DOM while in
                // composition stops the composition immediately and 'compositionend'
                // provides an emtpy string in event.originalEvnt.data. Therefore we use
                // the last 'compositionupdate' text to get the correct text.
                imeText = (imeText.length === 0) ? imeState.imeUpdateText : imeText;
            }

            if (imeText.length > 0) {
                // remove the inserted IME text from the DOM
                // and also remove the non-breaking space if it has been added in 'processCompositionStart'
                endPosition = Position.increaseLastIndex(startPosition, imeText.length - (imeState.imeAdditionalChar ? 0 : 1));
                implDeleteText(startPosition, endPosition);

                // We have to make sure that we always convert a implicit
                // paragraph. In some cases we have overlapping composition
                // events, which can produce an implicit pararaph with content
                // (inserted by the IME directly). So for this special
                // case we have to omit the length check in handleImplicitParagraph
                handleImplicitParagraph(startPosition, { ignoreLength: true });

                // insert the IME text provided by the event data
                self.insertText(imeText, startPosition, preselectedAttributes);

                // set the text seleection after the inserted IME text
                selection.setTextSelection(imeState.imeAdditionalChar ? endPosition : Position.increaseLastIndex(endPosition));

                // during the IME input we prevented calling 'restoreBrowserSelection', so we need to call it now
                selection.restoreBrowserSelection();
            }
        }

        function processTextInput(event) {
            dumpEventObject(event);

            if (Utils.IOS && !imeActive) {
                // Special code: The prevent default cancels the insertion of a suggested text.
                // Unfortunately the text is inserted for a short fraction and afterwards replaced
                // by the old text. Due to this process the cursor position is sometimes incorrectly
                // set. Previous code tried to restore the latest old cursor position, which was leading to internal errors or broken roundtrip.
                if (event.originalEvent.data.length && !lastKeyDownEvent) {
                    // detect that user taped on word suggestion, and not on actuall letter:
                    // if there is only textInput event, and no keyDown and keyUp.
                    // Workaround: instead of trying to set cursor position while element is or is not in the dom,
                    // close the ios keyboard by setting focus to a hidden node.
                    Utils.focusHiddenNode(); // #40185
                }
                event.preventDefault();
                return false;
            }
        }

        function processInput(event) {
            //Utils.log('processInput');

            dumpEventObject(event);

            // Special workaround for Chrome using the first input event after
            // we detected the "compositionend" event. This circumvent the broken
            // IME implementation, which sends a "compositionend" although not all
            // changes have been made to the DOM (breaks a MUST specification of the
            // W3C).
            if (imeCompositionEndReceived) {
                //Utils.log('call postProcessCompositionEnd');
                imeCompositionEndReceived = false;
                postProcessCompositionEnd();
            }
        }

        /**
         * Handler to process dragStart event from the browser.
         *
         * @param {Object} event
         *  The browser event sent via dragStart
         */
        function processDragStart(event) {
            var dataTransfer = event && event.originalEvent && event.originalEvent.dataTransfer,
                 dataTransferOperations = null;

            switch (selection.getSelectionType()) {

            case 'text':
            case 'drawing':
                dataTransferOperations = copyTextSelection();
                break;

            case 'cell':
                dataTransferOperations = copyCellRangeSelection();
                break;

            default:
                Utils.error('Editor.processDragStart(): unsupported selection type: ' + selection.getSelectionType());
            }

            // if browser supports DnD api add data to the event
            if (dataTransfer) {
                // add operation data
                if (!_.browser.IE) {
                    if (dataTransferOperations) {
                        dataTransfer.setData('text/ox-operations', JSON.stringify(dataTransferOperations));
                    }
                    // add plain text and html of the current browser selection
                    dataTransfer.setData('text/plain', selection.getTextFromBrowserSelection());
                    dataTransfer.setData('text/html', Export.getHTMLFromSelection(self));
                } else {
                    // IE just supports 'Text' & Url. Text is more generic
                    dataTransfer.setData('Text', selection.getTextFromBrowserSelection());
                }
            }
        }

        /**
         * Handler for 'dragover' event.
         * - draws a overlay caret on the 'dragover' event, which indicates in which position
         * the dragged content should be inserted to.
         *
         * @param {jQuery.Event} event
         *
         */
        var processDragOver = (function (event) {

            var dragOverEvent = event;

            function directCallback(event) {
                dragOverEvent = event;
            }

            function deferredCallback() {
                // inactive selection in header&footer
                if (dragOverEvent.target.classList.contains('inactive-selection') || $(dragOverEvent.target).parents('.inactive-selection').length > 0) {
                    return;
                }
                var caretOxoPosition = Position.getOxoPositionFromPixelPosition(editdiv, dragOverEvent.originalEvent.clientX, dragOverEvent.originalEvent.clientY);
                if (!caretOxoPosition || caretOxoPosition.start.length === 0) { return; }
                var dropCaret = editdiv.find('.drop-caret'),
                    zoom = app.getView().getZoomFactor(),
                    caretPoint = Position.getDOMPosition(editdiv, caretOxoPosition.start),
                    caretCoordinate = Utils.getCSSPositionFromPoint(caretPoint, editdiv, zoom),
                    caretLineHeight = caretPoint.node.parentNode.style.lineHeight;
                // adapt drop caret height to the content it is hovering, and reposition the caret
                dropCaret.css('height', caretLineHeight);
                dropCaret.css({
                    top: caretCoordinate.top + 'px',
                    left: caretCoordinate.left + 'px'
                });
                dropCaret.show();
            }

            return self.createDebouncedMethod(directCallback, deferredCallback,  { delay: 50, maxDelay: 100, infoString: 'Text: processDragOver' });

        })();

        /**
         * Handler for 'dragenter' event of the editor root node.
         * - create and appends drop caret, if user drags over editor root node
         *
         * @param {jQuery.Event} event
         *
         */
        function processDragEnter(event) {
            if (editdiv.find(event.originalEvent.target).length > 0 || DOM.isPageNode(event.originalEvent.target)) {
                var collabOverlay = editdiv.find('.collaborative-overlay'),
                    dropCaret = editdiv.find('.drop-caret');
                if (dropCaret.length === 0) {
                    dropCaret = $('<div>').addClass('drop-caret');
                    collabOverlay.append(dropCaret);
                }
            }
        }

        /**
         * Handler for 'dragleave' event of the editor root node.
         * - clears drop caret, if user drags out of the editor root node.
         *
         * @param {jQuery.Event} event
         *
         */
        function processDragLeave(event) {
            if (DOM.isPageNode(event.originalEvent.target)) {
                editdiv.find('.drop-caret').remove();
            }
        }

        /**
         * Handler to process drop events from the browser.
         *
         * @param {Object} event
         *  The drop event sent by the browser.
         */
        function processDrop(event) {

            event.preventDefault();

            if (self.getEditMode() !== true) {
                return false;
            }
            // inactive selection in header&footer
            if (event.target.classList.contains('inactive-selection') || $(event.target).parents('.inactive-selection').length > 0) {
                editdiv.find('.drop-caret').remove();
                return false;
            }

            var files = event.originalEvent.dataTransfer.files,
                dropX = event.originalEvent.clientX,
                dropY = event.originalEvent.clientY,
                dropPosition = Position.getOxoPositionFromPixelPosition(editdiv, dropX, dropY);

            // validating the drop position. This must be a text position. The parent must be a paragraph
            if (dropPosition && dropPosition.start && !Position.getParagraphElement(self.getNode(), _.initial(dropPosition.start))) {
                if (Position.getParagraphElement(self.getNode(), dropPosition.start)) {
                    dropPosition.start.push(0);
                } else {
                    return false;
                }
            }

            if (!files || files.length === 0) {

                // try to find out what type of data has been dropped
                var types = event.originalEvent.dataTransfer.types,
                    detectedDropDataType = null, div = null, url = null, text = null,
                    assuranceLevel = 99, lowerCaseType = null, operations = null;

                if (types && types.length > 0) {

                    _(types).each(function (type) {

                        lowerCaseType = type.toLowerCase();

                        if (lowerCaseType === 'text/ox-operations') {
                            operations = event.originalEvent.dataTransfer.getData(type);
                            // set the operations from the event to be used for the paste
                            clipboardOperations = (operations.length > 0) ? JSON.parse(operations) : [];
                            assuranceLevel = 0;
                            detectedDropDataType = 'operations';

                        } else if (lowerCaseType === 'text/html') {

                            var html = event.originalEvent.dataTransfer.getData(type);
                            if (html && assuranceLevel > 1) {
                                // clean up html to put only harmless html into the <div>
                                div = $('<div>').html(Utils.parseAndSanitizeHTML(html));
                                if (div.children().length > 0) {
                                    // Unfortunately we sometimes get broken html from Firefox (D&D from Chrome).
                                    // So it's better to use the plain text part.
                                    assuranceLevel = 1;
                                    detectedDropDataType = 'html';
                                }
                            }
                        } else if (lowerCaseType === 'text/uri-list' ||
                                    lowerCaseType === 'url') {

                            var list = event.originalEvent.dataTransfer.getData(type);
                            if (list && (list.length > 0) && (assuranceLevel > 3)) {
                                assuranceLevel = 3;
                                detectedDropDataType = 'link';
                                url = list;
                            }
                        } else if (lowerCaseType === 'text/x-moz-url') {
                            // FF sometimes (image D&D Chrome->FF) provides only this type
                            // instead of text/uri-list so we are forced to support it, too.
                            var temp = event.originalEvent.dataTransfer.getData(type);
                            // x-moz-url is defined as link + '\n' + caption
                            if (temp && temp.length > 0 && (assuranceLevel > 2)) {
                                var array = temp.split('\n');
                                url = array[0];
                                if (array.length > 1) {
                                    text = array[1];
                                }
                                assuranceLevel = 2;
                                detectedDropDataType = 'link';
                            }
                        } else if (lowerCaseType === 'text/plain' || lowerCaseType === 'text') {

                            var plainText = event.originalEvent.dataTransfer.getData(type);
                            if (plainText && (assuranceLevel > 4)) {
                                assuranceLevel = 4;
                                detectedDropDataType = 'text';
                                text = plainText;
                            }
                        }
                    });
                } else {
                    // IE sometimes don't provide any types but they are accessible getData()
                    // So try to check if we have a Url to check for.
                    url = event.originalEvent.dataTransfer.getData('Url');
                    if (url && url.length > 0) {
                        detectedDropDataType = 'link';
                    }
                }

                if (detectedDropDataType === 'operations') {
                    // use clipboard code to insert operations to document
                    self.pasteInternalClipboard(dropPosition);
                } else if (detectedDropDataType === 'html') {
                    if (div && div.children().length > 0) {
                        // drag&drop detected html
                        var ops = self.parseClipboard(div);
                        createOperationsFromExternalClipboard(ops, dropPosition);
                    }
                } else if (detectedDropDataType === 'link') {
                    // insert detected hyperlink
                    var setText = text || url;
                    if (setText && setText.length) {
                        if (Image.hasUrlImageExtension(url)) {
                            self.insertImageURL(url, dropPosition);
                        } else {
                            self.insertHyperlinkDirect(url, setText);
                        }
                    }
                } else if (detectedDropDataType === 'text') {
                    if (text && text.length > 0) {
                        insertPlainTextFormatted(text, dropPosition);
                    }
                } else {
                    // fallback try to use 'text' to at least get text
                    text = event.originalEvent.dataTransfer.getData('text');
                    if (text && text.length > 0) {
                        insertPlainTextFormatted(text, dropPosition);
                    }
                }
            } else {
                processDroppedImages(event, dropPosition);
            }

            // always clean caret on every drop
            self.executeDelayed(function () {
                editdiv.find('.drop-caret').remove();
                // set selection to dropped position
                if (dropPosition) {
                    selection.setTextSelection(dropPosition.start);
                }
            }, { delay: 500 }, 'Text: processDrop');

            return false;
        }

        function processDroppedImages(event, dropPosition) {

            var images = event.originalEvent.dataTransfer.files;

            // handle image date received from IO.readClientFileAsDataUrl
            function handleImageData(imgData) {
                self.insertImageURL(imgData, dropPosition);
            }

            //checks if files were dropped from the browser or the file system
            if (!images || images.length === 0) {
                self.insertImageURL(event.originalEvent.dataTransfer.getData('text'), dropPosition);
                return;
            } else {
                for (var i = 0; i < images.length; i++) {
                    var img = images[i];
                    var imgType = /image.*/;

                    //cancels insertion if the file is not an image
                    if (!img.type.match(imgType)) {
                        continue;
                    }

                    IO.readClientFileAsDataUrl(img).done(handleImageData);
                }
            }
        }

        /**
         * Uses a plain text string and interprets control characters, e.g.
         * \n, \t to have a formatted input for the editor.
         *
         * @param {String} plainText
         *  The plain text with optional control characters.
         */
        function insertPlainTextFormatted(plainText, dropPosition) {
            var lines, insertParagraph = false, result = [];

            if (_.isString(plainText) && plainText.length > 0) {

                lines = plainText.match(/[^\r\n]+/g);

                _(lines).each(function (line) {
                    if (insertParagraph) {
                        result.push({ operation: Operations.PARA_INSERT, depth: 0, listLevel: -1 });
                    }
                    result.push({ operation: Operations.TEXT_INSERT, data: line, depth: 0 });
                    insertParagraph = true;
                });

                createOperationsFromExternalClipboard(result, dropPosition);
            }
        }

        /**
         * Creates operations from the clipboard data returned by parseClipboard(...)
         * and applies them asynchronously.
         *
         * @param {Array} clipboardData
         *  The clipboard data array to create operations from.
         */
        function createOperationsFromExternalClipboard(clipboardData, dropPosition) {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // the operation start position
                start = dropPosition ? dropPosition.start : selection.getStartPosition(),
                // the operation end position
                end = _.clone(start),
                // indicates if the previous operation was insertHyperlink
                hyperLinkInserted,
                // used to cancel operation preparing in iterateArraySliced
                cancelled = false,
                // the current paragraph style sheet id
                styleId,
                // the next free list style number part
                listStyleNumber = 1,
                // the default paragraph list style sheet id
                listParaStyleId = null,
                // the list stylesheet id
                listStyleId = null,
                // indicates if previous operations inserted a list
                listInserted = false,
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // the ratio of operation generation to apply the operations
                generationRatio = 0.2,
                // current element container node
                rootNode = self.getCurrentRootNode(target);

            // the operation after insertHyperlink needs to remove the hyperlink style again
            function removeHyperLinkStyle(start, end, nextEntry) {
                if (hyperLinkInserted && (!nextEntry || nextEntry.operation !== 'insertHyperlink')) {
                    // generate the 'setAttributes' operation to remove the hyperlink style
                    generator.generateOperation(Operations.SET_ATTRIBUTES, {
                        attrs: Hyperlink.CLEAR_ATTRIBUTES,
                        start: _.clone(start),
                        end: Position.decreaseLastIndex(end)
                    });
                    hyperLinkInserted = false;
                }
            }

            // checks the given position for a text span with hyperlink attributes
            function isHyperlinkAtPosition(position) {
                var span = Position.getSelectedElement(rootNode, Position.decreaseLastIndex(position), 'span'),
                    characterAttrs = AttributeUtils.getExplicitAttributes(span, { family: 'character', direct: true });

                return !!(characterAttrs && _.isString(characterAttrs.url));
            }

            // handle implicit paragraph
            function doHandleImplicitParagraph(start) {
                var position = _.clone(start),
                    paragraph;

                if (position.pop() === 0) {  // is this an empty paragraph?
                    paragraph = Position.getParagraphElement(rootNode, position);
                    if ((paragraph) && (DOM.isImplicitParagraphNode(paragraph)) && (Position.getParagraphNodeLength(paragraph) === 0)) {
                        // creating new paragraph explicitely
                        generator.generateOperation(Operations.PARA_INSERT, {
                            start: position
                        });
                    }
                }
            }

            // the list collection is only aware of the list styles that have already been applied by an 'insertList' operation,
            // so we need to handle the next free list style id generation ourselfs.
            function getFreeListStyleId() {
                var sFreeId = 'L';

                while (listCollection.hasListStyleId('L' + listStyleNumber)) {
                    listStyleNumber++;
                }
                sFreeId += listStyleNumber;
                listStyleNumber++;
                return sFreeId;
            }

            // parse entry and generate operations
            function generateOperationCallback(entry, index, dataArray) {

                var def = null,
                    lastPos,
                    position,
                    endPosition = null,
                    attributes,
                    listOperation = null,
                    hyperlinkStyleId,
                    styleAttributes;

                // next operation's start is previous operation's end
                start = _.clone(end);
                end = _.clone(end);
                lastPos = start.length - 1;

                // cancel the clipboard paste if we loose the edit rigths or if the 'cancel' button is clicked.
                if (cancelled || !self.getEditMode()) { return Utils.BREAK; }

                switch (entry.operation) {

                case Operations.PARA_INSERT:
                    // generate the 'insertParagraph' operation
                    generator.generateOperation(Operations.PARA_SPLIT, {
                        start: _.clone(start)
                    });

                    end[lastPos - 1] += 1;
                    end[lastPos] = 0;

                    // set attributes only if it's not a list paragraph, these are handled in 'insertListElement'
                    if (entry.listLevel === -1) {
                        // use the nextStyleId if the current paragraph style defines one
                        styleAttributes = paragraphStyles.getStyleSheetAttributeMap(styleId, 'paragraph');
                        styleId = styleAttributes.paragraph && styleAttributes.paragraph.nextStyleId;

                        if (listInserted) {
                            // remove list style
                            if ((!_.isString(styleId)) || (styleId === self.getDefaultUIParagraphListStylesheet())) {
                                styleId = self.getDefaultUIParagraphStylesheet();
                            }
                            attributes = { styleId: styleId, paragraph: { listStyleId: null, listLevel: -1 } };
                            listInserted = false;

                        } else if (_.isString(styleId)) {
                            attributes = { styleId: styleId };
                        }

                        if (_.isString(styleId)) {
                            // generate operation to insert a dirty paragraph style into the document
                            generator.generateMissingStyleSheetOperation('paragraph', styleId);

                            // generate the 'setAttributes' operation
                            generator.generateOperation(Operations.SET_ATTRIBUTES, {
                                start: end.slice(0, -1),
                                attrs: attributes
                            });
                        }
                    }
                    break;

                case Operations.TEXT_INSERT:
                    // generate the 'insertText' operation (but not for empty strings)
                    if (entry.data) {
                        generator.generateOperation(Operations.TEXT_INSERT, {
                            text: entry.data,
                            start: _.clone(start),
                            attrs: { changes: { inserted: null, removed: null, modified: null }} // 41152
                        });

                        end[lastPos] += entry.data.length;
                    }

                    removeHyperLinkStyle(start, end, dataArray[index + 1]);
                    break;

                case Operations.TAB_INSERT:
                    // generate the 'insertTab' operation
                    generator.generateOperation(Operations.TAB_INSERT,
                        _.isObject(preselectedAttributes) ? { start: _.clone(start), attrs: preselectedAttributes } : { start: _.clone(start) }
                    );

                    end[lastPos] += 1;
                    removeHyperLinkStyle(start, end, dataArray[index + 1]);
                    break;

                case Operations.DRAWING_INSERT:
                    if ((!_.isString(entry.data)) || (entry.data.length < 1)) { break; }
                    // check if we got a webkit fake URL instead of a data URI,
                    // there's currently no way to access the image data of a webkit fake URL
                    // and check for local file url
                    if ((entry.data.substring(0, 15) === 'webkit-fake-url') || (entry.data.substring(0, 7) === 'file://')) {
                        app.rejectEditAttempt('image');
                        break;
                    }

                    def = $.Deferred();

                    getImageSize(entry.data).then(function (size) {

                        // exit silently if we lost the edit rights
                        if (!self.getEditMode()) {  return $.when(); }

                        // check for base64 image data or image url
                        attributes = {
                            drawing: _.extend({}, size, DEFAULT_DRAWING_MARGINS),
                            image: (entry.data.substring(0, 10) === 'data:image') ? { imageData: entry.data } : { imageUrl: entry.data }
                        };

                        // generate the 'insertDrawing' operation
                        generator.generateOperation(Operations.DRAWING_INSERT, {
                            start: _.clone(start),
                            type: 'image',
                            attrs: attributes
                        });

                        return $.when();
                    })
                    .then(function () {
                        end[lastPos] += 1;
                        removeHyperLinkStyle(start, end, dataArray[index + 1]);
                    }, function () {
                        app.rejectEditAttempt('image');
                    })
                    .always(function () {
                        // always resolve to continue processing paste operations
                        def.resolve();
                    });
                    break;

                case 'insertHyperlink':
                    if (entry.data && _.isNumber(entry.length) && (entry.length > 0) && HyperlinkUtils.hasSupportedProtocol(entry.data) && HyperlinkUtils.isValidURL(entry.data)) {
                        hyperlinkStyleId = self.getDefaultUIHyperlinkStylesheet();

                        // generate operation to insert a dirty character style into the document
                        if (_.isString(hyperlinkStyleId)) {
                            generator.generateMissingStyleSheetOperation('character', hyperlinkStyleId);
                        }

                        // the text for the hyperlink has already been inserted and the start position is right after this text,
                        // so the start for the hyperlink attribute is the start position minus the text length
                        position = _.clone(start);
                        position[lastPos] -= entry.length;
                        if (position[lastPos] < 0) {
                            position[lastPos] = 0;
                        }

                        // Task 35689, position of last character in range
                        endPosition = _.clone(end);
                        endPosition[lastPos] -= 1;

                        // generate the 'insertHyperlink' operation
                        generator.generateOperation(Operations.SET_ATTRIBUTES, {
                            attrs: { styleId: hyperlinkStyleId, character: { url: entry.data } },
                            start: position,
                            end: endPosition
                        });

                        hyperLinkInserted = true;
                    }
                    break;

                case 'insertList':
                    // for the list root level insert list style with all style levels
                    if (entry.listLevel === -1) {
                        listOperation = listCollection.getListOperationFromHtmlListTypes(entry.type);
                        listStyleId = getFreeListStyleId();

                        // generate the 'insertList' operation
                        generator.generateOperation(Operations.INSERT_LIST, {
                            listStyleId: listStyleId,
                            listDefinition: listOperation.listDefinition
                        });
                    }
                    break;

                case 'insertMSList':
                    listOperation = listCollection.getListOperationFromListDefinition(entry.data);
                    listStyleId = getFreeListStyleId();

                    // generate the 'insertList' operation
                    generator.generateOperation(Operations.INSERT_LIST, {
                        listStyleId: listStyleId,
                        listDefinition: listOperation.listDefinition
                    });
                    break;

                case 'insertListElement':
                    if (listStyleId && _.isNumber(entry.listLevel)) {
                        listParaStyleId = self.getDefaultUIParagraphListStylesheet();

                        // generate operation to insert a dirty paragraph style into the document
                        if (_.isString(listParaStyleId)) {
                            generator.generateMissingStyleSheetOperation('paragraph', listParaStyleId);
                        }

                        // generate the 'setAttributes' operation
                        generator.generateOperation(Operations.SET_ATTRIBUTES, {
                            start: start.slice(0, -1),
                            attrs: { styleId: listParaStyleId, paragraph: { listStyleId: listStyleId, listLevel: entry.listLevel } }
                        });

                        listInserted = true;
                    }
                    break;

                // needed for additional paragraph inside a list element, from the second paragraph on they have no bullet or numberring
                case 'insertListParagraph':
                    // generate the 'insertParagraph' operation
                    generator.generateOperation(Operations.PARA_SPLIT, {
                        start: _.clone(start)
                    });

                    end[lastPos - 1] += 1;
                    end[lastPos] = 0;

                    // set attributes only if it's a list paragraph
                    if (listInserted && _.isString(listParaStyleId) && entry.listLevel > -1) {

                        // generate the 'setAttributes' operation
                        generator.generateOperation(Operations.SET_ATTRIBUTES, {
                            start: end.slice(0, -1),
                            attrs: { styleId: listParaStyleId, paragraph: { listStyleId: null, listLevel: -1 } }
                        });
                    }
                    break;

                default:
                    Utils.log('createOperationsFromExternalClipboard(...) - unhandled operation: ' + entry.operation);
                    break;
                }

                return def;
            }

            // make sure that only one paste call is processed at the same time
            if (checkSetClipboardPasteInProgress()) { return $.when(); }

            // to paste at the cursor position don't create a paragraph as first operation
            if (clipboardData.length > 1 && clipboardData[0].operation === Operations.PARA_INSERT) {
                clipboardData.shift();
            } else if (clipboardData.length > 2 && clipboardData[0].operation === 'insertList' && clipboardData[1].operation === Operations.PARA_INSERT) {
                clipboardData.splice(1, 1);
            }

            // init the paragraph style sheet id
            styleId = AttributeUtils.getElementStyleId(Position.getLastNodeFromPositionByNodeName(rootNode, start, DOM.PARAGRAPH_NODE_SELECTOR));

            // make sure to remove the hyperlink style if we paste directly after a hyperlink
            hyperLinkInserted = isHyperlinkAtPosition(start);

            // init the next free list style number
            listStyleNumber = parseInt(listCollection.getFreeListId().slice(1, listCollection.getFreeListId().length), 10);
            if (!_.isNumber(listStyleNumber)) {
                listStyleNumber = 1;
            }

            return undoManager.enterUndoGroup(function () {

                var // the deferred to keep the undo group open until it is resolved or rejected
                    undoDef = $.Deferred(),
                    // the apply actions deferred
                    applyDef = null,
                    // the generate operations deferred
                    generateDef = null,
                    // the generated operations
                    operations = null,
                    // a snapshot object
                    snapshot = null,
                    // whether there is a selection range before pasting
                    hasRange = selection.hasRange(),
                    // the selection range before pasting
                    rangeStart = null, rangeEnd = null;

                // creating a snapshot
                if (hasRange) {
                    snapshot = new Snapshot(app);
                    rangeStart = _.clone(selection.getStartPosition());
                    rangeEnd = _.clone(selection.getEndPosition());
                }

                // delete current selection (if exists), has own progress bar
                self.deleteSelected({ alreadyPasteInProgress: true, snapshot: snapshot })
                .then(function () {

                    // create operation to replace an implicit pragraph with a real one
                    doHandleImplicitParagraph(start);

                    // creating a snapshot
                    if (!snapshot) { snapshot = new Snapshot(app); }

                    // show a nice message for operation creation and for applying operations with cancel button
                    app.getView().enterBusy({
                        cancelHandler: function () {
                            cancelled = true;
                            // restoring the old document state
                            snapshot.apply();
                            // calling abort function for operation promise
                            app.enterBlockOperationsMode(function () { if (applyDef && applyDef.abort) { applyDef.abort(); } });
                        },
                        warningLabel: gt('Sorry, pasting from clipboard will take some time.')
                    });

                    // generate operations
                    generateDef = self.iterateArraySliced(clipboardData, generateOperationCallback, { delay: 'immediate', infoString: 'Text: generateOperationCallback' });

                    // add progress handling
                    generateDef.progress(function (progress) {
                        // update the progress bar according to progress of the operations promise
                        app.getView().updateBusyProgress(generationRatio * progress);
                    });

                    return generateDef;
                })
                .then(function () {
                    // cancel the clipboard paste if we loose the edit rigths or if the 'cancel' button is clicked.
                    if (cancelled || !self.getEditMode()) { return $.Deferred().reject(); }
                })
                .then(function () {

                    var // the currently active target
                        target = self.getActiveTarget();

                    // cancel the clipboard paste if we loose the edit rigths or if the 'cancel' button is clicked.
                    if (cancelled || !self.getEditMode()) { return $.Deferred().reject(); }

                    operations = generator.getOperations();

                    // adding the information about the change tracking to the operations
                    if (changeTrack.isActiveChangeTracking()) {
                        changeTrack.handleChangeTrackingDuringPaste(operations);
                    } else {
                        changeTrack.removeChangeTrackInfoAfterSplitInPaste(operations);
                    }

                    // handling target positions (for example inside comments)
                    commentLayer.handlePasteOperationTarget(operations);
                    fieldManager.handlePasteOperationTarget(operations);

                    // if pasting into header or footer that are currently active, extend operations with target
                    if (target) {
                        _.each(operations, function (operation) {
                            if (operation.name !== Operations.INSERT_STYLESHEET) {
                                operation.target = target;
                            }
                        });
                    }

                    // apply generated operations
                    applyDef = self.applyOperations(operations, { async: true })
                    .progress(function (progress) {
                        // update the progress bar according to progress of the operations promise
                        app.getView().updateBusyProgress(generationRatio + (1 - generationRatio) * progress);
                    });

                    return applyDef;

                })
                .done(function () {
                    selection.setTextSelection(end);
                })
                .always(function () {
                    if (cancelled && hasRange) { selection.setTextSelection(rangeStart, rangeEnd); }
                    // no longer blocking page break calculations (40107)
                    self.setBlockOnInsertPageBreaks(false);
                    // leaving the blocked async mode
                    leaveAsyncBusy();
                    // close undo group
                    undoDef.resolve();
                    // deleting the snapshot
                    if (snapshot) { snapshot.destroy(); }
                });

                return undoDef.promise();

            }); // enterUndoGroup()
        }

        // ==================================================================
        // Private functions
        // ==================================================================

        // overwrite generic handler from EditModel: create undo for page attributes
        this.registerContextOperationHandler(Operations.SET_DOCUMENT_ATTRIBUTES, function (context) {

            var // the attribute set from the passed operation
                attributes = _.clone(context.getObj('attrs')),
                // old Attrs for comparing
                oldPageAttributes = null,
                // undo Object for local restoring
                undo = null;

            //only undo for page attributes
            if (undoManager.isLocalUndo() && attributes.page) {
                oldPageAttributes = pageStyles.getElementAttributes(self.getNode()).page;
                for (var key in oldPageAttributes) {
                    if (_.isUndefined(attributes.page[key])) {
                        delete oldPageAttributes[key];
                    }
                }
                undo = { name: Operations.SET_DOCUMENT_ATTRIBUTES, attrs: { page: oldPageAttributes } };
                undoManager.addUndo(undo, context.operation);
            }

            this.setDocumentAttributes(attributes);
        });

        this.registerOperationHandler(Operations.DELETE, function (operation) {

            var // node info about the paragraph to be deleted
                nodeInfo = null,
                // attribute type of the start node
                type = null,
                // generator for the undo/redo operations
                generator = undoManager.isUndoEnabled() ? this.createOperationsGenerator() : null,
                // if its header/footer editing, root node is header/footer node, otherwise editdiv
                // rootNode = self.getCurrentRootNode(operation.target),
                rootNode = self.getRootNode(operation.target),
                // undo operation for passed operation
                undoOperation = null;

            // undo/redo generation
            if (generator) {

                nodeInfo = Position.getDOMPosition(rootNode, operation.start, true);
                type = resolveElementType(nodeInfo.node);

                switch (type) {

                case 'text':
                    var position = operation.start.slice(0, -1),
                        paragraph = Position.getCurrentParagraph(rootNode, position),
                        start = operation.start[operation.start.length - 1],
                        end = _.isArray(operation.end) ? operation.end[operation.end.length - 1] : start;

                    undoOperation = { start: start, end: end, target: operation.target, clear: true };
                    generator.generateParagraphChildOperations(paragraph, position, undoOperation);
                    undoManager.addUndo(generator.getOperations(), operation);
                    break;

                case 'paragraph':
                    if (!DOM.isImplicitParagraphNode(nodeInfo.node)) {
                        generator.generateParagraphOperations(nodeInfo.node, operation.start, { target: operation.target });
                        undoManager.addUndo(generator.getOperations(), operation);
                    }
                    break;

                case 'cell':
                    generator.generateTableCellOperations(nodeInfo.node, operation.start, { target: operation.target });
                    undoManager.addUndo(generator.getOperations(), operation);
                    break;

                case 'row':
                    generator.generateTableRowOperations(nodeInfo.node, operation.start, { target: operation.target });
                    undoManager.addUndo(generator.getOperations(), operation);
                    break;

                case 'table':
                    if (!DOM.isExceededSizeTableNode(nodeInfo.node)) {
                        generator.generateTableOperations(nodeInfo.node, operation.start, { target: operation.target }); // generate undo operations for the entire table
                        undoManager.addUndo(generator.getOperations(), operation);
                    }
                    if (DOM.isTableInTableNode(nodeInfo.node)) {
                        if (($(nodeInfo.node).next().length > 0) && (DOM.isImplicitParagraphNode($(nodeInfo.node).next()))) {
                            $(nodeInfo.node).next().css('height', '');  // making paragraph behind the table visible after removing the table
                        }
                    }
                    break;
                }
            }

            // finally calling the implementation function to delete the content
            if (operation.target) {
                return implDelete(operation.start, operation.end, operation.target);
            } else {
                return implDelete(operation.start, operation.end);
            }
        });

        this.registerOperationHandler(Operations.MOVE, function (operation) {
            if (undoManager.isUndoEnabled()) {
                // Todo: Ignoring 'end', only 'start' === 'end' is supported
                var undoOperation = { name: Operations.MOVE, start: operation.to, end: operation.to, to: operation.start };

                extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }
            return implMove(operation.start, operation.end, operation.to, operation.target);
        });

        this.registerOperationHandler(Operations.TEXT_INSERT, function (operation, external) {

            if (!implInsertText(operation.start, operation.text, operation.attrs, operation.target, external)) {
                return false;
            }

            if (undoManager.isUndoEnabled()) {
                var end = Position.increaseLastIndex(operation.start, operation.text.length - 1),
                    undoOperation = { name: Operations.DELETE, start: operation.start, end: end };

                extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }
            return true;
        });

        this.registerOperationHandler(Operations.FIELD_INSERT, function (operation, external) {

            if (!implInsertField(operation.start, operation.type, operation.representation, operation.attrs, operation.target, external)) {
                return false;
            }

            if (undoManager.isUndoEnabled()) {
                var undoOperation = { name: Operations.DELETE, start: operation.start };

                extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }
            return true;
        });

        this.registerOperationHandler(Operations.TAB_INSERT, function (operation, external) {

            if (!implInsertTab(operation.start, operation.attrs, operation.target, external)) {
                return false;
            }

            if (undoManager.isUndoEnabled()) {
                var undoOperation = { name: Operations.DELETE, start: operation.start };
                extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }
            return true;
        });

        this.registerOperationHandler(Operations.HARDBREAK_INSERT, function (operation, external) {

            if (!implInsertHardBreak(operation.start, operation.type, operation.attrs, operation.target, external)) {
                return false;
            }

            if (undoManager.isUndoEnabled()) {
                var undoOperation = { name: Operations.DELETE, start: operation.start };
                extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }
            return true;
        });

        this.registerOperationHandler(Operations.DRAWING_INSERT, function (operation) {

            if (!implInsertDrawing(operation.type, operation.start, operation.attrs, operation.target)) {
                return false;
            }

            if (undoManager.isUndoEnabled()) {
                // not registering for undo if this is a drawing inside a drawing group (36150)
                var undoOperation = Position.isPositionInsideDrawingGroup(self.getNode(), operation.start) ? null : { name: Operations.DELETE, start: operation.start };

                if (undoOperation) {
                    extendPropertiesWithTarget(undoOperation, operation.target);
                }
                undoManager.addUndo(undoOperation, operation);
            }
            return true;
        });

        this.registerOperationHandler(Operations.INSERT_LIST, function (operation) {
            undoManager.addUndo({ name: Operations.DELETE_LIST, listStyleId: operation.listStyleId }, operation);
            listCollection.insertList(operation);
        });

        this.registerOperationHandler(Operations.DELETE_LIST, function (operation) {
            // no Undo, cannot be removed by UI
            listCollection.deleteList(operation.listStyleId);
        });

        this.registerOperationHandler(Operations.SET_ATTRIBUTES, function (operation, external) {
            // undo/redo generation is done inside implSetAttributes()
            return implSetAttributes(operation.start, operation.end, operation.attrs, operation.target, external);
        });

        this.registerOperationHandler(Operations.PARA_INSERT, function (operation) {

            var // the new paragraph
                paragraph = DOM.createParagraphNode(),
                // insert the paragraph into the DOM tree
                inserted,
                // text position at the beginning of the paragraph
                startPosition = null,
                //parents required for page breaks
                parents = paragraph.parents('table'),
                //page breaks
                currentElement = paragraph[0],
                undoOperation;

            if (parents.length > 0) {
                currentElement = parents.last();
            }

            if (operation.target) {
                inserted = insertContentNode(_.clone(operation.start), paragraph, operation.target);
            } else {
                inserted = insertContentNode(_.clone(operation.start), paragraph);
            }

            // insertContentNode() writes warning to console
            if (!inserted) { return false; }

            // insert required helper nodes
            validateParagraphNode(paragraph);

            // in Internet Explorer it is necessary to add new empty text nodes in paragraph again
            // newly, also in other browsers (new jQuery version?)
            repairEmptyTextNodes(paragraph);

            startPosition = Position.appendNewIndex(operation.start);

            // removing a following implicit paragraph (for example in a table cell)
            // -> exchanging an implicit paragraph with a non-implicit paragraph
            if (DOM.isImplicitParagraphNode(paragraph.next())) {
                paragraph.next().remove();
            }

            // generate undo/redo operations
            if (undoManager.isUndoEnabled()) {
                undoOperation = { name: Operations.DELETE, start: operation.start };
                extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }

            // apply the passed paragraph attributes
            if (_.isObject(operation.attrs)) {
                // No list style in comments in odt: 38829 (simply removing the list style id)
                if (app.isODF() && operation.target && operation.attrs && operation.attrs.paragraph && operation.attrs.paragraph.listStyleId && commentLayer.isCommentTarget(operation.target)) { delete operation.attrs.paragraph.listStyleId; }
                // saving attributes as paragraph node
                paragraphStyles.setElementAttributes(paragraph, operation.attrs);
            }

            // set cursor to beginning of the new paragraph
            // But not, if this was triggered by an insertColumn or insertRow operation (Task 30859)
            if (!guiTriggeredOperation) {
                lastOperationEnd = startPosition;
            }

            // register paragraph for deferred formatting, especially empty paragraphs
            // or update them immediately after the document import has finished (Task 28370)
            if (app.isImportFinished()) {
                implParagraphChangedSync($(paragraph));
            } else {
                implParagraphChanged(paragraph);
            }

            // updating lists, if required
            if (handleTriggeringListUpdate(paragraph, { paraInsert: true })) {
                // mark this paragraph for later list update. This is necessary, because it does not yet contain
                // the list label node (DOM.LIST_LABEL_NODE_SELECTOR) and is therefore ignored in updateLists.
                $(paragraph).data('updateList', 'true');
            }

            //render pagebreaks after insert
            insertPageBreaksDebounced(currentElement, DrawingFrame.getClosestTextFrameDrawingNode(paragraph));
            if (!pbState) { app.getView().recalculateDocumentMargin(); }

            // Performance: Saving paragraph info for following operations
            selection.setParagraphCache(paragraph, _.clone(operation.start), 0);

            // #39486
            if (operation.target) {
                pageLayout.markParagraphAsMarginal(paragraph, operation.target);
            }

            return true;
        });

        this.registerOperationHandler(Operations.PARA_SPLIT, function (operation) {
            if (undoManager.isUndoEnabled()) {
                var paragraphPos = operation.start.slice(0, -1),
                    undoOperation = { name: Operations.PARA_MERGE, start: paragraphPos };

                extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }
            return implSplitParagraph(operation.start, operation.target);
        });

        this.registerOperationHandler(Operations.PARA_MERGE, function (operation) {

            var
                // container root node of paragraph
                rootNode = self.getRootNode(operation.target),
                // the paragraph that will be merged with its next sibling
                paragraphInfo = Position.getDOMPosition(rootNode, operation.start, true),
                // current and next paragraph, as jQuery objects
                thisParagraph = null, nextParagraph = null,
                // text position at end of current paragraph, logical position of next paragraph
                paraEndPosition = null, nextParaPosition = null,
                // first child node of next paragraph
                firstChildNode = null,
                // the undo/redo operations
                generator = undoManager.isUndoEnabled() ? this.createOperationsGenerator() : null,
                // start node from where to run page break calculations
                currentElement,
                // the list label elements at the following paragraph
                listLabelInNextParagraph = null,
                // undo operation for passed operation
                undoOperation = {};

            // get current paragraph
            if (!paragraphInfo || !DOM.isParagraphNode(paragraphInfo.node)) {
                Utils.warn('Editor.mergeParagraph(): no paragraph found at position ' + JSON.stringify(operation.start));
                return false;
            }
            thisParagraph = $(paragraphInfo.node);
            currentElement = thisParagraph;
            paraEndPosition = Position.appendNewIndex(operation.start, Position.getParagraphLength(rootNode, operation.start));

            // get next paragraph
            nextParagraph = thisParagraph.next();
            // but make sure next paragraph is only valid node, and not pagebreak for example
            while (nextParagraph.length > 0 && !nextParagraph.is(DOM.CONTENT_NODE_SELECTOR)) {
                nextParagraph = nextParagraph.next();
            }

            if (!DOM.isParagraphNode(nextParagraph)) {
                Utils.warn('Editor.mergeParagraph(): no paragraph found after position ' + JSON.stringify(operation.start));
                return false;  // forcing an error, if there is no following paragraph
            }
            nextParaPosition = Position.increaseLastIndex(operation.start);

            // generate undo/redo operations
            if (generator) {
                undoOperation = { start: _.clone(paraEndPosition), target: operation.target };
                extendPropertiesWithTarget(undoOperation, operation.target);
                generator.generateOperation(Operations.PARA_SPLIT, undoOperation);

                undoOperation = { start: nextParaPosition };
                extendPropertiesWithTarget(undoOperation, operation.target);
                generator.generateSetAttributesOperation(nextParagraph, undoOperation, { clearFamily: 'paragraph' });
                undoManager.addUndo(generator.getOperations(), operation);
            }

            // remove dummy text node from current paragraph
            if (DOM.isDummyTextNode(thisParagraph[0].lastChild)) {
                $(thisParagraph[0].lastChild).remove();
            }

            // remove list label node from next paragraph (taking care of list update (35447)
            listLabelInNextParagraph = nextParagraph.children(DOM.LIST_LABEL_NODE_SELECTOR);
            if (listLabelInNextParagraph.length > 0) {
                listLabelInNextParagraph.remove();
                handleTriggeringListUpdate(nextParagraph);
            }

            // append all children of the next paragraph to the current paragraph, delete the next paragraph
            firstChildNode = nextParagraph[0].firstChild;
            thisParagraph.append(nextParagraph.children());
            nextParagraph.remove();

            // when merging par that contains MS hardbreak with type page, mark merged paragraph
            if (thisParagraph.find('.ms-hardbreak-page').length) {
                thisParagraph.addClass('manual-page-break contains-pagebreak');
            }

            // remove one of the sibling text spans at the concatenation point,
            // if one is empty; otherwise try to merge equally formatted text spans
            if (DOM.isTextSpan(firstChildNode) && DOM.isTextSpan(firstChildNode.previousSibling)) {
                if (DOM.isEmptySpan(firstChildNode)) {
                    $(firstChildNode).remove();
                } else if (DOM.isEmptySpan(firstChildNode.previousSibling)) {
                    $(firstChildNode.previousSibling).remove();
                } else {
                    Utils.mergeSiblingTextSpans(firstChildNode);
                }
            }

            // refresh DOM
            implParagraphChanged(thisParagraph);

            // checking paragraph attributes for list styles and updating lists, if required
            handleTriggeringListUpdate(thisParagraph);

            // new cursor position at merge position
            lastOperationEnd = _.clone(paraEndPosition);

            if ($(currentElement).data('lineBreaksData')) {
                $(currentElement).removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated
            }
            //render pagebreaks after merge
            insertPageBreaksDebounced(currentElement, DrawingFrame.getClosestTextFrameDrawingNode(thisParagraph));
            if (!pbState) { app.getView().recalculateDocumentMargin(); }

            return true;
        });

        this.registerOperationHandler(Operations.TABLE_SPLIT, function (operation) {
            if (undoManager.isUndoEnabled()) {
                var tablePos = operation.start.slice(0, -1),
                    undoOperation = { name: Operations.TABLE_MERGE, start: tablePos };
                extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }
            return implSplitTable(operation.start, operation.target);
        });

        this.registerOperationHandler(Operations.TABLE_MERGE, function (operation) {
            return implMergeTable(operation);
        });

        this.registerOperationHandler(Operations.TABLE_INSERT, function (operation) {

            var // the new table
                table = $('<table>').attr('role', 'grid').append($('<colgroup>')),
                // insert the table into the DOM tree
                inserted,
                // new implicit paragraph behind the table
                newParagraph = null,
                // number of rows in the table (only for operation.sizeExceeded)
                tableRows,
                // number of columns in the table (only for operation.sizeExceeded)
                tableColumns,
                // number of cells in the table (only for operation.sizeExceeded)
                tableCells,
                // element from which pagebreaks calculus is started downwards
                currentElement,
                // specifies which part of the table exceeds the size
                overflowPart,
                // the specific value which exceeds the size
                overflowValue,
                // the maximal allowed value for the part
                maxValue,
                // undo operation for handled operation
                undoOperation = {};

            if (operation.start.length > 2) {
                // if its paragraph creation inside of table
                currentElement = Position.getContentNodeElement(editdiv, operation.start.slice(0, 1));
            } else {
                currentElement = Position.getContentNodeElement(editdiv, operation.start);
            }

            if (operation.target) {
                inserted = insertContentNode(_.clone(operation.start), table, operation.target);
            } else {
                inserted = insertContentNode(_.clone(operation.start), table);
            }

            // insertContentNode() writes warning to console
            if (!inserted) { return false; }

            // generate undo/redo operations
            if (undoManager.isUndoEnabled()) {
                undoOperation = { name: Operations.DELETE, start: operation.start };
                extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }

            // Special replacement setting for a table, that cannot be displayed because of its size.
            if (_.isObject(operation.sizeExceeded)) {

                tableRows = Utils.getIntegerOption(operation.sizeExceeded, 'rows', 0);
                tableColumns = Utils.getIntegerOption(operation.sizeExceeded, 'columns', 0);
                tableCells = tableRows * tableColumns;

                if ((tableRows > 0) && (tableColumns > 0)) {

                    if (tableRows > Config.MAX_TABLE_ROWS) {
                        overflowPart = 'rows';
                        overflowValue = tableRows;
                        maxValue = Config.MAX_TABLE_ROWS;
                    } else if (tableColumns > Config.MAX_TABLE_COLUMNS) {
                        overflowPart = 'cols';
                        overflowValue = tableColumns;
                        maxValue = Config.MAX_TABLE_COLUMNS;
                    } else if (tableCells > Config.MAX_TABLE_CELLS) {
                        overflowPart = 'cols';
                        overflowValue = tableCells;
                        maxValue = Config.tableCells;
                    }
                }
                DOM.makeExceededSizeTable(table, overflowPart, overflowValue, maxValue);
                table.data('attributes', operation.attrs);  // setting attributes from operation directly to data without using 'setElementAttributes'

            // apply the passed table attributes
            } else if (_.isObject(operation.attrs) && _.isObject(operation.attrs.table)) {

                // We don't want operations to create operations themself, so we can't
                // call 'checkForLateralTableStyle' here.

                if (self.getBlockKeyboardEvent()  && _.browser.Firefox) { table.data('internalClipboard', true); }  // Fix for task 29401
                tableStyles.setElementAttributes(table, operation.attrs);
            }

            if (operation.target) {
                // trigger header/footer content update on other elements of same type, if change was made inside header/footer
                if (DOM.isMarginalNode(table) || DOM.getMarginalTargetNode(self.getNode(), table).length) {
                    updateEditingHeaderFooterDebounced(DOM.getMarginalTargetNode(self.getNode(), table));
                }
                self.setBlockOnInsertPageBreaks(true);
            }

            // call pagebreaks re-render after inserting table
            insertPageBreaksDebounced(currentElement, DrawingFrame.getClosestTextFrameDrawingNode(table));
            if (!pbState) { app.getView().recalculateDocumentMargin(); }

            // adding an implicit paragraph behind the table (table must not be the last node in the document)
            if (table.next().length === 0) {
                newParagraph = DOM.createImplicitParagraphNode();
                validateParagraphNode(newParagraph);
                // insert the new paragraph behind the existing table node
                table.after(newParagraph);
                implParagraphChanged(newParagraph);
                // in Internet Explorer it is necessary to add new empty text nodes in paragraph again
                // newly, also in other browsers (new jQuery version?)
                repairEmptyTextNodes(newParagraph);
            } else if (DOM.isImplicitParagraphNode(table.next())) {
                if (DOM.isTableInTableNode(table)) {
                    table.next().css('height', 0);  // hiding the paragraph behind the table inside another table
                }
            }

            if (undoRedoRunning  && _.browser.Firefox) { table.data('undoRedoRunning', true); }  // Fix for task 30477

            return true;
        });

        this.registerOperationHandler(Operations.COLUMNS_DELETE, function (operation) {

            var rootNode = self.getRootNode(operation.target),
                table = Position.getTableElement(rootNode, operation.start);

            if (table) {
                if (undoManager.isUndoEnabled()) {

                    var allRows = DOM.getTableRows(table),
                        allCellRemovePositions = Table.getAllRemovePositions(allRows, operation.startGrid, operation.endGrid),
                        generator = this.createOperationsGenerator();

                    allRows.each(function (index) {

                        var rowPos = operation.start.concat([index]),
                            cells = $(this).children(),
                            oneRowCellArray =  allCellRemovePositions[index],
                            end = oneRowCellArray.pop(),
                            start = oneRowCellArray.pop();  // more than one cell might be deleted in a row

                        // start<0: no cell will be removed in this row
                        if (start >= 0) {

                            if (end < 0) {
                                // remove all cells until end of row
                                end = cells.length;
                            } else {
                                // closed range to half-open range
                                end = Math.min(end + 1, cells.length);
                            }

                            // generate operations for all covered cells
                            cells.slice(start, end).each(function (index) {
                                generator.generateTableCellOperations(this, rowPos.concat([start + index]), { target: operation.target });
                            });
                        }
                    });
                    undoManager.addUndo(generator.getOperations(), operation);
                }
                implDeleteColumns(operation.start, operation.startGrid, operation.endGrid, operation.target);
            }

            return true;
        });

        this.registerOperationHandler(Operations.CELL_MERGE, function (operation) {
            if (undoManager.isUndoEnabled()) {
                var content = null,
                    gridSpan = null,
                    undoOperation = { name: Operations.CELL_SPLIT, start: operation.start, content: content, gridSpan: gridSpan };

                extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }
            return implMergeCell(_.copy(operation.start, true), operation.count, operation.target);
        });

        this.registerOperationHandler(Operations.CELLS_INSERT, function (operation) {

            if (undoManager.isUndoEnabled()) {
                var count = Utils.getIntegerOption(operation, 'count', 1, 1),
                    undoOperations = [],
                    undoOperation = null;

                // TODO: create a single DELETE operation for the cell range, once this is supported
                _(count).times(function () {
                    undoOperation = { name: Operations.DELETE, start: operation.start };
                    extendPropertiesWithTarget(undoOperation, operation.target);
                    undoOperations.push(undoOperation);
                });
                undoManager.addUndo(undoOperations, operation);
            }

            return implInsertCells(operation.start, operation.count, operation.attrs, operation.target);
        });

        this.registerOperationHandler(Operations.ROWS_INSERT, function (operation) {

            if (undoManager.isUndoEnabled()) {
                var count = Utils.getIntegerOption(operation, 'count', 1, 1),
                    undoOperations = [],
                    undoOperation = null;

                // TODO: create a single DELETE operation for the row range, once this is supported
                _(count).times(function () {
                    undoOperation = { name: Operations.DELETE, start: operation.start };
                    extendPropertiesWithTarget(undoOperation, operation.target);
                    undoOperations.push(undoOperation);
                });

                undoManager.addUndo(undoOperations, operation);
            }

            return implInsertRows(operation.start, operation.count, operation.insertDefaultCells, operation.referenceRow, operation.attrs, operation.target);

        });

        this.registerOperationHandler(Operations.COLUMN_INSERT, function (operation) {
            if (undoManager.isUndoEnabled()) {
                undoManager.enterUndoGroup(function () {

                    // COLUMNS_DELETE cannot be the answer to COLUMN_INSERT, because the cells of the new column may be inserted
                    // at very different grid positions. It is only possible to remove the new cells with deleteCells operation.
                    var localPos = _.clone(operation.start),
                        rootNode = self.getRootNode(operation.target),
                        table = Position.getDOMPosition(rootNode, localPos).node,  // -> this is already the new grid with the new column!
                        allRows = DOM.getTableRows(table),
                        allCellInsertPositions = Table.getAllInsertPositions(allRows, operation.gridPosition, operation.insertMode),
                        cellPosition = null,
                        undoOperation = {};

                    for (var i = (allCellInsertPositions.length - 1); i >= 0; i--) {
                        cellPosition = Position.appendNewIndex(localPos, i);
                        cellPosition.push(allCellInsertPositions[i]);
                        undoOperation = { name: Operations.DELETE, start: cellPosition };
                        extendPropertiesWithTarget(undoOperation, operation.target);
                        undoManager.addUndo(undoOperation);
                    }

                    undoManager.addUndo(null, operation);  // only one redo operation

                }, this); // enterUndoGroup()
            }
            return implInsertColumn(operation.start, operation.gridPosition, operation.insertMode, operation.target);
        });

        this.registerOperationHandler(Operations.INSERT_HEADER_FOOTER, function (operation) {
            if (undoManager.isUndoEnabled()) {
                var undoOperation = { name: Operations.DELETE_HEADER_FOOTER, id: operation.id };

                undoManager.addUndo(undoOperation, operation);
            }
            return implInsertHeaderFooter(operation.id, operation.type, operation.attrs);
        });

        this.registerOperationHandler(Operations.COMMENT_INSERT, function (operation) {

            var // the undo operation
                undoOperation = null;

            if (!implInsertComment(operation.start, operation.id, operation.author, operation.uid, operation.date, operation.target)) {
                return false;
            }

            if (undoManager.isUndoEnabled()) {
                undoOperation = { name: Operations.DELETE, start: operation.start };
                undoManager.addUndo(undoOperation, operation);
            }
            return true;
        });

        this.registerOperationHandler(Operations.RANGE_INSERT, function (operation) {

            var // the undo operation
                undoOperation = null;

            if (!implInsertRange(operation.start, operation.id, operation.type, operation.position, operation.target)) {
                return false;
            }

            if (undoManager.isUndoEnabled()) {
                undoOperation = { name: Operations.DELETE, start: operation.start };
                extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }
            return true;
        });

        this.registerOperationHandler(Operations.COMPLEXFIELD_INSERT, function (operation) {

            var // the undo operation
                undoOperation = null;

            if (!implInsertComplexField(operation.start, operation.id, operation.instruction, operation.target)) {
                return false;
            }

            if (undoManager.isUndoEnabled()) {
                undoOperation = { name: Operations.DELETE, start: operation.start };
                extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }
            return true;
        });

        this.registerOperationHandler(Operations.DELETE_HEADER_FOOTER, function (operation) {

            var generator = undoManager.isUndoEnabled() ? this.createOperationsGenerator() : null,
                containerNode = self.getRootNode(operation.id),
                // must be empty, so that generateContentOperations makes operations for all content inside node
                position = [],
                options = null,
                type = null,
                undoOperation = null;

            // undo/redo generation
            if (undoManager.isUndoEnabled()) {
                if (generator) {
                    options = { target: operation.id };

                    containerNode.children('.cover-overlay').remove();
                    // check if there are special fields for restoring before generating op
                    fieldManager.checkRestoringSpecialFieldsInContainer(containerNode);

                    generator.generateContentOperations(containerNode, position, options);
                    undoManager.addUndo(generator.getOperations());
                }

                type = pageLayout.getTypeFromId(operation.id);
                if (type) {
                    undoOperation = { name: Operations.INSERT_HEADER_FOOTER, id: operation.id, type: type };
                    undoManager.addUndo(undoOperation, operation);
                } else {
                    return;
                }

            }

            return implDeleteHeaderFooter(operation.id);
        });

        /**
         * call setAttributes with assigned listStyleId and listLevel
         * sets styleId to list-paragraph only if current styleId is standard
         */
        function setListStyle(listStyleId, listLevel) {

            var attrs = self.getAttributes('paragraph'),
                newAttrs = { paragraph: { listStyleId: listStyleId, listLevel: listLevel } };

            //TODO: workaround for broken paint update
            if (_.isUndefined(listLevel) && attrs.paragraph) {
                newAttrs.paragraph.listLevel = attrs.paragraph.listLevel;
            }

            // assigning list style id to paragraph, if it has the default style id (40787)
            if (attrs.styleId === self.getDefaultUIParagraphStylesheet()) {
                // checking, if the paragraph has a style id set
                newAttrs.styleId = self.getDefaultUIParagraphListStylesheet();
            }

            self.setAttributes('paragraph', newAttrs);
        }

        /**
         * Removes empty text nodes from the passed paragraph, checks whether
         * it needs a dummy text node, and converts consecutive white-space
         * characters.
         *
         * @param {HTMLElement|jQuery} paragraph
         *  The paragraph element to be validated. If this object is a jQuery
         *  collection, uses the first DOM node it contains.
         */
        function validateParagraphNode(paragraph) {

            var // current sequence of sibling text nodes
                siblingTextNodes = [],
                // array of arrays collecting all sequences of sibling text nodes
                allSiblingTextNodes = [siblingTextNodes],
                // whether the paragraph contains any text
                hasText = false,
                // whether the last child node is the dummy element
                hasLastDummy = false,
                // whether we have to add or preserve a dummy element for hard-breaks
                mustAddOrPreserveDummyForHardBreak = false;

            validateParagraphNode.DBG_COUNT = (validateParagraphNode.DBG_COUNT || 0) + 1;

            // convert parameter to a DOM node
            paragraph = Utils.getDomNode(paragraph);

            // whether last node is the dummy node
            hasLastDummy = DOM.isDummyTextNode(paragraph.lastChild);

            // remove all empty text spans which have sibling text spans, and collect
            // sequences of sibling text spans (needed for white-space handling)
            Position.iterateParagraphChildNodes(paragraph, function (node) {

                // visit all text spans embedded in text container nodes (fields, tabs, ... (NOT numbering labels))
                if (DOM.isTextSpan(node) || DOM.isTextComponentNode(node)) {
                    DOM.iterateTextSpans(node, function (span) {
                        if (DOM.isEmptySpan(span)) {
                            // remove this span, if it is an empty portion and has a sibling text portion (should not happen anymore)
                            if (DOM.isTextSpan(span.previousSibling) || DOM.isTextSpan(span.nextSibling)) {
                                Utils.warn('Editor.validateParagraphNode(): empty text span with sibling text span found');
                                $(span).remove();
                            }
                            // otherwise simply ignore the empty span
                        } else {
                            // append text node to current sequence
                            siblingTextNodes.push(span.firstChild);
                        }
                    });

                // anything else (no text span or text container node): start a new sequence of text nodes
                } else {
                    allSiblingTextNodes.push(siblingTextNodes = []);
                }

            }, undefined, { allNodes: true });

            // Special case for hyperlink formatting and empty paragraph (just one empty textspan)
            // Here we always don't want to have the hyperlink formatting, we hardly reset these attributes
            if (DOM.isTextSpan(paragraph.firstElementChild) && (paragraph.children.length === 1) &&
                (paragraph.firstElementChild.textContent.length === 0)) {
                var url = characterStyles.getElementAttributes(paragraph.firstElementChild).character.url;
                if ((url !== null) && (url.length > 0)) {
                    characterStyles.setElementAttributes(paragraph.firstElementChild, Hyperlink.CLEAR_ATTRIBUTES);
                }
            } else if (DOM.isListLabelNode(paragraph.firstElementChild) && (paragraph.children.length === 2) &&
                       (paragraph.firstElementChild.nextSibling.textContent.length === 0)) {
                var url = characterStyles.getElementAttributes(paragraph.firstElementChild.nextSibling).character.url;
                if ((url !== null) && (url.length > 0)) {
                    characterStyles.setElementAttributes(paragraph.firstElementChild.nextSibling, Hyperlink.CLEAR_ATTRIBUTES);
                }
            }

            // Special case for hard-breaks. Check for a hard-break as a node before the last child which
            // is not a dummy node. We need to have a dummy node for a hard-break at the end of the
            // paragraph to support correct formatting & cursor travelling.
            if (paragraph.children.length >= 2) {
                var prevNode = paragraph.lastChild.previousSibling;

                if (paragraph.children.length >= 3 && hasLastDummy) {
                    // Second case: We have a dummy node and must preserve it.
                    // prevNode is in that case the previous node of prev node.
                    // (hard-break) - (span) - (br)
                    prevNode = prevNode.previousSibling;
                }

                if (prevNode && DOM.isHardBreakNode(prevNode)) {
                    mustAddOrPreserveDummyForHardBreak = true;
                }
            }

            // Convert consecutive white-space characters to sequences of SPACE/NBSP
            // pairs. We cannot use the CSS attribute white-space:pre-wrap, because
            // it breaks the paragraph's CSS attribute text-align:justify. Process
            // each sequence of sibling text nodes for its own (the text node
            // sequences may be interrupted by other elements such as hard line
            // breaks, drawings, or other objects).
            // TODO: handle explicit NBSP inserted by the user (when supported)
            _(allSiblingTextNodes).each(function (siblingTextNodes) {

                var // the complete text of all sibling text nodes
                    text = '',
                    // offset for final text distribution
                    offset = 0;

                // collect the complete text in all text nodes
                _(siblingTextNodes).each(function (textNode) { text += textNode.nodeValue; });

                // ignore empty sequences
                if (text.length > 0) {
                    hasText = true;

                    // process all white-space contained in the text nodes
                    text = text
                        // normalize white-space (convert to SPACE characters)
                        .replace(/\s/g, ' ')
                        // text in the node sequence cannot start with a SPACE character
                        .replace(/^ /, '\xa0')
                        // convert SPACE/SPACE pairs to SPACE/NBSP pairs
                        .replace(/ {2}/g, ' \xa0')
                        // text in the node sequence cannot end with a SPACE character
                        .replace(/ $/, '\xa0');

                    // distribute converted text to the text nodes
                    _(siblingTextNodes).each(function (textNode) {
                        var length = textNode.nodeValue.length;
                        textNode.nodeValue = text.substr(offset, length);
                        offset += length;
                    });
                }
            });

            // insert an empty text span if there is no other content (except the dummy node)
            if (!paragraph.hasChildNodes() || (hasLastDummy && (paragraph.childNodes.length === 1))) {
                $(paragraph).prepend(DOM.createTextSpan());
            }

            // append dummy text node if the paragraph contains no text,
            // or remove it if there is any text
            if (!hasText && !hasLastDummy) {
                $(paragraph).append(DOM.createDummyTextNode());
            } else if (!hasLastDummy && mustAddOrPreserveDummyForHardBreak) {
                // we need to add a dummy text node after a last hard-break
                $(paragraph).append(DOM.createDummyTextNode());
            } else if (hasText && hasLastDummy && !mustAddOrPreserveDummyForHardBreak) {
                $(paragraph.lastChild).remove();
            }
        }

        // ==================================================================
        // Private image methods
        // ==================================================================

        /**
         * Draws the border(s) of drawings. Needed for localStorage case to draw lines into the canvas
         * or for automatically resized text frames or text frames within groups.
         *
         * @param {jQuery|Node[]} drawingNodes
         *  Collector (jQuery or array) for all drawing nodes, whose border canvas need to be
         *  repainted.
         */
        function drawCanvasBorders(drawingNodes) {

            _.each(drawingNodes, function (node) {

                var // one drawing node, must be 'jQuerified'
                    drawing = DrawingFrame.isDrawingFrame(node) ? $(node) : DrawingFrame.getDrawingNode(node),
                    // the attributes of the drawing (also using styles (38058))
                    drawingAttrs = drawingStyles.getElementAttributes(drawing);

                if (_.has(drawingAttrs, 'line') && drawingAttrs.line.type !== 'none') {
                    DrawingFrame.drawBorder(app, drawing, drawingAttrs.line);
                }
            });
        }

        /**
         * Calculates the size of the image defined by the given url,
         * limiting the size to the paragraph size of the current selection.
         *
         * @param {String} url
         *  The image url or base64 data url
         *
         * @returns {jQuery.Promise}
         *  A promise where the resolve result contains the size object
         *  including width and height in Hmm {width: 123, height: 456}
         */
        function getImageSize(url) {

            var // the result deferred
                def = $.Deferred(),
                // the image for size rendering
                image = $('<img>'),
                // the image url
                absUrl,
                // the clipboard holding the image
                clipboard = null;

            if (!url) {
                return def.reject();
            }

            absUrl = Image.getFileUrl(app, url);

            // append the clipboard div to the body and place the image into it
            clipboard = app.getView().createClipboardNode();
            clipboard.append(image);

            // if you set the load handler BEFORE you set the .src property on a new image, you will reliably get the load event.
            image.one('load', function () {

                var width, height, para, maxWidth, maxHeight, factor, start,
                    pageAttributes = null,
                    // a text frame node containing the image
                    textFrameDrawing = null;

                // Workaround for a strange Chrome behavior, even if we use .one() Chrome fires the 'load' event twice.
                // One time for the image node rendered and the other time for a not rendered image node.
                // We check for the rendered image node
                if (Utils.containsNode(clipboard, image)) {
                    width = Utils.convertLengthToHmm(image.width(), 'px');
                    height = Utils.convertLengthToHmm(image.height(), 'px');

                    // maybe the paragraph is not so big
                    start = selection.getStartPosition();
                    start.pop();
                    para = Position.getParagraphElement(editdiv, start);
                    if (para) {
                        maxWidth = Utils.convertLengthToHmm($(para).outerWidth(), 'px');
                        pageAttributes = pageStyles.getElementAttributes(self.getNode());
                        // reading page attributes, they are always available -> no need to check existence
                        maxHeight = pageAttributes.page.height - pageAttributes.page.marginTop - pageAttributes.page.marginBottom;

                        // adapting width and height also to text frames with fixed height (36329)
                        if (DOM.isNodeInsideTextFrame(para)) {
                            textFrameDrawing = DrawingFrame.getClosestTextFrameDrawingNode(para);
                            if (DrawingFrame.isFixedHeightDrawingFrame(textFrameDrawing)) {
                                maxHeight = Utils.convertLengthToHmm(DrawingFrame.getTextFrameNode(textFrameDrawing).height(), 'px');
                            }
                        }

                        if (width > maxWidth) {
                            factor = Utils.round(maxWidth / width, 0.01);
                            width = maxWidth;
                            height = Math.round(height * factor);
                        }

                        if ((maxHeight) && (height > maxHeight)) {
                            factor = Utils.round(maxHeight / height, 0.01);
                            height = maxHeight;
                            width = Math.round(width * factor);
                        }
                    }

                    def.resolve({ width: width, height: height });
                }
            })
            .error(function () {
                Utils.warn('Editor.getImageSize(): image load error');
                def.reject();
            })
            .attr('src', absUrl);

            // always remove the clipboard again
            def.always(function () {
                clipboard.remove();
            });

            return def.promise();
        }

        // ====================================================================
        //  IMPLEMENTATION FUNCTIONS
        // Private methods, that are called from method applyOperations().
        // The operations itself are never generated inside an impl*-function.
        // ====================================================================

        /**
         * Has to be called after changing the structure of a paragraph node, if
         * the update has to be done immediately (synchronously).
         *
         * @param {jQuery} paragraphs
         *  All paragraph nodes that need to be updated included into a
         *  jQuery object
         */
        function implParagraphChangedSync(paragraphs) {

            paragraphs.each(function () {

                var // the paragraph node
                    paragraph = this;

                // the paragraph may have been removed from the DOM in the meantime
                if (Utils.containsNode(self.getNode(), paragraph)) {
                    spellChecker.resetDirectly(paragraph); // reset to 'not spelled'
                    validateParagraphNode(paragraph);
                    paragraphStyles.updateElementFormatting(paragraph)
                    .done(function () { self.trigger('paragraphUpdate:after', paragraph); });
                }
            });
            // paragraph validation changes the DOM, restore selection
            // -> Not restoring browser selection, if the edit rights are not available (Task 29049)
            if ((!imeActive) && (self.getEditMode())) {
                selection.restoreBrowserSelection({ operationTriggered: true });
            }
        }

        /**
         * Has to be called every time after changing the structure of a
         * paragraph node.
         *
         * @param {HTMLElement|jQuery|Number[]} paragraph
         *  The paragraph element as DOM node or jQuery object, or the logical
         *  position of the paragraph or any of its child components.
         */
        var implParagraphChanged = $.noop;
        app.onImportSuccess(function () {

            var // all paragraph nodes that need to be updated
                paragraphs = $();

            // direct callback: called every time when implParagraphChanged() has been called
            function registerParagraph(paragraph) {
                if (_.isArray(paragraph)) {
                    paragraph = Position.getCurrentParagraph(self.getCurrentRootNode(), paragraph);
                }
                // store the new paragraph in the collection (jQuery keeps the collection unique)
                if (paragraph) {
                    // reset to 'not spelled'
                    spellChecker.resetDirectly(paragraph);
                    paragraphs = paragraphs.add(paragraph);
                }
            }

            // deferred callback: called once, after current script ends
            function updateParagraphs() {
                implParagraphChangedSync(paragraphs);
                paragraphs = $();
            }

            // create and return the deferred implParagraphChanged() method
            implParagraphChanged = self.createDebouncedMethod(registerParagraph, updateParagraphs, { delay: 200, maxDelay: 1000, infoString: 'Text: implParagraphChanged' });
        });

        /**
         * Has to be called every time after changing the cell structure of a
         * table. It recalculates the position of each cell in the table and
         * sets the corresponding attributes. This can be set for the first or
         * last column or row, or even only for the south east cell.
         *
         * @param {HTMLTableElement|jQuery} table
         *  The table element whose structure has been changed. If this object
         *  is a jQuery collection, uses the first node it contains.
         */
        var implTableChanged = $.noop;
        app.onImportSuccess(function () {

            var // all table nodes that need to be updated
                tables = $();

            // direct callback: called every time when implTableChanged() has been called
            function registerTable(table) {
                // store the new table in the collection (jQuery keeps the collection unique)
                tables = tables.add(table);
            }

            // deferred callback: called once, after current script ends
            function updateTables() {
                tables.each(function () {
                    var // the table node
                        table = this;
                    // the table may have been removed from the DOM in the meantime
                    // -> just checking, if the table is still somewhere below the page
                    if (Utils.containsNode(self.getNode(), table)) {
                        tableStyles.updateElementFormatting(table)
                        .done(function () { self.trigger('tableUpdate:after', table); });
                    }
                });
                tables = $();
            }

            // create and return the deferred implTableChanged() method
            implTableChanged = self.createDebouncedMethod(registerTable, updateTables, { infoString: 'Text: implParagraphChanged' });
        });

        /**
         * After paragraph modifications it might be necessary to update lists.
         * If this is necessary and which performance properties are required,
         * is checked within this function.
         *
         * @param {Node|jQuery|Null} [node]
         *  The DOM paragraph node to be checked. If this object is a jQuery
         *   collection, uses the first DOM node it contains. If missing or null,
         *   no list update is triggered and false is returned.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.checkSplitInNumberedList=false]
         *      If set to true, an additional check is executed, to make special
         *      performance improvements for numbered lists (32583). In this case
         *      a special marker is set at the following paragraph, if this function
         *      returns 'true'. In this case only all following paragraphs need to
         *      be updated.
         *  @param {Boolean} [options.paraInsert=false]
         *      If set to true, the 'updateList' data attribute is set at the (new)
         *      paragraph. This is necessary, if the paragraph does not already contain
         *      the list label node (DOM.LIST_LABEL_NODE_SELECTOR) and is therefore
         *      ignored in updateLists. Additionally 'updateListsDebounced' is 'informed
         *      about this setting. All this is important for performance reasons.
         *
         * @returns {Boolean}
         *  Whether the calling function can executed some specific code. This is
         *  only used for special options.
         */
        function handleTriggeringListUpdate(node, options) {

            var // paragraph attributes object
                paragraphAttrs = null,
                // the paragraph list style id
                listStyleId = null,
                // the paragraph list level
                listLevel = null,
                // whether a marker for a node is required
                runSpecificCode = false,
                // whether a split happened inside a numbered list (performance)
                splitInNumberedList = false,
                // whether the 'paraInsert' value needs to be set for updateListsDebounced
                paraInsert = Utils.getBooleanOption(options, 'paraInsert', false),
                // whether the property 'splitInNumberedList' needs to be checked
                checkSplitInNumberedList = Utils.getBooleanOption(options, 'checkSplitInNumberedList', false);

            // nothing to do, if the node is not a paragraph
            if (!DOM.isParagraphNode(node)) { return false; }

            // receiving the paragraph attributes
            // fix for Bug 37594: changed from explicit attrs to merged attrs,
            // because some para stylesheets have list styles inside
            paragraphAttrs = paragraphStyles.getElementAttributes(node);

            if (isListStyleParagraph(null, paragraphAttrs)) {
                if (paragraphAttrs && paragraphAttrs.paragraph) {
                    listStyleId = paragraphAttrs.paragraph.listStyleId;
                    listLevel = paragraphAttrs.paragraph.listLevel;

                    if (checkSplitInNumberedList) {
                        // marking paragraph for performance reasons, if this is a numbered list
                        if (listCollection.isNumberingList(listStyleId, listLevel)) {
                            runSpecificCode = true;
                            splitInNumberedList = true;
                        }
                    }

                    if (paraInsert) { runSpecificCode = true; }
                }

                // triggering list update debounced
                updateListsDebounced({ useSelectedListStyleIDs: true, listStyleId: listStyleId, listLevel: listLevel, paraInsert: paraInsert, splitInNumberedList: splitInNumberedList });
            }

            return runSpecificCode;
        }

        /**
         * After changing the page settings it is necessary to update specific elements. This are especially drawings
         * at paragraphs and tab stops. Drawings that are anchored to the page are updated by the event 'update:absoluteElements'
         * that comes later. Tables do not need to be updated, if their width is correctly set in percentage or 'auto' If they
         * have a fixed width, this must not be changed.
         */
        function updatePageDocumentFormatting() {

            var // updating tables and drawings at paragraphs (in page and header / footer)
                pageContentNode = DOM.getPageContentNode(editdiv),
                // all paragraph and all drawings
                formattingNodes = pageContentNode.find(DrawingFrame.NODE_SELECTOR + ', ' + DOM.PARAGRAPH_NODE_SELECTOR),
                // header and footer container nodes needs to be updated also, if there is content inside them
                headerFooterFormattingNodes = pageLayout.getHeaderFooterPlaceHolder().find(DrawingFrame.NODE_SELECTOR + ', ' + DOM.PARAGRAPH_NODE_SELECTOR);

            // update of tables is not required: width 'auto' (that is '100%') and percentage values works automatically

            // add content of comment nodes
            formattingNodes = formattingNodes.add(headerFooterFormattingNodes);

            // updating all elements
            _.each(formattingNodes, function (element) {

                if (DrawingFrame.isDrawingFrame(element) && DOM.isFloatingNode(element)) {
                    // Only handle drawings that are anchored to paragraphs
                    // -> absolute positioned drawings are updated later via 'update:absoluteElements'
                    drawingStyles.updateElementFormatting(element);
                    // also updating all paragraphs inside text frames (not searching twice for the drawings)
                    implParagraphChanged($(element).find(DOM.PARAGRAPH_NODE_SELECTOR));

                } else if (DOM.isParagraphNode(element)) {

                    // update the size of all tab stops in this paragraph (but only if the paragraph contains tabs (Performance))
                    if ($(element).find(DOM.TAB_NODE_SELECTOR).length > 0) {
                        paragraphStyles.updateTabStops(element);
                    }

                }
            });
        }

        /**
         * Prepares a 'real' paragraph after insertion of text, tab, drawing, ...
         * by exchanging an 'implicit' paragraph (in empty documents, empty cells,
         * behind tables, ...) with the help of an operation. Therefore the server is
         * always informed about creation and removal of paragraphs and implicit
         * paragraphs are only required for user input in the browser.
         *
         * @param {Number[]} position
         *  The logical text position.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.ignoreLength=false]
         *      If set to true, no check for the paragraph length is done. Normally
         *      a implicit paragraph must be empty. In special cases this check
         *      must be omitted.
         */
        function handleImplicitParagraph(position, options) {

            var // the searched implicit paragraph
                paragraph = null,
                // the new created paragraph
                newParagraph = null,
                // ignore length option
                ignoreLength = Utils.getBooleanOption(options, 'ignoreLength', false),
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // current element container node
                rootNode = self.getCurrentRootNode(target),
                // created operation
                operation = null;

            position = _.clone(position);
            if (position.pop() === 0) {  // is this an empty paragraph?
                paragraph = (useParagraphCache && paragraphCache) || Position.getParagraphElement(rootNode, position);
                if ((paragraph) && (DOM.isImplicitParagraphNode(paragraph)) && ((Position.getParagraphNodeLength(paragraph) === 0) || ignoreLength)) {
                    // removing implicit paragraph node
                    $(paragraph).remove();
                    // creating new paragraph explicitely
                    operation = { name: Operations.PARA_INSERT, start: position };
                    extendPropertiesWithTarget(operation, target);
                    self.applyOperations(operation);
                    // Setting attributes to new paragraph immediately (task 25670)
                    newParagraph = Position.getParagraphElement(rootNode, position);
                    paragraphStyles.updateElementFormatting(newParagraph);
                    // using the new paragraph as global cache
                    if (useParagraphCache && paragraphCache) { paragraphCache = newParagraph; }
                    // if implicit paragraph was marked as marginal, mark newly created also
                    if (DOM.isMarginalNode(paragraph)) {
                        $(newParagraph).addClass(DOM.MARGINAL_NODE_CLASSNAME);
                    }
                }
            }
        }

        /**
         * Prepares the text span at the specified logical position for
         * insertion of a new text component or character. Splits the text span
         * at the position, if splitting is required. Always splits the span,
         * if the position points between two characters of the span.
         * Additionally splits the span, if there is no previous sibling text
         * span while the position points to the beginning of the span, or if
         * there is no next text span while the position points to the end of
         * the span.
         *
         * @param {Number[]} position
         *  The logical text position.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.isInsertText=false]
         *      If set to true, this function was called from implInsertText.
         *  @param {Boolean} [options.useCache=false]
         *      If set to true, the paragraph element saved in the selection can
         *      be reused.
         *  @param {Boolean} [options.allowDrawingGroup=false]
         *      If set to true, the element can also be inserted into a drawing
         *      frame of type 'group'.
         *
         * @param {String|Node} [target]
         *  ID of root node or the root node itself.
         *
         * @returns {HTMLSpanElement|Null}
         *  The text span that precedes the passed offset. Will be the leading
         *  part of the original text span addressed by the passed position, if
         *  it has been split, or the previous sibling text span, if the passed
         *  position points to the beginning of the span, or the entire text
         *  span, if the passed position points to the end of a text span and
         *  there is a following text span available or if there is no following
         *  sibling at all.
         *  Returns null, if the passed logical position is invalid.
         */
        function prepareTextSpanForInsertion(position, options, target) {

            var // node info at passed position (DOM text node level)
                nodeInfo = null,
                // whether this text span is required for inserting text (performance)
                isInsertText = Utils.getBooleanOption(options, 'isInsertText', false),
                // whether the cached paragraph can be reused (performance)
                useCache = Utils.getBooleanOption(options, 'useCache', false),
                // the inline component before an empty text span
                inlineNode = null,
                // the character attributes of a previous inline component
                inlineAttrs = null,
                // the node, in which the content will be inserted
                insertNode = null;

            if (useCache && !target) {
                nodeInfo = Position.getDOMPosition(selection.getParagraphCache().node, [_.last(position)]);
            } else {
                if (target) {
                    if (_.isString(target)) {
                        nodeInfo = Position.getDOMPosition(self.getRootNode(target), position);
                    } else {
                        nodeInfo = Position.getDOMPosition(target, position);
                    }
                } else {
                    nodeInfo = Position.getDOMPosition(self.getNode(), position);
                }
            }

            // check that the parent is a text span
            if (!nodeInfo || !nodeInfo.node || !DOM.isPortionSpan(nodeInfo.node.parentNode)) {

                // maybe the nodeInfo is a drawing in the drawing layer
                if (nodeInfo && nodeInfo.node && DOM.isDrawingLayerNode(nodeInfo.node.parentNode)) {
                    nodeInfo.node = DOM.getDrawingPlaceHolderNode(nodeInfo.node).previousSibling.firstChild; // using previous sibling (see template doc)
                    nodeInfo.offset = 0;
                }

                if (!nodeInfo || !nodeInfo.node || !DOM.isPortionSpan(nodeInfo.node.parentNode)) {
                    Utils.warn('Editor.prepareTextSpanForInsertion(): expecting text span at position ' + JSON.stringify(position));
                    return null;
                }
            }
            nodeInfo.node = nodeInfo.node.parentNode;

            // return current span, if offset points to its end
            // without following node or with following text span
            if (nodeInfo.offset === nodeInfo.node.firstChild.nodeValue.length) {
                if ((isInsertText && !nodeInfo.node.nextSibling) || DOM.isTextSpan(nodeInfo.node.nextSibling)) {
                    if (isInsertText && (nodeInfo.offset === 0) && nodeInfo.node.previousSibling && DOM.isInlineComponentNode(nodeInfo.node.previousSibling)) {
                        // inheriting character attributes from previous inline component node, for example 'tabulator'
                        inlineNode = nodeInfo.node.previousSibling;
                        if (inlineNode.firstChild && DOM.isSpan(inlineNode.firstChild)) {
                            inlineAttrs = AttributeUtils.getExplicitAttributes(inlineNode.firstChild);
                            if (inlineAttrs && inlineAttrs.changes) { delete inlineAttrs.changes; } // -> no inheritance of change track attributes
                            if (!_.isEmpty(inlineAttrs)) {
                                characterStyles.setElementAttributes(nodeInfo.node, inlineAttrs);
                            }
                        }
                    }
                    return nodeInfo.node;
                }
            }

            // do not split at beginning with existing preceding text span
            if ((nodeInfo.offset === 0) && nodeInfo.node.previousSibling) {
                if (DOM.isTextSpan(nodeInfo.node.previousSibling)) {
                    return nodeInfo.node.previousSibling;
                } else if (isInsertText && DOM.isInlineComponentNode(nodeInfo.node.previousSibling)) {
                    // inheriting character attributes from previous inline component node, for example 'tabulator'
                    inlineNode = nodeInfo.node.previousSibling;
                    if (inlineNode.firstChild && DOM.isSpan(inlineNode.firstChild)) {
                        inlineAttrs = AttributeUtils.getExplicitAttributes(inlineNode.firstChild);
                        if (inlineAttrs && inlineAttrs.changes) { delete inlineAttrs.changes; } // -> no inheritance of change track attributes
                    }
                }
            }

            // otherwise, split the span
            insertNode = DOM.splitTextSpan(nodeInfo.node, nodeInfo.offset)[0];

            // checking, if there are attributes inherited from previous inline components
            if (isInsertText && inlineAttrs && !_.isEmpty(inlineAttrs)) {
                characterStyles.setElementAttributes(insertNode, inlineAttrs);
            }

            return insertNode;
        }

        /**
         * Helper function to insert the changes mode property into the
         * attribute object. This is necessary for performance reasons:
         * The 'mode: null' property and value shall not be sent with
         * every operation.
         * On the other this is necessary, so that an inserted text/tab/
         * field/hardbreak/... can be included into a change track span,
         * without using changeTrack. So the inserted text/tab.... must
         * not be marked as change tracked.
         * -> This is different to other character attributes.
         *
         * @param {Object} [attrs]
         *  Attributes transfered with the operation. These are checked
         *  for change track information and expanded, if necessary.
         *
         * @returns {Object}
         *  Attributes object, that is expanded with the change track
         *  attributes, if required.
         */
        function checkChangesMode(attrs) {

            var // a new object containing attributes
                newAttrs;

            // if no attributes are defined, set empty changes attributes
            if (!attrs) { return { changes: { inserted: null, removed: null, modified: null } }; }

            if (!attrs.changes) {
                newAttrs = _.copy(attrs, true);
                // if attributes are defined, but no change track attributes, also set empty changes attributes
                newAttrs.changes = { inserted: null, removed: null, modified: null };
                return newAttrs;
            }

            return attrs;
        }

        /**
         * Inserts a simple text portion into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new text portion.
         *
         * @param {String} text
         *  The text to be inserted.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied at the new text portion, as map of
         *  attribute maps (name/value pairs), keyed by attribute family.
         *
         * @param {String} [target]
         *  The target corresponding to the specified logical start position.
         *
         * @returns {Boolean}
         *  Whether the text portion has been inserted successfully.
         */
        function implInsertText(start, text, attrs, target) {

            var // text span that will precede the field
                span = null,
                // new new text span
                newSpan = null, newSpanNode = null,
                // variables for page breaks rendering
                currentElement = null, directParagraph = null, textFrameNode = null,
                // the height of the paragraph or table in that the text is inserted
                prevHeight = null, prevDirectParaHeight = null,
                // whether the height of a paragraph, table or dynamic text frame has changed
                updatePageBreak = false, updateTextFrame = false,
                // whether the height of a node can be changed by this operation handler
                fixedHeight = false,
                // whether the span for text insertion is a spell error span
                isSpellErrorSpan = false,
                // the paragraph node
                paragraphNode = null,
                // the previous operation before this current operation
                previousOperation = self.getPreviousOperation(),
                // whether the cache can be reused (performance)
                useCache = (!target && previousOperation && PARAGRAPH_CACHE_OPERATIONS[previousOperation.name] &&
                            selection.getParagraphCache() && _.isEqual(_.initial(start), selection.getParagraphCache().pos) && _.isUndefined(previousOperation.target));

            // helper function to determine a node, that can be used for measuring changes of height
            // triggered by the text insertion
            function setValidHeightNode() {

                currentElement = paragraphNode;

                // measuring heights to trigger update of page breaks or auto resizing text frames
                if (!$(currentElement.parentNode).hasClass('pagecontent')) {

                    // check, if the node inside a text frame
                    textFrameNode = DrawingFrame.getDrawingNode(currentElement.parentNode);

                    if (textFrameNode && textFrameNode.length > 0) {

                        if (DOM.isCommentNode(textFrameNode)) {
                            currentElement = textFrameNode[0]; // using the comment text frame for height measurement
                        } else {
                            // check, if the text frame node needs to grow dynamically
                            if (DrawingFrame.isAutoResizeHeightDrawingFrame(textFrameNode)) {
                                directParagraph = currentElement;
                                prevDirectParaHeight = currentElement.offsetHeight;

                                // finding a top level node for page break calculation
                                if (DOM.isInsideDrawingLayerNode(currentElement)) {
                                    currentElement = DOM.getTopLevelDrawingInDrawingLayerNode(currentElement);
                                    currentElement = DOM.getDrawingPlaceHolderNode(currentElement);
                                }

                                // searching for the top level paragraph or table
                                currentElement = $(currentElement).parents(DOM.PARAGRAPH_NODE_SELECTOR).last()[0];  // its a div.p inside paragraph

                                if (currentElement && !$(currentElement.parentNode).hasClass('pagecontent')) {
                                    currentElement = $(currentElement).parents(DOM.TABLE_NODE_SELECTOR).last()[0]; // its a text frame inside table(s)
                                }

                            } else {
                                // the height of the text frame will not change! It is not automatically resized.
                                fixedHeight = true;
                            }
                        }

                    } else if (target) {
                        if (!$(currentElement.parentNode).attr('data-container-id')) {
                            currentElement = $(currentElement).parents(DOM.TABLE_NODE_SELECTOR).last()[0]; //its a div.p inside table(s)
                        }
                        // TODO parent DrawingFrame
                    } else {
                        currentElement = $(currentElement).parents(DOM.TABLE_NODE_SELECTOR).last()[0]; // its a div.p inside table(s)
                    }
                }
            }

            // starting with preparing a text span for text insertion
            span = prepareTextSpanForInsertion(start, { isInsertText: true, useCache: useCache }, target);

            if (!span) { return false; }

            paragraphNode = span.parentNode;
            isSpellErrorSpan = DOM.isSpellerrorNode(span);

            // do not write text into an implicit paragraph, improvement for task 30906
            if (DOM.isImplicitParagraphNode(paragraphNode)) { return false; }

            // finding the correct element for measuring height changes caused by this text insertion
            setValidHeightNode();

            prevHeight = currentElement ? currentElement.offsetHeight : 0;

            // Splitting text span is only required, if attrs are available
            if (attrs) {
                attrs = checkChangesMode(attrs); // expanding attrs to disable change tracking, if not explicitely set
                // split the text span again to get actual character formatting for
                // the span, and insert the text
                newSpan = DOM.splitTextSpan(span, span.firstChild.nodeValue.length, { append: true }).contents().remove().end().text(text);
                if (!text) { DOM.ensureExistingTextNode(newSpan); } // empty text should never happen

                // apply the passed text attributes
                if (_.isObject(attrs)) { characterStyles.setElementAttributes(newSpan, attrs); }

                // removing spell error attribute
                if (isSpellErrorSpan) { spellChecker.clearSpellErrorAttributes(newSpan); }

                // removing empty neighboring text spans (merging fails -> avoiding warning in console)
                newSpanNode = Utils.getDomNode(newSpan);
                if (DOM.isChangeTrackNode(newSpanNode)) {
                    if (DOM.isEmptySpan(newSpanNode.previousSibling)) { $(newSpanNode.previousSibling).remove(); }
                    if (DOM.isEmptySpan(newSpanNode.nextSibling)) { $(newSpanNode.nextSibling).remove(); }
                }

            } else {

                // Performance: Simply adding new text into existing node
                // -> But not for change tracking and not if this is an empty span
                if (isSpellErrorSpan || DOM.isChangeTrackNode(span) || DOM.isEmptySpan(span)) {
                    attrs = checkChangesMode(); // expanding attrs to disable change tracking, if not explicitely set
                    newSpan = DOM.splitTextSpan(span, span.firstChild.nodeValue.length, { append: true }).contents().remove().end().text(text);
                    characterStyles.setElementAttributes(newSpan, attrs);
                    if (isSpellErrorSpan) { spellChecker.clearSpellErrorAttributes(newSpan); }
                } else {
                    span.firstChild.nodeValue += text;
                    newSpan = $(span);
                }
            }

            // try to merge with preceding and following span
            Utils.mergeSiblingTextSpans(newSpan, true);
            Utils.mergeSiblingTextSpans(newSpan);

            // validate paragraph, store new cursor position
            if ((guiTriggeredOperation) || (!app.isImportFinished()) || pasteInProgress) {
                implParagraphChanged(newSpan.parent());  // Performance and task 30587: Local client can defer attribute setting, Task 30603: deferring also during loading document
            } else {
                implParagraphChangedSync(newSpan.parent());  // Performance and task 30587: Remote client must set attributes synchronously
            }

            if (target) {
                self.setBlockOnInsertPageBreaks(true);
            }

            quitFromPageBreak = true; // quit from any currently running pagebreak calculus - performance optimization

            // checking changes of height for top level elements (for page breaks) and for direct paragraph (auto resize text frame)
            if (!fixedHeight) {
                if (directParagraph && prevDirectParaHeight !== $(directParagraph).height()) {
                    updateTextFrame = true;
                } else {
                    textFrameNode = null;
                }

                if ($(currentElement).data('lineBreaksData')) {
                    $(currentElement).removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated
                }

                if (prevHeight !== $(currentElement).height() || (DOM.isEmptySpan(span) && DOM.isHardBreakNode(newSpan.prev()))) { // #39468 - also update pb when typing after ms page hardbreak
                    updatePageBreak = true;
                }

                if (updatePageBreak || updateTextFrame) {
                    insertPageBreaksDebounced(currentElement, textFrameNode);
                    if (!pbState) { app.getView().recalculateDocumentMargin(); }
                }
            }

            lastOperationEnd = Position.increaseLastIndex(start, text.length);

            // Performance: Saving paragraph info for following operations
            selection.setParagraphCache(newSpan.parent(), _.clone(_.initial(lastOperationEnd)), _.last(lastOperationEnd));

            return true;
        }

        /**
         * Inserts a text field component into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new text field.
         *
         * @param {String} type
         *  A property describing the field type.
         *
         * @param {String} representation
         *  A fallback value, if the placeholder cannot be substituted with a
         *  reasonable value.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied at the new text field, as map of attribute
         *  maps (name/value pairs), keyed by attribute family.
         *
         * @param {Boolean} [external]
         *  Will be set to true, if the invocation of this method originates
         *  from an external operation.
         *
         * @returns {Boolean}
         *  Whether the text field has been inserted successfully.
         */
        function implInsertField(start, type, representation, attrs, target/*, external*/) {

            var // text span that will precede the field
                span = prepareTextSpanForInsertion(start, {}, target),
                // new text span for the field node
                fieldSpan = null,
                // the field node
                fieldNode = null;

            if (!span) { return false; }

            // expanding attrs to disable change tracking, if not explicitely set
            attrs = checkChangesMode(attrs);

            // split the text span again to get initial character formatting
            // for the field, and insert the field representation text

            // Fix for 29265: Removing empty text node in span with '.contents().remove().end()'.
            // Otherwise there are two text nodes in span after '.text(representation)' in IE.
            fieldSpan = DOM.splitTextSpan(span, 0).contents().remove().end().text(representation);
            if (!representation) { DOM.ensureExistingTextNode(fieldSpan); }

            // insert a new text field before the addressed text node, move
            // the field span element into the field node
            fieldNode = DOM.createFieldNode();
            fieldNode.append(fieldSpan).insertAfter(span);
            fieldNode.data('type', type);

            // odf - field types needed to be updated in frontend
            if (type === 'page-number' || type === 'page-count') {
                fieldNode.addClass('field-' + type);
                if (type === 'page-count') {
                    pageLayout.updatePageCountField(fieldNode);
                } else {
                    pageLayout.updatePageNumberField(fieldNode);
                }
            }

            // microsoft - field types needed to be updated in frontend
            if ((/NUMPAGES/i).test(type)) {
                fieldNode.addClass('field-NUMPAGES');
                pageLayout.updatePageCountField(fieldNode);
            }
            if ((/PAGENUM/i).test(type)) {
                fieldNode.addClass('field-page-number');
                pageLayout.updatePageNumberField(fieldNode);
            }

            if (!representation) {
                fieldNode.addClass('empty-field');
            }

            // apply the passed field attributes
            if (_.isObject(attrs)) {
                if (_.isObject(attrs.field)) {
                    _.each(attrs.field, function (element, name) {
                        fieldNode.data(name, element);
                        if (name === 'name') {
                            fieldNode.addClass('user-field');
                        }
                    });
                }
                characterStyles.setElementAttributes(fieldSpan, attrs);
            }

            fieldManager.addSimpleFieldToCollection(fieldNode, target);

            // validate paragraph, store new cursor position
            implParagraphChanged(span.parentNode);
            lastOperationEnd = Position.increaseLastIndex(start);
            return true;
        }

        /**
         * Inserts a horizontal tabulator component into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new tabulator.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied at the new tabulator component, as map of
         *  attribute maps (name/value pairs), keyed by attribute family.
         *
         * @param {Object} [target]
         *  If exists, defines node, to which start position is related.
         *  Used primarily to address headers/footers.
         *
         * @param {Boolean} [external]
         *  Will be set to true, if the invocation of this method originates
         *  from an external operation.
         *
         * @returns {Boolean}
         *  Whether the tabulator has been inserted successfully.
         */
        function implInsertTab(start, attrs, target/*, external*/) {

            var // text span that will precede the field
                span = prepareTextSpanForInsertion(start, {}, target),
                // new text span for the tabulator node
                tabSpan = null,
                // new tabulator node
                newTabNode,
                // element from which we calculate pagebreaks
                currentElement = span ? span.parentNode : '';

            if (!span) { return false; }

            // expanding attrs to disable change tracking, if not explicitely set
            attrs = checkChangesMode(attrs);

            // split the text span to get initial character formatting for the tab
            tabSpan = DOM.splitTextSpan(span, 0);
            if (_.browser.IE) {
                // Prevent resizing rectangle provided by IE for tab div even
                // set to contenteditable=false. Therefore we need to set IE
                // specific unselectable=on.
                tabSpan.attr('unselectable', 'on');
            } else if (_.browser.iOS && _.browser.Safari) {
                // Safari mobile allows to visit nodes which has contenteditable=false, therefore
                // we have to use a different solution. We use -webkit-user-select: none to force
                // Safari mobile to jump over the text span with the fill characters.
                tabSpan.css('-webkit-user-select', 'none');
            }

            newTabNode = DOM.createTabNode();

            // insert a tab container node before the addressed text node, move
            // the tab span element into the tab container node
            newTabNode.append(tabSpan).insertAfter(span);

            // apply the passed tab attributes
            if (_.isObject(attrs)) {
                characterStyles.setElementAttributes(tabSpan, attrs);
            }

            // validate paragraph, store new cursor position
            implParagraphChanged(span.parentNode);
            lastOperationEnd = Position.increaseLastIndex(start);

            // don't call explicitly page break rendering, if target for header/footer comes with operation
            if (target) {
                self.setBlockOnInsertPageBreaks(true);
            }

            // we changed paragraph layout, cached line breaks data needs to be invalidated
            if ($(currentElement).data('lineBreaksData')) {
                $(currentElement).removeData('lineBreaksData');
            }
            // call for debounced render of pagebreaks
            // TODO: Check if paragraph height was modified
            quitFromPageBreak = true; // quit from any currently running pagebreak calculus - performance optimization

            insertPageBreaksDebounced(currentElement, DrawingFrame.getClosestTextFrameDrawingNode(span.parentNode));
            if (!pbState) { app.getView().recalculateDocumentMargin(); }

            // immediately updating element formatting to underline the tab correctly
            // -> This is not necessary with localStorage and fastLoad during loading
            if ((app.isImportFinished()) && (changeTrack.isActiveChangeTracking())) {
                paragraphStyles.updateElementFormatting(span.parentNode);
            }

            return true;
        }

        /**
         * Inserts a hard break component into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new tabulator.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied at the new hard-break component, as map of
         *  attribute maps (name/value pairs), keyed by attribute family.
         *
         * @param {Boolean} [external]
         *  Will be set to true, if the invocation of this method originates
         *  from an external operation.
         *
         * @returns {Boolean}
         *  Whether the tabulator has been inserted successfully.
         */
        function implInsertHardBreak(start, type, attrs, target/*, external*/) {

            var // text span that will precede the hard-break node
                span = prepareTextSpanForInsertion(start, {}, target),
                // new text span for the hard-break node
                hardbreakSpan = null,
                // element from which we calculate pagebreaks
                currentElement = span ? span.parentNode : '',
                // the hard break type
                hardBreakType = type || 'textWrapping';  // defaulting to 'textWrapping'

            if (!span) { return false; }

            // expanding attrs to disable change tracking, if not explicitely set
            attrs = checkChangesMode(attrs);

            // split the text span to get initial character formatting for the hard-break
            hardbreakSpan = DOM.splitTextSpan(span, 0);

            // insert a hard-break container node before the addressed text node, move
            // the hard-break span element into the hard-break container node
            // Fix for 29265: Replacing '.empty()' of hardbreakSpan with '.contents().remove().end()'
            // to get rid of empty text node in span in IE.

            // TODO: implementing hard break type 'column'
            // -> fallback to text node with one space, so that counting is still valid.
            if ((hardBreakType === 'page') || (hardBreakType === 'column')) {
                hardbreakSpan.contents().remove().end().text('\xa0');  // non breakable space
                hardbreakSpan.addClass('ms-hardbreak-page');
                $(currentElement).addClass('manual-page-break');
            } else {
                hardbreakSpan.contents().remove().end().append($('<br>'));
            }

            DOM.createHardBreakNode(hardBreakType).append(hardbreakSpan).insertAfter(span);

            // apply the passed tab attributes
            if (_.isObject(attrs)) {
                characterStyles.setElementAttributes(hardbreakSpan, attrs);
            }

            // validate paragraph, store new cursor position
            implParagraphChanged(span.parentNode);
            lastOperationEnd = Position.increaseLastIndex(start);

            // don't call explicitly page break rendering, if target for header/footer comes with operation
            if (target) {
                self.setBlockOnInsertPageBreaks(true);
            }

            if ($(currentElement).data('lineBreaksData')) {
                $(currentElement).removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated
            }
            // call for debounced render of pagebreaks
            insertPageBreaksDebounced(currentElement, DrawingFrame.getClosestTextFrameDrawingNode(span.parentNode));
            if (!pbState) { app.getView().recalculateDocumentMargin(); }

            // immediately updating element formatting to underline the tab correctly
            // -> This is not necessary with localStorage and fastLoad during loading
            if ((app.isImportFinished()) && (changeTrack.isActiveChangeTracking())) {
                paragraphStyles.updateElementFormatting(span.parentNode);
            }

            return true;
        }

        /**
         * Inserts a comment into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new comment.
         *
         * @param {String} id
         *  The unique id of the comment.
         *
         * @param {String} author
         *  The author of the comment.
         *
         * @param {String} date
         *  The date of the comment.
         *
         * @param {String} [target]
         *  The target corresponding to the specified logical start position.
         *
         * @returns {Boolean}
         *  Whether the comment has been inserted successfully.
         */
        function implInsertComment(start, id, author, uid, date, target) {
            return commentLayer.insertCommentHandler(start, id, author, uid, date, target);
        }

        /**
         * Inserts a complex field into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new complex field.
         *
         * @param {String} id
         *  The unique id of the complex field.
         *
         * @param {String} instruction
         *  The instruction string of the complex field.
         *
         * @param {String} [target]
         *  The target corresponding to the specified logical start position.
         *
         * @returns {Boolean}
         *  Whether the complex field has been inserted successfully.
         */
        function implInsertComplexField(start, id, instruction, target) {
            return fieldManager.insertComplexFieldHandler(start, id, instruction, target);
        }

        /**
         * Inserts a range marker node into the document DOM. These ranges can be
         * used for several features like comments, complex fields, ...
         *
         * @param {Number[]} start
         *  The logical start position for the new range marker.
         *
         * @param {String} id
         *  The unique id of the range marker. This can be used to identify the
         *  corresponding node, that requires this range. This can be a comment
         *  node for example.
         *
         * @param {String} type
         *  The type of the range marker.
         *
         * @param {String} position
         *  The position of the range marker. Currently supported are 'start'
         *  and 'end'.
         *
         * @param {String} [target]
         *  The target corresponding to the specified logical start position.
         *
         * @returns {Boolean}
         *  Whether the range marker has been inserted successfully.
         */
        function implInsertRange(start, id, type, position, target) {
            return rangeMarker.insertRangeHandler(start, id, type, position, target);
        }

        /**
         * Inserts a drawing component into the document DOM.
         *
         * @param {String} type
         *  The type of the drawing. Supported values are 'shape', 'group',
         *  'image', 'diagram', 'chart', 'ole', 'horizontal_line', 'undefined'
         *
         * @param {Number[]} start
         *  The logical start position for the new tabulator.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied at the new drawing component, as map of
         *  attribute maps (name/value pairs), keyed by attribute family.
         *
         * @param {Object} [target]
         *  If exists, defines node, to which start position is related.
         *  Used primary to address headers/footers.
         *
         * @returns {Boolean}
         *  Whether the drawing has been inserted successfully.
         */
        function implInsertDrawing(type, start, attrs, target) {

            var // text span that will precede the field
                span = null,
                // deep copy of attributes, because they are modified in webkit browsers
                attributes = _.copy(attrs, true),
                // new drawing node
                drawingNode = null,
                // root node container
                rootNode = self.getRootNode(target),
                // image aspect ratio
                currentElement = Position.getContentNodeElement(rootNode, start.slice(0, -1), { allowDrawingGroup: true }),
                // whether the drawing is inserted into a drawing group
                insertIntoDrawingGroup = false,
                // the function used to insert the new drawing frame
                insertFunction = 'insertAfter',
                // the number of children in the drawing group
                childrenCount = 0;

            // helper function to insert a drawing frame into an existing drawing group
            function addDrawingFrameIntoDrawingGroup () {

                childrenCount = DrawingFrame.getGroupDrawingCount(currentElement);

                if (_.isNumber(childrenCount)) {
                    if (childrenCount === 0) {
                        if (_.last(start) === 0) {
                            span = $(currentElement).children().first(); // using 'span' for the content element in the drawing group
                            insertFunction = 'appendTo';
                        }
                    } else {
                        if (_.last(start) === 0) {
                            // inserting before the first element
                            span = DrawingFrame.getGroupDrawingChildren(currentElement, 0);
                            insertFunction = 'insertBefore';
                        } else if (_.last(start) <= childrenCount) {
                            // inserting after the span element
                            span = DrawingFrame.getGroupDrawingChildren(currentElement, _.last(start) - 1);
                        }
                    }
                    insertIntoDrawingGroup = true;
                }
            }

            try {
                span = prepareTextSpanForInsertion(start, {}, target);
            } catch (ex) {
                // do nothing, try to repair missing text spans
            }

            // check, if the drawing is inserted into a drawing group
            if (!span && DrawingFrame.isGroupDrawingFrame(currentElement)) { addDrawingFrameIntoDrawingGroup(); }

            if (!span) { return false; }

            // expanding attrs to disable change tracking, if not explicitely set
            attrs = checkChangesMode(attrs);

            // insert the drawing with default settings between the two text nodes (store original URL for later use)
            drawingNode = DrawingFrame.createDrawingFrame(type)[insertFunction](span);

            if (type === 'horizontal_line') { drawingNode.addClass('horizontal-line'); }

            // apply the passed drawing attributes
            if (_.isObject(attributes)) {
                drawingStyles.setElementAttributes(drawingNode, attributes);
            }

            // validate paragraph, store new cursor position
            if (!insertIntoDrawingGroup) { implParagraphChanged(span.parentNode); }

            lastOperationEnd = Position.increaseLastIndex(start);

            // don't call explicitly page break rendering, if target for header/footer comes with operation
            if (target) {
                self.setBlockOnInsertPageBreaks(true);
            }
            if ($(currentElement).data('lineBreaksData')) {
                $(currentElement).removeData('lineBreaksData'); // we changed paragraph layout, cached line breaks data needs to be invalidated
            }
            insertPageBreaksDebounced(currentElement, insertIntoDrawingGroup ? null : DrawingFrame.getClosestTextFrameDrawingNode(span.parentNode));
            if (!pbState) { app.getView().recalculateDocumentMargin(); }

            return true;
        }

        /**
         * Internet Explorer helper function. Repairs empty text nodes, that are
         * removed by IE after inserting a node into the DOM. This function
         * adds this empty text nodes again.
         *
         * After updating to a new version of jQuery, this seems to be
         * nessecary in other browsers too.
         *
         * @param {HTMLElement|jQuery} element
         *  The DOM element whose empty text nodes shall be added again.
         *
         * @param {Object} [options]
         *  {Boolean} allNodes - optional, default value = false
         *      If set to true, iterator will go through all children nodes, not only first level children.
         *      Use only if really needed, because it costs performance.
         */
        function repairEmptyTextNodes(element, options) {
            var // option to iterate all children, not only first level (costs performance, use only if really needed)
                iterateAllNodes = Utils.getBooleanOption(options, 'allNodes', false),
                // when element is paragraph, selects ether all or only first level children spans
                selector = null;

            if (!_.browser.IE) { return; }  // only necessary for IE and Spartan

            if ($(element).is('span')) {
                DOM.ensureExistingTextNode(element);
            } else if (DOM.isParagraphNode(element)) {
                selector = iterateAllNodes ? 'span' : '> span';
                $(element).find(selector).each(function () {
                    DOM.ensureExistingTextNode(this);
                });
            } else if (iterateAllNodes) {
                _.each($(element).find(DOM.PARAGRAPH_NODE_SELECTOR + ' > span'), function (span) {
                    DOM.ensureExistingTextNode(span);
                });
            } else if (DOM.isTableCellNode(element)) {
                DOM.getCellContentNode(element).find('> ' + DOM.PARAGRAPH_NODE_SELECTOR + ' > span').each(function () {
                    DOM.ensureExistingTextNode(this);
                });
            } else if (DOM.isTableRowNode(element)) {
                $(element).find('> td').each(function () {
                    DOM.getCellContentNode(this).find('> ' + DOM.PARAGRAPH_NODE_SELECTOR + ' > span').each(function () {
                        DOM.ensureExistingTextNode(this);
                    });
                });
            } else if (DOM.isTableNode(element)) {
                $(element).find('> tbody > tr > td').each(function () {
                    DOM.getCellContentNode(this).find('> ' + DOM.PARAGRAPH_NODE_SELECTOR + ' > span').each(function () {
                        DOM.ensureExistingTextNode(this);
                    });
                });
            }
        }

        /**
         * Returns the family of the attributes supported by the passed DOM
         * element.
         *
         * @param {HTMLElement|jQuery} element
         *  The DOM element whose associated attribute family will be returned.
         *  If this object is a jQuery collection, returns its first node.
         */
        function resolveElementFamily(element) {

            var // the element, as jQuery object
                $element = $(element),
                // the resulting style family
                family = null;

            if (DrawingFrame.isDrawingFrame($element)) {
                family = 'drawing'; // selecting drawing first (necessary after 39312)
            } else if (DOM.isPartOfParagraph($element)) {
                family = 'character';
            } else if (DOM.isParagraphNode($element)) {
                family = 'paragraph';
            } else if (DOM.isTableNode($element)) {
                family = 'table';
            } else if ($element.is('tr')) {
                family = 'row';
            } else if ($element.is('td')) {
                family = 'cell';
            } else {
                Utils.warn('Editor.resolveElementFamily(): unsupported element');
            }

            return family;
        }

        /**
         * Returns the type of the attributes supported by the passed DOM
         * element.
         *
         * @param {HTMLElement|jQuery} element
         *  The DOM element whose associated attribute type will be returned.
         *  If this object is a jQuery collection, returns its first node.
         */
        function resolveElementType(element) {

            var // the element, as jQuery object
                $element = $(element),
                // the resulting style type
                type = null;

            if (DOM.isPartOfParagraph($element)) {
                type = 'text';
            } else if (DOM.isParagraphNode($element)) {
                type = 'paragraph';
            } else if (DOM.isTableNode($element)) {
                type = 'table';
            } else if ($element.is('tr')) {
                type = 'row';
            } else if ($element.is('td')) {
                type = 'cell';
            } else {
                Utils.warn('Editor.resolveElementType(): unsupported element');
            }

            return type;
        }

        /**
         * Changes a specific formatting attribute of the specified element or
         * text range. The type of the attributes will be determined from the
         * specified range.
         *
         * @param {Number[]} start
         *  The logical start position of the element or text range to be
         *  formatted.
         *
         * @param {Number[]} [end]
         *  The logical end position of the element or text range to be
         *  formatted.
         *
         * @param {Object} attributes
         *  A map with formatting attribute values, mapped by the attribute
         *  names, and by attribute family names.
         */
        var implSetAttributes = (function () {

            // temporary array with only one entry, for use in Position functions
            var TMPPOSARRAY = [0];

            return function (start, end, attributes, target) {

                var // node info for start/end position
                    startInfo = null, endInfo = null,
                    // the main attribute family of the target components
                    styleFamily = null,
                    // the style sheet container for the target components
                    styleSheets = null,
                    // options for style collection method calls
                    options = null,
                    // the last text span visited by the character formatter
                    lastTextSpan = null,
                    // undo operations going into a single action
                    undoOperations = [],
                    // redo operation
                    redoOperation = null,
                    // Performance: Saving data for list updates
                    paraAttrs = null, listStyleId = null, listLevel = null, oldListStyleId = null,
                    // a helper attribute object with values set to null
                    nullAttributes = null,
                    // a helper iterator key
                    localkey = null,
                    //element which is passed to page breaks calculation
                    currentElement,
                    // the previous operation before this current operation
                    previousOperation = null,
                    // whether the paragraph cache can be reused (performance)
                    useCache = false,
                    // whether an update of formatting is required after assigning attributes (needed by spellcheck attribute)
                    forceUpdateFormatting = false,
                    //flags for triggering page breaks
                    isCharOrDrawingOrRow,
                    isClearFormatting,
                    isAttributesInParagraph,
                    isPageBreakBeforeAttributeInParagraph,
                    // if target is present, root node is header or footer, otherwise editdiv
                    rootNode = self.getRootNode(target);

                // sets or clears the attributes using the current style sheet container
                function setElementAttributes(element) {
                    styleSheets.setElementAttributes(element, attributes, options);
                }

                // change listener used to build the undo operations
                function changeListener(element, oldAttributes, newAttributes) {

                    var // selection object representing the passed element
                        range = Position.getPositionRangeForNode(rootNode, element),
                        // the attributes of the current family for the undo operation
                        undoAttributes = {},
                        // the operation used to undo the attribute changes
                        undoOperation = { name: Operations.SET_ATTRIBUTES, start: range.start, end: range.end, attrs: undoAttributes },
                        // last undo operation (used to merge character attributes of sibling text spans)
                        lastUndoOperation = ((undoOperations.length > 0) && (styleFamily === 'character')) ? _.last(undoOperations) : null;

                    // extend undoOperation if target is present
                    extendPropertiesWithTarget(undoOperation, target);

                    function insertUndoAttribute(family, name, value) {
                        undoAttributes[family] = undoAttributes[family] || {};
                        undoAttributes[family][name] = value;
                    }

                    // exceptional case for setting line type to 'none', to save the old color
                    if (_.has(attributes, 'line') && _.has(attributes.line, 'type') && attributes.line.type === 'none') {
                        if (_.has(oldAttributes, 'line') && _.has(oldAttributes.line, 'color') && !_.isUndefined(oldAttributes.line.color)) {
                            undoAttributes.line = {};
                            undoAttributes.line.color = oldAttributes.line.color;
                        }
                    }

                    // exceptional case for setting fill type to 'none', to save the old color
                    if (_.has(attributes, 'fill') && _.has(attributes.fill, 'type') && attributes.fill.type === 'none') {
                        if (_.has(oldAttributes, 'fill') && _.has(oldAttributes.fill, 'color') && !_.isUndefined(oldAttributes.fill.color)) {
                            undoAttributes.fill = {};
                            undoAttributes.fill.color = oldAttributes.fill.color;
                        }
                    }

                    // find all old attributes that have been changed or cleared
                    _(oldAttributes).each(function (attributeValues, family) {
                        if (family === 'styleId') {
                            // style sheet identifier is a string property
                            if (attributeValues !== newAttributes.styleId) {
                                undoAttributes.styleId = attributeValues;
                            }
                        } else {
                            // process attribute map of the current family
                            _(attributeValues).each(function (value, name) {
                                if (!(family in newAttributes) || !_.isEqual(value, newAttributes[family][name])) {
                                    insertUndoAttribute(family, name, value);
                                }
                            });
                        }
                    });

                    // find all newly added attributes
                    _(newAttributes).each(function (attributeValues, family) {
                        if (family === 'styleId') {
                            // style sheet identifier is a string property
                            if (!_.isString(oldAttributes.styleId)) {
                                undoAttributes.styleId = null;
                            }
                        } else {
                            // process attribute map of the current family
                            _(attributeValues).each(function (value, name) {
                                if (!(family in oldAttributes) || !(name in oldAttributes[family])) {
                                    insertUndoAttribute(family, name, null);
                                }
                            });
                        }
                    });

                    // try to merge 'character' undo operation with last array entry, otherwise add operation to array
                    if (lastUndoOperation && (_.last(lastUndoOperation.end) + 1 === _.last(undoOperation.start)) && _.isEqual(lastUndoOperation.attrs, undoOperation.attrs)) {
                        lastUndoOperation.end = undoOperation.end;
                    } else {
                        undoOperations.push(undoOperation);
                    }

                    // invalidate spell result if language attribute changes
                    spellChecker.resetClosest(element);

                }

                // fail if attributes is not of type object
                if (!_.isObject(attributes)) {
                    return false;
                }

                // do nothing if an empty attributes object has been passed
                if (_.isEmpty(attributes)) {
                    return;
                }

                // resolve start and end position
                if (!_.isArray(start)) {
                    Utils.warn('Editor.implSetAttributes(): missing start position');
                    return false;
                }

                // whether the cache can be reused (performance)
                previousOperation = self.getPreviousOperation();
                useCache = (!target && previousOperation && PARAGRAPH_CACHE_OPERATIONS[previousOperation.name] &&
                            selection.getParagraphCache() && _.isEqual(_.initial(start), selection.getParagraphCache().pos) && _.isUndefined(previousOperation.target));

                if (useCache) {
                    TMPPOSARRAY[0] = _.last(start);
                    startInfo = Position.getDOMPosition(selection.getParagraphCache().node, TMPPOSARRAY, true);
                    // setting paragraph cache is not required, it can simply be reused in next operation
                } else {
                    startInfo = Position.getDOMPosition(rootNode, start, true);

                    if (!startInfo) {
                        Utils.warn('Editor.implSetAttributes(): invalid start node!');
                    }

                    if (startInfo.family === 'character') {
                        selection.setParagraphCache(startInfo.node.parentElement, _.clone(_.initial(start)), _.last(start));
                    } else {
                        selection.setParagraphCache(null); // invalidating the cache (previous operation might not be in PARAGRAPH_CACHE_OPERATIONS)
                    }
                }

                if (!startInfo || !startInfo.node) {
                    Utils.warn('Editor.implSetAttributes(): invalid start position: ' + JSON.stringify(start));
                    return false;
                }
                // get attribute family of start and end node
                startInfo.family = resolveElementFamily(startInfo.node);

                if (!startInfo.family) { return; }

                if (_.isArray(end)) {
                    if (startInfo.family === 'character') {
                        //characters are children of paragraph, end is in the same paragraph, so we dont have to search the complete document for it
                        TMPPOSARRAY[0] = _.last(end);
                        endInfo = Position.getDOMPosition(startInfo.node.parentElement, TMPPOSARRAY, true);
                    } else {
                        endInfo = Position.getDOMPosition(rootNode, end, true);
                    }

                    endInfo.family = resolveElementFamily(endInfo.node);

                    if (!endInfo.family) { return; }

                } else {
                    end = start;
                    endInfo = startInfo;
                }

                // options for the style collection method calls (build undo operations while formatting)
                options = undoManager.isUndoEnabled() ? { changeListener: changeListener } : null;

                // characters (start or end may point to a drawing node, ignore that but format as
                // characters if the start object is different from the end object)
                if ((startInfo.family === 'character') || (endInfo.family === 'character') ||
                        ((startInfo.node !== endInfo.node) && DrawingFrame.isDrawingFrame(startInfo.node) && DrawingFrame.isDrawingFrame(endInfo.node))) {

                    // check that start and end are located in the same paragraph (and handling absolute positioned drawings correctly)
                    if (startInfo.node.parentNode !== endInfo.node.parentNode) {

                        // replacing the absolute positioned drawings by its place holder nodes
                        if (DOM.isDrawingLayerNode(startInfo.node.parentNode)) { startInfo.node = DOM.getDrawingPlaceHolderNode(startInfo.node); }
                        if (DOM.isDrawingLayerNode(endInfo.node.parentNode)) { endInfo.node = DOM.getDrawingPlaceHolderNode(endInfo.node); }

                        if (startInfo.node.parentNode !== endInfo.node.parentNode) {
                            Utils.warn('Editor.implSetAttributes(): end position in different paragraph');
                            return;
                        }
                    }

                    // visit all text span elements covered by the passed range
                    // (not only the direct children of the paragraph, but also
                    // text spans embedded in component nodes such as fields and tabs)
                    styleFamily = 'character';
                    styleSheets = characterStyles;
                    Position.iterateParagraphChildNodes(startInfo.node.parentNode, function (node) {

                        // visiting the span inside a hard break node
                        // -> this is necessary for change tracking attributes
                        if (DOM.isHardBreakNode(node)) {
                            setElementAttributes(node.firstChild);
                        }

                        // DOM.iterateTextSpans() visits the node itself if it is a
                        // text span, otherwise it visits all descendant text spans
                        // contained in the node except for drawings which will be
                        // skipped (they may contain their own paragraphs).
                        DOM.iterateTextSpans(node, function (span) {
                            // check for a spellchecked span (this needs to be checked, before attributes are applied)
                            if (!forceUpdateFormatting && app.isImportFinished() && DOM.isSpellerrorNode(span)) { forceUpdateFormatting = true; }
                            // assigning the new character attributes
                            setElementAttributes(span);
                            // try to merge with the preceding text span
                            Utils.mergeSiblingTextSpans(span, false);
                            // remember span (last visited span will be merged with its next sibling)
                            lastTextSpan = span;
                        });

                    }, undefined, {
                        // options for Position.iterateParagraphChildNodes()
                        allNodes: true,
                        start: _(start).last(),
                        end: _(end).last(),
                        split: true
                    });

                    // handling spell checking after modifying character attributes
                    if (forceUpdateFormatting) { implParagraphChanged(startInfo.node.parentNode); }

                    // try to merge last text span in the range with its next sibling
                    if (lastTextSpan) {
                        Utils.mergeSiblingTextSpans(lastTextSpan, true);
                    }

                // otherwise: only single components allowed at this time
                } else {

                    // check that start and end point to the same element
                    if (startInfo.node !== endInfo.node) {
                        Utils.warn('Editor.implSetAttributes(): no ranges supported for attribute family "' + startInfo.family + '"');
                        return;
                    }

                    // format the (single) element
                    styleFamily = startInfo.family;
                    styleSheets = self.getStyleCollection(styleFamily);

                    // Performance: Saving old list style id, before it is removed
                    if ((app.isImportFinished()) && (styleFamily === 'paragraph')) {
                        paraAttrs = AttributeUtils.getExplicitAttributes(startInfo.node);
                        if (paraAttrs) { oldListStyleId = getListStyleIdFromParaAttrs(paraAttrs); }
                    }

                    setElementAttributes(startInfo.node);
                }

                // create the undo action
                if (undoManager.isUndoEnabled()) {
                    redoOperation = { name: Operations.SET_ATTRIBUTES, start: start, end: end, attrs: attributes };
                    // extend redoOperation if target is present
                    extendPropertiesWithTarget(redoOperation, target);
                    undoManager.addUndo(undoOperations, redoOperation);
                }

                // update numberings and bullets (but updateListsDebounced can only be called after successful document import)
                if ((app.isImportFinished()) && (styleFamily === 'paragraph')) {
                    if ((('styleId' in attributes) || _.isObject(attributes.paragraph) && (('listLevel' in attributes.paragraph) || ('listStyleId' in attributes.paragraph)))) {

                        // determining listStyleId und listLevel
                        if ((_.isObject(attributes.paragraph)) && ('listStyleId' in attributes.paragraph) && (attributes.paragraph.listStyleId !== null)) {
                            listStyleId = attributes.paragraph.listStyleId;  // list style assigned to paragraph
                            listLevel = ('listLevel' in attributes.paragraph) ? attributes.paragraph.listLevel : 0;
                        } else if (attributes.styleId && isParagraphStyleWithListStyle(attributes.styleId)) {
                            listStyleId = getListStyleInfoFromStyleId(attributes.styleId, 'listStyleId');
                            listLevel = getListStyleInfoFromStyleId(attributes.styleId, 'listLevel') || 0;
                        } else {
                            // list level modified -> checking paragraph attributes for list styles to receive list style id
                            paraAttrs = AttributeUtils.getExplicitAttributes(startInfo.node);
                            // updating lists, if required
                            if (isListStyleParagraph(null, paraAttrs)) {
                                if (paraAttrs && paraAttrs.paragraph) {
                                    listStyleId = paraAttrs.paragraph.listStyleId;
                                    listLevel = paraAttrs.paragraph.listLevel;
                                }
                            }
                        }

                        // handling the case, that the list style id is removed
                        if (!listStyleId && oldListStyleId) { listStyleId = oldListStyleId; }

                        // defining list style Id or listLevel, that need to be updated
                        if (listStyleId || listLevel) {
                            if (!listStyleId) {
                                listStyleId = paragraphStyles.getElementAttributes(startInfo.node).paragraph.listStyleId;
                                //fix for Bug 37594 styleId is not at the para-attrs and not in the style it self. but by merging we get the correct stlyeId
                            }
                            // mark this paragraph for later list update. This is necessary, because it does not yet contain
                            // the list label node (DOM.LIST_LABEL_NODE_SELECTOR) and is therefore ignored in updateLists.
                            $(startInfo.node).data('updateList', 'true');
                            // registering this list style for update
                            // special handling for removal of listLevel, for example after 'Backspace'
                            if (listStyleId && (listLevel === null || listLevel === -1)) {
                                $(startInfo.node).data('removeLabel', 'true');
                            }

                            // not registering listStyle for updating, if this is a bullet list -> updating only specific paragraphs
                            if (listCollection.isAllLevelsBulletsList(listStyleId)) { listStyleId = null; }

                            updateListsDebounced({ useSelectedListStyleIDs: true, paraInsert: true, listStyleId: listStyleId, listLevel: listLevel });
                        }

                    } else if ('character' in attributes) {  // modified character attributes may lead to modified list labels

                        // checking paragraph attributes for list styles to receive list style id
                        paraAttrs = AttributeUtils.getExplicitAttributes(startInfo.node);
                        // updating lists, after character attributes were modified
                        if (isListStyleParagraph(null, paraAttrs)) {
                            if (paraAttrs && paraAttrs.paragraph) {
                                listStyleId = paraAttrs.paragraph.listStyleId;
                                listLevel = paraAttrs.paragraph.listLevel;
                                updateListsDebounced({ useSelectedListStyleIDs: true, paraInsert: true, listStyleId: listStyleId, listLevel: listLevel });
                            }
                        }
                    }
                }

                if ((styleFamily === 'paragraph') && (attributes.character) && (Position.getParagraphNodeLength(startInfo.node) === 0)) {
                    // Task 28187: Setting character attributes at empty paragraphs
                    // -> removing character attributes from text span so that character styles at paragraph become visible (also 30927)
                    nullAttributes = _.copy(attributes, true);
                    for (localkey in nullAttributes.character) { nullAttributes.character[localkey] = null; }
                    characterStyles.setElementAttributes(DOM.findFirstPortionSpan(startInfo.node), nullAttributes);
                }

                if (app.isImportFinished()) {
                    // adjust tabulators, if character or drawing attributes have been changed
                    // (changing paragraph or table attributes updates tabulators automatically)
                    if ((styleFamily === 'character') || (styleFamily === 'drawing')) {
                        paragraphStyles.updateTabStops(startInfo.node.parentNode);
                    }

                    // if target comes with operation, take care of targeted header/footer, otherwise, run normal page break rendering
                    if (target) {
                        // trigger header/footer content update on other elements of same type, if change was made inside header/footer
                        if (DOM.isMarginalNode(startInfo.node)) {
                            updateEditingHeaderFooterDebounced(DOM.getMarginalTargetNode(self.getNode(), startInfo.node));
                        } else if (DOM.isMarginalNode(startInfo.node.parentNode)) {
                            updateEditingHeaderFooterDebounced(DOM.getMarginalTargetNode(self.getNode(), startInfo.node.parentNode));
                        }
                        self.setBlockOnInsertPageBreaks(true);
                    }

                    // after document is loaded, trigger pagebreaks reposition on certain events like font size change, height of drawing or row changed
                    isCharOrDrawingOrRow = 'character' in attributes || 'drawing' in attributes || 'row' in attributes || ('line' in attributes && 'width' in attributes.line);
                    isClearFormatting = 'styleId' in attributes && attributes.styleId === null;
                    if (_.isObject(attributes.paragraph)) {
                        isAttributesInParagraph = ('lineHeight' in attributes.paragraph) || ('listLevel' in attributes.paragraph) || ('listStyleId' in attributes.paragraph);
                        isPageBreakBeforeAttributeInParagraph = ('pageBreakBefore' in attributes.paragraph);
                    }
                    if ((isCharOrDrawingOrRow || isClearFormatting || isAttributesInParagraph || isPageBreakBeforeAttributeInParagraph)) {
                        if (start.length > 1) {
                            // if its paragraph creation inside of table
                            currentElement = Position.getContentNodeElement(rootNode, start.slice(0, 1));
                        } else {
                            currentElement = Position.getContentNodeElement(rootNode, start);
                        }
                        if ($(currentElement).data('lineBreaksData')) {
                            $(currentElement).removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated
                        }
                        // attribute removed by user, needs to refresh page layout
                        if (isPageBreakBeforeAttributeInParagraph && !attributes.paragraph.pageBreakBefore) {
                            $(startInfo.node).removeClass('manual-page-break'); // remove class from div.p element
                            $(currentElement).removeClass('manual-page-break'); // remove class from parent table element also
                        }
                        insertPageBreaksDebounced(currentElement, DrawingFrame.getClosestTextFrameDrawingNode(startInfo.node));
                        if (!pbState) { app.getView().recalculateDocumentMargin(); }
                    }
                }

                return true;
            };
        }());

        /**
         * Inserts the passed content node at the specified logical position.
         *
         * @param {Number[]} position
         *  The logical position of the new content node.
         *
         * @param {HTMLElement|jQuery} node
         *  The new content node. If this object is a jQuery collection, uses
         *  the first node it contains.
         *
         * @param {String} target
         *  If exists, this string determines class name of container node.
         *  Usually is used for headers/footers
         *
         * @returns {Boolean}
         *  Whether the content node has been inserted successfully.
         */
        function insertContentNode(position, node, target) {

            var origPosition = _.clone(position),
                index = _.last(position),
                contentNode = null,
                insertBefore = true,
                insertAtEnd = false,
                parentNode = null,
                rootNode = null;

            function getContainerNode(position) {

                var // logical position of the paragraph container node
                    parentPosition = position.slice(0, -1),
                    // the container node for the paragraph
                    containerInfo = Position.getDOMPosition(rootNode, parentPosition, true),
                    // the parent container node
                    containerNode = null;

                // resolve component node to the correct node that contains the
                // child content nodes (e.g. resolve table cell elements to the
                // embedded div.cellcontent elements, or drawing elements to the
                // embedded div.content elements)
                if (containerInfo && containerInfo.node) {
                    containerNode = DOM.getChildContainerNode(containerInfo.node)[0];

                    // preparing text frame node, if this is a placeholder node inside a drawing frame of type 'shape'
                    // if (DrawingFrame.isPlaceHolderNode(containerNode) && DrawingFrame.isShapeDrawingFrame(containerInfo.node)) {
                    if (DrawingFrame.isShapeDrawingFrame(containerInfo.node)) {

                        // the child node can be a placeholder or an empty content node (empty during loading)
                        if (DrawingFrame.isPlaceHolderNode(containerNode) || DrawingFrame.isEmptyDrawingContentNode(containerNode)) {
                            // preparing text frame structure in drawing
                            containerNode = DrawingFrame.prepareDrawingFrameForTextInsertion(containerInfo.node);
                            // update formatting, otherwise styles (border, background, ...) will be lost
                            if (app.isImportFinished()) {
                                drawingStyles.updateElementFormatting(DrawingFrame.getDrawingNode(containerNode));
                            }
                            // #36327 - TF: Auto sizing for TF doesn't respect page size
                            containerNode.css('max-height', pageLayout.getDefPageActiveHeight({ convertToPixel: true }) - 15); // 15 px is rounding for borders and inner space of textframe
                        }
                    }
                }

                return containerNode;
            }

            // setting the correct root node dependent from an optional target
            rootNode = self.getRootNode(target);

            // trying to find existing paragraphs, tables or the parent element
            if (index > -1) {

                try {
                    contentNode = Position.getContentNodeElement(rootNode, position);
                } catch (e) {
                    contentNode = null;  // trying to repair in the following code
                }

                // content node does not exist -> maybe the new paragraph/table shall be added to the end of the document
                if (!contentNode) {

                    if (index === 0) {
                        // trying to find parent node, because this is the first paragraph/table in document or table cell or text frame or comment
                        parentNode = getContainerNode(position);

                        if (!parentNode) {
                            Utils.warn('Editor.insertContentNode(): cannot find parent node at position ' + JSON.stringify(position));
                            return false;
                        }
                        insertBefore = false;
                        insertAtEnd = true;

                    } else {
                        // Further try to find an existing element, behind that the new paragraph/table shall be appended
                        position[position.length - 1]--;
                        contentNode = Position.getContentNodeElement(rootNode, position, { allowImplicitParagraphs: false });

                        if (contentNode) {
                            insertBefore = false;
                        } else {
                            Utils.warn('Editor.insertContentNode(): cannot find content node at position ' + JSON.stringify(origPosition));
                            return false;
                        }
                    }
                }
            } else if (index === -1) { // Definition: Adding paragraph/table to the end if index is -1
                if (target) {
                    parentNode = rootNode;
                } else {
                    parentNode = getContainerNode(position);
                }
                if (!parentNode) {
                    Utils.warn('Editor.insertContentNode(): cannot find parent node at position ' + JSON.stringify(position));
                    return false;
                }
                insertBefore = false;
                insertAtEnd = true;
            } else {
                Utils.warn('Editor.insertContentNode(): invalid logical position ' + JSON.stringify(position));
                return false;
            }

            // modify the DOM
            if (insertAtEnd) {
                $(node).first().appendTo(parentNode);
            } else if (insertBefore) {
                $(node).first().insertBefore(contentNode);
            } else {
                $(node).first().insertAfter(contentNode);
            }

            return true;
        }

        /**
         * After splitting a paragraph with place holders in the new paragraph (that
         * was cloned before), it is necessary to update the place holder nodes in the
         * models. Also range marker and complex field models need to be updated.
         *
         * @param {Node|jQuery} para
         *  The paragraph node.
         */
        function updateAllModels(para) {

            var // whether the existence of the components in the specified node shall be checked
                noCheck = !DOM.isParagraphNode(para);

            // also update the links of the drawings in the drawing layer to their placeholder
            if (noCheck || DOM.hasDrawingPlaceHolderNode(para)) { drawingLayer.repairLinksToPageContent(para); }
            // also update the links of the comments in the comment layer to their placeholder
            if (noCheck || DOM.hasCommentPlaceHolderNode(para)) { commentLayer.repairLinksToPageContent(para); }
            // also update the collected range markers
            if (noCheck || DOM.hasRangeMarkerNode(para)) { rangeMarker.updateRangeMarkerCollector(para); }
            // also update the collected complex fields
            if (noCheck || DOM.hasComplexFieldNode(para)) { fieldManager.updateComplexFieldCollector(para); }
            // also update the collected simple fields
            if (noCheck || DOM.hasSimpleFieldNode(para)) { fieldManager.updateSimpleFieldCollector(para); }
        }

        /**
         * Splitting a paragraph at a specified logical position.
         *
         * @param {Number[]} position
         *  The logical position at which the paragraph shall be splitted.
         *
         * @returns {Boolean}
         *  Whether the paragraph has been splitted successfully.
         */
        function implSplitParagraph(position, target) {

            var // the last value of the position array
                offset = _.last(position),
                // the position of the 'old' paragraph
                paraPosition = position.slice(0, -1),
                // container root node of paragraph
                rootNode = self.getRootNode(target),
                // the 'old' paragraph node
                paragraph = (useParagraphCache && !target && paragraphCache) || Position.getParagraphElement(rootNode, paraPosition),
                // the position of the 'new' paragraph
                newParaPosition = Position.increaseLastIndex(paraPosition),
                // the 'new' paragraph node
                newParagraph = $(paragraph).clone(true).find('div.page-break, ' + DOM.DRAWING_SPACEMAKER_NODE_SELECTOR).remove().end().insertAfter(paragraph), //remove eventual page breaks before inserting to DOM
                // a specific position
                startPosition = null,
                // the length of the paragraph
                paraLength = Position.getParagraphNodeLength(paragraph),
                // whether the paragraph is splitted at the end
                isLastPosition = (paraLength === offset),
                // whether the paragraph contains floated drawings
                hasFloatedChildren = DOM.containsFloatingDrawingNode(paragraph),
                // Performance: Saving global paragraph cache
                paragraphCacheSafe = (useParagraphCache && paragraphCache) || null,
                // Performance: Saving data for list updates
                paraAttrs = null,
                // whether the split paragraph contains selected drawing(s)
                updateSelectedDrawing = false,
                // whether the document is in read-only mode
                readOnly = self.getEditMode() !== true,
                // the list label node inside a paragraph
                listLabelNode = null,
                // the current element
                currentElement,
                // the old drawing start position before splitting the paragraph
                drawingStartPosition = null, index = 0,
                // the last text span of the paragraph
                finalTextSpan = null;

            // checking if a selected drawing is affected by this split
            if (readOnly && selection.getSelectionType() === 'drawing') {
                // is the drawing inside the splitted paragraph and is the drawing behind the split?
                if (Utils.containsNode(paragraph, selection.getSelectedDrawing())) {
                    // split happens at position 'position'. Is this before the existing selection?
                    if (Utils.compareNumberArrays(position, selection.getStartPosition(), position.length) < 0) {
                        updateSelectedDrawing = true;
                        drawingStartPosition = _.clone(selection.getStartPosition());
                    }
                }
            }

            // caching current paragraph/table (if paragraph split inside table) for page break calculation
            currentElement = (position.length > 2) ? $(paragraph).parents('table').last() : $(paragraph);

            // Performance: If the paragraph is splitted at the end, the splitting process can be simplified
            if (isLastPosition && paraLength > 0 && DOM.isTextSpan(newParagraph.children(':last'))) {
                // the final textSpan must be reused to save character attributes
                finalTextSpan = newParagraph.children(':last').clone(true).text('');
                if (DOM.isListLabelNode(newParagraph.children(':first'))) { listLabelNode = newParagraph.children(':first'); }
                newParagraph.empty().end().prepend(finalTextSpan);
                DOM.ensureExistingTextNode(finalTextSpan);  // must be done after insertion into the DOM
                if (listLabelNode) { newParagraph.prepend(listLabelNode); }
                startPosition = Position.appendNewIndex(newParaPosition);
            } else {

                // delete trailing part of original paragraph
                if (offset !== -1) {
                    if (useParagraphCache) { paragraphCache = paragraph; } // Performance: implDeleteText can use cached paragraph
                    implDeleteText(position, Position.appendNewIndex(paraPosition, -1), { allowEmptyResult: true, keepDrawingLayer: true }, target);
                    if (hasFloatedChildren) {
                        // delete all image divs that are no longer associated with following floating drawings
                        Position.removeUnusedDrawingOffsetNodes(paragraph);
                    }
                    if (offset === 0) { // if split is at the beginning of paragraph, remove page breaks immediately for better user experience
                        $(paragraph).find('div.page-break').remove();
                        // handle place holder nodes for drawings and comment
                        updateAllModels(newParagraph);
                    }
                }

                // delete leading part of new paragraph
                startPosition = Position.appendNewIndex(newParaPosition);
                if (offset > 0) {
                    if (useParagraphCache) { paragraphCache = newParagraph; } // Performance: implDeleteText cannot use cached paragraph
                    implDeleteText(startPosition, Position.increaseLastIndex(startPosition, offset - 1), { keepDrawingLayer: true }, target);
                    if (hasFloatedChildren) {
                        // delete all empty text spans in cloned paragraph before floating drawings
                        // TODO: implDeleteText() should have done this already
                        Position.removeUnusedDrawingOffsetNodes(newParagraph);
                    }
                    // handle place holder nodes for drawings and comment
                    updateAllModels(newParagraph);
                }
                // drawings in the new paragraph need a repaint of the canvas border (also if offset is 0)
                self.trigger('drawingHeight:update', newParagraph.find(DrawingFrame.BORDER_NODE_SELECTOR));
            }

            // update formatting of the paragraphs
            implParagraphChanged(paragraph);
            implParagraphChanged(newParagraph);

            // invalidating an optional cache with spell results synchronously
            if (spellChecker.hasSpellResultCache(paragraph)) {
                spellChecker.clearSpellResultCache(paragraph);
                spellChecker.clearSpellResultCache(newParagraph);
            }

            if (useParagraphCache) { paragraphCache = paragraphCacheSafe; } // Performance: Restoring global paragraph cache

            // checking paragraph attributes for list styles
            if (handleTriggeringListUpdate(paragraph, { checkSplitInNumberedList: true })) {
                $(newParagraph).data('splitInNumberedList', 'true');
            }

            // checking paragraph attributes for list styles
            paraAttrs = AttributeUtils.getExplicitAttributes(paragraph);

            // !!!Notice - We do not send operation for removing this attribute, it is by default set to false for any paragraph in filter
            if (paraAttrs && paraAttrs.paragraph && paraAttrs.paragraph.pageBreakBefore && paraAttrs.paragraph.pageBreakBefore === true) {
                paragraphStyles.setElementAttributes(newParagraph, { paragraph: { pageBreakBefore: false } });
                newParagraph.removeClass('manual-page-break');
            }
            // !!!Notice - We do not send operation for removing this attribute, it is by default set to false for any paragraph in filter
            if (paraAttrs && paraAttrs.paragraph && paraAttrs.paragraph.pageBreakAfter && paraAttrs.paragraph.pageBreakAfter === true) {
                paragraphStyles.setElementAttributes(newParagraph, { paragraph: { pageBreakAfter: false } });
                newParagraph.removeClass('manual-pb-after');
            }

            // block page breaks render if operation is targeted
            if (target) {
                self.setBlockOnInsertPageBreaks(true);
            }

            if (!isLastPosition && currentElement.data('lineBreaksData')) {
                currentElement.removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated
            }
            if (newParagraph.data('lineBreaksData')) {
                newParagraph.removeData('lineBreaksData'); // for the cloned paragraph also
            }

            //quitFromPageBreak = true; // quit from any currently running pagebreak calculus - performance optimization
            insertPageBreaksDebounced(currentElement, DrawingFrame.getClosestTextFrameDrawingNode(paragraph));
            if (!pbState) { app.getView().recalculateDocumentMargin(); }

            lastOperationEnd = startPosition;

            // in Internet Explorer it is necessary to add new empty text nodes in paragraphs again
            // newly, also in other browsers (new jQuery version?)
            repairEmptyTextNodes(newParagraph);

            if (_.browser.IE) {
                repairEmptyTextNodes(paragraph); // also repair original paragraph in IE, #37652
            }

            // Performance: Saving paragraph info for following operations
            selection.setParagraphCache(newParagraph, _.clone(newParaPosition), 0);

            if (updateSelectedDrawing) {
                // selecting the previously selected drawing again
                index = position.length - 1;
                drawingStartPosition[index] = drawingStartPosition[index] - position[index];  // new text position inside paragraph
                drawingStartPosition[index - 1]++;  // new paragraph position
                selection.setTextSelection(drawingStartPosition, Position.increaseLastIndex(drawingStartPosition));
            }

            return true;
        }

        /**
         * Splitting a table at a specified logical position.
         *
         * @param {Number[]} position
         *  The logical position at which the table shall be splitted.
         *
         * @param {String[]} target
         *  ID of target root node for operation
         *
         * @returns {Boolean}
         *  Whether the table has been splitted successfully.
         */
        function implSplitTable(position, target) {
            var
                positionRowToSplit = _.last(position),
                // the position of the 'old' table
                tablePosition = position.slice(0, -1),
                // root node
                rootNode = self.getRootNode(target),
                // the 'old' table node
                tableNode = Position.getContentNodeElement(rootNode, tablePosition),
                // the position of the 'new' table
                //newTablePosition = Position.increaseLastIndex(tablePosition),
                newTableNode = $(tableNode).clone(true),
                // collection of first level table rows, excluding rows containing page break
                tableNodeRows = DOM.getTableRows(tableNode),
                // collection of first level new table rows, excluding rows containing page break
                newTableNodeRows = DOM.getTableRows(newTableNode).removeClass('break-above-tr');

            // if we clone table with manual page break, it should not be inherited in new table
            if (newTableNode.hasClass('manual-page-break')) {
                newTableNode.find(DOM.MANUAL_PAGE_BREAK_SELECTOR).removeClass('manual-page-break');
                newTableNode.removeClass('manual-page-break');
            }
            // remove ending rows from old table
            $(tableNodeRows[positionRowToSplit]).nextAll('tr').addBack().remove();
            // remove begining rows from new table
            $(newTableNodeRows[positionRowToSplit]).prevAll('tr').remove();
            newTableNode.insertAfter($(tableNode));

            implTableChanged(tableNode);
            implTableChanged(newTableNode);

            repairEmptyTextNodes(newTableNode, { allNodes: true }); // IE fix for empty table impl paragraphs after split, #34632

            // new cursor position at merge position
            lastOperationEnd = _.clone(Position.getFirstPositionInParagraph(rootNode, position));

            // run page break render if operation is not targeted
            if (target) {
                self.setBlockOnInsertPageBreaks(true);
            }

            insertPageBreaksDebounced(tableNode, DrawingFrame.getClosestTextFrameDrawingNode(tableNode));
            if (!pbState) { app.getView().recalculateDocumentMargin(); }

            return true;
        }

        /**
         * Merging two tables.
         *
         * @param {Object} operation - Contains name of the operation, start - the logical position
         *  of table from where merging should start, and optionaly target ID of root node
         *
         * @returns {Boolean}
         *  Whether the tables are merged successfully.
         */
        function implMergeTable(operation) {
            var
                // the position of the 'old' table
                tablePosition = operation.start,
                // currently active root node
                rootNode = self.getRootNode(operation.target),
                // the 'old' table node
                tableNode = Position.getContentNodeElement(rootNode, tablePosition),
                // new table node, next to old table
                nextTableNode = DOM.getAllowedNeighboringNode(tableNode, { next: true }),
                // cached collection of rows for next table
                nextTableNodeRows = DOM.getTableRows(nextTableNode).removeClass('break-above-tr'),
                // position of next table
                nextTablePosition = _.clone(tablePosition),
                // information for table split, if undo is called:
                // position of last row inside table
                rowPosInTable,
                // helper for stored value of row position in iteration
                rowPosition,
                // helper for stored value of cell position in iteration
                cellPosition,
                // logical position of table and its last row
                tableAndRowPosition,
                //generator of operations
                generator = undoManager.isUndoEnabled() ? self.createOperationsGenerator() : null,
                // created operation
                newOperation = null;

            nextTablePosition[nextTablePosition.length - 1] += 1;
            // calculate last row position from current table, used for undo table split operation
            rowPosInTable = Position.getLastRowIndexInTable(rootNode, tablePosition);
            if (rowPosInTable > -1) {
                // increased by 1 at the end, because split starts from above of passed row
                rowPosInTable += 1;
            } else { // invalid value
                rowPosInTable = 0;
            }
            tableAndRowPosition = Position.appendNewIndex(tablePosition, rowPosInTable);

            if (DOM.isTableNode(nextTableNode)) {
                if (generator) {
                    newOperation = { start: tableAndRowPosition };
                    extendPropertiesWithTarget(newOperation, operation.target);
                    generator.generateOperation(Operations.TABLE_SPLIT, newOperation);
                    generator.generateSetAttributesOperation(nextTableNode, { start: nextTablePosition }); // generate undo operation to set attributes to table
                    //generate set attributes for rows and cells
                    _.each(nextTableNodeRows, function (row, index) {
                        rowPosition = _.clone(nextTablePosition);
                        rowPosition.push(index);
                        generator.generateSetAttributesOperation(row, { start: rowPosition });
                        _.each(row.cells, function (cell, index) {
                            cellPosition = _.clone(rowPosition);
                            cellPosition.push(index);
                            generator.generateSetAttributesOperation(cell, { start: cellPosition });
                        });
                    });
                    undoManager.addUndo(generator.getOperations(), operation);
                }

                $(tableNode).find('> tbody').last().append($(nextTableNodeRows));
                $(nextTableNode).remove();

                implTableChanged(tableNode);

                // new cursor position at merge position
                lastOperationEnd = _.clone(Position.getFirstPositionInParagraph(rootNode, tablePosition));

                // run page break render if operation is not targeted
                if (operation.target) {
                    self.setBlockOnInsertPageBreaks(true);
                }

                insertPageBreaksDebounced(tableNode, DrawingFrame.getClosestTextFrameDrawingNode(tableNode));
                if (!pbState) { app.getView().recalculateDocumentMargin(); }

                return true;
            }
        }

        /**
         * Returns a valid text position at the passed component position
         * (paragraph or table). Used to calculate the new text cursor position
         * after deleting a component.
         */
        function getValidTextPosition(position) {
            var
                // container root node of table
                rootNode = self.getCurrentRootNode(),
                textPosition = Position.getFirstPositionInParagraph(rootNode, position);
            // fall-back to last position in document (e.g.: last table deleted)
            if (!textPosition) {
                textPosition = Position.getLastPositionInParagraph(rootNode, [rootNode[0].childNodes.length - 1]);
            }
            return textPosition;
        }

        /**
         * Deletes a table at the specified position.
         *
         * @param {Number[]} position
         *  The logical position of the table to be deleted.
         *
         * @param {String} target
         *  Id of the container node, where operation is applied.
         *
         * @returns {Boolean}
         *  TRUE if the function has been processed successfully
         *  otherwise FALSE.
         */
        function implDeleteTable(position, target) {

            var
                rootNode = self.getRootNode(target),
                tablePosition = Position.getLastPositionFromPositionByNodeName(rootNode, position, DOM.TABLE_NODE_SELECTOR),
                tableNode = Position.getTableElement(rootNode, tablePosition);

            if (tableNode) {
                $(tableNode).remove();
                lastOperationEnd = getValidTextPosition(tablePosition);
            } else {
                Utils.warning('Editor.implDeleteTable(): not tableNode found ' + JSON.stringify(position));
                return false;
            }

            // the deleted paragraphs can be part of a list, update all lists
            if (elementsContainListStyleParagraph(tableNode)) {
                updateListsDebounced({ useSelectedListStyleIDs: true, listStyleId: getListStyleIDsOfParagraphs(tableNode), listLevel: null });
            }

            return true;
        }

        function implDeleteRows(pos, startRow, endRow, target) {

            var localPosition = _.copy(pos, true),
                rootNode = self.getRootNode(target);

            if (!Position.isPositionInTable(rootNode, localPosition)) {
                Utils.warning('Editor.implDeleteRows(): position not in table ' + JSON.stringify(pos));
                return false;
            }

            var table = Position.getDOMPosition(rootNode, localPosition).node,
                rowNodes = DOM.getTableRows(table).slice(startRow, endRow + 1);

            rowNodes.remove();

            if (guiTriggeredOperation) {  // called from 'deleteRows' -> temporary branding for table
                $(table).data('gui', 'remove');
                // Performance: Simple tables do not need new formatting.
                // Tables with styles need to be updated below the removed row. Also the
                // row above need to be updated, because it may have become the final row.
                $(table).data('reducedTableFormatting', startRow > 0 ? startRow - 1 : 0);
            }

            // undo also needs table refresh for right border in Firefox (32374)
            if (undoRedoRunning  && _.browser.Firefox) { $(table).data('undoRedoRunning', true); }

            // Setting cursor
            var lastRow = Table.getRowCount(table) - 1;

            if (lastRow >= 0) {
                if (endRow > lastRow) {
                    endRow = lastRow;
                }
                localPosition.push(endRow);
                localPosition.push(0);  // using first cell in row
                localPosition.push(0);  // using first paragraph or table
                localPosition = Position.getFirstPositionInParagraph(rootNode, localPosition);

                // recalculating the attributes of the table cells
                if (requiresElementFormattingUpdate) {
                    implTableChanged(table);
                }

                lastOperationEnd = localPosition;
            }

            // the deleted paragraphs can be part of a list, update all lists
            if (elementsContainListStyleParagraph(rowNodes)) {
                updateListsDebounced({ useSelectedListStyleIDs: true, listStyleId: getListStyleIDsOfParagraphs(rowNodes), listLevel: null });
            }

            return true;
        }

        /**
         * Removing all cell contents
         */
        function implClearCell(cellNode) {

            var // the container for the content nodes
                container = DOM.getCellContentNode(cellNode),
                // the last paragraph in the cell
                paragraph = null;

            paragraph = DOM.createImplicitParagraphNode();
            container.empty().append(paragraph);
            // in Internet Explorer it is necessary to add new empty text nodes in paragraph again
            // newly, also in other browsers (new jQuery version?)
            repairEmptyTextNodes(paragraph);

            // validate the paragraph (add the dummy node)
            validateParagraphNode(paragraph);
            // and NOT formatting the implicit paragraph -> leads to 'jumping' of table after inserting row or column
            // paragraphStyles.updateElementFormatting(paragraph);
        }

        function implInsertRows(start, count, insertDefaultCells, referenceRow, attrs, target) {

            var localPosition = _.copy(start, true),
                useReferenceRow = _.isNumber(referenceRow) ? true : false,
                newRow = null,
                currentElement,
                rootNode = self.getRootNode(target);

            currentElement = Position.getContentNodeElement(editdiv, start.slice(0, 1));

            if (!Position.isPositionInTable(rootNode, localPosition)) {
                return false;
            }

            insertDefaultCells = insertDefaultCells ? true : false;

            if (!_.isNumber(count)) {
                count = 1; // setting default for number of rows
            }

            var tablePos = _.copy(localPosition, true);
            tablePos.pop();

            var table = Position.getDOMPosition(rootNode, tablePos).node,
                tableRowDomPos = Position.getDOMPosition(rootNode, localPosition),
                tableRowNode = null,
                row = null,
                cellsInserted = false;

            if (guiTriggeredOperation) {  // called from 'insertRow' -> temporary branding for table
                $(table).data('gui', 'insert');
            }

            if (tableRowDomPos) {
                tableRowNode = tableRowDomPos.node;
            }

            if (useReferenceRow) {

                if (guiTriggeredOperation) {
                    // Performance: Called from 'insertRow' -> simple table do not need new formatting.
                    // Tables with conditional styles only need to be updated below the referenceRow
                    // (and the row itself, too, for example southEast cell may have changed)
                    $(table).data('reducedTableFormatting', referenceRow);
                }

                row = DOM.getTableRows(table).eq(referenceRow).clone(true);

                // clear the cell contents in the cloned row
                row.children('td').each(function () {
                    implClearCell(this);
                });

                cellsInserted = true;

            } else if (insertDefaultCells) {

                var columnCount = Table.getColumnCount(table),
                    // prototype elements for row, cell, and paragraph
                    paragraph = DOM.createImplicitParagraphNode(),
                    cell = DOM.createTableCellNode(paragraph);

                // insert empty text node into the paragraph
                validateParagraphNode(paragraph);

                row = $('<tr>').attr('role', 'row');

                // clone the cells in the row element
                _.times(columnCount, function () { row.append(cell.clone(true).attr('role', 'gridcell')); });

                cellsInserted = true;

            } else {
                row = $('<tr>').attr('role', 'row');
            }

            _.times(count, function () {
                newRow = row.clone(true);

                // apply the passed attributes
                if (_.isObject(attrs)) {
                    tableRowStyles.setElementAttributes(newRow, attrs);
                }

                if (tableRowNode) {
                    // insert the new row before the existing row at the specified position
                    $(tableRowNode).before(newRow);
                } else {
                    // append the new row to the table
                    $(table).append(newRow);
                }

                // in Internet Explorer it is necessary to add new empty text nodes in rows again
                // newly, also in other browsers (new jQuery version?)
                repairEmptyTextNodes(newRow);

            });

            // recalculating the attributes of the table cells
            if (cellsInserted && requiresElementFormattingUpdate) {
                implTableChanged(table);
            }

            // Setting cursor
            if ((insertDefaultCells) || (useReferenceRow)) {
                localPosition.push(0);
                localPosition.push(0);
                localPosition.push(0);

                lastOperationEnd = localPosition;
            }

            // if operation is not targeted, render page breaks
            if (target) {
                self.setBlockOnInsertPageBreaks(true);
            }

            quitFromPageBreak = true; // quit from any currently running pagebreak calculus - performance optimization
            insertPageBreaksDebounced(currentElement, DrawingFrame.getClosestTextFrameDrawingNode(table));
            if (!pbState) { app.getView().recalculateDocumentMargin(); }

            return true;
        }

        function implInsertCells(start, count, attrs, target) {

            var localPosition = _.clone(start),
                tableNode,
                tableCellDomPos = null,
                tableCellNode = null,
                paragraph = null,
                cell = null,
                row = null,
                rootNode = self.getRootNode(target);

            tableNode = Position.getLastNodeFromPositionByNodeName(rootNode, start, DOM.TABLE_NODE_SELECTOR);

            if (!tableNode) {
                return;
            }

            if (!_.isNumber(count)) {
                count = 1; // setting default for number of rows
            }

            tableCellDomPos = Position.getDOMPosition(rootNode, localPosition);

            if (tableCellDomPos) {
                tableCellNode = tableCellDomPos.node;
            }

            // prototype elements for row, cell, and paragraph
            paragraph = DOM.createImplicitParagraphNode();

            // insert empty text node into the paragraph
            validateParagraphNode(paragraph);

            cell = DOM.createTableCellNode(paragraph);

            // apply the passed table cell attributes
            if (_.isObject(attrs)) {
                tableCellStyles.setElementAttributes(cell, attrs);
            }

            if (tableCellNode) {
                _.times(count, function () { $(tableCellNode).before(cell.clone(true)); });
                row = $(tableCellNode).parent();
            } else {
                var rowPos = localPosition.slice(0, -1);
                row = Position.getDOMPosition(rootNode, rowPos).node;
                _.times(count, function () { $(row).append(cell.clone(true)); });
            }

            // setting cursor to first paragraph in the cell
            localPosition.push(0);
            localPosition.push(0);
            lastOperationEnd = localPosition;

            // in Internet Explorer it is necessary to add new empty text nodes in rows again
            // newly, also in other browsers (new jQuery version?)
            repairEmptyTextNodes(row);

            // recalculating the attributes of the table cells
            if (requiresElementFormattingUpdate) {
                implTableChanged(tableNode);
            }

            return true;
        }

        /**
         * Delete table cells at the specified position.
         *
         * @param {Number[]} pos
         *  The logical position of the table to be deleted.
         *
         * @param start
         *  The logical start position of the cell range.
         *
         * @param end
         *  The logical end position of the cell range.
         *
         * @param {String} target
         *  Id of the container node, where operation is applied.
         *
         * @returns {Boolean}
         *  TRUE if the function has been processed successfully,
         *  otherwise FALSE.
         */
        function implDeleteCells(pos, start, end, target) {

            var localPosition = _.copy(pos, true),
                tableRowDomPos = null,
                row = null,
                table = null,
                maxCell = 0,
                cellNodes = null,
                allCellNodes = [],
                rootNode = self.getRootNode(target);

            if (!Position.isPositionInTable(rootNode, localPosition)) {
                Utils.warn('Editor.implDeleteCells(): position not in table ' + JSON.stringify(pos));
                return false;
            }

            tableRowDomPos = Position.getDOMPosition(rootNode, localPosition);

            if (tableRowDomPos) {
                row = tableRowDomPos.node;
            }

            if (row) {

                maxCell = $(row).children().length - 1;

                if (start <= maxCell) {

                    if (end > maxCell) {
                        cellNodes = $(row).children().slice(start);

                    } else {
                        cellNodes = $(row).children().slice(start, end + 1);
                    }

                    cellNodes.remove(); // removing all following cells
                    allCellNodes.push(cellNodes);
                }
            }

            // setting cursor position
            localPosition.push(0);
            localPosition.push(0);
            localPosition.push(0);

            if (row && requiresElementFormattingUpdate) {
                table = $(row).closest('table');
                // undo of insertColumn also needs table refresh for right border in Firefox
                if (undoRedoRunning  && _.browser.Firefox) { table.data('undoRedoRunning', true); }
                implTableChanged(table);
            }

            lastOperationEnd = localPosition;

            // the deleted paragraphs can be part of a list, update all lists
            if (elementsContainListStyleParagraph(allCellNodes)) {
                updateListsDebounced({ useSelectedListStyleIDs: true, listStyleId: getListStyleIDsOfParagraphs(allCellNodes), listLevel: null });
            }

            return true;
        }

        function implDeleteColumns(start, startGrid, endGrid, target) {

            var localPosition = _.copy(start, true),
                cellNodes = null,
                allCellNodes = [],
                rootNode = self.getRootNode(target);

            if (!Position.isPositionInTable(rootNode, localPosition)) {
                return;
            }

            var table = Position.getDOMPosition(rootNode, localPosition).node,
                allRows = DOM.getTableRows(table),
                endColInFirstRow = -1,
                lastColInFirstRow = 0;

            if (guiTriggeredOperation) {  // called from 'deleteColumns' -> temporary branding for table
                $(table).data('gui', 'remove');
            }

            allRows.each(
                function (i, row) {
                    var startCol = Table.getCellPositionFromGridPosition(row, startGrid, false),
                        endCol = Table.getCellPositionFromGridPosition(row, endGrid, false);

                    if ((i === 0) && (endCol !== -1)) {
                        endColInFirstRow = endCol;
                    }

                    if (startCol !== -1) {  // do nothing if startCol is out of range for this row

                        if (endCol === -1) {
                            // checking whether undo of operations is possible and remove cells
                            cellNodes = $(row).children().slice(startCol);
                        } else {
                            // checking whether undo of operations is possible and remove all cells in the range
                            cellNodes = $(row).children().slice(startCol, endCol + 1);
                        }

                        cellNodes.each(function (i, cell) {
                            checkDisableUndoStack('cell', cell);
                            // updating all models, for drawings, comments, range markers, ...
                            self.updateCollectionModels(cell);
                        }).remove();  // removing cell nodes
                        allCellNodes.push(cellNodes);
                    }
                }
            );

            // Setting cursor
            lastColInFirstRow = DOM.getTableRows(table).first().children().length - 1;

            if ((endColInFirstRow > lastColInFirstRow) || (endColInFirstRow === -1)) {
                endColInFirstRow = lastColInFirstRow;
            }
            localPosition.push(0);
            localPosition.push(endColInFirstRow);
            localPosition.push(0);
            localPosition.push(0);

            // delete undo stack immediately if this is necessary and not a part of an undo group
            if (!isInUndoGroup && deleteUndoStack) {
                undoManager.clearUndoActions();
                deleteUndoStack = false;
            }

            // recalculating the attributes of the table cells
            if (requiresElementFormattingUpdate) {
                implTableChanged(table);
            }

            // the deleted paragraphs can be part of a list, update all lists
            if (elementsContainListStyleParagraph(allCellNodes)) {
                updateListsDebounced({ useSelectedListStyleIDs: true, listStyleId: getListStyleIDsOfParagraphs(allCellNodes), listLevel: null });
            }

            lastOperationEnd = localPosition;
        }

        function implInsertColumn(start, gridPosition, insertMode, target) {

            var localPosition = _.copy(start, true),
                rootNode = self.getRootNode(target);

            if (!Position.isPositionInTable(rootNode, localPosition)) {
                return false;
            }

            var table = Position.getDOMPosition(rootNode, localPosition).node,
                allRows = DOM.getTableRows(table);

            if (guiTriggeredOperation) {  // called from 'insertColumn' -> temporary branding for table
                $(table).data('gui', 'insert');
            }

            allRows.each(
                function (i, row) {
                    var cellPosition = Table.getCellPositionFromGridPosition(row, gridPosition),
                        cellClone = $(row).children(DOM.TABLE_CELLNODE_SELECTOR).slice(cellPosition, cellPosition + 1).clone(true);

                    implClearCell(cellClone);

                    if (insertMode === 'behind') {
                        cellClone.insertAfter($(row).children().get(cellPosition));
                    } else {
                        cellClone.insertBefore($(row).children().get(cellPosition));
                    }

                    // in Internet Explorer it is necessary to add new empty text nodes in columns again
                    // newly, also in other browsers (new jQuery version?)
                    repairEmptyTextNodes(cellClone);
                }
            );

            // recalculating the attributes of the table cells
            if (requiresElementFormattingUpdate) {
                implTableChanged(table);
            }

            // Setting cursor to first position in table
            localPosition.push(0);
            localPosition.push(gridPosition);
            localPosition.push(0);
            localPosition.push(0);

            lastOperationEnd = localPosition;

            return true;
        }

        function implDeleteText(startPosition, endPosition, options, target) {

            var // info about the parent paragraph node
                position = null, paragraph = null,
                // last index in start and end position
                startOffset = 0, endOffset = 0,
                // next sibling text span of last visited child node
                nextTextSpan = null,
                // a counter for the removed elements
                removeCounter = 0,
                // the number of text positions to be removed
                removePositions = 0,
                // in internal calls (from other operations except delete), it can be allowed,
                // that there is nothing to delete
                allowEmptyResult = Utils.getBooleanOption(options, 'allowEmptyResult', false),
                // whether the drawings in the drawing layer or comments in the comment layer shall not be removed
                // -> this is necessary after splitting a paragraph
                keepDrawingLayer = Utils.getBooleanOption(options, 'keepDrawingLayer', false),
                // whether a character was deleted within this function
                noCharDeleted = false,
                // root node used for calculating position - if target is present - root node is header or footer
                rootNode = self.getRootNode(target);

            // get paragraph node from start position
            if (!_.isArray(startPosition) || (startPosition.length < 2)) {
                Utils.warn('Editor.implDeleteText(): missing start position');
                return false;
            }
            position = startPosition.slice(0, -1);
            paragraph = (useParagraphCache && paragraphCache) || Position.getParagraphElement(rootNode, position);
            if (!paragraph) {
                Utils.warn('Editor.implDeleteText(): no paragraph found at position ' + JSON.stringify(position));
                return false;
            }

            // validate end position
            if (_.isArray(endPosition) && !Position.hasSameParentComponent(startPosition, endPosition)) {
                Utils.warn('Editor.implDeleteText(): range not in same paragraph');
                return false;
            }

            // start and end offset in paragraph
            startOffset = _.last(startPosition);
            endOffset = _.isArray(endPosition) ? _.last(endPosition) : startOffset;
            if (endOffset === -1) { endOffset = undefined; }
            if (endOffset !== undefined) {
                removePositions = endOffset - startOffset + 1;
            }

            // visit all covered child nodes (iterator allows to remove the visited node)
            Position.iterateParagraphChildNodes(paragraph, function (node, nodeStart, nodeLength) {

                var // previous text span of current node
                    prevTextSpan = null;

                // remove preceding position offset node of floating drawing objects
                if (DOM.isFloatingDrawingNode(node)) {
                    $(node).prev(DOM.OFFSET_NODE_SELECTOR).remove();
                }

                // get sibling text spans
                prevTextSpan = DOM.isTextSpan(node.previousSibling) ? node.previousSibling : null;
                nextTextSpan = DOM.isTextSpan(node.nextSibling) ? node.nextSibling : null;

                // fix for msie, where empty text spans migth be removed
                // newly, also in other browsers (new jQuery version?)
                if (DOM.isTextSpanWithoutTextNode(node)) {
                    repairEmptyTextNodes(node);
                }
                if ((!prevTextSpan) && (DOM.isTextSpanWithoutTextNode(node.previousSibling))) {
                    prevTextSpan = node.previousSibling;
                    repairEmptyTextNodes(prevTextSpan);
                }
                if ((!nextTextSpan) && (DOM.isTextSpanWithoutTextNode(node.nextSibling))) {
                    nextTextSpan = node.previousSibling;
                    repairEmptyTextNodes(nextTextSpan);
                }

                // clear text in text spans
                if (DOM.isTextSpan(node)) {

                    // only remove the text span, if it has a sibling text span
                    // (otherwise, it separates other component nodes)
                    if (prevTextSpan || nextTextSpan) {
                        $(node).remove();
                    } else {
                        // remove text, but keep text span element
                        node.firstChild.nodeValue = '';
                        // but remove change track information immediately (38791)
                        if (DOM.isChangeTrackNode(node)) { changeTrack.updateChangeTrackAttributes($(node), {}); }
                    }
                    removeCounter += nodeLength;
                    return;
                }

                // other component nodes (drawings or text components)
                if (DOM.isTextComponentNode(node) || DrawingFrame.isDrawingFrame(node) || DOM.isCommentPlaceHolderNode(node) || DOM.isRangeMarkerNode(node)) {

                    // if we are removing manualy inserted page break by user, trigger the pagebreak recalculation on document
                    if (DOM.isTextComponentNode(node) && $(node.parentNode).hasClass('manual-page-break')) {
                        $(node.parentNode).removeClass('manual-page-break');
                        insertPageBreaksDebounced(node.parentNode, DrawingFrame.getClosestTextFrameDrawingNode(node.parentNode));
                    }

                    // node is the image place holder -> remove also the drawing from the drawing layer
                    if (DOM.isDrawingPlaceHolderNode(node)) {
                        drawingLayer.removeFromDrawingLayer(node, { keepDrawingLayer: keepDrawingLayer });
                    } else if (DOM.isCommentPlaceHolderNode(node)) {
                        commentLayer.removeFromCommentLayer(node, { keepDrawingLayer: keepDrawingLayer });
                    } else if (DOM.isRangeMarkerNode(node) && !keepDrawingLayer) {
                        rangeMarker.removeRangeMarker(node);
                    } else if (DOM.isComplexFieldNode(node) && !keepDrawingLayer) {
                        fieldManager.removeFromComplexFieldCollection(node);
                    } else if (DOM.isFieldNode(node)) {
                        fieldManager.removeSimpleFieldFromCollection(node);
                    }

                    // safely release image data (see comments in BaseApplication.destroyImageNodes())
                    app.destroyImageNodes(node);

                    // remove the visited node
                    $(node).remove();

                    // remove previous empty sibling text span (not next sibling which would break iteration)
                    if (DOM.isEmptySpan(prevTextSpan) && nextTextSpan) {
                        $(prevTextSpan).remove();
                    }
                    removeCounter += nodeLength;
                    return;
                }

                Utils.error('Editor.implDeleteText(): unknown paragraph child node');
                return Utils.BREAK;

            }, undefined, { start: startOffset, end: endOffset, split: true });

            // removed must be > 0 and it must be the difference from endOffset - startOffset + 1
            // check, if at least one element was removed
            if (!allowEmptyResult && (removeCounter === 0)) {
                Utils.warn('Editor.implDeleteText(): no component found from position: ' + JSON.stringify(startPosition) + ' to ' + JSON.stringify(endPosition));
                return false;
            }

            // check, if the removed part has the correct length
            if (!allowEmptyResult && (removePositions > 0) && (removePositions !== removeCounter)) {
                Utils.warn('Editor.implDeleteText(): incorrect number of removals: Required: ' + removeCounter + ' Done: ' + removeCounter);
                return false;
            }

            // remove next sibling text span after deleted range, if empty,
            // otherwise try to merge with equally formatted preceding text span
            if (nextTextSpan && DOM.isTextSpan(nextTextSpan.previousSibling)) {
                if (DOM.isEmptySpan(nextTextSpan)) {
                    $(nextTextSpan).remove();
                } else {
                    Utils.mergeSiblingTextSpans(nextTextSpan);
                }
            }

            // validate paragraph node, store operation position for cursor position
            if ((guiTriggeredOperation) || (undoRedoRunning) || (!app.isImportFinished())) {
                implParagraphChanged(paragraph);  // Performance and task 30597: Local client can defer attribute setting, Task 30603: deferring also during loading document
            } else {
                implParagraphChangedSync($(paragraph));  // Performance and task 30597: Remote client must set attributes synchronously
            }

            lastOperationEnd = _.clone(startPosition);
            if (noCharDeleted) {
                if (_.isArray(endPosition) && !_.isEqual(startPosition, endPosition)) { lastOperationEnd = _.clone(endPosition); }
                lastOperationEnd[lastOperationEnd.length - 1] += 1;
            }

            return true;
        }

        /**
         * Creates and inserts Header or footer with passes id and type
         *
         * @param {String} id
         * @param {String} type
         * @param {Object} attrs
         *
         * @returns {Boolean}
         */
        function implInsertHeaderFooter(id, type, attrs) {
            pageLayout.implCreateHeaderFooter(id, type, attrs, undoRedoRunning);

            return true;
        }

        /**
         * Deletes and removes from DOM node with passed id
         *
         * @param {String} id
         *
         * @returns {Boolean}
         */
        function implDeleteHeaderFooter(id) {
            return pageLayout.implDeleteHeaderFooter(id);
        }

        /**
         * Checking whether a node contains an element, that cannot be restored
         * in undo action. This are tables with exceeded size or drawings that
         * are not of type 'image'.
         *
         * @param {Node|jQuery|Number[]} node
         *  The node, whose descendants are investigated. For convenience reasons
         *  this can also be a logical position that is transformed into a node
         *  automatically.
         *
         * @param {Number} [startOffset]
         *  An optional offset that can be used to reduce the search range inside
         *  the specified node.
         *
         * @param {Number} [endOffset]
         *  An optional offset that can be used to reduce the search range inside
         *  the specified node.
         *
         * @returns {Boolean}
         *  Returns true, if the node contains an unrestorable descendant,
         *  otherwise false.
         */
        function containsUnrestorableElements(node, startOffset, endOffset) {

            var // whether the node contains an unrestorable descendant
                contains = false,
                // a dom point calculated from logical position
                domPos = null,
                // a locally defined helper node
                searchNode = null,
                // the logical position of a non-image drawing
                drawingPos = null;

            // supporting for convenience also logical positions, that are automatically
            // converted into dom nodes
            if (_.isArray(node)) {
                domPos = Position.getDOMPosition(self.getCurrentRootNode(), node, true);
                if (domPos && domPos.node) { searchNode = domPos.node; }
            } else {
                searchNode = node;
            }

            if (searchNode) {
                if ($(searchNode).find(DOM.TABLE_SIZE_EXCEEDED_NODE_SELECTOR).length > 0) {
                    contains = true;
                } else {
                    // checking if the table or paragraph contains drawings, that are not of type 'image'
                    // or if the table contains another table with exceeded size
                    $(DrawingFrame.NODE_SELECTOR + ', ' + DOM.DRAWINGPLACEHOLDER_NODE_SELECTOR, searchNode).each(function (i, drawing) {

                        if (DOM.isDrawingPlaceHolderNode(drawing)) { drawing = DOM.getDrawingPlaceHolderNode(drawing); }

                        if (DOM.isUnrestorableDrawingNode(drawing)) {
                            // if only a part of a paragraph is deleted, it has to be checked,
                            // if the non-image and non-textframe drawing is inside this selected part.
                            if (_.isNumber(startOffset) && (_.isNumber(endOffset))) {
                                // the logical position of the non-image drawing
                                drawingPos = Position.getOxoPosition(self.getCurrentRootNode(), drawing, 0);
                                if (Position.isNodePositionInsideRange(drawingPos, startOffset, endOffset)) {
                                    contains = true;
                                }
                            } else {
                                contains = true;
                            }
                        }
                    });
                }
                // if node contains special page number fields in header and footer restore original state before delete
                fieldManager.checkRestoringSpecialFields(searchNode, startOffset, endOffset);
            }

            return contains;
        }

        /**
         * Checking whether there can be an undo for the delete operation.
         * Undo is not possible, if a table with exceeded size is removed,
         * or if a drawing, that is not of type image is removed.
         *
         * @param {String} type
         *  The type of the node to be removed. This can be 'text', 'paragraph',
         *  'cell', 'row' and table.
         *
         * @param {Node|jQuery} node
         *  The node to be removed.
         *
         * @param {Number[]} start
         *  The logical start position of the element or text range to be
         *  deleted.
         *
         * @param {Number[]} [end]
         *  The logical end position of the element or text range to be
         *  deleted. This is only relevant for type 'text'.
         *
         * @param {String[]} [target]
         *  If element is marginal, target is needed in group with
         *  start and end to determine root container.
         */
        function checkDisableUndoStack(type, node, start, end, target) {

            var // info about the parent paragraph or table node
                position = null, paragraph = null,
                // last index in start and end position
                startOffset = 0, endOffset = 0;

            // deleting the different types:
            switch (type) {

            case 'text':
                // get paragraph node from start position
                if (!_.isArray(start) || (start.length === 0)) {
                    Utils.warn('Editor.disableUndoStack(): missing start position');
                    return false;
                }
                position = start.slice(0, -1);
                paragraph = Position.getParagraphElement(self.getCurrentRootNode(target), position);
                if (!paragraph) {
                    Utils.warn('Editor.disableUndoStack(): no paragraph found at position ' + JSON.stringify(position));
                    return false;
                }

                // validate end position
                if (_.isArray(end) && !Position.hasSameParentComponent(start, end)) {
                    Utils.warn('Editor.disableUndoStack(): range not in same paragraph');
                    return false;
                }

                // start and end offset in paragraph
                startOffset = _.last(start);
                endOffset = _.isArray(end) ? _.last(end) : startOffset;
                if (endOffset === -1) { endOffset = undefined; }

                // visit all covered child nodes (iterator allows to remove the visited node)
                Position.iterateParagraphChildNodes(paragraph, function (localnode) {

                    if (DOM.isDrawingPlaceHolderNode(localnode)) { localnode = DOM.getDrawingPlaceHolderNode(localnode); }

                    if (DOM.isUnrestorableDrawingNode(localnode)) {
                        deleteUndoStack = true;
                    }
                }, undefined, { start: startOffset, end: endOffset });
                break;

            case 'paragraph':
                if (containsUnrestorableElements(node)) {
                    deleteUndoStack = true;
                }
                break;

            case 'cell':
                if (containsUnrestorableElements(node)) {
                    deleteUndoStack = true;
                }
                break;

            case 'row':
                if (containsUnrestorableElements(node)) {
                    deleteUndoStack = true;
                }
                break;

            case 'table':
                if (DOM.isExceededSizeTableNode(node) || containsUnrestorableElements(node)) {
                    deleteUndoStack = true;
                }
                break;
            }
        }

        /**
         * Removes a specified element or a text range. The type of the element
         * will be determined from the parameters start and end.
         * specified range. Currently, only single components can be deleted,
         * except for text ranges in a paragraph. A text range can include
         * characters, fields, and drawing objects, but must be contained in a
         * single paragraph.
         *
         * @param {Number[]} start
         *  The logical start position of the element or text range to be
         *  deleted.
         *
         * @param {Number[]} [end]
         *  The logical end position of the element or text range to be
         *  deleted. Can be omitted, if the end position is equal to the start
         *  position (single component)
         *
         * @param {String} target - optional
         *  If target is existing, rootNode is not editdiv, but currently edited header/footer
         *
         *  @return {Boolean}
         *   TRUE if the function has been processed successfully, otherwise
         *   FALSE.
         */
        function implDelete(start, end, target) {

            var // node info for start/end position
                startInfo = null, endInfo = null,
                // position description for cells
                rowPosition, startCell, endCell,
                // position description for rows
                tablePosition, startRow, endRow,
                // a new implicit paragraph node
                newParagraph = null,
                // a temporary helper position
                localPosition = null,
                // result of function (default true)
                result = true,
                // starting point for inserting page breaks downwards
                position = start.length > 1 ? start.slice(0, 1) : start.slice(0),
                // used for pagebreaks
                currentElement = null,
                // the height of the paragraph before deleting content
                prevHeight = null,
                // the space maker node and the paragraph containing the space maker node
                spaceMakerNode = null, spaceMakerParagraph = null,
                // a top level drawing inside the drawing layer
                topLevelDrawing = null,
                // whether content in the drawing layer in a text frame was removed
                isDrawingLayerText = false,
                // a text frame node, that might contain the start position
                textFrameNode = null,
                // container element with target id
                rootNode = self.getRootNode(target);

            // resolve start and end position
            if (!_.isArray(start)) {
                Utils.warn('Editor.implDelete(): missing start position');
                return false;
            }
            startInfo = Position.getDOMPosition(rootNode, start, true);
            if (!startInfo || !startInfo.node) {
                Utils.warn('Editor.implDelete(): invalid start position: ' + JSON.stringify(start));
                return false;
            }
            endInfo = _.isArray(end) ? Position.getDOMPosition(rootNode, end, true) : startInfo;
            if (!endInfo || !endInfo.node) {
                Utils.warn('Editor.implDelete(): invalid end position: ' + JSON.stringify(end));
                return false;
            }

            end = end || start;

            // get attribute type of start and end node
            startInfo.type = resolveElementType(startInfo.node);
            endInfo.type = resolveElementType(endInfo.node);

            // check that start and end point to the same element type
            if ((!startInfo.type || !endInfo.type) || (startInfo.type !== endInfo.type)) {
                Utils.warn('Editor.implDelete(): problem with node types: ' + startInfo.type + ' and ' + endInfo.type);
                return false;
            }

            // check that start and end point to the same element for non text types (only one cell, row, paragraph, ...)
            if ((startInfo.type !== 'text') && (startInfo.node !== endInfo.node)) {
                Utils.warn('Editor.implDelete(): no ranges supported for node type "' + startInfo.type + '"');
                return false;
            }

            // check that for text nodes start and end point have the same parent
            if ((startInfo.type === 'text') && (startInfo.node.parentNode !== endInfo.node.parentNode)) {
                if (DOM.isDrawingLayerNode(endInfo.node.parentNode)) {
                    if (startInfo.node.parentNode !== DOM.getDrawingPlaceHolderNode(endInfo.node).parentNode) {
                        Utils.warn('Editor.implDelete(): deleting range only supported inside one paragraph.');
                        return false;
                    }
                } else if (DOM.isDrawingLayerNode(startInfo.node.parentNode)) {
                    if (endInfo.node.parentNode !== DOM.getDrawingPlaceHolderNode(startInfo.node).parentNode) {
                        Utils.warn('Editor.implDelete(): deleting range only supported inside one paragraph.');
                        return false;
                    }
                } else {
                    Utils.warn('Editor.implDelete(): deleting range only supported inside one paragraph.');
                    return false;
                }
            }

            // checking whether undo of operations is possible
            checkDisableUndoStack(startInfo.type, startInfo.node, _.clone(start), _.clone(end), target);

            // check for a text frame node
            textFrameNode = DrawingFrame.getClosestTextFrameDrawingNode(startInfo.node.parentNode);

            // deleting the different types:
            switch (startInfo.type) {

            case 'text':
                var isLastCharInPar = !startInfo.node.nextSibling && (startInfo.offset === startInfo.node.textContent.length - 1);
                // setting the paragraph node (required for page breaks)
                currentElement = startInfo.node.parentNode;
                if (DOM.isDrawingLayerNode(currentElement)) { currentElement = DOM.getDrawingPlaceHolderNode(startInfo.node).parentNode; }
                if (!$(currentElement.parentNode).hasClass('pagecontent')) {
                    if (DrawingFrame.isTextFrameNode(currentElement.parentNode)) {
                        // is the text frame inside a drawing in the drawing layer?
                        if (DOM.isInsideDrawingLayerNode(currentElement)) {
                            isDrawingLayerText = true;
                            // measuring height changes at the current paragraph (that is already currentElement
                            // But for the top level measurement the paragraph containing the space maker node is required
                            topLevelDrawing = DOM.getTopLevelDrawingInDrawingLayerNode(currentElement);
                            spaceMakerNode = DOM.getDrawingSpaceMakerNode(topLevelDrawing);
                            if (DOM.getDrawingSpaceMakerNode(topLevelDrawing)) { spaceMakerParagraph = $(spaceMakerNode).parentsUntil(self.getNode(), DOM.PARAGRAPH_NODE_SELECTOR).last()[0]; }
                            // div.textframe height needs to be measured synchronously
                            currentElement = currentElement.parentNode;
                        } else if (DOM.isTopLevelNodeInComment(currentElement)) {
                            // comments are outside of the main document. Therefore it is only necessary to check the paragraph
                            // inside the comment, not the paragraph containing the placeholder for the comment.
                            currentElement = currentElement.parentNode;
                        } else {
                            currentElement = $(currentElement).parentsUntil(self.getNode(), DOM.PARAGRAPH_NODE_SELECTOR).last()[0];  // its a div.p inside paragraph
                        }
                    } else if (target) {
                        if (!DOM.isMarginalContentNode(currentElement.parentNode)) {
                            currentElement = $(currentElement).parents(DOM.TABLE_NODE_SELECTOR).last()[0]; //its a div.p inside table(s)
                        }
                        // TODO parent DrawingFrame
                    } else {
                        currentElement = $(currentElement).parentsUntil(self.getNode(), DOM.TABLE_NODE_SELECTOR).last()[0]; //its a div.p inside table(s)
                    }
                }
                // caching paragraph height before removing text
                prevHeight = currentElement ? currentElement.offsetHeight : 0;
                // removing text
                result = implDeleteText(start, end, {}, target);

                // block page breaks render if operation is targeted
                if (target) {
                    self.setBlockOnInsertPageBreaks(true);
                }

                // using the same paragraph node again
                if ((!isDrawingLayerText) && ($(currentElement).data('lineBreaksData'))) {
                    $(currentElement).removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated
                }
                quitFromPageBreak = true; // quit from any currently running pagebreak calculus - performance optimization
                // trigger repainting only if height changed or element is on two pages,
                // or paragraph is on more pages, and last char in paragraph is not deleted (performance improvement)
                if (prevHeight !== currentElement.offsetHeight || ($(currentElement).hasClass('contains-pagebreak')) && !isLastCharInPar) {
                    insertPageBreaksDebounced(isDrawingLayerText ? spaceMakerParagraph : currentElement, textFrameNode);
                    if (!pbState) { app.getView().recalculateDocumentMargin(); }
                }

                break;

            case 'paragraph':
                if (DOM.isImplicitParagraphNode(startInfo.node)) {
                    Utils.warn('Editor.implDelete(): Error: Operation tries to delete an implicit paragraph!');
                    return false;
                }
                // used to check if next neighbour is table, and after deletion, selection has to be updated
                var nextNodeNeighbour = $(startInfo.node).next();

                // Setting lastOperationEnd for the special case, that insertParagraph was used instead of splitParagraph
                // -> if there is a previous paragraph, use the last position of this previous paragraph (task 30742)
                if (($(startInfo.node).prev().length > 0) && (DOM.isParagraphNode($(startInfo.node).prev()))) {
                    localPosition = _.clone(start);
                    localPosition[localPosition.length - 1]--;
                    localPosition = Position.getLastPositionInParagraph(rootNode, localPosition);
                    lastOperationEnd = localPosition;
                } else if ($(startInfo.node).prev().length === 0) {
                    // special case: Removing the first paragraph (undo-redo handling)
                    localPosition = _.clone(start);
                    localPosition = Position.getFirstPositionInParagraph(rootNode, localPosition);
                    lastOperationEnd = localPosition;
                }

                if ((DOM.isParagraphWithoutNeighbour(startInfo.node)) || (DOM.isFinalParagraphBehindTable(startInfo.node))) {
                    newParagraph = DOM.createImplicitParagraphNode();
                    $(startInfo.node).parent().append(newParagraph);
                    validateParagraphNode(newParagraph);
                    // if original paragraph was marked as marginal, mark newly created implicit as marginal, too
                    if (DOM.isMarginalNode(startInfo.node)) {
                        $(newParagraph).addClass(DOM.MARGINAL_NODE_CLASSNAME);
                    }
                    implParagraphChanged(newParagraph);
                    // in Internet Explorer it is necessary to add new empty text nodes in paragraph again
                    // newly, also in other browsers (new jQuery version?)
                    repairEmptyTextNodes(newParagraph);

                }

                // checking, if the paragraph contains drawing place holder nodes, comment place holder nodes or range
                // marker nodes. In this case all models need to be updated. Furthermore the comments and the drawings
                // in their layers need to be removed, too.
                self.updateCollectionModels(startInfo.node);

                $(startInfo.node).remove(); // remove the paragraph from the DOM
                // the deleted paragraphs can be part of a list, update all lists
                handleTriggeringListUpdate(startInfo.node);

                if (DOM.isTableNode(nextNodeNeighbour)) { // #34685 properly set cursor inside table after delete paragraph
                    selection.setTextSelection(Position.getFirstPositionInParagraph(rootNode, start), null, { simpleTextSelection: false, splitOperation: false });
                }

                // block page breaks render if operation is targeted
                if (target) {
                    self.setBlockOnInsertPageBreaks(true);
                }

                // get currentElement again (maybe whole paragraph has deleted), and pass it to repaint pagebreaks from that page only
                currentElement = Position.getContentNodeElement(rootNode, position);
                insertPageBreaksDebounced(currentElement, textFrameNode);
                if (!pbState) { app.getView().recalculateDocumentMargin(); }

                break;

            case 'cell':

                // checking for absolute drawings, comments and range markers
                self.updateCollectionModels(startInfo.node);

                rowPosition = _.clone(start);
                startCell = rowPosition.pop();
                endCell = startCell;
                result = implDeleteCells(rowPosition, startCell, endCell, target);
                break;

            case 'row':

                // checking for absolute drawings, comments and range markers
                self.updateCollectionModels(startInfo.node);

                tablePosition = _.clone(start);
                startRow = tablePosition.pop();
                endRow = startRow;
                result = implDeleteRows(tablePosition, startRow, endRow, target);

                // block page breaks render if operation is targeted
                if (target) {
                    self.setBlockOnInsertPageBreaks(true);
                }

                // get currentElement again (maybe whole paragraph has deleted), and pass it to repaint pagebreaks from that page only
                currentElement = Position.getContentNodeElement(rootNode, position);
                insertPageBreaksDebounced(currentElement, textFrameNode);
                if (!pbState) { app.getView().recalculateDocumentMargin(); }

                break;

            case 'table':

                // checking for absolute drawings, comments and range markers
                self.updateCollectionModels(startInfo.node);

                result = implDeleteTable(start, target);

                // block page breaks render if operation is targeted
                if (target) {
                    // trigger header/footer content update on other elements of same type, if change was made inside header/footer
                    updateEditingHeaderFooterDebounced(self.getRootNode(target));
                    self.setBlockOnInsertPageBreaks(true);
                }

                // get currentElement again (maybe whole paragraph has deleted), and pass it to repaint pagebreaks from that page only
                currentElement = Position.getContentNodeElement(rootNode, position);
                insertPageBreaksDebounced(currentElement, textFrameNode);
                if (!pbState) { app.getView().recalculateDocumentMargin(); }

                break;

            default:
                Utils.error('Editor.implDelete(): unsupported node type: ' + startInfo.type);
            }

            // delete undo stack immediately if this is necessary and not a part of an undo group
            if (!isInUndoGroup && deleteUndoStack) {
                undoManager.clearUndoActions();
                deleteUndoStack = false;
            }

            return result;
        }

        function implMove(_start, _end, _to, target) {

            var start = _.copy(_start, true),
                to = _.copy(_to, true),
                activeRootNode = self.getRootNode(target),
                sourcePos = Position.getDOMPosition(activeRootNode, start, true),
                destPos = Position.getDOMPosition(activeRootNode, to, true),
                insertBefore = true,
                splitNode = false,
                // a text span before the moved node before moving the node
                prevTextSpan = 0;

            // Fix for 28634 -> Moving a drawing to position with tab or hard break
            if (DOM.isHardBreakNode(destPos.node) || DOM.isTabNode(destPos.node) || DOM.isFieldNode(destPos.node)) {
                destPos.node = destPos.node.previousSibling;
            } else if (DrawingFrame.isDrawingFrame(destPos.node) && DOM.isDrawingLayerNode(destPos.node.parentNode)) {
                destPos.node = DOM.getDrawingPlaceHolderNode(destPos.node).previousSibling;
            }

            if (destPos.offset === 0) {
                insertBefore = true;
            } else if ((destPos.node.length) && (destPos.offset === (destPos.node.length - 1))) {
                insertBefore = false;
            } else if ((DrawingFrame.isDrawingFrame(destPos.node)) && (destPos.offset === 1)) {
                insertBefore = false;
            } else {
                splitNode = true;  // splitting node is required
                insertBefore = false;  // inserting after new created text node
            }

            if ((sourcePos) && (destPos)) {

                var sourceNode = sourcePos.node,
                    destNode = destPos.node,
                    useOffsetDiv = true,
                    offsetDiv = sourceNode.previousSibling,
                    doMove = true;

                if ((sourceNode) && (destNode)) {

                    if (!DrawingFrame.isDrawingFrame(sourceNode)) {
                        doMove = false; // supporting only drawings at the moment
                        Utils.warn('Editor.implMove(): moved  node is not a drawing: ' + Utils.getNodeName(sourceNode));
                    } else {
                        // also move the offset divs
                        if ((!offsetDiv) || (!DOM.isOffsetNode(offsetDiv))) {
                            useOffsetDiv = false;
                        }
                    }

                    if (doMove) {

                        if (splitNode) {
                            destNode = DOM.splitTextSpan(destNode, destPos.offset + 1)[0];
                        } else {
                            if (destNode.nodeType === 3) {
                                destNode = destNode.parentNode;
                            }
                        }

                        // using empty span as reference for inserting new components
                        if ((DrawingFrame.isDrawingFrame(destNode)) && (DOM.isOffsetNode(destNode.previousSibling))) {
                            destNode = destNode.previousSibling;  // switching temporary to offset
                        }

                        // there can be empty text spans before the destination node
                        if ((DOM.isTextSpan(destNode)) || (DrawingFrame.isDrawingFrame(destNode)) || (DOM.isOffsetNode(destNode))) {
                            while (DOM.isEmptySpan(destNode.previousSibling)) {
                                destNode = destNode.previousSibling;
                            }
                        }

                        if ((insertBefore) && (DOM.isTextSpan(destNode))) {
                            destNode = DOM.splitTextSpan(destNode, 0)[0]; // taking care of empty text span before drawing
                            insertBefore = false;  // insert drawing behind new empty text span
                        }

                        // removing empty text spans behind or after the source node
                        if ((sourceNode.previousSibling) && (sourceNode.nextSibling)) {
                            if ((DOM.isTextSpan(sourceNode.previousSibling)) && (DOM.isEmptySpan(sourceNode.nextSibling))) {
                                $(sourceNode.nextSibling).remove();
                            } else if ((DOM.isEmptySpan(sourceNode.previousSibling)) && (DOM.isTextSpan(sourceNode.nextSibling))) {
                                $(sourceNode.previousSibling).remove();
                            }
                        }

                        if ((sourceNode.previousSibling) && (sourceNode.previousSibling.previousSibling) && (sourceNode.nextSibling) && (DOM.isOffsetNode(sourceNode.previousSibling))) {
                            if ((DOM.isTextSpan(sourceNode.previousSibling.previousSibling)) && (DOM.isEmptySpan(sourceNode.nextSibling))) {
                                $(sourceNode.nextSibling).remove();
                            } else if ((DOM.isEmptySpan(sourceNode.previousSibling.previousSibling)) && (DOM.isTextSpan(sourceNode.nextSibling))) {
                                $(sourceNode.previousSibling.previousSibling).remove();
                            }
                        }

                        // saving the old previous text span for later merge
                        if (sourceNode.previousSibling && DOM.isTextSpan(sourceNode.previousSibling)) { prevTextSpan = sourceNode.previousSibling; }

                        // moving the drawing
                        if (insertBefore) {
                            $(sourceNode).insertBefore(destNode);
                        } else {
                            $(sourceNode).insertAfter(destNode);
                        }

                        // moving also the corresponding div before the moved drawing
                        if (useOffsetDiv) {
                            $(offsetDiv).insertBefore(sourceNode);
                        }

                        // merging text spans, if possible
                        if (prevTextSpan) { Utils.mergeSiblingTextSpans(prevTextSpan, true); }

                        implParagraphChanged(to);
                    }
                }
            }

            return true;
        }

        function implMergeCell(start, count, target) {

            var rowPosition = _.copy(start, true),
                localStartCol = rowPosition.pop(),
                localEndCol = localStartCol + count,
                rootNode = self.getRootNode(target),
                // Counting the colSpan off all cells in the range
                row = Position.getDOMPosition(rootNode, rowPosition).node,
                allSelectedCells = $(row).children().slice(localStartCol, localEndCol + 1),
                colSpanSum = Table.getColSpanSum(allSelectedCells),
                // Shifting the content of all following cells to the first cell
                targetCell = $(row).children().slice(localStartCol, localStartCol + 1),
                sourceCells = $(row).children().slice(localStartCol + 1, localEndCol + 1);

            Table.shiftCellContent(targetCell, sourceCells);

            sourceCells.remove();

            // apply the passed table cell attributes
            tableCellStyles.setElementAttributes(targetCell, { cell: { gridSpan: colSpanSum } });

            return true;
        }

        /**
         * modify current listlevel - if any - depending on the increase flag
         */
        function implModifyListLevel(currentlistLevel, increase, minLevel) {
            undoManager.enterUndoGroup(function () {
                if (increase && currentlistLevel < 8) {
                    currentlistLevel += 1;
                    self.setAttribute('paragraph', 'listLevel', currentlistLevel);
                } else if (!increase && currentlistLevel > minLevel) {
                    currentlistLevel -= 1;
                    self.setAttribute('paragraph', 'listLevel', currentlistLevel);
                    if (currentlistLevel < 0) {
                        self.setAttribute('paragraph', 'listStyleId', null);
                    }
                }
            });
        }

        /**
         * Updates all paragraphs that are part of any bullet or numbering
         * list.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.async=false]
         *      If set to true, all lists are updated asynchronously. This
         *      should happen only once when the document is loaded.
         *
         * @returns {jQuery.Promise}
         *  A promise that is resolved, when all paragraphs are updated.
         */
        function updateLists(options) {

            var listItemCounter = [],
                listParagraphIndex = [], // paragraph index in a list
                // list of all paragraphs in the document
                paragraphNodes = null,
                // the deferred, whose promise will be returned
                def = null,
                // Performance: Whether this list update was triggered by an action with specified style id
                useSelectedListStyleIDs = updateListsDebouncedOptions.useSelectedListStyleIDs,
                // Performance: Whether there are list styles registered for updating
                doNothing = false,
                // Performance: An array with listStyleIds of splitted paragraphs
                allListStyleIds = null,  // Performance: Only for splitted paragraphs
                // Performance: An array with listStyleLevels of splitted paragraphs (same order as in listStyleIds)
                // allListStyleLevels = null;
                // Performance: If only paragraphs with numbered lists were splitted in the registration phase,
                // it is sufficient to update all paragraphs starting from the one paragraph that was marked
                // with 'splitInNumberedList'.
                splitInNumberedList = updateListsDebouncedOptions.splitInNumberedList,
                // whether the splitted paragraph in the numbered list was found
                splittedNumberedListFound = false,
                // whether DOM manipulations can be suppressed. This is the case for numbered
                suppressDomManipulation = false;

            function updateListInParagraph(para) {

                // always remove an existing label
                var elementAttributes = paragraphStyles.getElementAttributes(para),
                    paraAttributes = elementAttributes.paragraph,
                    listStyleId = paraAttributes.listStyleId,
                    listLevel = paraAttributes.listLevel,
                    oldLabel = null,
                    updateParaTabstops = false,
                    removeLabel = false,
                    updateList = false,
                    // an array with all list style ids, that share the common base (and are counted together (40792))
                    listFamily = listStyleId !== '' ? listCollection.getAllListStylesWithSameBaseStyle(listStyleId) : null,
                    // using the first list id of a list family as counter (inside listItemCounter and listParagraphIndex)
                    listCounterId = (listFamily && listFamily.length > 1) ? listFamily[0] : listStyleId;

                // Updating paragraphs, that are no longer part of lists (for example after 'Backspace')
                if ($(para).data('removeLabel')) {
                    removeLabel = true;
                    $(para).removeData('removeLabel');
                }

                if ($(para).data('updateList')) {
                    updateList = true;
                    $(para).removeData('updateList');
                }

                if ($(para).data('splitInNumberedList')) {  // might be set at more than one paragraph
                    splittedNumberedListFound = true;
                    $(para).removeData('splitInNumberedList');
                }

                // Performance II: Do nothing, if the list style ID does not fit -> only modify paragraphs with correct list style ID
                // -> could be improved with check of list level (taking care of styles including upper levels)
                if (useSelectedListStyleIDs && !_.contains(allListStyleIds, listStyleId) && (!listFamily || !_.contains(listFamily, listStyleId)) && !removeLabel && !updateList) { return; }

                // Performance III: No DOM manipulation, if the split happened in a numbered list and the marked
                // paragraph (marked with 'splitInNumberedList') was not found yet. Only for following paragraphs
                // the DOM manipulations are necessary. But the number counting is necessary for all paragraphs.
                suppressDomManipulation = (splitInNumberedList && !splittedNumberedListFound);

                if (!suppressDomManipulation) {
                    oldLabel = $(para).children(DOM.LIST_LABEL_NODE_SELECTOR);
                    updateParaTabstops = oldLabel.length > 0;
                    oldLabel.remove();
                }

                if (listStyleId !== '' || listLevel) {
                    var noListLabel = paraAttributes.listLabelHidden === true;
                    if (listLevel < 0) {
                        // is a numbering level assigned to the current paragraph style?
                        listLevel = listCollection.findIlvl(listStyleId, elementAttributes.styleId);
                    }
                    if (listLevel !== -1 && listLevel < 10) {
                        if (!listItemCounter[listCounterId]) {
                            listItemCounter[listCounterId] = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0];
                            listParagraphIndex[listCounterId] = 0;
                        }
                        if (!noListLabel) {
                            listItemCounter[listCounterId][listLevel]++;
                            listParagraphIndex[listCounterId]++;
                        }
                        if (paraAttributes.listStartValue >= 0) {
                            listItemCounter[listCounterId][listLevel] = paraAttributes.listStartValue;
                        }
                        // TODO: reset sub-levels depending on their 'levelRestartValue' attribute
                        var subLevelIdx = listLevel + 1;
                        for (; subLevelIdx < 10; subLevelIdx++) {
                            listItemCounter[listCounterId][subLevelIdx] = 0;
                        }
                        // fix level counts of non-existing upper levels
                        subLevelIdx = listLevel - 1;
                        for (; subLevelIdx >= 0; subLevelIdx--) {
                            if (listItemCounter[listCounterId][subLevelIdx] === 0) {
                                listItemCounter[listCounterId][subLevelIdx] = 1;
                            }
                        }

                        // Performance III: After updating the listItemCounter and listParagraphIndex, the following
                        // DOM manipulations can be ignored, if 'suppressDomManipulation' is set to true. This is the
                        // case for numbered lists in which the added paragraph marked with 'splitInNumberedList' was
                        // not found yet.
                        if (suppressDomManipulation) { return; }

                        updateParaTabstops = true;

                        var listObject = listCollection.formatNumber(listCounterId, listLevel, listItemCounter[listCounterId], listParagraphIndex[listCounterId]),
                            tab = !listObject.labelFollowedBy || listObject.labelFollowedBy === 'listtab';

                        if (!tab && (listObject.labelFollowedBy === 'space')) {
                            listObject.text += String.fromCharCode(0x00a0);//add non breaking space
                        }

                        var numberingElement = DOM.createListLabelNode(noListLabel ? '' : listObject.text);

                        var span = DOM.findFirstPortionSpan(para);
                        var elementCharAttributes = characterStyles.getElementAttributes(span).character;
                        var listSpans = numberingElement.children('span');
                        if (listObject.imgsrc && !noListLabel) {
                            var absUrl = app.getServerModuleUrl(IO.FILTER_MODULE_NAME, { action: 'getfile', get_filename: listObject.imgsrc }),
                            imageWidth =  Utils.convertHmmToLength(listObject.imgwidth, 'pt');
                            if (!imageWidth) {
                                imageWidth = elementCharAttributes.fontSize;
                            }

                            var image = $('<div>', { contenteditable: false })
                            .addClass('drawing')
                            .data('url', listObject.imgsrc)
                            .append($('<div>').addClass('content')
                                .append($('<img>', { src: absUrl }).css('width', imageWidth + 'pt'))
                            );

                            characterStyles.updateElementLineHeight(image, paraAttributes.lineHeight, elementCharAttributes);
                            $(image).css('height', elementCharAttributes.fontSize + 'pt');
                            $(image).css('width', elementCharAttributes.fontSize + 'pt');
                            numberingElement.prepend(image);

                        } else if (elementAttributes.character || elementAttributes.styleId) {
                            var paraCharStyles = paragraphStyles.getStyleAttributeSet(elementAttributes.styleId);
                            // Map certain character styles from the paragraph style to the label ignoring others
                            // which doesn't fit this special use case. We prefer paragraph character attributes over
                            // style character attributes.
                            // Don't map background color and underline as MS Word also don't support them for
                            // list-labels.
                            if (elementAttributes.character.fontSize || paraCharStyles.character.fontSize) {
                                listSpans.css('font-size', (elementAttributes.character.fontSize || paraCharStyles.character.fontSize) + 'pt');
                            }
                            if (elementAttributes.character.fontName || paraCharStyles.character.fontName) {
                                listSpans.css('font-family', self.getCssFontFamily(elementAttributes.character.fontName || paraCharStyles.character.fontName));
                            }
                            if (elementAttributes.character.bold || paraCharStyles.character.bold) {
                                listSpans.css('font-weight', (elementAttributes.character.bold || paraCharStyles.character.bold) ? 'bold' : 'normal');
                            }
                            if (elementAttributes.character.italic || paraCharStyles.character.italic) {
                                listSpans.css('font-style', (elementAttributes.character.italic || paraCharStyles.character.italic) ? 'italic' : 'normal');
                            }

                            listSpans.css('text-decoration', self.getCssTextDecoration(elementAttributes.character));

                            if (elementAttributes.character.color || paraCharStyles.character.color) {
                                listSpans.css('color', self.getCssColor((elementAttributes.character.color || paraCharStyles.character.color), 'text'));
                            }
                            if (paraCharStyles.character.fillColor) {
                                listSpans.css('background-color', self.getCssColor(paraCharStyles.character.fillColor, 'fill'));
                            }
                        }

                        if (listObject.color) {
                            listSpans.css('color', self.getCssTextColor(listObject.color, [paraAttributes.fillColor, listObject.fillColor]));
                        }
                        characterStyles.updateElementLineHeight(numberingElement, paraAttributes.lineHeight, elementAttributes.character);
                        var minWidth = 0,
                            isNegativeIndent = listObject.firstLine < listObject.indent;

                        if (isNegativeIndent) {
                            var labelWidth = listObject.indent - listObject.firstLine;
                            if (tab) {
                                minWidth = labelWidth;
                            }
                            numberingElement.css('margin-left', (-listObject.indent + listObject.firstLine) / 100 + 'mm');
                        } else {
                            numberingElement.css('margin-left', (listObject.firstLine - listObject.indent) / 100 + 'mm');
                        }
                        numberingElement.css('min-width', minWidth / 100 + 'mm');

                        // #35649 - MS Word pagebreak inside list title not positioned correctly
                        if (DOM.isManualPageBreakNode($(para)) && DOM.hasMSPageHardbreakNode($(para)) && $(para).find(DOM.PAGE_BREAK_SELECTOR).length > 0) {
                            $(para).find(DOM.PAGE_BREAK_SELECTOR).first().after(numberingElement);
                        } else {
                            $(para).prepend(numberingElement);
                        }

                        if (tab || noListLabel) {
                            var realWidth = Utils.convertLengthToHmm(numberingElement[0].offsetWidth, 'px'),
                                realEndPos = listObject.firstLine + realWidth,
                                defaultTabstop = self.getDocumentAttributes().defaultTabStop,
                                paraAttributes = paragraphStyles.getElementAttributes(para).paragraph,
                                targetPosition = 0,
                                minWidthValue = 0;

                            if (isNegativeIndent && listObject.indent >= realEndPos) {
                                targetPosition = listObject.indent;
                            } else {
                                var tabstop = null;
                                if (paraAttributes && paraAttributes.tabStops) {
                                    tabstop = _.find(paraAttributes.tabStops, function (tab) {
                                        return realEndPos <= tab.pos;
                                    });
                                }
                                if (tabstop) {
                                    targetPosition = tabstop.pos;
                                } else if (defaultTabstop > 0) {
                                    // using factor '0.95' to avoid ugly large list-labels, if they are not really required (40792)
                                    targetPosition = (1 + (Math.floor((0.95 * realEndPos) / defaultTabstop))) * defaultTabstop;
                                } else {
                                    targetPosition = realEndPos;
                                }
                            }

                            minWidthValue = (targetPosition - listObject.firstLine);
                            minWidthValue = minWidthValue ? minWidthValue : 0;
                            if (minWidthValue > 0) { numberingElement.css('min-width', minWidthValue / 100 + 'mm'); }
                        }

                        // in draft mode, margins of list paragraphs have to be converted from mm to %
                        if (Utils.SMALL_DEVICE && para.style.marginLeft.slice(-1) !== '%') {
                            var ratio = 100  / (pageWidth - (2 * pagePaddingLeft));
                            $(para).css('margin-left', (ratio * parseInt($(para).css('margin-left'), 10)) + '%');
                            $(para).data('draftRatio', ratio);
                        }
                    }
                }

                if (updateParaTabstops) {
                    paragraphStyles.updateTabStops(para);
                }
            }

            // receiving list of all document paragraphs
            paragraphNodes = self.getCurrentRootNode().find(DOM.PARAGRAPH_NODE_SELECTOR);

            // Performance I: Simplified process, if list update was triggered by splitting or deleting of paragraphs
            // Only check those paragraphs, that have a list label node.
            if (useSelectedListStyleIDs) {
                // Reducing the number of paragraphs, that need to be updated. This can happen now before it is checked,
                // if the paragraph has a list style ID set. Affected are only paragraphs, that contain already the
                // list label node (DOM.LIST_LABEL_NODE_SELECTOR), or, if they are newly inserted, they are branded
                // with the 'updateList' or 'removeLabel' (after backspace) data.
                if (updateListsDebouncedOptions.paraInsert) {
                    paragraphNodes = paragraphNodes.filter(function () {
                        return (DOM.isListLabelNode(this.firstChild) || $(this).data('updateList'));
                    });
                } else {
                    paragraphNodes = paragraphNodes.filter(function () {
                        return DOM.isListLabelNode(this.firstChild);
                    });
                }

                allListStyleIds = updateListsDebouncedOptions.listStyleIds;

                // Update of bullet lists can be completely ignored, except new paragraphs were inserted
                if (!updateListsDebouncedOptions.paraInsert) {
                    allListStyleIds = _.reject(allListStyleIds, function (styleID) {
                        return listCollection.isAllLevelsBulletsList(styleID);
                    });
                }

                // If the list is empty, there is nothing to do
                doNothing = _.isEmpty(allListStyleIds);
            }

            if (doNothing) {
                def = $.when();
            } else if (Utils.getBooleanOption(options, 'async', false)) {
                def = self.iterateArraySliced(paragraphNodes, updateListInParagraph, { delay: 'immediate', infoString: 'Text: updateListInParagraph' });
            } else {
                paragraphNodes.each(function () { updateListInParagraph(this); });
                def = $.when();
            }

            // enabling new registration for debounced list update
            updateListsDebouncedOptions = {};

            return def.promise();
        }

        /**
         * Returns the value of a specified paragraph attribute of a given
         * paragraph style.
         *
         * @param {String} styleId
         *  The paragraph style id, that will be checked checked.
         *
         * @param {String} paraAttr
         *  The attribute of the paragraph attributes of the style, whose value
         *  shall be returned.
         *
         * @returns {String}
         *  The value of the specified paragraph attribute of the specified style.
         */
        function getListStyleInfoFromStyleId(styleId, paraAttr) {

            var // the style attributes
                styleAttrs = paragraphStyles.getStyleSheetAttributeMap(styleId);

            return (styleAttrs && styleAttrs.paragraph && styleAttrs.paragraph[paraAttr]);
        }

        /**
         * Check, wheter a given paragraph style id is adding list style information
         * to a paragraph (bullet or numbering list).
         *
         * @param {String} styleId
         *  The paragraph style id, that will be checked for a list style id.
         *
         * @returns {Boolean}
         *  Whether the specified styleId contains a list style id.
         */
        function isParagraphStyleWithListStyle(styleId) {

            var // the style attributes
                styleAttrs = paragraphStyles.getStyleSheetAttributeMap(styleId);

            return (styleAttrs && styleAttrs.paragraph && (styleAttrs.paragraph.listStyleId || styleAttrs.paragraph.listLevel));
        }

        /**
         * Check, wheter a given paragraph node describes a paragraph that
         * is part of a list (bullet or numbering list).
         *
         * @param {HTMLElement|jQuery} node
         *  The paragraph whose attributes will be checked. If this object is a
         *  jQuery collection, uses the first DOM node it contains.
         *
         * @param {Object} [attributes]
         *  An optional map of attribute maps (name/value pairs), keyed by attribute.
         *  It this parameter is defined, the parameter 'paragraph' can be omitted.
         *
         * @returns {Boolean}
         *  Whether the content node is a list paragraph node.
         */
        function isListStyleParagraph(paragraph, attributes) {

            var // the attributes directly assigned to the paragraph
                paraAttrs = attributes || AttributeUtils.getExplicitAttributes(paragraph);

            // shortcut, not handling style attributes
            if (paraAttrs && paraAttrs.paragraph && paraAttrs.paragraph.listStyleId) { return true; }

            // checking paragraph style
            if (attributes && attributes.styleId && isParagraphStyleWithListStyle(attributes.styleId)) { return true; }

            return false;
        }

        /**
         * Returning the list style id from specified paragraph attributes. The list
         * style can be set directly or via a style id at the paragraph.
         *
         * @param {Object} [attributes]
         *  An optional map of attribute maps (name/value pairs), keyed by attribute.
         *  It this parameter is defined, the parameter 'paragraph' can be omitted.
         *
         * @returns {String|null}
         *  The list style id or null, if no list style defined in specified attributes.
         */
        function getListStyleIdFromParaAttrs(attributes) {

            // shortcut, not handling style attributes
            if (attributes && attributes.paragraph && attributes.paragraph.listStyleId) { return attributes.paragraph.listStyleId; }

            // checking paragraph style
            if (attributes && attributes.styleId && isParagraphStyleWithListStyle(attributes.styleId)) { return getListStyleInfoFromStyleId(attributes.styleId, 'listStyleId'); }

            return null;
        }

        /**
         * Check, wheter a given collection of nodes contain paragraphs with list
         * styles.
         *
         * @param {Array|HTMLElement|jQuery} nodes
         *  The collection of nodes, that will be checked for paragraphs with list styles.
         *  This can be an array with html nodes or jQuery nodes. It can also be a html node
         *  or a jQuery object not included into an array. Above all nodes is iterated.
         *
         * @returns {Boolean}
         *  Whether at least one of the specified nodes contains at least one list paragraph node.
         */
        function elementsContainListStyleParagraph(nodes) {

            var isListParagraph = false;

            if ((!nodes) || (_.isArray(nodes) && _.isEmpty(nodes))) { return isListParagraph; }

            _.chain(nodes).getArray().any(function (oneNode) {
                $(oneNode).each(function (index, node) {
                    $(node).find(DOM.PARAGRAPH_NODE_SELECTOR).each(function (i, paragraph) {
                        if (isListStyleParagraph(paragraph)) {
                            isListParagraph = true;
                            return !isListParagraph; // stopping 'each' iteration, if list paragraph was found
                        }
                    });
                    return !isListParagraph; // stopping 'each' iteration, if list paragraph was found
                });
                return isListParagraph; // stopping 'any' iteration, if list paragraph was found
            });

            return isListParagraph;
        }

        /**
         * Collecting all list style IDs of the paragraphs inside the specified elements.
         *
         * @param {Array|HTMLElement|jQuery} nodes
         *  The collection of nodes, that will be checked for paragraphs with list styles.
         *  This can be an array with html nodes or jQuery nodes. It can also be a html node
         *  or a jQuery object not included into an array. Above all nodes is iterated.
         *
         * @returns {[Array]}
         *  An array containing the list style IDs of the paragraphs inside the specified elements.
         */
        function getListStyleIDsOfParagraphs(nodes) {

            var listStyleIDs = null;

            if ((!nodes) || (_.isArray(nodes) && _.isEmpty(nodes))) { return listStyleIDs; }

            _.chain(nodes).getArray().any(function (oneNode) {
                $(oneNode).each(function (index, node) {
                    $(node).find(DOM.PARAGRAPH_NODE_SELECTOR).each(function (i, paragraph) {
                        // checking paragraph attributes for list styles
                        var paraAttrs = AttributeUtils.getExplicitAttributes(paragraph);
                        // updating lists, if required
                        if (isListStyleParagraph(null, paraAttrs) && paraAttrs && paraAttrs.paragraph) {
                            listStyleIDs = listStyleIDs || [];
                            if (!_.contains(listStyleIDs, paraAttrs.paragraph.listStyleId)) { listStyleIDs.push(paraAttrs.paragraph.listStyleId); }
                        }
                    });
                });
            });

            return listStyleIDs;
        }

        /**
         * Collecting information about the event, that triggered the list update. This update can be significantly
         * accelerated by modifying only paragraphs with specified list IDs.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.useSelectedListStyleIDs=false]
         *      If set to true, the update list process can be accelerated by
         *      only modifying paragraphs with correct list style ID set.
         *  @param {String|String[]} [options.listStyleId, '']
         *      The listStyleId of the paragraph.
         *  @param {Number} [options.listStyleLevel = -1]
         *      The listStyleLevel of the paragraph.
         *  @param {Boolean} [options.paraInsert = false]
         *      Whether a paragraph was inserted.
         *  @param {Boolean} [options.splitInNumberedList = false]
         *      Whether a paragraph with a numbered was splitted.
         */
        function registerListUpdate(options) {

            // Performance: Registering whether only specified style IDs need to be updated in this list update (never overwriting, after 'false' was set).
            // This is the case after splitting of a paragraph or after deleting a paragraph
            if ((updateListsDebouncedOptions.useSelectedListStyleIDs === undefined) || (updateListsDebouncedOptions.useSelectedListStyleIDs === true)) {
                updateListsDebouncedOptions.useSelectedListStyleIDs = Utils.getBooleanOption(options, 'useSelectedListStyleIDs', false);

                // checking, whether new paragraphs with list style IDs were inserted.
                // It is sufficient, that only one call of this function sets the value to 'true'.
                if ((updateListsDebouncedOptions.paraInsert === undefined) || (updateListsDebouncedOptions.paraInsert === false)) {
                    updateListsDebouncedOptions.paraInsert = Utils.getBooleanOption(options, 'paraInsert', false);
                }

                // check, if only operations with 'splitInList' triggered this registerListUpdate. In this case the paragraph is marked that was
                // created by the split. Only this and the following paragraphs need to be updated with new numbers. This new created paragraph
                // has the flag 'splitInList'.
                if ((updateListsDebouncedOptions.splitInNumberedList === undefined) || (updateListsDebouncedOptions.splitInNumberedList === true)) {
                    updateListsDebouncedOptions.splitInNumberedList = Utils.getBooleanOption(options, 'splitInNumberedList', false);
                }

                // also collecting all the affected listStyleIDs (listStyleLevels are ignored yet)
                if (Utils.getBooleanOption(options, 'useSelectedListStyleIDs', false)) {
                    updateListsDebouncedOptions.listStyleIds = updateListsDebouncedOptions.listStyleIds || [];
                    if (_.isArray(options.listStyleId)) {
                        updateListsDebouncedOptions.listStyleIds = updateListsDebouncedOptions.listStyleIds.concat(options.listStyleId);
                    } else {
                        updateListsDebouncedOptions.listStyleIds.push(Utils.getStringOption(options, 'listStyleId', ''));
                    }
                }
            }
        }

        /**
         * Removing the artificial selection of empty paragraphs
         */
        function removeArtificalHighlighting() {
            _.each(highlightedParagraphs, function (element) {
                $(element).removeClass('selectionHighlighting');
            });
            highlightedParagraphs = [];
        }

        /**
         * After reloading a document, some updates are necessary in the model.
         * For example refreshing the collector for artificial paragraphs (IE and
         * Firefox).
         */
        function documentReloadHandler() {

            var paragraphs = null;

            if (_.isEmpty(highlightedParagraphs)) { return; }  // nothing to do

            if ((_.browser.Firefox || _.browser.IE) && selection.hasRange()) {
                paragraphs = DOM.getPageContentNode(self.getNode()).find('div.selectionHighlighting');
                // converting to array
                highlightedParagraphs = $.makeArray(paragraphs);
            }
        }

        /**
         * Collecting data about event that triggered inserting page breaks
         *
         * @param {HTMLElement|jQuery} currentNode
         *  DOM node where the cursor position is - we process DOM from that element downward
         *
         */
        function registerPageBreaksInsert(currentNode, textFrameNode) {
            currentProcessingNode = currentNode;
            if (textFrameNode && $(textFrameNode).length > 0) { currentProcessingTextFrameNode = textFrameNode; }
        }

        /**
         * Collecting node for update of all other header/footer nodes of same type
         *
         * @param {jQuery} targetNode
         *  header or footer node that is beeing updated
         *
         */
        function registerTargetNodeUpdate(targetNode) {
            targetNodeForUpdate = targetNode;
        }

        /**
         * Inserts a collaborative overlay element.
         */
        function insertCollaborativeOverlay() {
            editdiv.append($('<div>').addClass('collaborative-overlay').attr('contenteditable', false));
        }

        /**
         * Updating the selection of a remote client without write privileges.
         */
        function updateRemoteSelection() {

            var // whether the selection before applying the external operation was a range selection
                wasRange = false,
                // whether a drawing is selected
                drawingSelected = false,
                // whether the browser selection can be evaluated
                useBrowserSelection = true,
                // whether a cell range is selected
                cellSelected = false,
                // old logical start and end positions
                oldStartPosition = null, oldEndPosition = null,
                // the drawing node of a selected drawing
                drawingNode,
                // logical position of the selected drawing
                drawingStartPosition, drawingEndPosition,
                // a new calculated valid logical position
                validPos,
                // the node containing the focus
                focusNode = null;

            // checking if the selection before applying the external operation was a range selection
            if (selection.hasRange()) {
                wasRange = true;
                oldStartPosition = _.clone(selection.getStartPosition());
                oldEndPosition = _.clone(selection.getEndPosition());
                drawingSelected = (selection.getSelectionType() === 'drawing');
                cellSelected = (selection.getSelectionType() === 'cell');
            }

            if (drawingSelected || cellSelected) { useBrowserSelection = false; }

            // using the browser selection as new selection
            if (useBrowserSelection) {
                selection.updateSelectionAfterBrowserSelection();
                // check if it is necessary to restore a selection range (maybe the browser removed a selection range)
                if (wasRange && !selection.hasRange()) {
                    validPos = Position.findValidSelection(editdiv, oldStartPosition, oldEndPosition, selection.getLastDocumentPosition());
                    selection.setTextSelection(validPos.start, validPos.end, { remoteSelectionUpdate: true });
                }
            } else if (drawingSelected) {

                drawingNode = selection.getSelectedDrawing();

                if ($(drawingNode).closest(DOM.PAGECONTENT_NODE_SELECTOR).length > 0) {
                    // is the drawing still in the dom?
                    drawingStartPosition = Position.getOxoPosition(editdiv, drawingNode, 0);
                    drawingEndPosition = Position.increaseLastIndex(drawingStartPosition);

                    if (!_.isEqual(drawingStartPosition), selection.getStartPosition()) {
                        selection.setTextSelection(drawingStartPosition, drawingEndPosition, { remoteSelectionUpdate: true });
                    }
                } else {
                    validPos = Position.findValidSelection(editdiv, selection.getStartPosition(), selection.getEndPosition(), selection.getLastDocumentPosition());
                    selection.setTextSelection(validPos.start, validPos.end, { preserveFocus: false, remoteSelectionUpdate: true });

                    // The div.page must get the focus. Unfortunately this fails in Chrome,
                    // if Chrome does not have the focus.
                    if (!$(window.document.activeElement).is(DOM.PAGE_NODE_SELECTOR)) {
                        focusNode = window.document.activeElement;
                        Utils.info('Missing browser focus -> changing focus failed. Focus at ' + focusNode.nodeName + '.' + focusNode.className);
                    }
                }

            } else if (cellSelected) {

                // Updating position for cell selection
                selection.updateSelectionAfterBrowserSelection();

                // maybe the cell selection was destroyed by the operation -> selection.getSelectedCellRange() returns null
                if (!selection.getSelectedCellRange()) {
                    validPos = Position.findValidSelection(editdiv, oldStartPosition, oldEndPosition, selection.getLastDocumentPosition());
                    selection.setTextSelection(validPos.start, validPos.end, { remoteSelectionUpdate: true });
                }

            }
        }

        /**
         * Shows a warning dialog with Yes/No buttons before deleting document
         * contents that cannot be restored.
         *
         * @param {String} title
         *  The title of the dialog box.
         *
         * @param {String} message
         *  The message text shown in the dialog box.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved if the Yes
         *  button has been pressed, or rejected if the No button has been pressed.
         */
        function showDeleteWarningDialog(title, message) {

            var // the displayed dialog
                dialog = new Dialogs.ModalQueryDialog(message, { title: title, width: 650 });

            // disable modifications of div.page node (36435)
            self.getNode().attr('contenteditable', false);

            // close dialog automatically after losing edit rights
            app.getView().closeDialogOnReadOnlyMode(dialog);

            return dialog.show().always(function () {
                self.getNode().attr('contenteditable', true);
            });
        }

        /**
         * Removes pageBreakBefore paragraph attribute from passed table element. It is called when
         * cursor is in first position inside first table paragraph and BACKSPACE is pressed, or
         * when DELETE is pressed on cursor placed in paragraph's last position right above table
         *
         * @param {jQuery|DOM} table - table element that has paragraph with pageBreakBefore attribute,
         *  and has to be removed.
         *
         */
        function removePageBreakBeforeAttribute(table) {
            var
                paragraphWithManualPageBreak,
                paragraphPos,
                operation;

            paragraphWithManualPageBreak = $(table).find(DOM.MANUAL_PAGE_BREAK_SELECTOR);
            if (paragraphWithManualPageBreak.length > 0) {
                paragraphPos = Position.getOxoPosition(editdiv, paragraphWithManualPageBreak);

                operation = { name: Operations.SET_ATTRIBUTES, attrs: { paragraph: { pageBreakBefore: false } }, start: paragraphPos };
                self.applyOperations(operation);
            } else {
                if (app.isODF()) {
                    paragraphPos = Position.getFirstPositionInParagraph(editdiv, Position.getOxoPosition(editdiv, table));
                    operation = { name: Operations.SET_ATTRIBUTES, attrs: { paragraph: { pageBreakBefore: null } }, start: paragraphPos };
                    self.applyOperations(operation);
                } else {
                    // #35358 - if we delete paragraph, we cannot remove attr from it, so just clean up table marker for displaying
                    $(table).removeClass(DOM.MANUAL_PAGE_BREAK_CLASS_NAME);
                }
            }
        }

        /**
         * Clean up after the busy mode of a local client with edit prileges. After the
         * asynchronous operations were executed, the busy mode can be left.
         */
        function leaveAsyncBusy() {
            // closing the dialog
            app.getView().leaveBusy().grabFocus();
            // end of gui triggered operation
            self.setGUITriggeredOperation(false);
            // allowing keyboard events again
            self.setBlockKeyboardEvent(false);
            // always leaving clip board paste mode, also this is not always required
            setClipboardPasteInProgress(false);
            // restore the original selection (this is necessary for directly following key press)
            selection.restoreBrowserSelection();
        }

        /**
         * Asynchronous execution of operations from the local client with edit privileges.
         * This is necessary for the operations that depend on a (huge) selection. This is
         * very important for setAttributes() and deleteSelected(). At least the applying
         * of operations needs to be done asynchronously.
         * TODO: Generation of operations for setAttributes and deleteSelected() also needs
         * to be done asynchronously.
         *
         * @param {OperationsGenerator} generator
         *  The generated operations.
         *
         * @param {String} label
         *  The label that is shown while applying the operations.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.showProgress=true]
         *      Whether the progress bar shall be handled inside this function. This is not
         *      necessary, if there is already an outer function that handles the progress
         *      bar. The latter is for example the case for deleting the selection before
         *      pasting the content of the external clipboard.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved after all
         *  actions have been applied successfully, or rejected immediately
         *  after applying an action has failed. See editModel.applyActions()
         *  for more information.
         */
        function applyTextOperationsAsync(generator, label, options) {

            var // the promise for the asychronous execution of operations
                operationsPromise = null,
                // whehter the visibility of the progress bar shall be handled by this function
                // -> this is typically not necessary during pasting, because there is already
                //    a visible progress bar
                showProgress = Utils.getBooleanOption(options, 'showProgress', true),
                // closing progress bar at the end, if not a user abort occurred
                leaveOnSuccess = Utils.getBooleanOption(options, 'leaveOnSuccess', false),
                // an optional start value for the progress bar
                progressStart = Utils.getNumberOption(options, 'progressStart', 0);

            // setting a default label, if not specified from caller
            if (!label) { label = gt('Sorry, applying changes will take some time.'); }

            // blocking keyboard input during applying of operations
            self.setBlockKeyboardEvent(true);

            // Wait for a second before showing the progress bar (entering busy mode).
            // Users applying minimal or 'normal' amount of change tracks will not see this progress bar at all.
            if (showProgress) {
                app.getView().enterBusy({
                    cancelHandler: function () {
                        if (operationsPromise) { operationsPromise.abort(); }
                    },
                    delay: 1000,
                    warningLabel: label
                }).grabFocus();
            }

            // fire apply actions asynchronously
            operationsPromise = self.applyOperations(generator, { async: true });

            // handle the result of change track operations
            operationsPromise
                .progress(function (progress) {
                    // update the progress bar according to progress of the operations promise
                    app.getView().updateBusyProgress(progressStart + progress * (1 - progressStart));
                })
                .always(function () {
                    if (showProgress || leaveOnSuccess) {
                        leaveAsyncBusy();
                    }
                });

            return operationsPromise;
        }

        /**
         * iterate over _all_ paragraphs and update numbering symbols and index
         */
        var updateListsDebounced = $.noop;
        app.onImportSuccess(function () {
            // Adding delay for debounced list update, so that the browser gets the chance to render the new page (iPad mini)
            updateListsDebounced = self.createDebouncedMethod(registerListUpdate, updateLists, { delay: 10, infoString: 'Text: updateLists' });
        });

        var insertPageBreaksDebounced = $.noop;
        app.onImportSuccess(function () {
            insertPageBreaksDebounced = self.createDebouncedMethod(registerPageBreaksInsert, function () {
                if (currentProcessingTextFrameNode) { self.trigger('drawingHeight:update', $(currentProcessingTextFrameNode)); }
                currentProcessingTextFrameNode = null;
                if (self.getBlockOnInsertPageBreaks()) { // if operation is inside header/footer, block page breaks. They will be triggered from other function if necessary.
                    self.setBlockOnInsertPageBreaks(false);
                    self.trigger('update:absoluteElements');  // updating comments and absolutely positioned drawings
                } else {
                    pageLayout.insertPageBreaks();
                }
            }, { delay: 200, infoString: 'Text: insertPageBreaksDebounced' });
        });

        // Debouncing distribution of updated data from editing node to all other headers&footers with same type
        var updateEditingHeaderFooterDebounced = $.noop;
        app.onImportSuccess(function () {
            updateEditingHeaderFooterDebounced = self.createDebouncedMethod(registerTargetNodeUpdate, pageLayout.updateEditingHeaderFooter, { delay: 200, infoString: 'Text: pageLayout.updateEditingHeaderFooter' });
        });

        // Remote client needs to get all changes in headers and footers, too.
        // On remote client, all impl functions with header/footer target make changes to header footer template node.
        // To get changes to all nodes with same type in document, this function needs to be triggered.
        // Usually called from this.getRootNode - when target is present, and no edit rights
        var replaceAllTypesOfHeaderFootersDebounced = $.noop;
        app.onImportSuccess(function () {
            replaceAllTypesOfHeaderFootersDebounced = self.createDebouncedMethod(function () {}, pageLayout.replaceAllTypesOfHeaderFooters, { delay: 200, infoString: 'Text: pageLayout.replaceAllTypesOfHeaderFooters' });
        });

        // public handler function for updating change tracks debounced.
        // This handler invalidates the existing side bar. This needs to be done
        // after operations or after 'refresh:layout' event.
        this.updateChangeTracksDebounced = $.noop;

        // public handler function for updating change tracks debounced.
        // This handler does not invalidate the existing side bar. This can happen
        // for example after 'scroll' events.
        this.updateChangeTracksDebouncedScroll = $.noop;

        // public handler function for updating comment ranges debounced. This is
        // required, if a comment range is visualized (with hover) and then it is
        // written inside this comment range. This happens after the event
        // 'paragraphUpdate:after' was triggered.
        var updateHighligtedRangesDebounced = $.noop;
        app.onImportSuccess(function () {
            updateHighligtedRangesDebounced = self.createDebouncedMethod($.noop, rangeMarker.updateHighlightedRanges, { delay: 100, infoString: 'Text: rangeMarker.updateHighlightedRanges' });
        });

        /**
         * Dumps the passed event object to the browser console.
         */
        var dumpEventObject = LOG_EVENTS ? function (event) {
            Utils.log('type=' + event.type + ' keyCode=' + event.keyCode + ' charCode=' + event.charCode + ' shift=' + event.shiftKey + ' ctrl=' + event.ctrlKey + ' alt=' + event.altKey);
        } : $.noop;

        // initialization -----------------------------------------------------

        undoManager = this.getUndoManager();
        characterStyles = this.getStyleCollection('character');
        paragraphStyles = this.getStyleCollection('paragraph');
        tableStyles = this.getStyleCollection('table');
        tableRowStyles = this.getStyleCollection('row');
        tableCellStyles = this.getStyleCollection('cell');
        drawingStyles = this.getStyleCollection('drawing');
        pageStyles = this.getStyleCollection('page');

        // the logical selection, synchronizes with browser DOM selection
        selection = new Selection(app, editdiv);
        remoteSelection = new RemoteSelection(app, editdiv);
        changeTrack = new ChangeTrack(app, editdiv, self);
        fieldManager = new FieldManager(app);
        drawingLayer = new DrawingLayer(app, editdiv);
        commentLayer = new CommentLayer(app, editdiv);
        rangeMarker = new RangeMarker(app, editdiv);
        pageLayout = new PageLayout(this, editdiv);
        numberFormatter = new NumberFormatter(this);
        listCollection = new ListCollection(this);
        searchHandler = new SearchHandler(app, this);
        spellChecker = new SpellChecker(app, this);

        // initialize document contents
        app.onInit(initDocument);

        // initialize selection after import
        app.onImport(function () {
            selection.selectDocumentLoadPosition();
            app.getView().grabFocus();
        });

        // more initialization after successful import
        app.onImportSuccess(documentLoaded);

        // disable browser editing capabilities if import fails
        app.onImportFailure(function () {
            editdiv.attr('contenteditable', false);
        });

        // restore selection after undo/redo operations
        undoManager.on('undo:before redo:before', function () {
            undoRedoRunning = true;
        });

        // restore selection after undo/redo operations
        undoManager.on('undo:after redo:after', function (event, operations) {
            var para = null,
                newParagraph = null,
                pageContentNode = null,
                lastOperation = _.last(operations),
                target = lastOperation.target,
                isCreateDeleteHeaderFooterOp = lastOperation.name === 'insertHeaderFooter' || lastOperation.name === 'deleteHeaderFooter',
                $rootNode = self.getRootNode(target),
                // the node at the last position
                lastPositionNode = null;

            // not blocking page break calculations after undo (40169)
            self.setBlockOnInsertPageBreaks(false);

            // group of undo operations, replace headers&footers after all operations are finnished
            if (_.findWhere(operations, { name: 'insertHeaderFooter' })) {
                pageLayout.replaceAllTypesOfHeaderFooters();
                pageLayout.updateStartHeaderFooterStyle();
                self.setBlockOnInsertPageBreaks(false); // release block on insertPageBreaks to register debounced call
                insertPageBreaksDebounced();
            }
            // odf needs also replacing of headers after delete (switching type)
            if (app.isODF() && _.findWhere(operations, { name: 'deleteHeaderFooter' })) {
                pageLayout.replaceAllTypesOfHeaderFooters();
            }

            // return from edit mode of header/footer if undo/redo operation doesnt have target,
            // or is not create or delete headerFooter operation
            if (self.isHeaderFooterEditState() && !(target || isCreateDeleteHeaderFooterOp)) {
                pageLayout.leaveHeaderFooterEditMode();
                selection.setNewRootNode(self.getNode()); // restore original rootNode
                updateEditingHeaderFooterDebounced();
            } else if (target && pageLayout.isIdOfMarginalNode(target)) {
                // jump into editing mode of header/footer with given target (first occurance in doc)
                if (self.isHeaderFooterEditState()) {
                    // do nothing if we are already inside that node
                    if (self.getHeaderFooterRootNode()[0] !== $rootNode[0]) {
                        // first leave current node, and enter another
                        pageLayout.leaveHeaderFooterEditMode(self.getHeaderFooterRootNode());

                        // must be direct call
                        pageLayout.updateEditingHeaderFooter();

                        if (!$rootNode.parent().hasClass(DOM.HEADER_FOOTER_PLACEHOLDER_CLASSNAME)) {
                            // only if h/f type is not in placeholder
                            pageLayout.enterHeaderFooterEditMode($rootNode);
                            selection.setNewRootNode($rootNode);
                            app.getView().scrollToChildNode($rootNode);
                        }
                    }
                } else if (!$rootNode.parent().hasClass(DOM.HEADER_FOOTER_PLACEHOLDER_CLASSNAME)) {
                    // only if h/f type is not in placeholder
                    pageLayout.enterHeaderFooterEditMode($rootNode);
                    selection.setNewRootNode($rootNode);
                    app.getView().scrollToChildNode($rootNode);
                }
            }

            // setting text selection to 'lastOperationEnd' after undo/redo.
            // Special behaviour for setAttributes operations, where an existing
            // browser selection needs to be restored (useSelectionRangeInUndo).
            // -> Fix for task 28520: An setAttributes after a splitParagraph can have an invalid text selection
            // and therefore must use lastOperationEnd
            if ((useSelectionRangeInUndo) && (!selection.isTextLevelSelection() || selection.isValidTextSelection())) {

                if (lastOperationEnd) {
                    lastPositionNode = Position.getDOMPosition($rootNode, lastOperationEnd);
                    lastPositionNode = (lastPositionNode && lastPositionNode.node) ? lastPositionNode.node : null;
                    lastPositionNode = (lastPositionNode && lastPositionNode.nodeType === 3) ? lastPositionNode.parentNode : null;
                }

                if (lastOperationEnd && lastPositionNode && DOM.isInsidePlaceHolderComplexField(lastPositionNode)) {
                    selection.setTextSelection(lastOperationEnd); // behavior for task 40085
                } else {
                    selection.restoreBrowserSelection({ preserveFocus: true });
                }

            } else {
                // #31971, #32322
                pageContentNode = DOM.getPageContentNode(editdiv);
                if (pageContentNode.find(DOM.CONTENT_NODE_SELECTOR).length === 0 && pageContentNode.find(DOM.PAGE_BREAK_SELECTOR).length > 0) { //cleanup of dom
                    pageContentNode.find(DOM.PAGE_BREAK_SELECTOR).remove();

                    newParagraph = DOM.createImplicitParagraphNode();
                    pageContentNode.append(newParagraph);
                    validateParagraphNode(newParagraph);
                    implParagraphChanged(newParagraph);
                    // in Internet Explorer it is necessary to add new empty text nodes in paragraph again
                    // newly, also in other browsers (new jQuery version?)
                    repairEmptyTextNodes(newParagraph);
                }

                if (_.browser.IE) {
                    // 29397: Not using implicit paragraph in IE
                    para = Position.getLastNodeFromPositionByNodeName($rootNode, lastOperationEnd, DOM.PARAGRAPH_NODE_SELECTOR);
                    if (DOM.isImplicitParagraphNode(para) && para.previousSibling) {
                        lastOperationEnd = Position.getLastPositionInParagraph($rootNode, Position.getOxoPosition($rootNode, para.previousSibling, 0));
                    }
                }

                if (lastOperationEnd) {

                    // undo might have inserted a table -> checking lastOperationEnd (Firefox)
                    if (!Position.getLastNodeFromPositionByNodeName($rootNode, lastOperationEnd, DOM.PARAGRAPH_NODE_SELECTOR)) {
                        lastOperationEnd = Position.getFirstPositionInParagraph($rootNode, lastOperationEnd);
                    }

                    // setting selection before calling 'implParagraphChangedSync' (35350)
                    selection.setTextSelection(lastOperationEnd);

                    // Avoiding cursor jumping, if new paragraph is empty (32454), indicated by setting cursor to position ending with '0'
                    if (_.last(lastOperationEnd) === 0) {
                        para = para || Position.getLastNodeFromPositionByNodeName($rootNode, lastOperationEnd, DOM.PARAGRAPH_NODE_SELECTOR);
                        if (para) { implParagraphChangedSync($(para)); }
                    }
                }

            }

            // this must be executed, never leave function before!
            undoRedoRunning = false;
            // hide old/invalid change track popups after undo/redo
            app.getView().getChangeTrackPopup().hide();
        });

        // undogroup:open : checking if the operations in a group can be undone
        undoManager.on('undogroup:open', function () {
            deleteUndoStack = false;  // default: undo stack must not be deleted
            isInUndoGroup = true;
        });

        // undogroup:closed : delete undo stack, if required
        undoManager.on('undogroup:close', function () {
            if (deleteUndoStack) { undoManager.clearUndoActions(); }
            deleteUndoStack = false;
            isInUndoGroup = false;
        });

        // remove highlighting before changing the DOM which invalidates the positions in highlightRanges
        this.on('operations:before', function (event, operations, external) {

            // checking operations for change track attributes
            if (!changeTrack.isSideBarHandlerActive() && changeTrack.hasChangeTrackOperation(operations)) {
                self.trigger('changeTrack:stateInfo', { state: true });
            }

            // reset read-only paragraph
            if (roIOSParagraph) {
                if (self.getEditMode() === true) {
                    $(roIOSParagraph).removeAttr('contenteditable');
                }
                roIOSParagraph = null;
            }

            // clear preselected attributes before applying any operations
            if (!keepPreselectedAttributes) {
                preselectedAttributes = null;
            }

            // Insert drawing operations with a document URL as imageUrl attribute also have the imageData attribute set.
            // Before the operation is applied we need to determine which attribute to use and which to remove.
            if (!external) {
                _(operations).each(function (operation) {
                    if (operation.name === Operations.DRAWING_INSERT && (operation.type === 'image' || operation.type === 'ole') && operation.attrs && operation.attrs.image) {
                        Image.postProcessOperationAttributes(operation.attrs.image, app.getFileDescriptor().id);
                    }
                });
            }
        });

        this.on('operations:success', function (event, operations) {
            // saving information about the type of operation
            useSelectionRangeInUndo = (_.last(operations).name === Operations.SET_ATTRIBUTES);

            // useSelectionRangeInUndo also needs to be true, if after setAttributes an operation like deleteListStyle comes (32005)
            // -> maybe there needs to be a list of operations that can be ignored like 'Operations.DELETE_LIST' following the setAttributes operation
            // -> leaving this simple now because of performance reasons
            if ((!useSelectionRangeInUndo) && (operations.length > 1) && (_.last(operations).name === Operations.DELETE_LIST) && (operations[operations.length - 2].name === Operations.SET_ATTRIBUTES)) {
                useSelectionRangeInUndo = true;
            }

            // update change track sidebar after successfully applied operation
            self.updateChangeTracksDebounced();

            // In read-only mode update the selection after an external operation (and after the document is loaded successfully)
            // -> maybe this need to be deferred
            if (self.getEditMode() !== true && app.isImportFinished()) {
                updatingRemoteSelection = true;
                updateRemoteSelection();
                updatingRemoteSelection = false;
            }
        });

        // edit mode changed
        this.on('change:editmode', function (event, editMode) {

            // set CSS class for formatting dependent on edit mode (mouse pointers)
            editdiv.toggleClass('edit-mode', editMode);

            // Fix for 30992, setting valid text position after receiving edit rights
            if (editMode) {
                if (Position.isValidTextPosition(editdiv, selection.getStartPosition())) {
                    if (Position.isValidTextPosition(editdiv, selection.getEndPosition())) {
                        selection.setTextSelection(selection.getStartPosition(), selection.getEndPosition());
                    } else {
                        selection.setTextSelection(selection.getStartPosition());
                    }
                } else if ((!_.isEmpty(selection.getStartPosition())) && (Position.isValidTextPosition(editdiv, Position.getFirstPositionInParagraph(editdiv, [selection.getStartPosition()[0]])))) {
                    selection.setTextSelection(Position.getFirstPositionInParagraph(editdiv, [selection.getStartPosition()[0]]));
                } else if (lastOperationEnd && Position.isValidTextPosition(editdiv, lastOperationEnd)) {
                    selection.setTextSelection(lastOperationEnd);
                } else {
                    selection.selectTopPosition();
                }
            }

            // in read-only mode change drawing selection to text selection
            if (!editMode) {
                selection.selectDrawingAsText();
                if (self.isHeaderFooterEditState()) {
                    pageLayout.leaveHeaderFooterEditMode();
                    selection.setNewRootNode(editdiv);
                    selection.setTextSelection(selection.getStartPosition());
                }

            }

            // We are not able to disable the context menu in read-only mode on iPad
            // or other touch devices. Therefore we just set contenteditable dependent
            // on the current editMode.
            if (Utils.TOUCHDEVICE) {

                // prevent virtual keyboard on touch devices in read-only mode
                editdiv.attr('contenteditable', editMode);
            }

            // disable IE table manipulation handlers in edit mode
            // Utils.getDomNode(editdiv).onresizestart = function () { return false; };
            // The resizestart event does not appear to bubble in IE9+, so we use the selectionchange event to bind
            // the resizestart handler directly once the user selects an object (as this is when the handles appear).
            // The MS docs (http://msdn.microsoft.com/en-us/library/ie/ms536961%28v=vs.85%29.aspx) say it's not
            // cancelable, but it seems to work in practice.

            // The oncontrolselect event fires when the user is about to make a control selection of the object.
            // Non-standard event defined by Microsoft for use in Internet Explorer.
            // This event fires before the element is selected, so inspecting the selection object gives no information about the element to be selected.
            if (_.browser.IE) {
                Utils.getDomNode(editdiv).oncontrolselect = function (e) {
                    if (DOM.isImageNode(e.srcElement)) {
                        // if an image was selected, return browser selection to the image clipboard
                        selection.restoreBrowserSelection();
                    }
                    // return false to suppress IE default size grippers
                    return false;
                };
            }

            // focus back to editor
            app.getView().grabFocus();
        });

        /**
         * Registering the listener for the 'changeTrack:stateInfo' event. This event is
         * triggered with the option 'state: true' after loading the document and the
         * document contains change tracked elements or after receiving an operation
         * with change track attributes.
         * The event is triggered with 'state: false', if no change tracked element was
         * found inside the document.
         */
        this.on('changeTrack:stateInfo', function (event, options) {

            var // whether the document contains change tracking content
                state = Utils.getBooleanOption(options, 'state', false);

            if (state) {
                if (!changeTrack.isSideBarHandlerActive() && !Utils.SMALL_DEVICE) {
                    changeTrack.setSideBarHandlerActive(true);
                    self.updateChangeTracksDebounced = self.createDebouncedMethod($.noop, function () { changeTrack.updateSideBar({ invalidate: true }); }, { delay: 500, maxDelay: 1500, infoString: 'Text: changeTrack.updateSideBar(true)' });
                    self.updateChangeTracksDebouncedScroll = self.createDebouncedMethod($.noop, function () { changeTrack.updateSideBar({ invalidate: false }); }, { delay: 500, maxDelay: 1500, infoString: 'Text: changeTrack.updateSideBar(false)' });
                }
            } else {
                if (changeTrack.isSideBarHandlerActive()) {
                    changeTrack.setSideBarHandlerActive(false);
                    self.updateChangeTracksDebounced = $.noop;
                    self.updateChangeTracksDebouncedScroll = $.noop;
                }
            }

        });

        /**
         * Registering the listener for the events 'paragraphUpdate:after' and
         * 'tableUpdate:after'. These events are triggered, when the deferred
         * updating of paragraphs or tables has finished. The listener receives
         * the formatted node. This is required, because of task 36495.
         *
         * @param {jQuery.Event} event
         *  The jQuery browser event object.
         *
         * @param {Node|Node[]|jQuery} nodes
         *  The node or list of nodes that can be handled in this event handler.
         */
        this.on('paragraphUpdate:after tableUpdate:after drawingHeight:update', function (event, nodes) {

            _.each($(nodes), function (node) {

                var drawingNode = null;

                if (DOM.isImageDrawingNode(node) || DrawingFrame.isBorderNode(node) || DOM.isNodeInsideTextFrame(node) || DrawingFrame.isTextFrameShapeDrawingFrame(node)) {
                    drawingNode = DrawingFrame.getDrawingNode(node);
                    drawCanvasBorders(drawingNode);
                    if (DrawingFrame.isGroupedDrawingFrame(drawingNode) && DrawingFrame.isAutoResizeHeightDrawingFrame(drawingNode)) {
                        DrawingFrame.updateDrawingGroupHeight(drawingNode);
                    }
                }
                if (event.type !== 'drawingHeight:update' && pageLayout.isHeaderFooterActivated()) { // #37250 - Looping request when exit header edit mode
                    if (DOM.isMarginalNode(node)) {
                        updateEditingHeaderFooterDebounced(DOM.getMarginalTargetNode(self.getNode(), node));
                    } else if (DOM.isHeaderOrFooter(node)) {
                        updateEditingHeaderFooterDebounced(node);
                    } else if ($(node).find(DOM.TAB_NODE_SELECTOR).length && pageLayout.getHeaderFooterPlaceHolder().find(node).length) {
                        // if updated paragraph is inside headerfooter, but in template node, and it has tabs inside,
                        // redistribute to all visible headerfooter nodes
                        pageLayout.replaceAllTypesOfHeaderFooters();
                    }
                }

                // updating an visualized (comment) range, if necessary
                updateHighligtedRangesDebounced();
            });
        });

        /**
         * Registering the handler for document reload event. This happens, if the
         * user cancels a long running action.
         */
        this.on('document:reloaded', documentReloadHandler);

        // scroll to cursor, forward selection change events to own listeners
        selection.on('change', function (event, options) {

            var // whether an implicit paragraph behind a table has to be increased
                increaseParagraph = false,
                domNode = null,
                paraNode = null,
                rowNode = null,
                tableNode = null,
                explicitAttributes = null,
                startPosition = null,
                nodeInfo = null,
                insertOperation = Utils.getBooleanOption(options, 'insertOperation', false),
                splitOperation = Utils.getBooleanOption(options, 'splitOperation', false),
                keepChangeTrackPopup = Utils.getBooleanOption(options, 'keepChangeTrackPopup', false),
                browserEvent = Utils.getObjectOption(options, 'event', null),
                readOnly = self.getEditMode() !== true,
                // selection hightlighting: dom point of start and end position
                startNodeInfo = null, endNodeInfo = null,
                // selection hightlighting: paragraph dom nodes
                nextParagraph = null, startParagraph = null, endParagraph = null,
                // whether the iteration reached the end paragraph
                reachedEndParagraph = false,
                // helper nodes for explicit paragraphs inside text frames
                drawing = null, drawingBorder = null,
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // current element container node
                thisRootNode = self.getCurrentRootNode(target);

            // handling height of implicit paragraphs, but not for insertText operations or splitting of paragraphs
            // and also not in readonly mode (28563)
            if (!insertOperation && !splitOperation && !readOnly) {

                startPosition = selection.getStartPosition();
                nodeInfo = Position.getDOMPosition(thisRootNode, startPosition);

                // if this is the last row of a table, a following implicit paragraph has to get height zero or auto
                if (Position.isPositionInTable(thisRootNode, startPosition)) {
                    domNode = nodeInfo.node;
                    paraNode = $(domNode).closest(DOM.PARAGRAPH_NODE_SELECTOR);
                    rowNode = $(paraNode).closest(DOM.TABLE_ROWNODE_SELECTOR);
                    tableNode = $(rowNode).closest(DOM.TABLE_NODE_SELECTOR);

                    if (paraNode !== null) {
                        if (paraNode.get(0) === increasedParagraphNode) {
                            increaseParagraph = true;  // avoid, that height is set to 0, if cursor is in this paragraph
                            if (paraNode.prev().length === 0) {  // maybe the previous table was just removed
                                paraNode.css('height', '');  // the current paragraph always has to be increased
                                increasedParagraphNode = null;
                            }
                        } else if (DOM.isIncreasableParagraph(paraNode)) {
                            if (increasedParagraphNode !== null) { $(increasedParagraphNode).css('height', 0); } // useful for tables in tables in tables in ...
                            paraNode.css('height', '');
                            increaseParagraph = true;   // cursor travelling from right/button directly into the explicit paragraph
                            increasedParagraphNode = paraNode.get(0);
                            // in Internet Explorer it is necessary to restore the browser selection, so that no frame is drawn around the paragraph (28132)
                            if (_.browser.IE) { selection.restoreBrowserSelection(); }
                        } else if ((DOM.isChildTableNode(tableNode)) && ($(rowNode).next().length === 0) && (DOM.isIncreasableParagraph($(tableNode).next()))) {
                            if (increasedParagraphNode !== null) { $(increasedParagraphNode).css('height', 0); } // useful for tables in tables in tables in ...
                            $(tableNode).next().css('height', '');
                            increaseParagraph = true;
                            increasedParagraphNode = $(tableNode).next().get(0);
                        } else if ((paraNode.prev().length === 0) && (paraNode.height() === 0)) {
                            // if the cursor was not in the table, when the inner table was deleted -> there was no increasedParagraphNode behind the table
                            paraNode.css('height', '');  // the current paragraph always has to be increased, maybe a previous table just was deleted (undo)
                        }
                    }
                } else {

                    // check if the currently increased paragraph behind a table is clicked (for example inside a text frame)
                    if (increasedParagraphNode) {
                        domNode = nodeInfo.node;
                        paraNode = domNode && $(domNode).closest(DOM.PARAGRAPH_NODE_SELECTOR);
                        if (paraNode && Utils.getDomNode(paraNode) === increasedParagraphNode) {
                            increaseParagraph = true;  // avoid, that height is set to 0, if cursor is in this paragraph
                        }
                    }

                    if (_.browser.IE) {
                        // in Internet Explorer it is necessary to restore the browser selection, so that no frame is drawn around the paragraph (28132)
                        paraNode = nodeInfo ? $(nodeInfo.node).closest(DOM.PARAGRAPH_NODE_SELECTOR) : null;
                        if (DOM.isIncreasableParagraph(paraNode)) {
                            paraNode.css('height', '');
                            selection.restoreBrowserSelection();
                        }
                    }
                }

                // is the increased paragraph node inside a text frame?
                drawing = $(increasedParagraphNode).closest(DrawingFrame.NODE_SELECTOR);

                // if the implicit paragraph no longer has to get an increase height
                if (!increaseParagraph && increasedParagraphNode) {
                    // can the increased paragraph node still be decreased (not if it is no longer increasable)
                    if (DOM.isIncreasableParagraph(increasedParagraphNode)) {
                        $(increasedParagraphNode).css('height', 0);
                    }
                    increasedParagraphNode = null;

                }

                // check, if the drawing border needs to be repainted (canvas)
                if (drawing.length > 0) {
                    drawingBorder = DrawingFrame.getBorderNode(drawing);
                    if (drawingBorder.length > 0) {
                        self.trigger('drawingHeight:update', drawing);
                    }
                }

            }

            // in Internet Explorer it is necessary to restore the browser selection, so that no frame is drawn around the paragraph (28132),
            // this part for the read only mode
            if (readOnly && _.browser.IE && !Position.isPositionInTable(thisRootNode, selection.getStartPosition())) {
                nodeInfo = Position.getDOMPosition(thisRootNode, selection.getStartPosition());
                if (nodeInfo) {
                    paraNode = $(nodeInfo.node).closest(DOM.PARAGRAPH_NODE_SELECTOR);
                    if ((DOM.isImplicitParagraphNode(paraNode)) && (DOM.isFinalParagraphBehindTable(paraNode))) {
                        paraNode.css('height', '');
                        selection.restoreBrowserSelection();
                    }
                }
            }

            // clear preselected attributes when selection changes
            if (!keepPreselectedAttributes) {
                preselectedAttributes = null;
            }

            // Special behaviour for cursor positions directly behind fields.
            // -> the character attributes of the field have to be used
            // -> using preselectedAttributes
            if (!insertOperation && nodeInfo && (nodeInfo.offset === 0) && nodeInfo.node && nodeInfo.node.parentNode && DOM.isFieldNode(nodeInfo.node.parentNode.previousSibling)) {
                explicitAttributes = AttributeUtils.getExplicitAttributes(nodeInfo.node.parentNode.previousSibling.firstChild);
                self.addPreselectedAttributes(explicitAttributes);
            }

            // removing highlighting of selection of empty paragraphs in Firefox and IE
            if (!_.isEmpty(highlightedParagraphs)) { removeArtificalHighlighting(); }

            // highlighting selection of empty paragraphs in Firefox and IE
            if ((_.browser.Firefox || _.browser.IE) && selection.hasRange() && selection.getSelectionType() !== 'cell') {

                startNodeInfo = Position.getDOMPosition(thisRootNode, selection.getStartPosition());
                endNodeInfo = Position.getDOMPosition(thisRootNode, selection.getEndPosition());

                if (startNodeInfo && startNodeInfo.node && endNodeInfo && endNodeInfo.node) {
                    startParagraph = Utils.getDomNode($(startNodeInfo.node).closest(DOM.PARAGRAPH_NODE_SELECTOR));
                    endParagraph = Utils.getDomNode($(endNodeInfo.node).closest(DOM.PARAGRAPH_NODE_SELECTOR));

                    if (startParagraph && endParagraph && startParagraph !== endParagraph) {
                        nextParagraph = startParagraph;

                        while (nextParagraph && !reachedEndParagraph) {

                            if (nextParagraph) {

                                // only highlight empty paragraphs
                                if (Position.getParagraphNodeLength(nextParagraph) === 0) {
                                    // collecting all modified paragraphs in an array
                                    highlightedParagraphs.push(nextParagraph);
                                    // modify the background color of the empty paragraph to simulate selection
                                    // Task 35375: Using class, so that background-color is not included into copy/paste
                                    $(nextParagraph).addClass('selectionHighlighting');
                                }

                                if (nextParagraph === endParagraph) {
                                    reachedEndParagraph = true;
                                } else {
                                    nextParagraph = Utils.findNextNode(thisRootNode, nextParagraph, DOM.PARAGRAPH_NODE_SELECTOR);
                                }
                            }
                        }
                    }
                }
            }

            // forward selection change events to own listeners (view and controller)
            self.trigger('selection', selection, { insertOperation: insertOperation, browserEvent: browserEvent, keepChangeTrackPopup: keepChangeTrackPopup, updatingRemoteSelection: updatingRemoteSelection });
        });

        // Register selectionchange handler to get notification if user changes
        // selection via handles. No other way to detect these selection changes on
        // mobile Safari.
        if (Utils.IOS) {
            app.registerGlobalEventHandler(document, 'selectionchange', function (event) {
                var sel = window.getSelection();

                if (sel && sel.rangeCount >= 1) {
                    // Check that the selection is inside our editdiv node. We are listing to
                    // the whole document!
                    if (Utils.containsNode(editdiv, sel.getRangeAt(0).startContainer)) {
                        selection.processBrowserEvent(event);
                    }
                }
            });
        }

        // registering keypress handler at document for Chrome to avoid problems with direct keypress to document
        // -> using app.registerGlobalEventHandler the handler is registered with 'show' and deregistered
        // with 'hide' event of the app automatically and does not influence other apps.
        if (_.browser.WebKit) {
            app.registerGlobalEventHandler(document, 'keypress', processKeypressOnDocument);
        }

        // hybrid edit mode
        editdiv.on(listenerList = {
            keydown: processKeyDown,
            keypress: processKeyPressed,
            keyup: processKeyUp,
            compositionstart: processCompositionStart,
            compositionupdate: processCompositionUpdate,
            compositionend: processCompositionEnd,
            textInput: processTextInput,
            input: processInput,
            'mousedown touchstart': processMouseDown,
            'mouseup touchend': processMouseUp,
            blur: processBlur,
            dragstart: processDragStart,
            drop: processDrop,
            dragover: processDragOver,
            dragenter: processDragEnter,
            dragleave: processDragLeave,
            cut: _.bind(this.cut, this),
            copy: _.bind(this.copy, this),
            'paste beforepaste': _.bind(this.paste, this)     // for IE we need to handle 'beforepaste', on all other browsers 'paste'
        });

        if (Utils.SMALL_DEVICE) {
            editdiv.on('touchstart touchmove touchend', DOM.TABLE_NODE_SELECTOR, processTouchEventsForTableDraftMode);
        }

        // Fix for 29751: IE support double click for word selection
        if (_.browser.IE || _.browser.WebKit) {
            editdiv.on({
                dblclick: processDoubleClick
            });
        }

        // header/footer editing on doubleclick
        editdiv.on('dblclick', '.header.inactive-selection, .footer.inactive-selection, .cover-overlay', processHeaderFooterEdit);

        // user defined fields
        editdiv.on('mouseenter', '.field[class*=user-field]', pageLayout.createTooltipForField);
        editdiv.on('mouseleave', '.field[class*=user-field]', pageLayout.destroyTooltipForField);

        // complex fields
        editdiv.on('mouseenter', '.complex-field', fieldManager.createHoverHighlight);
        editdiv.on('mouseleave', '.complex-field', fieldManager.removeHoverHighlight);

        // mouseup events can be anywhere -> binding to $(document)
        this.listenTo($(document), 'mouseup touchend', processMouseUpOnDocument);

        // register handler for changes of page settings
        this.on('change:pageSettings', updatePageDocumentFormatting);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            // remove event handler from document
            $(document).off('keypress', processKeypressOnDocument);

            changeTrack.destroy();
            drawingLayer.destroy();
            commentLayer.destroy();
            fieldManager.destroy();
            rangeMarker.destroy();
            remoteSelection.destroy();
            selection.destroy();
            pageLayout.destroy();
            searchHandler.destroy();
            spellChecker.destroy();
            listCollection.destroy();

            self = changeTrack = drawingLayer = commentLayer = fieldManager = rangeMarker = selection = remoteSelection = pageLayout = searchHandler = spellChecker = listCollection = null;
        });

    } // class Editor

    // exports ================================================================

    // derive this class from class EditModel
    return EditModel.extend({ constructor: Editor });

});
