/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/drawingLayer', [
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/text/utils/textutils',
    'io.ox/office/text/utils/operations',
    'io.ox/office/text/dom',
    'io.ox/office/text/position'
], function (TriggerObject, TimerMixin, DrawingFrame, AttributeUtils, Utils, Operations, DOM, Position) {

    'use strict';

    // class DrawingLayer =====================================================

    /**
     * An instance of this class represents the model for all drawings in the
     * drawing layer of the edited document.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {TextApplication} app
     *  The application instance.
     *
     * @param {HTMLElement|jQuery} rootNode
     *  The root node of the document. If this object is a jQuery collection,
     *  uses the first node it contains.
     *
     * @param {TextModel} model
     *  The text model instance.
     */
    function DrawingLayer(app, rootNode) {

        var // self reference
            self = this,
            // a list of all drawings (jQuery) in the main document in the drawing layer (model)
            drawings = [],
            // a counter for drawings in the margins (model)
            marginCounter = 0,
            // the page layout
            pageLayout = null,
            // the page styles of the document
            pageAttributes = null,
            // a place holder ID for absolute positioned drawings
            placeHolderDrawingID = 1000,
            // the text model object
            model = null,
            // the drawing styles object
            drawingStyles = null,
            // whether the OX Text drawing layer is active (disable this for testing
            // or performance reasons)
            isActive = true;

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Adding one drawing into the drawing collections (the model).
         *
         * @param {HTMLElement|jQuery} drawing
         *  One drawing node that is positioned absolutely.
         *
         * @param {Boolean} isMarginalNode
         *  Whether the drawing is inside header or footer
         *
         * @returns {Number}
         *  The number of drawings in the main document.
         */
        function addIntoDrawingModel(drawing, isMarginalNode) {

            if (isMarginalNode) {
                marginCounter++;
            } else {
                // inserting drawing into the drawings list
                drawings.push($(drawing));
            }

            return drawings.length;
        }

        /**
         * Removing one drawing from the drawing collections (the model).
         *
         * @param {HTMLElement|jQuery} drawing
         *  One drawing node.
         *
         * @param {Boolean} isMarginalNode
         *  Whether the drawing is inside header or footer
         *
         * @returns {Number}
         *  The number of drawings in the main document.
         */
        function removeFromDrawingModel(drawing, isMarginalNode) {

            if (isMarginalNode) {
                marginCounter--;
            } else {
                // remove drawing from the drawings list
                if (!_.isEmpty(drawings)) {
                    // ... and finally removing the drawing from the collector 'drawings'
                    drawings = _.filter(drawings, function (node) {
                        return Utils.getDomNode(node) !== Utils.getDomNode(drawing);
                    });
                }
            }

            return drawings.length;
        }

        /**
         * Initializing page attributes needed for calculation of page breaks.
         * In some scenarios, like ex. loading from local storage,
         * initialPageBreaks method is not called, and variables are not initialized for later usage.
         *
         */
        function initializePageAttributes() {
            pageAttributes = model.getStyleCollection('page').getElementAttributes(rootNode);
        }

        /**
         * Returns whether a drawing overlaps with the page content node. In this case
         * it is necessary to create a space maker element below the drawing. If the drawing
         * is completely located in the border region, no space maker element is required.
         *
         * @param {Object} drawingAttributes
         *  The explicit attributes at the drawing node.
         */
        function drawingOverlapsPageContent(drawingAttrs) {

            // drawingAttrs: {"left":0,"top":0,"width":3970,"height":2980,
            // "name":"Grafik 1","description":"","flipH":true,"flipV":false,"replacementData":"","imageUrl":"word/media/image1.jpeg",
            // "imageData":"","cropLeft":0,"cropRight":0,"cropTop":0,"cropBottom":0,
            // "marginTop":0,"marginBottom":0,"marginLeft":317,"marginRight":317,
            // "borderLeft":{"style":"none"},"borderRight":{"style":"none"},"borderTop":{"style":"none"},"borderBottom":{"style":"none"},
            // "inline":false,"anchorHorBase":"page","anchorHorAlign":"offset","anchorHorOffset":4000,"anchorVertBase":"page",
            // "anchorVertAlign":"offset","anchorVertOffset":6000,"textWrapMode":"topAndBottom","textWrapSide":"both"}

            if (!drawingAttrs) { return false; }

            // pageAttributes: {"width":21000,"height":29700,"marginLeft":2499,"marginRight":2499,"marginTop":2499,"marginBottom":2000}
            if (!pageAttributes) {
                initializePageAttributes();
            }

            // drawing in top page frame
            if ((drawingAttrs.anchorVertBase === 'page') && (drawingAttrs.anchorVertOffset + drawingAttrs.height <= pageAttributes.page.marginTop)) { return false; }
            // drawing in left page frame
            if ((drawingAttrs.anchorHorBase === 'page') && (drawingAttrs.anchorHorOffset + drawingAttrs.width <= pageAttributes.page.marginLeft)) { return false; }
            // drawing in right page frame
            if ((drawingAttrs.anchorHorBase === 'page') && (drawingAttrs.anchorHorOffset >= pageAttributes.page.width - pageAttributes.page.marginRight)) { return false; }

            // TODO: Drawing overlaps with header and footer -> also no space maker element required
            // TODO: anchorVertBase can also be 'column'
            // TODO: anchorHorBase can be any other type

            return true;
        }

        /**
         * Returns whether the passed 'textWrapMode' attribute allows to wrap the
         * text around the drawing.
         *
         * @param {String} textWrapMode
         *  The text wrap mode of a drawing element
         */
        function isTextWrapped(textWrapMode) {

            var // values for the 'textWrapMode' attribute allowing to wrap the text around the drawing
                WRAPPED_TEXT_VALUES = /^(square|tight|through)$/;

            return WRAPPED_TEXT_VALUES.test(textWrapMode);
        }

        /**
         * Inserting the space maker node below the absolutely positioned drawing.
         *
         * @param {HTMLElement|jQuery} targetNode
         *  The current target node for the logical positions
         *
         * @param {Number[]} position
         *  The logical position, at which the space maker element needs to be inserted.
         *
         * @param {jQuery} drawing
         *  The drawing node, already jQuerified.
         *
         * @param {Object} drawingAttributes
         *  The explicit drawing attributes at the drawing node.
         *
         * @param {Object} lineAttributes
         *  The explicit line (border) attributes at the drawing node.
         *
         * @param {Number} zoomFactor
         *  The current zoom factor in percent.
         */
        function insertDrawingSpaceMaker(targetNode, position, drawing, drawingAttributes, lineAttributes, zoomFactor) {

            var // the space maker node below the drawing
                spaceMakerNode = null,
                // the width of the space maker
                width = 0,
                // the height of the space maker
                height = 0,
                // the paragraph element, parent of the drawing
                paragraph = $(Position.getParagraphElement(targetNode, _.initial(position))), // not drawing.parent(), because this is the drawing layer node
                // the width of the paragraph in px
                paraWidth = paragraph.outerWidth(true),
                // the zoomFactor used for calculations
                zoomValue = zoomFactor / 100,
                // the left distance of the paragraph to the root node (taking care of tables)
                paragraphOffset = Position.getPixelPositionToRootNodeOffset(targetNode, paragraph, zoomFactor),
                // the distance from left border of paragraph to the left border of the drawing in pixel
                leftDistance = Math.round((drawing.position().left - paragraphOffset.x) / zoomValue),
                // the distance from right border of the drawing to the right border of the paragraph in pixel
                rightDistance = Math.round((paragraphOffset.x / zoomValue) + paraWidth - ((drawing.position().left / zoomValue) + drawing.outerWidth(true))),
                // the distance from top border of paragraph to the top border of the drawing in pixel
                topDistance = Math.round((drawing.position().top - paragraphOffset.y) / zoomValue),
                // text wrapping side (left/right/none)
                wrapMode = 'none',
                // the id of the drawing (required for restoring from local storage)
                drawingID = drawing.attr('data-drawingID'),
                // whether the target node is a footer
                isFooterNode = DOM.isFooterNode(targetNode),
                // the old and the new height of a footer node
                oldHeight = 0, newHeight = 0, newBottomOffset = 0,
                // the change of height of the footer node after inserting a drawing space maker node
                spaceMakerShift = 0;

            if (drawingAttributes && isTextWrapped(drawingAttributes.textWrapMode)) {
                switch (drawingAttributes.textWrapSide) {
                case 'left':
                    wrapMode = 'left';
                    break;
                case 'right':
                    wrapMode = 'right';
                    break;
                case 'both':
                case 'largest':
                    // no support for 'wrap both sides' in CSS, default to 'largest'
                    wrapMode = (leftDistance >= rightDistance) ? 'left' : 'right';
                    break;
                default:
                    Utils.warn('insertDrawingSpaceMaker(): invalid text wrap side: ' + drawingAttributes.textWrapSide);
                    wrapMode = 'none';
                }
            } else {
                // text does not float beside drawing
                wrapMode = 'none';
            }

            // calculating the width of the space maker
            // -> outerWidth already contains the border width, although it is drawing with canvas. But an additional
            //    margin was set to the drawing before, representing the border.
            switch (wrapMode) {
            case 'none':
                width = paraWidth;
                break;
            case 'left':
                width = drawing.outerWidth(true) + rightDistance;
                break;
            case 'right':
                width = drawing.outerWidth(true) + leftDistance;
                break;
            }

            // setting the height of the space maker node
            height = drawing.outerHeight(true);

            // it might be necessary to reduce the height, if the drawing is above the paragraph
            if (topDistance < 0) { height += topDistance; }

            // inserting div element to create space for the drawing
            spaceMakerNode = $('<div>').addClass('float ' + ((wrapMode === 'left') ? 'right ' : 'left ') + DOM.DRAWING_SPACEMAKER_CLASS)
                                   .css('position', 'relative')
                                   .height(height + 10)  // TODO: Adding 10 px is not precise enough
                                   .width(width + 10)
                                   .attr('data-drawingID', drawingID);  // Adding the drawing id (required for local storage)

            // saving the old height of the footer node
            if (isFooterNode) { oldHeight = Utils.round($(targetNode).height(), 1); }

            if (_.last(position) === 0) {
                // inserting at the beginning of the paragraph, no empty span before
                paragraph.prepend(spaceMakerNode);
            } else {
                // splitting text span
                spaceMakerNode.insertAfter(model.prepareTextSpanForInsertion(position, null, targetNode));
            }

            // if the drawing is inside the footer, the vertical alignment relative to the bottom need to be updated
            if (isFooterNode) {
                newHeight = Utils.round($(targetNode).height(), 1);
                if (newHeight !== oldHeight) {
                    spaceMakerShift = newHeight - oldHeight;
                    newBottomOffset = Math.max(Utils.convertCssLength(drawing.css('bottom'), 'px', 1) + spaceMakerShift, 0);
                    drawing.css({ bottom: newBottomOffset });
                    // saving shift at the drawing node so that it can be used, when the space maker node is removed
                    // and when the bottom offset is set in 'this.setAbsoluteDrawingPosition'.
                    drawing.data(DOM.DRAWING_SPACEMAKER_SHIFT, spaceMakerShift);
                }
            }

            // registering the space maker at the absolute positioned drawing, but not inside header or footer
            if (!DOM.isHeaderOrFooter(targetNode)) {
                drawing.data(DOM.DRAWING_SPACEMAKER_LINK, Utils.getDomNode(spaceMakerNode));
            }
        }

        /**
         * Sorting the drawings in the order from top to bottom in the document. This is not
         * necessarily the order inside the DOM.
         *
         * @param {jQuery[]} drawings
         *  The list with all drawings in the drawing layer.
         */
        function sortFromTopToBottomOnOnePage(sortDrawings) {
            return _.sortBy(sortDrawings, function (drawing) {
                var attrs = AttributeUtils.getExplicitAttributes(drawing);
                return (attrs && attrs.drawing && attrs.drawing.anchorVertOffset) || 0;
            });
        }

        /**
         * Removing the space maker node that is assigned to a specified drawing node.
         * Additionally it might be necessary to merge the neighboring text nodes
         *
         * @param {Node} drawingSpaceMaker
         *  The space maker element of a drawing.
         */
        function removeDrawingSpaceMaker(drawingSpaceMaker) {

            if (!drawingSpaceMaker) { return; }

            var prev = drawingSpaceMaker.previousSibling,
                tryToMerge = prev && DOM.isTextSpan(prev);
            $(drawingSpaceMaker).remove();
            if (tryToMerge) { Utils.mergeSiblingTextSpans(prev, true); }
        }

        /**
         * Collecting all absolutely positioned drawings that are located inside the
         * margin specified by the update node, whose change triggered the update of
         * absolute positioned drawings.
         *
         * @param {jQuery[]} updateNode
         *  The node, whose change triggered the update of absolute drawings. This node
         *  is not necessarily inside an activated node, because this process could be
         *  triggered by a remote client or with an undo operation.
         *
         * @returns {Object}
         *  An object with the two properties 'drawings' and 'marginals'.
         *  The 'drawings' property contains a list with all absolute positioned drawings
         *  inside the specified margin. Inside this margin, the drawings are sorted in
         *  vertical direction from top to bottom. Drawings in the template margins are
         *  not part of this collection.
         *  The 'marginals' property contains a list of all header or footer nodes, whose
         *  drawings are in the list of the 'drawings'. In this case, it contains only
         *  the marginal node corresponding to the specified updateNode.
         */
        function getAllDrawingsFromSelectedMargin(updateNode) {

            var // the currently activated root node
                activeRootNode = null,
                // the text drawing layer inside the margin
                marginalDrawingLayer = null,
                // the collector for all absolute drawings
                marginalDrawingsArray = [],
                // the header or footer node corresponding to the specified update node
                marginalRoot = updateNode && DOM.isMarginalNode(updateNode) && DOM.getClosestMarginalTargetNode(updateNode),
                // a collector for all affected header or footer nodes containing collected drawings
                allMarginalNodes = [];

            // using active marginal node, if the update node cannot be used
            if (!marginalRoot) {
                activeRootNode = model.getCurrentRootNode();
                if (activeRootNode && DOM.isHeaderOrFooter(activeRootNode)) { marginalRoot = activeRootNode; }
            }

            if (marginalRoot) {
                marginalDrawingLayer = marginalRoot.children(DOM.DRAWINGLAYER_NODE_SELECTOR);
                if (marginalDrawingLayer.length > 0) {
                    _.each(marginalDrawingLayer.children(DrawingFrame.NODE_SELECTOR), function (drawing) { marginalDrawingsArray.push($(drawing)); });
                    if (marginalDrawingsArray.length > 1) {
                        marginalDrawingsArray = sortFromTopToBottomOnOnePage(marginalDrawingsArray);
                    }
                }
                allMarginalNodes.push(marginalRoot);
            }

            return { drawings: marginalDrawingsArray, marginals: allMarginalNodes };
        }

        /**
         * Collecting all absolutely positioned drawings in all margins. But every id
         * is only handled once. Not collected are drawings inside the template margins,
         * because in the template margins the positions cannot be calculated. The pixel
         * API fails in the template folder.
         *
         * @returns {Object}
         *  An object with the two properties 'drawings' and 'marginals'.
         *  The 'drawings' property contains a list with all absolute positioned drawings
         *  inside all margins, that are displayed in the document. Inside each margin, the
         *  drawings are sorted in vertical direction from top to bottom. Drawings in the
         *  template margins are not part of this collection.
         *  The 'marginals' property contains a list of all header or footer nodes, whose
         *  drawings are in the list of the 'drawings'.
         */
        function getAllDrawingsFromMargins() {

            var // the top level content nodes under div.page
                allContentNodes = rootNode.find(DOM.HEADER_WRAPPER_SELECTOR + ',' + DOM.FOOTER_WRAPPER_SELECTOR + ',' + DOM.PAGECONTENT_NODE_SELECTOR),
                // all text drawing layer that are inside headers or footers, but not in the templates
                allTextDrawingLayer = allContentNodes.find(DOM.HEADER_SELECTOR + ' > ' + DOM.DRAWINGLAYER_NODE_SELECTOR + ', ' + DOM.FOOTER_SELECTOR + ' > ' + DOM.DRAWINGLAYER_NODE_SELECTOR),
                // the collector for all absolute drawings
                marginalDrawingsArray = [],
                // a collector for all container ids
                allContainerIds = [],
                // a collector for all affected header or footer nodes containing collected drawings
                allMarginalNodes = [];

            if (allTextDrawingLayer.length > 0) {

                _.each(allTextDrawingLayer, function (drawingLayer) {

                    var oneLayerDrawings = null,
                        containerId = DOM.getTargetContainerId(drawingLayer.parentNode); // collecting every id only once

                    if (!_.contains(allContainerIds, containerId)) { // collecting drawings for every id only once

                        oneLayerDrawings = [];
                        _.each($(drawingLayer).children(DrawingFrame.NODE_SELECTOR), function (drawing) {
                            oneLayerDrawings.push($(drawing));
                        });
                        if (oneLayerDrawings.length > 1) {
                            oneLayerDrawings = sortFromTopToBottomOnOnePage(oneLayerDrawings);
                        }
                        marginalDrawingsArray = marginalDrawingsArray.concat(oneLayerDrawings);

                        allContainerIds.push(containerId); // not collecting each ID more than once, if forceAll is true
                        allMarginalNodes.push(drawingLayer.parentNode); // collecting all headers or footers
                    }
                });
            }

            return { drawings: marginalDrawingsArray, marginals: allMarginalNodes };
        }

        /**
         * Refreshing the links between a drawing from the drawing layer and its
         * place holder node and its space maker node.
         *
         * @param {Node|jQuery} absoluteDrawing
         *  The absolute positioned drawing in the drawing layer node.
         *
         * @param {Node|jQuery} contentNode
         *  The node, that contains the place holder nodes. This can be the page
         *  content node or any footer or header node.
         */
        function refreshLinkForAbsoluteDrawing(absoluteDrawing, contentNode) {

            var // the jQuery version of the drawing
                $absoluteDrawing = $(absoluteDrawing),
                // the drawing id of the drawing in the drawing layer
                drawingID = $absoluteDrawing.attr('data-drawingID'),
                // the selector string for the search for place holder nodes and space maker nodes
                selectorString = '[data-drawingID=' + drawingID + ']',
                // finding the place holder and optionally the space maker node, that are located inside the page content node
                foundNodes = $(contentNode).find(selectorString),
                // whether the corresponding place holder node was found (this is required)
                placeHolderFound = false;

            if (_.isString(drawingID)) { drawingID = parseInt(drawingID, 10); }

            // updating the value for the global drawing ID, so that new drawings in drawing layer get an increased number.
            if (_.isNumber(drawingID) && (drawingID >= placeHolderDrawingID)) { placeHolderDrawingID = drawingID + 1; }

            if (foundNodes.length > 0) {

                _.each(foundNodes, function (oneNode) {

                    if (DOM.isDrawingPlaceHolderNode(oneNode)) {
                        // creating links in both directions
                        $absoluteDrawing.data(DOM.DRAWINGPLACEHOLDER_LINK, oneNode);
                        $(oneNode).data(DOM.DRAWINGPLACEHOLDER_LINK, Utils.getDomNode(absoluteDrawing));
                        placeHolderFound = true;
                    } else if (DOM.isDrawingSpaceMakerNode(oneNode)) {
                        // there might be a space maker node with this drawingID, too
                        // -> but not every drawing has a space maker node
                        // creating links in both directions
                        $absoluteDrawing.data(DOM.DRAWING_SPACEMAKER_LINK, oneNode);
                    }
                });

                if (!placeHolderFound) {
                    Utils.error('DrawingLayer.refreshLinkForAbsoluteDrawing(): failed to find place holder node with drawing ID: ' + drawingID);
                }

            } else {
                Utils.error('DrawingLayer.refreshLinkForAbsoluteDrawing(): failed to find place holder node with drawingID: ' + drawingID);
            }

        }

        /**
         * Updating the space maker element below the drawings in the drawing layer.
         * This function is called very often and is therefore performance critical.
         * Also needs to update the vertical pixel position of the absolute drawings,
         * because this changes, if the drawing changes the page.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.forceUpdateAllDrawings=false]
         *      If set to true, all drawing positions are updated, even if the page
         *      did not change. This is especially necessary after loading the
         *      document.
         */
        var updateAbsoluteDrawings = Utils.profileMethod('DrawingLayer.updateAbsoluteDrawings(): ', function (options) {

            var // setting zoom factor, keeping percentage
                zoomFactor = app.getView().getZoomFactor(),
                // a collector for all page numbers
                allPageNumbers = [],
                // a helper object for all drawing nodes
                allSortedDrawings = {},
                // an ordered collection of the absolute positioned drawings
                sortedDrawings = [],
                // the key name for saving the current page number at the drawing node
                dataPageKey = 'currentPageNumber',
                // check if a processing node is registered, so that only following drawings
                currentNode = model.getCurrentProcessingNode(),
                // setting a page number until which the drawings need to be updated
                startPageNumber = 1,
                // whether the update of all drawings is forced
                forceUpdateAllDrawings =  Utils.getBooleanOption(options, 'forceUpdateAllDrawings', false),
                // an array with all absolutely positioned drawings in a currently activated header or footer
                marginalDrawingsArray = null,
                // an array with all header and footer nodes, that are updated in this update process
                marginalsArray = null,
                // an object with properties 'drawings' and 'marginals' for all drawings and header/footer node
                marginalDrawingsCollector = null,
                // whether marginal drawings also need to be updated
                updateDrawingsInMarginalNode = false;

            if (!pageLayout) { pageLayout = model.getPageLayout(); }

            // header-footer handling
            if (forceUpdateAllDrawings) {
                // collecting all affected marginal drawing nodes
                marginalDrawingsCollector = getAllDrawingsFromMargins();
            } else if (DOM.isMarginalNode(currentNode)) {
                // collecting all marginal drawing nodes affected defined by the current node
                marginalDrawingsCollector = getAllDrawingsFromSelectedMargin(currentNode);
            }

            marginalDrawingsArray = (marginalDrawingsCollector && marginalDrawingsCollector.drawings) || [];
            marginalsArray = (marginalDrawingsCollector && marginalDrawingsCollector.marginals) || [];
            updateDrawingsInMarginalNode = marginalDrawingsArray && marginalDrawingsArray.length > 0;

            // Performance: Update only following pages
            if (!currentNode) {
                startPageNumber = pageLayout.getPageNumber(currentNode);
                // was is successful to determine the start page number?
                startPageNumber = startPageNumber || 1;

                if (startPageNumber > 1) { startPageNumber--; }
            }

            // no updates and space maker for drawings in header or footer
            // TODO: This needs to be modified, if header and footer become writable
            sortedDrawings = _.filter(drawings, function (drawing) {
                return !drawing.hasClass(DOM.TARGET_NODE_CLASSNAME);
            });

            // sorting the drawings corresponding to their appearance in the document
            _.each(sortedDrawings, function (drawing) {
                // on which page is the place holder node?
                var page = pageLayout.getPageNumber(DOM.getDrawingPlaceHolderNode(drawing));
                // was is successful to determine the start page number?
                if (!page) { page = 0; }

                // updating only following drawings
                if (page >= startPageNumber) {
                    if (!_.contains(allPageNumbers, page)) { allPageNumbers.push(page); }
                    if (!allSortedDrawings[page]) { allSortedDrawings[page] = []; }
                    allSortedDrawings[page].push(drawing);
                    // saving the page number at the drawing
                    $(drawing).data(dataPageKey, page);
                }
            });

            // sorting the pages
            allPageNumbers.sort(function (a, b) { return a - b; });

            // reset the sorted drawings helper array
            sortedDrawings = [];

            // sorting all drawings on one page corresponding to their vertical offset
            _.each(allPageNumbers, function (page) {
                if (allSortedDrawings[page].length > 1) {
                    allSortedDrawings[page] = sortFromTopToBottomOnOnePage(allSortedDrawings[page]);  // order is now from top to down in the visibility
                }
                // concatenating all drawings into one array
                sortedDrawings = sortedDrawings.concat(allSortedDrawings[page]);
            });

            // check, if a drawing needs a new vertical offset, caused by page change
            _.each(sortedDrawings, function (drawing) {
                self.setAbsoluteDrawingPosition(drawing, { forceUpdateAllDrawings: forceUpdateAllDrawings });
            });

            // also handling all drawings inside header or footer -> put them to the start of the sorted drawings collector
            if (updateDrawingsInMarginalNode) { sortedDrawings = marginalDrawingsArray.concat(sortedDrawings); }

            // removing the existing space maker nodes -> maybe they can be reused in the future?
            _.each(sortedDrawings, function (drawing) {
                var // the space maker node below the drawing
                    spaceMakerNode = DOM.getDrawingSpaceMakerNode(drawing),
                    // new bottom offset in pixel for drawings in footer
                    newBottomOffset = 0;

                if (spaceMakerNode) {

                    removeDrawingSpaceMaker(spaceMakerNode);
                    drawing.removeData(DOM.DRAWING_SPACEMAKER_LINK);

                    // fix for position of drawings in footer, that are aligned to 'bottom'
                    if (drawing.data(DOM.DRAWING_SPACEMAKER_SHIFT)) {
                        newBottomOffset = Math.max(Utils.convertCssLength(drawing.css('bottom'), 'px', 1) - drawing.data(DOM.DRAWING_SPACEMAKER_SHIFT), 0);
                        drawing.css({ bottom: newBottomOffset });
                        drawing.removeData(DOM.DRAWING_SPACEMAKER_SHIFT);
                    }
                }
            });

            // iterate over all drawings -> is a space maker node element required below the drawing?
            _.each(sortedDrawings, function (drawing) {

                var // the merged attributes of the drawing
                    allAttrs = drawingStyles.getElementAttributes(drawing),
                    // the drawing attributes set at the drawing
                    drawingAttrs = allAttrs.drawing,
                    // the line attributes set at the drawing
                    lineAttrs = allAttrs.line || {},
                    // the horizontal and vertical offset of the drawing relative to the page
                    pos = null,
                    // the page info object for a specified pixel position
                    pageInfo = null,
                    // the page number at which the drawing is positioned
                    // page = drawing.data(dataPageKey);
                    // the current target node for the logical positions
                    targetNode = rootNode,
                    // whether the drawing is inside header or footer
                    isMarginalDrawing = DOM.isDrawingInsideMarginalDrawingLayerNode(drawing);

                // immediately remove the current page number
                drawing.removeData(dataPageKey);

                if (drawingOverlapsPageContent(drawingAttrs) || isMarginalDrawing) {

                    if (updateDrawingsInMarginalNode && isMarginalDrawing) {
                        targetNode = drawing.parent().parent();
                    }

                    // -> update or create the space maker element
                    // calculating pixel position relative to the page
                    pos = Position.getPixelPositionToRootNodeOffset(targetNode, drawing, zoomFactor);

                    try {
                        // calculating the logical position at the specified position (first position in line)
                        pageInfo = Position.getPositionFromPagePixelPosition(targetNode, pos.x, pos.y, zoomFactor, { getFirstTextPositionInLine: true });
                    } catch (ex) {}

                    if (pageInfo && pageInfo.pos) {
                        insertDrawingSpaceMaker(targetNode, pageInfo.pos, drawing, drawingAttrs, lineAttrs, zoomFactor);
                    }
                }
            });

            // transferring the changes into the header or footer templates, so that all header or footer nodes can be updated.
            if (marginalsArray.length > 0) { pageLayout.updateDataFromFirstNodeInDoc(marginalsArray); }

            Utils.log('Number of drawings: ' + sortedDrawings.length);
        });

        /**
         * After load from local storage the links between drawings and drawing place holders
         * need to restored.
         * Additionally the global 'drawings' collector is filled with the drawings.
         * This is only done for the drawings that are not located inside the header or footer.
         * Absolute positioned drawings inside a header or footer are only counted in the global
         * variable 'marginCounter'.
         */
        function refreshDrawingLayer() {

            // creating all links between drawing in drawing layer and the corresponding
            // drawing place holder
            // -> for this the data-drawingid is required
            // This process is not done for margins, where the links are not used. Instead
            // the number of absolute drawings in margins is counted.

            var // the page content node
                pageContentNode = DOM.getPageContentNode(model.getNode()),
                // collecting all drawings in the drawing layer
                allDrawings = self.getDrawingLayerNode().children('div.drawing'),
                // a list of all header and footer nodes
                allMargins = pageLayout.getHeaderFooterPlaceHolder().children();

            // reset model
            drawings = [];
            marginCounter = 0;

            // filling the list of absolute positioned drawings
            _.each(allDrawings, function (oneDrawing) {
                drawings.push($(oneDrawing));
            });

            if (drawings.length > 0) {
                _.each(drawings, function (absoluteDrawing) {
                    refreshLinkForAbsoluteDrawing(absoluteDrawing, pageContentNode);
                });
            }

            // counting the absolute positioned drawings in header or footer
            _.each(allMargins, function (margin) {

                var // the drawings in one drawing layer in one header or footer
                    allDrawings = null,
                    // the drawing layer inside the header or footer
                    drawingLayer = $(margin).children(DOM.DRAWINGLAYER_NODE_SELECTOR);

                if (drawingLayer.length > 0) {

                    allDrawings = drawingLayer.children('div.drawing');

                    // only counting the number of absolute drawings in header and footer
                    // -> no updating of links required
                    marginCounter += allDrawings.length;
                }
            });
        }

        /**
         * Drawing layer specific handling after reloading the document (for example after
         * canceling a long running action).
         */
        function handleDocumentReload() {
            refreshDrawingLayer();
            updateDrawings();
        }

        /**
         * Function that handles the code to be executed after receiving the import success
         * event.
         */
        function handleImportSuccess() {
            if (model.isLocalStorageImport()) { refreshDrawingLayer(); }
            updateDrawings({ forceUpdateAllDrawings: true });
        }

        /**
         * Triggering the general update function for absolutely positioned drawings after
         * checking, if there are absolutely positioned drawings.
         */
        function updateDrawings(options) {
            if (!self.isEmpty()) { updateAbsoluteDrawings(options); }
        }

        // methods ------------------------------------------------------------

        /**
         * Whether the drawing layer is active. This can be disabled for testing
         * or performance reasons.
         *
         * @returns {Boolean}
         *  Whether the drawing layer is active.
         */
        this.isActive = function () {
            return isActive;
        };

        /**
         * Whether the document contains absolutely positioned drawings in the
         * main document (those are collected inside the drawings container).
         *
         * @returns {Boolean}
         *  Whether the document contains at least one absolutely positioned
         *  drawing.
         */
        this.isEmptyMainDoc = function () {
            return drawings.length === 0;
        };

        /**
         * Whether the document contains absolutely positioned drawings in the
         * margins (not collected in drawings container).
         *
         * @returns {Boolean}
         *  Whether the document contains at least one absolutely positioned
         *  drawing in the margins.
         */
        this.isEmptyMargin = function () {
            return marginCounter === 0;
        };

        /**
         * Whether the document contains absolutely positioned drawings in the
         * margins or in the main document.
         *
         * @returns {Boolean}
         *  Whether the document contains at least one absolutely positioned
         *  drawing in the margins or in the main document.
         */
        this.isEmpty = function () {
            return (marginCounter === 0 && drawings.length === 0);
        };

        /**
         * Whether the document contains absolutely positioned drawings
         * inside the page content node. This excludes all drawings, that
         * are located in the header or footer.
         *
         * @returns {Boolean}
         *  Whether the document contains at least one absolutely positioned
         *  drawing in the page content node.
         */
        this.containsDrawingsInPageContentNode = function () {

            var // the drawing layer node inside the page div.page
                drawingLayerNode = self.returnDrawingLayerNode();

            return drawingLayerNode && drawingLayerNode.length > 0 && drawingLayerNode.children().length > 0;
        };

        /**
         * Provides the array with all drawings in the drawing layer.
         *
         * @returns {jQuery[]}
         *  The list with all drawings in the drawing layer.
         */
        this.getDrawings = function () {
            return drawings;
        };

        /**
         * Provides a unique ID for the drawings in the drawing layer and
         * its placeholder elements.
         *
         * @returns {Number}
         *  A unique id.
         */
        this.getNextDrawingID = function () {
            return placeHolderDrawingID++;
        };

        /**
         * Setting the 'left' and the 'top' css values for absolutely
         * positioned drawings. This function is called, if drawings
         * are inserted into the drawing layer ('newValues' is true) or
         * if they are removed from the drawing layer('doClean' is true)
         * or during update of the positions (called from
         * updateAbsoluteDrawings). An update of the vertical
         * position can be necessary, if a paragraph containing a drawing
         * place holder element moves to another page. In this case the
         * absolute positioned drawing has to change its position, too.
         *
         * @param {jQuery} drawingNode
         *  One drawing node that is positioned absolutely. For convenience
         *  reasons this can also be the place holder node.
         *
         * @param {Number} leftValue
         *  The left page offset in pixel.
         *
         * @param {Number} topValue
         *  The top page offset in pixel. This is only the offset to
         *  the current page, not to the div.page element.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.newValues=false]
         *      If set to true, the drawing was inserted, set 'toPage' or modified
         *      in size or position. In this case the data values stored at the
         *      element need to be updated.
         *  @param {Boolean} [options.doClean=false]
         *      If set to true, the drawing was removed from the drawing layer. In
         *      this case the data values stored at the element need to be deleted.
         *  @param {Boolean} [options.useBottomOffset=false]
         *      If set to true, the top value for the vertical offset is interpreted
         *      as bottom offset. This is used for positions in footers.
         *  @param {Boolean} [options.forceUpdateAllDrawings=false]
         *      If set to true, all drawing positions are updated, even if the page
         *      did not change. This is especially necessary after loading the
         *      document.
         *  @param {leftValue} [options.leftValue=0]
         *      The left page offset in pixel.
         *  @param {topValue} [options.topValue=0]
         *      The top page offset in pixel.
         *  @param {jQuery} [options.targetNode]
         *      An optional target node as an alternative to the page div.page. This
         *      is used for header or footer nodes.
         */
        this.setAbsoluteDrawingPosition = function (drawingNode, options) {

            var // whether the drawing offsets shall be cleaned
                doClean = Utils.getBooleanOption(options, 'doClean', false),
                // whether the values have changed (called from drawingStyles.updateDrawingFormatting)
                newValues = Utils.getBooleanOption(options, 'newValues', false),
                // whether the update of all drawings is forced
                forceUpdateAllDrawings =  Utils.getBooleanOption(options, 'forceUpdateAllDrawings', false),
                // the left page offset in pixel
                leftValue = Utils.getNumberOption(options, 'leftValue', 0),
                // the top page offset in pixel
                topValue = Utils.getNumberOption(options, 'topValue', 0),
                // the page number, at which the drawing place holder is located
                pageNumber = 1,
                // the vertical and horizontal offsets relative to the page
                horizontalPageOffset = 0, verticalPageOffset = 0,
                // the vertical offset in the document
                documentOffset = 0,
                // the drawing in the drawing layer
                drawing = DOM.isDrawingPlaceHolderNode(drawingNode) ? $(DOM.getDrawingPlaceHolderNode(drawingNode)) : drawingNode,
                // an optional target node in header or footer
                inHeaderOrFooter = options && options.targetNode && options.targetNode.length > 0,
                // whether the topValue needs to be set as bottom offset (used in footers)
                useBottomOffset = Utils.getBooleanOption(options, 'useBottomOffset', false);

            if (newValues) {

                // called from drawingStyles.updateDrawingFormatting
                horizontalPageOffset = leftValue;
                verticalPageOffset = topValue;

                if (!pageLayout) { pageLayout = model.getPageLayout(); }

                // determining the page number of the drawing, after the document is loaded completely.
                // The page number is always 1 inside headers or footers.
                if (app.isImportFinished() && !inHeaderOrFooter) {
                    pageNumber = pageLayout.getPageNumber(DOM.getDrawingPlaceHolderNode(drawing));
                    drawing.data('pageNumber', pageNumber);
                }

                // saving the values at the drawing node
                drawing.data('verticalPageOffset', verticalPageOffset);
                drawing.data('horizontalPageOffset', horizontalPageOffset);

                documentOffset = (pageNumber === 1) ? verticalPageOffset : Position.getVerticalPagePixelPosition(rootNode, pageLayout, pageNumber, app.getView().getZoomFactor(), verticalPageOffset);

            } else if (doClean) {

                // removing the values at the drawing node
                drawing.removeData('verticalPageOffset');
                drawing.removeData('horizontalPageOffset');
                drawing.removeData('pageNumber');

                documentOffset = 0;
                horizontalPageOffset = 0;

            } else {
                // this is the standard update case, only drawing is defined
                // -> modifications are only necessary, if the page has changed
                if (!pageLayout) { pageLayout = model.getPageLayout(); }

                if (drawing.data('currentPageNumber')) {
                    pageNumber = drawing.data('currentPageNumber');
                } else {
                    pageNumber = pageLayout.getPageNumber(DOM.getDrawingPlaceHolderNode(drawing));
                }

                // if it failed to determine the page, simply do nothing (leave drawing where it is)
                if (!pageNumber) { return; }

                // page number not changed, nothing to do
                // -> but updating after first load is required, to be more precise!
                if (!forceUpdateAllDrawings && (pageNumber === drawing.data('pageNumber'))) { return; }

                drawing.data('pageNumber', pageNumber); // setting updated value

                horizontalPageOffset = drawing.data('horizontalPageOffset');
                verticalPageOffset = drawing.data('verticalPageOffset');

                documentOffset = (pageNumber === 1) ? verticalPageOffset : Position.getVerticalPagePixelPosition(rootNode, pageLayout, pageNumber, app.getView().getZoomFactor(), verticalPageOffset);
            }

            if (useBottomOffset) {
                if (drawing.data(DOM.DRAWING_SPACEMAKER_SHIFT)) { documentOffset += drawing.data(DOM.DRAWING_SPACEMAKER_SHIFT); }
                drawing.css({ bottom: documentOffset, top: '', left: horizontalPageOffset });
            } else {
                drawing.css({ top: documentOffset, left: horizontalPageOffset });
            }
        };

        /**
         * Shifting a drawing from the page content node into the drawing
         * layer node. This happens during loading the document or if a
         * drawing mode is switched from 'asCharacter' or 'toParagraph' to
         * 'toPage'.
         *
         * @param {jQuery} drawing
         *  One drawing node that is positioned absolutely.
         *
         * @param {jQuery} [target]
         *  An optional alternative for the destination of the drawing layer. If not
         *  specified, div.page needs to be the parent of the drawing layer. For
         *  headers and footers, it is also possible, that div.header or div.footer
         *  are the parent for the drawing layer.
         */
        this.shiftDrawingIntoDrawingLayer = function (drawing, target) {

            var // the drawing layer node for absolute positioned drawings (will be created, if required)
                drawingLayerNode = self.getDrawingLayerNode(target),
                // the place holder element for the drawing in the page content
                drawingPlaceHolder = $('<div>', { contenteditable: false }).addClass('inline ' + DOM.DRAWINGPLACEHOLDER_CLASS),
                // a drawing ID to connect drawing and drawingPlaceHoder node
                drawingID = self.getNextDrawingID(),
                // the string identifier for the target
                targetString = target ? $(target).attr('data-container-id') : null;

            // adding the target string to the drawing ID
            if (targetString) { drawingID = drawingID + '_' + targetString; }

            // creating and inserting the drawing place holder behind the drawing
            drawing.after(drawingPlaceHolder);

            // removing an offset element before the drawing, if there is one
            if (DOM.isOffsetNode(drawing.prev())) { drawing.prev().remove(); }

            // shifting the drawing itself into the drawing layer node
            drawingLayerNode.append(drawing);

            drawing.attr('data-drawingID', drawingID);
            drawingPlaceHolder.attr('data-drawingID', drawingID);

            // direct link between the drawing and drawing place holder (not used in header or footer)
            if (!DOM.isHeaderOrFooter(target)) {
                drawing.data(DOM.DRAWINGPLACEHOLDER_LINK, Utils.getDomNode(drawingPlaceHolder));
                drawingPlaceHolder.data(DOM.DRAWINGPLACEHOLDER_LINK, Utils.getDomNode(drawing));
            }

            // and finally registering it at the drawing layer handler (the model)
            // -> do not register drawings in header or footer, that are NOT located
            // inside 'div.header-footer-placeholder'. All other drawings are
            // 'throw-away' drawings.
            addIntoDrawingModel(drawing, DOM.isHeaderOrFooter(target));
        };

        /**
         * Shifting a drawing from the drawing layer node into the page content
         * node. This happens if a drawing mode is switched from 'toPage' to
         * 'asCharacter' or 'toParagraph'.
         *
         * @param {jQuery} drawing
         *  One drawing node that is no longer positioned absolutely.
         *  This can be the place holder node or the drawing in the
         *  drawing layer node for convenience reasons.
         */
        this.shiftDrawingAwayFromDrawingLayer = function (drawing) {

            var // the drawing layer node for absolute positioned drawings
                drawingLayerNode = self.getDrawingLayerNode(),
                // the drawing node in the drawing layer
                drawingNode = null,
                // the drawing place holder in the page content
                placeHolderNode = null,
                // the space maker node for the absolute positioned drawing
                spaceMakerNode = null,
                // whether the drawing is inside header or footer
                isMarginalNode = false;

            // supporting space holder node and drawing node in drawing layer
            if (DOM.isDrawingPlaceHolderNode(drawing)) {
                drawingNode = DOM.getDrawingPlaceHolderNode(drawing);
                placeHolderNode = drawing;
            } else if (DOM.isDrawingLayerNode($(drawing).parent())) {
                drawingNode = drawing;
                placeHolderNode = DOM.getDrawingPlaceHolderNode(drawing);
            } else {
                return null;
            }

            // checking for header and footer
            isMarginalNode = DOM.isMarginalNode($(placeHolderNode).parent());

            // ... removing the space maker node
            spaceMakerNode = DOM.getDrawingSpaceMakerNode(drawingNode);
            if (spaceMakerNode) {
                removeDrawingSpaceMaker(spaceMakerNode);
                $(drawingNode).removeData(DOM.DRAWING_SPACEMAKER_LINK);
            }

            // shifting the drawing behind the place holder node
            $(placeHolderNode).after(drawingNode);

            // ... removing the place holder node
            $(placeHolderNode).remove();
            $(drawingNode).removeData(DOM.DRAWINGPLACEHOLDER_LINK);
            $(drawingNode).removeAttr('data-drawingID');

            // delete drawing layer, if it is no longer required
            if (drawingLayerNode.length === 0) {
                // creating the drawing layer
                $(drawingLayerNode).remove();

                if (!isMarginalNode) {
                    // no more drawings in collector
                    drawings = [];
                }
            }

            // updating the model
            removeFromDrawingModel(drawing, isMarginalNode);
        };

        /**
         * Removing all drawings from the drawing layer, that have place holder nodes
         * located inside the specified node. The node can be a paragraph that will be
         * removed. In this case it is necessary, that the drawings in the drawing layer
         * are removed, too, and that the model is updated.
         * This is a simplified function, that simply removes the drawing nodes
         * located inside the drawing layer. It does not take care of space maker
         * nodes or place holder nodes. This is not necessary, because it is assumed,
         * that the specified node will be removed, too.
         *
         * @param {HTMLElement|jQuery} node
         *  The node, for which all drawings in the drawing layer will be removed. This
         *  assumes that the node contains place holder nodes. Only the drawings in the
         *  drawing layer will be removed. This function does not take care of place
         *  holder nodes and space maker nodes.
         */
        this.removeAllInsertedDrawingsFromDrawingLayer = function (node) {

            var // the collection of all place holder nodes
                allPlaceHolder = $(node).find(DOM.DRAWINGPLACEHOLDER_NODE_SELECTOR);

            // starting to remove the drawings, if there are place holder nodes
            if (allPlaceHolder.length > 0) {
                _.each(allPlaceHolder, function (placeHolderNode) {

                    var // the drawing node corresponding to the place holder node
                        drawingNode = DOM.getDrawingPlaceHolderNode(placeHolderNode),
                        // the optional space maker node corresponding to the drawing node
                        spaceMakerNode = null,
                        // whether the drawing place holder node is inside header or footer
                        isMarginalNode = false;

                    if (drawingNode) {

                        // whether the drawing place holder node is inside header or footer
                        isMarginalNode = DOM.isMarginalNode($(placeHolderNode).parent());

                        // ... removing the space maker node
                        spaceMakerNode = DOM.getDrawingSpaceMakerNode(drawingNode);

                        if (spaceMakerNode) {
                            removeDrawingSpaceMaker(spaceMakerNode);
                            $(drawingNode).removeData(DOM.DRAWING_SPACEMAKER_LINK);
                        }

                        // updating the model
                        removeFromDrawingModel(drawingNode, isMarginalNode);

                        // ... and removing the node in the drawing layer from the DOM
                        $(drawingNode).remove();
                    } else {
                        Utils.warn('removeAllInsertedDrawingsFromDrawingLayer(): failed to find drawing node for place holder node!');
                    }
                });
            }
        };

        /**
         * Removing one drawing element from the drawing layer node. The parameter
         * is the place holder node in the page content.
         *
         * @param {jQuery} placeHolderNode
         *  The place holder node in the page content
         *
         *  Optional parameters:
         *  @param {Boolean} [options.keepDrawingLayer=false]
         *      If set to true, the drawing in the drawing layer corresponding to the
         *      drawing place holder is NOT removed. This is for example necessary after
         *      a split of paragraph. The default is 'false' so that place holder node,
         *      space maker node and drawing in drawing layer are removed together.
         */
        this.removeFromDrawingLayer = function (placeHolderNode, options) {

            var // the drawing node in the drawing layer
                drawing = DOM.getDrawingPlaceHolderNode(placeHolderNode),
                // whether the drawings in the drawing layer shall not be removed
                // -> this is necessary after splitting a paragraph
                keepDrawingLayer = Utils.getBooleanOption(options, 'keepDrawingLayer', false),
                // the space maker node for the absolute positioned drawing
                spaceMakerNode = null,
                // whether the drawing is inside the margins
                isMarginalNode = DOM.isMarginalNode($(placeHolderNode).parent());

            if (keepDrawingLayer) {

                // removing only the place holder node
                $(placeHolderNode).removeData(DOM.DRAWINGPLACEHOLDER_LINK);
                $(placeHolderNode).remove();

            } else {

                // removing links to other nodes
                $(placeHolderNode).removeData(DOM.DRAWINGPLACEHOLDER_LINK);
                $(drawing).removeData(DOM.DRAWINGPLACEHOLDER_LINK);

                // safely release image data (see comments in BaseApplication.destroyImageNodes())
                app.destroyImageNodes(drawing);

                // ... removing the space maker node
                spaceMakerNode = DOM.getDrawingSpaceMakerNode(drawing);

                if (spaceMakerNode) {
                    removeDrawingSpaceMaker(DOM.getDrawingSpaceMakerNode(drawing));
                    $(drawing).removeData(DOM.DRAWING_SPACEMAKER_LINK);
                }

                // ... removing the drawing in the drawing layer
                $(drawing).remove();

                // ... removing the place holder node
                $(placeHolderNode).remove();

                // ... and finally updating the model
                removeFromDrawingModel(drawing, isMarginalNode);
            }
        };

        /**
         * After splitting a paragraph with the place holder in the new paragraph (that
         * was cloned before), it is necessary that the drawing in the drawing layer
         * updates its link to the place holder drawing in the new paragraph.
         * This is not necessary in headers or footers, where the links are not used.
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node.
         */
        this.repairLinksToPageContent = function (paragraph) {

            var // the function that is used to find the place holder nodes. If only a paragraph is updated
                // it is sufficient to search for children. In the case of a page content node, 'find' needs to be used
                searchFunction = DOM.isParagraphNode(paragraph) ? 'children' : 'find',
                // whether each single node needs to be checked
                checkMarginal = (searchFunction === 'find');

            if (!DOM.isMarginalNode(paragraph)) {
                _.each($(paragraph)[searchFunction](DOM.DRAWINGPLACEHOLDER_NODE_SELECTOR), function (placeHolder) {
                    var drawing = null;
                    if (!checkMarginal || !DOM.isMarginalNode(placeHolder)) {
                        drawing = DOM.getDrawingPlaceHolderNode(placeHolder);
                        $(drawing).data(DOM.DRAWINGPLACEHOLDER_LINK, placeHolder);
                    }
                });
            }

        };

        /**
         * Receiving the drawing layer node for absolute positioned drawings. It
         * is not created within this function. If it does not exist, null is
         * returned.
         *
         * @param {jQuery} [target]
         *  An optional alternative for the destination of the drawing layer. If not
         *  specified, div.page needs to be the parent of the drawing layer. For
         *  headers and footers, it is also possible, that div.header or div.footer
         *  are the parent for the drawing layer.
         *
         * @returns {jQuery|null} drawingLayerNode
         *  The drawing layer node, if is exists. Otherwise null.
         */
        this.returnDrawingLayerNode = function (target) {

            var // the document page node or the target in header or footer
                pageNode = (target && target.length > 0) ? target : model.getNode(),
                // the drawing layer node in the dom
                drawingLayerNode = pageNode.children(DOM.DRAWINGLAYER_NODE_SELECTOR);

            return (drawingLayerNode && drawingLayerNode.length > 0) ? drawingLayerNode : null;
        };

        /**
         * Receiving the drawing layer node for absolute positioned drawings.
         * It is created, if it does not exist yet.
         *
         * @param {jQuery} [target]
         *  An optional alternative for the destination of the drawing layer. If not
         *  specified, div.page needs to be the parent of the drawing layer. For
         *  headers and footers, it is also possible, that div.header or div.footer
         *  are the parent for the drawing layer.
         *
         * @returns {jQuery} drawingLayerNode
         *  The drawing layer node
         */
        this.getDrawingLayerNode = function (target) {

            var // the document page node or the target in header or footer
                pageNode = (target && target.length > 0) ? target : model.getNode(),
                // the drawing layer node in the dom
                drawingLayerNode = self.returnDrawingLayerNode(pageNode),
                // the page content node
                pageContentNode = null;

            // create drawing layer node, if necessary
            if (!drawingLayerNode || (drawingLayerNode && drawingLayerNode.length === 0)) {
                // the drawing layer needs to be created next to the page content node. In the case of
                // a header or footer node, it will be created directly below the element div.header
                // or div.footer. In the latter case, the drawing layer is created directly behind
                // the last paragraph.
                pageContentNode = (target && target.length > 0) ? DOM.getMarginalContentNode(pageNode) : DOM.getPageContentNode(pageNode);
                drawingLayerNode = $('<div>').addClass(DOM.DRAWINGLAYER_CLASS);
                if (_.browser.Firefox) {
                    //workaround for abs pos element, which get the "moz-dragger" but we dont want it
                    drawingLayerNode.attr('contenteditable', 'false');
                }
                // Adding the text drawing layer behind an optional footer wrapper
                if (!target && DOM.isFooterWrapper(pageContentNode.next())) { pageContentNode = pageContentNode.next(); }
                // inserting the text drawing layer node
                pageContentNode.after(drawingLayerNode);
            }

            return drawingLayerNode;
        };

        // initialization -----------------------------------------------------

        app.onInit(function () {

            app.onImportSuccess(handleImportSuccess);

            model = app.getModel();

            drawingStyles = model.getStyleCollection('drawing');

            pageLayout = model.getPageLayout();

            model.one('pageBreak:after', updateDrawings);

            model.on('update:absoluteElements', updateDrawings);

            model.on('document:reloaded', handleDocumentReload);
        });

    } // class DrawingLayer

    // export =================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: DrawingLayer });

});
