/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/text/components/field/complexField', [
    'io.ox/office/text/components/field/basefield',
    'io.ox/office/text/utils/textutils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/text/utils/operations',
    'io.ox/office/text/dom',
    'io.ox/office/text/position'
], function (BaseField, Utils, LocaleData, Operations, DOM, Position) {

    'use strict';

    // class ComplexField =====================================================

    /**
     * An instance of this class represents the model for all complex fields in the
     * edited document.
     *
     * @constructor
     *
     * @extends Field
     *
     * @param {TextApplication} app
     *  The application instance.
     */
    function ComplexField(app) {

        var // self reference
            self = this,
            // an object with target string as key and target node as value, the model
            allFields = {},
            // a counter for the complex field ID
            fieldID = 0,
            // the text model object
            model = null,
            // the selection object
            selection = null,
            // page layout object
            pageLayout = null,
            // number formatter object
            numberFormatter = null,
            // range marker object
            rangeMarker = null;

        // base constructors --------------------------------------------------

        BaseField.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Registering one complex field in the collection (the model).
         *
         * @param {HTMLElement|jQuery} field
         *  One complex field node.
         *
         * @param {String} id
         *  The unique id of the complex field node.
         *
         * @param {String} [target]
         *  The target, where the complex field is located.
         */
        function addIntoComplexFieldCollection(field, id, target) {

            var // the complex field node in jQuery format
                $field = $(field),
                // whether the complex field is inside header or footer
                // -> in this case it is useless to save the node itself, but saving
                //    the target allows to find the range with specific id fast
                isMarginalId = target && pageLayout.isIdOfMarginalNode(target);

            // adding the node also into the type specific collector
            allFields[id] = isMarginalId ? target : $field;
        }

        /**
         * Removing one complex field with specified id from the collection (the model).
         *
         * @param {String} id
         *  The unique id of the complex field node.
         */
        function removeFromComplexFieldCollection(id) {
            delete allFields[id];
        }

        /**
         * Finding in a header or footer specified by the target the complex field
         * with the specified id.
         *
         * @param {String} id
         *  The id string.
         *
         * @param {String} target
         *  The target string to identify the header or footer node
         *
         * @param {Number} [index]
         *  Cardinal number of target node in the document from top to bottom. If passed,
         *  forces to return node with that index in document, from top to bottom.
         *
         * @returns {jQuery|null}
         *  The range marker node with the specified id and type, or null, if no such
         *  range marker exists.
         */
        function getMarginalComplexField(id, target, index) {

            var // the specified header or footer node
                marginalNode = null,
                // the searched complex field with the specified id
                field = null;

            marginalNode = model.getRootNode(target, index);

            if (marginalNode) {
                field = _.find(marginalNode.find(DOM.COMPLEXFIELDNODE_SELECTOR), function (field) { return DOM.getComplexFieldId(field) === id; });
            }

            return field ? $(field) : null;
        }

        /**
         * Private method to mark all nodes that belong to complex field,
         * or, so to say, are inside range start and range end of a field.
         *
         * @param {HTMLElement|jQuery} marker
         *  One range marker node.
         *
         * @param {String} id
         *  Id of processed field.
         */
        function markComplexFieldForHighlighting(cxField, id) {
            var $node = $(cxField).next(),
                $helperNode;
            while ($node.length && !DOM.isRangeMarkerEndNode($node)) {
                $node.addClass(DOM.COMPLEXFIELDMEMBERNODE_CLASS + ' complex' + id).data('fieldId', id);
                $helperNode = $node.next();
                if (!$helperNode.length) {
                    $helperNode = $node.parent().next().children().first(); // if field is in more than one paragraph, search for range-end in next p
                    if (!$helperNode.length) {
                        Utils.warn('complexField.markComplexFieldForHighlighting(): rangeEnd not found!');
                        return;
                    } else {
                        $node = $helperNode;
                    }
                } else {
                    $node = $helperNode;
                }
            }
        }

        /**
         * Removing empty text spans between the range marker start node and the
         * following complex field node.
         *
         * @param {HTMLElement|jQuery} node
         *  The complex field node.
         */
        function removePreviousEmptyTextSpan(field) {

            var // the node preceding the complex field node
                precedingNode = $(field).prev();

            if (precedingNode.length > 0 && precedingNode.prev().length > 0 && DOM.isEmptySpan(precedingNode) && DOM.isRangeMarkerStartNode(precedingNode.prev())) {
                precedingNode.remove();
            }
        }

        /**
         * Helper function to keep the global fieldID up-to-date. After inserting
         * a new complex field or after loading from local storage, this number needs to be updated.
         * Then a new complex field can be inserted from the client with a valid and unique id.
         *
         * @param {String} id
         *  The id of the complex field node.
         */
        function updateFieldID(id) {

            // does the id end with a number?
            // -> then the fieldID needs to be checked

            var // resulting array of the regular expression
                matches = /(\d+)$/.exec(id),
                // the number value at the end of the id
                number = 0;

            if (_.isArray(matches)) {
                number = parseInt(matches[1], 10);

                if (number >= fieldID) {
                    fieldID = number + 1;
                }
            }
        }

        /**
         * Update of content for complex field: Page number
         *
         * @param {jQuery|Node} node
         *  Complex field node.
         *
         * @param {String} format
         *  Format of the number.
         */
        function updatePageNumberFieldCx(node, format) {
            var number = pageLayout.getPageNumber(node),
                id = DOM.getComplexFieldMemberId(node),
                startPos = Position.increaseLastIndex(Position.getOxoPosition(selection.getRootNode(), node)),
                endPos = Position.getOxoPosition(selection.getRootNode(), rangeMarker.getEndMarker(id));

            if (model.isHeaderFooterEditState()) { // no explicit update of pageNumber field in header/footer!
                return;
            }

            number = self.formatPageFieldInstruction(number, format);

            // TODO: Nested complex fields

            if (startPos && endPos && number) {
                return model.getUndoManager().enterUndoGroup(function () {
                    selection.setTextSelection(startPos, endPos);
                    model.deleteSelected();
                    model.insertText(number, startPos);
                });
            } else {
                Utils.error('complexField.updatePageNumberFieldCx(): Wrong start end postitions, or text: ', startPos, endPos, number);
            }
        }

        /**
        * Update of content for complex field: Number of pages.
        *
        * @param {jQuery|Node} node
        *   Complex field node
        *
        * @param {String} format
        *  Format of the number.
        */
        function updateNumPagesFieldCx(node, format) {
            var numPages = pageLayout.getNumberOfDocumentPages(),
                id = DOM.getComplexFieldMemberId(node),
                startPos = Position.increaseLastIndex(Position.getOxoPosition(selection.getRootNode(), node)),
                endPos = Position.getOxoPosition(selection.getRootNode(), rangeMarker.getEndMarker(id));

            if (model.isHeaderFooterEditState()) { // no explicit update of numPages field in header/footer!
                return;
            }

            numPages = self.formatPageFieldInstruction(numPages, format, true);

            // TODO: Nested complex fields

            if (startPos && endPos && numPages) {
                return model.getUndoManager().enterUndoGroup(function () {
                    selection.setTextSelection(startPos, endPos);
                    model.deleteSelected();
                    model.insertText(numPages, startPos);
                });
            } else {
                Utils.error('complexField.updateNumPagesFieldCx(): Wrong start end postitions, or text: ', startPos, endPos, numPages);
            }
        }

        /**
         * Update of content for complex field: Date
         *
         * @param {jQuery|Node} node
         *  Complex field node
         *
         * @param {String} format
         *  Format of date
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.time=false]
         *      If it's time format only.
         */
        function updateDateTimeFieldCx(node, format, options) {
            var rawDate = numberFormatter.makeDateTime(numberFormatter.getDateTimeComponents(new Date(), true)),
                date = numberFormatter.convertDateToNumber(rawDate),
                time = Utils.getBooleanOption(options, 'time', false),
                format = (format && format !== 'default') ? format : (time ? LocaleData.SHORT_TIME : LocaleData.SHORT_DATE),
                formattedDate = numberFormatter.formatValue(date, format),
                id = DOM.getComplexFieldMemberId(node),
                isMarginal = DOM.isMarginalNode(node),
                rootNode = isMarginal ? DOM.getClosestMarginalTargetNode(node) : selection.getRootNode(),
                startPos = Position.increaseLastIndex(Position.getOxoPosition(rootNode, node)),
                endPos = Position.getOxoPosition(rootNode, rangeMarker.getEndMarker(id));

            if (startPos && endPos) {
                return model.getUndoManager().enterUndoGroup(function () {
                    if (isMarginal) {
                        pageLayout.enterHeaderFooterEditMode(rootNode);
                        selection.setNewRootNode(rootNode);
                        //selection.resetSelection();
                    }
                    selection.setTextSelection(startPos, endPos);
                    model.deleteSelected();
                    model.insertText(formattedDate, startPos);
                });
            } else {
                Utils.error('complexField.updateDateTimeFieldCx(): Wrong start and end postitions: ', startPos, endPos);
            }

            // TODO: Nested complex fields
        }

        /**
         * Update the content of complex field: FILENAME
         *
         * @param {jQuery|Node} node
         *  Complex field node
         *
         * @param {String} format
         *  Format of the field: Upper case, lower case, first capitalize, all first letters in words capitalized.
         *
         */
        function updateFileNameFieldCx(node, format) {
            var fileName = app.getFullFileName(),
                id = DOM.getComplexFieldMemberId(node),
                startPos = Position.increaseLastIndex(Position.getOxoPosition(selection.getRootNode(), node)),
                endPos = Position.getOxoPosition(selection.getRootNode(), rangeMarker.getEndMarker(id));

            if (format) {
                if ((/Lower/).test(format)) {
                    fileName = fileName.toLowerCase();
                } else if ((/Upper/).test(format)) {
                    fileName = fileName.toUpperCase();
                } else if ((/FirstCap/).test(format)) {
                    fileName = Utils.capitalize(fileName);
                } else if ((/Caps/).test(format)) {
                    fileName = Utils.capitalizeWords(fileName);
                }
            }

            // TODO: Nested complex fields

            if (startPos && endPos && fileName) {
                return model.getUndoManager().enterUndoGroup(function () {
                    selection.setTextSelection(startPos, endPos);
                    model.deleteSelected();
                    model.insertText(fileName, startPos);
                });
            } else {
                Utils.error('complexField.updateNumPagesFieldCx(): Wrong start end postitions, or text: ', startPos, endPos, fileName);
            }
        }

        /**
         * Local helper function that reverts conversion from special field to normal,
         * which is necessary before creating undo operations.
         *
         * @param {jQuery|Node} specialField
         */
        function restoreSpecialField (specialField) {
            var content = $(specialField).children().detach();

            $(specialField).data('length', 1).removeClass(DOM.SPECIAL_FIELD_NODE_CLASSNAME).after(content);
        }

        // methods ------------------------------------------------------------

        /**
         * Whether the document contains complex fields in the document.
         *
         * @returns {Boolean}
         *  Whether the document contains at least one complex field.
         */
        this.isEmpty = function () {
            return _.isEmpty(allFields);
        };

        /**
         * Public method to get all complex fields from model.
         *
         * @returns {Object}
         */
        this.getAllFields = function () {
            return allFields;
        };

        /**
         * Provides a jQuerified range start marker node.
         *
         * @returns {jQuery|null}
         *  The start marker with the specified id, or null, if no such start marker exists.
         */
        this.getComplexField = function (id) {

            var // the value saved under the specified id
                field = allFields[id] || null;

            return _.isString(field) ? getMarginalComplexField(id, field) : field;
        };

        /**
         * Check, whether the specified target string is a valid target for a complex field.
         *
         * @param {String} target
         *  The target string.
         *
         * @returns {Boolean}
         *  Whether the specified id string is a valid id for a complex field node.
         */
        this.isComplexFieldId = function (id) {
            return self.getComplexField(id) !== null;
        };

        /**
         * Provides a unique ID for the complex field.
         *
         * @returns {String}
         *  A unique id.
         */
        this.getNextComplexFieldID = function () {

            var // the next free complex field id
                fieldIdString = 'fld' + fieldID++,
                // the range marker object
                rangeMarker = model.getRangeMarker();

            // checking, if there also exist no range markers with this id
            while (rangeMarker.isUsedRangeMarkerId(fieldIdString)) {
                fieldIdString = 'fld' + fieldID++;
            }

            return fieldIdString;
        };

        /**
         * Public method to remove complex field from model.
         *
         * @param {HTMLElement|jQuery} field
         *  One complex field node.
         *
         */
        this.removeFromComplexFieldCollection = function (node) {
            removeFromComplexFieldCollection(DOM.getComplexFieldId(node));
        };

        /**
         * Handler for insertComplexField operations. Inserts a complex field node into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new complex field.
         *
         * @param {String} id
         *  The unique id of the complex field.
         *
         * @param {String} instruction
         *  The instructions of the complex field.
         *
         * @param {String} target
         *  The target string corresponding to the specified start position.
         *
         * @returns {Boolean}
         *  Whether the complex field has been inserted successfully.
         */
        this.insertComplexFieldHandler = function (start, id, instruction, target) {

            var // the range marker element
                complexFieldNode = $('<div>', { contenteditable: false }).addClass('inline ' + DOM.COMPLEXFIELDNODE_CLASS),
                // the text span needed for inserting the range marker node
                textSpan = model.prepareTextSpanForInsertion(start, null, target),
                // the start marker node, that belongs to this complex field
                startMarker = model.getRangeMarker().getStartMarker(id);

            if (!textSpan) { return false; }

            // saving also the id at the complex field node
            complexFieldNode.attr('data-field-id', id);
            complexFieldNode.attr('data-field-instruction', instruction.replace(/\"/g, '\"'));

            // inserting the complex field node into the DOM
            if (startMarker) {
                if (DOM.isEmptySpan(textSpan) && Utils.getDomNode($(textSpan).prev()) === Utils.getDomNode(startMarker)) { $(textSpan).remove(); }
                complexFieldNode.insertAfter(startMarker);  // no empty span between start marker and complex field node
            } else {
                complexFieldNode.insertAfter(textSpan);
            }

            // adding the complex field node in the collection of all complex fields
            // -> this also needs to be done after loading document from fast load or using local storage
            addIntoComplexFieldCollection(complexFieldNode, id, target);

            // handling the global counter
            updateFieldID(id);

            return true;
        };

        /**
         * Inserting a complex field. This function generates the operations for the ranges and
         * the complex field node. It does not modify the DOM. This function is only executed in
         * the client with edit privileges, not in remote clients.
         */
        this.insertComplexField = function () {

            return model.getUndoManager().enterUndoGroup(function () {

                // TODO:
                // This function is only necessary, when the client can insert complex fields
                // actively without using the clip board

            });
        };

        /**
         * After splitting a paragraph, it is necessary, that all complex field nodes in the cloned
         * 'new' paragraph are updated in the collectors.
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node.
         */
        this.updateComplexFieldCollector = function (para) {

            var // whether each single node needs to be checked
                checkMarginal = false,
                // all complex fields inside the paragraph
                allFields = null;

            // not necessary for paragraphs in header or footer -> only the target are stored, not the nodes
            if (DOM.isMarginalNode(para)) { return; }

            // if this is not a paragraph, each single node need to be checked
            checkMarginal = !DOM.isParagraphNode(para);

            allFields = $(para).find(DOM.COMPLEXFIELDNODE_SELECTOR);

            // update  the complex field nodes in the collection objects
            _.each(allFields, function (oneField) {
                if (!checkMarginal || !DOM.isMarginalNode(oneField)) {
                    // simply overwriting the old complex fields with the new values
                    addIntoComplexFieldCollection(oneField, DOM.getComplexFieldId(oneField));
                }
            });
        };

        /**
         * After load from local storage or fast load the collectors for the complex fields need to be filled.
         *
         * @param {Boolean} usedLocalStorage
         *  Whether the document was loaded from the local storage.
         *
         * @param {Boolean} usedFastLoad
         *  Whether the document was loaded with the fast load process.
         */
        this.refreshComplexFields = function (usedLocalStorage, usedFastLoad) {

            var // the page content node of the document
                pageContentNode = DOM.getPageContentNode(model.getNode()),
                // all range marker nodes in the document
                allComplexFieldNodes = pageContentNode.find(DOM.COMPLEXFIELDNODE_SELECTOR),
                // a collector for all marginal template nodes
                allMargins = pageLayout.getHeaderFooterPlaceHolder().children(),
                // the target string node
                target = '',
                // whether an update of header and footer is required
                updateMarginalNodes = false;

            // helper function to add a collection of range marker nodes into the model
            function addMarginalNodesIntoCollection(collection) {
                _.each(collection, function (oneComplexField) {
                    addIntoComplexFieldCollection(oneComplexField, DOM.getComplexFieldId(oneComplexField), target);
                    updateFieldID(DOM.getComplexFieldId(oneComplexField)); // updating the value for the global field id, so that new fields get an increased number.
                    if (usedFastLoad) {
                        removePreviousEmptyTextSpan(oneComplexField);
                        updateMarginalNodes = true;
                    }
                });
                pageLayout.replaceAllTypesOfHeaderFooters();
            }

            // reset model
            allFields = {};

            // 1. Search in page content node
            // 2. Search in header/footer template node

            // adding all range marker nodes into the collector (not those from header or footer)
            _.each(allComplexFieldNodes, function (oneComplexField) {
                if (!DOM.isMarginalNode($(oneComplexField).parent())) {
                    addIntoComplexFieldCollection(oneComplexField, DOM.getComplexFieldId(oneComplexField));
                    updateFieldID(DOM.getComplexFieldId(oneComplexField)); // updating the value for the global field id, so that new fields get an increased number.
                    if (usedFastLoad) { removePreviousEmptyTextSpan(oneComplexField); }
                }
            });

            // adding the range markers that are located inside header or footer (using the template node)
            _.each(allMargins, function (margin) {

                var // a collector for all range marker nodes inside one margin
                    allMarginComplexFieldNodes = $(margin).find(DOM.COMPLEXFIELDNODE_SELECTOR);

                if (allMarginComplexFieldNodes.length > 0) {
                    target = DOM.getTargetContainerId(margin);
                    addMarginalNodesIntoCollection(allMarginComplexFieldNodes);
                }
            });

            // if empty text spans were removed in header or footer node, an update of the header and
            // footer nodes in the document is required
            if (updateMarginalNodes) {
                // TODO:
                // Refreshing the marginal nodes in the document with the nodes from the template node (not available yet)
            }
        };

        /**
         * After deleting a complete paragraph, it is necessary, that all complex field nodes
         * in the deleted paragraph are also removed from the model collectors.
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node.
         */
        this.removeAllInsertedComplexFields = function (para) {

            var // all range markers inside the paragraph
                allFields = $(para).find(DOM.COMPLEXFIELDNODE_SELECTOR);

            // update  the marker nodes in the collection objects
            _.each(allFields, function (field) {

                var // the id of the marker node
                    fieldId = DOM.getComplexFieldId(field);

                if (_.isString(fieldId)) {
                    removeFromComplexFieldCollection(fieldId);
                }
            });
        };

        /**
         * When pasting content, that contains complex fields, it is necessary, that the
         * complex fields get a new unique id.
         *
         * @param {Object[]} operations
         *  The collection with all paste operations.
         */
        this.handlePasteOperationTarget = function (operations) {

            var // a container for all complex field ids
                allIDs = [],
                // a container for all range marker ids of complex fields
                allRangeMarkerIDs = {};

            // replacing the complex field ids
            function replaceID(ops, id1, id2) {

                _.each(ops, function (op) {
                    if (op) {
                        if (op.id === id1) { op.id = id2; }
                        if (op.target === id1) { op.target = id2; }
                    }
                });
            }

            // collecting all 'id's of insertComplexField operations and assign new target values
            _.each(operations, function (operation) {
                if (operation && operation.name === Operations.COMPLEX_FIELD_INSERT) {
                    allIDs.push(operation.id);
                } else if (operation && operation.name === Operations.RANGE_INSERT && operation.type === 'field') {
                    allRangeMarkerIDs[operation.id] = 1;
                }
            });

            // iterating over all registered complex field ids
            if (allIDs.length > 0) {
                _.each(allIDs, function (oldID) {
                    var newID = self.getNextComplexFieldID();
                    // iterating over all operations, so that also the range markers will be updated
                    replaceID(operations, oldID, newID);
                    delete allRangeMarkerIDs[oldID];
                });
            }

            // are there still unmodified range marker targets (maybe only the start marker was pasted)
            _.each(_.keys(allRangeMarkerIDs), function (oldID) {
                var newID = self.getNextComplexFieldID();
                // iterating over all operations
                replaceID(operations, oldID, newID);
            });

        };

        /**
         * Checks instruction, and depending of extracted field type,
         * updates passed complex field node.
         *
         * @param {jQuery|Node} node
         *  Node that is going to be updated.
         *
         * @param {String} instruction
         *  Field instruction.
         */
        this.updateByInstruction = function (node, instruction) {
            var parsedFormatInstruction,
                dividedData = self.cleanUpAndExtractType(instruction),
                type = dividedData.type,
                instruction = dividedData.instruction;

            if (type) {
                if (self.isNumPages(type)) {
                    updateNumPagesFieldCx(node, self.parseFormatInstruction(instruction));
                } else if (self.isPageNumber(type)) {
                    updatePageNumberFieldCx(node, self.parseFormatInstruction(instruction));
                } else if (self.isCurrentDate(type)) {
                    parsedFormatInstruction = self.parseFormatInstruction(instruction, { dateTime: true });
                    updateDateTimeFieldCx(node, parsedFormatInstruction);
                } else if (self.isCurrentTime(type)) {
                    parsedFormatInstruction = self.parseFormatInstruction(instruction, { dateTime: true });
                    updateDateTimeFieldCx(node, parsedFormatInstruction, { time: true });
                } else if (self.isFileName(type)) {
                    parsedFormatInstruction = self.parseFormatInstruction(instruction);
                    updateFileNameFieldCx(node, parsedFormatInstruction);
                }
            }
        };

        /**
         * Update date and time fields before running download or print action.
         *
         * @param {jQuery|Node} field
         *  Field to be updated.
         *
         * @param {String} instruction
         *  Instruction for updating.
         */
        this.updateDateTimeFieldOnDownload = function (field, instruction) {
            var dividedData = self.cleanUpAndExtractType(instruction),
                type = dividedData.type,
                instruction = dividedData.instruction;

            if (type) {
                if ((self.isCurrentDate(type) || self.isCurrentTime(type))) {
                    instruction = self.parseFormatInstruction(instruction, { dateTime: true });
                    updateDateTimeFieldCx(field, instruction, { time: self.isCurrentTime(type) });
                }
            }
        };

        /**
         * Converts normal complex field to special complex field, when field is inside header/footer and has type PAGE.
         * Content of field is detached and inserted inside complex field.
         * Inverse function from this function is restoreSpecialFields.
         *
         * @param {String} fieldId
         *  Id of the field
         *
         * @param {String} marginalTarget
         *  Id of the marginal target node
         *
         * @param {String} type
         *  Type of field: num pages or page number
         */
        this.convertToSpecialField = function (fieldId, marginalTarget, type) {
            var innerContent,
                rootNode = model.getRootNode(marginalTarget),
                rangeStart = rootNode.find('[data-range-id="' + fieldId + '"]').filter(DOM.RANGEMARKER_STARTTYPE_SELECTOR),
                rangeEnd = rootNode.find('[data-range-id="' + fieldId + '"]').filter(DOM.RANGEMARKER_ENDTYPE_SELECTOR),
                startPos = Position.getOxoPosition(rootNode, rangeStart),
                endPos = Position.getOxoPosition(rootNode, rangeEnd),
                innerLen = _.last(endPos) - _.last(startPos),
                isNumPages = (/NUMPAGES/i).test(type),
                className = isNumPages ? 'cx-numpages' : 'cx-page',
                replacementNode;

            if (innerLen > 0) {
                innerLen -= 1;
            } else {
                Utils.warn('complexField.convertToSpecialField(): inner length is < 1');
            }

            innerContent = rangeEnd.prev().detach().empty();
            rangeEnd.prev().append(innerContent).data('length', innerLen).addClass(DOM.SPECIAL_FIELD_NODE_CLASSNAME).addClass(className);

            if (!DOM.isMarginalPlaceHolderNode(rootNode.parent())) {
                replacementNode = rootNode.children().clone(true);
                replacementNode.find('[contenteditable="true"]').attr('contenteditable', false);
                pageLayout.getHeaderFooterPlaceHolder().children('[data-container-id="' + marginalTarget + '"]').empty().append(replacementNode);
            }
        };

        /**
         * Check if there are special fields of type page in header, inside given searchNode,
         * and if yes, restore them before deleting them.
         *
         * @param {Node} searchNode
         *  Node which we search for special fields.
         *
         * @param {Number} startOffset
         *  Start offset of given node.
         *
         * @param {Number} endOffset
         *  End offset of given node.
         */
        this.checkRestoringSpecialFields = function (searchNode, startOffset, endOffset) {

            if (DOM.isMarginalNode(searchNode) && $(searchNode).find(DOM.SPECIAL_FIELD_NODE_SELECTOR).length) {
                _.each($(searchNode).find(DOM.SPECIAL_FIELD_NODE_SELECTOR), function (specialField) {
                    var fieldPos = Position.getOxoPosition(model.getRootNode(), specialField, 0);
                    if (Position.isNodePositionInsideRange(fieldPos, startOffset, endOffset)) {
                        restoreSpecialField(specialField);
                    }
                });
            }
        };

        /**
         * Check if there are special fields in container node, and revert conversion from special field to normal,
         * which is necessary before creating undo operations.
         *
         * @param {jQuery|Node} containerNode
         *  Can be whole header or footer, not restricted to paragraph parts.
         *  For that, see function: checkRestoringSpecialFields
         */
        this.checkRestoringSpecialFieldsInContainer = function (containerNode) {
            _.each($(containerNode).find(DOM.SPECIAL_FIELD_NODE_SELECTOR), function (specialField) {
                restoreSpecialField(specialField);
            });
        };

        /**
         * After cloning headers&footers special page fields have to be updated with proper page number.
         */
        this.updateCxPageFieldsInMarginals = function () {
            var $pageContentNode = null,
                firstHeadNum = 1,
                lastFootNum = null,
                // first header in doc
                $firstHeaderFields = null,
                // last footer in doc
                $lastFooterFields = null,
                numPagesNodes = null;

            if (self.isEmpty() || !pageLayout.getHeaderFooterPlaceHolder().find(DOM.SPECIAL_FIELD_NODE_SELECTOR).length) {
                return; // no special fields in the document, don't make unnecessary loops
            }

            // performance: assign variables only if really needed
            $pageContentNode = DOM.getPageContentNode(model.getNode());
            lastFootNum = pageLayout.getNumberOfDocumentPages();
            // first header in doc - page number always 1
            $firstHeaderFields = pageLayout.getFirstHeaderWrapperNode().children(DOM.HEADER_SELECTOR).find('.cx-page');
            // last footer in doc gets total page count as page number
            $lastFooterFields = pageLayout.getLastFooterWrapperNode().children(DOM.FOOTER_SELECTOR).find('.cx-page');

            _.each($firstHeaderFields, function (headerField) {
                var $headerField = $(headerField),
                    headerFormat = DOM.getComplexFieldInstruction($headerField);

                $headerField.children().empty().html(self.formatPageFieldInstruction(firstHeadNum, headerFormat));
            });

            _.each($lastFooterFields, function (footerField) {
                var $footerField = $(footerField),
                    footerFormat = DOM.getComplexFieldInstruction($footerField);

                $footerField.children().empty().html(self.formatPageFieldInstruction(lastFootNum, footerFormat));
            });

            _.each(pageLayout.getPageBreaksCollection(), function (pageBreak) {
                var $pageBreak = $(pageBreak),
                    headerNumber = $pageBreak.data('page-num') + 1,
                    footerNumber = $pageBreak.data('page-num'),
                    $headerFields = $pageBreak.children(DOM.HEADER_SELECTOR).find('.cx-page'),
                    $footerFields = $pageBreak.children(DOM.FOOTER_SELECTOR).find('.cx-page');

                _.each($headerFields, function (headerField) {
                    var $headerField = $(headerField),
                        headerFormat = DOM.getComplexFieldInstruction($headerField);

                    $headerField.children().empty().html(self.formatPageFieldInstruction(headerNumber, headerFormat));
                });

                _.each($footerFields, function (footerField) {
                    var $footerField = $(footerField),
                        footerFormat = DOM.getComplexFieldInstruction($footerField);

                    $footerField.children().empty().html(self.formatPageFieldInstruction(footerNumber, footerFormat));
                });
            });

            // update number of pages fields, which all have same value
            if (pageLayout.getHeaderFooterPlaceHolder().find('.cx-numpages').length) {
                numPagesNodes = $pageContentNode.find(DOM.PAGE_BREAK_SELECTOR).add(pageLayout.getFirstHeaderWrapperNode()).add(pageLayout.getLastFooterWrapperNode()).find('.cx-numpages');
                _.each(numPagesNodes, function (fieldNode) {
                    var $fieldNode = $(fieldNode),
                        fieldFormat = DOM.getComplexFieldInstruction($fieldNode);

                    $fieldNode.children().empty().html(self.formatPageFieldInstruction(lastFootNum, fieldFormat));
                });
            }
        };

        /**
         * Function called after importing document.
         * It loops all fields in collection and:
         *  - marks them for highlighting,
         *  - updates date and time fields
         *  - converts page fields in header&footer to special fields
         */
        this.updateDateTimeAndSpecialFields = function () {
            var updatePageFields = false,
                returnCursorToBegining = true;

            _.each(allFields, function (entry, fieldId) {
                var field = self.getComplexField(fieldId),
                    instruction = DOM.getComplexFieldInstruction(field),
                    dividedData = self.cleanUpAndExtractType(instruction),
                    type = dividedData.type,
                    instruction = dividedData.instruction,
                    format = null;

                markComplexFieldForHighlighting(field, fieldId);

                if (type) {
                    if ((self.isCurrentDate(type) || self.isCurrentTime(type))) {
                        format = self.parseFormatInstruction(instruction, { dateTime: true });
                        // if previously updated field is marginal, and this is not, leave header edit state
                        if (!DOM.isMarginalNode(field) && model.isHeaderFooterEditState()) {
                            pageLayout.leaveHeaderFooterEditMode();
                            selection.setNewRootNode(model.getNode());
                        }
                        updateDateTimeFieldCx(field, format, { time: self.isCurrentTime(type) });
                        if (pageLayout.isIdOfMarginalNode(entry)) {
                            markComplexFieldForHighlighting(self.getComplexField(fieldId), fieldId);
                            pageLayout.getHeaderFooterPlaceHolder().children('.' + entry).empty().append(selection.getRootNode().children().clone(true));
                        }
                        returnCursorToBegining = true;
                    } else if (_.isString(entry) && pageLayout.isIdOfMarginalNode(entry) && self.isPageNumber(type)) {
                        if (!DOM.isSpecialField(field)) {
                            self.convertToSpecialField(fieldId, entry, type);
                        }
                        updatePageFields = true;
                    }
                } else {
                    Utils.warn('complexField.updateDateTimeAndSpecialFields: invalid type: ', type);
                }
            });
            if (updatePageFields) {
                // update reference in collection because of replaced node
                pageLayout.headerFooterCollectionUpdate();
                pageLayout.replaceAllTypesOfHeaderFooters(); // complex fields in header/footers are refreshed in this method call, if any
            }
            if (returnCursorToBegining) {
                // if there was update of date fields, return cursor to first document position
                if (model.isHeaderFooterEditState()) {
                    pageLayout.leaveHeaderFooterEditMode();
                    selection.setNewRootNode(model.getNode());
                }
                selection.setTextSelection(selection.getFirstDocumentPosition());
            }
        };

        /**
         * Public method for updating current page number field(s) in given node.
         *
         * @param {jQuery} $node
         * @param {Boolean} isHeader
         *  If node is header or not.
         * @param {Number} pageCount
         *  Number of total pages in document.
         */
        this.updatePageNumInCurrentNodeCx = function ($node, isHeader, pageCount) {
            if (isHeader) {
                if (DOM.isHeaderWrapper($node.parent())) {
                    self.updatePageNumInCurrentMarginalCx($node, isHeader, 1);
                } else {
                    self.updatePageNumInCurrentMarginalCx($node, isHeader);
                }
            } else {
                if (DOM.isFooterWrapper($node.parent())) {
                    self.updatePageNumInCurrentMarginalCx($node, isHeader, pageCount);

                } else {
                    self.updatePageNumInCurrentMarginalCx($node, isHeader);
                }
            }
        };

        /**
         *
         *
         * @param $node
         * @param pageNum
         */
        this.updatePageNumInCurrentMarginalCx = function ($node, isHeader, pageNum) {
            _.each($node.find('.cx-page'), function (field) {
                var $field = $(field),
                    format = DOM.getComplexFieldInstruction($field),
                    pageNumber = pageNum || ($(field).parentsUntil('.pagecontent', '.page-break').data('page-num') + (isHeader ? 1 : 0));

                $field.children().empty().html(self.formatPageFieldInstruction(pageNumber, format));
            });
        };

        /**
         * Public method for updating number of pages field(s) in given node.
         *
         * @param {jQuery} $node
         * @param {Number} pageCount
         *  Number of total pages in document.
         */
        this.updatePageCountInCurrentNodeCx = function ($node, pageCount) {
            _.each($node.find('.cx-numpages'), function (fieldNode) {
                var $fieldNode = $(fieldNode),
                    fieldFormat = DOM.getComplexFieldInstruction($fieldNode);

                $fieldNode.children().empty().html(self.formatPageFieldInstruction(pageCount, fieldFormat));
            });
        };

        // initialization -----------------------------------------------------

        app.onInit(function () {
            model = app.getModel();
            selection = model.getSelection();
            pageLayout = model.getPageLayout();
            numberFormatter = model.getNumberFormatter();
            rangeMarker = model.getRangeMarker();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            model = selection = pageLayout = numberFormatter = rangeMarker = null;
        });

    } // class ComplexField

    // constants --------------------------------------------------------------

    // export =================================================================

    // derive this class from class BaseField
    return BaseField.extend({ constructor: ComplexField });
});
