/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/drawingrenderer', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/spreadsheet/view/render/renderutils'
], function (Utils, Forms, TriggerObject, TimerMixin, DrawingFrame, RenderUtils) {

    'use strict';

    // class DrawingRenderer ==================================================

    /**
     * Renders the drawing objects of the active sheet into the DOM drawing
     * layer shown in a single grid pane.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {GridPane} gridPane
     *  The grid pane instance that owns this drawing layer renderer.
     */
    function DrawingRenderer(gridPane) {

        var // self reference
            self = this,

            // the spreadsheet model and view
            docView = gridPane.getDocView(),
            docModel = docView.getDocModel(),

            // the layer node containing all drawing frames
            drawingLayerNode = gridPane.createLayerNode('drawing-layer'),

            // current layer range covered by this renderer
            layerRange = null,

            // all existing drawing frames, mapped by UID (faster access than DOM lookups)
            drawingFrameMap = {},

            // all drawing models whose drawing frames need to be refreshed
            pendingModelMap = {},

            // the current background task for refreshing drawing frames
            refreshTimer = null;

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Returns all selected DOM drawing frame nodes.
         *
         * @returns {jQuery}
         *  The selected drawing frames, as jQuery object.
         */
        function getSelectedDrawingFrames() {
            return drawingLayerNode.find(DrawingFrame.NODE_SELECTOR + Forms.SELECTED_SELECTOR);
        }

        /**
         * Updates the position of the passed drawing object according to its
         * anchor attributes.
         *
         * @returns {jQuery|Null}
         *  The DOM drawing frame of the drawing model, if it is visible, and
         *  located inside the layer range of this renderer; otherwise null.
         */
        function updateDrawingFramePosition(drawingModel) {

            // the DOM drawing frame (ignore drawing models without a created drawing frame)
            var drawingFrame = drawingFrameMap[drawingModel.getUid()];
            if (!drawingFrame) { return null; }

            // the effective drawing rectangle (null if the drawing object is hidden)
            var rectangle = drawingModel.getRectangle();

            // drawing object visible: show the drawing frame, and set its position and size
            if (rectangle && layerRange.overlaps(drawingModel.getRange())) {
                drawingFrame.show().css(gridPane.convertToLayerRectangle(rectangle));
                return drawingFrame;
            }

            // hide the drawing frame if the drawing object is hidden, or outside the layer range
            drawingFrame.hide();
            return null;
        }

        /**
         * Starts a background task that refreshes the formatting of all
         * drawing frames registered in the variable 'pendingModelMap' (unless
         * such a task is already running). The background task will refresh
         * the drawing frames in several time slices if necessary.
         *
         * @param {Number} [refreshDelay=200]
         *  The initial delay time before the first drawing frame will be
         *  updated, in milliseconds.
         */
        function refreshDrawingFramesDelayed(refreshDelay) {

            // nothing to do, if the timer is running, or if there are no dirty drawings
            if (refreshTimer || _.isEmpty(pendingModelMap)) { return; }

            // Start a background loop that will process all drawing models that are contained
            // in the object 'pendingModelMap'. Drawing models added while the background loop
            // is running will not be handled. Therefore, the background loop will be started
            // again and again, until the map is empty (see below).
            refreshTimer = self.iterateObjectSliced(pendingModelMap, function (drawingModel, drawingUid) {

                // the DOM drawing frame (undefined, if the drawing object has been deleted in the meantime)
                var drawingFrame = drawingFrameMap[drawingUid];

                // update all drawing frames that still exist, and are visible
                if (drawingModel && drawingFrame) {
                    DrawingFrame.updateFormatting(docView.getApp(), drawingFrame, drawingModel.getMergedAttributes());
                }

                // remove processed drawing models from the 'pendingModelMap' object
                delete pendingModelMap[drawingUid];

            }, { delay: _.isNumber(refreshDelay) ? refreshDelay : 200, interval: 200 });

            // try to restart the timer in case new dirty objects have been registered in the meantime
            refreshTimer.done(function () {
                refreshTimer = null;
                refreshDrawingFramesDelayed(); // continue with default delay
            });

            // clean up after the timer has been aborted
            refreshTimer.fail(function () {
                refreshTimer = null;
            });
        }

        /**
         * Updates the position and formatting of the drawing frame that
         * represents the passed drawing model.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model instance whose drawing frame will be updated.
         *
         * @param {Number} [refreshDelay=200]
         *  The delay time before the contents and formatting of the drawing
         *  frame will be updated, in milliseconds (the position and visibility
         *  of the drawing frame will be updated immediately).
         */
        function updateDrawingFrame(drawingModel, refreshDelay) {

            var // the DOM drawing frame with updated position
                drawingFrame = updateDrawingFramePosition(drawingModel);

            // update formatting of visible drawing frames
            if (drawingFrame) {
                pendingModelMap[drawingModel.getUid()] = drawingModel;
                refreshDrawingFramesDelayed(refreshDelay);
            } else {
                delete pendingModelMap[drawingModel.getUid()];
            }
        }

        /**
         * Creates and inserts a new DOM drawing frame node that represents the
         * passed drawing model.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model instance to create a DOM drawing frame for.
         *
         * @param {Array<Number>} position
         *  The document position of the drawing model in the sheet.
         *
         * @param {Number} [refreshDelay=200]
         *  The delay time before the contents and formatting of the drawing
         *  frame will be updated, in milliseconds (the position and visibility
         *  of the drawing frame will be initialized immediately).
         */
        function createDrawingFrame(drawingModel, position, refreshDelay) {

            var // the new drawing frame, as jQuery object
                drawingFrame = DrawingFrame.createDrawingFrame(drawingModel),
                // all existing drawing frames, as plain DOM collection
                drawingNodes = drawingLayerNode[0].childNodes,
                // the DOM child index (Z order) specified by the position
                index = position[0];

            // store the drawing frame in a local map (faster access than DOM lookup)
            drawingFrameMap[drawingModel.getUid()] = drawingFrame;

            // insert the drawing frame with the correct Z order
            if (drawingNodes.length === index) {
                drawingLayerNode.append(drawingFrame);
            } else {
                drawingFrame.insertBefore(drawingNodes[index]);
            }

            // update position and formatting of the drawing frame
            updateDrawingFrame(drawingModel, refreshDelay);
        }

        /**
         * Updates the selection of all drawing frames, and triggers a
         * 'render:drawingselection' event, if the drawing selection has
         * changed.
         */
        function renderDrawingSelection() {

            var // all drawing frames selected in the DOM (before running this method)
                oldSelectedFrames = getSelectedDrawingFrames(),
                // all drawing frames selected in the DOM (after running this method)
                newSelectedFrames = $(),
                // all existing drawing frames, as plain DOM collection
                drawingNodes = drawingLayerNode[0].childNodes,
                // disable tracking in read-only mode, or if sheet is protected
                editMode = docModel.getEditMode() && !docView.isSheetLocked();

            // process all drawings to be selected
            docView.getSelectedDrawings().forEach(function (position) {

                var // the root drawing frame addressed by the current position
                    drawingFrame = drawingNodes[position[0]];

                // drawing frame may be missing, e.g. while remotely inserting new drawings
                // TODO: find embedded drawing frames
                if (drawingFrame && (position.length === 1)) {
                    DrawingFrame.drawSelection(drawingFrame, { movable: editMode, resizable: editMode });
                    newSelectedFrames = newSelectedFrames.add(drawingFrame);
                }
            });

            // remove selection border from all drawing frames not selected anymore
            oldSelectedFrames.not(newSelectedFrames).each(function () { DrawingFrame.clearSelection(this); });

            // notify listeners
            if (!Utils.equalArrays(oldSelectedFrames.get(), newSelectedFrames.get())) {
                self.trigger('render:drawingselection', newSelectedFrames);
            }
        }

        /**
         * Updates the drawing selection after specific sheet view attributes
         * have been changed.
         */
        function changeSheetViewAttributesHandler(event, attributes) {

            // render drawing selection (triggers a 'render:drawingselection' event)
            if ('selection' in attributes) {
                renderDrawingSelection();
            }

            // trigger a 'render:drawingselection' event if active pane changes
            // (no actual rendering needed, border colors will change via CSS)
            if ('activePane' in attributes) {
                self.trigger('render:drawingselection', getSelectedDrawingFrames());
            }
        }

        /**
         * Processes a new drawing object inserted into the drawing collection.
         */
        function insertDrawingHandler(event, drawingModel, position) {
            createDrawingFrame(drawingModel, position);
            renderDrawingSelection();
        }

        /**
         * Processes a drawing object removed from the drawing collection.
         */
        function deleteDrawingHandler(event, drawingModel) {

            var // the unique identifier of the drawing, used as map key
                modelUid = drawingModel.getUid(),
                // the DOM drawing frame of the drawing model
                drawingFrame = drawingFrameMap[modelUid];

            if (drawingFrame) {
                drawingFrame.remove();
                delete drawingFrameMap[modelUid];
                delete pendingModelMap[modelUid];
                renderDrawingSelection();
            }
        }

        /**
         * Processes a drawing object that has been changed in any way.
         */
        function changeDrawingHandler(event, drawingModel) {
            updateDrawingFrame(drawingModel);
        }

        // protected methods --------------------------------------------------

        /**
         * Changes the drawing layer according to the passed layer range.
         */
        this.setLayerRange = RenderUtils.profileMethod('DrawingRenderer.setLayerRange()', function (newLayerRange) {

            var // the drawing collection of the active sheet
                drawingCollection = docView.getDrawingCollection(),
                // the old layer range
                oldLayerRange = layerRange;

            // store new layer range in internal member
            layerRange = newLayerRange;

            // repaint all visible drawing frames; or create all drawing frames if
            // not done yet (e.g. active sheet changed, enabled split in current sheet)
            if (oldLayerRange) {
                drawingCollection.iterateModelsByPosition(function (drawingModel) {
                    updateDrawingFrame(drawingModel, 1000);
                });
            } else {
                drawingCollection.iterateModelsByPosition(function (drawingModel, position) {
                    createDrawingFrame(drawingModel, position, 1000);
                });
                renderDrawingSelection();
            }
        });

        /**
         * Resets this renderer, clears the DOM layer node.
         */
        this.hideLayerRange = function () {

            // reset internal settings
            layerRange = null;
            pendingModelMap = {};

            // safely destroy all image nodes to prevent memory problems on iPad
            docView.getApp().destroyImageNodes(drawingLayerNode);
            drawingLayerNode.empty();
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the DOM root node of this drawing layer renderer containing
         * all drawing frames.
         *
         * @returns {jQuery}
         *  The DOM root node of this drawing layer renderer, as jQuery object.
         */
        this.getLayerNode = function () {
            return drawingLayerNode;
        };

        /**
         * Returns the DOM drawing frame node at the passed document position.
         *
         * @param {Array<Number>} position
         *  The document position of the drawing model. May specify a position
         *  inside another drawing object.
         *
         * @returns {jQuery}
         *  The DOM node of the drawing frame, as jQuery object; or an empty
         *  jQuery collection, if the drawing frame does not exist.
         */
        this.getDrawingFrame = function (position) {
            var drawingModel = docView.getDrawingCollection().findModel(position),
                drawingFrame = drawingModel ? drawingFrameMap[drawingModel.getUid()] : null;
            return drawingFrame ? drawingFrame : $();
        };

        // initialization -----------------------------------------------------

        // update grid pane when document edit mode changes
        gridPane.listenToWhenVisible(this, docModel, 'change:editmode', renderDrawingSelection);

        // update drawing layer node (only, if this grid pane is visible)
        gridPane.listenToWhenVisible(this, docView, 'change:sheet:viewattributes', changeSheetViewAttributesHandler);
        gridPane.listenToWhenVisible(this, docView, 'insert:drawing', insertDrawingHandler);
        gridPane.listenToWhenVisible(this, docView, 'delete:drawing', deleteDrawingHandler);
        gridPane.listenToWhenVisible(this, docView, 'change:drawing', changeDrawingHandler);

        // bug 31479: suppress double-clicks for drawings
        drawingLayerNode.on('dblclick', false);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docView.getApp().destroyImageNodes(drawingLayerNode);
            self = gridPane = docModel = docView = null;
            drawingLayerNode = drawingFrameMap = pendingModelMap = refreshTimer = null;
        });

    } // class DrawingRenderer

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: DrawingRenderer });

});
