/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/cellrenderer', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/canvas/canvas',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/model/cellcollection',
    'io.ox/office/spreadsheet/view/render/renderutils'
], function (Utils, Canvas, TriggerObject, TimerMixin, Color, Border, SheetUtils, PaneUtils, CellCollection, RenderUtils) {

    'use strict';

    var // convenience shortcuts
        Interval = SheetUtils.Interval,
        Range = SheetUtils.Range,
        RangeArray = SheetUtils.RangeArray,

        // line width for grid colors in the canvas
        GRID_LINE_WIDTH = RenderUtils.GRID_LINE_WIDTH,

        // automatic grid color (dark gray instead of black, to make it visible on white and black backgrounds)
        AUTO_GRID_COLOR = new Color('rgb', '666666'),

        // reduction of clipping area size at leading border for canvas rendering,
        // needed to protect the existing outer cell borders
        LEADING_CLIP_SIZE_REDUCTION = Utils.roundDown(SheetUtils.MIN_CELL_SIZE / 2, GRID_LINE_WIDTH),

        // reduction of clipping area size at trailing border for canvas rendering,
        // needed to protect the existing outer cell borders
        TRAILING_CLIP_SIZE_REDUCTION = SheetUtils.MIN_CELL_SIZE - LEADING_CLIP_SIZE_REDUCTION;

    // class CellRenderer =====================================================

    /**
     * Renders the cell contents, cell formatting, and the cell grid lines into
     * a single grid pane.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {GridPane} gridPane
     *  The grid pane instance that owns this cell layer renderer.
     */
    function CellRenderer(gridPane) {

        var // self reference
            self = this,

            // document model and view, and the rendering cache of the document view
            docView = gridPane.getDocView(),
            docModel = docView.getDocModel(),
            renderCache = docView.getRenderCache(),

            // current zoom factor, for scaling of canvas element
            sheetZoom = 1,

            // canvas element for all cell contents
            cellCanvas = new Canvas({ classes: 'cell-layer' }),

            // the layer node containing outlines for table ranges
            tableLayerNode = gridPane.createLayerNode('table-layer'),

            // the cell range and absolute position covered by the layer nodes in the sheet area
            layerRange = null,
            layerRectangle = null,

            // all pending cell ranges to be rendered debounced
            pendingRanges = new RangeArray();

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Renders the grid lines between the cells into the canvas element.
         */
        var renderGridLayer = RenderUtils.profileMethod('rendering grid layer', function (clip, range) {

            // nothing to do, if grid line are not enabled in the active sheet
            if (!docView.getSheetViewAttribute('showGrid')) { return; }

            // render grid lines into the canvas
            cellCanvas.render(clip, function (context) {

                var // the model of the active sheet
                    sheetModel = docView.getSheetModel(),
                    // the current grid color
                    gridColor = sheetModel.getGridColor(),
                    // the path object containing all grid line positions
                    path = context.createPath(),
                    // all merged ranges with more than one visible column
                    mergedColRanges = new RangeArray(),
                    // all merged ranges with more than one visible row
                    mergedRowRanges = new RangeArray(),
                    // sub-pixel offset to keep grid lines on canvas pixels
                    subPxOffset = -GRID_LINE_WIDTH / 2;

                // fill the arrays of merged ranges in one pass
                renderCache.iterateMergedRanges(function (mergedRange, shrunkenRange) {
                    if (shrunkenRange.start[0] < shrunkenRange.end[0]) { mergedColRanges.push(shrunkenRange); }
                    if (shrunkenRange.start[1] < shrunkenRange.end[1]) { mergedRowRanges.push(shrunkenRange); }
                });

                // adds either all vertical or all horizontal grid lines to the path
                function addGridLines(columns) {

                    var // the column/row collection in the other direction (for intervals of merged ranges)
                        collection = columns ? docView.getRowCollection() : docView.getColCollection(),
                        // merged ranges spanning over multiple columns/rows (causing multiple line segments)
                        mergedRanges = columns ? mergedColRanges : mergedRowRanges,
                        // layer start offset in rendering direction
                        layerOffset = columns ? layerRectangle.top : layerRectangle.left,
                        // layer size in rendering direction
                        layerSize = columns ? layerRectangle.height : layerRectangle.width,
                        // function to push a line to the path
                        pushLineFunc = columns ?
                            function (x, y1, y2) { path.pushLine(x, y1 - GRID_LINE_WIDTH, x, y2); } :
                            function (y, x1, x2) { path.pushLine(x1 - GRID_LINE_WIDTH, y, x2, y); };

                    // restrict layer size to sheet size
                    layerSize = Math.min(layerSize, collection.getTotalSize() - layerOffset);

                    // add the grid lines for all visible columns/rows to the path
                    renderCache.iterateInterval(range, columns, function (entry) {

                        var // use the end position of the current column/row, translate by half a pixel to paint on whole pixels
                            offset = entry.offset + entry.size + subPxOffset,
                            // all merged ranges that will break the grid line
                            coveredMergedRanges = mergedRanges.filter(function (mergedRange) {
                                return (mergedRange.getStart(columns) <= entry.index) && (entry.index < mergedRange.getEnd(columns));
                            }),
                            // the merged and sorted column/row intervals of the merged ranges
                            mergedIntervals = coveredMergedRanges.intervals(!columns).merge(),
                            // the coordinates of the line segments, as flat array
                            lineCoords = null;

                        // calculate the line start/end coordinates
                        lineCoords = mergedIntervals
                            // get the pixel positions of the merged column/row intervals
                            .map(collection.getIntervalPosition, collection)
                            // filter hidden merged ranges
                            .filter(function (position) { return position.size > 0; })
                            // adjust layer offset in positions and convert to (nested) integer array
                            .map(function (position) { return [position.offset, position.offset + position.size]; });

                        // convert to a flat array, add start and end position of the layer rectangle
                        lineCoords = _.flatten(lineCoords);
                        lineCoords.unshift(layerOffset);
                        lineCoords.push(layerOffset + layerSize);

                        // draw all line segments
                        for (var index = 0, length = lineCoords.length; index < length; index += 2) {
                            if (lineCoords[index] < lineCoords[index + 1]) {
                                pushLineFunc(offset, lineCoords[index], lineCoords[index + 1]);
                            }
                        }
                    });
                }

                // add all grid lines to the path
                addGridLines(false);
                addGridLines(true);

                // paint everything with 20% opacity
                context.setGlobalAlpha(0.2);
                context.setLineStyle({ style: docModel.resolveColor(gridColor, AUTO_GRID_COLOR).css, width: GRID_LINE_WIDTH });
                context.drawPath(path);
            });
        });

        /**
         * Renders the fill styles of all cells into the canvas element.
         */
        var renderFillLayer = RenderUtils.profileMethod('rendering fill layer', function (clip, range) {

            // render cell background areas into the canvas
            cellCanvas.render(clip, function (context) {

                var // the model of the active sheet
                    sheetModel = docView.getSheetModel(),
                    // all columns with an explicit fill color, mapped by column index
                    colFills = {},
                    // all rows with an explicit fill color (or transparent), mapped by row index
                    rowFills = {},
                    // all cells that need to be cleared (in order to delete default column/row formatting)
                    clearCells = [],
                    // all cells that need to be filled
                    fillCells = [];

                // paints or clears a rectangle in the canvas (also fills the trailing pixel of the preceding left/top cells)
                function drawRect(left, top, width, height, style) {
                    if (style) {
                        context.setFillStyle({ style: style });
                        context.drawRect(left - GRID_LINE_WIDTH, top - GRID_LINE_WIDTH, width + GRID_LINE_WIDTH, height + GRID_LINE_WIDTH);
                    } else {
                        context.clearRect(left - GRID_LINE_WIDTH, top - GRID_LINE_WIDTH, width + GRID_LINE_WIDTH, height + GRID_LINE_WIDTH);
                    }
                }

                // paints the rectangle of the passed cell data
                function drawCellElement(element) {
                    var fill = element.style ? element.style.fill : null;
                    if (element.isMergedOrigin()) {
                        var rect = sheetModel.getRangeRectangle(element.merged.shrunkenRange);
                        drawRect(rect.left, rect.top, rect.width, rect.height, fill);
                    } else {
                        drawRect(element.col.offset, element.row.offset, element.col.size, element.row.size, fill);
                    }
                }

                // render entire filled columns (skip transparent columns, canvas has been cleared)
                renderCache.iterateInterval(range, true, function (element) {
                    var fill = element.style ? element.style.fill : null;
                    if (fill) {
                        drawRect(element.offset, layerRectangle.top, element.size, layerRectangle.height, fill);
                        colFills[element.index] = fill;
                    }
                });

                // render entire filled rows (but skip rows without attributes property, which do not have the 'customFormat' flag)
                renderCache.iterateInterval(range, false, function (element) {
                    if (element.style) {
                        // render transparent rows too, need to clean areas with column formatting
                        drawRect(layerRectangle.left, element.offset, layerRectangle.width, element.size, element.style.fill);
                        rowFills[element.index] = element.style.fill;
                    }
                });

                // Collect all cell to be filled or cleared (transparent cells may need to clear the default column/row
                // background). Collect cells to be cleared in a separate array to be able to draw them before the filled
                // cells which want to expand the filled area by 1 pixel to the left/top into their predecessors.
                renderCache.iterateCells(range, function (element) {

                    // skip undefined cells (column/row default formatting remains as is)
                    if (!element.style) { return; }

                    // target array to push the current cell data object to
                    var targetArray = element.style.fill ? fillCells : clearCells;

                    // bug 40429: always render merged ranges over default column/row formatting
                    if (element.isMergedOrigin()) {
                        targetArray.push(element);
                        return;
                    }

                    var // column and row index of the current cell
                        col = element.address[0],
                        row = element.address[1],
                        // current default fill color from column or row (already rendered above)
                        defaultFillStyle = (row in rowFills) ? rowFills[row] : (col in colFills) ? colFills[col] : null;

                    // Bug 37041: Improve performance by checking the default column/row fill color
                    // (do not render cells that do not change the existing fill color).
                    if (defaultFillStyle !== element.style.fill) {
                        targetArray.push(element);
                    }
                }, { origins: true });

                // first, clear all transparent cells (done before drawing all filled cells, otherwise
                // the trailing pixel of preceding filled cells would be cleared by these cells)
                clearCells.forEach(drawCellElement);
                fillCells.forEach(drawCellElement);
            });
        });

        /**
         * Renders the border styles of all cells into the canvas element.
         */
        var renderBorderLayer = RenderUtils.profileMethod('rendering border layer', function (clip, range) {

            // render cell background areas into the canvas
            cellCanvas.render(clip, function (context) {

                var // current border line to be painted next (will be expanded across multiple cells before)
                    cachedBorder = null;

                // returns whether the passed border descriptors are considered to be equal for expanding
                function equalBorders(border1, border2) {
                    // - never expand dashed borders to prevent moving dashes/dots during partial rendering
                    // - no need to check the 'hair' property which already influences the CSS color
                    return border1.solid && border2.solid && (border1.width === border2.width) && (border1.double === border2.double) && (border1.color === border2.color);
                }

                // paints the border line stored in the variable 'cachedBorder', and resets it
                function paintCachedBorder() {

                    if (!cachedBorder) { return; }

                    var // shortcut to the border style
                        border = cachedBorder.border,
                        // shortcuts to start/end coordinate of the line
                        start = cachedBorder.start,
                        end = cachedBorder.end,
                        // sub-pixel offset to paint the line on entire canvas pixels
                        subPxOffset = -((border.width / 2) % GRID_LINE_WIDTH),
                        // a new context path object
                        path = context.createPath();

                    // pushes a single line to the rendering path
                    function pushLine(offset) {
                        if (cachedBorder.columns) {
                            path.pushLine(offset, start - GRID_LINE_WIDTH, offset, end);
                        } else {
                            path.pushLine(start - GRID_LINE_WIDTH, offset, end, offset);
                        }
                    }

                    // initialize context line style for the border line
                    context.setLineStyle({ style: border.color, width: border.lineWidth, pattern: border.pattern });

                    // add all border lines to the path
                    if (border.double) {
                        var leadOffset = cachedBorder.offset - Math.round((border.width - border.lineWidth) / 2 - subPxOffset) - subPxOffset;
                        pushLine(leadOffset);
                        pushLine(leadOffset + Math.round(border.width / 3 * 2));
                    } else {
                        pushLine(cachedBorder.offset + subPxOffset);
                    }

                    // draw the border, reset the 'cachedBorder' variable
                    context.drawPath(path);
                    cachedBorder = null;
                }

                // renders all border lines of the passed range in one direction
                function renderBorderLines(columns, leading) {

                    var // property name for the border style of the current cell
                        borderProp = columns ? (leading ? 'l' : 'r') : (leading ? 't' : 'b'),
                        // property name for the opposite border style of the adjacent cell
                        oppositeProp = columns ? (leading ? 'r' : 'l') : (leading ? 'b' : 't');

                    // visit all cells
                    renderCache.iterateCells(range, function (element) {

                        var // the border style of the cell
                            elementBorder = element.getBorder(borderProp),
                            // the next element containing the concurrent border line
                            adjacentElement = element[borderProp],
                            // the border style of the adjacent cell
                            adjacentBorder = adjacentElement && adjacentElement.getBorder(oppositeProp),
                            // get the stronger of the two borders
                            border = RenderUtils.BorderDescriptor.getStrongerBorder(elementBorder, adjacentBorder);

                        // no border available: nothing to do
                        if (!border) { return; }

                        var // matrix header element in drawing direction
                            mainHeader = columns ? element.col : element.row,
                            // line coordinate for main drawing direction
                            mainOffset = mainHeader.offset + (leading ? 0 : mainHeader.size),
                            // matrix header element in opposite direction (for start/end coordinate of the line)
                            oppositeHeader = columns ? element.row : element.col,
                            // start/end coordinates of the line from opposite header
                            startOffset = oppositeHeader.offset,
                            endOffset = startOffset + oppositeHeader.size;

                        // try to expand the existing cached border (never expand dashed borders to prevent moving dashes/dots during partial rendering)
                        if (cachedBorder && (cachedBorder.columns === columns) && (cachedBorder.offset === mainOffset) && (cachedBorder.end === startOffset) && equalBorders(cachedBorder.border, border)) {
                            cachedBorder.end = endOffset;
                        } else {
                            // paint current cached border, and start a new cached border
                            paintCachedBorder();
                            cachedBorder = { offset: mainOffset, start: startOffset, end: endOffset, border: border, columns: columns };
                        }

                    }, { columns: columns, first: leading });
                }

                // paint vertical borders of all cells (first left borders of left cells, then right borders of all cells)
                renderBorderLines(true, true);
                renderBorderLines(true, false);

                // paint horizontal borders of all cells (first top borders of top cells, then bottom borders of all cells)
                renderBorderLines(false, true);
                renderBorderLines(false, false);

                // paint the remaining border
                paintCachedBorder();
            });
        });

        /**
         * Renders the cell content texts into the canvas element.
         */
        var renderTextLayer = RenderUtils.profileMethod('rendering text layer', function (clip, range) {

            // render text cells into the canvas
            cellCanvas.render(clip, function (context) {

                // Collect text cells separated by writing direction, to be able to render BiDi text correctly.
                // The property 'direction' of the canvas rendering context is hardly supported in real-life,
                // but as a workaround, setting the 'dir' element attribute at the canvas element will do the
                // job. In order to prevent changing that attribute per text cell (performance!), texts will be
                // rendered separated by writing direction, resulting in two DOM modifications only.
                var textDescMap = { ltr: [], rtl: [] };
                renderCache.iterateTextCells(range, function (element) {
                    textDescMap[element.text.font.direction].push(element.text);
                });

                // ensure to render the text at font base line position
                context.setFontStyle({ align: 'left', baseline: 'alphabetic' });

                // render text cells separated by writing direction
                _.each(textDescMap, function (textDescs, textDir) {

                    // set 'dir' attribute at the canvas element which will influence
                    // the writing direction of the rendering context (see above)
                    cellCanvas.getNode().attr('dir', textDir);

                    // draw the texts and additional decorations into the canvas
                    textDescs.forEach(function (textDesc) {
                        context.clip(textDesc.clip, function () {

                            // initialize all settings for canvas rendering (text will be rendered in fill mode, without outline)
                            context.setLineStyle(null).setFillStyle(textDesc.fill).setFontStyle(textDesc.font);

                            // draw the text lines (or single words in justified alignment mode)
                            textDesc.lines.forEach(function (lineDesc) {
                                if (lineDesc.words) {
                                    lineDesc.words.forEach(function (wordDesc) {
                                        context.drawText(wordDesc.text, lineDesc.x + wordDesc.x, lineDesc.y);
                                    });
                                } else {
                                    context.drawText(lineDesc.text, lineDesc.x, lineDesc.y);
                                }
                            });

                            // render text decoration
                            if (textDesc.decoration) {
                                context.setLineStyle(textDesc.decoration.line).drawPath(textDesc.decoration.path);
                            }
                        });
                    });
                });
            });
        });

        /**
         * Creates the HTML mark-up for the range elements of the passed table
         * ranges.
         *
         * @param {Array<TableModel>|TableModel> tableModels
         *  An array with models of table ranges, or a single table model.
         */
        function createTableOutlineMarkup(tableModels) {

            // collect all table ranges (expand auto-filter to available content range)
            var tableRanges = RangeArray.map(tableModels, function (tableModel) {

                var // the original table range, as contained in the model
                    tableRange = tableModel.getRange(),
                    // the data range, expanded at bottom border for auto filters
                    dataRange = tableModel.getDataRange();

                // expand resulting range to data range (dynamically expanded for auto filters)
                if (dataRange) { tableRange = tableRange.boundary(dataRange); }

                // add name of the table range
                tableRange.name = tableModel.getName().toLowerCase();
                return tableRange;
            });

            // create the HTML mark-up for all visible table ranges
            var markup = '';
            gridPane.iterateRangesForRendering(tableRanges, function (range, rectangle) {
                markup += '<div class="range" style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '"';
                if (!range.name) { markup += ' data-autofilter="true"'; }
                markup += '></div>';
            }, { alignToGrid: true });

            return markup;
        }

        /**
         * Creates elements for the outlines of all visible table ranges.
         */
        function renderTableOutlines() {

            // get all table ranges in the active sheet
            var tableModels = docView.getTableCollection().getAllTables();

            // create the HTML mark-up for all visible table ranges
            tableLayerNode[0].innerHTML = createTableOutlineMarkup(tableModels);
        }

        /**
         * Updates the outline node of the auto filter. The auto filter changes
         * dynamically after adding or deleting cell contents below at its
         * bottom border.
         */
        function updateAutoFilterOutline() {

            // remove the current auto filter outline node
            tableLayerNode.find('>[data-autofilter]').remove();

            // create and insert the new HTML mark-up for the auto filter
            var tableModel = docView.getTableCollection().getTable('');
            if (tableModel) { tableLayerNode.append(createTableOutlineMarkup(tableModel)); }
        }

        /**
         * Returns a copy of the passed column/row interval, which will be
         * expanded to include the next available visible entry in both
         * directions.
         *
         * @param {Interval} interval
         *  The column/row interval to be expanded.
         *
         * @param {Boolean} columns
         *  Whether the passed interval is a column interval (true), or a row
         *  interval (false).
         *
         * @returns {Interval}
         *  The expanded interval. If there are no other visible columns/rows
         *  available before or after the interval, the respective index will
         *  not be changed.
         */
        function expandIntervalToNextVisible(interval, columns) {

            var // the column/row collection
                collection = columns ? docView.getColCollection() : docView.getRowCollection(),
                // preceding visible column/row entry
                prevEntry = collection.getPrevVisibleEntry(interval.first - 1),
                // following visible column/row entry
                nextEntry = collection.getNextVisibleEntry(interval.last + 1);

            return new Interval(prevEntry ? prevEntry.index : interval.first, nextEntry ? nextEntry.index : interval.last);
        }

        /**
         * Renders the cell background styles, the grid lines, the cell border
         * lines, and the cell display texts of the passed cell range into the
         * canvas element. The grid lines will be rendered above the cell
         * background rectangles, but below the border lines. The text contents
         * will be rendered on top of everything else.
         *
         * @param {Range} range
         *  The address of the cell range to be rendered into the canvas.
         */
        var renderCellLayers = RenderUtils.profileMethod('CellRenderer.renderCellLayers()', function (range) {

            // expand the passed range by one column/row in each direction to be able
            // to paint or delete thick borders covering the pixels of adjacent cells
            var colInterval = expandIntervalToNextVisible(range.colInterval(), true),
                rowInterval = expandIntervalToNextVisible(range.rowInterval(), false),
                renderRange = Range.createFromIntervals(colInterval, rowInterval);

            // restrict to own layer range
            renderRange = renderRange.intersect(layerRange);
            if (!renderRange) { return; }
            RenderUtils.log('original range: ' + range);
            RenderUtils.log('render range: ' + renderRange);

            // the rectangle of the rendering range, for clipping
            var clipRect = docView.getRangeRectangle(renderRange);

            // reduce the clipping rectangle by half of the minimum cell size (which is the maximum
            // border width) in each direction to not clear the outer borders of the rendered range
            if (layerRange.start[0] < renderRange.start[0]) {
                clipRect.left += LEADING_CLIP_SIZE_REDUCTION;
                clipRect.width -= LEADING_CLIP_SIZE_REDUCTION;
            }
            if (renderRange.end[0] < layerRange.end[0]) {
                clipRect.width -= TRAILING_CLIP_SIZE_REDUCTION;
            }
            if (layerRange.start[1] < renderRange.start[1]) {
                clipRect.top += LEADING_CLIP_SIZE_REDUCTION;
                clipRect.height -= LEADING_CLIP_SIZE_REDUCTION;
            }
            if (renderRange.end[1] < layerRange.end[1]) {
                clipRect.height -= TRAILING_CLIP_SIZE_REDUCTION;
            }

            // clear the rendering area
            cellCanvas.render(function (context) {
                context.clearRect(clipRect);
            });

            // render all cell background rectangles, grid lines, border lines, and texts
            renderFillLayer(clipRect, renderRange);
            renderGridLayer(clipRect, renderRange);
            renderBorderLayer(clipRect, renderRange);
            renderTextLayer(clipRect, renderRange);

            // update the auto filter outline range which may change after every cell change
            updateAutoFilterOutline();
        });

        /**
         * Registers the specified cell ranges to be rendered in a background
         * task.
         *
         * @param {RangeArray|Range} dirtyRanges
         *  The addresses of all cell ranges to be rendered, or the address of
         *  a single cell range.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.direct=false]
         *      Whether the cells in the passed ranges should be rendered
         *      immediately.
         */
        var registerPendingRanges = (function () {

            // direct callback: register passed ranges
            function registerRanges(ranges) {
                RenderUtils.log('CellRenderer.registerPendingRanges(): (1) store pending ranges');
                pendingRanges.append(ranges);
                RenderUtils.log('pending ranges: ' + pendingRanges);
            }

            // deferred callback: renders all registered ranges into the canvas element
            var renderRanges = RenderUtils.profileMethod('CellRenderer.registerPendingRanges(): (2) render pending ranges', function () {

                // the boundary range of all pending range to be rendered
                var renderRange =  pendingRanges.boundary();

                // performance: wait for more dirty ranges in the rendering cache
                // (it will trigger an update event that causes to run this method again)
                if (!renderRange || renderCache.hasDirtyRanges(renderRange)) { return; }

                // for simplicity, render all cells in the entire bounding range
                renderCellLayers(renderRange);

                // reset the pending ranges for next rendering cycle
                pendingRanges.clear();
            });

            // debounced version of the method CellRenderer.registerPendingRanges()
            var registerPendingRangesDebounced = self.createDebouncedMethod(registerRanges, renderRanges);

            // the actual method CellRenderer.registerPendingRanges() to be returned from the local scope
            function registerPendingRanges(ranges, options) {

                // render cell immediately if specified via the passed options, and if there
                // are no pending ranges, and if the passed ranges consist of a single row only
                if (Utils.getBooleanOption(options, 'direct', false) && pendingRanges.empty() && (ranges.length === 1) && (ranges.first().rows() === 1)) {
                    registerRanges(ranges);
                    renderRanges();
                } else {
                    registerPendingRangesDebounced(ranges);
                }
            }

            return registerPendingRanges;
        }());

        /**
         * Handler for the 'update:matrix' events triggered from the rendering
         * cache. Renders the specified cell ranges into the cell layers.
         */
        function updateMatrixHandler(event, ranges, options) {
            registerPendingRanges(ranges, options);
        }

        /**
         * Updates the cell grid lines after specific sheet view attributes
         * have been changed.
         */
        function changeViewAttributesHandler(event, attributes) {

            // visibility of cell grid has changed (grid color already handled via rendering cache)
            if ('showGrid' in attributes) {
                registerPendingRanges(layerRange);
                tableLayerNode.toggle(attributes.showGrid);
            }
        }

        // protected methods (renderer API) -----------------------------------

        /**
         * Changes the layers according to the passed layer range.
         */
        this.setLayerRange = RenderUtils.profileMethod('CellRenderer.setLayerRange()', function (newLayerRange, newLayerRectangle, dirtyRanges) {

            var // the model of the active sheet
                sheetModel = docView.getSheetModel(),
                // current zoom factor of active sheet
                newSheetZoom = sheetModel.getEffectiveZoom(),
                // the exact position of the new empty space in the rendering layer
                emptyRectangles = null,
                // the new ranges that became visible in the rendering layer
                renderRanges = null;

            // initialize canvas element (automatic scrolling of retained contents)
            RenderUtils.takeTime('initialize canvas element', function () {

                // render entire rectangle, if rendering layer was hidden before
                if (!layerRectangle) {

                    cellCanvas.initialize(newLayerRectangle, RenderUtils.CANVAS_RESOLUTION);
                    emptyRectangles = [newLayerRectangle];
                    renderRanges = new RangeArray(newLayerRange);

                } else {

                    // the zoom scaling factor, according to old and new zoom factor
                    var scale =  newSheetZoom / sheetZoom;

                    // 'scroll' canvas element to the new layer rectangle
                    emptyRectangles = cellCanvas.relocate(newLayerRectangle, scale);

                    // get cell ranges to be rendered from empty rectangles (layer rectangle does not match range boundaries)
                    renderRanges = RangeArray.map(emptyRectangles, function (rectangle) {
                        return sheetModel.getRangeFromRectangle(rectangle, { pixel: true });
                    });

                    // add the passed dirty ranges
                    renderRanges.append(dirtyRanges);

                    // Bug 38311: Clear canvas areas that are outside the sheet limits. Parent
                    // container node is in state overflow:visible in order to show elements
                    // that are party outside the sheet area, e.g. selection range elements
                    // with their tracking handles. If the lower or right boundary of the sheet
                    // is visible, the canvas may contain some rendering artifacts that will
                    // not be overdrawn or cleared later.
                    var sheetWidth = sheetModel.getColCollection().getTotalSize(),
                        sheetHeight = sheetModel.getRowCollection().getTotalSize(),
                        oversizeWidth = newLayerRectangle.left + newLayerRectangle.width - sheetWidth,
                        oversizeHeight = newLayerRectangle.top + newLayerRectangle.height - sheetHeight;
                    if ((oversizeWidth > 0) || (oversizeHeight > 0)) {
                        cellCanvas.render(function (context) {
                            if (oversizeWidth > 0) {
                                context.clearRect(sheetWidth, newLayerRectangle.top, oversizeWidth, newLayerRectangle.height);
                            }
                            if (oversizeHeight > 0) {
                                context.clearRect(newLayerRectangle.left, sheetHeight, newLayerRectangle.width, oversizeHeight);
                            }
                        });
                    }
                }
            });

            // store new layer range and rectangle for convenience
            layerRange = newLayerRange;
            layerRectangle = newLayerRectangle;
            sheetZoom = newSheetZoom;

            // immediately render the grid lines to reduce the 'empty space' effect when scrolling quickly
            emptyRectangles.forEach(function (clipRect, index) {
                renderGridLayer(clipRect, renderRanges[index]);
            });

            // register the new ranges for debounced rendering (needed to render missing parts of existing
            // cells at the old layer boundaries which will not be notified from the rendering cache)
            registerPendingRanges(renderRanges);

            // repaint all table range outlines
            renderTableOutlines();
        });

        /**
         * Resets this renderer, clears the DOM layer nodes.
         */
        this.hideLayerRange = function () {
            layerRange = layerRectangle = null;
            cellCanvas.initialize({ width: 0, height: 0 });
            tableLayerNode.empty();
            pendingRanges.clear();
        };

        // initialization -----------------------------------------------------

        // insert the canvas element into the DOM
        cellCanvas.getNode().insertBefore(tableLayerNode);

        // start rendering after any cell elements in the rendering cache have changed
        gridPane.listenToWhenVisible(this, docView.getRenderCache(), 'update:matrix', updateMatrixHandler);

        // update grid lines after switching on/off, or changing their color
        gridPane.listenToWhenVisible(this, docView, 'change:sheet:viewattributes', changeViewAttributesHandler);

        // render table range outlines
        gridPane.listenToWhenVisible(this, docView, 'insert:table delete:table change:table', self.createDebouncedMethod($.noop, renderTableOutlines));

        // destroy all class members on destruction
        this.registerDestructor(function () {
            cellCanvas.destroy();
            self = gridPane = docModel = docView = renderCache = null;
            cellCanvas = tableLayerNode = pendingRanges = null;
        });

    } // class CellRenderer

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: CellRenderer });

});
