/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/headerpane', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/view/popup/headercontextmenu',
    'io.ox/office/spreadsheet/view/mixin/headertrackingmixin',
    'io.ox/office/spreadsheet/view/render/renderutils',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, Forms, Tracking, TriggerObject, TimerMixin, SheetUtils, PaneUtils, HeaderContextMenu, HeaderTrackingMixin, RenderUtils, gt) {

    'use strict';

    var // convenience shortcuts
        Address = SheetUtils.Address,
        Interval = SheetUtils.Interval,

        // margin at trailing border of the entire sheet area, in pixels
        SCROLL_AREA_MARGIN = 30,

        // size of the resizer nodes, in pixels
        RESIZER_SIZE = 7,

        // tooltip labels for resizers
        COL_RESIZER_VISIBLE = gt('Drag to change column width'),
        COL_RESIZER_HIDDEN = gt('Drag to show hidden column'),
        ROW_RESIZER_VISIBLE = gt('Drag to change row height'),
        ROW_RESIZER_HIDDEN = gt('Drag to show hidden row');

    // class HeaderPane =======================================================

    /**
     * Represents a single row or column header pane in the spreadsheet view.
     * If the view has been split or frozen at a specific cell position, the
     * view will consist of up to four header panes (left and right column
     * header panes, and top and bottom row header panes).
     *
     * Triggers the following events:
     * - 'change:scrollsize'
     *      After the size of the scrollable area in this header pane has been
     *      changed. Event handlers receive the new scroll area size.
     * - 'change:scrollpos'
     *      After the absolute scroll position of this header pane has been
     *      changed. Event handlers receive the new scroll position.
     * - 'select:start'
     *      After selection tracking has been started. Event handlers receive
     *      the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number} index
     *          The zero-based index of the start column/row.
     *      (3) {String} mode
     *          The selection mode according to the keyboard modifier keys:
     *          - 'select': Standard selection without modifier keys.
     *          - 'append': Append new range to current selection (CTRL/META).
     *          - 'extend': Extend current active range (SHIFT).
     * - 'select:move'
     *      After the index of the tracked column/row has changed while
     *      selection tracking is active. Event handlers receive the following
     *      parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number} index
     *          The zero-based index of the current column/row.
     * - 'select:end'
     *      After selection tracking has been finished (either successfully or
     *      not). Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number} index
     *          The zero-based index of the last column/row.
     *      (3) {Boolean} success
     *          True if selecting has finished successfully, false if selecting
     *          has been canceled.
     * - 'resize:start'
     *      After column/row resize tracking has been started. Receives the
     *      current offset and initial size of the column/row.
     * - 'resize:move'
     *      After the new column/row size has changed while resize tracking is
     *      active. Receives the current offset and size of the column/row.
     * - 'resize:end'
     *      After column/row resize tracking has been finished (either
     *      successfully or not).
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view that contains this header pane.
     *
     * @param {String} paneSide
     *  The position of this header pane (one of 'left', 'right', 'top',
     *  'bottom', or 'corner').
     */
    function HeaderPane(docView, paneSide) {

        var // self reference
            self = this,

            // the spreadsheet model and view
            docModel = docView.getDocModel(),

            // whether this header pane contains columns
            columns = PaneUtils.isColumnSide(paneSide),

            // whether this header pane precedes another header pane
            leading = PaneUtils.isLeadingSide(paneSide),

            // the name of the scroll anchor sheet view attribute
            anchorName = PaneUtils.getScrollAnchorName(paneSide),

            // the container node of this header pane (must be focusable for tracking support)
            rootNode = $('<div class="header-pane" tabindex="-1" data-side="' + paneSide + '" data-orientation="' + (columns ? 'columns' : 'rows') + '">'),

            // the content node with the header cell and resizer nodes
            contentNode = $('<div class="header-content noI18n">').appendTo(rootNode),

            // the context menu for the header cells
            contextMenu = null,

            // size of the trailing scroll bar shown in associated grid panes (trailing panes only)
            scrollBarSize = leading ? 0 : columns ? Utils.SCROLLBAR_WIDTH : Utils.SCROLLBAR_HEIGHT,

            // the size of this header pane usable to show column/row header cells
            paneSize = 0,

            // whether this header pane is frozen (not scrollable)
            frozen = false,

            // distance between cell A1 and the top-left visible cell of this pane
            hiddenSize = 0,

            // the column/row interval to be displayed in the DOM (null indicates hidden header pane)
            interval = null,

            // the absolute position and size of the displayed column/row interval in the sheet
            position = { offset: 0, size: 0 },

            // the absolute scroll position in the associated grid panes
            scrollPos = 0,

            // the total size of the scroll area
            scrollSize = 0;

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Refreshes the appearance of this header pane according to the
         * position of the grid pane that is currently active.
         */
        function updateFocusDisplay() {

            var // which header pane is currently focused (while selection tracking is active)
                focusPaneSide = docView.getSheetViewAttribute('activePaneSide'),
                // which grid pane is currently focused
                focusPanePos = docView.getSheetViewAttribute('activePane'),
                // whether this grid pane appears focused
                focused = false;

            if (docView.hasFrozenSplit()) {
                // always highlight all header panes in frozen split mode
                focused = true;
            } else if (_.isString(focusPaneSide)) {
                // a header pane is active (selection tracking): focus this
                // header pane if it is active or in the opposite direction
                focused = (focusPaneSide === paneSide) || (PaneUtils.isColumnSide(focusPaneSide) !== columns);
            } else {
                // a single grid pane is focused
                focused = (PaneUtils.getColPaneSide(focusPanePos) === paneSide) || (PaneUtils.getRowPaneSide(focusPanePos) === paneSide);
            }

            rootNode.toggleClass(Forms.FOCUSED_CLASS, focused);
        }

        /**
         * Refreshes the position of the touch resizer handle node.
         *
         * @param {Number} [size]
         *  A custom size of the column/row the resizer node is attached to. If
         *  omitted, uses the current size of the column/row.
         */
        function updateTouchResizerPosition(size) {

            var resizerNode = contentNode.find('>.touchresizer'),
                index = Utils.getElementAttributeAsInteger(resizerNode, 'data-index'),
                entry = self.getColRowCollection().getEntry(index),
                offset = entry.offset + (_.isNumber(size) ? size : entry.size) - position.offset;

            resizerNode.css(columns ? 'left' : 'top', offset + 'px');
        }

        /**
         * Redraws the selection highlighting in the header cells of this
         * header pane.
         */
        function renderSelection() {

            var // the selected ranges
                ranges = docView.getSelectedRanges(),
                // the merged ranges of selected columns/rows
                selectionIntervals = ranges.intervals(columns).merge(),
                // current index into the intervals array
                arrayIndex = 0;

            // find the first selection interval that ends at or before the passed index
            function findSelectionInterval(index) {
                while ((arrayIndex < selectionIntervals.length) && (selectionIntervals[arrayIndex].last < index)) { arrayIndex += 1; }
                return selectionIntervals[arrayIndex];
            }

            // find the first interval, and iterate all cell nodes
            findSelectionInterval(interval.first);
            Utils.iterateSelectedDescendantNodes(contentNode, '.cell', function (cellNode) {
                var index = Utils.getElementAttributeAsInteger(cellNode, 'data-index'),
                    selectionInterval = findSelectionInterval(index),
                    selected = _.isObject(selectionInterval) && (selectionInterval.first <= index);
                Forms.selectNodes(cellNode, selected);
            }, undefined, { children: true });

            // on touch devices, update the resize handle for entire columns/rows
            if (Utils.TOUCHDEVICE) {
                if ((ranges.length === 1) && docModel.isFullRange(ranges.first(), columns)) {
                    contentNode.find('>.touchresizer').show().attr('data-index', ranges.first().getEnd(columns));
                    updateTouchResizerPosition();
                } else {
                    contentNode.find('>.touchresizer').hide();
                }
            }
        }

        /**
         * Renders all header cells in the current column/row interval.
         */
        function renderCells() {

            var // the column/row collection
                collection = self.getColRowCollection(),
                // the entry descriptors needed for rendering
                entries = [],
                // the HTML mark-up for all header cells to be rendered
                markup = '',
                // whether the previous entry was hidden
                prevHidden = false,
                // name of specific CSS attribute
                OFFSET_NAME = columns ? 'left' : 'top',
                SIZE_NAME = columns ? 'width' : 'height';

            // returns the mark-up of an offset CSS property
            function getOffset(offset) {
                return OFFSET_NAME + ':' + offset + 'px;';
            }

            // returns the mark-up of a size CSS property
            function getSize(size) {
                return SIZE_NAME + ':' + size + 'px;';
            }

            // returns the mark-up of an offset and a size CSS property
            function getPosition(offset, size) {
                return getOffset(offset) + getSize(size);
            }

            // creates a new entry in the 'entries' array according to the passed column/row descriptor
            function createEntry(colRowData) {

                var // the descriptor for the current entry
                    entry = { index: colRowData.index, size: colRowData.size };

                // start offset of the cell node and resizer node
                entry.offset = colRowData.offset - position.offset;
                entry.resizerOffset = entry.offset + colRowData.size;

                // adjust resizer position according to hidden state
                if (entry.size > 0) {
                    entry.resizerOffset -= Math.ceil(RESIZER_SIZE / 2);
                } else if (entries.length > 0) {
                    _.last(entries).resizerOffset -= Math.ceil(RESIZER_SIZE / 2);
                }

                // store new descriptor in result array
                entries.push(entry);
            }

            // create a descriptor array first (needed to manipulate preceding entries before rendering)
            collection.iterateEntries(interval, createEntry, { hidden: 'last' });

            // add another entry for the trailing hidden columns/rows
            if ((entries.length === 0) || (_.last(entries).index < interval.last)) {
                createEntry(collection.getEntry(interval.last));
            }

            // set font size and line height (column headers only, at cell level for row headers) at content node
            contentNode.css('font-size', docView.getHeaderFontSize() + 'pt');
            if (columns) { contentNode.css('line-height', (rootNode.height() - 1) + 'px'); }

            // generate HTML mark-up text for all header cells
            entries.forEach(function (entry) {

                var // whether the current entry is hidden and precedes a visible entry
                    hidden = entry.size === 0,
                    // tool tip for the resizer
                    resizerTooltip = columns ? (hidden ? COL_RESIZER_HIDDEN : COL_RESIZER_VISIBLE) : (hidden ? ROW_RESIZER_HIDDEN : ROW_RESIZER_VISIBLE),
                    // entry index as element attribute
                    dataAttr = ' data-index="' + entry.index + '"';

                // generate mark-up for the header cell node
                if (hidden) {
                    markup += '<div class="marker" style="' + getOffset(entry.offset) + '"><div><div></div></div></div>';
                } else {
                    markup += '<div class="cell' + (prevHidden ? ' prev-hidden' : '') + '"';
                    markup += ' style="' + getPosition(entry.offset, entry.size);
                    if (!columns) { markup += 'line-height:' + (entry.size - 1) + 'px;'; }
                    markup += '"' + dataAttr + '>' + Address.stringifyIndex(entry.index, columns) + '</div>';
                }

                // generate mark-up for the resizer node
                if (!Utils.TOUCHDEVICE) {
                    markup += '<div class="resizer' + (hidden ? ' collapsed' : '') + '"';
                    markup += ' style="' + getPosition(entry.resizerOffset, RESIZER_SIZE) + '"';
                    markup += ' title="' + resizerTooltip + '"' + dataAttr + '></div>';
                }

                // update the prevHidden flag
                prevHidden = hidden;
            });

            // generate mark-up for global resizer node for touch devices
            if (Utils.TOUCHDEVICE) {
                markup += '<div class="touchresizer">' + Forms.createIconMarkup(columns ? 'fa-arrows-h' : 'fa-arrows-v') + '</div>';
            }

            // insert entire HTML mark-up into the cell container node
            contentNode[0].innerHTML = markup;
        }

        /**
         * Returns the maximum possible size of the scrolling area, according
         * to the size of the entire sheet, ignoring the current used area.
         */
        function getMaxScrollSize() {
            // add fixed margin after the sheet, add scroll bar size in trailing header panes
            return self.getColRowCollection().getTotalSize() - hiddenSize + SCROLL_AREA_MARGIN + scrollBarSize;
        }

        /**
         * Returns the scrolling position reduced to the valid scrolling range,
         * according to the size of the entire sheet, ignoring the current used
         * area.
         */
        function getValidScrollPos(pos) {
            return Math.max(0, Math.min(pos, getMaxScrollSize() - paneSize));
        }

        /**
         * Updates the size of the scrollable area according to the used area
         * in the sheet, and the current scroll position. Tries to reserve
         * additional space for the full size of this header pane at the
         * trailing border, to be able to scroll page-by-page into the unused
         * area of the sheet.
         */
        function updateScrollAreaSize(newScrollPos) {

            var // the column/row collection
                collection = self.getColRowCollection(),
                // number of used columns/rows in the sheet
                usedCount = docView.getCellCollection().getUsedCount(columns),
                // the total size of the used area in the sheet, in pixels
                usedSize = Math.max(0, collection.getEntry(usedCount).offset - hiddenSize),
                // the absolute size of the scrollable area
                newScrollSize = newScrollPos + paneSize;

            // scrollable area includes the used area, and the passed scroll position,
            // expanded by the own client size to be able to scroll forward
            newScrollSize = Math.max(usedSize, newScrollSize) + paneSize;

            // restrict to the maximum scrolling size of the sheet
            newScrollSize = Math.min(newScrollSize, getMaxScrollSize());

            // trigger event, if the new scroll area size has changed
            if (scrollSize !== newScrollSize) {
                scrollSize = newScrollSize;
                self.trigger('change:scrollsize', scrollSize);
            }
        }

        /**
         * Updates the relative scroll position and size of the content node,
         * and notifies listeners about the changed scroll position.
         */
        function updateScrollPosition(newScrollPos) {

            // restrict scroll position to the maximum possible scroll position
            // (prevent that header pane scrolls more than the grid panes can scroll)
            scrollPos = getValidScrollPos(newScrollPos);

            var // the column/row collection
                collection = self.getColRowCollection(),
                // the relative offset of the content node
                contentOffset = position.offset - (scrollPos + hiddenSize),
                // maximum size (restrict to sheet size)
                contentSize = Math.min(position.size, collection.getTotalSize() - position.offset);

            // set the position and size of the content node
            if (columns) {
                contentNode.css({ left: contentOffset, width: contentSize });
            } else {
                contentNode.css({ top: contentOffset, height: contentSize });
            }

            // notify listeners (do not check whether the scroll position really has
            // changed, depending grid panes may have a wrong scroll position initially
            // due to wrong size of the scroll area)
            self.trigger('change:scrollpos', scrollPos);
        }

        /**
         * Returns the complete column/row interval that can be displayed in
         * this header pane. If the active sheet does not contain a frozen
         * column/row split, this interval will contain all columns/rows of the
         * sheet, otherwise the interval will contain the available columns or
         * rows according to the position of this header pane, and the current
         * split settings.
         *
         * @returns {Interval}
         *  The complete column/row interval available in this header pane.
         */
        function getAvailableInterval() {

            var // the model of the active sheet
                sheetModel = docView.getSheetModel(),
                // the frozen column/row interval
                frozenInterval = null;

            // get frozen interval from sheet model
            if (sheetModel.hasFrozenSplit()) {
                frozenInterval = columns ? sheetModel.getSplitColInterval() : sheetModel.getSplitRowInterval();
            }

            // return entire column/row interval of the sheet in unfrozen views
            if (!frozenInterval) {
                return self.getColRowCollection().getFullInterval();
            }

            // build the available interval according to the own position
            return leading ? frozenInterval : new Interval(frozenInterval.last + 1, docModel.getMaxIndex(columns));
        }

        /**
         * Registers an event handler at the specified instance that will only
         * be invoked when this header pane is visible.
         */
        function listenToWhenVisible(source, type, handler) {
            self.listenTo(source, type, function () {
                if (interval) {
                    handler.apply(self, _.toArray(arguments));
                }
            });
        }

        /**
         * Updates this header pane after specific sheet view attributes have
         * been changed.
         */
        function changeSheetViewAttributesHandler(event, attributes) {

            // refresh focus display
            if (Utils.hasProperty(attributes, /^activePane/)) {
                updateFocusDisplay();
            }

            // repaint the selected header cells
            if ('selection' in attributes) {
                renderSelection();
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the document view that contains this header pane.
         *
         * @returns {SpreadsheetView}
         *  The document view that contains this header pane.
         */
        this.getDocView = function () {
            return docView;
        };

        /**
         * Returns the root DOM node of this header pane.
         *
         * @returns {jQuery}
         *  The root node of this header pane, as jQuery object.
         */
        this.getNode = function () {
            return rootNode;
        };

        /**
         * Returns the content node of this header pane, containing the cell
         * nodes and resize tracking nodes.
         *
         * @returns {jQuery}
         *  The content node of this header pane, as jQuery object.
         */
        this.getContentNode = function () {
            return contentNode;
        };

        /**
         * Returns the column or row collection of the active sheet, depending
         * on the direction of this header pane.
         *
         * @returns {ColRowCollection}
         *  The column collection of the active sheet, if this is a column
         *  header pane; otherwise the row collection of the active sheet.
         */
        this.getColRowCollection = function () {
            return columns ? docView.getColCollection() : docView.getRowCollection();
        };

        /**
         * Initializes the settings and layout of this header pane.
         *
         * @param {Object} settings
         *  The view settings of the pane side represented by this header pane.
         *  Supports the following properties:
         *  @param {Number} settings.offset
         *      The absolute offset of the header pane in the view root node,
         *      in pixels.
         *  @param {Number} settings.size
         *      The size of the header pane (width of column panes, height of
         *      row panes), in pixels. The value zero will have the effect to
         *      hide the entire header pane.
         *  @param {Boolean} settings.frozen
         *      Whether the header pane and the associated grid panes are
         *      frozen (not scrollable in the main direction).
         *  @param {Number} settings.hiddenSize
         *      The total size of the hidden columns/rows in front of the sheet
         *      in frozen view mode.
         *
         * @returns {HeaderPane}
         *  A reference to this instance.
         */
        this.initializePaneLayout = function (settings) {

            // copy passed settings
            paneSize = settings.size;
            frozen = settings.frozen;
            hiddenSize = settings.hiddenSize;

            // initialize according to visibility
            if (paneSize > 0) {

                // show pane root node and initialize auto-scrolling
                rootNode.show();
                Tracking.enableTracking(rootNode, Utils.extendOptions(PaneUtils.DEFAULT_TRACKING_OPTIONS, {
                    autoScroll: frozen ? false : (columns ? 'horizontal' : 'vertical'),
                    borderNode: rootNode
                }));

                // position and size
                rootNode.css(columns ?
                    { left: settings.offset, width: paneSize, height: docView.getHeaderHeight() } :
                    { top: settings.offset, width: docView.getHeaderWidth(), height: paneSize }
                );

                // update CSS classes at root node for focus display
                updateFocusDisplay();

            } else {

                // hide pane root node and deinitialize auto-scrolling
                rootNode.hide();
                Tracking.disableTracking(rootNode);
            }

            return this;
        };

        /**
         * Recalculates the current row/column interval and its position and
         * size in the sheet, and renders all visible header cells and the
         * selection on demand.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.hide=false]
         *      If set to true, the layer interval will be explicitly hidden.
         *  @param {Boolean} [options.force=false]
         *      If set to true, the layer interval will always be recalculated
         *      regardless of the current scroll position. This option will be
         *      ignored, if the option 'hide' has been set.
         *
         * @returns {HeaderBoundary|Null}
         *  A descriptor containing the new layer interval and position (if the
         *  properties in this object are null, the layer interval has been
         *  hidden); or null, if nothing has changed in this header pane.
         */
        this.updateLayerInterval = function (options) {

            // invalidates the layer interval and other members, and returns the descriptor for the caller
            function invalidateIntervalAndReturn() {

                // nothing to do, if the interval is already invalid
                if (!interval) { return null; }

                // reset members, return 'hide' flag
                interval = position = null;
                scrollPos = scrollSize = 0;
                return new PaneUtils.HeaderBoundary();
            }

            // invalidate existing layer interval (header pane hidden, or 'hide' flag passed)
            if ((paneSize === 0) || Utils.getBooleanOption(options, 'hide', false)) {
                return invalidateIntervalAndReturn();
            }

            var // the column/row collection
                collection = self.getColRowCollection(),
                // the current scroll anchor from the sheet attributes
                scrollAnchor = docView.getSheetViewAttribute(anchorName),
                // the current effective scroll position
                newScrollPos = getValidScrollPos(collection.convertScrollAnchorToPixel(scrollAnchor) - hiddenSize),
                // the absolute position of the first visible pixel
                visibleOffset = newScrollPos + hiddenSize,
                // the absolute offset behind the last visible pixel
                visibleEndOffset = visibleOffset + paneSize,
                // the additional size before and after the visible area
                additionalSize = frozen ? 0 : Math.ceil(paneSize * PaneUtils.ADDITIONAL_SIZE_RATIO),
                // whether updating the interval is required due to passed refresh settings
                forceRefresh = Utils.getBooleanOption(options, 'force', false);

            // check if the layer interval does not need to be recalculated: layer interval exists, update not forced,
            // and the visible area of the header pane is inside the layer interval (with a small distance)
            if (interval && position && !forceRefresh && (frozen ||
                (((position.offset <= 0) || (visibleOffset - position.offset >= additionalSize / 3)) &&
                ((collection.getTotalSize() <= position.offset + position.size) || (position.offset + position.size - visibleEndOffset >= additionalSize / 3))
            ))) {
                updateScrollPosition(newScrollPos);
                return null;
            }

            var // exact pixel size for the new interval with additional buffer size
                // (round up to multiples of 100 to reduce expensive resizing while rendering)
                boundSize = Utils.roundUp(paneSize + 2 * additionalSize, 100),
                // exact pixel position for the new interval
                boundOffset = visibleOffset - Math.round((boundSize - paneSize) / 2),
                // the available interval
                availableInterval = getAvailableInterval(),
                // the data of the new first column/row
                firstEntryData = collection.getEntryByOffset(boundOffset, { pixel: true }),
                // the data of the new last column/row
                lastEntryData = collection.getEntryByOffset(boundOffset + boundSize - 1, { pixel: true });

            // calculate the new visible column/row interval
            interval = new Interval(
                Math.max(availableInterval.first, (firstEntryData.offset === 0) ? 0 : firstEntryData.index),
                Math.min(availableInterval.last, (lastEntryData.offset + lastEntryData.size >= collection.getTotalSize()) ? collection.getMaxIndex() : lastEntryData.index)
            );

            // set the new exact pixel position
            position = { offset: boundOffset, size: boundSize };

            // sanity check
            if (interval.last < interval.first) {
                Utils.warn('HeaderPane.updateLayerInterval(): invalid interval: side=' + paneSide + ' interval=' + interval.stringifyAs(columns));
                return invalidateIntervalAndReturn();
            }

            // update scroll size and position according to the new interval
            updateScrollAreaSize(newScrollPos);
            updateScrollPosition(newScrollPos);

            // render header cells and the selection
            renderCells();
            renderSelection();

            // return the new interval and position
            RenderUtils.log('HeaderPane.updateLayerInterval(): side=' + paneSide + ' size=' + paneSize + ' interval=' + interval.stringifyAs(columns) + ' position=' + JSON.stringify(position));
            return new PaneUtils.HeaderBoundary(interval, position);
        };

        /**
         * Returns whether this header pane is currently visible (depending on
         * its position, and the view split settings of the active sheet).
         *
         * @returns {Boolean}
         *  Whether this header pane is currently visible.
         */
        this.isVisible = function () {
            return (paneSize > 0) && _.isObject(interval);
        };

        /**
         * Returns the complete column/row interval that can be displayed in
         * this header pane. If the active sheet does not contain a frozen
         * column/row split, this interval will contain all columns/rows of the
         * sheet, otherwise the interval will contain the available columns or
         * rows according to the position of this header pane, and the current
         * split settings.
         *
         * @returns {Interval|Null}
         *  The complete column/row interval available in this header pane, in
         *  the zero-based index properties 'first' and 'last'; or null, if
         *  this header pane is currently hidden.
         */
        this.getAvailableInterval = function () {
            return this.isVisible() ? getAvailableInterval() : null;
        };

        /**
         * Returns the column/row interval currently shown in this header pane.
         *
         * @returns {Interval|Null}
         *  The column/row interval currently displayed in this header pane; or
         *  null, if this header pane is currently hidden.
         */
        this.getRenderInterval = function () {
            return interval.clone();
        };

        /**
         * Returns the offset and size of the entire column/row interval
         * currently covered by this header pane, including the columns/rows
         * around the visible area.
         *
         * @returns {Object|Null}
         *  The offset and size of the displayed column/row interval in pixels;
         *  or null, if this header pane is currently hidden.
         */
        this.getRenderPosition = function () {
            return _.clone(position);
        };

        /**
         * Returns the offset and size of the visible area currently shown in
         * this header pane.
         *
         * @returns {Object}
         *  The offset and size of the visible area in this header pane, in
         *  pixels.
         */
        this.getVisiblePosition = function () {
            return { offset: hiddenSize + scrollPos, size: paneSize };
        };

        /**
         * Returns the current scroll position of this header pane.
         *
         * @returns {Number}
         *  The current scroll position, in pixels.
         */
        this.getScrollPos = function () {
            return scrollPos;
        };

        /**
         * Returns the size of the scrollable area according to the used area
         * in the sheet, and the current scroll position.
         *
         * @returns {Number}
         *  The size of the scrollable area, in pixels.
         */
        this.getScrollSize = function () {
            return scrollSize;
        };

        /**
         * Returns the zero-based index of the first visible column/row in this
         * header pane. The column/row is considered visible, it at least half
         * of its total size is really visible.
         *
         * @returns {Number}
         *  The zero-based index of the first visible column/row.
         */
        this.getFirstVisibleIndex = function () {

            var // the column/row collection of the active sheet
                collection = this.getColRowCollection(),
                // the current scroll anchor
                scrollAnchor = docView.getSheetViewAttribute(anchorName),
                // the column/row index of the scroll anchor
                scrollIndex = Math.floor(scrollAnchor),
                // the layout data of the next column/row
                nextEntry = null;

            // check if the cell is visible less than half, go to next cell in that case
            if ((scrollAnchor % 1 > 0.5) || !collection.isEntryVisible(scrollIndex)) {
                nextEntry = collection.getNextVisibleEntry(scrollIndex + 1);
            }

            return nextEntry ? nextEntry.index : scrollIndex;
        };

        /**
         * Changes the scroll position of this header pane, and triggers a
         * 'change:scrollpos' event that causes all associated grid panes to
         * update their own scroll position, and optionally a debounced
         * 'change:scrollsize' event, if the total size of the scroll area has
         * been changed due to the new scroll position.
         *
         * @param {Number} pos
         *  The new scroll position in the sheet, in pixels (the value 0
         *  corresponds to the first available position; in frozen panes this
         *  value does NOT include the leading hidden part of the sheet).
         *
         * @returns {HeaderPane}
         *  A reference to this instance.
         */
        this.scrollTo = this.createDebouncedMethod(function (pos) {

            var // the column/row collection of the active sheet
                collection = this.getColRowCollection(),
                // the new scroll position
                newScrollPos = getValidScrollPos(pos);

            // do nothing if scroll position did not change
            if (frozen || (scrollPos === newScrollPos)) { return this; }

            // recalculate scroll area size immediately, if new scroll position is outside
            if (newScrollPos > scrollSize - paneSize) {
                updateScrollAreaSize(newScrollPos);
            }

            // immediately update internal scroll position, this prevents unnecessary updates
            scrollPos = newScrollPos;

            // update scroll anchor view properties, this will trigger change
            // handler from the view that repositions the content node
            docView.setSheetViewAttribute(anchorName, collection.getScrollAnchorByOffset(newScrollPos + hiddenSize, { pixel: true }));

            return this;

        }, function () { updateScrollAreaSize(scrollPos); }, { delay: 500 });

        /**
         * Changes the scroll position of this header pane relative to the
         * current scroll position.
         *
         * @param {Number} diff
         *  The difference for the scroll position, in pixels.
         *
         * @returns {HeaderPane}
         *  A reference to this instance.
         */
        this.scrollRelative = function (diff) {
            return this.scrollTo(scrollPos + diff);
        };

        /**
         * Scrolls this header pane to make the specified area visible.
         *
         * @param {Number} offset
         *  The start offset of the area to be made visible, in pixels.
         *
         * @param {Number} size
         *  The size of the area to be made visible, in pixels.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.forceLeading=false]
         *      If set to true, the passed offset will be scrolled to the
         *      leading border of the header pane, also if the passed area is
         *      already visible.
         *
         * @returns {HeaderPane}
         *  A reference to this instance.
         */
        this.scrollToPosition = function (offset, size, options) {

            // move the area offset to the leading border if specified
            if (Utils.getBooleanOption(options, 'forceLeading', false)) {
                return this.scrollTo(offset - hiddenSize);
            }

            var // the end offset of the area (with little extra space to make selection border visible)
                endOffset = offset + size + 2,
                // the absolute offset and size of the visible area
                visibleOffset = this.getVisiblePosition().offset,
                // the inner pane side (without scroll bars in grid panes)
                innerSize = paneSize - scrollBarSize,
                // whether the leading border is outside the visible area
                leadingOutside = offset < visibleOffset,
                // whether the trailing border is outside the visible area
                trailingOutside = endOffset > visibleOffset + innerSize,
                // new scrolling position
                newScrollPos = scrollPos;

            if (leadingOutside && !trailingOutside) {
                newScrollPos = Math.max(offset, endOffset - innerSize) - hiddenSize;
            } else if (!leadingOutside && trailingOutside) {
                newScrollPos = Math.min(offset, endOffset - innerSize) - hiddenSize;
            }

            // update scroll position
            return this.scrollTo(newScrollPos);
        };

        /**
         * Scrolls this header pane to make the specified column/row visible.
         *
         * @param {Number} index
         *  The zero-based column/row index to be made visible.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.forceLeading=false]
         *      If set to true, the passed offset will be scrolled to the
         *      leading border of the header pane, also if the passed area is
         *      already visible.
         *
         * @returns {HeaderPane}
         *  A reference to this instance.
         */
        this.scrollToEntry = function (index, options) {

            var // the position and size of the target column/row
                entryData = this.getColRowCollection().getEntry(index);

            // update scroll position
            return this.scrollToPosition(entryData.offset, entryData.size, options);
        };

        // initialization -----------------------------------------------------

        // tracking mix-in expects fully initialized header pane
        HeaderTrackingMixin.call(this, paneSide);

        // marker for touch devices and browser types
        Forms.addDeviceMarkers(rootNode);

        // create the context menu
        contextMenu = new HeaderContextMenu(this, columns);

        // update scroll area size according to the size of the used area in the sheet
        this.listenTo(docView, 'change:usedarea', function () { updateScrollAreaSize(scrollPos); });

        // listen to view events (only if this pane is visible)
        listenToWhenVisible(docView, 'change:sheet:viewattributes', changeSheetViewAttributesHandler);

        // update the position of the touch resizer node while tracking
        if (Utils.TOUCHDEVICE) {
            self.on({
                'resize:move': function (event, offset, size) { updateTouchResizerPosition(size); },
                'resize:end': function () { updateTouchResizerPosition(); }
            });
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            contextMenu.destroy();
            self = docModel = docView = contextMenu = null;
        });

    } // class HeaderPane

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: HeaderPane });

});
