/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Kai Ahrens <kai.ahrens@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/dialog/functiondialog', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/dialogs',
    'io.ox/office/spreadsheet/view/labels',
    'gettext!io.ox/office/spreadsheet/main',
    'less!io.ox/office/spreadsheet/view/dialog/functiondialog'
], function (Utils, Forms, Dialogs, Labels, gt) {

    'use strict';

    var // dialog width according to device type
        DIALOG_WIDTH = ox.mobile ? 320 : 540,

        // special drop-down list entry value for 'show all functions'
        CATEGORY_ALL = '__all__',

        DIALOG_TITLE =
            //#. 'Insert Function' dialog title
            gt.pgettext('function-dialog', 'Insert Function'),

        DIALOG_OK_LABEL =
            //#. 'Insert Function' dialog: label for OK button
            gt.pgettext('function-dialog', 'Insert'),

        PICK_CATEGORY_LABEL =
            //#. 'Insert Function' dialog: label for a drop-down menu for categories
            //#. %1$s is the selected function category (displayed as clickable link)
            //#, c-format
            gt.pgettext('function-dialog', 'Pick a category: %1$s'),

        // special drop-down list entry for 'all categories'
        ALL_FUNCTIONS_LABEL =
            //#. 'Insert Function' dialog: list entry to show functions from all categories
            gt.pgettext('function-dialog', 'All'),

        PICK_FUNCTION_LABEL =
            //#. 'Insert Function' dialog: label for a list box containing function names
            gt.pgettext('function-dialog', 'Pick a function'),

        SYNTAX_LABEL =
            //#. 'Insert Function' dialog: subtitle for the short syntax description of a function
            gt.pgettext('function-dialog', 'Syntax'),

        PARAMETERS_LABEL =
            //#. 'Insert Function' dialog: subtitle for the parameter list of a function
            gt.pgettext('function-dialog', 'Parameters'),

        OPTIONAL_PARAM_NAME =
            //#. 'Insert Function' dialog: name of an optional parameter in the description of a function
            //#. %1$s is the name of the parameter to be marked as optional
            //#, c-format
            gt.pgettext('function-dialog', '%1$s (optional)');

    // class FunctionDialog ===================================================

    /**
     * A modal dialog showing an overview of all function categories, all
     * functions and their appropriate description, function syntax and
     * parameters. The user can select a function from one of the categories.
     *
     * @constructor
     *
     * @extends Dialogs.ModalDialog
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view instance containing this dialog.
     *
     * @param {String} [text]
     *  An optional cell text. If specified, it will be tried to extract a
     *  function name from, which will be selected initially in the dialog.
     */
    var FunctionDialog = Dialogs.ModalDialog.extend({ constructor: function (docView, text) {

        var // the application instance, needed to resolve translated function names
            app = docView.getApp(),

            // the descriptors of all supported functions
            descriptors = docView.getDocModel().getFormulaInterpreter().getFunctionDescriptors(),

            // container node for first column (category drop-down, function list)
            controlRootNode = $('<div class="list">'),

            // container node for second column (description text)
            descRootNode = $('<div class="desc">'),

            // the <select> control with the function names
            categorySelectNode = null,
            functionSelectNode = null;

        // base constructor ---------------------------------------------------

        Dialogs.ModalDialog.call(this, { title: DIALOG_TITLE, width: DIALOG_WIDTH, okLabel: DIALOG_OK_LABEL });

        // private methods ----------------------------------------------------

        /**
         * Returns the fully formatted syntax string for the passed function
         * help descriptor.
         *
         * @param {Object} functionHelp
         *  The function help descriptor, as returned by the method
         *  SpreadsheetApplication.getFunctionHelp().
         *
         * @returns {String}
         *  The formatted syntax for the specified function.
         */
        function getFunctionSyntax(functionHelp) {

            var syntax = '',
                SEP = app.getListSeparator() + ' ';

            if (_.isObject(functionHelp)) {

                var cycleCount = functionHelp.cycleCount,
                    cycleLength = functionHelp.cycleLength;

                syntax += functionHelp.name + '(';

                for (var i = 0, len = functionHelp.params.length; i < len; ++i) {

                    var param = functionHelp.params[i];

                    if ((cycleCount === 0) || (functionHelp.repeatStart < 0) || (i < functionHelp.repeatStart)) {
                        syntax += (param.optional ? '[' : '') + ((i > 0) ? SEP : '') + param.name + (param.optional ? ']' : '');
                        continue;
                    }

                    var cycleParams =  [];

                    for (var last = i + cycleLength; i < last; ++i) {
                        param = functionHelp.params[i];
                        cycleParams.push(param.name);
                        syntax += ((i > 0) ? ', ' : '') + param.name + '1';
                    }

                    syntax += '[...';

                    for (var n = 0, lastCycleNumber = cycleCount; n < cycleLength; ++n) {
                        syntax += SEP + cycleParams[n] + lastCycleNumber;
                    }

                    syntax += ']';
                    break;
                }

                syntax += ')';
            }

            return syntax;
        }

        /**
         * Updates the function description, syntax and parameters, based on
         * the given native function name.
         *
         * @param {String} nativeFuncName
         *  The native function name, as used by filling the function list box.
         */
        function updateFunctionDescription(nativeFuncName) {

            // clean the function description node
            descRootNode.empty();

            // fetch the function help data for the specified function
            var functionHelp = app.getFunctionHelp(nativeFuncName);
            if (!functionHelp) { return; }

            // create the general function description (name, description, syntax)
            descRootNode.append(
                $('<p class="desc-title main">').text(_.noI18n(functionHelp.name)),
                $('<p>').text(_.noI18n(functionHelp.description)),
                $('<p class="desc-title sub">').text(SYNTAX_LABEL),
                $('<p>').text(_.noI18n(getFunctionSyntax(functionHelp)))
            );

            // nothing more to do for parameterless functions
            if (functionHelp.params.length === 0) { return; }

            // create detailed parameter description
            var jqParamList = $('<ul class="desc-params">');
            descRootNode.append($('<p class="desc-title sub">').text(PARAMETERS_LABEL), jqParamList);
            functionHelp.params.forEach(function (param) {
                var paramName = param.optional ? gt.format(OPTIONAL_PARAM_NAME, _.noI18n(param.name)) : _.noI18n(param.name);
                jqParamList.append(
                    $('<li>').append(
                        $('<span class="param-name">').text(paramName),
                        $('<span>&mdash;</span>'),
                        $('<span>').text(_.noI18n(param.description))
                    )
                );
            });
        }

        /**
         * Updates the function list. To be called, whenever the category has
         * been changed or initially after executing the dialog. This function
         * itself calls all other functions to update the function description,
         * syntax and parameters.
         *
         * @param {String} category
         *  The category identifier, as used by filling the category drop-down
         *  menu.
         */
        function updateFunctionList(category) {

            var // whether to show all functions
                showAll = category === CATEGORY_ALL,
                // the resulting function names (native and translated)
                functionNames = [];

            // filter functions by category, unless the passed category is the special 'all functions' identifier
            _.each(descriptors, function (descriptor, nativeFuncName) {
                if (showAll || descriptor.category[category]) {
                    functionNames.push({ native: nativeFuncName, localized: app.getLocalizedFunctionName(nativeFuncName) });
                }
            });

            // sort all functions alphabetically
            functionNames = _.sortBy(functionNames, 'localized');

            // fill the function list box
            var listMarkup = '';
            functionNames.forEach(function (functionName) {
                listMarkup += Forms.createElementMarkup('option', {
                    attributes: { value: functionName.native },
                    content: functionName.localized
                });
            });
            functionSelectNode[0].innerHTML = listMarkup;

            // select the first entry in the list

            functionSelectNode.val(functionNames[0].native);
            updateFunctionDescription(functionNames[0].native);
        }

        /**
         * Creates and layouts all dialog controls, contained within the body of the
         * dialog.
         * This function, in conjunction with the dialogs' less file,
         * determines the general visual layout and functionality of the dialog
         */
        function initControls() {
            var jqMenuList = null,
                jqMenuAnchor = null,
                jqCategoriesCtl = null;

            if (!_.browser.iOS) {
                // the anchor element used to open the drop-down menu
                jqMenuAnchor = $('<a href="#" tabindex="1" aria-haspopup="true" data-toggle="dropdown">');
                jqMenuAnchor.text(ALL_FUNCTIONS_LABEL);

                // the drop-down list attached to the anchor element
                jqMenuList = $('<ul class="dropdown-menu categories" role="menu">');
            }

            // collect all used category identifiers
            var categories = {};
            _.each(descriptors, function (functionDesc) {
                _.extend(categories, functionDesc.category);
            });

            // resolve category UI labels
            categories = _.keys(categories).map(function (category) {
                return { value: category, label: Labels.getCategoryLabel(category) };
            });

            // sort by category UI labels, TODO: define a fixed category order?
            categories = _.sortBy(categories, 'label');

            // create the drop-down list entries
            categories.unshift({ value: CATEGORY_ALL, label: ALL_FUNCTIONS_LABEL });

            if (!_.browser.iOS) {
                categories.forEach(function (categoryData) {
                    var anchorNode = $('<a href="#" data-value="' + categoryData.value + '" role="menuitem">').text(categoryData.label);
                    jqMenuList.append($('<li>').append(anchorNode));
                });

                // click handler for the category list entries
                jqMenuList.on('click', function (evt) {
                    var jqClicked = $(evt.target).closest('a'),
                        category = jqClicked.attr('data-value');

                    evt.preventDefault();

                    jqMenuList.find('.selected').removeClass('selected');
                    jqClicked.addClass('selected');

                    jqMenuAnchor.text(jqClicked.text());
                    updateFunctionList(category);
                    _.defer(function () { functionSelectNode.focus(); });
                });
            } else {
                jqCategoriesCtl = $(Forms.createSelectBoxMarkup(categories, {
                        name: 'Categories',
                        classes: 'control functions categories',
                        size: ox.mobile ? 4 : 1

                    }));

                categorySelectNode = jqCategoriesCtl.find('select');
                categorySelectNode.val(CATEGORY_ALL);

                // click handler for the category list entries
                categorySelectNode.on('change', function (evt) {
                    evt.preventDefault();
                    updateFunctionList($(this).val());
                });
            }

            // the texts surrounding the category drop-down menu (there may
            // be leading and trailing text portions in some languages)
            var pickCategoryLabels = PICK_CATEGORY_LABEL.split(/%1\$s/);

            // the complete category drop-down control
            var jqCategoryCtl = $('<div class="control">').append(
                    pickCategoryLabels[0] ? $('<label>').text(pickCategoryLabels[0]) : $(),
                    (!_.browser.iOS) ? $('<span class="dropdown">').append(jqMenuAnchor, jqMenuList) : $('<span class="dropdown">').append(jqCategoriesCtl),
                    pickCategoryLabels[1] ? $('<label>').text(pickCategoryLabels[1]) : $()
                );

            // the function list
            var jqFunctionCtl = $(Forms.createSelectBoxMarkup([], {
                    name: 'Functions',
                    label: PICK_FUNCTION_LABEL,
                    classes: 'control functions',
                    size: ox.mobile ? 4 : 13
                }));

            // shortcut to the function list <select> elements
            functionSelectNode = jqFunctionCtl.find('select');

            // update function description on changing the selection in the list control
            functionSelectNode.on('change', function () {
                updateFunctionDescription(functionSelectNode.val());
            });

            // append controls to first control column
            controlRootNode.append(jqCategoryCtl, jqFunctionCtl);

            // preselect the 'all functions' list entry
            if (!_.browser.iOS) { jqMenuList.find('a[data-value="' + CATEGORY_ALL + '"]').addClass('selected'); }
            updateFunctionList(CATEGORY_ALL);
        }

        // public methods -----------------------------------------------------

        /**
         * Shows the dialog.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved with the native (English) name of
         *  the function selected in the dialog; or that will be rejected when
         *  canceling the dialog.
         */
        this.show = _.wrap(this.show, function (show) {
            return show.call(this).then(function () {
                var funcName = functionSelectNode.val();
                return funcName ? funcName : $.Deferred().reject();
            });
        });

        // initialization -----------------------------------------------------

        // close dialog when losing edit rights
        docView.closeDialogOnReadOnlyMode(this);

        // initialize the body element of the dialog
        this.getBody()
            .addClass('io-ox-office-spreadsheet-function-dialog')
            .toggleClass('mobile', !!ox.mobile)
            .append(controlRootNode, descRootNode);

        // create the contents of the dialog
        initControls();

        // try to preselect a function from the passed text
        if (text) { text = app.getNativeFunctionName(text); }
        if (text && app.isNativeFunctionName(text)) {
            functionSelectNode.find('option[value="' + text + '"]').prop('selected', true);
            functionSelectNode.val(text);
            updateFunctionDescription(text);
        }

        // set initial control focus
        this.on('show', function () {

            // us 102078426: dialogs on small devices should come up w/o keyboard,
            // so the focus shoud not be in an input element
            if (Utils.SMALL_DEVICE) {
                this.getOkButton().focus();
            } else {
                functionSelectNode.focus();
            }

        });

        // clean up after closing
        this.on('close', function () {
            app = docView = descriptors = null;
            controlRootNode = descRootNode = functionSelectNode = categorySelectNode = null;
        });

    }}); // class FunctionDialog

    // exports ================================================================

    return FunctionDialog;

});
