/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/controls', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/view/editcontrols',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/view/labels',
    'io.ox/office/spreadsheet/view/control/alignmentpicker',
    'io.ox/office/spreadsheet/view/control/formatcategorygroup',
    'io.ox/office/spreadsheet/view/control/formatcategorypicker',
    'io.ox/office/spreadsheet/view/control/formatcodepicker',
    'io.ox/office/spreadsheet/view/control/cellbordermodepicker',
    'io.ox/office/spreadsheet/view/control/cellbordercolorpicker',
    'io.ox/office/spreadsheet/view/control/cellstylepicker',
    'io.ox/office/spreadsheet/view/control/mergecellspicker',
    'io.ox/office/spreadsheet/view/control/namedrangesmenubutton',
    'io.ox/office/spreadsheet/view/control/sortmenubutton',
    'io.ox/office/spreadsheet/view/control/activesheetgroup',
    'io.ox/office/spreadsheet/view/control/activesheetlist',
    'io.ox/office/spreadsheet/view/control/subtotallist',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, EditControls, SheetUtils, Labels, AlignmentPicker, FormatCategoryGroup, FormatCategoryPicker, FormatCodePicker, CellBorderModePicker, CellBorderColorPicker, CellStylePicker, MergeCellsPicker, NamedRangesMenuButton, SortMenuButton, ActiveSheetGroup, ActiveSheetList, SubtotalList, gt) {

    'use strict';

    // static class SpreadsheetControls =======================================

    /**
     * Provides different classes for GUI form controls. Collects all standard
     * control group classes defined in the toolkit and the base framework, and
     * adds more controls in a single map for convenience.
     *
     * @extends EditControls
     */
    var SpreadsheetControls = _.extend({}, EditControls, {
        AlignmentPicker: AlignmentPicker,
        FormatCategoryGroup: FormatCategoryGroup,
        FormatCategoryPicker: FormatCategoryPicker,
        FormatCodePicker: FormatCodePicker,
        CellBorderModePicker: CellBorderModePicker,
        CellBorderColorPicker: CellBorderColorPicker,
        CellStylePicker: CellStylePicker,
        MergeCellsPicker: MergeCellsPicker,
        NamedRangesMenuButton: NamedRangesMenuButton,
        SortMenuButton: SortMenuButton,
        ActiveSheetGroup: ActiveSheetGroup,
        ActiveSheetList: ActiveSheetList,
        SubtotalList: SubtotalList
    });

    // class ColRowSizeLabel ==================================================

    /**
     * A label control showing an icon in front of a unit field which is used
     * to manipulate the width of a column, or the height of a row.
     *
     * @constructor
     *
     * @extends Label
     */
    SpreadsheetControls.ColRowSizeLabel = EditControls.Label.extend({ constructor: function (columns) {

        EditControls.Label.call(this, {
            icon: columns ? 'docs-table-resize-column' : 'docs-table-resize-row',
            tooltip: columns ? gt('Column width') : gt('Row height'),
            smallerVersion: { hide: true }
        });

    }}); // class ColRowSizeLabel

    // class ColRowSizeField ==================================================

    /**
     * A unit field control used to manipulate the width of a column, or the
     * height of a row.
     *
     * @constructor
     *
     * @extends UnitField
     */
    SpreadsheetControls.ColRowSizeField = EditControls.UnitField.extend({ constructor: function (columns) {

        EditControls.UnitField.call(this, {
            width: 70,
            tooltip: columns ? gt('Column width') : gt('Row height'),
            min: 0,
            max: columns ? SheetUtils.MAX_COLUMN_WIDTH : SheetUtils.MAX_ROW_HEIGHT,
            smallerVersion: { hide: true }
        });

    }}); // class ColRowSizeField

    // class CellBorderStylePicker ============================================

    /**
     * A picker control for cell border styles. The available list entries will
     * depend on the current file format.
     *
     * @constructor
     *
     * @extends BorderStylePicker
     */
    SpreadsheetControls.CellBorderStylePicker = EditControls.BorderStylePicker.extend({ constructor: function (docView, initOptions) {

        EditControls.BorderStylePicker.call(this, docView.getApp().isOOXML() ? Labels.BORDER_OOXML_PRESET_STYLES : Labels.BORDER_ODF_LINE_STYLES, initOptions);

    }}); // class CellBorderStylePicker

    // class DynamicSplitCheckBox =============================================

    /**
     * A check box control for a dynamic (movable) column/row split in a sheet.
     *
     * @constructor
     *
     * @extends CheckBox
     */
    SpreadsheetControls.DynamicSplitCheckBox = EditControls.CheckBox.extend({ constructor: function (initOptions) {

        EditControls.CheckBox.call(this, Utils.extendOptions({
            label: /*#. check box: split a spreadsheet into 2 or 4 different parts that can be scrolled independently */ gt.pgettext('sheet-split', 'Split sheet'),
            tooltip: gt.pgettext('sheet-split', 'Split the sheet above and left of the current cursor position')
        }, initOptions));

    }}); // class DynamicSplitCheckBox

    // class FrozenSplitCheckBox ==============================================

    /**
     * A check box control with a combined drop-down menu with different
     * settings for frozen columns and rows in a sheet.
     *
     * @constructor
     *
     * @extends CompoundCheckBox
     */
    SpreadsheetControls.FrozenSplitCheckBox = EditControls.CompoundCheckBox.extend({ constructor: function (docView, initOptions) {

        EditControls.CompoundCheckBox.call(this, docView, Utils.extendOptions({
            label: /*#. check box: split a spreadsheet into 2 or 4 different parts, the leading (left/upper) parts are frozen and cannot be scrolled */ gt.pgettext('sheet-split', 'Freeze sheet'),
            tooltip: gt.pgettext('sheet-split', 'Freeze the rows above and the columns left of the cursor')
        }, initOptions));

        this.addGroup('view/split/frozen/fixed', new EditControls.Button({ label: gt('Freeze first row'), tooltip: gt('Freeze the first visible row'), value: { cols: 0, rows: 1 } }))
            .addGroup('view/split/frozen/fixed', new EditControls.Button({ label: gt('Freeze first column'), tooltip: gt('Freeze the first visible column'), value: { cols: 1, rows: 0 } }))
            .addGroup('view/split/frozen/fixed', new EditControls.Button({ label: gt('Freeze first row and column'), tooltip: gt('Freeze the first visible row and the first visible column'), value: { cols: 1, rows: 1 } }));

    }}); // class FrozenSplitCheckBox

    // exports ================================================================

    return SpreadsheetControls;

});
