/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/control/subtotallist', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/control/radiolist',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, RadioList, SheetUtils, gt) {

    'use strict';

    // class SubtotalList =====================================================

    /**
     * A drop-down list control for displaying different subtotal values.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view containing this instance.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Will be passed to the base class RadioList.
     */
    function SubtotalList(docView, initOptions) {

        var // self reference
            self = this,

            // the number formatter of the document
            numberFormatter = docView.getDocModel().getNumberFormatter();

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            tooltip: gt('Type of the subtotal value')
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * Returns caption options of a list entry for the specified subtotal
         * type and value.
         */
        function getCaptionOptions(type, value) {

            var // contents and formatting of the active cell (instance of CellDescriptor)
                cellDesc = docView.getActiveCellEntry(),
                // whether the active cell is formatted as date or time
                dateTime = /^(date|time|datetime)$/.test(cellDesc.format.cat),
                // the resulting label text
                label = '';

            // format finite numbers with the number format of the active cell
            if (_.isNumber(value) && _.isFinite(value)) {

                // different formatting and special handling according to subtotal type
                if ((type === 'cells') || (type === 'numbers')) {
                    // subtotals of type 'cells' and 'numbers' are integers
                    label = String(value);
                } else if ((type === 'sum') && dateTime) {
                    // type 'sum' with date/time: show localized #N/A error
                    label = numberFormatter.formatValue(SheetUtils.ErrorCodes.NA);
                } else {
                    // format number according to the active cell's number format
                    var formatCode = numberFormatter.resolveFormatCode(cellDesc.attributes.cell.numberFormat);
                    label = numberFormatter.formatValue(value, formatCode) || String(value);
                }
            }

            switch (type) {
            case 'sum':
                return {
                    label: /*#. Subtotal result in OX Spreadsheet's status bar: Sum of all selected numeric cells */ gt('Sum: %1$s', _.noI18n(label)),
                    tooltip: gt('Sum of selected cells')
                };
            case 'min':
                return {
                    label: /*#. Subtotal result in OX Spreadsheet's status bar: Minimum of all selected numeric cells */ gt('Min: %1$s', _.noI18n(label)),
                    tooltip: gt('Minimum value in selection')
                };
            case 'max':
                return {
                    label: /*#. Subtotal result in OX Spreadsheet's status bar: Maximum of all selected numeric cells */ gt('Max: %1$s', _.noI18n(label)),
                    tooltip: gt('Maximum value in selection')
                };
            case 'cells':
                return {
                    label: /*#. Subtotal result in OX Spreadsheet's status bar: Count of all selected numeric cells AND text cells */ gt('Count: %1$s', _.noI18n(label)),
                    tooltip: gt('Number of selected cells that contain data')
                };
            case 'numbers':
                return {
                    label: /*#. Subtotal result in OX Spreadsheet's status bar: Count of all selected numeric cells only (NOT text cells) */ gt('Numerical count: %1$s', _.noI18n(label)),
                    tooltip: gt('Number of selected cells that contain numerical data')
                };
            case 'average':
                return {
                    label: /*#. Subtotal result in OX Spreadsheet's status bar: Average of all selected numeric cells */ gt('Average: %1$s', _.noI18n(label)),
                    tooltip: gt('Average of selected cells')
                };
            }
        }

        /**
         * Updates the visible label of this control.
         */
        function updateHandler(type) {

            // get fresh subtotals
            var subtotals = docView.getSubtotals();

            // if selection only consists of string values, show cell count
            if (('numbers' in subtotals) && (subtotals.numbers === 0)) { type = 'cells'; }

            // update the caption of the menu button
            if (type in subtotals) {
                self.setLabel(getCaptionOptions(type, subtotals[type]).label);
            }
        }

        /**
         * Recreates all sheet entries in the drop-down menu.
         */
        function fillList() {

            // get fresh subtotals
            var subtotals = docView.getSubtotals();

            // creates an option button for the current subtotals
            function createOptionButton(section, type) {
                self.createOptionButton(type, _.extend({ section: section }, getCaptionOptions(type, subtotals[type])));
            }

            // always create the cell count entry
            self.clearMenu().createMenuSection('cells');
            createOptionButton('cells', 'cells');

            // create numerical entries on demand
            if (subtotals.numbers > 0) {
                self.createMenuSection('numbers');
                createOptionButton('numbers', 'numbers');
                createOptionButton('numbers', 'sum');
                createOptionButton('numbers', 'min');
                createOptionButton('numbers', 'max');
                createOptionButton('numbers', 'average');
            }
        }

        /**
         * A debounced version of the method fillList().
         */
        var fillListDebounced = this.createDebouncedMethod(_.noop, fillList, { delay: 200 });

        // initialization -----------------------------------------------------

        this.registerUpdateHandler(updateHandler);

        // initialize the menu after successful import
        this.listenTo(docView.getApp().getImportPromise(), 'done', function () {

            // be lazy and do nothing as long as the menu is closed
            self.getMenu().on({
                'popup:beforeshow': function () {
                    fillList();
                    self.listenTo(docView, 'change:layoutdata', fillListDebounced);
                },
                'popup:hide': function () {
                    self.stopListeningTo(docView, 'change:layoutdata', fillListDebounced);
                }
            });
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = initOptions = numberFormatter = null;
        });

    } // class SubtotalList

    // exports ================================================================

    // derive this class from class RadioList
    return RadioList.extend({ constructor: SubtotalList });

});
