/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/validationcollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/editframework/model/attributedmodel',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/tokenarray'
], function (Utils, ModelObject, AttributedModel, SheetUtils, TokenArray) {

    'use strict';

    var // convenience shortcuts
        Address = SheetUtils.Address;

    // class ValidationModel ==================================================

    /**
     * Stores data validation settings for one or more cell ranges in a sheet.
     *
     * @constructor
     *
     * @extends AttributedModel
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this instance.
     *
     * @param {RangeArray} targetRanges
     *  The addresses of all cell ranges that contain the data validation
     *  settings.
     *
     * @param {Object} [attrs]
     *  The data validation attribute set.
     */
    var ValidationModel = AttributedModel.extend({ constructor: function (sheetModel, targetRanges, attrs) {

        var // self reference
            self = this,

            // the spreadsheet document model
            docModel = sheetModel.getDocModel(),

            // token array representing the validation attribute 'value1'
            tokenArray1 = new TokenArray(sheetModel),

            // token array representing the validation attribute 'value2'
            tokenArray2 = new TokenArray(sheetModel);

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, docModel, attrs, { families: 'validation' });

        // private methods ----------------------------------------------------

        /**
         * Updates internal settings after the validation attributes of this
         * validation model have been changed.
         */
        function changeAttributesHandler(event, newAttributes, oldAttributes) {

            // extract the validation attributes from the passed attribute sets
            newAttributes = (newAttributes || self.getMergedAttributes()).validation;
            oldAttributes = oldAttributes ? oldAttributes.validation : {};

            // parse a changed formula expression in attribute 'value1'
            if (newAttributes.value1 !== oldAttributes.value1) {
                tokenArray1.invokeSilently('parseFormula', 'op', newAttributes.value1);
            }

            // parse a changed formula expression in attribute 'value2'
            if (newAttributes.value2 !== oldAttributes.value2) {
                tokenArray2.invokeSilently('parseFormula', 'op', newAttributes.value2);
            }
        }

        // protected methods --------------------------------------------------

        /**
         * Creates and returns a cloned instance of this validation model for
         * the specified sheet.
         *
         * @internal
         *  Used by the class ValidationCollection during clone construction.
         *  DO NOT CALL from external code!
         *
         * @param {SheetModel} targetModel
         *  The model instance of the new cloned sheet that will own the clone
         *  returned by this method.
         *
         * @returns {ValidationModel}
         *  A clone of this validation model, initialized for ownership by the
         *  passed sheet model.
         */
        this.clone = function (targetModel) {
            // construct a new model instance and relocate the sheet references
            var valModel = new ValidationModel(targetModel, targetRanges, this.getExplicitAttributes(true));
            valModel.relocateSheet(sheetModel.getIndex(), targetModel.getIndex());
            return valModel;
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the addresses of all target cell ranges of this validation
         * model.
         *
         * @returns {RangeArray}
         *  The addresses of all cell ranges.
         */
        this.getRanges = function () {
            return targetRanges.clone(true);
        };

        /**
         * Returns whether the passed cell address is contained by the target
         * cell ranges of this validation model.
         *
         * @param {Address} address
         *  The address of the cell to be checked.
         *
         * @returns {Boolean}
         *  Whether the specified cell is contained in this validation model.
         */
        this.containsCell = function (address) {
            return targetRanges.containsAddress(address);
        };

        /**
         * Returns the reference addresses of this validation model (the
         * top-left cell of the bounding range of all target cell ranges of
         * this validation model).
         *
         * @returns {Address}
         *  The address of the reference cell of this validation model.
         */
        this.getRefAddress = function () {
            return targetRanges.empty() ? Address.A1.clone() : targetRanges.boundary().start;
        };

        /**
         * Returns the list contents for validations of type 'list' or 'source'
         * according to the formula expression contained in the attribute
         * 'value1'.
         *
         * @param {Address} targetAddress
         *  The target address used to resolve relative references in the
         *  formula expression.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved with an array of strings and/or
         *  null values, if this model is a data validation of type 'list' or
         *  'source', and the formula expression contained in the attribute
         *  'value1' could be resolved successfully to a list. The array
         *  elements set to the value null represent cells that cannot be
         *  formatted to a valid display string with their number format.
         *  Otherwise, if no list could be generated, the promise will be
         *  rejected. Additionally, the promise will contain the method abort()
         *  that is able to abort a running server request. In this case, the
         *  promise will be rejected immediately.
         */
        this.queryListValues = function (targetAddress) {

            var // the original query for 'source' validation
                sourceQuery = null,
                // the resulting promise
                resultPromise = null;

            // resolve formula to constant strings (validation type 'list')
            function resolveList() {
                var listValues = tokenArray1.resolveStringList();
                return _.isArray(listValues) ? $.when(listValues) : $.Deferred().reject();
            }

            // resolve formula to source ranges (validation type 'source')
            function resolveSource() {

                var // the address of the reference cell, used to resolve range addresses
                    refAddress = self.getRefAddress(),
                    // get the cell range addresses contained in the first token array
                    sourceRanges = tokenArray1.resolveRangeList({ refAddress: refAddress, targetAddress: targetAddress, resolveNames: true });

                if (sourceRanges.empty()) { return $.Deferred().reject(); }
                sourceQuery = docModel.queryCellContents([sourceRanges], { hidden: true, maxCount: 1000 });
                return sourceQuery.then(function (resultContents) {
                    return _.pluck(resultContents[0], 'display');
                });
            }

            // resolve source data by validation type
            switch (this.getMergedAttributes().validation.type) {
            case 'list':
                resultPromise = resolveList();
                break;
            case 'source':
                resultPromise = resolveSource();
                break;
            default:
                // wrong validation type
                resultPromise = $.Deferred().reject();
            }

            // add an 'abort()' method to the promise that aborts the server query
            return _.extend(resultPromise, {
                abort: function () {
                    if (sourceQuery) { sourceQuery.abort(); }
                    return this;
                }
            });
        };

        /**
         * Changes the target cell ranges of this validation model.
         *
         * @param {RangeArray} newRanges
         *  The addresses of the new target cell ranges. The old target ranges
         *  of this validation model will be replaced completely.
         *
         * @returns {ValidationModel}
         *  A reference to this instance.
         */
        this.setRanges = function (newRanges) {
            targetRanges = newRanges.merge();
            return this;
        };

        /**
         * Reduces the target cell ranges containing the data validation
         * settings to the difference of these ranges and the passed ranges.
         *
         * @param {RangeArray} otherRanges
         *  The addresses of the cell ranges to be removed from the target cell
         *  ranges of this model.
         *
         * @returns {Boolean}
         *  Whether the reduced target cell ranges are still valid (not all
         *  deleted). If false is returned, this validation model must be
         *  deleted from the validation collection.
         */
        this.reduceRanges = function (otherRanges) {
            targetRanges = targetRanges.difference(otherRanges);
            return !targetRanges.empty();
        };

        /**
         * Transforms the target cell ranges of this validation model, after
         * columns or rows have been inserted into or removed from the sheet.
         *
         * @returns {Boolean}
         *  Whether the transformed target cell ranges are still valid (not all
         *  deleted). If false is returned, this validation model must be
         *  deleted from the validation collection.
         */
        this.transformRanges = function (interval, insert, columns) {
            targetRanges = docModel.transformRanges(targetRanges, interval, insert, columns, 'expandEnd');
            return !targetRanges.empty();
        };

        /**
         * Recalculates the formula expressions stored in the validation
         * attributes 'value1' and 'value2', after columns or rows have been
         * inserted into or deleted from any sheet in the document.
         *
         * @returns {ValidationModel}
         *  A reference to this instance.
         */
        this.transformValues = function (sheet, interval, insert, columns) {
            tokenArray1.transformRanges(sheet, interval, insert, columns);
            tokenArray2.transformRanges(sheet, interval, insert, columns);
            return this;
        };

        /**
         * Changes all formula expressions stored in the validation attributes
         * 'value1' and 'value2' with sheet references containing the specified
         * old sheet index to the new sheet index.
         *
         * @returns {ValidationModel}
         *  A reference to this instance.
         */
        this.relocateSheet = function (oldSheet, newSheet) {
            tokenArray1.relocateSheet(oldSheet, newSheet);
            tokenArray2.relocateSheet(oldSheet, newSheet);
            return this;
        };

        // initialization -----------------------------------------------------

        // merge the target range addresses
        targetRanges = targetRanges.merge();

        // parse the source formulas after changing value attributes
        this.on('change:attributes', changeAttributesHandler);
        changeAttributesHandler();

        // update source values after one of the token array has changed (but prevent updating the token array again)
        tokenArray1.on('triggered', function () {
            self.off('change:attributes', changeAttributesHandler);
            self.setAttributes({ validation: { value1: tokenArray1.getFormula('op') } });
            self.on('change:attributes', changeAttributesHandler);
        });
        tokenArray2.on('triggered', function () {
            self.off('change:attributes', changeAttributesHandler);
            self.setAttributes({ validation: { value2: tokenArray2.getFormula('op') } });
            self.on('change:attributes', changeAttributesHandler);
        });

        // destroy all class members
        this.registerDestructor(function () {
            tokenArray1.destroy();
            tokenArray2.destroy();
            self = docModel = sheetModel = tokenArray1 = tokenArray2 = null;
        });

    }}); // class ValidationModel

    // class ValidationCollection =============================================

    /**
     * Stores data validation settings in a specific sheet. Data validation can
     * be used to restrict the values allowed to be entered in specific cell
     * ranges, and to show tooltips and error messages.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */
    function ValidationCollection(sheetModel) {

        var // the spreadsheet document model
            docModel = sheetModel.getDocModel(),

            // all model instances with validation settings
            valModels = [];

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // private methods ----------------------------------------------------

        /**
         * Returns the validation model that covers the specified cell.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {ValidationModel|Null}
         *  The validation for the specified cell. If the cell is not validated
         *  at all, returns null.
         */
        function findModelByAddress(address) {
            var valModel = _.find(valModels, function (model) {
                return model.containsCell(address);
            });
            return valModel ? valModel : null;
        }

        /**
         * Returns the validation model addressed by the property 'index' in
         * the passed document operation.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing a document operation for validations,
         *  expected to contain an integer property 'index'.
         *
         * @returns {Object}
         *  A descriptor for the validation model and its index, in the
         *  following properties:
         *  - {Number} index
         *      The array index of the validation model (the value of the
         *      'index' property in the passed operation).
         *  - {ValidationModel} model
         *      The validation model referenced by the document operation.
         *
         * @throws {OperationError}
         *  If the operation does not contain a valid 'index' property, or if
         *  the validation model with such an index does not exist.
         */
        function getModelData(context) {

            // get the index from the operation
            var index = context.getInt('index');

            // check existence of the validation model
            var valModel = valModels[index];
            context.ensure(valModel, 'missing validation');

            return { index: index, model: valModel };
        }

        /**
         * Invokes the specified method on all validation models.
         *
         * @param {String} methodName
         *  The name of the method to be invoked on each validation model. If
         *  the method returns the value false, the respective model will be
         *  removed from the collection.
         *
         * @param {ValidationModel|Null} excludeModel
         *  A validation model that will be skipped (its method will not be
         *  invoked).
         *
         * @param {Any} ...
         *  Other parameters that will be passed to the specified method.
         */
        function invokeModelMethod(methodName, excludeModel) {
            var args = _.toArray(arguments).slice(invokeModelMethod.length);
            Utils.iterateArray(valModels, function (valModel, index) {
                if ((valModel !== excludeModel) && (valModel[methodName].apply(valModel, args) === false)) {
                    valModel.destroy();
                    valModels.splice(index, 1);
                }
            }, { reverse: true }); // reverse to be able to delete in-place
        }

        /**
         * Recalculates the position of all validation target ranges, after
         * columns or rows have been inserted into or deleted from the sheet.
         */
        function transformRanges(interval, insert, columns) {
            invokeModelMethod('transformRanges', null, interval, insert, columns);
        }

        /**
         * Recalculates the formula expressions stored in the validation
         * attributes 'value1' and 'value2', after columns or rows have been
         * inserted into or deleted from any sheet in the document.
         */
        function transformValues(sheet, interval, insert, columns) {
            _.invoke(valModels, 'transformValues', sheet, interval, insert, columns);
        }

        // protected methods --------------------------------------------------

        /**
         * Returns the internal contents of this collection, needed for cloning
         * into another collection.
         *
         * @internal
         *  Used during clone construction. DO NOT CALL from external code!
         *
         * @returns {Object}
         *  The internal contents of this collection.
         */
        this.getCloneData = function () {
            return { valModels: valModels };
        };

        /**
         * Clones all contents from the passed collection into this collection.
         *
         * @internal
         *  Used by the class SheetModel during clone construction. DO NOT CALL
         *  from external code!
         *
         * @param {ValidationCollection} collection
         *  The source collection whose contents will be cloned into this
         *  collection.
         *
         * @returns {ValidationCollection}
         *  A reference to this instance.
         */
        this.cloneFrom = function (collection) {

            var // the internal contents of the source collection
                cloneData = collection.getCloneData();

            // clone the contents of the source collection
            valModels = _.invoke(cloneData.valModels, 'clone', sheetModel);

            return this;
        };

        /**
         * Callback handler for the document operation 'insertValidation'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'insertValidation' document operation.
         */
        this.applyInsertOperation = function (context) {

            // get and validate the insertion index
            var index = context.getInt('index');
            context.ensure((index >= 0) && (index <= valModels.length), 'invalid index');

            // the new validation model
            var valModel = new ValidationModel(sheetModel, context.getRangeArray(), { validation: context.operation });

            // reduce ranges of existing models (this may result in removing models completely)
            invokeModelMethod('reduceRanges', null, valModel.getRanges());

            // insert the new validation model
            valModels.splice(index, 0, valModel);

            // TODO: notify listeners
        };

        /**
         * Callback handler for the document operation 'deleteValidation'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'deleteValidation' document operation.
         */
        this.applyDeleteOperation = function (context) {

            // resolve the validation model addressed by the operation (throws on error)
            var modelData = getModelData(context);

            // destroy and remove the validation model
            modelData.model.destroy();
            valModels.splice(modelData.index, 1);

            // TODO: notify listeners
        };

        /**
         * Callback handler for the document operation 'changeValidation'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'changeValidation' document operation.
         */
        this.applyChangeOperation = function (context) {

            // resolve the validation model addressed by the operation (throws on error)
            var modelData = getModelData(context);

            // resolve optional 'ranges' property (throws on error), change target ranges
            var targetRanges = context.getOptRangeArray();
            if (targetRanges) {
                // reduce ranges of other existing models (this may result in removing models completely)
                invokeModelMethod('reduceRanges', modelData.model, targetRanges);
                // set the new target ranges at the validation model
                modelData.model.setRanges(targetRanges);
            }

            // change all other validation attributes
            modelData.model.setAttributes({ validation: context.operation });

            // TODO: notify listeners
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the validation settings for the specified cell.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {Object|Null}
         *  The validation settings of the specified cell, in the following
         *  properties:
         *  - {RangeArray} ranges
         *      The addresses of all related cell ranges containing the same
         *      validation settings.
         *  - {Object} attributes
         *      All validation attributes.
         *  If the cell is not validated at all, returns null.
         */
        this.getValidationSettings = function (address) {
            var valModel = findModelByAddress(address);
            return valModel ? {
                ranges: valModel.getRanges(),
                attributes: valModel.getMergedAttributes().validation
            } : null;
        };

        /**
         * Returns the list contents for a data validation of type 'list' or
         * 'source' at the specified cell address.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved with an
         *  array of strings, if the data validation at the address is of type
         *  'list' or 'source', and the formula expression contained in the
         *  attribute 'value1' could be resolved successfully to a list of
         *  strings. Otherwise, the promise will be rejected. Additionally, the
         *  promise will contain the method 'abort()' that is able to abort a
         *  running server request. In this case, the promise will be rejected
         *  immediately.
         */
        this.queryListValues = function (address) {
            var valModel = findModelByAddress(address);
            return valModel ? valModel.queryListValues(address) : $.Deferred().reject();
        };

        // initialization -----------------------------------------------------

        // update target ranges after inserting/deleting columns or rows in the own sheet
        sheetModel.registerTransformationHandler(transformRanges);

        // update source formulas after inserting/deleting columns or rows in any sheet
        docModel.registerTransformationHandler(transformValues);

        // destroy all class members
        this.registerDestructor(function () {
            docModel.unregisterTransformationHandler(transformValues);
            _.invoke(valModels, 'destroy');
            docModel = sheetModel = valModels = null;
        });

    } // class ValidationCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: ValidationCollection });

});
