/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/modelattributesmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/drawinglayer/model/drawingstylecollection',
    'io.ox/office/spreadsheet/model/cellstylecollection',
    'io.ox/office/spreadsheet/model/tablestylecollection'
], function (Utils, Color, Border, DrawingStyleCollection, CellStyleCollection, TableStyleCollection) {

    'use strict';

    // definitions for global document attributes -----------------------------

    var DOCUMENT_ATTRIBUTE_DEFINITIONS = {

        /**
         * Number of columns available in each sheet of the document.
         */
        cols: { def: 1024 },

        /**
         * Number of rows available in each sheet of the document.
         */
        rows: { def: 1048576 },

        // view attributes

        /**
         * Index of the active (displayed) sheet.
         */
        activeSheet: { def: 0 }
    };

    // definitions for character attributes in cells --------------------------

    var CHARACTER_ATTRIBUTE_DEFINITIONS = {

        /**
         * Name of the font family used to render the cell contents. Bound to
         * the apply flag 'font' of a cell style sheet.
         */
        fontName: { def: 'Arial', apply: 'font' },

        /**
         * Size of the font used to render the cell contents, in points. Bound
         * to the apply flag 'font' of a cell style sheet.
         */
        fontSize: { def: 11, apply: 'font' },

        /**
         * Specifies whether to render the cell contents with bold characters.
         * Bound to the apply flag 'font' of a cell style sheet.
         */
        bold: { def: false, apply: 'font' },

        /**
         * Specifies whether to render the cell contents with italic
         * characters. Bound to the apply flag 'font' of a cell style sheet.
         */
        italic: { def: false, apply: 'font' },

        /**
         * Specifies whether to underline the cell contents. Bound to the apply
         * flag 'font' of a cell style sheet.
         */
        underline: { def: false, apply: 'font' },

        /**
         * Specifies whether to strike the cell contents. Supported values are
         * 'none', 'single', and 'double'. Bound to the apply flag 'font' of a
         * cell style sheet.
         */
        strike: { def: 'none', apply: 'font' },

        /**
         * Text color used to render the cell contents, underlines, and strike
         * lines. Bound to the apply flag 'font' of a cell style sheet.
         */
        color: { def: Color.AUTO, apply: 'font' }
    };

    // definitions for cell attributes ----------------------------------------

    var CELL_ATTRIBUTE_DEFINITIONS = {

        /**
         * Horizontal alignment of the text contents in the cell. Bound to the
         * apply flag 'align' of a cell style sheet.
         */
        alignHor: { def: 'auto', apply: 'align' },

        /**
         * Vertical alignment of the text contents in the cell. Bound to the
         * apply flag 'align' of a cell style sheet.
         */
        alignVert: { def: 'bottom', apply: 'align' },

        /**
         * Specifies whether to break long text automatically into multiple
         * text lines. Has no effect on non-text cell results. Bound to the
         * apply flag 'align' of a cell style sheet.
         */
        wrapText: { def: false, apply: 'align' },

        /**
         * The number format code, and/or identifier of a predefined number
         * format, as object with the string property 'code', and index
         * property 'id'. Bound to the apply flag 'number' of a cell style
         * sheet.
         */
        numberFormat: { def: { id: 0 }, apply: 'number' },

        /**
         * The fill color for the cell area. Bound to the apply flag 'fill' of
         * a cell style sheet.
         */
        fillColor: { def: Color.AUTO, apply: 'fill' },

        /**
         * The style of the left border of the cell. Bound to the apply flag
         * 'border' of a cell style sheet.
         */
        borderLeft: { def: Border.NONE, apply: 'border' },

        /**
         * The style of the right border of the cell. Bound to the apply flag
         * 'border' of a cell style sheet.
         */
        borderRight: { def: Border.NONE, apply: 'border' },

        /**
         * The style of the top border of the cell. Bound to the apply flag
         * 'border' of a cell style sheet.
         */
        borderTop: { def: Border.NONE, apply: 'border' },

        /**
         * The style of the bottom border of the cell. Bound to the apply flag
         * 'border' of a cell style sheet.
         */
        borderBottom: { def: Border.NONE, apply: 'border' },

        /**
         * The style of the diagonal border of the cell from top-left corner to
         * bottom-right corner. Bound to the apply flag 'border' of a cell
         * style sheet.
         */
        borderDown: { def: Border.NONE, apply: 'border' },

        /**
         * The style of the diagonal border of the cell from bottom-left corner
         * to top-right corner. Bound to the apply flag 'border' of a cell
         * style sheet.
         */
        borderUp: { def: Border.NONE, apply: 'border' },

        /**
         * The style of the inner horizontal borders of an entire cell range.
         * Will be copied to the attributes 'topBorder' and/or 'bottomBorder'
         * according to the position of a cell in the range to be formatted.
         * Cannot be inserted into a cell style sheet.
         */
        borderInsideHor: { def: Border.NONE, scope: 'element' },

        /**
         * The style of the inner vertical borders of an entire cell range.
         * Will be copied to the attributes 'leftBorder' and/or 'rightBorder'
         * according to the position of a cell in the range to be formatted.
         * Cannot be inserted into a cell style sheet.
         */
        borderInsideVert: { def: Border.NONE, scope: 'element' },

        /**
         * Specifies whether the cell is unlocked. Locked cells in locked
         * sheets cannot be modified. Has no effect in unlocked sheets. Bound
         * to the apply flag 'protect' of a cell style sheet.
         */
        unlocked: { def: false, apply: 'protect' },

        /**
         * Specifies whether the formula expression of a formula cell will be
         * hidden in the user interface of a locked sheet. Has no effect in
         * unlocked sheets. Bound to the apply flag 'protect' of a cell style
         * sheet.
         */
        hidden: { def: false, apply: 'protect' }
    };

    // definitions for apply group flags --------------------------------------

    var APPLY_ATTRIBUTE_DEFINITIONS = {

        /**
         * If part of a cell style sheet, specifies whether the font attributes
         * of the style will overwrite explicit attributes of the target cell.
         */
        font: { def: true },

        /**
         * If part of a cell style sheet, specifies whether the fill color of
         * the style will overwrite the explicit fill color of the target cell.
         */
        fill: { def: true },

        /**
         * If part of a cell style sheet, specifies whether the border
         * attributes of the style will overwrite the border attributes of the
         * target cell.
         */
        border: { def: true },

        /**
         * If part of a cell style sheet, specifies whether the alignment
         * attributes of the style will overwrite the alignment attributes of
         * the target cell.
         */
        align: { def: true },

        /**
         * If part of a cell style sheet, specifies whether the number format
         * of the style will overwrite the number format of the target cell.
         */
        number: { def: true },

        /**
         * If part of a cell style sheet, specifies whether the protection
         * flags of the style will overwrite the protection flags of the target
         * cell.
         */
        protect: { def: true }
    };

    // definitions for column attributes --------------------------------------

    var COLUMN_ATTRIBUTE_DEFINITIONS = {

        /**
         * Specifies whether the column is visible in the GUI.
         */
        visible: { def: true },

        /**
         * The width of the column, in 1/100 mm.
         */
        width: { def: 2000 },

        /**
         * Specifies whether the width of the column has been set or changed in
         * the GUI and is fixed.
         */
        customWidth: { def: false }
    };

    // definitions for row attributes -----------------------------------------

    var ROW_ATTRIBUTE_DEFINITIONS = {

        /**
         * Specifies whether the row is visible in the GUI.
         */
        visible: { def: true },

        /**
         * The height of the row, in 1/100 mm.
         */
        height: { def: 500 },

        /**
         * Specifies whether the height of the row has been set or changed in
         * the GUI and is fixed. If set to false, the row height will be
         * adjusted automatically due to cell contents and font settings.
         */
        customHeight: { def: false },

        /**
         * Specifies whether the row contains default formatting attributes for
         * all undefined cells. If set to false, undefined cells will be
         * formatted according to the default column formatting.
         */
        customFormat: { def: false },

        /**
         * Specifies whether the row has been hidden implicitly by a filter
         * (auto-filter, or filtered table range). The row will be hidden
         * although the attribute 'visible' is true. Used by ODF only!
         */
        filtered: { def: false }
    };

    // definitions for sheet attributes ---------------------------------------

    var SHEET_ATTRIBUTE_DEFINITIONS = {

        /**
         * Specifies whether the sheet is visible in the GUI.
         */
        visible: { def: true },

        /**
         * Specifies whether the sheet is locked. Contents in locked sheets
         * cannot be changed (except unlocked cells).
         */
        locked: { def: false },

        /**
         * Base column width for undefined columns, in number of digits of the
         * default document font. Used by OOXML only.
         */
        baseColWidth: { def: 8 },

        // view attributes

        /**
         * The current zoom factor. The value 1 represents the standard zoom of
         * 100%.
         */
        zoom: { def: 1 },

        /**
         * The selected cell ranges, as array of JSON cell range addresses.
         */
        selectedRanges: { def: [{ start: [0, 0], end: [0, 0] }] },

        /**
         * The array index of the active cell range in the 'selectedRanges'
         * attribute value (the index of the cell range containing the active
         * cell).
         */
        activeIndex: { def: 0 },

        /**
         * The address of the active cell in the selected cell ranges, as JSON
         * cell position (array with column and row index). Must be located
         * inside the active cell range (as specified by the attributes
         * 'selectedRanges' and 'activeIndex').
         */
        activeCell: { def: [0, 0] },

        /**
         * The positions of all selected drawing objects. Each element of this
         * array is a sheet-local drawing position (an array of integers
         * without leading sheet index).
         */
        selectedDrawings: { def: [] },

        /**
         * Whether the grid lines between the cells will be rendered.
         */
        showGrid: { def: true },

        /**
         * The color of the grid lines between the cells.
         */
        gridColor: { def: Color.AUTO },

        /**
         * The split mode used when split view is active. Must be one of the
         * following values:
         * - 'split': Dynamic split (grid panes are resizable).
         * - 'frozen': Frozen split (grid panes are not resizable).
         * - 'frozenSplit': Like 'frozen', but thawing will change split mode
         *      to 'split' instead of removing the split.
         */
        splitMode: { def: 'split' },

        /**
         * Width of the left grid panes, as 1/100 mm for dynamic split mode, or
         * number of columns in frozen split mode. The value zero hides the
         * right grid panes.
         */
        splitWidth: { def: 0 },

        /**
         * Height of the upper grid panes, as 1/100 mm for dynamic split mode,
         * or number of rows in frozen split mode. The value zero hides the
         * lower grid panes.
         */
        splitHeight: { def: 0 },

        /**
         * Identifier of the active (focused) grid pane. Must be one of
         * 'topLeft', 'topRight', 'bottomLeft', or 'bottomRight'. Must be a
         * visible grid pane, according to the current split settings.
         */
        activePane: { def: 'topLeft' },

        /**
         * Horizontal scroll position in the left grid panes, as zero-based
         * column index.
         */
        scrollLeft: { def: 0 },

        /**
         * Horizontal scroll position in the right grid panes, as zero-based
         * column index.
         */
        scrollRight: { def: 0 },

        /**
         * Vertical scroll position in the upper grid panes, as zero-based row
         * index.
         */
        scrollTop: { def: 0 },

        /**
         * Vertical scroll position in the lower grid panes, as zero-based row
         * index.
         */
        scrollBottom: { def: 0 }
    };

    // definitions for table attributes ---------------------------------------

    var TABLE_ATTRIBUTE_DEFINITIONS = {

        /**
         * Whether the first row of the table is a header row. Header rows will
         * not be filtered or sorted.
         */
        headerRow: { def: false, scope: 'element' },

        /**
         * Whether the last row of the table is a footer row. Footer rows will
         * not be filtered or sorted.
         */
        footerRow: { def: false, scope: 'element' },

        /**
         * Whether the table can be filtered (this flag does not mean that the
         * table has actual filter rules applied, but the GUI may provide
         * visual feedback and control elements for filtering).
         */
        filtered: { def: false, scope: 'element' }
    };

    // definitions for filter attributes in table columns ---------------------

    var TABLE_FILTER_ATTRIBUTE_DEFINITIONS = {

        /**
         * The type of the filter rule in this column. The following values are
         * supported:
         * - 'none': No filter rule is active.
         * - 'discrete': Discrete filter. The attribute 'entries' contains
         *      all cell entries to be shown for the table column.
         */
        type: { def: 'none' },

        /**
         * All entries (cell display strings) that will be shown for this table
         * column. Only used, if the filter type is 'discrete'.
         */
        entries: { def: [], parent: { name: 'type', value: 'discrete' } },

        /**
         * Whether the drop-down button of the filter column is currently
         * hidden by a merged cell range.
         */
        hideMergedButton: { def: false }
    };

    // definitions for sorting attributes in table columns --------------------

    var TABLE_SORT_ATTRIBUTE_DEFINITIONS = {

        /**
         * Whether to sort the column entries in descending order.
         */
        descending: { def: false }
    };

    // definitions for data validation attributes -----------------------------

    var VALIDATION_ATTRIBUTE_DEFINITIONS = {

        /**
         * The type of the data validation. Supports all types used in the
         * document operation 'insertValidation'.
         */
        type: { def: 'all' },

        /**
         * The comparison operator for most validation types, used together
         * with the attributes 'value1' and 'value2'.
         */
        compare: { def: '', parent: { name: 'type', value: function (value) { return (/^(integer|number|date|time|length)$/).test(value); } } },

        /**
         * The first value used together with the comparison operator. Must be
         * a formula expression as string, without the leading equality sign.
         */
        value1: { def: '', parent: { name: 'type', value: function (value) { return value !== 'all'; } } },

        /**
         * The second value used together with the comparison operators
         * 'between' and 'notBetween'. Must be a formula expression as string,
         * without the leading equality sign.
         */
        value2: { def: '', parent: { name: 'type', value: function (value) { return value !== 'all'; } } },

        /**
         * Specifies whether to show an informational tooltip when a cell with
         * this data validation is selected.
         */
        showInfo: { def: true },

        /**
         * The headline for the informational tooltip.
         */
        infoTitle: { def: '' },

        /**
         * The message text for the informational tooltip.
         */
        infoText: { def: '' },

        /**
         * Specifies whether to show an alert box when an invalid value has
         * been entered into a cell with this data validation.
         */
        showError: { def: true },

        /**
         * The headline for the alert box.
         */
        errorTitle: { def: '' },

        /**
         * The message text for the alert box.
         */
        errorText: { def: '' },

        /**
         * The type of the alert box. Must be one of 'info', 'warning', or
         * 'error'.
         */
        errorType: { def: 'error' },

        /**
         * Specifies whether to show a drop-down button at the selected cell,
         * if the data validation type is 'list' or 'source'.
         */
        showDropDown: { def: true },

        /**
         * Specifies whether to allow blank cells regardless of the actual
         * validation type and conditions.
         */
        ignoreEmpty: { def: true }
    };

    // definitions for formatting rule attributes -----------------------------

    var FORMAT_RULE_ATTRIBUTE_DEFINITIONS = {

        /**
         * The type of the formatting rule. Supports all types used in the
         * document operation 'insertCondFormatRule'.
         */
        type: { def: 'formula' },

        /**
         * The first value, used by most conditional rules. Depending on the
         * rule type, must either be a formula expression as string (without
         * the leading equality sign), or a constant number (for 'average'
         * rules only).
         */
        value1: { def: '' },

        /**
         * The second value, used by interval rules of type 'between' and
         * 'notBetween' only. Must be a formula expression as string, without
         * the leading equality sign.
         */
        value2: { def: '' },

        /**
         * The sheet-global priority of this rule among all formatting rules in
         * the sheet. The lower this value, the higher the priority. Rules with
         * equal priority will be applied in insertion order.
         */
        priority: { def: 0 },

        /**
         * Specifies whether to stop evaluating other formatting rules matching
         * a specific cell, if the conditions of this rule has matched.
         */
        stop: { def: false }
    };

    // additional definitions for drawing attributes --------------------------

    var DRAWING_ATTRIBUTE_DEFINITIONS = {

        /**
         * Type of the drawing anchor (specifies which anchor attributes have
         * to be used), and runtime behavior of the drawing while inserting,
         * removing, or changing columns and rows. Supported values are
         * 'twoCell', 'oneCell', 'absolute', 'twoCellAsOneCell', and
         * 'twoCellAsAbsolute'.
         */
        anchorType: { def: 'twoCell', scope: 'element' },

        /**
         * Zero-based index of the column containing the leading border of the
         * drawing object.
         */
        startCol: { def: 0, scope: 'element' },

        /**
         * The exact position of the leading border in the column, as ratio of
         * the column width (in the range [0,1]).
         */
        startColOffset: { def: 0, scope: 'element' },

        /**
         * Zero-based index of the row containing the top border of the drawing
         * object.
         */
        startRow: { def: 0, scope: 'element' },

        /**
         * The exact position of the top border in the row, as ratio of the row
         * height (in the range [0,1]).
         */
        startRowOffset: { def: 0, scope: 'element' },

        /**
         * Zero-based index of the column containing the trailing border of the
         * drawing object.
         */
        endCol: { def: 0, scope: 'element' },

        /**
         * The exact position of the trailing border in the column, as ratio of
         * the column width (in the range [0,1]).
         */
        endColOffset: { def: 0, scope: 'element' },

        /**
         * Zero-based index of the row containing the bottom border of the
         * drawing object.
         */
        endRow: { def: 0, scope: 'element' },

        /**
         * The exact position of the bottom border in the row, as ratio of the
         * row height (in the range [0,1]).
         */
        endRowOffset: { def: 0, scope: 'element' }
    };

    // class ModelAttributesMixin =============================================

    /**
     * A mix-in class for the document model class SpreadsheetModel providing
     * the style sheet containers for all attribute families used in a
     * spreadsheet document.
     *
     * @constructor
     */
    function ModelAttributesMixin() {

        var // the style sheet collection for cells, columns, and rows
            cellStyles = null,

            // the style sheet collection for table ranges
            tableStyles = null,

            // the style sheet collection for drawing objects
            drawingStyles = null;

        // public methods -----------------------------------------------------

        /**
         * Returns the cell style collection of this document.
         *
         * @returns {StyleCollection}
         *  The cell style collection of this document.
         */
        this.getCellStyles = function () {
            return cellStyles;
        };

        /**
         * Returns the default column width for all undefined columns in a
         * sheet, according to the font settings in the current default cell
         * style, and the passed base column width.
         *
         * @param {Number} baseWidth
         *  The base column width, as number of zero digits.
         *
         * @returns {Number}
         *  The default column width for all undefined columns, in 1/100 mm.
         */
        this.getDefaultColWidthHmm = function (baseWidth) {

            var charAttributes = cellStyles.getDefaultStyleAttributeSet().character,
                digitWidth = this.getFontCollection().getDigitWidth(charAttributes),
                colWidth = digitWidth * baseWidth + 5;

            return Utils.convertLengthToHmm(colWidth, 'px');
        };

        /**
         * Calculates the row height in 1/100 of millimeters to be used for the
         * passed character attributes. The row height is about 10% larger than
         * the normal line height for the passed font settings, as returned
         * e.g. by the method FontMetrics.getNormalLineHeight().
         *
         * @param {Object} charAttributes
         *  Character formatting attributes influencing the line height, as
         *  used in document operations.
         *
         * @returns {Number}
         *  The row height for the passed character attributes, in 1/100 mm.
         */
        this.getRowHeightHmm = function (charAttributes) {

            var // the normal line height according to the font settings
                lineHeight = this.getFontCollection().getNormalLineHeight(charAttributes);

            // use at least 125% of the font size, to compensate for browsers
            // reporting very small line heights (especially Chrome)
            // Bug 40718: increased 120% to 125% to cover the standard case 'Arial 11pt'.
            lineHeight = Math.max(lineHeight, Utils.convertLength(charAttributes.fontSize, 'pt', 'px') * 1.25);

            // enlarge the resulting 'normal' line height by another 10% to add top/bottom cell padding
            return Utils.convertLengthToHmm(Math.floor(lineHeight * 1.1), 'px');
        };

        /**
         * Calculates the effective scaled row height in pixels to be used for
         * the passed character attributes.
         *
         * @param {Object} charAttributes
         *  Character formatting attributes influencing the line height, as
         *  used in document operations.
         *
         * @param {Number} zoom
         *  The zoom factor used to scale the font size in the passed character
         *  attributes.
         *
         * @returns {Number}
         *  The row height for the passed character attributes, in pixels.
         */
        this.getRowHeight = function (charAttributes, zoom) {
            return Utils.convertHmmToLength(this.getRowHeightHmm(charAttributes) * zoom, 'px', 1);
        };

        /**
         * Returns the default row height for all undefined rows in a sheet,
         * according to the font settings in the current default cell style.
         *
         * @returns {Number}
         *  The default row height for all undefined rows, in 1/100 mm.
         */
        this.getDefaultRowHeightHmm = function () {
            return this.getRowHeightHmm(cellStyles.getDefaultStyleAttributeSet().character);
        };

        // initialization -----------------------------------------------------

        this.registerAttributeDefinitions('document', DOCUMENT_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('character', CHARACTER_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('cell', CELL_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('apply', APPLY_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('column', COLUMN_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('row', ROW_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('sheet', SHEET_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('table', TABLE_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('filter', TABLE_FILTER_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('sort', TABLE_SORT_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('validation', VALIDATION_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('rule', FORMAT_RULE_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('drawing', DRAWING_ATTRIBUTE_DEFINITIONS);

        this.addStyleCollection(cellStyles = new CellStyleCollection(this))
            .addStyleCollection(tableStyles = new TableStyleCollection(this))
            .addStyleCollection(drawingStyles = new DrawingStyleCollection(this));

        // destroy class members on destruction
        this.registerDestructor(function () {
            cellStyles = tableStyles = drawingStyles = null;
        });

    } // class ModelAttributesMixin

    // exports ================================================================

    return ModelAttributesMixin;

});
