/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/model', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/model/editmodel',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/operationcontext',
    'io.ox/office/spreadsheet/model/numberformatter',
    'io.ox/office/spreadsheet/model/namecollection',
    'io.ox/office/spreadsheet/model/sheetmodel',
    'io.ox/office/spreadsheet/model/modelattributesmixin',
    'io.ox/office/spreadsheet/model/viewsettingsmixin',
    'io.ox/office/spreadsheet/model/formula/tokenizer',
    'io.ox/office/spreadsheet/model/formula/tokenarray',
    'io.ox/office/spreadsheet/model/formula/compiler',
    'io.ox/office/spreadsheet/model/formula/interpreter'
], function (Utils, EditModel, Config, Operations, SheetUtils, SheetOperationContext, NumberFormatter, NameCollection, SheetModel, ModelAttributesMixin, ViewSettingsMixin, Tokenizer, TokenArray, Compiler, Interpreter) {

    'use strict';

    var // convenience shortcuts
        Address = SheetUtils.Address,
        Interval = SheetUtils.Interval,
        Range = SheetUtils.Range,
        AddressArray = SheetUtils.AddressArray,
        IntervalArray = SheetUtils.IntervalArray,
        RangeArray = SheetUtils.RangeArray,
        Range3DArray = SheetUtils.Range3DArray,

        // definitions for document view attributes
        DOCUMENT_VIEW_ATTRIBUTES = {

            /**
             * Index of the active (visible) sheet in the document.
             */
            activeSheet: {
                def: -1,
                validate: function (sheet) {
                    return ((sheet >= 0) && (sheet < this.getSheetCount())) ? sheet : Utils.BREAK;
                }
            },

            /**
             * An array of active remote clients with selection settings. See
             * method EditApplication.getActiveClients() for details about the
             * contents of this array. The local client must not be part of the
             * array.
             */
            remoteClients: {
                def: [],
                validate: function (clients) { return _.isArray(clients) ? clients : []; }
            }
        },

        // maps sheet event names to different document event names
        SHEET_TO_DOCUMENT_EVENTS = {
            'change:attributes': 'change:sheet:attributes',
            'change:viewattributes': 'change:sheet:viewattributes'
        },

        // whether to let each 'insertSheet' operation fail in debug mode
        DEBUG_FAILING_IMPORT = Config.getDebugUrlFlag('spreadsheet:failing-import');

    // class SpreadsheetModel =================================================

    /**
     * Represents a 'workbook', the spreadsheet document model. Implements
     * execution of all supported operations.
     *
     * Triggers the events supported by the base class EditModel, and the
     * following additional events:
     *  - 'insert:sheet:before':
     *      Before a new sheet will be inserted into the document. The event
     *      handler receives the zero-based index of the new sheet it will have
     *      in the collection of all sheets.
     *  - 'insert:sheet:after':
     *      After a new sheet has been inserted into the document. The event
     *      handler receives the zero-based index of the new sheet in the
     *      collection of all sheets, and the new sheet model instance.
     *  - 'delete:sheet:before':
     *      Before a sheet will be removed from the document. The event handler
     *      receives the zero-based index of the sheet in the collection of all
     *      sheets, and the model instance of the sheet to be removed.
     *  - 'delete:sheet:after':
     *      After a sheet has been removed from the document. The event handler
     *      receives the zero-based index of the sheet in the collection of all
     *      sheets before it has been removed.
     *  - 'move:sheet:before':
     *      Before a sheet will be moved to a new position. The event handler
     *      receives the zero-based index of the new position in the collection
     *      of all sheets, the zero-based index of the old position, and the
     *      model instance of the moved sheet.
     *  - 'move:sheet:after':
     *      After a sheet has been moved to a new position. The event handler
     *      receives the zero-based index of the new position in the collection
     *      of all sheets, the zero-based index of the old position, and the
     *      model instance of the moved sheet.
     *  - 'rename:sheet':
     *      After a sheet has been renamed. The event handler receives the
     *      zero-based index of the renamed sheet in the collection of all
     *      sheets, the new name of the sheet, and the sheet model instance.
     * - 'change:viewattributes'
     *      After at least one view attribute of the document has been changed.
     *      Event handlers receive an incomplete attribute map containing all
     *      changed view attributes with their new values.
     *
     * Additionally, events of all sheets will be forwarded to the own model
     * listeners. The sheet event 'change:attributes' will be triggered as
     * 'change:sheet:attributes', and the sheet event 'change:viewattributes'
     * will be triggered as 'change:sheet:viewattributes'. The second parameter
     * passed to all event listeners will be the sheet index (following the
     * jQuery event object). The actual sheet event parameters will be appended
     * to the parameters of the own model listeners. Example: The sheet event
     * 'insert:columns' with the parameters (event, interval) will be triggered
     * as document model event 'insert:columns' with the parameters (event,
     * sheet, interval).
     *
     * Events will always be triggered, also during importing the document.
     *
     * @constructor
     *
     * @extends EditModel
     * @extends ModelAttributesMixin
     * @extends ViewSettingsMixin
     *
     * @param {SpreadsheetApplication} app
     *  The application containing this document model.
     */
    function SpreadsheetModel(app) {

        var // self reference
            self = this,

            // all existing sheets and their names, by zero-based index
            sheetsCollection = [],

            // the number formatter for this document
            numberFormatter = null,

            // the formula tokenizers for this document, mapped by grammar
            formulaTokenizers = {},

            // the formula compiler for this document
            formulaCompiler = null,

            // the formula interpreter for this document
            formulaInterpreter = null,

            // the collection of globally defined names
            nameCollection = null,

            // handlers to adjust range addresses after inserting/deleting columns/rows
            transformationHandlers = [],

            // the maximum column/row index in a sheet
            maxCol = 0, maxRow = 0;

        // base constructors --------------------------------------------------

        EditModel.call(this, app, { contextClass: SheetOperationContext, remoteUndo: true });
        ModelAttributesMixin.call(this);
        ViewSettingsMixin.call(this, DOCUMENT_VIEW_ATTRIBUTES);

        // private methods ----------------------------------------------------

        /**
         * Updates the maximum available column and row index.
         */
        function updateMaxColRow(attributes) {
            maxCol = attributes.cols - 1;
            maxRow = attributes.rows - 1;
        }

        /**
         * Returns the sheet info structure at the specified index.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {Object|Null}
         *  The info structure of the sheet at the passed index or with the
         *  passed name, containing the sheet name and the reference to the
         *  sheet model instance, or null if the passed index or name is
         *  invalid.
         */
        function getSheetInfo(sheet) {
            sheet = _.isNumber(sheet) ? sheet : _.isString(sheet) ? self.getSheetIndex(sheet) : -1;
            return ((sheet >= 0) && (sheet < sheetsCollection.length)) ? sheetsCollection[sheet] : null;
        }

        /**
         * Returns an error code if the passed sheet name is not valid (empty,
         * or with invalid characters, or already existing in the document).
         *
         * @param {String} sheetName
         *  The desired name of the sheet (case-insensitive).
         *
         * @param {Number} [sheet]
         *  If specified, the zero-based index of the sheet that is allowed to
         *  contain the passed name (used while renaming an existing sheet).
         *
         * @returns {String}
         *  The empty string, if the sheet name is valid; otherwise one of the
         *  following error codes:
         *  - 'sheetname:empty': The passed sheet name is empty.
         *  - 'sheetname:invalid': The passed sheet name contains invalid
         *      characters.
         *  - 'sheetname:used': The passed sheet name is already used in the
         *      document.
         */
        function validateSheetName(sheetName, sheet) {

            var // an existing sheet with the passed name
                sheetModelByName = self.getSheetModel(sheetName),
                // the model of the specified sheet
                sheetModelByIndex = _.isNumber(sheet) ? self.getSheetModel(sheet) : null;

            // name must not be empty
            if (sheetName.length === 0) {
                return 'sheetname:empty';
            }

            // name must not start nor end with an apostrophe (embedded apostrophes
            // are allowed) and must not contain any NPCs or other invalid characters
            if (/^'|[\x00-\x1f\x80-\x9f\[\]*?:\/\\]|'$/.test(sheetName)) {
                return 'sheetname:invalid';
            }

            // name must not be used already for another sheet
            if (_.isObject(sheetModelByName) && (!_.isObject(sheetModelByIndex) || (sheetModelByName !== sheetModelByIndex))) {
                return 'sheetname:used';
            }

            return '';
        }

        /**
         * Returns whether the passed sheet name is valid (non-empty, no
         * invalid characters, and not existing in the document).
         *
         * @param {String} sheetName
         *  The desired name of the sheet (case-insensitive).
         *
         * @param {Number} [sheet]
         *  If specified, the zero-based index of the sheet that is allowed to
         *  contain the passed name (used while renaming an existing sheet).
         *
         * @returns {Boolean}
         *  Whether the sheet name is valid and not used.
         */
        function isValidSheetName(sheetName, sheet) {
            return validateSheetName(sheetName, sheet).length === 0;
        }

        /**
         * Creates and inserts a new empty sheet into this document, and
         * triggers the appropriate document events.
         *
         * @param {Number} sheetIndex
         *  The insertion index of the new sheet. MUST be valid (not negative,
         *  and not greater than the number of existing sheets).
         *
         * @param {String} sheetType
         *  The type identifier of the new sheet.
         *
         * @param {String} sheetName
         *  The name of the new sheet. MUST NOT be used yet in this document.
         *
         * @param {Object} [attributes]
         *  The initial sheet attributes.
         *
         * @returns {SheetModel|Null}
         *  The model instance of the new sheet; or null, if no more sheets can
         *  be inserted into this document.
         */
        function createSheetAndTrigger(sheetIndex, sheetType, sheetName, attributes) {

            // first, check maximum sheet count
            if (self.getSheetCount() === Config.MAX_SHEET_COUNT) { return null; }

            var // the model instance of the new sheet
                sheetModel = new SheetModel(self, sheetType, attributes);

            self.trigger('insert:sheet:before', sheetIndex);
            sheetsCollection.splice(sheetIndex, 0, { model: sheetModel, name: sheetName });
            self.trigger('insert:sheet:after', sheetIndex, sheetModel);

            // trigger all events of the new sheet at this document model
            sheetModel.on('triggered', function (event2, type) {
                var triggerType = SHEET_TO_DOCUMENT_EVENTS[type] || type;
                self.trigger.apply(self, [triggerType, sheetModel.getIndex()].concat(_.toArray(arguments).slice(2)));
            });

            // invoke all range transformation handlers
            sheetModel.registerTransformationHandler(function (interval, insert, columns) {
                var sheetIndex = sheetModel.getIndex();
                transformationHandlers.forEach(function (handler) {
                    handler.call(self, sheetIndex, interval, insert, columns);
                });
            });

            return sheetModel;
        }

        /**
         * Callback handler for the document operation 'insertSheet'. Creates
         * and inserts a new sheet in this spreadsheet document.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'insertSheet' document operation.
         */
        function applyInsertSheetOperation(context) {

            var // the sheet index passed in the operation
                sheet = context.getInt('sheet'),
                // the sheet name passed in the operation
                sheetName = context.getStr('sheetName'),
                // the sheet type passed in the operation
                sheetType = context.getOptEnum('type', /^(work|chart|macro|dialog)sheet$/, 'worksheet'),
                // the sheet attributes passed in the operation (optional)
                attributes = context.getOptObj('attrs');

            // let the 'insertSheet' operation fail on purpose if specified in URL options
            context.ensure(!DEBUG_FAILING_IMPORT, 'debug mode: insert sheet fails');

            // check the sheet index, name (must not exist yet), and type
            context.ensure((sheet >= 0) && (sheet <= self.getSheetCount()), 'invalid sheet index');
            context.ensure(isValidSheetName(sheetName), 'invalid sheet name');

            // create and insert the new sheet
            var sheetModel = createSheetAndTrigger(sheet, sheetType, sheetName, attributes);
            context.ensure(sheetModel, 'cannot create sheet');
        }

        /**
         * Callback handler for the document operation 'deleteSheet'. Deletes
         * an existing sheet from this spreadsheet document.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'deleteSheet' document operation.
         */
        function applyDeleteSheetOperation(context, sheetModel, sheet) {

            // there must remain at least one sheet in the document
            context.ensure(self.getSheetCount() > 1, 'cannot delete last sheet');

            // remove the sheet from the document model, notify all listeners
            self.trigger('delete:sheet:before', sheet, sheetModel);
            sheetModel.destroy();
            sheetsCollection.splice(sheet, 1);
            self.trigger('delete:sheet:after', sheet);
        }

        /**
         * Callback handler for the document operation 'setSheetName'. Renames
         * an existing sheet in this spreadsheet document.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'setSheetName' document operation.
         */
        function applyRenameSheetOperation(context, sheetModel, sheet) {

            // the sheet object
            var sheetInfo = getSheetInfo(sheet);

            // the sheet name passed in the operation (no other sheet must contain this name)
            var sheetName = context.getStr('sheetName');
            context.ensure(isValidSheetName(sheetName, sheet), 'invalid sheet name');

            // rename the sheet (no event, if passed name matches old name)
            if (sheetInfo.name !== sheetName) {
                sheetInfo.name = sheetName;
                self.trigger('rename:sheet', sheet, sheetName, sheetModel);
            }
        }

        /**
         * Callback handler for the document operation 'moveSheet'. Moves an
         * existing sheet in this document to a new position.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'moveSheet' document operation.
         */
        function applyMoveSheetOperation(context, sheetModel, fromSheet) {

            // the target position of the sheet
            var toSheet = context.getInt('to');
            context.ensure((toSheet >= 0) && (toSheet <= self.getSheetCount()), 'invalid target index');

            // move the sheet, and notify all listeners
            var sheetInfo = sheetsCollection[fromSheet];
            self.trigger('move:sheet:before', fromSheet, toSheet, sheetModel);
            sheetsCollection.splice(fromSheet, 1);
            sheetsCollection.splice(toSheet, 0, sheetInfo);
            self.trigger('move:sheet:after', fromSheet, toSheet, sheetModel);
        }

        /**
         * Callback handler for the document operation 'copySheet'. Creates a
         * copy of an existing sheet in this spreadsheet document, and inserts
         * that sheet to a new position.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'copySheet' document operation.
         */
        function applyCopySheetOperation(context, sheetModel) {

            // the target position for the new sheet
            var toSheet = context.getInt('to');
            context.ensure((toSheet >= 0) && (toSheet <= self.getSheetCount()), 'invalid target index');

            // the new sheet name passed in the operation (no existing sheet must contain this name)
            var sheetName = context.getStr('sheetName');
            context.ensure(isValidSheetName(sheetName), 'invalid sheet name');

            // insert a new sheet model into the collection, clone all contents from the existing sheet
            var cloneModel = createSheetAndTrigger(toSheet, sheetModel.getType(), sheetName);
            context.ensure(cloneModel, 'cannot create sheet');
            cloneModel.cloneFrom(sheetModel);
        }

        /**
         * Creates the operation properties for the passed formula expression,
         * intended to be used in the document operations 'insertName', and
         * 'changeName'. The formula expression will be checked for syntax and
         * semantic errors, and it will be relocated for usage in document
         * operations.
         *
         * @param {String} grammar
         *  The identifier of the formula grammar to be used to parse the
         *  passed formula expression. See constructor of the class Tokenizer
         *  for details.
         *
         * @param {String} formula
         *  The formula expression to be bound to a defined name.
         *
         * @param {Number} refSheet
         *  The index of the reference sheet the formula is related to.
         *
         * @param {Address} refAddress
         *  The address of the reference cell used to interpret relative cell
         *  references in the formula expression.
         *
         * @returns {Object|Null}
         *  On success, an object with the property 'formula' containing the
         *  relocated formula expression, the property 'parse' with the current
         *  locale code, and an optional property 'ref' with the passed
         *  reference address (as formula expression) for ODF documents.
         */
        function getNameFormulaProperties(grammar, formula, refSheet, refAddress) {

            // a temporary token array for the passed formula, and for the reference address (TODO: sheet-local names)
            var tokenArray = new TokenArray(self, { type: 'name', temp: true });

            // the resulting operation properties
            var properties = null;

            // parse the formula; bug 40293: add sheet name to references without sheet (OOXML only)
            tokenArray.parseFormula(grammar, formula, app.isOOXML() ? { extendSheet: refSheet } : null);

            // interpret the formula to find syntax errors or semantic errors
            var formulaResult = tokenArray.interpretFormula('any');
            if (formulaResult.type !== 'error') {

                // TODO (bug 40438): Currently, the CalcEngine EXPECTS translated formula expressions, even
                // without 'parse' property. For now, always pass translated formulas and the parse property.

                // OOXML: relocate formula expression from reference cell to A1; ODF: pass reference cell with operation
                if (app.isOOXML()) {
                    properties = { formula: tokenArray.getFormula('ui', { refAddress: refAddress, targetAdddress: Address.A1 }) };
                } else {
                    properties = { formula: tokenArray.getFormula('ui') };
                    // store reference cell as formula expression with sheet name
                    tokenArray.clear().appendRange(new Range(refAddress), { sheet: refSheet, abs: true });
                    properties.ref = tokenArray.getFormula('op');
                }

                // TODO (bug 40438): remove the parse property
                properties.parse = Config.LOCALE;
            }

            tokenArray.destroy();
            return properties;
        }

        /**
         * Sends the own selection settings to remote clients.
         */
        var sendActiveSelectionDebounced = this.createDebouncedMethod(_.noop, function () {

            var // index of the active sheet
                activeSheet = self.getActiveSheet(),
                // the model of the active sheet
                sheetModel = self.getSheetModel(activeSheet);

            // do not send anything while settings are invalid
            if (sheetModel) {
                app.updateUserData({ sheet: activeSheet, selection: sheetModel.getViewAttribute('selection') });
            }
        }, { delay: 250 });

        /**
         * Handles 'docs:users' events containing the selections of all remote
         * clients.
         *
         * @param {Array} activeClients
         *  All active clients viewing or editing this document.
         */
        function changeActiveClientsHandler(activeClients) {

            var // filter for remote clients with existing selection data
                remoteClients = activeClients.filter(function (client) {
                    return client.remote && _.isObject(client.userData) && _.isNumber(client.userData.sheet) && _.isObject(client.userData.selection);
                });

            // store selections in the document (this triggers a change event which causes rendering)
            self.setViewAttribute('remoteClients', remoteClients);
        }

        // operation handler registration -------------------------------------

        /**
         * Registers a callback function for the specified document operation
         * addressing an existing sheet in this document. The operation must
         * contain a property 'sheet' with the index of an existing sheet,
         * unless the operation can be used to address this document model as
         * well as any sheet model (see option 'allowDoc' below).
         *
         * @param {String} name
         *  The name of a document operation to be handled by the passed
         *  callback function.
         *
         * @param {Function} handler
         *  The callback function that will be invoked for every operation with
         *  the specified name, and a property 'sheet' referring to an existing
         *  sheet. Receives the following parameters:
         *  (1) {SheetOperationContext} context
         *      The operation context wrapping the JSON operation object.
         *  (2) {SheetModel|SpreadsheetModel} targetModel
         *      The model of the sheet addressed by the operation. May be this
         *      document model instead of a sheet model, if the operation
         *      property 'sheet' is missing, and the operation is allowed to
         *      address the document (option 'allowDoc' has been set to true).
         *  (3) {Number|Null} sheet
         *      The zero-based index of the sheet addressed by the operation,
         *      or null, if the operation addresses this document model.
         *  Will be called in the context of this document model instance.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.allowDoc=false]
         *      If set to true, the operation property 'sheet' can be omitted
         *      to allow to address the entire document with the operation.
         */
        function registerSheetOperationHandler(name, handler, options) {
            var allowDocModel = Utils.getBooleanOption(options, 'allowDoc', false);
            return self.registerContextOperationHandler(name, function (context) {
                var isDocModel = allowDocModel && !context.has('sheet'),
                    sheet = isDocModel ? null : context.getInt('sheet'),
                    targetModel = isDocModel ? self : self.getSheetModel(sheet);
                context.ensure(targetModel, 'missing sheet');
                return handler.call(self, context, targetModel, sheet);
            });
        }

        /**
         * Registers a public method of a sheet model as callback function for
         * the specified document operation. The operation must contain a
         * property 'sheet' with the index of an existing sheet, unless the
         * operation can be used to address this document model as well as any
         * sheet model (see option 'allowDoc' below).
         *
         * @param {String} name
         *  The name of a document operation to be handled by a sheet model.
         *
         * @param {String} method
         *  The name of a public instance method of the sheet model addressed
         *  by the operation. Receives the following parameters:
         *  (1) {SheetOperationContext} context
         *      The operation context wrapping the JSON operation object.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.allowDoc=false]
         *      If set to true, the operation property 'sheet' can be omitted
         *      to allow to address the entire document with the operation.
         */
        function registerModelMethod(name, method, options) {
            registerSheetOperationHandler(name, function (context, targetModel) {
                targetModel[method](context);
            }, options);
        }

        /**
         * Registers a public method of a collection contained by a sheet model
         * as callback function for the specified document operation. The
         * operation must contain a property 'sheet' with the index of an
         * existing sheet, unless the operation can be used to address this
         * document model as well as any sheet model (see option 'allowDoc'
         * below).
         *
         * @param {String} name
         *  The name of a document operation to be handled by a collection of a
         *  sheet model.
         *
         * @param {String} getter
         *  The name of a public instance method of the sheet model returning
         *  one of its collection instances, e.g. 'getCellCollection'.
         *
         * @param {String} method
         *  The name of a public instance method of the collection returned by
         *  the sheet model addressed by the operation. Receives the following
         *  parameters:
         *  (1) {SheetOperationContext} context
         *      The operation context wrapping the JSON operation object.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.allowDoc=false]
         *      If set to true, the operation property 'sheet' can be omitted
         *      to allow to address the entire document with the operation. The
         *      collection getter method will be invoked on this document model
         *      instance.
         */
        function registerCollectionMethod(name, getter, method, options) {
            registerSheetOperationHandler(name, function (context, targetModel) {
                targetModel[getter]()[method](context);
            }, options);
        }

        /**
         * Registers a callback function for the specified document operation
         * addressing a drawing model in a specific sheet of this spreadsheet
         * document. The operation MUST contain a property 'start' (non-empty
         * array of integers) with the position of the drawing object in the
         * document. The position may point to a non-existing drawing (e.g. to
         * insert a new drawing), but it must point to an existing sheet (the
         * first array element is the sheet index).
         *
         * @param {String} name
         *  The name of a document operation for drawing objects to be handled
         *  by the passed callback function.
         *
         * @param {Function} handler
         *  The callback function that will be invoked for every operation with
         *  the specified name, and a property 'start' referring to an existing
         *  sheet. Receives the following parameters:
         *  (1) {SheetOperationContext} context
         *      The operation context wrapping the JSON operation object.
         *  (2) {SheetDrawingCollection} drawingCollection
         *      The drawing collection of the sheet addressed by the operation.
         *  (3) {Array<Number>} position
         *      The drawing position (the property 'start' from the operation
         *      without the first array element which was the sheet index).
         *  Will be called in the context of this document model instance.
         */
        function registerDrawingOperationHandler(name, handler) {
            return self.registerContextOperationHandler(name, function (context) {
                var position = context.getArr('start');
                context.ensure((position.length >= 2) && position.every(_.isNumber), 'invalid drawing position');
                var sheetModel = self.getSheetModel(position[0]);
                context.ensure(sheetModel, 'missing sheet');
                return handler.call(self, context, sheetModel.getDrawingCollection(), position.slice(1));
            });
        }

        /**
         * Registers a public method of the drawing collection of a sheet as
         * callback function for the specified drawing operation. See the
         * description of the method registerDrawingOperationHandler() for more
         * details about drawing collections.
         *
         * @param {String} name
         *  The name of a document operation for drawing objects to be handled
         *  by a drawing collection.
         *
         * @param {String} method
         *  The name of a public instance method of the drawing collection of
         *  the sheet model addressed by the operation. Receives the following
         *  parameters:
         *  (1) {SheetOperationContext} context
         *      The operation context wrapping the JSON operation object.
         *  (2) {Array<Number>} position
         *      The drawing position (the property 'start' from the operation
         *      without the first array element which was the sheet index).
         */
        function registerDrawingCollectionMethod(name, method) {
            registerDrawingOperationHandler(name, function (context, collection, position) {
                collection[method](context, position);
            });
        }

        /**
         * Registers a public method of a drawing model as callback function
         * for the specified drawing operation. See the description of the
         * method registerDrawingOperationHandler() for more details about
         * drawing collections.
         *
         * @param {String} name
         *  The name of a document operation for drawing objects to be handled
         *  by the addressed drawing model.
         *
         * @param {String} method
         *  The name of a public instance method of the drawing model addressed
         *  by the operation. Receives the following parameters:
         *  (1) {SheetOperationContext} context
         *      The operation context wrapping the JSON operation object.
         *
         * @param {String} [type]
         *  If specified, the drawing model addressed by the operation must be
         *  of the specified type. If a drawing model with another type exists
         *  at the specified document position, the operation will fail.
         */
        function registerDrawingModelMethod(name, method, type) {
            registerDrawingOperationHandler(name, function (context, collection, position) {
                var drawingModel = collection.findModel(position, { deep: true, type: type });
                context.ensure(drawingModel, 'invalid drawing position');
                drawingModel[method](context);
            });
        }

        // protected methods --------------------------------------------------

        /**
         * Called from the applications import-fail handler when document
         * import has failed. Inserts an empty sheet, if this model does not
         * contain any sheets. This helps preventing to write if-else
         * statements for empty documents at thousand places in the view code
         * (there will always be an active sheet with selection etc.).
         *
         * @internal
         *  Not intended to be called from any other code than the import
         *  callback handlers of the application.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.prepareInvalidDocument = function () {
            if (sheetsCollection.length === 0) {
                createSheetAndTrigger(0, 'worksheet', 'Sheet1');
            }
            return this;
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the number formatter of this document.
         *
         * @returns {NumberFormatter}
         *  The number formatter of this document.
         */
        this.getNumberFormatter = function () {
            return numberFormatter;
        };

        /**
         * Returns the formula tokenizer of this document for the specified
         * formula grammar.
         *
         * @param {String} grammar
         *  The formula grammar supported by the tokenizer. See constructor of
         *  the class Tokenizer for details.
         *
         * @returns {Tokenizer}
         *  The formula compiler of this document for the specified formula
         *  grammar.
         */
        this.getFormulaTokenizer = function (grammar) {
            return formulaTokenizers[grammar] || (formulaTokenizers[grammar] = new Tokenizer(this, grammar));
        };

        /**
         * Returns the formula compiler of this document.
         *
         * @returns {Compiler}
         *  The formula compiler of this document.
         */
        this.getFormulaCompiler = function () {
            return formulaCompiler;
        };

        /**
         * Returns the formula interpreter of this document.
         *
         * @returns {Interpreter}
         *  The formula interpreter of this document.
         */
        this.getFormulaInterpreter = function () {
            return formulaInterpreter;
        };

        /**
         * Returns the collection of all global defined names contained in this
         * document.
         *
         * @returns {NameCollection}
         *  The collection of all defined names in this document.
         */
        this.getNameCollection = function () {
            return nameCollection;
        };

        // sheet range --------------------------------------------------------

        /**
         * Returns the column index of the rightmost cells in a sheet.
         *
         * @returns {Number}
         *  The column index of the rightmost cells in a sheet.
         */
        this.getMaxCol = function () {
            return maxCol;
        };

        /**
         * Returns the row index of the bottom cells in a sheet.
         *
         * @returns {Number}
         *  The row index of the bottom cells in a sheet.
         */
        this.getMaxRow = function () {
            return maxRow;
        };

        /**
         * Returns the maximum column or row index in a sheet.
         *
         * @param {Boolean} columns
         *  Whether to return the maximum column index (true), or the maximum
         *  row index (false).
         *
         * @returns {Number}
         *  The maximum column or row index in a sheet.
         */
        this.getMaxIndex = function (columns) {
            return columns ? maxCol : maxRow;
        };

        /**
         * Returns the range address of the entire area in a sheet, from the
         * top-left cell to the bottom-right cell.
         *
         * @returns {Range}
         *  The range address of an entire sheet.
         */
        this.getSheetRange = function () {
            return Range.create(0, 0, maxCol, maxRow);
        };

        /**
         * Returns whether the passed range address covers one or more entire
         * columns in the active sheet.
         *
         * @param {Range} range
         *  A cell range address.
         *
         * @returns {Boolean}
         *  Whether the passed range address covers one or more entire columns.
         */
        this.isColRange = function (range) {
            return (range.start[1] === 0) && (range.end[1] === maxRow);
        };

        /**
         * Returns the address of the cell range covering the specified column
         * interval.
         *
         * @param {Interval|Number} interval
         *  The column interval, or a single zero-based column index.
         *
         * @returns {Range}
         *  The address of the column range.
         */
        this.makeColRange = function (interval) {
            return Range.create(_.isNumber(interval) ? interval : interval.first, 0, _.isNumber(interval) ? interval : interval.last, maxRow);
        };

        /**
         * Returns whether the passed range address covers one or more entire
         * rows in the active sheet.
         *
         * @param {Range} range
         *  A cell range address.
         *
         * @returns {Boolean}
         *  Whether the passed range address covers one or more entire rows.
         */
        this.isRowRange = function (range) {
            return (range.start[0] === 0) && (range.end[0] === maxCol);
        };

        /**
         * Returns the address of the cell range covering the specified row
         * interval.
         *
         * @param {Interval|Number} interval
         *  The row interval, or a single zero-based row index.
         *
         * @returns {Range}
         *  The address of the row range.
         */
        this.makeRowRange = function (interval) {
            return Range.create(0, _.isNumber(interval) ? interval : interval.first, maxCol, _.isNumber(interval) ? interval : interval.last);
        };

        /**
         * Returns whether the passed range address covers one or more entire
         * columns or rows in the active sheet.
         *
         * @param {Range} range
         *  A range address.
         *
         * @param {Boolean} columns
         *  Whether to check for a full column range (true), or for a full row
         *  range (false).
         *
         * @returns {Boolean}
         *  Whether the passed range address covers one or more entire columns
         *  or rows.
         */
        this.isFullRange = function (range, columns) {
            return columns ? this.isColRange(range) : this.isRowRange(range);
        };

        /**
         * Returns the address of the cell range covering the specified column
         * or row interval.
         *
         * @param {Interval|Number} interval
         *  The column or row interval, or a single column or row index.
         *
         * @param {Boolean} columns
         *  Whether to create a full column range (true), or a full row range
         *  (false).
         *
         * @returns {Range}
         *  The address of the full column or row range.
         */
        this.makeFullRange = function (interval, columns) {
            return columns ? this.makeColRange(interval) : this.makeRowRange(interval);
        };

        /**
         * Returns whether the passed cell address contains valid column and
         * row indexes (inside the limits of the sheets in this document).
         *
         * @param {Address} address
         *  The cell address to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed cell address contains valid column and row
         *  indexes.
         */
        this.isValidAddress = function (address) {
            return (address[0] >= 0) && (address[0] <= maxCol) && (address[1] >= 0) && (address[1] <= maxRow);
        };

        /**
         * Returns whether the passed cell range address contains valid start
         * and end addresses (inside the limits of the sheets in this document)
         * that are in correct order (start before end).
         *
         * @param {Range} range
         *  The cell range address to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed cell range address contains valid start and end
         *  addresses in correct order.
         */
        this.isValidRange = function (range) {
            return (range.start[0] >= 0) && (range.start[0] <= range.end[0]) && (range.end[0] <= maxCol) &&
                (range.start[1] >= 0) && (range.start[1] <= range.end[1]) && (range.end[1] <= maxRow);
        };

        /**
         * Converts the passed JSON data to a column/row interval (an instance
         * of the class Interval) if possible.
         *
         * @param {Any} json
         *  A value that will be converted to a column/row interval. To create
         *  an interval object, the data MUST be an object with the property
         *  'first' containing a valid column/row index (non-negative, less
         *  than or equal to the maximum index of this document). Optionally,
         *  the object may contain a property 'last' containing a valid index
         *  (non-negative, equal to or greater than 'first', less than or equal
         *  to the maximum index of this document).
         *
         * @param {Boolean} columns
         *  Whether to convert the passed JSON data to a column interval
         *  (true), or to a row interval (false).
         *
         * @param {Boolean} [operationMode=false]
         *  If set to true, the JSON interval is expected to contain the
         *  properties 'start' and 'end' (as used in document operations),
         *  instead of 'first' and 'last'.
         *
         * @returns {Interval|Null}
         *  A column/row interval object, if the passed JSON data can be
         *  converted to an interval; otherwise null.
         */
        this.createInterval = function (json, columns, operationMode) {

            // data must be an object
            if (!_.isObject(json)) { return null; }

            var firstName = operationMode ? 'start' : 'first',
                lastName = operationMode ? 'end' : 'last',
                maxIndex = this.getMaxIndex(columns);

            // 'first' property must be a valid index
            var first = json[firstName];
            if (!_.isNumber(first) || !isFinite(first) || (first < 0) || (first > maxIndex)) { return null; }

            // 'last' property may be missing
            if (!(lastName in json)) { return new Interval(first); }

            // 'last' property must be a valid index following 'first'
            var last = json[lastName];
            if (!_.isNumber(last) || !isFinite(last) || (last < first) || (last > maxIndex)) { return null; }

            return new Interval(first, last);
        };

        /**
         * Converts the passed JSON data to an array of column/row intervals
         * (an instance of the class IntervalArray) if possible.
         *
         * @param {Any} json
         *  A value that will be converted to an array of index intervals. To
         *  be able to create an interval array, the data MUST be an array with
         *  valid JSON representations of index intervals (see method
         *  SpreadsheetModel.createInterval() for details).
         *
         * @param {Boolean} columns
         *  Whether to convert the passed JSON data to column intervals (true),
         *  or to row intervals (false).
         *
         * @param {Boolean} [operationMode=false]
         *  If set to true, the JSON intervals are expected to contain the
         *  properties 'start' and 'end' (as used in document operations),
         *  instead of 'first' and 'last'.
         *
         * @returns {IntervalArray|Null}
         *  An array of index intervals, if the passed JSON data can be
         *  converted to such an array; otherwise null.
         */
        this.createIntervalArray = function (json, columns, operationMode) {

            // data must be an array
            if (!_.isArray(json)) { return null; }

            // process array elements (must be valid intervals), early exit on first invalid interval
            var intervals = new IntervalArray();
            json.every(function (interval) {
                interval = this.createInterval(interval, columns, operationMode);
                if (interval) { intervals.push(interval); return true; }
            }, this);

            // ensure that all intervals have been converted successfully
            return (json.length === intervals.length) ? intervals : null;
        };

        /**
         * Converts the passed JSON data to a cell address object (an instance
         * of the class Address) if possible.
         *
         * @param {Any} json
         *  A value that will be converted to a cell address. To create a cell
         *  address object, the data MUST be an array with two elements. Each
         *  element must be a finite number. The first array element MUST be a
         *  valid column index (non-negative, less than or equal to the maximum
         *  column index of this document). The second array element MUST be a
         *  valid row index (non-negative, less than or equal to the maximum
         *  row index of this document).
         *
         * @returns {Address|Null}
         *  A cell address object, if the passed JSON data can be converted to
         *  a cell address; otherwise null.
         */
        this.createAddress = function (json) {

            // data must be an array with exactly two elements
            if (!_.isArray(json) || (json.length !== 2)) { return null; }

            // elements must valid column/row indexes
            if (!_.isNumber(json[0]) || !isFinite(json[0]) || (json[0] < 0) || (json[0] > maxCol)) { return null; }
            if (!_.isNumber(json[1]) || !isFinite(json[1]) || (json[1] < 0) || (json[1] > maxRow)) { return null; }

            return new Address(json[0], json[1]);
        };

        /**
         * Converts the passed JSON data to an array of cell addresses (an
         * instance of the class AddressArray) if possible.
         *
         * @param {Any} json
         *  A value that will be converted to an array of cell addresses. To
         *  successfully create an array, the data MUST be an array with valid
         *  JSON representations of cell addresses (see method
         *  SpreadsheetModel.createAddress() for details).
         *
         * @returns {AddressArray|Null}
         *  An array of cell addresses, if the passed JSON data can be
         *  converted to such an array; otherwise null.
         */
        this.createAddressArray = function (json) {

            // data must be an array
            if (!_.isArray(json)) { return null; }

            // process array elements (must be valid addresses), early exit on first invalid address
            var addresses = new AddressArray();
            json.every(function (address) {
                address = this.createAddress(address);
                if (address) { addresses.push(address); return true; }
            }, this);

            // ensure that all addresses have been converted successfully
            return (json.length === addresses.length) ? addresses : null;
        };

        /**
         * Converts the passed JSON data to a cell range address object (an
         * instance of the class Range) if possible.
         *
         * @param {Any} json
         *  A value that will be converted to a cell range address. To create a
         *  cell range address, the data MUST be an object with the property
         *  'start', containing a valid JSON representation of the start cell
         *  address of the range (see method SpreadsheetModel.createAddress()
         *  for details), and an optional property 'end' with another JSON cell
         *  address for the end of the range (defaults to 'start').
         *
         * @returns {Range|Null}
         *  A cell range address object, if the passed JSON data can be
         *  converted to a cell range address; otherwise null.
         */
        this.createRange = function (json) {

            // data must be an object
            if (!_.isObject(json)) { return null; }

            // 'start' property must be a valid address
            var start = this.createAddress(json.start);
            if (!start) { return null; }

            // 'end' property may be missing
            if (!('end' in json)) { return new Range(start); }

            // 'end' property must be a valid address following the start address
            var end = this.createAddress(json.end);
            if (!end || (end[0] < start[0]) || (end[1] < start[1])) { return null; }

            return new Range(start, end);
        };

        /**
         * Converts the passed JSON data to an array of cell range addresses
         * (an instance of the class RangeArray) if possible.
         *
         * @param {Any} json
         *  A value that will be converted to an array of cell range addresses.
         *  To create an array, the data MUST be an array with valid JSON
         *  representations of cell range addresses (see method
         *  SpreadsheetModel.createRange() for details).
         *
         * @returns {RangeArray|Null}
         *  An array of cell range addresses, if the passed JSON data can be
         *  converted to such an array; otherwise null.
         */
        this.createRangeArray = function (json) {

            // data must be an array
            if (!_.isArray(json)) { return null; }

            // process array elements (must be valid ranges), early exit on first invalid range
            var ranges = new RangeArray();
            json.every(function (range) {
                range = this.createRange(range);
                if (range) { ranges.push(range); return true; }
            }, this);

            // ensure that all ranges have been converted successfully
            return (json.length === ranges.length) ? ranges : null;
        };

        /**
         * Returns a new range that has been moved by the specified distance,
         * but always keeping the range inside the sheet limits (see method
         * SpreadsheetModel.getCroppedMovedRange() for an alternative keeping
         * the specified move distance intact).
         *
         * @param {Range} range
         *  The address of the range to be moved.
         *
         * @param {Number} cols
         *  The number of columns the range will be moved. Negative values will
         *  move the range towards the leading border of the sheet, positive
         *  values will move the range towards the trailing border.
         *
         * @param {Number} rows
         *  The number of rows the range will be moved. Negative values will
         *  move the range towards the top border of the sheet, positive values
         *  will move the range towards the bottom border.
         *
         * @returns {Range}
         *  The address of the moved range. If the move distances passed to
         *  this method are too large, the range will only be moved to the
         *  sheet borders.
         */
        this.getBoundedMovedRange = function (range, cols, rows) {

            // reduce move distance to keep the range address valid
            cols = Utils.minMax(cols, -range.start[0], maxCol - range.end[0]);
            rows = Utils.minMax(rows, -range.start[1], maxRow - range.end[1]);

            // return the new range
            return Range.create(range.start[0] + cols, range.start[1] + rows, range.end[0] + cols, range.end[1] + rows);
        };

        /**
         * Returns a new range that has been moved by the specified distance.
         * If the moved range leaves the sheet area, it will be cropped at the
         * sheet borders (see method SpreadsheetModel.getBoundedMovedRange()
         * for an alternative keeping the original size of the range intact).
         *
         * @param {Range} range
         *  The address of the range to be moved.
         *
         * @param {Number} cols
         *  The number of columns the range will be moved. Negative values will
         *  move the range towards the leading border of the sheet, positive
         *  values will move the range towards the trailing border.
         *
         * @param {Number} rows
         *  The number of rows the range will be moved. Negative values will
         *  move the range towards the top border of the sheet, positive values
         *  will move the range towards the bottom border.
         *
         * @returns {Range|Null}
         *  The address of the moved range. If the specified move distances are
         *  too large, the range will be cropped to the sheet area. If the
         *  range has been moved completely outside the sheet, this method will
         *  return null.
         */
        this.getCroppedMovedRange = function (range, cols, rows) {

            var // the address components of the moved range
                col1 = Math.max(0, range.start[0] + cols),
                row1 = Math.max(0, range.start[1] + rows),
                col2 = Math.min(maxCol, range.end[0] + cols),
                row2 = Math.min(maxRow, range.end[1] + rows);

            // return null, if the resulting range becomes invalid
            return ((col1 <= col2) && (row1 <= row2)) ? Range.create(col1, row1, col2, row2) : null;
        };

        /**
         * Transforms the passed column or row index according to the specified
         * column or row operation. Used to get the effective column/row index
         * after columns or rows have been insert into or deleted from a sheet.
         *
         * @param {Number} sourceIndex
         *  The column/row index to be transformed.
         *
         * @param {Interval} operationInterval
         *  The column/row interval of the operation.
         *
         * @param {Boolean} insert
         *  Whether the columns/rows have been inserted into the sheet (true),
         *  or deleted from the sheet (false).
         *
         * @param {Boolean} columns
         *  Whether the passed index is a column index and the passed operation
         *  interval is a column interval (true), or index and interval are for
         *  rows (false).
         *
         * @returns {Number|Null}
         *  The transformed column/row index. If the column/row has been
         *  deleted completely while deleting columns/rows, or has been shifted
         *  outside the sheet completely while inserting columns/rows, null
         *  will be returned instead.
         */
        this.transformIndex = function (sourceIndex, operationInterval, insert, columns) {

            var // the length of the operation interval
                size = operationInterval.size();

            if (insert) {
                // insert rows/columns (increase the index)
                if (operationInterval.first <= sourceIndex) {
                    sourceIndex += size;
                }
            } else {
                // delete rows/columns (decrease or invalidate the address index)
                if (operationInterval.last < sourceIndex) {
                    sourceIndex -= size;
                } else if (operationInterval.first <= sourceIndex) {
                    sourceIndex = -1;
                }
            }

            return (sourceIndex >= 0) && (sourceIndex <= this.getMaxIndex(columns)) ? sourceIndex : null;
        };

        /**
         * Transforms the passed cell address according to the specified column
         * or row operation. Used to get the effective cell address after
         * columns or rows have been insert into or deleted from a sheet.
         *
         * @param {Address} sourceAddress
         *  The address of the cell to be transformed.
         *
         * @param {Interval} operationInterval
         *  The column/row interval of the operation.
         *
         * @param {Boolean} insert
         *  Whether the columns/rows have been inserted into the sheet (true),
         *  or deleted from the sheet (false).
         *
         * @param {Boolean} columns
         *  Whether the passed operation interval is a column interval (true),
         *  or a row interval (false).
         *
         * @returns {Address|Null}
         *  The address of the transformed cell. If the cell has been deleted
         *  completely while deleting columns/rows, or has been shifted outside
         *  the sheet while inserting columns/rows, null will be returned
         *  instead.
         */
        this.transformAddress = function (sourceAddress, operationInterval, insert, columns) {

            var // the transformed index
                newIndex = this.transformIndex(sourceAddress.get(columns), operationInterval, insert, columns);

            // create a new address with adjusted column/row index
            return _.isNumber(newIndex) ? sourceAddress.clone().set(newIndex, columns) : null;
        };

        /**
         * Transforms the passed column/row interval according to the specified
         * operation. Used to get the effective interval after columns or rows
         * have been insert into or deleted from a sheet.
         *
         * @param {Interval} sourceInterval
         *  The column/row interval to be transformed.
         *
         * @param {Interval} operationInterval
         *  The column/row interval of the operation.
         *
         * @param {Boolean} insert
         *  Whether the columns/rows have been inserted into the sheet (true),
         *  or deleted from the sheet (false).
         *
         * @param {Boolean} columns
         *  Whether the passed operation interval is a column interval (true),
         *  or a row interval (false). Influences the maximum index allowed in
         *  the transformed interval.
         *
         * @param {Boolean} [expandEnd=false]
         *  If set to true, the end position of the interval will be expanded,
         *  if new columns/rows will be inserted exactly behind the interval.
         *
         * @returns {Interval|Null}
         *  The transformed column/row interval. If the interval has been
         *  deleted completely while deleting columns/rows, or has been shifted
         *  outside the sheet completely while inserting columns/rows, null
         *  will be returned instead.
         */
        this.transformInterval = function (sourceInterval, operationInterval, insert, columns, expandEnd) {

            var // the length of the interval
                size = operationInterval.size(),
                // the resulting transformed interval
                targetInterval = sourceInterval.clone();

            if (insert) {
                // insert rows/columns (shift, or enlarge the range)
                if (operationInterval.first <= sourceInterval.first) { targetInterval.first += size; }
                if (operationInterval.first <= sourceInterval.last + (expandEnd ? 1 : 0)) { targetInterval.last = Math.min(sourceInterval.last + size, this.getMaxIndex(columns)); }
            } else {
                // delete rows/columns (shift, shrink, or delete the range)
                if (operationInterval.first < sourceInterval.first) { targetInterval.first = Math.max(operationInterval.first, sourceInterval.first - size); }
                if (operationInterval.first <= sourceInterval.last) { targetInterval.last = Math.max(sourceInterval.last - size, operationInterval.first - 1); }
            }

            // delete interval, if resulting start position exceeds end position (happens when deleting
            // the entire interval, or when shifting it outside the sheet while inserting)
            return (targetInterval.first <= targetInterval.last) ? targetInterval : null;
        };

        /**
         * Transforms the passed cell range address according to the specified
         * column or row operation. Used to get the effective range after
         * columns or rows have been insert into or deleted from a sheet.
         *
         * @param {Range} sourceRange
         *  The address of the cell range to be transformed.
         *
         * @param {Interval} operationInterval
         *  The column/row interval of the operation.
         *
         * @param {Boolean} insert
         *  Whether the columns/rows have been inserted into the sheet (true),
         *  or deleted from the sheet (false).
         *
         * @param {Boolean} columns
         *  Whether the passed operation interval is a column interval (true),
         *  or a row interval (false).
         *
         * @param {Boolean} [expandEnd=false]
         *  If set to true, the end position of the range will be expanded, if
         *  new columns will be inserted exactly right of the range, or new
         *  rows will be inserted exactly below the range.
         *
         * @returns {Range|Null}
         *  The address of the transformed cell range. If the range has been
         *  deleted completely while deleting columns/rows, or has been shifted
         *  outside the sheet completely while inserting columns/rows, null
         *  will be returned instead.
         */
        this.transformRange = function (sourceRange, operationInterval, insert, columns, expandEnd) {

            var // the source column/row intervals from the range
                colInterval = sourceRange.colInterval(),
                rowInterval = sourceRange.rowInterval(),
                // the transformed interval
                resultInterval = this.transformInterval(columns ? colInterval : rowInterval, operationInterval, insert, columns, expandEnd);

            // assign result to the correct interval
            if (columns) { colInterval = resultInterval; } else { rowInterval = resultInterval; }

            // build and return the resulting range
            return resultInterval ? Range.createFromIntervals(colInterval, rowInterval) : null;
        };

        /**
         * Transforms the passed array of column/row intervals according to the
         * specified operation. Used to get the effective intervals after
         * columns or rows have been inserted into or deleted from a sheet.
         *
         * @param {IntervalArray|Interval} sourceIntervals
         *  An array of index intervals to be transformed, or a single index
         *  interval.
         *
         * @param {Interval} operationInterval
         *  The column/row interval of the operation.
         *
         * @param {Boolean} insert
         *  Whether the columns/rows have been inserted into the sheet (true),
         *  or deleted from the sheet (false).
         *
         * @param {Boolean} columns
         *  Whether the passed operation interval is a column interval (true),
         *  or a row interval (false). Influences the maximum index allowed in
         *  the transformed interval.
         *
         * @param {String} [insertMode='expand']
         *  Specifies how to modify intervals for insert operations (parameter
         *  'insert' set to true). The following modes are supported:
         *  - 'expand' (default): If columns/rows will be inserted into the
         *      middle of an interval (between first and last entry of the
         *      interval), it will be expanded.
         *  - 'expandEnd': Similar to 'expand', but columns/rows inserted
         *      exactly at the end of the interval will cause to expand the
         *      interval too.
         *  - 'split': If columns/rows will be inserted into the middle of an
         *      interval (between first and last entry of the interval), it
         *      will be split into two intervals; the inserted columns/rows
         *      will not be part of the new intervals.
         *
         * @returns {IntervalArray}
         *  The transformed column/row intervals. Source intervals that have
         *  been deleted completely while deleting columns/rows, or that have
         *  been shifted outside the sheet completely while inserting
         *  columns/rows, will not be included (the resulting array may even be
         *  empty).
         */
        this.transformIntervals = function (sourceIntervals, operationInterval, insert, columns, insertMode) {

            var // whether to expand intervals at the end when inserting behind the interval
                expandEnd = insertMode === 'expandEnd',
                // whether to split intervals on insertion in the middle
                split = insertMode === 'split',
                // the resulting transformed intervals
                resultIntervals = new IntervalArray();

            // transform all passed intervals
            IntervalArray.forEach(sourceIntervals, function (sourceInterval) {

                var // basic transformation of the source interval
                    resultInterval = self.transformInterval(sourceInterval, operationInterval, insert, columns, expandEnd);

                // split expanded intervals if specified
                if (resultInterval) {
                    if (insert && split && (resultInterval.first < operationInterval.first) && (operationInterval.last < resultInterval.last)) {
                        resultIntervals.push(new Interval(resultInterval.first, operationInterval.first - 1), new Interval(operationInterval.last + 1, resultInterval.last));
                    } else {
                        resultIntervals.push(resultInterval);
                    }
                }
            });

            return resultIntervals;
        };

        /**
         * Transforms the passed array of cell range addresses according to the
         * specified operation. Used to get the effective range addresses after
         * columns or rows have been insert into or deleted from a sheet.
         *
         * @param {RangeArray|Range} sourceRanges
         *  An array of cell range addresses to be transformed, or a single
         *  cell range address.
         *
         * @param {Interval} operationInterval
         *  The column/row interval of the operation.
         *
         * @param {Boolean} insert
         *  Whether the columns/rows have been inserted into the sheet (true),
         *  or deleted from the sheet (false).
         *
         * @param {Boolean} columns
         *  Whether the passed operation interval is a column interval (true),
         *  or a row interval (false).
         *
         * @param {String} [insertMode='expand']
         *  Specifies how to modify intervals for insert operations (parameter
         *  'insert' set to true). The following modes are supported:
         *  - 'expand' (default): If columns/rows will be inserted into the
         *      middle of a range (between first and last column/row), it will
         *      be expanded.
         *  - 'expandEnd': Similar to 'expand', but columns/rows inserted
         *      exactly at the end of the range will cause to expand the range
         *      too.
         *  - 'split': If columns/rows will be inserted into the middle of a
         *      range (between first and last column/row), it will be split
         *      into two ranges; the inserted columns/rows will not be part of
         *      the new ranges.
         *
         * @returns {RangeArray}
         *  The transformed cell range addresses. Source ranges that have been
         *  deleted completely while deleting columns/rows, or that have been
         *  shifted outside the sheet completely while inserting columns/rows,
         *  will not be included (the resulting array may even be empty).
         */
        this.transformRanges = function (sourceRanges, operationInterval, insert, columns, insertMode) {

            var // the resulting transformed ranges
                resultRanges = new RangeArray();

            // transform all passed cell ranges
            RangeArray.forEach(sourceRanges, function (sourceRange) {

                var // the source column/row intervals from the range
                    colIntervals = sourceRange.colInterval(),
                    rowIntervals = sourceRange.rowInterval(),
                    // the transformed interval (may be split into two parts)
                    resultIntervals = self.transformIntervals(columns ? colIntervals : rowIntervals, operationInterval, insert, columns, insertMode);

                // assign result to the correct interval
                if (columns) { colIntervals = resultIntervals; } else { rowIntervals = resultIntervals; }

                // build and return the resulting range
                resultRanges.append(RangeArray.createFromIntervals(colIntervals, rowIntervals));
            });

            return resultRanges;
        };

        /**
         * Registers a callback function that will be invoked after columns or
         * rows have been inserted into or deleted from any sheet in this
         * document.
         *
         * @param {Function} handler
         *  The callback handler function. Will be invoked in the context of
         *  this model instance. Receives the following parameters:
         *  (1) {Number} sheet
         *      The zero-based sheet index.
         *  (2) {Interval} interval
         *      The column/row interval of the insert/delete operation, with
         *      the index properties 'first' and 'last'.
         *  (3) {Boolean} insert
         *      Whether the specified column/row interval has been inserted
         *      into this sheet (true), or deleted from this sheet (false).
         *  (4) {Boolean} columns
         *      Whether the specified interval is a column interval (true), or
         *      a row interval (false).
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.registerTransformationHandler = function (handler) {
            transformationHandlers.push(handler);
            return this;
        };

        /**
         * Unregisters a callback function that has been registered with the
         * method SpreadsheetModel.registerTransformationHandler() before.
         *
         * @param {Function} handler
         *  The callback handler function to be unregistered.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.unregisterTransformationHandler = function (handler) {
            transformationHandlers = _.without(transformationHandlers, handler);
            return this;
        };

        // sheets -------------------------------------------------------------

        /**
         * Returns the maximum length of a sheet name according to the current
         * file format.
         *
         * @returns {Number}
         *  The maximum length of a sheet name.
         */
        this.getMaxSheetNameLength = function () {
            return app.isOOXML() ? 31 : 65535;
        };

        /**
         * Returns the number of sheets in the spreadsheet document.
         *
         * @returns {Number}
         *  The number of sheets in the document.
         */
        this.getSheetCount = function () {
            return sheetsCollection.length;
        };

        /**
         * Returns the names of all sheets in the spreadsheet document.
         *
         * @returns {Array<String>}
         *  The names of all sheets in the document.
         */
        this.getSheetNames = function () {
            return _.pluck(sheetsCollection, 'name');
        };

        /**
         * Returns whether a sheet at the specified index or with the specified
         * case-insensitive name exists in the spreadsheet document.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {Boolean}
         *  Whether a sheet exists at the passed index or with the passed name.
         */
        this.hasSheet = function (sheet) {
            return _.isObject(getSheetInfo(sheet));
        };

        /**
         * Returns the exact name of the sheet at the specified index or with
         * the specified case-insensitive name.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {String|Null}
         *  The name of the sheet at the passed index or with the passed name,
         *  or null if the passed index or name is invalid.
         */
        this.getSheetName = function (sheet) {
            var sheetInfo = getSheetInfo(sheet);
            return _.isObject(sheetInfo) ? sheetInfo.name : null;
        };

        /**
         * Returns the current index of the sheet with the specified name.
         *
         * @param {String} sheetName
         *  The case-insensitive sheet name.
         *
         * @returns {Number}
         *  The current zero-based index of the sheet with the specified name,
         *  or -1 if a sheet with the passed name does not exist.
         */
        this.getSheetIndex = function (sheetName) {
            sheetName = sheetName.toLowerCase();
            return Utils.findFirstIndex(sheetsCollection, function (sheetInfo) {
                return sheetInfo.name.toLowerCase() === sheetName;
            });
        };

        /**
         * Returns the current sheet index of the passed sheet model instance.
         *
         * @param {SheetModel} sheetModel
         *  The sheet model whose index will be returned.
         *
         * @returns {Number}
         *  The current zero-based sheet index of the passed sheet model, or -1
         *  if the passed object is not a valid sheet model.
         */
        this.getSheetIndexOfModel = function (sheetModel) {
            return Utils.findFirstIndex(sheetsCollection, function (sheetInfo) {
                return sheetInfo.model === sheetModel;
            });
        };

        /**
         * Returns the model instance for the sheet at the specified index or
         * with the specified name.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {SheetModel|Null}
         *  The model instance for the sheet at the passed index or with the
         *  specified name, or null if the passed index or name is invalid.
         */
        this.getSheetModel = function (sheet) {
            var sheetInfo = getSheetInfo(sheet);
            return _.isObject(sheetInfo) ? sheetInfo.model : null;
        };

        /**
         * Invokes the passed iterator function for all sheet models contained
         * in this document.
         *
         * @param {Function} iterator
         *  The iterator function invoked for all sheet models. Receives the
         *  following parameters:
         *  (1) {SheetModel} sheetModel
         *      The current sheet model instance.
         *  (2) {Number} sheet
         *      The zero-based sheet index.
         *  (3) {String} name
         *      The name of the sheet.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the sheets will be visited in reversed order.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateSheetModels = function (iterator, options) {
            return Utils.iterateArray(sheetsCollection, function (sheetInfo, sheet) {
                return iterator.call(this, sheetInfo.model, sheet, sheetInfo.name);
            }, options);
        };

        /**
         * Creates a new sheet in this document.
         *
         * @param {Number} sheet
         *  The zero-based insertion index of the sheet.
         *
         * @param {String} sheetName
         *  The name of the new sheet. Must not be empty. Must not be equal to
         *  the name of any existing sheet.
         *
         * @param {Object} [attrs]
         *  Attribute set containing initial formatting attributes for the new
         *  sheet.
         *
         * @returns {String}
         *  The empty string, if the sheet has been inserted successfully;
         *  otherwise one of the following error codes:
         *  - 'sheetname:empty': The passed sheet name is empty.
         *  - 'sheetname:invalid': The passed sheet name contains invalid
         *      characters.
         *  - 'sheetname:used': The passed sheet name is already used in the
         *      document.
         *  - 'internal': Internal error while applying the operation.
         */
        this.insertSheet = function (sheet, sheetName, attrs) {

            var // the operation object
                operation = {
                    name: Operations.INSERT_SHEET,
                    sheet: sheet,
                    sheetName: Utils.cleanString(sheetName) // remove all NPCs from passed sheet name
                },
                // check that the sheet name can be used (prevent internal application error on failing operation)
                result = validateSheetName(operation.sheetName);

            // prevent internal application error on invalid sheet name
            if (result.length > 0) { return result; }

            // add optional attributes
            if (_.isObject(attrs)) { operation.attrs = attrs; }

            // generate and apply the operation
            return this.applyOperations(operation) ? '' : 'internal';
        };

        /**
         * Removes a sheet from this document.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet.
         *
         * @returns {Boolean}
         *  Whether deleting the sheet was successful.
         */
        this.deleteSheet = function (sheet) {
            // check that the sheet can be removed (prevent internal application error on failing operation)
            return (this.getSheetCount() > 1) && this.applyOperations({ name: Operations.DELETE_SHEET, sheet: sheet });
        };

        /**
         * Moves a sheet in the spreadsheet document to a new position.
         *
         * @param {Number} fromSheet
         *  The zero-based index of the existing sheet to be moved.
         *
         * @param {Number} toSheet
         *  The new zero-based index of the sheet.
         *
         * @returns {Boolean}
         *  Whether moving the sheet was successful.
         */
        this.moveSheet = function (fromSheet, toSheet) {
            // generate and apply the operation
            return this.applyOperations({ name: Operations.MOVE_SHEET, sheet: fromSheet, to: toSheet });
        };

        /**
         * Creates a copy of an existing sheet in the spreadsheet document.
         *
         * @param {Number} fromSheet
         *  The zero-based index of the existing sheet to be copied.
         *
         * @param {Number} toSheet
         *  The zero-based insertion index of the new sheet.
         *
         * @param {String} sheetName
         *  The name of the new sheet. Must not be empty. Must not be equal to
         *  the name of any other sheet (case-insensitive).
         *
         * @returns {String}
         *  The empty string, if the sheet has been copied successfully;
         *  otherwise one of the following error codes:
         *  - 'sheetname:empty': The passed sheet name is empty.
         *  - 'sheetname:invalid': The passed sheet name contains invalid
         *      characters.
         *  - 'sheetname:used': The passed sheet name is already used in the
         *      document.
         *  - 'internal': Internal error while applying the operation.
         */
        this.copySheet = function (fromSheet, toSheet, sheetName) {

            var // the error code for invalid sheet names
                result = null;

            // remove all NPCs from passed sheet name
            sheetName = Utils.cleanString(sheetName);

            // check that the sheet can be renamed (prevent internal application error on failing operation)
            result = validateSheetName(sheetName);
            if (result.length > 0) { return result; }

            // generate and apply the operation
            return this.applyOperations({ name: Operations.COPY_SHEET, sheet: fromSheet, to: toSheet, sheetName: sheetName }) ? '' : 'internal';
        };

        /**
         * Renames a sheet in the spreadsheet document.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet to be renamed.
         *
         * @param {String} sheetName
         *  The new name of the sheet. Must not be empty. Must not be equal to
         *  the name of any other sheet (case-insensitive).
         *
         * @returns {String}
         *  The empty string, if the sheet has been renamed successfully;
         *  otherwise one of the following error codes:
         *  - 'sheetname:empty': The passed sheet name is empty.
         *  - 'sheetname:invalid': The passed sheet name contains invalid
         *      characters.
         *  - 'sheetname:used': The passed sheet name is already used in the
         *      document.
         *  - 'internal': Internal error while applying the operation.
         */
        this.renameSheet = function (sheet, sheetName) {

            var // the error code for invalid sheet names
                result = '';

            // remove all NPCs from passed sheet name
            sheetName = Utils.cleanString(sheetName);

            // check that the sheet can be renamed (prevent internal application error on failing operation)
            result = validateSheetName(sheetName, sheet);
            if (result.length > 0) { return result; }

            // do not generate an operation, if passed name matches old name
            if (this.getSheetName(sheet) === sheetName) { return ''; }

            // generate and apply the operation
            return this.applyOperations({ name: Operations.RENAME_SHEET, sheet: sheet, sheetName: sheetName }) ? '' : 'internal';
        };

        /**
         * Returns the zero-based index of the active (visible) sheet in this
         * document (the value of the 'activeSheet' view attribute).
         *
         * @returns {Number}
         *  The zero-based index of the active sheet.
         */
        this.getActiveSheet = function () {
            return this.getViewAttribute('activeSheet');
        };

        /**
         * Changes the active (visible) sheet in this document (the value of
         * the 'activeSheet' view attribute).
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet to be activated.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.setActiveSheet = function (sheet) {
            return this.setViewAttribute('activeSheet', sheet);
        };

        // defined names ------------------------------------------------------

        /**
         * Creates and applies an 'insertName' document operation to insert a
         * new defined name into the spreadsheet document.
         *
         * @param {String} label
         *  The exact label for the new defined name.
         *
         * @param {String} grammar
         *  The identifier of the formula grammar to be used to parse the
         *  passed formula expression. See constructor of the class Tokenizer
         *  for details.
         *
         * @param {String} formula
         *  The formula expression to be bound to the new defined name.
         *
         * @param {Number} refSheet
         *  The index of the reference sheet the formula is related to.
         *
         * @param {Address} refAddress
         *  The address of the reference cell used to interpret relative cell
         *  references in the formula expression.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'name:used': A defined name with the passed label exists already.
         *  - 'name:empty': The specified label is empty.
         *  - 'name:invalid': The specified label contains invalid characters.
         *  - 'name:address': The specified label would be valid, but conflicts
         *      with the representation of a relative cell reference in A1
         *      notation, or a cell reference in R1C1 notation (either English,
         *      e.g. 'R1C1', or according to current UI language, e.g. 'Z1S1'
         *      in German).
         *  - 'formula:invalid': The passed formula expression is invalid.
         *  - 'operation': Internal error while applying the operation.
         */
        this.insertName = function (label, grammar, formula, refSheet, refAddress) {

            // check that the defined name does not exist yet
            if (nameCollection.hasNameModel(label)) {
                return SheetUtils.makeRejected('name:used');
            }

            // the label must be valid (use grammar 'ui' to check for localized R1C1 notation)
            var labelError = this.getFormulaTokenizer('ui').validateNameLabel(label);
            if (labelError) { return SheetUtils.makeRejected(labelError); }

            // parse and validate the formula expression
            var properties = getNameFormulaProperties(grammar, formula, refSheet, refAddress);
            if (!properties) { return SheetUtils.makeRejected('formula:invalid'); }

            // generate and apply the 'insertName' operation
            properties.exprName = label;
            return this.createAndApplyOperations(function (generator) {
                generator.generateOperation(Operations.INSERT_NAME, properties);
            });
        };

        /**
         * Creates and applies a 'changeName' document operation to change the
         * label or formula definition of an existing defined name.
         *
         * @param {String} label
         *  The label of the defined name to be changed.
         *
         * @param {String|Null} newLabel
         *  The new label for the defined name. If set to null, the label of
         *  the defined name will not be changed.
         *
         * @param {String} grammar
         *  The identifier of the formula grammar to be used to parse the new
         *  formula expression. See constructor of the class Tokenizer for
         *  details. Ignored, if parameter 'newFormula' is null.
         *
         * @param {String|Null} newFormula
         *  The new formula expression to be bound to the defined name. If
         *  omitted, the formula expression of the defined name will not be
         *  changed (an invalid formula expression in the defined name will be
         *  retained).
         *
         * @param {Number} refSheet
         *  The index of the reference sheet the formula is related to.
         *
         * @param {Address} refAddress
         *  The address of the reference cell used to interpret relative cell
         *  references in the formula expression.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'name:used': A defined name with the new label exists already.
         *  - 'name:empty': The new label is empty.
         *  - 'name:invalid': The new label contains invalid characters.
         *  - 'name:address': The new label would be valid, but conflicts with
         *      the representation of a relative cell reference in A1 notation,
         *      or a cell reference in R1C1 notation (either English, e.g.
         *      'R1C1', or according to current UI language, e.g. 'Z1S1' in
         *      German).
         *  - 'formula:invalid': The passed formula expression is invalid.
         *  - 'operation': Internal error while applying the operation.
         */
        this.changeName = function (label, newLabel, grammar, newFormula, refSheet, refAddress) {

            // check that the name exists
            if (!nameCollection.hasNameModel(label)) {
                return $.Deferred().reject();
            }

            // the properties for the 'changeName' operation
            var properties = {};

            // check the passed new label (ignore unchanged existing invalid labels, e.g. from import filter)
            if (_.isString(newLabel) && (label !== newLabel)) {

                // new label must not exist yet in the document (but allow to change case of an existing label)
                if ((label.toUpperCase() !== newLabel.toUpperCase()) && nameCollection.hasNameModel(newLabel)) {
                    return SheetUtils.makeRejected('name:used');
                }

                // the label must be valid (use grammar 'ui' to check for localized R1C1 notation)
                var labelError = this.getFormulaTokenizer('ui').validateNameLabel(newLabel);
                if (labelError) { return SheetUtils.makeRejected(labelError); }

                // add the new label to the operation properties
                properties.newName = newLabel;
            }

            // parse and validate the formula expression
            if (_.isString(newFormula)) {
                _.extend(properties, getNameFormulaProperties(grammar, newFormula, refSheet, refAddress));
                if (!('formula' in properties)) { return SheetUtils.makeRejected('formula:invalid'); }
            }

            // nothing to do without any changed properties
            if (_.isEmpty(properties)) { return $.when(); }

            // generate and apply the 'changeName' operation
            properties.exprName = label;
            return this.createAndApplyOperations(function (generator) {
                generator.generateOperation(Operations.CHANGE_NAME, properties);
            });
        };

        /**
         * Creates and applies a 'deleteName' document operation to delete a
         * defined name from the document.
         *
         * @param {String} label
         *  The label of the defined name to be deleted.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'operation': Internal error while applying the operation.
         */
        this.deleteName = function (label) {

            // check that the name exists
            if (!nameCollection.hasNameModel(label)) {
                return $.Deferred().reject();
            }

            // generate and apply the 'deleteName' operation
            return this.createAndApplyOperations(function (generator) {
                generator.generateOperation(Operations.DELETE_NAME, { exprName: label });
            });
        };

        // range contents -----------------------------------------------------

        /**
         * Returns the results and display strings of all cells contained in
         * the passed cell ranges. If any of the sheet cell collections does
         * not cover all passed ranges, the missing data will be requested from
         * the server.
         *
         * @param {Array<Range3DArray>} rangesArray
         *  An array of arrays (!) of cell range addresses. Each array element
         *  is treated as an independent range list. Each cell range address in
         *  the inner arrays MUST be an instance of the class Range3D including
         *  the sheet indexes of the range. Each range MUST refer to a single
         *  sheet only (no sheet intervals), but different ranges in an array
         *  may refer to different sheets.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.hidden=false]
         *      If set to true, all cells contained in hidden columns or rows
         *      will be included into the result. By default, only visible
         *      cells will be returned.
         *  @param {Boolean} [options.attributes=false]
         *      If set to true, the cell content objects will contain the
         *      merged formatting attributes of the cells.
         *  @param {Boolean} [options.compressed=false]
         *      If set to true, the returned cell arrays will be optimized:
         *      Consecutive cells with equal contents and formatting will be
         *      represented by a single array element with an additional
         *      property 'count'. Especially useful, if large ranges located in
         *      the unused area of the sheet are queried.
         *  @param {Number} [options.maxCount]
         *      If specified, the maximum number of cells per range list that
         *      will be returned in the result, regardless how large the passed
         *      ranges are (the passed ranges will be shortened before the cell
         *      contents will be requested). Example: The first array element
         *      in the parameter 'rangesArray' covers 2000 cells (in several
         *      ranges), the second array element covers 500 cells. With this
         *      option set to the value 1000, the first range list will be
         *      shortened to 1000 cells, the second range list will be resolved
         *      completely. If omitted, and the ranges are too large (as
         *      defined by the constant SheetUtils.MAX_QUERY_CELL_COUNT), the
         *      entire request will be rejected (in compressed mode, the size
         *      of the compressed result array will be checked against that
         *      constant, not the number of cells in the original ranges).
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved with the
         *  cell contents referred by the passed ranges. The result will be an
         *  array of arrays of cell content objects (the length of the outer
         *  result array will be equal to the length of the 'rangesArray'
         *  parameter). Each cell object in the inner arrays will contain the
         *  following properties:
         *  - {String} cellData.display
         *      The display string of the cell result (will be the empty string
         *      for blank cells).
         *  - {Number|String|Boolean|ErrorCode|Null} cellData.result
         *      The typed result value (will be null for blank cells).
         *  - {Object} [cellData.attrs]
         *      The merged formatting attributes of the cell. Will only be set,
         *      if the option 'attributes' has been set to true (see above).
         *  - {Object} [cellData.format]
         *      Additional number format information for the cell format code.
         *      Will only be set, if the option 'attributes' has been set (see
         *      above).
         *  - {Number} [cellData.count]
         *      Always set in compressed mode (see option 'compressed' above).
         *      Contains the number of consecutive cells with equal contents
         *      and formatting represented by this array element.
         *  The cells in the inner arrays will be in order of the range lists.
         *  Cells from a single range will be ordered row-by-row.
         */
        this.queryCellContents = function (rangesArray, options) {

            var // the Deferred object that will be resolved with the cell contents
                def = $.Deferred(),
                // the abortable promise returned by this method
                promise = this.createAbortablePromise(def),
                // whether to include hidden cells
                hidden = Utils.getBooleanOption(options, 'hidden', false),
                // the maximum number of cells in the result
                maxCount = Utils.getIntegerOption(options, 'maxCount', null, 1),
                // the shortened ranges, mapped by sheet index
                rangesMap = {},
                // the promises of all cell collection queries
                allQueries = [];

            // check sheet index, reduce the ranges to their visible parts
            rangesArray = rangesArray.map(function (ranges) {

                // check sheet index contained in the range
                ranges = ranges.filter(function (range) {
                    if (this.hasSheet(range.sheet1) && range.singleSheet()) { return true; }
                    Utils.warn('SpreadsheetModel.queryCellContents(): invalid sheet index in range address');
                    return false;
                }, this);

                // reduce the ranges to their visible parts
                if (!hidden) {
                    ranges = Range3DArray.map(ranges, function (range) {
                        var visibleRanges = self.getSheetModel(range.sheet1).getVisibleRanges(range);
                        return Range3DArray.createFromRanges(visibleRanges, range.sheet1);
                    });
                }

                // shorten the ranges if specified
                if (_.isNumber(maxCount)) {
                    ranges = ranges.shortenTo(maxCount);
                }

                return ranges;
            }, this);

            // reject the entire request, if the ranges are too large
            if (rangesArray.some(function (ranges) { return ranges.cells() > SheetUtils.MAX_QUERY_CELL_COUNT; })) {
                def.reject('overflow');
                return promise;
            }

            // build separate arrays of range lists (instances of RangeArray without sheet indexes) per sheet
            rangesArray.forEach(function (ranges, arrayIndex) {
                ranges.forEach(function (range) {
                    // insert the range into the correct sub-array in the map
                    var mapEntry = rangesMap[range.sheet1] || (rangesMap[range.sheet1] = _.times(rangesArray.length, function () { return new RangeArray(); }));
                    // 3D ranges must be converted to simple ranges without sheet indexes
                    mapEntry[arrayIndex].push(range.toRange());
                });
            });

            // adjust processed options before calling cell collection methods (performance)
            options = options || {};
            options.hidden = true; // ranges already reduced to visible parts, do not reduce them again
            delete options.maxCount; // ranges already reduced to maximum count

            // query cell contents from all cell collections, and store all relevant data in the array and map
            _.each(rangesMap, function (sheetRangesArray, sheetKey) {

                var // zero-based index of the current sheet
                    sheet = parseInt(sheetKey, 10),
                    // cell collection instance of the current sheet
                    cellCollection = self.getSheetModel(sheet).getCellCollection();

                // store array index of the promise in the map, to be able to map from sheet index to promise index quickly
                rangesMap[sheetKey] = { sheet: sheet, index: allQueries.length };

                // query all cell contents from the cell collection of the current sheet
                allQueries.push(cellCollection.queryCellContents(sheetRangesArray, options));
            });

            // wait for all sheet promises before building the result
            $.when.apply($.when, allQueries).done(function () {

                var // $.when() passes the result arrays of all sheets in its function parameters
                    queryResultArrays = arguments,
                    // the resulting cell contents of all sheets, in the correct order
                    resultsArray = [];

                // resolve cell data in order of the original ranges
                rangesArray.forEach(function (ranges, arrayIndex) {

                    var // the cell contents for the current range list
                        result = [];

                    // pick all cell results in order of the requested ranges
                    ranges.forEach(function (range) {

                        var // array index of the associated cell collection query (parameter index)
                            queryIndex = rangesMap[range.sheet1].index,
                            // the array of cell contents for the sheet
                            sheetResults = queryResultArrays[queryIndex][arrayIndex];

                        // move the appropriate number of cell entries into the result array
                        result = result.concat(sheetResults.splice(0, range.cells()));
                    });

                    resultsArray.push(result);
                });

                // resolve the Deferred object with the complete result array
                def.resolve(resultsArray);

            }).fail(_.bind(def.reject, def));

            // return an abortable promise
            return promise;
        };

        // initialization -----------------------------------------------------

        // create instance members
        numberFormatter = new NumberFormatter(this);
        formulaCompiler = new Compiler(this);
        formulaInterpreter = new Interpreter(this);
        nameCollection = new NameCollection(this);

        // register operation handlers for manipulating the sheet collection
        this.registerContextOperationHandler(Operations.INSERT_SHEET, applyInsertSheetOperation);
        registerSheetOperationHandler(Operations.DELETE_SHEET, applyDeleteSheetOperation);
        registerSheetOperationHandler(Operations.RENAME_SHEET, applyRenameSheetOperation);
        registerSheetOperationHandler(Operations.MOVE_SHEET, applyMoveSheetOperation);
        registerSheetOperationHandler(Operations.COPY_SHEET, applyCopySheetOperation);

        // register operation handlers for a sheet in the spreadsheet document
        registerModelMethod(Operations.CHANGE_SHEET, 'applySetAttributesOperation');

        // register operation handlers for cell collections
        registerCollectionMethod(Operations.SET_CELL_CONTENTS, 'getCellCollection', 'applySetCellContentsOperation');
        registerCollectionMethod(Operations.FILL_CELL_RANGE, 'getCellCollection', 'applyFillCellRangeOperation');
        registerCollectionMethod(Operations.CLEAR_CELL_RANGE, 'getCellCollection', 'applyClearCellRangeOperation');
        registerCollectionMethod(Operations.AUTO_FILL, 'getCellCollection', 'applyAutoFillOperation');
        registerCollectionMethod(Operations.INSERT_CELLS, 'getCellCollection', 'applyInsertCellsOperation');
        registerCollectionMethod(Operations.DELETE_CELLS, 'getCellCollection', 'applyDeleteCellsOperation');
        registerCollectionMethod(Operations.INSERT_HYPERLINK, 'getCellCollection', 'applyInsertHyperlinkOperation');
        registerCollectionMethod(Operations.DELETE_HYPERLINK, 'getCellCollection', 'applyDeleteHyperlinkOperation');

        // register operation handlers for merged range collections
        registerCollectionMethod(Operations.MERGE_CELLS, 'getMergeCollection', 'applyMergeCellsOperation');

        // register operation handlers for column collections
        registerCollectionMethod(Operations.INSERT_COLUMNS, 'getColCollection', 'applyInsertOperation');
        registerCollectionMethod(Operations.DELETE_COLUMNS, 'getColCollection', 'applyDeleteOperation');
        registerCollectionMethod(Operations.CHANGE_COLUMNS, 'getColCollection', 'applyChangeOperation');

        // register operation handlers for row collections
        registerCollectionMethod(Operations.INSERT_ROWS, 'getRowCollection', 'applyInsertOperation');
        registerCollectionMethod(Operations.DELETE_ROWS, 'getRowCollection', 'applyDeleteOperation');
        registerCollectionMethod(Operations.CHANGE_ROWS, 'getRowCollection', 'applyChangeOperation');

        // register operation handlers for name collections (allow operations to address global names as well as sheet names)
        registerCollectionMethod(Operations.INSERT_NAME, 'getNameCollection', 'applyInsertOperation', { allowDoc: true });
        registerCollectionMethod(Operations.DELETE_NAME, 'getNameCollection', 'applyDeleteOperation', { allowDoc: true });
        registerCollectionMethod(Operations.CHANGE_NAME, 'getNameCollection', 'applyChangeOperation', { allowDoc: true });

        // register operation handlers for table collections
        registerCollectionMethod(Operations.INSERT_TABLE, 'getTableCollection', 'applyInsertOperation');
        registerCollectionMethod(Operations.DELETE_TABLE, 'getTableCollection', 'applyDeleteOperation');
        registerCollectionMethod(Operations.CHANGE_TABLE, 'getTableCollection', 'applyChangeOperation');
        registerCollectionMethod(Operations.CHANGE_TABLE_COLUMN, 'getTableCollection', 'applyChangeColumnOperation');

        // register operation handlers for validation collections
        registerCollectionMethod(Operations.INSERT_VALIDATION, 'getValidationCollection', 'applyInsertOperation');
        registerCollectionMethod(Operations.DELETE_VALIDATION, 'getValidationCollection', 'applyDeleteOperation');
        registerCollectionMethod(Operations.CHANGE_VALIDATION, 'getValidationCollection', 'applyChangeOperation');

        // register operation handlers for conditional formatting collections
        registerCollectionMethod(Operations.INSERT_COND_FORMAT, 'getCondFormatCollection', 'applyInsertOperation');
        registerCollectionMethod(Operations.DELETE_COND_FORMAT, 'getCondFormatCollection', 'applyDeleteOperation');
        registerCollectionMethod(Operations.CHANGE_COND_FORMAT, 'getCondFormatCollection', 'applyChangeOperation');
        registerCollectionMethod(Operations.INSERT_COND_FORMAT_RULE, 'getCondFormatCollection', 'applyInsertRuleOperation');
        registerCollectionMethod(Operations.DELETE_COND_FORMAT_RULE, 'getCondFormatCollection', 'applyDeleteRuleOperation');
        registerCollectionMethod(Operations.CHANGE_COND_FORMAT_RULE, 'getCondFormatCollection', 'applyChangeRuleOperation');

        // register operation handlers for generic drawing models
        registerDrawingCollectionMethod(Operations.INSERT_DRAWING, 'applyInsertDrawingOperation');
        registerDrawingCollectionMethod(Operations.DELETE_DRAWING, 'applyDeleteDrawingOperation');
        registerDrawingModelMethod(Operations.SET_DRAWING_ATTRIBUTES, 'applySetAttributesOperation');

        // register operation handlers for chart models
        registerDrawingModelMethod(Operations.INSERT_CHART_DATASERIES, 'applyInsertSeriesOperation', 'chart');
        registerDrawingModelMethod(Operations.DELETE_CHART_DATASERIES, 'applyDeleteSeriesOperation', 'chart');
        registerDrawingModelMethod(Operations.SET_CHART_DATASERIES_ATTRIBUTES, 'applyChangeSeriesOperation', 'chart');
        registerDrawingModelMethod(Operations.SET_CHART_AXIS_ATTRIBUTES, 'applyChangeAxisOperation', 'chart');
        registerDrawingModelMethod(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, 'applyChangeGridOperation', 'chart');
        registerDrawingModelMethod(Operations.SET_CHART_TITLE_ATTRIBUTES, 'applyChangeTitleOperation', 'chart');
        registerDrawingModelMethod(Operations.SET_CHART_LEGEND_ATTRIBUTES, 'applyChangeLegendOperation', 'chart');

        // set the column/row count, listen to document attribute changes
        updateMaxColRow(this.getDocumentAttributes());
        this.on('change:attributes', function (event, attributes) {
            updateMaxColRow(attributes);
        });

        // send changed selection to remote clients after activating another sheet
        this.on('change:viewattributes', function (event, attributes) {
            if ('activeSheet' in attributes) {
                sendActiveSelectionDebounced();
            }
        });

        // send changed selection in the active sheet to remote clients
        this.on('change:sheet:viewattributes', function (event, sheet, attributes) {
            if (('selection' in attributes) && (sheet === self.getActiveSheet())) {
                sendActiveSelectionDebounced();
            }
        });

        // application notifies changed data of remote clients
        this.listenTo(app, 'docs:users', changeActiveClientsHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.chain(sheetsCollection).pluck('model').invoke('destroy');
            nameCollection.destroy();
            formulaInterpreter.destroy();
            formulaCompiler.destroy();
            _.invoke(formulaTokenizers, 'destroy');
            numberFormatter.destroy();
            // base class owns undo manager and document styles
            formulaTokenizers = formulaCompiler = formulaInterpreter = null;
            sheetsCollection = numberFormatter = nameCollection = null;
            transformationHandlers = null;
        });

    } // class SpreadsheetModel

    // exports ================================================================

    // derive this class from class EditModel
    return EditModel.extend({ constructor: SpreadsheetModel });

});
