/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/impl/textfuncs', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils'
], function (Utils, SheetUtils, FormulaUtils) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions dealing with text.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     *************************************************************************/

    var // shortcut for the map of error code literals
        ErrorCodes = SheetUtils.ErrorCodes;

    // exports ================================================================

    return {

        ASC: {
            category: 'text',
            hidden: true,
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        BAHTTEXT: {
            category: 'text',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        CHAR: {
            category: 'text',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:int',
            resolve: function (value) {
                if ((value < 1) || (value > 255)) { throw ErrorCodes.VALUE; }
                // TODO: not exact, needs code page conversion
                return String.fromCharCode(value);
            }
        },

        CLEAN: {
            category: 'text',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (value) {
                // TODO: other Unicode characters?
                return value.replace(/[\x00-\x1f\x81\x8d\x8f\x90\x9d]/g, '');
            }
        },

        CODE: {
            category: 'text',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (value) {
                // TODO: not exact, needs code page conversion
                return (value.length === 0) ? ErrorCodes.VALUE : value.charCodeAt(0);
            }
        },

        CONCATENATE: {
            category: 'text',
            minParams: 1,
            type: 'val',
            // each parameter is a single value (function does not concatenate cell ranges or matrixes!)
            signature: 'val:str',
            resolve: function () {
                return _.reduce(arguments, FormulaUtils.add, '');
            }
        },

        DBCS: {
            category: 'text',
            supported: 'ooxml', // ODF name: JIS
            hidden: true,
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        DOLLAR: {
            category: 'text',
            minParams: 1,
            maxParams: 2,
            type: 'val'
        },

        EXACT: {
            category: 'text',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:str',
            resolve: function (value1, value2) {
                return value1 === value2; // exact case-sensitive comparison
            }
        },

        FIND: {
            category: 'text',
            altNamesHidden: { ooxml: 'FINDB' }, // TODO: different behavior for multi-byte character sets?
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'val:str val:str val:int',
            resolve: function (searchText, text, index) {
                if (_.isUndefined(index)) { index = 1; }
                this.checkStringIndex(index);
                index -= 1;
                // do not search, if index is too large (needed if searchText is empty)
                if (index > text.length - searchText.length) { throw ErrorCodes.VALUE; }
                // FIND searches case-sensitive
                index = text.indexOf(searchText, index);
                if (index < 0) { throw ErrorCodes.VALUE; }
                return index + 1;
            }
        },

        FIXED: {
            category: 'text',
            minParams: 1,
            maxParams: 3,
            type: 'val'
        },

        JIS: {
            category: 'text',
            supported: 'odf', // OOXML name: DBCS
            hidden: true,
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        LEFT: {
            category: 'text',
            altNamesHidden: 'LEFTB', // TODO: different behavior for multi-byte character sets?
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: function (text, count) {
                if (_.isUndefined(count)) { count = 1; }
                this.checkStringLength(count);
                return (count === 0) ? text : text.substr(0, count);
            }
        },

        LEN: {
            category: 'text',
            altNamesHidden: 'LENB', // TODO: different behavior for multi-byte character sets?
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: _.property('length')
        },

        LOWER: {
            category: 'text',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (text) { return text.toLowerCase(); }
        },

        MID: {
            category: 'text',
            altNamesHidden: 'MIDB', // TODO: different behavior for multi-byte character sets?
            minParams: 3,
            maxParams: 3,
            type: 'val',
            signature: 'val:str val:int val:int',
            resolve: function (text, start, count) {
                this.checkStringIndex(start);
                this.checkStringLength(count);
                return text.substr(start - 1, count);
            }
        },

        NUMBERVALUE: {
            category: 'text',
            minParams: 1,
            maxParams: 3,
            type: 'val'
        },

        PHONETIC: {
            category: 'text',
            supported: 'ooxml',
            hidden: true,
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'ref:single'
        },

        PROPER: {
            category: 'text',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (text) {
                // TODO: the reg-exp does not include all characters that count as letters
                return text.replace(/[A-Za-z\xAA\xB5\xBA\xC0-\xD6\xD8-\xF6\xF8-\u02B8\u02BB-\u02C1\u02D0-\u02D1\u02E0-\u02E4\u02EE\u0370-\u0373\u0376-\u0377\u037A-\u037D\u0386\u0388-\u038A\u038C\u038E-\u03A1\u03A3-\u03F5\u03F7-\u0481\u048A-\u0523]+/g, Utils.capitalize);
            }
        },

        REPLACE: {
            category: 'text',
            altNamesHidden: { ooxml: 'REPLACEB' }, // TODO: different behavior for multi-byte character sets?
            minParams: 4,
            maxParams: 4,
            type: 'val',
            signature: 'val:str val:int val:int val:str',
            resolve: function (oldText, start, count, newText) {
                this.checkStringIndex(start);
                this.checkStringLength(count);
                start -= 1;
                return Utils.replaceSubString(oldText, start, start + count, newText);
            }
        },

        REPT: {
            category: 'text',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: function (text, repeat) {
                this.checkStringLength(text.length * repeat);
                return (text.length === 0) ? '' : Utils.repeatString(text, repeat);
            }
        },

        RIGHT: {
            category: 'text',
            altNamesHidden: 'RIGHTB', // TODO: different behavior for multi-byte character sets?
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: function (text, count) {
                if (_.isUndefined(count)) { count = 1; }
                this.checkStringLength(count);
                return (count === 0) ? text : text.substr(-count);
            }
        },

        ROT13: {
            category: 'text',
            supported: 'odf',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        SEARCH: {
            category: 'text',
            altNamesHidden: { ooxml: 'SEARCHB' }, // TODO: different behavior for multi-byte character sets?
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'val:str val:str val:int',
            resolve: function (searchText, text, index) {

                // prepare and check the passed index
                if (_.isUndefined(index)) { index = 1; }
                this.checkStringIndex(index);
                index -= 1;

                // return index, if searchText is empty or a 'match all' selector
                if (/^\**$/.test(searchText)) {
                    if (text.length <= index) { throw ErrorCodes.VALUE; }
                    return index + 1;
                }

                // shorten the passed text according to the start index
                text = text.substr(index);

                var // create the regular expression
                    regExp = this.convertPatternToRegExp(searchText),
                    // find first occurrence of the pattern
                    matches = regExp.exec(text);

                // nothing found: throw #VALUE! error code
                if (!_.isArray(matches)) { throw ErrorCodes.VALUE; }

                // return the correct index of the search text
                return matches.index + index + 1;
            }
        },

        SUBSTITUTE: {
            category: 'text',
            minParams: 3,
            maxParams: 4,
            type: 'val',
            signature: 'val:str val:str val:str val:int',
            resolve: function (text, replaceText, newText, index) {

                // check passed index first (one-based index)
                if (_.isNumber(index) && (index <= 0)) { throw ErrorCodes.VALUE; }

                // do nothing, if the text to be replaced is empty
                if (replaceText.length === 0) { return text; }

                var // the regular expression that will find the text(s) to be replaced (SUBSTITUTE works case-sensitive)
                    regExp = new RegExp(_.escapeRegExp(replaceText), 'g');

                // replace all occurrences
                if (_.isUndefined(index)) {
                    return text.replace(regExp, newText);
                }

                // replace the specified occurrence of the text
                var splits = text.split(regExp);
                if (index < splits.length) {
                    splits[index - 1] += newText + splits[index];
                    splits.splice(index, 1);
                    return splits.join(replaceText);
                }

                // index too large: return original text
                return text;
            }
        },

        T: {
            category: 'text',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val',
            resolve: function (value) {
                return _.isString(value) ? value : '';
            }
        },

        TEXT: {
            category: 'text',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        TRIM: {
            category: 'text',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (text) {
                // TRIM removes space characters (U+0020) only, no other white-space or control characters
                return text.replace(/^ +| +$/, '').replace(/  +/g, ' ');
            }
        },

        UNICHAR: {
            category: 'text',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:int',
            resolve: function (value) {
                if ((value < 1) || (value > 65535)) { throw ErrorCodes.VALUE; }
                var char = String.fromCharCode(value);
                if (/[\ud800-\udfff\ufdd0-\ufdef\ufffe\uffff]/.test(char)) { throw ErrorCodes.NA; }
                return char;
            }
        },

        UNICODE: {
            category: 'text',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (value) {
                return (value.length === 0) ? ErrorCodes.VALUE : value.charCodeAt(0);
            }
        },

        UPPER: {
            category: 'text',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (text) { return text.toUpperCase(); }
        },

        USDOLLAR: {
            category: 'text',
            supported: 'ooxml',
            hidden: true,
            minParams: 1,
            maxParams: 2,
            type: 'val'
        },

        VALUE: {
            category: 'text',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: function (text) {
                // Do not use 'val:num' as signature, this would convert Booleans to numbers automatically.
                // The VALUE function does not accept Booleans (the text "TRUE" cannot be converted to a number).
                // Therefore, use convertToNumber() explicitly on a text argument which will throw with Booleans.
                return this.convertToNumber(text);
            }
        }
    };

});
