/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/impl/mathfuncs', [
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils'
], function (SheetUtils, FormulaUtils) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all mathematical spreadsheet functions.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     *************************************************************************/

    var // shortcut for the map of error code literals
        ErrorCodes = SheetUtils.ErrorCodes,

        // mathematical constants
        PI = Math.PI,
        PI_2 = PI / 2,
        PI_180 = PI / 180,
        LN10 = Math.LN10,

        // shortcuts to mathematical functions
        floor = Math.floor,
        ceil = Math.ceil,
        round = Math.round,
        abs = Math.abs,
        pow = Math.pow,
        sqrt = Math.sqrt,
        exp = Math.exp,
        log = Math.log,
        sin = Math.sin,
        cos = Math.cos,
        tan = Math.tan,
        atan = Math.atan,

        // standard options for numeric parameter iterators (method FormulaContext.iterateNumbers())
        SKIP_MAT_SKIP_REF_SKIP_EMPTY = { // id: CSS4
            valMode: 'convert', // value operands: convert strings and Booleans to numbers
            matMode: 'skip', // matrix operands: skip strings and Booleans
            refMode: 'skip', // reference operands: skip strings and Booleans
            // emptyParam: false, // skip all empty parameters
            complexRef: true // accept multi-range and multi-sheet references
        },

        // standard options for numeric parameter iterators (method FormulaContext.iterateNumbers())
        RCONVERT_ALL_SKIP_EMPTY = { // id: RRR0
            valMode: 'rconvert', // value operands: convert strings to numbers (but not Booleans)
            matMode: 'rconvert', // matrix operands: convert strings to numbers (but not Booleans)
            refMode: 'rconvert' // reference operands: convert strings to numbers (but not Booleans)
            // emptyParam: false, // skip all empty parameters
            // complexRef: false // do not accept multi-range and multi-sheet references
        };

    // private global functions ===============================================

    /**
     * Returns the factorial of the passed number. Throws the #NUM! error code,
     * if the passed number is negative or too large.
     */
    var factorial = (function () {
        var FACTS = [], fact = 1;
        while (isFinite(fact)) { FACTS.push(fact); fact *= FACTS.length; }
        return function factorial(number) {
            var fact = FACTS[floor(number)];
            if (!fact) { throw ErrorCodes.NUM; }
            return fact;
        };
    }());

    /**
     * Returns the greatest common divisor of the passed integer numbers.
     *
     * @param {Number} int1
     *  The first integer number.
     *
     * @param {Number} int2
     *  The second integer number.
     *
     * @returns {Number}
     *  The greatest common divisor of the passed numbers.
     */
    function gcd(int1, int2) {
        while (int2 > 0) {
            var tmp = int2;
            int2 = round(int1 % int2);
            int1 = tmp;
        }
        return int1;
    }

    /**
     * Returns the binomial coefficient of the passed input values.
     *
     * @param {Number} n
     *  The upper value of the binomial coefficient (size of the set to be
     *  chosen from).
     *
     * @param {Number} k
     *  The lower value of the binomial coefficient (number of choices from the
     *  set).
     *
     * @returns {Number}
     *  The binomial coefficient, if both values are non-negative, and 'n' is
     *  greater than or equal to 'k'.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed values are invalid, or if the
     *  resulting binomial coefficient is too large.
     */
    function binomial(n, k) {

        // check input values
        if ((n < 0) || (k < 0) || (n < k)) { throw ErrorCodes.NUM; }

        // special cases (n 0) and (n n)
        if ((k === 0) || (k === n)) { return 1; }

        // binomial coefficient is symmetric: (n k) = (n n-k); use lower k for performance
        if (k * 2 > n) { k = n - k; }

        var // the resulting binomial coefficient
            result = n - k + 1;

        // The function isFinite() is quite expensive, by using two nested loops (the inner
        // loop runs up to 10 iterations without checking the result) the entire algorithm
        // becomes about 30% to 40% faster.
        for (var i = 2; (i <= k) && isFinite(result);) {
            for (var j = 0; (j < 10) && (i <= k); i += 1, j += 1) {
                result *= (n - k + i);
                result /= i;
            }
        }

        // check final result
        if (isFinite(result)) { return result; }
        throw ErrorCodes.NUM;
    }

    /**
     * @param {Number} value
     * the value to be rounded
     * @param {Number} mult
     * multiple to round to
     * @returns {Number} the value rounded to a multiple of mult
     * @throws {ErrorCode}
     *  if signs of value and mult are different
     */
    function mround(value, mult) {
        if (value * mult < 0) {
            throw ErrorCodes.NUM;
        }
        var mod = abs(value % mult),
            result = abs(value),
            amult = abs(mult);
        if (mod) {
            if (mod > amult / 2) {
                result += amult - mod;
            } else {
                result -= mod;
            }
        }
        return value < 0 ? -result : result;
    }

    // exports ================================================================

    return {

        ABS: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: abs
        },

        ACOS: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: Math.acos
        },

        ACOSH: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: FormulaUtils.acosh
        },

        ACOT: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return PI_2 - atan(number);
            }
        },

        ACOTH: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: FormulaUtils.acoth
        },

        ASIN: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: Math.asin
        },

        ASINH: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: FormulaUtils.asinh
        },

        ATAN: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: atan
        },

        ATAN2: {
            category: 'math',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: FormulaUtils.arctan2 // parameter order is (x,y), Math.atan2 expects (y,x)
        },

        ATANH: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: FormulaUtils.atanh
        },

        CEILING: {
            category: 'math',
            minParams: { ooxml: 2, odf: 1 },
            maxParams: { ooxml: 2, odf: 3 },
            type: 'val',
            signature: 'val:num val:num val:int',
            resolve: function (number, significance, mode) {
                var result, modulo, minBorder = 1e-16;
                if (_.isUndefined(significance)) {
                    significance = 1;
                }
                if (_.isUndefined(mode)) {
                    mode = this.isODF() ? 0 : 1;
                }
                if (significance === 0) {
                    throw ErrorCodes.VALUE;
                } else if (significance < 0 && number > 0) {
                    throw ErrorCodes.NUM;
                }
                modulo = abs(significance - (number % significance));
                if (number === 0 || modulo < minBorder) {
                    result = number;
                } else {
                    result = (floor(number / significance) + 1) * significance;
                }
                return result;
            }
        },

        'CEILING.MATH': {
            category: 'math',
            supported: 'ooxml',
            minParams: 1,
            maxParams: 3,
            type: 'val'
        },

        'CEILING.PRECISE': {
            category: 'math',
            supported: 'ooxml',
            hidden: true, // replaced by CEILING.MATH in XL2013
            altNamesHidden: 'ISO.CEILING', // replaced by CEILING.MATH in XL2013
            minParams: 1,
            maxParams: 2,
            type: 'val'
        },

        COMBIN: {
            category: 'math',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:int val:int',
            resolve: binomial
        },

        COMBINA: {
            category: 'math',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:int val:int',
            resolve: function (number1, number2) {
                return binomial(number1 + number2 - 1, number2);
            }
        },

        COS: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: cos
        },

        COSH: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: FormulaUtils.cosh
        },

        COT: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return FormulaUtils.divide(1, tan(number));
            }
        },

        COTH: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return FormulaUtils.divide(1, FormulaUtils.tanh(number));
            }
        },

        CSC: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return FormulaUtils.divide(1, sin(number));
            }
        },

        CSCH: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return FormulaUtils.divide(2 * exp(-number), 1 - exp(-2 * number));
            }
        },

        DEGREES: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return number / PI_180;
            }
        },

        EVEN: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                var result = 2 * ceil(abs(number) / 2);
                return (number < 0) ? -result : result;
            }
        },

        EXP: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: exp
        },

        FACT: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:int',
            resolve: factorial
        },

        FACTDOUBLE: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:int',
            resolve: (function () {
                var FACTS = [1, 1];
                while (isFinite(_.last(FACTS))) { FACTS.push(FACTS[FACTS.length - 2] * FACTS.length); }
                return function (number) {
                    return (number in FACTS) ? FACTS[number] : ErrorCodes.NUM;
                };
            }())
        },

        FLOOR: {
            category: 'math',
            minParams: { ooxml: 2, odf: 1 },
            maxParams: { ooxml: 2, odf: 3 },
            type: 'val',
            signature: 'val:num val:num val:int',
            resolve: function (number, significance, mode) {
                var result, modulo, minBorder = 1e-16;
                if (_.isUndefined(mode)) {
                    mode = this.isODF() ? 0 : 1;
                }
                if (_.isUndefined(significance)) {
                    significance = 1;
                }
                if (significance === 0 || (number > 0 && significance < 0)) {
                    throw ErrorCodes.VALUE;
                }
                modulo = abs(significance - (number % significance));
                if (number === 0 || modulo < minBorder) {
                    result = number;
                } else {
                    if (mode === 0 && number < 0) {
                        result = (floor(number / significance) + 1) * significance;
                    } else {
                        result = (floor(number / significance)) * significance;
                    }
                }
                return result;
            }
        },

        'FLOOR.MATH': {
            category: 'math',
            supported: 'ooxml',
            minParams: 1,
            maxParams: 3,
            type: 'val'
        },

        'FLOOR.PRECISE': {
            category: 'math',
            supported: 'ooxml',
            hidden: true, // replaced by FLOOR.MATH in XL2013
            minParams: 1,
            maxParams: 2,
            type: 'val'
        },

        GCD: {
            category: 'math',
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // returns the GCD of the result and the new number
                function aggregate(result, number) {
                    // round down to integers, fail on negative numbers
                    number = floor(number);
                    if (number < 0) { throw ErrorCodes.NUM; }
                    return gcd(result, number);
                }

                // throw #VALUE! without any number (reference to empty cells)
                function finalize(result, count) {
                    if (count === 0) { throw ErrorCodes.VALUE; }
                    return result;
                }

                // start with 0, needed to force calling aggregate() for all function
                // parameters for integer conversion and check for negative numbers
                // Booleans lead to error: GCD(6,TRUE) = #VALUE!
                // empty parameters are ignored: GCD(6,) = 6
                return FormulaUtils.implementNumericAggregation(0, aggregate, finalize, RCONVERT_ALL_SKIP_EMPTY);
            }())
        },

        INT: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:int',
            resolve: _.identity // always rounds down, e.g. INT(-1.5) = -2
        },

        LCM: {
            category: 'math',
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // updates the LCM in the passed result according to the passed number
                function aggregate(result, number) {
                    // round down to integers, fail on negative numbers
                    number = floor(number);
                    if (number < 0) { throw ErrorCodes.NUM; }
                    return ((result === 0) || (number === 0)) ? 0 : (result * number / gcd(number, result));
                }

                // throw #VALUE! without any number (reference to empty cells)
                function finalize(result, count) {
                    if (count === 0) { throw ErrorCodes.VALUE; }
                    return result;
                }

                // start with 1, needed to force calling aggregate() for all function
                // parameters for integer conversion and check for negative numbers
                // Booleans lead to error: LCM(6,TRUE) = #VALUE!
                // empty parameters are ignored: LCM(6,) = 6 while LCM(6,0) = 0
                return FormulaUtils.implementNumericAggregation(1, aggregate, finalize, RCONVERT_ALL_SKIP_EMPTY);
            }())
        },

        LN: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: log
        },

        LOG: {
            category: 'math',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: function (number, base) {
                return log(number) / (_.isNumber(base) ? log(base) : LN10);
            }
        },

        LOG10: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return log(number) / LN10;
            }
        },

        MOD: {
            category: 'math',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: FormulaUtils.modulo
        },

        MROUND: {
            category: 'math',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: mround
        },

        MULTINOMIAL: {
            category: 'math',
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // use the same simple implementation as Excel: if the sum of all numbers exceeds
                // the value 170, getting factorial in the numerator will fail with a #NUM! error,
                // although the final result of the function would be small enough
                function finalize(numbers) {
                    if (numbers.length === 0) { throw ErrorCodes.VALUE; }
                    var sum = 0, denom = 1;
                    _.each(numbers, function (number) {
                        sum += floor(number);
                        denom *= factorial(number);
                    });
                    return factorial(sum) / denom;
                }

                // create and return the resolver function to be assigned to the 'resolve' option of a function descriptor
                return FormulaUtils.implementNumericAggregationWithArray(finalize, RCONVERT_ALL_SKIP_EMPTY);
            }())
        },

        N: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val',
            resolve: function (value) {
                return _.isNumber(value) ? value : (value === true) ? 1 : 0;
            }
        },

        NEG: {
            category: 'math',
            supported: 'odf',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) { return -number; }
        },

        ODD: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                var result = 2 * ceil((abs(number) + 1) / 2) - 1;
                return (number < 0) ? -result : result;
            }
        },

        PI: {
            category: 'math',
            minParams: 0,
            maxParams: 0,
            type: 'val',
            resolve: _.constant(PI)
        },

        POWER: {
            category: 'math',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: { ooxml: FormulaUtils.power, odf: pow }
        },

        PRODUCT: {
            category: 'math',
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // default result is 0, if no numbers have been found, e.g. =PRODUCT(,)
                function finalize(result, count) { return (count === 0) ? 0 : result; }

                // empty parameters are ignored: PRODUCT(1,) = 1
                return FormulaUtils.implementNumericAggregation(1, FormulaUtils.multiply, finalize, SKIP_MAT_SKIP_REF_SKIP_EMPTY);
            }())
        },

        QUOTIENT: {
            category: 'math',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: function (number1, number2) {
                return FormulaUtils.trunc(FormulaUtils.divide(number1, number2));
            }
        },

        RADIANS: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return number * PI_180;
            }
        },

        RAND: {
            category: 'math',
            minParams: 0,
            maxParams: 0,
            type: 'val',
            resolve: Math.random
        },

        RANDBETWEEN: {
            category: 'math',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: function (min, max) {
                min = ceil(min);
                max = floor(max);
                if (max < min) { throw ErrorCodes.NUM; }
                return floor(Math.random() * (max - min + 1)) + min;
            }
        },

        ROUND: {
            category: 'math',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:int',
            resolve: function (number, digits) {
                var mult = pow(10, -digits);
                return mround(number, number < 0 ? -mult : mult);
            }
        },

        ROUNDDOWN: {
            category: 'math',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:int',
            resolve: function (number, digits) {
                var mult = pow(10, -digits),
                    smult = (number < 0) ? -mult : mult,
                    result = mround(number, smult);
                return ((result !== number) && (abs(result) > abs(number))) ? (result - smult) : result;
            }
        },

        ROUNDUP: {
            category: 'math',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:int',
            resolve: function (number, digits) {
                var mult = pow(10, -digits),
                    smult = (number < 0) ? -mult : mult,
                    result = mround(number, smult);
                return ((result !== number) && (abs(result) < abs(number))) ? (result + smult) : result;
            }
        },

        SEC: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return FormulaUtils.divide(1, cos(number));
            }
        },

        SECH: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return FormulaUtils.divide(2 * exp(-number), 1 + exp(-2 * number));
            }
        },

        SERIESSUM: {
            category: 'math',
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        SIGN: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return (number < 0) ? -1 : (number > 0) ? 1 : 0;
            }
        },

        SIN: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: sin
        },

        SINH: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: FormulaUtils.sinh
        },

        SQRT: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: sqrt
        },

        SQRTPI: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return sqrt(PI * number);
            }
        },

        SUBTOTAL: {
            category: 'math',
            minParams: 2,
            type: 'val',
            signature: 'val ref'
        },

        SUM: {
            category: 'math',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters are zero and can be ignored
            resolve: FormulaUtils.implementNumericAggregation(0, FormulaUtils.add, _.identity, SKIP_MAT_SKIP_REF_SKIP_EMPTY)
        },

        SUMIF: {
            category: 'math',
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'ref val:any ref',
            resolve: FormulaUtils.implementFilterAggregation(0, FormulaUtils.add, _.identity)
        },

        SUMIFS: {
            category: 'math',
            supported: 'ooxml',
            minParams: 3,
            repeatParams: 2,
            type: 'val',
            signature: 'ref ref val:any'
        },

        SUMPRODUCT: {
            category: 'math',
            minParams: 1,
            type: 'val'
        },

        SUMSQ: {
            category: 'math',
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // adds the square of the number to the sum
                function aggregate(result, number) { return result + number * number; }

                // empty parameters are zero and can be ignored
                return FormulaUtils.implementNumericAggregation(0, aggregate, _.identity, SKIP_MAT_SKIP_REF_SKIP_EMPTY);
            }())
        },

        TAN: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: tan
        },

        TANH: {
            category: 'math',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: FormulaUtils.tanh
        },

        TRUNC: {
            category: 'math',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:int',
            resolve: function (number, digits) {

                // nothing to do for zeros
                if (number === 0) { return 0; }

                // default value for missing digits
                digits = digits || 0;
                // simpler implementation for integer truncation
                if (digits === 0) { return FormulaUtils.trunc(number); }

                var // absolute number
                    absNum = abs(number),
                    // exponent of the passed number
                    exp10 = floor(log(absNum) / LN10),
                    // mantissa of the passed number
                    mant = absNum / pow(10, exp10);

                // ignore the exponent when truncating the mantissa to the specified digits
                digits += exp10;
                // truncating mantissa (which is always less than 10) to entire tens results in zero
                if (digits < 0) { return 0; }
                // truncating to more than 14 decimal places does not change the passed number
                if (digits >= 14) { return number; }
                // truncate and construct the result
                return ((number < 0) ? -1 : 1) * floor(mant * pow(10, digits)) * pow(10, exp10 - digits);
            }
        }
    };

});
