/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/impl/complexfuncs', [
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/complex'
], function (SheetUtils, FormulaUtils, Complex) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions dealing with complex
     * numbers.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     * In spreadsheet documents, complex numbers are represented by real
     * numbers (complex number without imaginary part), or by strings looking
     * like complex numbers, e.g. '2+3i' or '-4j'. The client-side formula
     * engine provides the function signature type 'val:comp', and the class
     * Complex whose instances represent complex numbers converted from real
     * numbers, or from string representations of complex numbers.
     *
     *************************************************************************/

    var // shortcut for the map of error code literals
        ErrorCodes = SheetUtils.ErrorCodes,

        // shortcuts to mathematical functions
        sin = Math.sin,
        cos = Math.cos,
        sinh = FormulaUtils.sinh,
        cosh = FormulaUtils.cosh;

    // private global functions ===============================================

    /**
     * Returns a resolver function for IMSUM and IMPRODUCT that reduces all
     * complex numbers of all function parameters to their sum or product.
     *
     * @param {Function} callback
     *  The aggregation callback function. Receives two complex numbers, and
     *  must return the resulting complex number (imaginary unit can be omitted
     *  in the resulting complex number).
     *
     * @returns {Function}
     *  The resulting function implementation for IMSUM and IMPRODUCT.
     */
    function implementComplexAggregation(callback) {

        // create the resolver callback function, will be called with FormulaContext context
        return function complexAggregator() {

            var // the resulting complex product
                result = null,
                // whether any parameter was non-empty
                hasAny = false;

            this.iterateOperands(0, function (operand) {
                this.iterateValues(operand, function (value) {
                    var complex = this.convertToComplex(value);
                    if (result) {
                        this.checkComplexUnits(result, complex);
                        result = callback.call(this, result, complex);
                        result.unit = result.unit || complex.unit;
                    } else {
                        result = new Complex(complex.real, complex.imag, complex.unit);
                    }
                }, {
                    emptyParam: false, // empty parameters are skipped: IMSUM("1+i",) = "1+i"
                    complexRef: false // do not accept multi-range and multi-sheet references
                });
                if (!operand.isEmpty()) { hasAny = true; }
            });

            // only empty arguments: return #N/A instead of zero
            if (!hasAny) { throw ErrorCodes.NA; }
            // only references to empty cells: return 0
            return _.isObject(result) ? result : new Complex(0, 0);
        };
    }

    // exports ================================================================

    return {

        COMPLEX: {
            category: 'complex',
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'val:num val:num val:str',
            resolve: function (real, imag, unit) {
                // use 'i', if imaginary unit is missing (undefined), empty parameter (null), or empty string
                if (!unit) { unit = 'i'; }
                // unit must be lower-case 'i' or 'j'
                if ((unit !== 'i') && (unit !== 'j')) { throw ErrorCodes.VALUE; }
                // create a complex number (will be converted to a string)
                return new Complex(real, imag, unit);
            }
        },

        IMABS: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) { return c.abs(); }
        },

        IMAGINARY: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) { return c.imag; }
        },

        IMARGUMENT: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) { return c.arg(); }
        },

        IMCONJUGATE: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                return new Complex(c.real, -c.imag, c.unit);
            }
        },

        IMCOS: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                return new Complex(cos(c.real) * cosh(c.imag), -sin(c.real) * sinh(c.imag), c.unit);
            }
        },

        IMCOSH: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                return new Complex(cosh(c.real) * cos(c.imag), sinh(c.real) * sin(c.imag), c.unit);
            }
        },

        IMCOT: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                var quot = cos(2 * c.real) - cosh(2 * c.imag);
                return new Complex(-sin(2 * c.real) / quot, sinh(2 * c.imag) / quot, c.unit);
            }
        },

        IMCSC: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                var quot = cos(2 * c.real) - cosh(2 * c.imag);
                return new Complex(-2 * sin(c.real) * cosh(c.imag) / quot, 2 * cos(c.real) * sinh(c.imag) / quot, c.unit);
            }
        },

        IMCSCH: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                var quot = cosh(2 * c.real) - cos(2 * c.imag);
                return new Complex(2 * sinh(c.real) * cos(c.imag) / quot, -2 * cosh(c.real) * sin(c.imag) / quot, c.unit);
            }
        },

        IMDIV: {
            category: 'complex',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:comp val:comp',
            resolve: function (c1, c2) {
                this.checkComplexUnits(c1, c2);
                var quot = c2.real * c2.real + c2.imag * c2.imag;
                // division of complex number by zero results in #NUM! instead of #DIV/0!
                // -> do not use the divide() helper function here, invalid coefficients
                // in the returned complex number will be resolved to #NUM! automatically
                return new Complex(
                    (c1.real * c2.real + c1.imag * c2.imag) / quot,
                    (c1.imag * c2.real - c1.real * c2.imag) / quot,
                    c1.unit
                );
            }
        },

        IMEXP: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) { return c.exp(); }
        },

        IMLN: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) { return c.log(1); }
        },

        IMLOG10: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) { return c.log(Math.LN10); }
        },

        IMLOG2: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) { return c.log(Math.LN2); }
        },

        IMPOWER: {
            category: 'complex',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:comp val:num',
            resolve: function (c, exp) { return c.pow(exp); }
        },

        IMPRODUCT: {
            category: 'complex',
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: implementComplexAggregation(function (c1, c2) {
                return new Complex(c1.real * c2.real - c1.imag * c2.imag, c1.real * c2.imag + c1.imag * c2.real);
            })
        },

        IMREAL: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) { return c.real; }
        },

        IMSEC: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                var quot = cos(2 * c.real) + cosh(2 * c.imag);
                return new Complex(2 * cos(c.real) * cosh(c.imag) / quot, 2 * sin(c.real) * sinh(c.imag) / quot, c.unit);
            }
        },

        IMSECH: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                var quot = cosh(2 * c.real) + cos(2 * c.imag);
                return new Complex(2 * cosh(c.real) * cos(c.imag) / quot, -2 * sinh(c.real) * sin(c.imag) / quot, c.unit);
            }
        },

        IMSIN: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                return new Complex(sin(c.real) * cosh(c.imag), cos(c.real) * sinh(c.imag), c.unit);
            }
        },

        IMSINH: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                return new Complex(sinh(c.real) * cos(c.imag), cosh(c.real) * sin(c.imag), c.unit);
            }
        },

        IMSQRT: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) { return c.pow(0.5); }
        },

        IMSUB: {
            category: 'complex',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:comp val:comp',
            resolve: function (c1, c2) {
                this.checkComplexUnits(c1, c2);
                return new Complex(c1.real - c2.real, c1.imag - c2.imag, c1.unit);
            }
        },

        IMSUM: {
            category: 'complex',
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: implementComplexAggregation(function (c1, c2) {
                return new Complex(c1.real + c2.real, c1.imag + c2.imag);
            })
        },

        IMTAN: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:comp',
            resolve: function (c) {
                var quot = cos(2 * c.real) + cosh(2 * c.imag);
                return new Complex(sin(2 * c.real) / quot, sinh(2 * c.imag) / quot, c.unit);
            }
        }
    };

});
