/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/complex', [
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils'
], function (SheetUtils, FormulaUtils) {

    'use strict';

    var // shortcut for the map of error code literals
        ErrorCodes = SheetUtils.ErrorCodes;

    // class Complex ==========================================================

    /**
     * Representation of a complex number literal: a data structure with a
     * floating-point property for the real coefficient, and a floating-point
     * property for the imaginary coefficient.
     *
     * @constructor
     *
     * @property {Number} real
     *  The real coefficient of the complex number.
     *
     * @property {Number} imag
     *  The imaginary coefficient of the complex number.
     *
     * @property {String|Null} unit
     *  The imaginary unit (either 'i' or 'j'). If set to null, the imaginary
     *  unit to be used is undetermined yet (by default, 'i' will be used).
     */
    function Complex(real, imag, unit) {

        this.real = real;
        this.imag = imag;
        this.unit = _.isString(unit) ? unit : null;

    } // class Complex

    // public methods ---------------------------------------------------------

    /**
     * Returns the absolute value of this complex number.
     *
     * @returns {Number}
     *  The absolute value of this complex number.
     */
    Complex.prototype.abs = function () {
        return Math.sqrt(this.real * this.real + this.imag * this.imag);
    };

    /**
     * Returns the argument of this complex number, as radiant.
     *
     * @returns {Number}
     *  The argument of this complex number, as radiant.
     *
     * @throws {ErrorCode}
     *  Throws a #DIV/0! error, if this complex number is 0.
     */
    Complex.prototype.arg = function () {
        return FormulaUtils.arctan2(this.real, this.imag);
    };

    /**
     * Returns the exponential of this complex number.
     *
     * @returns {Complex}
     *  The complex exponential (as new instance of the class Complex), using
     *  the imaginary unit of this complex number.
     */
    Complex.prototype.exp = function () {
        var cabs = Math.exp(this.real);
        return new Complex(cabs * Math.cos(this.imag), cabs * Math.sin(this.imag), this.unit);
    };

    /**
     * Returns the power of this complex number to the passed real exponent.
     *
     * @param {Number} exp
     *  The exponent, as a real number.
     *
     * @returns {Complex}
     *  The complex power (as new instance of the class Complex), using the
     *  imaginary unit of this complex number.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if both numbers are zero.
     */
    Complex.prototype.pow = function (exp) {

        // 0^exp is undefined for non-positive exponents, and 0 otherwise
        if ((this.real === 0) && (this.imag === 0)) {
            if (exp <= 0) { throw ErrorCodes.NUM; }
            return new Complex(0, 0, this.unit);
        }

        // calculate complex power via polar representation
        var cabs = Math.pow(this.abs(), exp), carg = exp * this.arg();
        return new Complex(cabs * Math.cos(carg), cabs * Math.sin(carg), this.unit);
    };

    /**
     * Returns the logarithm of this complex number to the passed base.
     *
     * @param {Number} base
     *  The natural logarithm of the base, as a real number (e.g., for the
     *  value 1, this method will return the natural logarithm of this complex
     *  number; for the value Math.LN10, this method will return the decimal
     *  logarithm of this complex number).
     *
     * @returns {Complex}
     *  The complex logarithm (as new instance of the class Complex), using the
     *  imaginary unit of this complex number.
     *
     * @throws {ErrorCode}
     *  Throws a #DIV/0! error, if this complex number is 0.
     */
    Complex.prototype.log = function (base) {
        return new Complex(Math.log(this.abs()) / base, this.arg() / base, this.unit);
    };

    /**
     * Returns the string representation of this complex number for debug
     * logging.
     */
    Complex.prototype.toString = function () {
        return this.real + ((this.imag >= 0) ? '+' : '') + this.imag + (this.unit || 'i');
    };

    // exports ================================================================

    return Complex;

});
