/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/chartmodel', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/drawinglayer/model/drawingmodel',
    'io.ox/office/drawinglayer/view/chartstyleutil',
    'io.ox/office/drawinglayer/view/chartformatter',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/model/formula/tokenarray',
    'io.ox/office/spreadsheet/model/drawing/drawingmodelmixin',
    'io.ox/office/spreadsheet/model/drawing/manualcolorchart',
    'io.ox/office/spreadsheet/model/drawing/axismodel',
    'io.ox/office/spreadsheet/model/drawing/dataseriesmodel',
    'io.ox/office/spreadsheet/model/drawing/legendmodel',
    'io.ox/office/spreadsheet/model/drawing/axistitlemodel',
    'io.ox/office/drawinglayer/lib/canvasjs.min'
], function (Utils, DateUtils, Color, DrawingModel, ChartStyleUtil, ChartFormatter, SheetUtils, Operations, TokenArray, DrawingModelMixin, ManualColorHandler, AxisModel, DataSeriesModel, LegendModel, AxisTitleModel, CanvasJS) {

    'use strict';

    var // convenience shortcuts
        Range3D = SheetUtils.Range3D,
        RangeArray = SheetUtils.RangeArray,
        Range3DArray = SheetUtils.Range3DArray,

        TYPES = {
            'column standard': 'column',
            'column clustered': 'column',
            'column stacked': 'stackedColumn',
            'column percentStacked': 'stackedColumn100',
            'bar standard': 'bar',
            'bar clustered': 'bar',
            'bar stacked': 'stackedBar',
            'bar percentStacked': 'stackedBar100',
            'area standard': 'area',
            'area stacked': 'stackedArea',
            'area percentStacked': 'stackedArea100',
            'line standard': 'line',
            'line clustered': 'line',
            'line stacked': 'line',
            'line percentStacked': 'spline',
            'line standard curved': 'spline',
            'line clustered curved': 'spline',
            'line stacked curved': 'spline',
            'line percentStacked curved': 'spline',
            'scatter standard': 'line',
            'scatter standard curved': 'spline',
            'bubble standard': 'bubble',
            'pie standard': 'pie',
            'donut standard': 'doughnut'
        },

        MARKER_TYPES = {
            circle: 'circle',
            dot: 'circle',
            square: 'square',
            triangle: 'triangle',
            x: 'cross',
            none: 'none'
        },

        MARKER_LIST = ['circle', 'square', 'triangle', 'cross'];

    // class SheetChartModel ==================================================

    /**
     * A class that implements a chart model to be inserted into a sheet of a
     * spreadsheet document.
     *
     * Additionally to the events triggered by the base class DrawingModel,
     * instances of this class trigger the following 'change:drawing' events:
     * - 'insert:series'
     *      After a new data series has been inserted into this chart. Event
     *      handlers receive the logical index of the new data series.
     * - 'delete:series'
     *      After a new data series has been deleted from this chart. Event
     *      handlers receive the last logical index of the data series.
     *
     * @constructor
     *
     * @extends DrawingModel
     * @extends DrawingModelMixin
     *
     * @param {SpreadsheetApplication} app
     *  The application containing this chart model.
     *
     * @param {SheetModel} sheetModel
     *  The model instance of the sheet that contains this chart object.
     *
     * @param {Object} [initAttributes]
     *  Initial formatting attribute set for this chart model.
     */
    function SheetChartModel(sheetModel, initAttributes) {

        var self = this;

        var docModel = sheetModel.getDocModel();

        var controllerToken = null;

        var data = {
            animationEnabled: false,
            culture:  'en',
            // zoomEnabled: true,
            bg: 'white',
            backgroundColor: 'transparent',
            axisX: { stripLines: [] },
            axisY: { stripLines: [] },
            axisZ: { stripLines: [] },
            creditHref: '',
            creditText: '',
            title: { text: '' },
            legend: {},
            data: []
        };

        data.axisY.stripLines[0] = {
            value: 0,
            color: 'black',
            label: '',
            zero: true
        };

        var indexLabelAttrs = {
            fontSize: 10,
            color: { type: 'auto' },
            fontName: 'Arial'
        };

        var colorHandler = null;
        var chartFormatter = null;

        // base constructors --------------------------------------------------

        DrawingModel.call(this, docModel, 'chart', initAttributes, { families: 'chart fill line' });
        DrawingModelMixin.call(this, sheetModel, cloneConstructor);

        /**
         * this is only called by the Drawingframe to initialize the REAL CanvasJS-chart
         */
        this.getModelData = function () {
            return data;
        };

        var axes = {
            x: new AxisModel(self, 'x', 'y'),
            y: new AxisModel(self, 'y', 'x'),
            z: new AxisModel(self, 'z', 'x')
        };

        var mainName = null;
        var mainTitleModel = new AxisTitleModel(self);

        var legendModel = new LegendModel(this, {}, data.legend);

        // private methods ----------------------------------------------------

        /**
         * Returns a clone of this chart model for the specified sheet. Used by
         * the implementation of the public clone() method defined by the class
         * DrawingModelMixin.
         *
         * @param {SheetModel} targetModel
         *  The sheet model that will contain the cloned chart model.
         *
         * @returns {SheetChartModel}
         *  A clone of this chart model, initialized for ownership by the
         *  passed target sheet model.
         */
        function cloneConstructor(targetModel) {

            var // the index of the target sheet
                newSheet = targetModel.getIndex(),
                // new created copy of all attributemodels attributes
                cloneData = self.getCloneData(newSheet);

            // pass private data as hidden argument to the constructor
            return new SheetChartModel(targetModel, cloneData.attrs, cloneData);
        }

        /**
         * Updates this chart model if it refers to data in the specified cell
         * range addresses.
         *
         * @param {Range3DArray|Range3D} ranges
         *  The addresses of the changed cell ranges, or a single cell range
         *  address. The cell range addresses MUST be instances of the class
         *  Range3D with sheet indexes.
         */
        function handleUpdatedRanges(ranges) {

            var // whether any data series overlaps with the changed ranges
                changed = false;

            // check passed ranges if they overlap with the source ranges of the data series
            _(data.data).each(function (dataSeries) {
                if (dataSeries.needUpdate) {
                    changed = true;
                } else if (dataSeries.model.rangesOverlap(ranges)) {
                    changed = true;
                    dataSeries.needUpdate = true;
                }
            });

            if (changed) { self.refresh(); }
        }

        function colRowChangeHandler(event, sheet, interval, attrs) {

            function transformRanges(insert, columns) {
                data.data.forEach(function (dataSeries) {
                    var changed = dataSeries.model.transformRanges(sheet, interval, insert, columns);
                    if (changed) {
                        dataSeries.needUpdate = true;
                    }
                });
            }

            function updateRange(range) {
                handleUpdatedRanges(Range3D.createFromRange(range, sheet));
            }

            switch (event.type) {
            case 'insert:columns':
                transformRanges(true, true);
                break;
            case 'delete:columns':
                transformRanges(false, true);
                break;
            case 'change:columns':
                // attributes may be missing, e.g. after changing column default attributes at the sheet
                if (!attrs || (attrs.column && ('visible' in attrs.column))) {
                    updateRange(docModel.makeColRange(interval));
                }
                break;
            case 'insert:rows':
                transformRanges(true, false);
                break;
            case 'delete:rows':
                transformRanges(false, false);
                break;
            case 'change:rows':
                // attributes may be missing, e.g. after changing row default attributes at the sheet
                if (!attrs || (attrs.row && ('visible' in attrs.row))) {
                    updateRange(docModel.makeRowRange(interval));
                }
                break;
            }
        }

        /**
         * Handles 'docs:update:cells' notifications. If this chart refers to
         * changed cells, the respective data series will be updated.
         */
        function updateNotificationHandler(changedData, allChangedRanges) {
            handleUpdatedRanges(allChangedRanges);
        }

        /**
         * Handles 'change:cells' events triggered by the document model (from
         * any sheet), and refreshes this chart model debounced, if it refers
         * to any of the changed cells.
         */
        var changeCellsHandler = (function () {

            var // whether any data series overlaps with the changed ranges
                changed = false;

            function eventHandler(event, sheet, ranges) {

                // chart is already dirty, no need to check the ranges
                if (changed) { return; }

                // insert sheet index into the ranges (required by series model)
                ranges = Range3DArray.createFromRanges(ranges, sheet);

                // check passed ranges if they overlap with the source ranges of the data series
                _.each(data.data, function (dataSeries) {
                    if (dataSeries.needUpdate) {
                        changed = true;
                    } else if (dataSeries.model.rangesOverlap(ranges)) {
                        dataSeries.needUpdate = changed = true;
                    }
                });
            }

            function refreshChart() {
                if (changed) {
                    changed = false;
                    self.refresh();
                }
            }

            // wait quite a long time before refreshing the chart, cell ranges
            // may be queried in a background loop
            return self.createDebouncedMethod(eventHandler, refreshChart, { delay: 500 });
        }());

        function changeAttributes(evt, mergedAttributes, oldMergedAttributes) {
            var mc = mergedAttributes.chart;
            var oc = oldMergedAttributes.chart;
            var mf = mergedAttributes.fill;
            var of = oldMergedAttributes.fill;
            var ml = mergedAttributes.line;
            var ol = oldMergedAttributes.line;

            if (!_.isEqual(mc, oc) || !_.isEqual(mf, of) || !_.isEqual(ml, ol)) {

                updateChartType();

                if (mc.chartStyleId !== null && mc.chartStyleId !== undefined && mf.color.type === 'auto') {
                    self.updateBackgroundColor();
                }

                changeDrawing(evt, 'series');
            }
        }

        function isXYType() {
            var attributes = self.getMergedAttributes();
            return attributes.chart.type.indexOf('bubble') === 0 || attributes.chart.type.indexOf('scatter') === 0;
        }

        function isLineType() {
            var type = self.getMergedAttributes().chart.type;
            if (type.indexOf('line') >= 0 || type.indexOf('scatter') >= 0) {
                return true;
            }
            return false;
        }

        function changeDrawing(evt, subtype) {
            var lAttrs = legendModel.getMergedAttributes();
            var attributes = self.getMergedAttributes();

            var cjst = getCJSType();

            _(data.data).each(function (dataSeries, index) {
                if (dataSeries.needUpdate) {
                    return;
                }
                if (subtype === 'series') {
                    var attrs = dataSeries.model.getMergedAttributes();
                    if (isLineType()) {
                        colorHandler.handleColor(attrs.line, dataSeries, dataSeries.seriesIndex, 'color', data.data.length);
                    } else {
                        colorHandler.handleColor(attrs.fill, dataSeries, dataSeries.seriesIndex, 'color', data.data.length);
                    }
                    colorHandler.handleColor(attrs.fill, dataSeries, dataSeries.seriesIndex, 'markerColor', data.data.length);
                }
                if (cjst === 'pie' || cjst === 'doughnut') {
                    dataSeries.startAngle = attributes.chart.rotation - 90;
                }
                if (lAttrs.legend.pos === 'off') {
                    dataSeries.showInLegend = false;
                } else {
                    dataSeries.showInLegend = true;
                    if (cjst === 'pie' || cjst === 'doughnut') {
                        if (index > 0) {
                            //we only see the first series
                            dataSeries.showInLegend = false;
                        } else if (dataSeries.dataPoints.length > 50) {
                            //canvasjs has rendering problems with to big legend-sizes
                            dataSeries.showInLegend = false;
                        }
                    }
                }

                if (dataSeries.showInLegend && (!dataSeries.dataPoints || !dataSeries.dataPoints.length)) {
                    dataSeries.showInLegend = false;
                }

                ChartStyleUtil.handleCharacterProps(self, indexLabelAttrs, dataSeries, 'indexLabel');

                if (attributes.chart.dataLabel) {
                    dataSeries.indexLabel = '{name}';
                    dataSeries.indexLabelLineColor = null;
                    dataSeries.indexLabelPlacement = 'inside';
                } else {
                    dataSeries.indexLabel = ' ';
                    dataSeries.indexLabelLineColor = 'transparent';
                    dataSeries.indexLabelPlacement = null;
                }

                if (isLineType()) {
                    if (!dataSeries.marker) {
                        var markerIndex = index % MARKER_LIST.length;
                        dataSeries.marker = MARKER_LIST[markerIndex];
                    }

                    dataSeries.legendMarkerType = dataSeries.marker;
                } else {
                    dataSeries.legendMarkerType = 'square';
                }

                _.each(dataSeries.dataPoints, function (dataPoint, pointIndex) {
                    if (isLineType()) {
                        if (data.data.length === 1 && self.isVaryColorEnabled()) {
                            var markerIndex = pointIndex % MARKER_LIST.length;
                            dataPoint.markerType = MARKER_LIST[markerIndex];
                        } else {
                            dataPoint.markerType = dataSeries.marker;
                        }
                        dataPoint.markerSize = sheetModel.getEffectiveZoom() * 7;
                    } else {
                        dataPoint.markerType = null;
                        dataPoint.markerSize = null;
                    }
                });
            });

            chartFormatter.format();

            var mnt = mainTitleModel.getMergedAttributes();
            data.title.text = mnt.text.link[0] || '';
            if (!data.title.text && data.data.length === 1) {
                data.title.text = mainName || '';
            }
            if (data.title.text) {
                ChartStyleUtil.handleCharacterProps(self, mnt.character, data.title);
            }

            legendModel.refreshInfo();
            refreshAxis();
        }

        function getCJSType() {
            var type = self.getChartType();
            var res = TYPES[type];

            if (!res) {
                Utils.warn('no correct charttype found', type);
                if (!type) {
                    res = 'column';
                } else if (!type.indexOf('radar')) {
                    res = 'line';
                } else {
                    res = 'column';
                }
            }

            if (res === 'line' && self.getMergedAttributes().chart.curved === true) {
                res = 'spline';
            }
            return res;
        }

        function updateChartType() {
            var cjs = getCJSType();
            for (var i = 0; i < data.data.length; i++) {
                var series = data.data[i];
                series.type = cjs;
            }
            self.trigger('change:drawing');
        }

        /**
         * Inserts a new data series model into this chart model.
         */
        function insertDataSeries(index, attrs) {

            var dataSeries = {
                type: getCJSType(),
                fillOpacity: 1
            };
            data.data.splice(index, 0, dataSeries);

            dataSeries.marker = MARKER_TYPES[attrs.series.marker];

            var model = new DataSeriesModel(sheetModel, attrs);

            dataSeries.model = model;
            dataSeries.seriesIndex = index;

            dataSeries.needUpdate = true;

            dataSeries.dataPoints = [];
            dataSeries.name = '';

            self.trigger('change:drawing');

            if (controllerToken) {
                updateTokenArrays();
            }

            return true;
        }

        // protected methods --------------------------------------------------

        /**
         * Handler for the document operation 'insertChartDataSeries'.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'insertChartDataSeries' operation.
         */
        this.applyInsertSeriesOperation = function (context) {
            var index = context.getOptInt('series', data.data.length);
            insertDataSeries(index, context.getOptObj('attrs'));
        };

        /**
         * Handler for the document operation 'deleteChartDataSeries'.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'deleteChartDataSeries' operation.
         */
        this.applyDeleteSeriesOperation = function (context) {

            var index = context.getInt('series'),
                seriesData = data.data[index];
            context.ensure(seriesData, 'invalid series index');

            seriesData.model.destroy();
            data.data.splice(index, 1);
            this.trigger('change:drawing');

            if (controllerToken) {
                updateTokenArrays();
            }
        };

        /**
         * Handler for the document operation 'setChartDataSeriesAttributes'.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'setChartDataSeriesAttributes' operation.
         */
        this.applyChangeSeriesOperation = function (context) {

            var seriesData = data.data[context.getInt('series')];
            context.ensure(seriesData, 'invalid series index');

            seriesData.model.setAttributes(context.getObj('attrs'));
            seriesData.needUpdate = true;
            updateTokenArrays();
            this.trigger('change:drawing', 'series');
        };

        /**
         * Handler for the document operation 'setChartAxisAttributes'.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'setChartAxisAttributes' operation.
         */
        this.applyChangeAxisOperation = function (context) {

            var axisModel = axes[context.getStr('axis')];
            context.ensure(axisModel, 'invalid axis identifier');

            axisModel.setAttributes(context.getObj('attrs'));
            this.trigger('change:drawing');
        };

        /**
         * Handler for the document operation 'setChartGridlineAttributes'.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'setChartGridlineAttributes' operation.
         */
        this.applyChangeGridOperation = function (context) {

            var axisModel = axes[context.getStr('axis')];
            context.ensure(axisModel, 'invalid axis identifier');

            var gridModel = axisModel.getGrid();
            context.ensure(gridModel, 'missing grid line model');

            gridModel.setAttributes(context.getObj('attrs'));
            this.trigger('change:drawing');
        };

        /**
         * Handler for the document operation 'setChartTitleAttributes'.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'setChartTitleAttributes' operation.
         */
        this.applyChangeTitleOperation = function (context) {

            var axisId = context.getStr('axis'),
                titleModel = (axisId === 'main') ? mainTitleModel : (axisId in axes) ? axes[axisId].getTitle() : null;
            context.ensure(titleModel, 'invalid axis identifier');

            titleModel.setAttributes(context.getObj('attrs'));
            this.trigger('change:drawing');
        };

        /**
         * Handler for the document operation 'setChartLegendAttributes'.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'setChartLegendAttributes' operation.
         */
        this.applyChangeLegendOperation = function (context) {
            legendModel.setAttributes(context.getObj('attrs'));
            this.trigger('change:drawing', 'series');
        };

        function reduceBigRange(source, index, name) {

            function compareFormats(group, other) {

                var groupFormat = ChartStyleUtil.getNumberFormat(docModel, group);
                var otherFormat = ChartStyleUtil.getNumberFormat(docModel, other);
                if (otherFormat === groupFormat) {
                    group.display = null;
                    group.result += other.result;
                } else if (ChartStyleUtil.isDate(groupFormat) && ChartStyleUtil.isDate(otherFormat)) {
                    group.display = null;
                    group.result += other.result;

                    if (DateUtils.isDateFormat(groupFormat) && DateUtils.isDateFormat(otherFormat)) {
                        //take org format
                    } else if (DateUtils.isTimeFormat(groupFormat) && DateUtils.isTimeFormat(otherFormat)) {
                        //take org format
                    } else if (DateUtils.isDateFormat(groupFormat) && DateUtils.isTimeFormat(otherFormat)) {
                        groupFormat = DateUtils.getDateFormat() + ' ' + DateUtils.getTimeFormat();
                        group.format.cat = 'date';
                    } else if (DateUtils.isTimeFormat(groupFormat) && DateUtils.isDateFormat(otherFormat)) {
                        groupFormat = DateUtils.getDateFormat() + ' ' + DateUtils.getTimeFormat();
                        group.format.cat = 'date';
                    }
                }

                delete group.attrs.cell.numberFormat.id;
                group.attrs.cell.numberFormat.code = groupFormat;
            }

            var range = data.data[index].model.getRange(name);
            var start = range.start;
            var end = range.end;

            var xDist = 1 + end[0] - start[0];
            var yDist = 1 + end[1] - start[1];

            if (xDist > yDist) {
                Utils.warn('chart with 2d ranges as "' + name + '" reduceBigRange() must be implemented!');
                return source;
            } else if (yDist > xDist) {
                var newList = [];
                for (var y = 0; y < yDist; y++) {
                    var group = source[y * xDist];
                    for (var x = 1; x < xDist; x++) {
                        var other = source[y * xDist + x];
                        compareFormats(group, other);
                    }
                    group.display = docModel.getNumberFormatter().formatValue(group.result, ChartStyleUtil.getNumberFormat(docModel, group));
                    newList.push(group);
                }
                return newList;
            }

        }

        // public methods -----------------------------------------------------

        this.setMainName = function (name) {
            mainName = name;
        };

        /**
         * is called by the DrawingController for a selected chart
         */
        this.getChartType = function () {
            var attributes = self.getMergedAttributes();

            var type = attributes.chart.type;
            type = type.replace('2d', '').replace('3d', '');

            var stacking = attributes.chart.stacking;
            if (stacking.length) {
                type = type + ' ' + stacking;
            }
            var curved = attributes.chart.curved;
            if (curved && /(line|scatter)/.test(type)) {
                type = type + ' curved';
            }
            return type;
        };

        this.getChartTypeForGui = function () {
            var chartType = this.getChartType();
            return chartType.replace('clustered', 'standard');
        };

        /**
         * refresh checks if something is changed in the series,
         * if true it calls the 'updateview'-request to the server
         */
        this.refresh = this.createDebouncedMethod($.noop, function () {

            if (!docModel.getApp().isImportFinished()) {
                return;
            }

            var // all ranges to be querried from server, as array of arrays of range addresses
                sourceRanges = null,
                // maps series indexes and source link types to array elements in 'sourceRanges'
                collectInfo = null;

            _(data.data).each(function (seriesData, series) {
                if (!seriesData.needUpdate) { return; }

                if (!sourceRanges) {
                    sourceRanges = [];
                    collectInfo = [];
                }

                var info = { series: series, indexes: {} };

                seriesData.model.iterateTokenArrays(function (tokenArray, name) {
                    var entryRanges = tokenArray.resolveRangeList({ resolveNames: true });
                    if (!entryRanges.empty()) {
                        info.indexes[name] = sourceRanges.length;
                        sourceRanges.push(entryRanges);
                    }
                });
                collectInfo.push(info);

                seriesData.needUpdate = false;
            });

            if (!sourceRanges) {
                changeDrawing();
                return;
            }

            if (sourceRanges.length > 0) {

                var // query cell contents for the data series
                    request = docModel.queryCellContents(sourceRanges, { maxCount: 1000, attributes: true });

                // do not call the handler function, if the chart has been deleted in the meantime
                self.listenTo(request, 'done', function (allContents) {

                    _(collectInfo).each(function (info) {

                        // returns the cell contents for the specified source link type
                        function getCellContents(key) {
                            return _.isNumber(info.indexes[key]) ? allContents[info.indexes[key]] : [];
                        }

                        // series values must exist
                        var valuesSource = getCellContents('values');
                        var titleSource = getCellContents('title');
                        var namesSource = getCellContents('names');
                        var bubblesSource = getCellContents('bubbles');

                        if (namesSource.length && namesSource.length >= valuesSource.length * 2) {
                            // workaround for TOO BIG ranges (2d range)
                            namesSource = reduceBigRange(namesSource, info.series, 'names');
                        }
                        if (titleSource.length && titleSource.length >= valuesSource.length * 2) {
                            // workaround for TOO BIG ranges (2d range)
                            titleSource = reduceBigRange(titleSource, info.series, 'title');
                        }

                        chartFormatter.update(info.series, valuesSource, titleSource, namesSource, bubblesSource);

                    });
                    refreshAxis();
                    self.trigger('change:drawing', 'series');
                });
            } else {
                _(data.data).each(function (seriesData) {
                    seriesData.needUpdate = false;
                });
                self.trigger('change:drawing', 'series');
            }
        }, { delay: 50, mayDelay: 250 });

        this.getLegendModel = function () {
            return legendModel;
        };

        this.getAxisModel = function (axis) {
            return axes[axis];
        };

        this.getTitleModel = function (type) {
            return (type === 'main') ? mainTitleModel : this.getAxisModel(type).getTitle();
        };

        /**
         *  Adds formula tokens to the given containing the source ranges of all data
         *  series of this chart object.
         *
         *  @param {Array} array where tokenarrays are pushed in
         */
        this.getTokenArrays = function (res) {
            updateTokenArrays();
            res.push(controllerToken);
        };

        function updateTokenArrays() {
            if (!controllerToken) {
                controllerToken = new TokenArray(sheetModel);
            }

            var tmpList = {};
            _.each(data.data, function (dataSerie) {
                dataSerie.model.iterateTokenArrays(function (tokenArray) {
                    var formula = tokenArray.getFormula('op');
                    if (formula.length) {
                        tmpList[formula] = true;
                    }
                });
            });

            var formula = _.keys(tmpList).join(',');
            controllerToken.parseFormula('op', formula);
        }

        function refreshAxis() {
            if (!data.data.length) {
                return;
            }
            axes.x.refreshInfo();
            axes.y.refreshInfo();
        }

        function getStyleId() {
            return self.getMergedAttributes().chart.chartStyleId;
        }

        function clearSeriesColor(generator, position) {
            _(data.data).each(function (dataSeries, index) {
                var at = dataSeries.model.getMergedAttributes();
                if (at.fill.color.type !== 'auto' || at.line.color.type !== 'auto') {
                    generator.generateDrawingOperation(Operations.SET_CHART_DATASERIES_ATTRIBUTES, position, { series: index, attrs: { fill: { color: Color.AUTO }, line: { color: Color.AUTO } } });
                }
            });
        }

        function createBackgroundColorOp(styleId) {
            var drAttrs = { chart: { chartStyleId: styleId } };

            var bgColor = colorHandler.getBackgroundColor(styleId);
            if (bgColor) {
                drAttrs.fill = { color: bgColor };
            }
            return drAttrs;
        }

        function sourceExistsComplete(name) {
            if (!data.data.length) {
                return false;
            }
            var exists = true;
            _.find(data.data, function (dataSeries) {
                if (_.isEmpty(dataSeries.model.getMergedAttributes().series[name])) {
                    exists = false;
                    return true;
                }
            });
            return exists;
        }

        function isFirstLabel(axisId) {
            var axis = self.getAxisIdForDrawing(axisId);
            if (axis === 'x') {
                return sourceExistsComplete('title');
            } else {
                return sourceExistsComplete('names');
            }
        }

        this.updateBackgroundColor = function () {
            //only fallback for broken background restore!!!
            var bgColor = colorHandler.getBackgroundColor(getStyleId());
            if (bgColor) {
                var drAttrs = {};
                drAttrs.fill = { color: bgColor };
                self.setAttributes(drAttrs);
            }
        };

        this.isXYType = isXYType;

        /**
         * return true if the chart has only one series or is type of bubble or donut (because there you see only the first series)
         * @returns {Boolean}
         */
        this.isVaryColorEnabled = function () {
            return colorHandler.isVaryColorEnabled();
        };

        this.isAxesEnabled = function () {
            var type = getCJSType();
            return type !== 'pie' && type !== 'doughnut';
        };

        /**
         * it generates a unique-color-ID out of the internal chartStyleId
         * for describing the color patterns
         * return null, if one or more series have a unique fill color
         * @returns {String|null}
         */
        this.getColorSet = function () {
            var indiv = _(data.data).find(function (dataSeries) {
                var at = dataSeries.model.getMergedAttributes();
                if (at.fill.color.type !== 'auto') {
                    return true;
                }
            });

            if (indiv) {
                return null;
            }

            var styleId = getStyleId() - 1;
            var colorSet = styleId % 8;
            return 'cs' + colorSet;
        };

        /**
         * it generates a unique-style-ID out of the internal chartStyleId
         * for describing the background and the highlight of the chart
         * it ignores unique fill colors of the series
         * @returns {String}
         */
        this.getStyleSet = function () {
            var styleSet = Math.floor((getStyleId() - 1) / 8);
            return 'ss' + ((styleSet < 4) ? 0 : styleSet);
        };

        /**
         * updates the interal chartStyleId by the unique ID
         * deletes all series colors if existing
         * @param colorSet
         */
        this.changeColorSet = function (colorSet) {
            sheetModel.createAndApplyOperations(function (generator) {

                var styleId = getStyleId() - 1;
                var csId = colorSet.replace('cs', '') | 0;
                var styleSet = (styleId / 8) | 0;

                styleId = styleSet * 8 + csId + 1;

                var position = self.getSheetPosition(),
                    attrs = createBackgroundColorOp(styleId);

                clearSeriesColor(generator, position);
                generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position, { attrs: attrs });
            });
        };

        /**
         * updates the interal chartStyleId by the unique ID
         * does not touch the series colors
         * @param colorSet
         */
        this.changeStyleSet = function (styleSet) {
            sheetModel.createAndApplyOperations(function (generator) {

                var styleId = getStyleId() - 1;
                var ssId = styleSet.replace('ss', '') | 0;
                var colorSet = styleId % 8;

                styleId = ssId * 8 + colorSet + 1;

                var position = self.getSheetPosition(),
                    attrs = createBackgroundColorOp(styleId);

                generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position, { attrs: attrs });
            });
        };

        /**
         * changes the varycolors flag
         * deletes all series colors if existing, so the colors only orient on the patterns
         * @param colorSet
         */
        this.changeVaryColors = function (state) {
            sheetModel.createAndApplyOperations(function (generator) {

                var position = self.getSheetPosition();

                clearSeriesColor(generator, position);
                generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position, { attrs: { chart: { varyColors: state } } });
            });
        };

        this.getSeriesCount = function () {
            return data.data.length;
        };

        /**
         * must be called for having correct behavior mit "varyColors" in combination with more than one series
         */
        this.firstInit = function () {
            updateChartType();
        };

        /**
         * must be called for having correct behavior mit "new CanvasJS.Chart" is called, because of a crazy bug inside there
         */
        this.resetData = function () {
            data.axisX._oldOptions = null;
            data.axisY._oldOptions = null;
        };

        /**
         * normally the assigned axisId (x or y) will be returned
         * except chart is a bar-type, then x & y are interchanged
         *
         * @param axisId
         * @returns
         */
        this.getAxisIdForDrawing = function (axisId) {
            var chartType = self.getMergedAttributes().chart.type;
            if (chartType.indexOf('bar') === 0) {
                if (axisId === 'x') { return 'y'; }
                if (axisId === 'y') { return 'x'; }
            }
            return axisId;
        };

        /**
         * transfers model-data to CanvasJs-data
         */
        this.updateRenderInfo = function () {
            changeDrawing();
            self.refresh();
        };

        /**
         * iterates all sources and calculates min and max positions,
         * direction of the series-values and the sheetindex
         *
         * @returns {Object}
         *  A descriptor with several properties of the data source:
         *  - {String} [warn]
         *      If existing, a specific warning code for invalid source data:
         *      - 'nodata': Source ranges are not available at all.
         *      - 'sheets': Source ranges are not on the same sheet.
         *      - 'directions': Source ranges are not in the same direction.
         * - {Range} range
         *      The bounding range of all source ranges in the targeted sheet.
         * - {Number} sheet
         *      Sheet index of all source ranges.
         * - {Number} axis
         *      Identifier of the main axis of the source ranges.
         */
        this.getDataSourceInfo = function () {

            var ranges = new RangeArray();
            var mainAxis = null;
            var sheet = null;
            var warn = null;

            _.each(data.data, function (dataSeries) {
                dataSeries.model.iterateTokenArrays(function (tokenArray, name) {
                    var rangeList = tokenArray.resolveRangeList({ resolveNames: true });
                    if (rangeList.empty()) { return; }
                    var range = rangeList.first();
                    if (sheet === null) {
                        sheet = range.sheet1;
                    }
                    if (!range.isSheet(sheet)) {
                        warn = 'sheets';
                    }
                    if (name === 'values') {
                        var axis = null;
                        var col = range.cols();
                        var row = range.rows();
                        if (col > row) {
                            axis = 1;
                        } else if (row > 1) {
                            axis = 0;
                        }
                        if (_.isNumber(axis)) {
                            if (mainAxis === null) {
                                mainAxis = axis;
                            } else if (mainAxis !== axis) {
                                warn = 'directions';
                            }
                        }
                    }
                    // convert the 3D range (with sheet indexes) to a 2D range (without sheet indexes)
                    ranges.push(range.toRange());
                });
            });

            if (warn) { return { warn: warn }; }
            if (sheet === null || ranges.empty()) { return { warn: 'nodata' }; }
            return { range: ranges.boundary(), axis: mainAxis, sheet: sheet };
        };

        /**
         * checks if all dataseries have a sourcelink 'title'
         * which is the first row in a normal chart
         *
         * 'title' in all charts except bar chart, there it is 'names'
         */
        this.isFirstRowLabel = function () {
            return isFirstLabel('x');
        };

        /**
         * checks if all dataseries have a sourcelink 'names'
         * which is the first column in a normal chart
         *
         * 'names' in all charts except bar chart, there it is 'title'
         */
        this.isFirstColLabel = function () {
            return isFirstLabel('y');
        };

        /**
         *
         * @returns {Boolean}
         */
        this.isMarkerOnly = function () {
            var marker = false;
            _.find(data.data, function (dataSeries) {
                var att = dataSeries.model.getMergedAttributes();
                if (att.line.type === 'none') {
                    marker = true;
                }
                return true;
            });
            return marker;
        };

        /**
         * Creates an image replacement from this model with the given
         * extent and image mimeType (either image/png or image/jpeg).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.mimeType]
         *   If specified, the given image mime type will
         *   be used for image creation. Allowed values ATM
         *   are image/png (default) and image/jpeg
         *  @param {Object} [options.size]
         *   If specified, the chart replacement will be
         *   rendered with the given size, otherwise, its
         *   size is retrieved from the model's drawing
         *   attributes.
         *
         * @returns {jQuery|Null}
         *  If successful, a floating jquery img node, containg the image data
         *  as src dataUrl and with the requested width and height set
         *  accordingly.
         */
        this.getReplacementNode = function (options) {

            var drawingRectPx = this.getRectangle();
            if (!drawingRectPx) { return null; }

            var jqImgNode = null,
                modelData = this.getModelData(),
                chartId = 'io-ox-documents-chart-frame' + _.uniqueId(),
                widthHeightAttr = '"width="' + drawingRectPx.width + 'px" height="' + drawingRectPx.height + 'px"',
                jqDrawingDiv = $('<div class="chartnode chartholder" id="' + chartId + '" ' + widthHeightAttr + '>').css({
                    position: 'absolute',
                    left: -drawingRectPx.width,
                    top: -drawingRectPx.height,
                    width: drawingRectPx.width,
                    height: drawingRectPx.height
                });

            $(':root').append(jqDrawingDiv);

            var chartRenderer = new CanvasJS.Chart(chartId, _.extend(modelData, { backgroundColor: modelData.cssBackgroundColor })),
                jqCanvas = $('#' + chartId + ' .canvasjs-chart-canvas'),
                canvas = jqCanvas.get(0);

            try {
                // render chart
                chartRenderer.render();

                var dataUrl = canvas.toDataURL(Utils.getStringOption(options, 'mimeType', 'image/png'));

                if (dataUrl) {
                    jqImgNode = $('<img>').attr({
                        width: drawingRectPx.width,
                        height: drawingRectPx.height,
                        src: dataUrl
                    });

                    jqCanvas = dataUrl = chartRenderer = null;
                }
            } catch (ex) {
                Utils.exception(ex, 'replacement chart image rendering error');
            }

            jqDrawingDiv.remove();
            jqDrawingDiv = null;

            return jqImgNode;
        };

        /**
         *
         * @param state
         */
        this.setMarkerOnly = function (state) {
            sheetModel.createAndApplyOperations(function (generator) {

                var attrs = { line: {} };
                attrs.line.type = (state ? 'none' : 'solid');
                var position = self.getSheetPosition();

                _(data.data).each(function (dataSeries, index) {
                    generator.generateDrawingOperation(Operations.SET_CHART_DATASERIES_ATTRIBUTES, position, { series: index, attrs: attrs });
                });
            });
        };

        this.getCloneData = function (newSheet) {

            var // a data object passed as hidden parameter to the constructor of the clone
                cloneData = { series: [], axes: {}, legend: legendModel.getExplicitAttributes(), title: mainTitleModel.getExplicitAttributes() },
                // a temporary formula token array used to relocate sheet references of chart source links
                tokenArray = new TokenArray(sheetModel, { temp: true }),
                // the index of the own sheet
                oldSheet = sheetModel.getIndex();

            if (_.isUndefined(newSheet)) {
                newSheet = oldSheet;
            }

            _.each(axes, function (axisModel, name) {
                var aClone = {
                    axis: axisModel.getExplicitAttributes()
                };
                if (axisModel.getGrid()) {
                    aClone.grid = axisModel.getGrid().getExplicitAttributes();
                }
                if (axisModel.getTitle()) {
                    aClone.title = axisModel.getTitle().getExplicitAttributes();
                }
                cloneData.axes[name] = aClone;
            });

            _.each(data.data, function (dataSeries) {
                var series = {};

                dataSeries.model.iterateTokenArrays(function (token, name) {
                    tokenArray.parseFormula('op', token.getFormula('op'));
                    tokenArray.relocateSheet(oldSheet, newSheet);
                    series[name] = tokenArray.getFormula('op');
                });
                var nsp = dataSeries.model.getExplicitAttributes();
                nsp.series = series;
                cloneData.series.push(nsp);
            });
            tokenArray.destroy();
            tokenArray = null;

            cloneData.attrs = this.getExplicitAttributes();

            return cloneData;
        };

        // initialization -----------------------------------------------------

        colorHandler = new ManualColorHandler(this, data);
        chartFormatter = new ChartFormatter(docModel, this);

        this.on('change:attributes', changeAttributes);
        this.on('change:drawing', changeDrawing);
        this.listenTo(docModel.getApp(), 'docs:update:cells', updateNotificationHandler);
        this.listenTo(docModel, 'insert:columns delete:columns change:columns insert:rows delete:rows change:rows', colRowChangeHandler);
        this.listenTo(docModel, 'change:cells', changeCellsHandler);

        this.listenTo(docModel.getApp().getImportPromise(), 'done', function () {
            self.executeDelayed(function () {
                //small hack for fastcalc-loading
                if (sheetModel.getIndex() === docModel.getActiveSheet()) {
                    self.refresh();
                }
            });
        });

        // clone private data passed as hidden argument to the c'tor
        (function (args) {
            var cloneData = args[SheetChartModel.length];
            if (_.isObject(cloneData)) {
                self.executeDelayed(function () {
                    _.each(cloneData.axes, function (axisData, axisType) {
                        var targetAxisModel = axes[axisType];
                        targetAxisModel.setAttributes(axisData.axis);
                        if (!_.isEmpty(axisData.grid)) {
                            targetAxisModel.getGrid().setAttributes(axisData.grid);
                        }
                        if (!_.isEmpty(axisData.title)) {
                            targetAxisModel.getTitle().setAttributes(axisData.title);
                        }
                    });
                    _.each(cloneData.series, function (attrs, index) {
                        insertDataSeries(index, attrs);
                    });
                    mainTitleModel.setAttributes(cloneData.title);
                    legendModel.setAttributes(cloneData.legend);
                });
            }
        }(arguments));

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.invoke(axes, 'destroy');
            mainTitleModel.destroy();
            legendModel.destroy();
            chartFormatter.destroy();

            for (var key in data.data) {
                var ds = data.data[key];
                ds.model.destroy();
                //must delete, otherwise cansjs has still reference on this
                delete ds.model;
            }

            self = docModel = sheetModel = mainTitleModel = legendModel = data = axes = colorHandler = chartFormatter = null;
        });

    } // class SheetChartModel

    // exports ================================================================

    // derive this class from class DrawingModel
    return DrawingModel.extend({ constructor: SheetChartModel });

});
