/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/app/controller', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/mixedborder',
    'io.ox/office/editframework/app/editcontroller',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/app/drawingcontroller'
], function (Utils, Border, MixedBorder, EditController, Config, SheetUtils, PaneUtils, DrawingControllerMixin) {

    'use strict';

    // class SpreadsheetController ============================================

    /**
     * The controller of a OX Spreadsheet application.
     *
     * @constructor
     *
     * @extends EditController
     * @extends DrawingControllerMixin
     */
    function SpreadsheetController(app, docModel, docView) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        EditController.call(this, app, docModel, docView);
        DrawingControllerMixin.call(this, app, docModel, docView);

        // initialization -----------------------------------------------------

        // register all controller items
        this.registerDefinitions({

            // selection mode -------------------------------------------------

            // enabled if the document is editable, and no drawing objects are selected
            'document/editable/cell': {
                parent: 'document/editable',
                enable: function () { return !docView.hasDrawingSelection(); }
            },

            // enabled if the document is editable, and at least one drawing object is selected
            'document/editable/drawing': {
                parent: 'document/editable',
                enable: function () { return docView.hasDrawingSelection(); }
            },

            // document view --------------------------------------------------

            // parent item, enabled if the in-place cell edit mode is NOT active
            'view/cell/editmode/off': {
                parent: 'app/valid',
                enable: function () { return !docView.isCellEditMode(); }
            },

            'view/statuspane/show': {
                parent: 'app/valid',
                get: function () { return docView.getStatusPane().isVisible(); },
                set: function (state) { docView.getStatusPane().toggle(state); }
            },

            // index of the active sheet in the array of visible sheets (not model sheet index!)
            'view/sheet/active': {
                parent: 'app/valid',
                get: function () { return docView.getActiveSheet({ visible: true }); },
                set: function (index) { docView.setActiveSheet(index, { visible: true }); }
            },

            'view/sheet/prev': {
                parent: 'app/valid',
                enable: function () { return docView.getActiveSheet({ visible: true }) > 0; },
                set: function () { docView.activatePreviousSheet(); },
                shortcut: { keyCode: 'PAGE_UP', ctrlOrMeta: true, alt: true }
            },

            'view/sheet/next': {
                parent: 'app/valid',
                enable: function () { return docView.getActiveSheet({ visible: true }) + 1 < docView.getVisibleSheetCount(); },
                set: function () { docView.activateNextSheet(); },
                shortcut: { keyCode: 'PAGE_DOWN', ctrlOrMeta: true, alt: true }
            },

            'view/subtotals/type': {
                parent: 'app/valid',
                enable: function () {
                    var subtotals = docView.getSubtotals();
                    return (subtotals.cells > 1) && (this.getValue() in subtotals);
                },
                get: function () { return app.getUserSettingsValue('subtotalType', 'sum'); },
                set: function (subtotalType) { app.setUserSettingsValue('subtotalType', subtotalType); }
            },

            'view/status': {
                parent: 'app/valid',
                enable: function () { return _.isString(docView.getStatusLabel()); },
                get: function () { return docView.getStatusLabel(); }
            },

            'view/formula': {
                parent: 'view/cell/editmode/off',
                enable: function () { return docView.isSingleCellSelection() && !docView.hasDrawingSelection() && _.isString(this.getValue()); },
                get: function () { return docView.getActiveCellEntry().formula; }
            },

            'view/sheet/more': {
                parent: 'app/valid'
            },

            'view/grid/show': {
                parent: 'app/valid',
                get: function () { return docView.getSheetViewAttribute('showGrid'); },
                set: function (state) { docView.setSheetViewAttribute('showGrid', state); }
            },

            'view/split/dynamic': {
                parent: 'view/cell/editmode/off',
                get: function () { return docView.hasDynamicSplit(); },
                set: function (state) { docView.setDynamicSplit(state); }
            },

            'view/split/frozen': {
                parent: 'view/cell/editmode/off',
                get: function () { return docView.hasFrozenSplit(); },
                set: function (value) {
                    if (value === 'toggle') {
                        value = docView.hasFrozenSplit() ? false : true;
                    }
                    docView.setFrozenSplit(value);
                }
            },

            'view/split/frozen/fixed': {
                parent: 'view/cell/editmode/off',
                set: function (value) { docView.setFixedFrozenSplit(value.cols, value.rows); }
            },

            'view/zoom': {
                parent: 'app/valid',
                get: function () { return docView.getZoom(); },
                set: function (zoom) { docView.setZoom(zoom); }
            },

            'view/zoom/dec': {
                parent: 'view/zoom',
                enable: function (zoom) { return zoom > SheetUtils.MIN_ZOOM; },
                set: function () { docView.decZoom(); }
            },

            'view/zoom/inc': {
                parent: 'view/zoom',
                enable: function (zoom) { return zoom < SheetUtils.MAX_ZOOM; },
                set: function () { docView.incZoom(); }
            },

            // sheet operations -----------------------------------------------

            'sheet/attributes': {
                parent: 'document/editable',
                get: function () { return docView.getSheetAttributes().sheet; }
            },

            'sheet/visible': {
                parent: 'sheet/attributes',
                enable: function () { return docView.getVisibleSheetCount() > 1; },
                get: function (attributes) { return attributes.visible; },
                set: function (visible) { docView.setSheetAttributes({ sheet: { visible: visible } }); }
            },

            'sheet/showall': {
                parent: 'document/editable',
                enable: function () { return docView.getHiddenSheetCount() > 0; },
                set: function () { docView.showAllSheets(); }
            },

            'sheet/insert': {
                parent: 'document/editable',
                enable: function () { return docModel.getSheetCount() < Config.MAX_SHEET_COUNT; },
                set: function () { docView.insertSheet(); }
            },

            'sheet/delete': {
                parent: 'document/editable',
                enable: function () { return docView.getVisibleSheetCount() > 1; },
                set: function () { docView.deleteSheet(); }
            },

            'sheet/name': {
                parent: 'document/editable',
                get: function () { return docView.getSheetName(); },
                set: function (sheetName) { docView.renameSheet(sheetName); }
            },

            'sheet/rename/dialog': {
                parent: 'sheet/name',
                set: function () { return docView.showRenameSheetDialog(); }
            },

            'sheet/copy/dialog': {
                parent: ['sheet/insert', 'document/ooxml'],
                set: function () { return docView.showCopySheetDialog(); }
            },

            'sheet/reorder/dialog': {
                parent: 'document/editable',
                enable: function () { return docView.getVisibleSheetCount() > 1; },
                set: function () { return docView.showReorderSheetsDialog(); }
            },

            'sheet/formulas/refresh': {
                parent: 'view/cell/editmode/off',
                set: function () { return docView.getCellCollection().refreshAllFormulaCells(); },
                shortcut: { keyCode: 'F9', shift: true, ctrlOrMeta: true }
            },

            // Parent item for various sheet-level operations in the active sheet, enabled if:
            // - document is in edit mode,
            // - in-place cell edit mode is not active,
            // - the active sheet is not locked.
            // The type of the selection (cells or drawing objects) does not matter.
            'sheet/operation/unlocked': {
                parent: ['document/editable', 'view/cell/editmode/off'],
                enable: function () { return !docView.isSheetLocked(); }
            },

            // Parent item for various sheet-level operations in the active sheet, enabled if:
            // - document is in edit mode,
            // - cell selection (no drawing selection),
            // - in-place cell edit mode is not active.
            // Note that the active sheet may be locked.
            'sheet/operation/cell': {
                parent: ['document/editable/cell', 'view/cell/editmode/off']
            },

            // Parent item for various sheet-level operations in the active sheet. Works similar
            // to 'sheet/operation/cell', but the active sheet must be unlocked to enable this item.
            'sheet/operation/unlocked/cell': {
                parent: 'sheet/operation/cell',
                enable: function () { return !docView.isSheetLocked(); }
            },

            // hyperlink operations -------------------------------------------

            'hyperlink/edit/dialog': {
                parent: 'sheet/operation/unlocked/cell',
                set: function () { return docView.editHyperlink(); }
            },

            'hyperlink/delete': {
                parent: 'sheet/operation/unlocked/cell',
                enable: function () { return _.isString(docView.getCellURL()); },
                set: function () { return docView.deleteHyperlink(); }
            },

            // defined names operations ---------------------------------------

            'names/menu/toggle': {
                parent: 'document/editable/cell',
                get: function () { return docView.getDefinedNamesMenu().isVisible(); },
                set: function (state) { docView.getDefinedNamesMenu().toggle(state); }
            },

            'name/insert/dialog': {
                parent: 'document/editable/cell',
                set: function () { return docView.showInsertNameDialog(); }
            },

            'name/edit/dialog': {
                parent: 'document/editable/cell',
                set: function (label) { return docView.showChangeNameDialog(label); }
            },

            'name/delete': {
                parent: 'document/editable/cell',
                set: function (label) { return docView.deleteName(label); }
            },

            // table operations -----------------------------------------------

            // toggles the table filter, if a table is selected in the active
            // sheet, otherwise toggles the auto filter of the active sheet
            'table/filter': {
                parent: 'sheet/operation/unlocked/cell',
                // TODO: filtering real tables has been disabled, until OOXML filter supports import/export of filter settings
                enable: function () { return !docView.isRealTableSelected(); },
                get: function () { return docView.hasTableFilter(); },
                set: function (state) { return docView.toggleTableFilter(state); }
            },

            // refreshes the filter and sorting of the selected table range
            'table/refresh': {
                parent: 'sheet/operation/unlocked/cell',
                enable: function () { return docView.canRefreshTable(); },
                set: function () { return docView.refreshTable(); }
            },

            // Applies formatting attributes for the specified table column.
            // Expected value is an object with the properties 'tableName', 'tableCol',
            // and 'attributes' (incomplete attribute set with filter and sorting attributes).
            'table/column/attributes': {
                parent: 'sheet/operation/unlocked/cell',
                set: function (data) { return docView.changeTableColumn(data.tableName, data.tableCol, data.attributes); }
            },

            // column operations ----------------------------------------------

            // parent item to get the mixed column attributes
            'column/attributes/mixed': {
                get: function () { return docView.getColumnAttributes(); }
            },

            // parent item to get the column attributes of the active cell
            'column/attributes/active': {
                get: function () { return docView.getActiveColumnAttributes(); }
            },

            'column/insert': {
                parent: 'sheet/operation/unlocked/cell',
                enable: function () { return docView.canInsertColumns(); },
                set: function () { return docView.insertColumns(); }
            },

            'column/delete': {
                parent: 'sheet/operation/unlocked/cell',
                enable: function () { return docView.canDeleteColumns(); },
                set: function () { return docView.deleteColumns(); }
            },

            'column/hide': {
                parent: ['column/attributes/mixed', 'sheet/operation/unlocked/cell'],
                // enabled when any visible columns are selected
                enable: function (attributes) { return attributes.visible !== false; },
                set: function () { return docView.setColumnAttributes({ visible: false }); }
            },

            'column/show': {
                parent: 'sheet/operation/unlocked/cell',
                enable: function () { return docView.canShowColumns(); },
                set: function () { return docView.showColumns(); }
            },

            'column/width/optimal': {
                parent: 'column/hide', // only enabled when any visible columns (that CAN BE hidden) are selected
                set: function () { return docView.setOptimalColumnWidth(); }
            },

            'column/width/active': {
                parent: ['column/attributes/active', 'sheet/operation/unlocked/cell'],
                get: function (attributes) { return attributes.visible ? attributes.width : 0; },
                set: function (value) { return docView.setColumnWidth(value); }
            },

            // row operations -------------------------------------------------

            // parent item to get the mixed row attributes
            'row/attributes/mixed': {
                get: function () { return docView.getRowAttributes(); }
            },

            // parent item to get the row attributes of the active cell
            'row/attributes/active': {
                get: function () { return docView.getActiveRowAttributes(); }
            },

            'row/insert': {
                parent: 'sheet/operation/unlocked/cell',
                enable: function () { return docView.canInsertRows(); },
                set: function () { return docView.insertRows(); }
            },

            'row/delete': {
                parent: 'sheet/operation/unlocked/cell',
                enable: function () { return docView.canDeleteRows(); },
                set: function () { return docView.deleteRows(); }
            },

            'row/hide': {
                parent: ['row/attributes/mixed', 'sheet/operation/unlocked/cell'],
                // enabled when any visible rows are selected
                enable: function (attributes) { return attributes.visible !== false; },
                set: function () { return docView.setRowAttributes({ visible: false }); }
            },

            'row/show': {
                parent: 'sheet/operation/unlocked/cell',
                enable: function () { return docView.canShowRows(); },
                set: function () { return docView.showRows(); }
            },

            'row/height/optimal': {
                parent: 'row/hide', // only enabled when any visible rows (that CAN BE hidden) are selected
                set: function () { return docView.setOptimalRowHeight(); }
            },

            'row/height/active': {
                parent: ['row/attributes/active', 'sheet/operation/unlocked/cell'],
                get: function (attributes) { return attributes.visible ? attributes.height : 0; },
                set: function (value) { return docView.setRowHeight(value); }
            },

            // cell operations on active cell ---------------------------------

            // inserts some text into the active cell
            'cell/active/value': {
                parent: 'sheet/operation/cell', // can be used in locked sheets
                set: function (value) { return docView.setCellContents(value, null, { parse: true }); }
            },

            'function/insert/dialog': {
                parent: 'sheet/operation/cell', // can be used in locked sheets
                set: function () { return docView.insertFunction(); }
            },

            'cell/painter': {
                parent: 'sheet/operation/cell', // can be used in locked sheets
                get: function () { return docView.isFormatPainterActive(); },
                set: function (state) { return docView.activateFormatPainter(state); }
            },

            // cell operations on selection -----------------------------------

            'cell/clear/values': {
                parent: 'sheet/operation/cell', // can be used in locked sheets
                set: function () { return docView.fillCellRanges(null); }
            },

            'cell/clear/all': {
                parent: 'sheet/operation/cell', // can be used in locked sheets
                set: function () { return docView.clearCellRanges(); }
            },

            'cell/merge': {
                parent: 'sheet/operation/unlocked/cell',
                enable: function () { return docView.hasAnyRangeSelected(); },
                get: function () { return docView.hasMergedRangeSelected(); },
                set: function (type) { return docView.mergeRanges(type); }
            },

            'cell/autofill': {
                parent: 'sheet/operation/cell', // can be used in locked sheets
                set: function (fillData) { return docView.autoFill(fillData.border, fillData.count); }
            },

            'cell/autoformula': {
                parent: 'sheet/operation/cell', // can be used in locked sheets
                enable: function () { return docView.getSelectedRanges().length === 1; },
                set: function (funcName) { return docView.insertAutoFormula(funcName); }
            },

            'cell/sort/menu/show': {
                parent: 'sheet/operation/unlocked/cell',
                set: function () { return docView.showSortMenu(); },
                preserveFocus: true
            },

            'cell/sort': {
                parent: 'sheet/operation/unlocked/cell',
                // TODO: sorting real tables has been disabled, until GUI and OOXML filter support sorting tables
                enable: function () { return !docView.isRealTableSelected(); },
                get: function () { return docView.getSortState(); },
                set: function (value) { return docView.sortCellRange('standard', value); }
            },

            'cell/sort/custom': {
                parent: 'sheet/operation/unlocked/cell',
                get: function () { return docView.getSortState('custom'); },
                set: function (value) { return docView.sortCellRange('custom', value); }
            },

            'cell/sort/clear': {
                parent: 'sheet/operation/unlocked/cell',
                set: function () { docView.resetSortOptions('custom'); },
                preserveFocus: true
            },

            'cell/sort/hasheadline': {
                parent: 'sheet/operation/unlocked/cell',
                get: function () { return docView.getSortOption('hasheadline', 'custom'); },
                set: function (value) { docView.setSortOption('hasheadline', value, 'custom'); },
                preserveFocus: true
            },

            'cell/sort/casesensitive': {
                parent: 'sheet/operation/unlocked/cell',
                get: function () { return docView.getSortOption('casesensitive', 'custom'); },
                set: function (value) { docView.setSortOption('casesensitive', value, 'custom'); },
                preserveFocus: true
            },

            'cell/sort/direction': {
                parent: 'sheet/operation/unlocked/cell',
                get: function () { return docView.getSortOption('direction', 'custom'); },
                set: function (value) { docView.setSortOption('direction', value, 'custom'); },
                preserveFocus: true
            },

            'cell/sort/order': {
                parent: 'sheet/operation/unlocked/cell',
                get: function () { return docView.getSortOption('order', 'custom'); },
                set: function (value) { docView.setSortOption('order', value, 'custom'); },
                preserveFocus: true
            },

            'cell/sort/colrow': {
                parent: 'sheet/operation/unlocked/cell',
                get: function () { return docView.getSortOption('colrow', 'custom'); },
                set: function (value) { docView.setSortOption('colrow', value, 'custom'); },
                preserveFocus: true
            },

            // cell attributes ------------------------------------------------

            'cell/attributes': {
                parent: 'document/editable/cell',
                get: function () { return docView.getCellAttributes(); },
                set: function (attributes) { return docView.fillCellRanges(undefined, attributes); }
            },

            'cell/stylesheet': {
                parent: ['cell/attributes', 'view/cell/editmode/off'],
                get: function (attributes) { return attributes.styleId; },
                set: function (styleId) { return docView.fillCellRanges(undefined, { styleId: styleId }); }
            },

            'cell/reset': {
                parent: ['cell/attributes', 'view/cell/editmode/off'],
                set: function () { return docView.clearAttributes(); }
            },

            // cell attributes (family 'cell') ----------------------------

            'cell/attributes/cell': {
                parent: ['cell/attributes', 'view/cell/editmode/off'],
                get: function (attributes) { return attributes.cell; }
            },

            'cell/fillcolor': {
                parent: 'cell/attributes/cell',
                get: function (attributes) { return attributes.fillColor; },
                set: function (color) { return docView.setCellAttribute('fillColor', color); }
            },

            'cell/linebreak': {
                parent: 'cell/attributes/cell',
                get: function (attributes) { return attributes.wrapText; },
                set: function (wrap) { return docView.setCellAttribute('wrapText', wrap); }
            },

            'cell/alignhor': {
                parent: 'cell/attributes/cell',
                get: function (attributes) { return attributes.alignHor; },
                set: function (alignment) { return docView.setCellAttribute('alignHor', alignment); }
            },

            'cell/alignvert': {
                parent: 'cell/attributes/cell',
                get: function (attributes) { return attributes.alignVert; },
                set: function (alignment) { return docView.setCellAttribute('alignVert', alignment); }
            },

            // border mode, as Boolean map with position keys (left, top, ...)
            'cell/border/mode': {
                parent: 'sheet/operation/cell', // can be used in locked sheets
                get: function () { return docView.getBorderMode(); },
                set: function (borderMode) { return docView.setBorderMode(borderMode); }
            },

            // all mixed border attributes of the selection
            'cell/border/attributes': {
                parent: 'document/editable/cell',
                get: function () { return docView.getBorderAttributes(); }
            },

            // enabled if at least one border in the selection is visible
            'cell/border/visible': {
                parent: 'cell/border/attributes',
                enable: function (borderAttributes) { return _.any(borderAttributes, MixedBorder.isVisibleBorder); }
            },

            // a single mixed border for all borders in the selection
            'cell/border/value': {
                parent: ['cell/border/visible', 'view/cell/editmode/off'],
                get: function (borderAttributes) { return MixedBorder.mixBorders(_.values(borderAttributes)); }
            },

            'cell/border/style': {
                parent: 'cell/border/value',
                get: function (border) { return border.style; },
                set: function (value) { return docView.changeVisibleBorders({ style: value }); }
            },

            // width of the border lines, in points (rounded to 1/2 points)
            'cell/border/width': {
                parent: 'cell/border/value',
                get: function (border) { return !MixedBorder.isVisibleBorder(border) ? 0 : _.isNumber(border.width) ? Utils.convertHmmToLength(border.width, 'pt', 0.5) : null; },
                set: function (value) { return docView.changeVisibleBorders({ width: Utils.convertLengthToHmm(value, 'pt') }); }
            },

            'cell/border/color': {
                parent: 'cell/border/value',
                get: function (border) { return border.color; },
                set: function (value) { return docView.changeVisibleBorders({ color: value }); }
            },

            // combined style and width as enumeration as used by Excel
            'cell/border/style/preset': {
                parent: 'cell/border/value',
                get: function (border) { return PaneUtils.getPresetStyleForBorder(border); },
                set: function (style) { return docView.changeVisibleBorders(PaneUtils.getBorderForPresetStyle(style)); }
            },

            'cell/numberformat/category': {
                parent: 'sheet/operation/cell', // can be used in locked sheets
                get: function () { return docView.getNumberFormatCategory(); },
                set: function (category) { return docView.setNumberFormatCategory(category); },
                shortcut: [
                    { keyCode: '1', shift: true, ctrlOrMeta: true, value: 'number' },
                    { keyCode: '2', shift: true, ctrlOrMeta: true, value: 'scientific' },
                    { keyCode: '3', shift: true, ctrlOrMeta: true, value: 'date' },
                    { keyCode: '4', shift: true, ctrlOrMeta: true, value: 'currency' },
                    { keyCode: '5', shift: true, ctrlOrMeta: true, value: 'percent' },
                    { keyCode: '6', shift: true, ctrlOrMeta: true, value: 'standard' }
                ]
            },

            'cell/numberformat/code': {
                parent: 'cell/numberformat/category',
                enable: function (category) { return !/^(standard|custom)$/.test(category); },
                get: function () { return docView.getNumberFormatCode(); },
                set: function (formatCode) { return docView.setNumberFormatCode(formatCode); }
            },

            // cell attributes (family 'character') ---------------------------

            // also enabled in in-place cell edit mode
            'cell/attributes/character': {
                parent: 'cell/attributes',
                get: function (attributes) { return attributes.character; }
            },

            'character/bold': {
                parent: 'cell/attributes/character',
                get: function (attributes) { return attributes.bold; },
                set: function (state) { return docView.setCharacterAttribute('bold', state); },
                shortcut: { keyCode: 'B', ctrlOrMeta: true, value: function (state) { return !state; } }
            },

            'character/italic': {
                parent: 'cell/attributes/character',
                get: function (attributes) { return attributes.italic; },
                set: function (state) { return docView.setCharacterAttribute('italic', state); },
                shortcut: { keyCode: 'I', ctrlOrMeta: true, value: function (state) { return !state; } }
            },

            'character/underline': {
                parent: 'cell/attributes/character',
                get: function (attributes) { return attributes.underline; },
                set: function (state) { return docView.setCharacterAttribute('underline', state); },
                shortcut: { keyCode: 'U', ctrlOrMeta: true, value: function (state) { return !state; } }
            },

            'character/strike': {
                parent: 'cell/attributes/character',
                get: function (attributes) { return _.isString(attributes.strike) ? (attributes.strike !== 'none') : null; },
                set: function (state) { return docView.setCharacterAttribute('strike', state ? 'single' : 'none'); }
            },

            'character/color': {
                parent: 'cell/attributes/character',
                get: function (attributes) { return attributes.color; },
                set: function (color) { return docView.setCharacterAttribute('color', color); }
            },

            'character/fontname': {
                parent: 'cell/attributes/character',
                get: function (attributes) { return attributes.fontName; },
                set: function (fontName) { return docView.setCharacterAttribute('fontName', fontName); }
            },

            'character/fontsize': {
                parent: 'cell/attributes/character',
                get: function (attributes) { return attributes.fontSize; },
                set: function (fontSize) { return docView.setCharacterAttribute('fontSize', fontSize); }
            },

            'character/format': {
                parent: 'app/valid'
            },

            'document/ooxml': {
                enable: function () { return app.isOOXML(); }
            }
        });

        // update GUI after changed selection, and after receiving view updates
        this.listenTo(docView, 'change:sheet:viewattributes change:layoutdata celledit:enter celledit:change celledit:leave', function () { self.update(); });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = app = docModel = docView = null;
        });

    } // class SpreadsheetController

    // exports ================================================================

    // derive this class from class EditController
    return EditController.extend({ constructor: SpreadsheetController });

});
