/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */
define('io.ox/office/portal/portalutils', [
    'io.ox/core/capabilities',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/utils/driveutils',
    'io.ox/office/baseframework/app/extensionregistry',
    'gettext!io.ox/office/portal/main'
], function (Capabilities, Utils, KeyCodes, DateUtils, DriveUtils, ExtensionRegistry, gt) {

    'use strict';

    // TODO maybe there is a better place for this. Maybe backend office settings?
    var OFFICE_CAPABILITY_NAMES = ['text', 'spreadsheet', 'presentation'];

    // static class PortalUtils ===============================================

    // copy tk/utils methods
    var PortalUtils = _.clone(Utils);

    // constants --------------------------------------------------------------

    // static methods ---------------------------------------------------------

    /**
     * Returns the full module name of the specified portal application.
     *
     * @param {String} appBaseName
     *  The base name of the portal application ('text', 'spreadsheet', etc.).
     *
     * @returns {String}
     *  The full module name of the specified portal application.
     */
    PortalUtils.getPortalModuleName = function (appBaseName) {
        return 'io.ox/office/portal/' + appBaseName;
    };

    /**
     * Returns the full module name of the specified editor application
     * associated to a portal application.
     *
     * @param {String} appBaseName
     *  The base name of the editor application ('text', 'spreadsheet', etc.).
     *
     * @returns {String}
     *  The full module name of the specified editor application.
     */
    PortalUtils.getEditorModuleName = function (appBaseName) {
        return 'io.ox/office/' + appBaseName;
    };

    /**
     * Returns the application module name for the given filename.
     *
     * @param {String} filename
     *  Filename as string
     *
     * @returns {String | Null}
     *  Application module name string if it exists, otherwise null.
     */
    PortalUtils.getModuleNameForFile = function (filename) {
        return ExtensionRegistry.getEditModule(filename);
    };

    /**
     * Detects file type and creates an corresponding icon node.
     *
     * @param fileName
     *  da filename
     *
     * @returns {jQuery}
     *  the icon node as jQuery object
     */
    PortalUtils.createDocumentIcon = function (filename) {

        var moduleName = PortalUtils.getModuleNameForFile(filename),
            iconClasses = 'document-icon fa ';

        switch (moduleName) {
        case 'io.ox/office/spreadsheet':
            iconClasses += 'fa-table spreadsheet';
            break;
        case 'io.ox/office/text':
            iconClasses += 'fa-align-left text';
            break;
        case 'io.ox/office/presentation':
            iconClasses += 'fa-picture presentation';
            break;
        default:
            iconClasses += 'fa-file';
            break;
        }

        return $('<i>').addClass(iconClasses);

    };

    /**
     * Formats the timestamp representation for the recents document list:
     * - show only time, if the document is created today
     * - show year only, if the document is not modified in this year.
     *
     * @param {Number} timestamp
     * - milliseconds since Jan 1, 1970
     *
     * @returns {String}
     *  the formatted date as string
     */
    PortalUtils.formatDate = function (timestamp) {

        if (!_.isNumber(timestamp)) { return gt('unknown'); }

        var now = new DateUtils.getLocalDate(),
            timeStampDate = new DateUtils.getLocalDate(timestamp);

        function getTimeString() {
            return DateUtils.format(timeStampDate, DateUtils.getTimeFormat());
        }

        function getDateString() {
            // get locale country default date format
            var formatString = DateUtils.getDateFormat();
            // hide year if the document is modified in this year
            if (timeStampDate.getYear() === now.getYear()) {
                formatString = DateUtils.getNoYearFormat();
            }
            return DateUtils.format(timeStampDate, formatString);
        }

        function isToday(date) {
            return date.getDate() === now.getDate() &&
                date.getMonth() === now.getMonth() &&
                date.getYear() === now.getYear();
        }

        // show only time if the document is last modifed today
        return isToday(timeStampDate) ? getTimeString() : getDateString();

    };

    /**
     * Returns the application base name for the given filename.
     *
     * @param {String} filename
     *  Filename as string
     *
     * @returns {String | Null}
     *  Application base name string if it exists, otherwise null.
     */
    PortalUtils.getAppBaseName = function (filename) {

        var moduleName = PortalUtils.getModuleNameForFile(filename);

        if (!moduleName) { return null; }

        return moduleName.substr(moduleName.lastIndexOf('/') + 1, moduleName.length);
    };

    /**
     * Returns the given file name without the file extension.
     *
     * @param {String} filename
     *  Filename as string
     *
     * @returns {String}
     *  Specified file name as string without file extension.
     */
    PortalUtils.removeFileExtension = function (filename) {

        var shortFileName = filename,
            pos = filename.lastIndexOf('.');

        if (pos > -1) {
            shortFileName = shortFileName.substr(0, pos);
        }

        return shortFileName;
    };

    /**
     * Returns whether the given string ends with a given suffix.
     *
     * @param {String} str
     *  A specified string
     *
     * @param {String} suffix
     *  A specified suffix
     *
     * @returns {Boolean}
     *  Whether the given string ends with the specified suffix.
     */
    PortalUtils.stringEndsWith = function (str, suffix) {
        return str && suffix && str.indexOf(suffix, str.length - suffix.length) > -1;
    };

    /**
     * Returns application base names that are enabled in the Appsuite Capabilities.
     *
     * @returns {Array} appBaseNames
     *  An array of application base name strings.
     */
    PortalUtils.getEnabledApps = function () {

        var enabledApps = [];

        // get enabled office apps from Capabilities
        _.each(OFFICE_CAPABILITY_NAMES, function (name) {
            if (Capabilities.has(name)) { enabledApps.push(name); }
        });

        return enabledApps;

    };

    /**
     * Returns the name of the current active application, so that it corresponds to the
     * type of the file.
     *
     * @returns {String} appBaseName
     *  The application base name string.
     */
    PortalUtils.getActiveApp = function (app) {
        return (app.attributes.title).toLowerCase();
    };

    /**
     * Move focus to from given node to its next or previous sibling. (keyboard accessibility)
     *
     * @param {jQuery} startNode
     *  Starting node.
     *
     * @param {String = 'right'} direction
     *  Available direction strings: 'right','left', 'up', 'down'.
     */
    PortalUtils.moveFocus = function (startNode, direction) {

        if (!startNode) { return; }

        var nodeToFocus = startNode;
        direction = direction || 'right';

        /**
         * Tries to find a vertical neighbor according to cursor direction (up/down)
         *
         * @returns {jQuery} || null
         *  the vertical neighbor object or null no vertical neighbours found.
         */
        function findVerticalSibling() {
            var startNodePosition = startNode.position(),
                startNodeHeight = startNode.outerHeight(true),
                targetNodes = startNode.siblings(),
                targetNode = _.find(targetNodes, function (node) {
                    if (!$(node).attr('tabindex')) { return false; }
                    var nodePosition = $(node).position();
                    if (nodePosition.left !== startNodePosition.left) { return false; }
                    switch (direction) {
                        case 'down':
                            return nodePosition.top === (startNodePosition.top + startNodeHeight);
                        case 'up':
                            return nodePosition.top === (startNodePosition.top - startNodeHeight);
                        default :
                            return false;
                    }
                });
            if (!targetNode) { return null; }
            return targetNode;
        }

        switch (direction) {
            case 'right':
                nodeToFocus = startNode.next();
                // temp workaround: jump over inserted Bootstrap tooltip element if there is any.
                if (!nodeToFocus.attr('tabindex')) { nodeToFocus = nodeToFocus.next(); }
                break;
            case 'left':
                nodeToFocus = startNode.prev();
                break;
            case 'down':
                nodeToFocus = findVerticalSibling('down');
                break;
            case 'up':
                nodeToFocus = findVerticalSibling('up');
                break;
        }

        if (!nodeToFocus) { return; }

        nodeToFocus.focus();
    };

    /**
     * Keyboard accessibility handler for recent documents and templates list.
     *
     * @param {jQuery.Event} event
     *  {Boolean} event.data.grid - Whether grid navigation (up,down,left,right) is required.
     */
    PortalUtils.keydownHandler = function (event) {

        if (!event || !event.target) { return; }

        var startNode = $(event.target),
            isGrid = event.data ? event.data.grid : false;

        if (KeyCodes.matchKeyCode(event, 'ENTER') || KeyCodes.matchKeyCode(event, 'SPACE')) { startNode.trigger('click'); }
        if (KeyCodes.matchKeyCode(event, 'LEFT_ARROW')) { PortalUtils.moveFocus(startNode, 'left'); }
        if (KeyCodes.matchKeyCode(event, 'RIGHT_ARROW')) { PortalUtils.moveFocus(startNode, 'right'); }
        if (KeyCodes.matchKeyCode(event, 'DOWN_ARROW')) { PortalUtils.moveFocus(startNode, isGrid ? 'down' : 'right'); }
        if (KeyCodes.matchKeyCode(event, 'UP_ARROW')) { PortalUtils.moveFocus(startNode, isGrid ? 'up' : 'left'); }
    };

    /**
     * calls CoreMetrics Class (io.ox/metrics/main)
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.moduleName]
     *      moduleName of the current app
     *      with help of the current module the metrics params "app" will be created
     *      and the param "action" gets the app name as prefix except actionModuleName is assigned
     *  @param {String} [options.actionModuleName=options.moduleName]
     *      and the param "action" gets the app name as prefix
     *  @param {String} [options.target]
     *      target is used as "target" and is directly sent to CoreMetrics
     *  @param {String} [options.action]
     *      action gets the prefix of the app and is sent as "action" to the CoreMetrics
     */
    PortalUtils.trackEvent = function (options) {
        require(['io.ox/metrics/main']).done(function (CoreMetrics) {
            var app = Utils.getStringOption(options, 'moduleName', '').replace('io.ox/office/', '');
            var actionApp = Utils.getStringOption(options, 'actionModuleName', Utils.getStringOption(options, 'moduleName', '')).replace('io.ox/office/', '');
            var send = {
                app: app + '-portal',
                target: Utils.getStringOption(options, 'target'),
                action: actionApp + '-' + Utils.getStringOption(options, 'action')
            };
            //console.warn('track', send);
            CoreMetrics.trackEvent(send);
        });
    };

    /**
     * Launch options to create a new document in the default 'Documents'
     * folder in Drive.
     *
     * @returns {Object}
     *  Necessary parameters for creating a new document in the default folder.
     */
    PortalUtils.getNewLaunchOptions = function () {
        return { action: 'new', folderId: DriveUtils.getStandardDocumentsFolderId() };
    };

    // exports ================================================================

    return PortalUtils;

});
